// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_VIDEO_SINK_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_VIDEO_SINK_H

#include <vector>

#include "common.h"
#include "MediaSinkBase.h"
#include "IVideoSinkCallbacks.h"

/**
 * This sink handles video frames that must be projected on the head unit. The general flow of
 * how such a video sink would operate is described below:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      VideoSink* endpoint = new VideoSink(serviceId, galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IVideoSinkCallbacks.
 *      endpoint->addSupportedConfiguration(...);
 *      endpoint->addSupportedConfiguration(...);
 *      endpoint->setMaxUnackedFrames(...);
 *
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 *
 *      setupCallback() gets invoked.
 *          perform any initialization necessary for playback for this codec.
 *
 *      codecConfigCallback() gets invoked.
 *          perform codec specific setup steps.
 *
 *      sourceVideoConfigCallback() gets invoked.
 *          ensure you have a surface of the appropriate resolution to begin playback.
 *
 *      playbackStartCallback gets invoked.
 *
 *      dataAvailableCallback gets invoked.
 *          schedule new frame for decoding
 *
 *      A frame completes rendering.
 *          call endpoint->ackFrames(1); // MediaSinkBase::ackFrames();
 * </pre>
 */
class VideoSink : public MediaSinkBase {
public:
    VideoSink(uint8_t id, MessageRouter* router, bool autoStartProjection)
            : MediaSinkBase(id, router),
              mAutoStartProjection(autoStartProjection),
              mViewingDistance(0),
              mDisplayId(0),
              mDisplayType(DISPLAY_TYPE_MAIN),
              mDisplayIdAndTypeSet(false) {}
    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr);
    void registerCallbacks(const shared_ptr<IVideoSinkCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Set the maximum unacked frames for this sink.
     * @param maxUnackedFrames How many frames should be kept outstanding. Once this limit is hit
     *        the other side will stop sending new video frames.
     */
    void setMaxUnackedFrames(int maxUnackedFrames) {
        mMaxUnackedFrames = maxUnackedFrames;
    }

    /**
     * Set video focus. Should be called when changing video focus in an unsolicited manner.
     * To 'grant' a focus request, set the same focus as was requested. To 'deny' a request,
     * just set the video focus request back to whatever it already was and the phone will
     * adjust its behavior accordingly.
     * @param focusMode The focus mode to set. Can be one of VIDEO_FOCUS_PROJECTED,
     *        VIDEO_FOCUS_NATIVE, or VIDEO_FOCUS_NATIVE_TRANSIENT.
     * @param unsolicited true if responding to a video focus request, false otherwise.
     */
    void setVideoFocus(int focusMode, bool unsolicited);

    /**
     * Sets the viewing distance of this video sink. Should be invoked before this service is
     * registered with the GalReceiver.
     * @param viewingDistance the average screen-to-eye distance in millimeters.
     */
    void setViewingDistance(uint32_t viewingDistance) { mViewingDistance = viewingDistance; }

    /**
     * Sets the display ID and type for multi-display configurations. Should be
     * invoked before this service is registered with the GalReceiver.
     * @param displayId The zero-based display ID. Display IDs must be
     * consecutive, without gaps, and the primary display must have ID 0.
     * @param displayType The type of this display.
     */
    void setDisplayIdAndType(uint32_t displayId, DisplayType displayType) {
        mDisplayId = displayId;
        mDisplayType = displayType;
        mDisplayIdAndTypeSet = true;
    }

    /**
     * Sets the initial_content_keycode. Should only be used on Auxiliary
     * displays to set the content that will be shown on the display at startup.
     * defaults to KEYCODE_NAVIGATION
     * @param keycode the initial content keycode
     */
    void setInitialContentKeycode(KeyCode keycode) {
      mInitialContentKeycode = keycode;
    }

    /**
     * Add a supported video configuration. Multiple video configurations may be
     * supported so this method should be called once for each supported
     * configuration. Note that the order in which you call these methods
     * indicates your preference order unless overridden by calling
     * MediaSinkBase::resetAllowedConfigurations().
     * @param codecResolution This can be one of VIDEO_800x480, VIDEO_1280x720
     * or VIDEO_1920x1080.
     * @param fps frame rate of video. Either 60 or 30.
     * @param widthMargin The number of pixels in the x axis that content wont
     * be rendered to.
     * @param heightMargin The number of pixels in the y axis that content wont
     * be rendered to.
     * @param density The density in dpi that allows proper scaling of the
     * projected screen.
     * @param realDensity The pixel resolution of the video / the rendered video
     * size.
     * @param decoderAdditionalDepth additional frames decoder needs to display
     * a frame.
     * @param pixelAspectRatioE4 The ratio of the physical pixel width over
     * height multiplied by 1e4
     * @param mediaCodecType H264 or H265?
     * @param ui_config the UiConfig for this configuration
     * @return index number of the configuration passed.
     */
    uint32_t addSupportedConfiguration(int codecResolution, int fps,
                                       int32_t density, int32_t realDensity,
                                       int32_t decoderAdditionalDepth,
                                       int32_t pixelAspectRatioE4,
                                       int32_t mediaCodecType,
                                       const UiConfig& ui_config);
    uint32_t addSupportedConfiguration(
        int codecResolution, int fps, int32_t widthMargin, int32_t heightMargin,
        int32_t density, int32_t realDensity, int32_t decoderAdditionalDepth,
        int32_t pixelAspectRatioE4, int32_t mediaCodecType);

    void updateUiConfig(const UiConfig& ui_config);

   protected:
    virtual int handleSetup(int mediaCodecType);
    void sendConfig(int status) override;
    virtual int handleCodecConfig(void* config, size_t len);
    void playbackStart(int32_t sessionId);
    void playbackStop(int32_t sessionId);
    virtual int handleMediaConfiguration(int index);

    bool handleVideoFocusRequest(const VideoFocusRequestNotification& req);
    bool handleUpdateUiConfigReply(const UpdateUiConfigReply& req) override;
    unsigned int getNumberOfConfigurations();
private:
    void handleDataAvailable(uint64_t timestamp, const shared_ptr<IoBuffer>& data, size_t offset);
    void filterConfig(unsigned int index, Config* resp);

    bool mAutoStartProjection;
    shared_ptr<IVideoSinkCallbacks> mCallbacks;
    std::vector<shared_ptr<VideoConfiguration> > mVideoConfigurations;
    uint32_t mViewingDistance;
    uint32_t mDisplayId;
    DisplayType mDisplayType;
    bool mDisplayIdAndTypeSet;
    KeyCode mInitialContentKeycode = KeyCode::KEYCODE_UNKNOWN;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_VIDEO_SINK_H
