#ifndef WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_PINGSERVICE_H_
#define WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_PINGSERVICE_H_

#include <cstdint>
#include <list>

#include "IConnectionConfigurationListener.h"

#include "common.h"
#include "util/GalMutex.h"
#include "util/IntervalTimer.h"

#define DEFAULT_PING_INTERVAL_MS 1000  // HUIG Requirement R03-140 and R05-380.
#define DEFAULT_PING_TIMEOUT_MS 3000  // HUIG Requirement R03-160 and R05-400.
#define DEFAULT_TRACKED_PING_COUNT 5  // HUIG Requirement R05-410
#define DEFAULT_HIGH_LATENCY_THRESHOLD_MS 200  // HUIG Requirement R05-410

class Controller;

namespace GalService {

using GalUtil::IntervalTimer;

class IPingControllerCallbacks {
public:
    virtual ~IPingControllerCallbacks() { }
    virtual void sendPingRequest(int64_t timestamp, bool bugReport) = 0;
    virtual void unrecoverableError(MessageStatus err) = 0;
    virtual void highLatencyCallback(std::vector<int> latencies) = 0;
    virtual void latencyBackToNormalCallback(std::vector<int> latencies) = 0;
};

/**
 * A service which continiously sends PING requests to the MD, and monitors
 * that PING responses are returned within the specified timeout.
 */
class PingService : public IConnectionConfigurationListener {
public:
    PingService(IPingControllerCallbacks* controller) :
        mController(controller),
        mIntervalMs(DEFAULT_PING_INTERVAL_MS),
        mTimeoutMs(DEFAULT_PING_TIMEOUT_MS),
        mHighLatencyThresholdMs(DEFAULT_HIGH_LATENCY_THRESHOLD_MS),
        mTrackedLatencyLimit(DEFAULT_TRACKED_PING_COUNT),
        mSentHighLatencyWarning(false),
        mIntervalTimer(mIntervalMs, std::bind(&PingService::heartBeat, this)) {}

    ~PingService() override;

    /** Starts sending pings. */
    void start();

    /** Stop sending pings and tear down the PingService. */
    void shutdown();

    /**
     * Called by Controller to indicate that the MD has sent PING response.
     * Returns true if timestamp belongs to a PING by this class.
     **/
    bool pingResponseCallback(int64_t timestamp);

    /** Sets the frequency at which PINGs should be sent. */
    void setIntervalMs(int intervalMs);

    /** Sets the maximum delay that can occur before connection is torn down. */
    void setTimeoutMs(int timeoutMs);

    /** Sets the threshold at which a high latency warning will be sent. */
    void setHighLatencyThresholdMs(int thresholdMs);

    /**
     * Sets the maximum number of pings that should be tracked for high latency
     * detection.
     */
    void setTrackedPingCount(int numberOfPings);

    /** Gets the interval between PING's being sent. */
    int getIntervalMs();

    /** Gets the maximum delay that can occur before connection is torn down. */
    int getTimeoutMs();

    /** Gets the threshold value for a high latency warning. */
    int getHighLantecyThresholdMs();

    /** Gets how many pings should be evaluated for high latency. */
    int getTrackedPingCount();

    /**
     * Sets timeout/interval and other settings from a ConnectionConfiguration.
     */
    void applyConnectionConfiguration(
        const ConnectionConfiguration& config) override;

    /**
     * Gets timeout/interval and other settings from a ConnectionConfiguration.
     */
    ConnectionConfiguration getConnectionConfiguration() override;

protected:
    /**
     * heartBeat is called at intervalMs frequency. It will send a PING and
     * ensure that all existing pending PINGs have been responded to.
     *
     * Note: This method is marked protected to assist in unit testing.
     */
    void heartBeat();

private:
    /**
     * Verifies that the oldest pending PING is not older than
     * timestamp - mTimeoutMs.
     **/
    void detectPingTimeout(int64_t timestamp);

    /**
     * Verifies that the last tracked ping count pings are below
     * mHighLatencyThresholdMs. If there is a change, the method will notify
     * IControllerCallbacks.
     */
    void detectHighLatency();

    /**
     * Adds a latency to be tracked for high latency detection. Once the number
     * of tracked latencies are above mTrackedLatencyLimit, the oldest latency
     * is removed.
     */
    void addTrackedLatency(int latency);

    /** Adds a ping that was sent out to the MD. */
    void addPendingPing(int64_t timestamp);

    /** Removes a ping which received a ping response from the MD. */
    bool removePendingPing(int64_t timestamp);

    /** Returns true if there are at least mTrackedLatencyLimit latencies. */
    bool hasEnoughTrackedLatencies();

    IPingControllerCallbacks* mController;
    int mIntervalMs;
    int mTimeoutMs;
    int mHighLatencyThresholdMs;
    int mTrackedLatencyLimit;
    bool mSentHighLatencyWarning;

    IntervalTimer mIntervalTimer;
    GalMutex mLock;
    std::list<int64_t> mPendingPings;
    std::list<int> mTrackedLatencies;
};
}  // namespace GalService
#endif  // WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_PINGSERVICE_H_
