// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_SINK_BASE_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_SINK_BASE_H

#include <vector>

#include "common.h"
#include "ProtocolEndpointBase.h"

#define INVALID_SESSION_ID  (-1)

/**
 * Base class for all media sinks. This is the code that handles most of
 * the state machine required for video and audio. The video and audio
 * sink inherit from this and provide callbacks which get called when
 * messages arrive that need specific handling.
 */
class MediaSinkBase : public ProtocolEndpointBase {
public:
    MediaSinkBase(uint8_t id, MessageRouter* router) :
            ProtocolEndpointBase(id, router, false),
            mCodecType(static_cast<MediaCodecType>(0)),
            mCurrentSessionId(INVALID_SESSION_ID),
            mMaxUnackedFrames(0) { }
    virtual int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    /**
     * @DEPRECATED Procol 1.5 and earlier.
     * This message is used to send an ack for frames that have been processed.
     * This forms the core of the flow control mechanism.
     * @param sessionId The session id of the frames that were processed.
     * @param numFrames The number of frames to be acknowledged.
     * @return true if the message was transmitted, false otherwise.
     */
    bool ackFrames(int32_t sessionId, uint32_t numFrames);
    /**
     * This message is used to send an ack for frames that have been processed.
     * This forms the core of the flow control mechanism.
     * @param sessionId The session id of the frames that were processed.
     * @param receive_timestamp_ns When each frame was received.
     * @return true if the message was transmitted, false otherwise.
     */
    bool ackFrames(int32_t sessionId,
                   const std::vector<uint64_t>& receive_timestamp_ns);
    /**
     * Set the codec type of this media sink.
     * @param codecType See the MediaCodecType proto.
     */
    void setCodecType(int codecType) {
        mCodecType = (MediaCodecType) codecType;
    }

    /**
     * Call this *only* if you want to change the allowed configurations from what was
     * set during registration time. Among the allowed configurations, the one in the
     * beginning will have higher priority. This will cause all future media setup transactions
     * to have the new allowed configurations. It will not affect any already started
     * media channels. To have this take effect, you must stop and start the channel.
     * If this is not specified, configuration will be preferred by the order of addition: First
     * added will have higher priority.
     * Note that calling this is meaningless for audio as all audio configurations declared in
     * service discovery should be always supported in audio case.
     * @param configIndices Array of configuration indices.
     * @param size number of indices in configIndices.
     * @return true if the index is valid, false if not.
     */
    bool resetAllowedConfigurations(uint32_t* configIndices, unsigned int size);

protected:
    /**
     * Handles a setup request. These messages are sink specific so must be implemented by the
     * specific sink types.
     * @param mediaCodecType The codec type. See enum MediaCodecType for details.
     */
    virtual int handleSetup(int mediaCodecType) = 0;
    /**
     * Each individual codec has its own configuration, which needs to be handled by the specific
     * sinks. This callback is called when codec information is available and must be implemented.
     * @param config The raw config data.
     * @param len The length of the config data.
     * @return STATUS_SUCCESS if the data was handled successfully, an appropriate error otherwise.
     */
    virtual int handleCodecConfig(void* config, size_t len) = 0;
    /**
     * This function gets called when a new audio or video frame is available.
     * @param timestamp The timestamp of this message.
     * @param data An IoBuffer that contains the data.
     * @param offset The offset of data within the IoBuffer.
     */
    virtual void handleDataAvailable(uint64_t timestamp, const shared_ptr<IoBuffer>& data,
            size_t offset) = 0;
    /**
     * Send Config to phone
     * @param status new status.
     */
    virtual void sendConfig(int status) = 0;
    /**
     * Notifies the upper layers that audio / video playback is about to begin.
     * @param sessionId The session id associated with this audio / video stream.
     */
    virtual void playbackStart(int32_t sessionId) = 0;
    /**
     * Notifies that playback of this audio / video stream has terminated.
     * @param sessionId The session id of the stream that just terminated.
     */
    virtual void playbackStop(int32_t sessionId) = 0;
    /**
     * The subclass should implement the correct function to check whether the config
     * sent is the correct one and stub out the irrelevant one.
     */
    virtual int handleMediaConfiguration(int index) = 0;

    /**
     * Called when the remote end notifies us that they wish to acquire video focus.
     * @return false if sink does not support video focus.
     */
    virtual bool handleVideoFocusRequest(const VideoFocusRequestNotification& req) = 0;

    virtual bool handleUpdateUiConfigReply(const UpdateUiConfigReply& req) = 0;

    /** return number of configurations added to the sink */
    virtual unsigned int getNumberOfConfigurations() = 0;

private:
    int handleStart(const Start& req);
    int handleStop(const Stop& req);

protected:
    MediaCodecType mCodecType;
    int32_t mCurrentSessionId;
    /** max unacked frames, to be set by child */
    uint32_t mMaxUnackedFrames;

    std::vector<uint32_t> mConfigurationPreference;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_SINK_BASE_H
