// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_INPUT_SOURCE_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_INPUT_SOURCE_H

#include <set>

#include "common.h"
#include "ProtocolEndpointBase.h"
#include "util/GalMutex.h"
#include "IInputSourceCallbacks.h"

using std::set;

/**
 * This endpoint is used to send input events from the car to the phone.
 * A lot of the methods are self explanatory, documentation is provided for those
 * that arent.
 */
class InputSource : public ProtocolEndpointBase {
public:
    InputSource(uint8_t id, MessageRouter *router) :
            ProtocolEndpointBase(id, router, false),
            mHasTouchScreen(false),
            mTouchScreenWidth(0),
            mTouchScreenHeight(0),
            mTouchScreenType(static_cast<TouchScreenType>(0)),
            mTouchScreenSecondary(false),
            mHasTouchPad(false),
            mTouchPadWidth(0),
            mTouchPadHeight(0),
            mTouchPadUiNavigation(false),
            mTouchPadPhysWidth(0),
            mTouchPadPhysHeight(0),
            mTouchPadUiAbsolute(false),
            mTouchPadTapAsSelect(false),
            mTouchPadUiSensitivity(0),
            mHasDisplayId(false),
            mDisplayId(0) {}

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<IInputSourceCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Call this method during the initialization phase to register a touchscreen.
     * For now, the touchscreen is assumed to be related to the primary display.
     * @param width The width of the touch screen.
     * @param height The height of the touch screen.
     */
    void registerTouchScreen(int32_t width, int32_t height, TouchScreenType type) {
      registerTouchScreen(width, height, type, false);
    }

    /**
     * Call this method during the initialization phase to register a
     * touchscreen. For now, the touchscreen is assumed to be related to the
     * primary display.
     * @param width The width of the touch screen.
     * @param height The height of the touch screen.
     * @param isSecondary Set to true if the touch screen should be treated as a
     * secondary input
     */
    void registerTouchScreen(int32_t width, int32_t height,
                             TouchScreenType type, bool isSecondary) {
      mTouchScreenWidth = width;
      mTouchScreenHeight = height;
      mTouchScreenType = type;
      mTouchScreenSecondary = isSecondary;
      mHasTouchScreen = true;
    }

    /**
     * Call this method during the initialization phase to register a touchpad that can
     * be used by the user to draw simple gestures.
     * @param width The width of the touchpad.
     * @param height The height of the touchpad.
     * @param physicalWidth The width of the touchpad in mm.
     * @param physicalHeight The height of the touchpad in mm.
     */
    void registerTouchPad(int32_t width, int32_t height, int32_t physicalWidth,
                          int32_t physicalHeight) {
      registerTouchPad(width, height, false, physicalWidth, physicalHeight,
                       false, true);
    }
    /**
     * @param uiNavigation Set to true if the touchpad should be used to control
     * the UI
     * @param uiAbsolute If both uiNavigation and this flag are true, then the
     * touchpad will map to the screen in terms absolute coordinates when
     * navigating through the UI. If uiNavigation is true and this flag is
     * false, then touchpad movements will be interpreted as gestures. If
     * uiNavigation is false, this flag has no effect.
     * @param tapAsSelect Set to false if a tap on the touchpad should not be a
     * select event.
     */
    void registerTouchPad(int32_t width, int32_t height, bool uiNavigation,
                int32_t physicalWidth, int32_t physicalHeight, bool uiAbsolute,
                bool tapAsSelect) {
      registerTouchPad(width, height, uiNavigation, physicalWidth,
                       physicalHeight, uiAbsolute, tapAsSelect,
                       DEFAULT_TOUCHPAD_SENSITIVITY);
    }
    /**
     * @param uiNavigation Set to true if the touchpad should be used to control
     * the UI
     * @param uiAbsolute If both uiNavigation and this flag are true, then the
     * touchpad will map to the screen in terms absolute coordinates when
     * navigating through the UI. If uiNavigation is true and this flag is
     * false, then touchpad movements will be interpreted as gestures. If
     * uiNavigation is false, this flag has no effect.
     * @param tapAsSelect Set to false if a tap on the touchpad should not be a
     * select event.
     * @param uiSensitivity A number from 1 to 10 (inclusive) to configure the touchpad's
     * sensitivity. Larger numbers are more sensitive, requiring smaller
     * inputs to trigger movement.
     */
    void registerTouchPad(int32_t width, int32_t height, bool uiNavigation,
                int32_t physicalWidth, int32_t physicalHeight, bool uiAbsolute,
                bool tapAsSelect, int32_t uiSensitivity) {
      mTouchPadWidth = width;
      mTouchPadHeight = height;
      mTouchPadUiNavigation = uiNavigation;
      mTouchPadPhysWidth = physicalWidth;
      mTouchPadPhysHeight = physicalHeight;
      mTouchPadUiAbsolute = uiAbsolute;
      mTouchPadTapAsSelect = tapAsSelect;
      mTouchPadUiSensitivity = uiSensitivity;
      mHasTouchPad = true;
    }
    /**
     * Call this method during the initialization phase to register a set of keycodes
     * that are supported, which is used in the service discovery response.
     * @param keycodes The set of supported keycodes.
     */
    void registerKeycodes(const std::set<int32_t>& keycodes) {
      mKeycodes = keycodes;
    }
    /**
     * Call this method during the initialization phase to register a set of
     * feedback events that are supported, which is used in the service
     * discovery response.
     * @param events The set of supported feedback events.
     */
    void registerFeedbackEvents(const std::set<FeedbackEvent>& events) {
      mFeedbackEvents = events;
    }

    /**
     * Call this method during the initialization phase to set the display ID
     * associated with this input source.
     * @param displayId The display ID for this input.
     */
    void setDisplayId(uint32_t displayId) {
      mDisplayId = displayId;
      mHasDisplayId = true;
    }

    /**
     * Use this method to report a keypress or release. This is the preferred interface for
     * signaling key presses.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param keycode The keycode of the key that was pressed or released.
     * @param isDown true if it is a press, false if it is a release.
     * @param metastate Can be used to represent shift, alt etc. Currently unused.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportKey(uint64_t timestamp, uint32_t keycode, bool isDown, uint32_t metaState);
    /**
     * Use this method to signal a long press of a button. Use this interface only if you
     * have to. What this does is simulates a sequence of key down, hold, key up on the
     * phone. The only case when you should do this is if your system prevents you from
     * being able to detect the initial key down. Do not report a key up after this. Contact
     * the GAL team before you decide to use this interface.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param keycode The keycode of the key that was pressed or released.
     * @param metastate Can be used to represent shift, alt etc. Currently unused.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportKeyLongPress(uint64_t timestamp, uint32_t keycode, uint32_t metastate);
    /**
     * Use this method to report a value from a device that sends out an absoute value.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param keycode While these devices aren't keys per se, they use a keycode as a
     *        device identifier, which should be registered using registerKeycodes.
     * @param value The absolute value associated with that device. For example, if a
     *        wheel goes from 0-10, the value would be that.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportAbsolute(uint64_t timestamp, uint32_t keycode, int32_t value);
    /**
     * Use this method to report a value from a device that can output a relative value.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param keycode While these devices aren't keys per se, they use a keycode as a
     *        device identifier, which should be registered using registerKeycodes.
     * @param delta The value associated with this event. You can indicate a magnitude and
     *        direction using the value and its sign. For instance, if a jog dial were
     *        turned two steps to the left, delta would be -2. If it were turned 5 steps
     *        to the right, delta would be +5.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportRelative(uint64_t timestamp, uint32_t keycode, int32_t delta);
    /**
     * Use this method to report a touch event on the touch screen that is associcated with
     * the primary display. Coordinates reported must be translated and/or scaled as appropriate
     * such that they are within the coordinate space of the display and so if the display
     * is running a scaled resolution, the x & y values should be scaled appropriately too.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param numPointers The number of pointers being described.
     * @param pointerIds The pointer ids of the pointers.
     * @param x The x values of the pointers.
     * @param y The y values of the pointers.
     * @param action The action that took place in this event.
     * @param actionIndex The index of the pointer to which action corresponds.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportTouch(uint64_t timestamp, int32_t numPointers, const uint32_t* pointerIds,
            const uint32_t* x, const uint32_t* y, int action, int actionIndex);
    /**
     * Use this method to report a touch event on the touch pad. The coordinates must
     * be absolute and within the coordinate space of the touch pad itself.
     * @param timestamp The timestamp of the event in nanoseconds.
     * @param numPointers The number of pointers being described.
     * @param pointerIds The pointer ids of the pointers.
     * @param x The x values of the pointers.
     * @param y The y values of the pointers.
     * @param action The action that took place in this event.
     * @param actionIndex The index of the pointer to which action corresponds.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
    int reportTouchPad(uint64_t timestamp, int32_t numPointers, const uint32_t* pointerIds,
            const uint32_t* x, const uint32_t* y, int action, int actionIndex);
    /**
     * Use this method to report a change in the ui sensitivity of the touch
     * pad. To send the updated service configuration to the mobile device, the
     * GalReceiver::updateService function must be called with this InputSource
     * as its argument after a call to the
     * InputSource::updateTouchPadUiSensitivity function.
     * @param uiSensitivity A number from 1 to 10 (inclusive) to configure the
     * touchpad's sensitivity. Larger numbers are more sensitive, requiring
     * smaller inputs to trigger movement.
     * @return STATUS_SUCCESS on success, STATUS_INVALID_INPUT if there's no
     * touch pad configured or if the input is outside the valid range.
     */
    int updateTouchPadUiSensitivity(int32_t uiSensitivity);
private:
    int handleKeyBindingRequest(const KeyBindingRequest& req);
    int handleInputFeedback(const InputFeedback& feedback);
    bool isBound(int32_t keycode);
    void sendInputReport(uint64_t timestamp, InputReport* report);
    void populateTouchEvent(TouchEvent* event, uint64_t timestamp, int32_t numPointers,
            const uint32_t* pointerIds, const uint32_t* x, const uint32_t* y, int action,
            int actionIndex);
    int sendKey(uint64_t timestamp, uint32_t keycode, bool isDown,
            bool longPress, uint32_t metaState);

    bool mHasTouchScreen;
    int32_t mTouchScreenWidth;
    int32_t mTouchScreenHeight;
    TouchScreenType mTouchScreenType;
    bool mTouchScreenSecondary;

    bool mHasTouchPad;
    int32_t mTouchPadWidth;
    int32_t mTouchPadHeight;
    bool mTouchPadUiNavigation;
    int32_t mTouchPadPhysWidth;
    int32_t mTouchPadPhysHeight;
    bool mTouchPadUiAbsolute;
    bool mTouchPadTapAsSelect;
    int32_t mTouchPadUiSensitivity;

    bool mHasDisplayId;
    uint32_t mDisplayId;

    std::set<int32_t> mKeycodes;
    std::set<int32_t> mBoundKeycodes;
    std::set<FeedbackEvent> mFeedbackEvents;

    shared_ptr<IInputSourceCallbacks> mCallbacks;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_INPUT_SOURCE_H
