// Copyright 2020 Google Inc. All Rights Reserved.
#ifndef WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_UTIL_INTERVALTIMER_H_
#define WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_UTIL_INTERVALTIMER_H_

#include <condition_variable>
#include <functional>
#include <thread>

namespace GalUtil {

/**
 * A class that repeatedly calls a callback method at a set frequency.
 *
 * This class may need be modified in order to be supported on your platform.
 */
class IntervalTimer {
public:
    IntervalTimer(int interval, std::function<void()> callback) :
        mRunning(false),
        mCallback(callback),
        mIntervalMs(interval) {}

    virtual ~IntervalTimer();

    /**
     * Starts the timer.
     *
     * Note: SetInterval must be called before start(), otherwise the timer will
     * stop immediately.
     */
    void start();

    /**
     * Stops the timer.
     *
     * Note: This method may block until mCallback has completed.
     */
    void stop();

    /**
     * Sets the interval at which this timer makes a callback, defined in ms.
     *
     * Note: If SetInterval is called after start(), the following callback will
     * happen at the previous interval, and any subsequent callbacks will happen
     * at the new frequency.
     */
    void updateIntervalMs(int intervalMs);

protected:
    /** The run method is called in a separate thread. */
    void run();

    /**
     * The wait method wait for interval_ms and return true if that time period
     * has passed, or false if it was cancelled (due to the timer being
     * stopped).
     *
     * Note: This method is marked virtual to assist with unit testing.
     */
    virtual bool wait(int intervalMs);

    /** Whether the timer is currently running. */
    bool mRunning;

    /** The method to be called every interval. */
    std::function<void()> mCallback;

private:
    int mIntervalMs;
    std::thread mThread;
    std::condition_variable mWaitCondition;
    std::mutex mWaitLock;
};
};      // namespace GalUtil
#endif  // WIRELESS_ANDROID_AUTO_PROJECTION_PROTOCOL_RECEIVER_LIB_SRC_UTIL_INTERVALTIMER_H_
