// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_PLAYBACK_STATUS_ENDPOINT_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_PLAYBACK_STATUS_ENDPOINT_H

#include "common.h"
#include "ProtocolEndpointBase.h"
#include "util/GalMutex.h"
#include "IMediaPlaybackStatusCallbacks.h"

/**
 * This class implements an endpoint that receives media playback status over
 * the GAL protocol. You must also subclass IMediaPlaybackStatusCallbacks and implement all
 * the methods so that the GAL receiver can interface with your hardware. The call
 * sequence would look something like this:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      MediaPlaybackStatusEndpoint* endpoint = new MediaPlaybackStatusEndpoint(serviceId,
 *              galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IMediaPlaybackStatusCallbacks.
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 * </pre>
 */
class MediaPlaybackStatusEndpoint : public ProtocolEndpointBase {
public:
    MediaPlaybackStatusEndpoint(uint8_t id, MessageRouter *router) :
            ProtocolEndpointBase(id, router, false) { }

    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void registerCallbacks(const shared_ptr<IMediaPlaybackStatusCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Reports an action associated with media playback.
     * @param action See enum InstrumentClusterAction in protos.proto for possible values.
     * @return STATUS_SUCCESS if the call succeeded, an error code otherwise.
     */
    int reportAction(uint32_t action);

private:
    shared_ptr<IMediaPlaybackStatusCallbacks> mCallbacks;
    void handleMediaPlaybackStatus(const MediaPlaybackStatus &status);
    void handleMediaPlaybackMetadata(const MediaPlaybackMetadata &metadata);
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_MEDIA_PLAYBACK_STATUS_ENDPOINT_H
