// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_INAVIGATION_STATUS_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_INAVIGATION_STATUS_CALLBACKS_H

#include "common.h"

/**
 * Callbacks to indicate Navigation Status events.
 */
class INavigationStatusCallbacks {
public:
    virtual ~INavigationStatusCallbacks() { }

    /**
     * Called when navigation status message comes through.
     *
     * @param status navigation status (active, inactive, unavailable)
     */
    virtual void navigationStatusCallback(int status) = 0;

    /**
     * Called when a Navigation Next Turn message comes through.
     *
     * @param road name of the road
     * @param turnSide turn side (may be TurnSide values, defined in
     *        protos.pb.h: LEFT, RIGHT or UNSPECIFIED).
     * @param event event type (may be NextTurnEnum values, defined in
     *        protos.pb.h: TURN, U_TURN, ROUNDABOUT_ENTER_AND_EXIT, etc).
     * @param image image to be shown in the instrument cluster (PNG format).
     *        Will be null if instrument cluster type is ENUM (defined in
     *        protos.pb.h), or if the image parameters are malformed (length or
     *        width non-positive, or illegal imageColorDepthBits) in the initial
     *        NavigationStatusService call.
     * @param turnAngle turn angle in degrees between the roundabout entry and
     *        exit (0..359).  Only used for event type ROUNDABOUT_ENTER_AND_EXIT.
     *        -1 if unused.
     * @param turnNumber turn number, counting around from the roundabout entry
     *        to the exit.  Only used for event type ROUNDABOUT_ENTER_AND_EXIT.
     *        -1 if unused.
     *
     * @note For an example of a roundabout: if a roundabout has 4 exits,
     *        spaced evenly, then the first exit will have turnNumber=1,
     *        turnAngle=90; the second will have turnNumber=2, turnAngle=180;
     *        the third will have turnNumber=3, turnAngle=270.  turnNumber and
     *        turnAngle are counted in the direction of travel around the
     *        roundabout (clockwise for roads where the car drives on the
     *        left-hand side of the road, such as Australia; anti-clockwise for
     *        roads where the car drives on the right, such as the USA).
     */
    virtual void navigationNextTurnCallback(
            const std::string& road, int turnSide, int event,
            const std::string& image, int turnAngle, int turnNumber) = 0;

    /**
     * Called when a Navigation Next Turn Distance message comes through.
     *
     * @param distanceMeters Distance to next turn event in meters.
     * @param timeSeconds Time to next turn event in seconds.
     * @param displayDistanceE3 The distance to the next turn, rounded and converted to the
     *        units used when this distance is displayed by the MD's navigation UI. This value
     *        is then scaled by 1e3. After correcting for the scaling factor, the HU should use
     *        this value as is (e.g. no further rounding/padding, no reinterpreting units).
     * @param displayDistanceUnit The units of (displayDistanceE3 * 1e-3).
     */
    virtual void navigationNextTurnDistanceCallback(
            int distanceMeters, int timeSeconds, int displayDistanceE3,
            NavigationNextTurnDistanceEvent_DistanceUnits displayDistanceUnit) = 0;

    /**
     * Called when a NavigationState message comes through.
     *
     * @param event Navigation state data to be displayed on the instrument
     *        cluster of a car.
     */
    virtual void navigationStateCallback(const NavigationState& event) = 0;

    /**
     * Called when a NavigationCurrentPosition message comes through.
     *
     * @param event The distance and estimated time of arrival info for next
     *        steps and destinations, and current road information to describe
     *        the current position of navigation.
     */
    virtual void navigationCurrentPositionCallback(
        const NavigationCurrentPosition& event) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_INAVIGATION_STATUS_CALLBACKS_H
