// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H

#include "common.h"

struct ConnectedDeviceStruct {
    std::string deviceName;
    uint32_t deviceId;
};

/**
 * This class represents a general set of callbacks that must be set up for the GAL receiver
 * library to be able to function properly.
 */
class IControllerCallbacks {
public:
    virtual ~IControllerCallbacks() { }
    /**
     * Called when service discovery request is received. This call sends across
     * an icon set and a label that can be used by the native UI to display a
     * button that allows users to switch back to projected mode.
     * @param smallIcon 32x32 png image.
     * @param mediumIcon 64x64 png image.
     * @param largeIcon 128x128 png image.
     * @param label A label that may be displayed alongside the icon.
     * @param deviceName A friendly device name.
     * @param instanceId a unique instance Id represents a specific MD
     */
    virtual void serviceDiscoveryRequestCallback(
        const std::string& smallIcon, const std::string& mediumIcon,
        const std::string& largeIcon, const std::string& label,
        const std::string& deviceName, const std::string& instanceId) = 0;

    /**
     * Called when an unrecoverable error has been encountered. The recommended course
     * of action is to reset the usb link at this point. A re-establishment of the
     * GAL connection may be attempted after that.
     *
     * The current set of error codes is:
     *  - STATUS_AUTHENTICATION_FAILURE if the SSL handshake fails.
     *  - STATUS_FRAMING_ERROR if an error occurs during communication with the
     *    MD. Examples of errors include (but are not limited to) IO errors and
     *    SSL decryption errors.
     */
    virtual void unrecoverableErrorCallback(MessageStatus err) = 0;

    /**
     * Indicates that the MD wants the HU to capture a bugreport. The HU can
     * choose to ignore this if it is sent with high frequency - it is the HU's
     * responsibility to ensure that the system cannot be DoS'd.
     * @param timestamp The (remote) timestamp of the request.
     */
    virtual void bugreportRequestCallback(int64_t timestamp) = 0;

    /**
     * Called when a navigation focus request is received from the phone. You must respond
     * to this by calling Controller::setNavigationFocus() (even if there is no change in
     * navigation focus). If navigation focus is given to the mobile device, all native
     * turn by turn guidance systems must be stopped.
     * @param type The type requested (can be NAV_FOCUS_NATIVE or NAV_FOCUS_PROJECTED).
     */
    virtual void navigationFocusCallback(NavFocusType type) = 0;

    /**
     * Called when ByeByeRequest is received from phone. After taking necessary steps,
     * car side should send ByeByeResponse.
     * @param reason The reason for the disconnection request.
     */
    virtual void byeByeRequestCallback(ByeByeReason reason) = 0;

    /**
     * Called when ByeByeResponse is received from phone. Normally this is a reply for
     * ByeByeRequest message sent from car.
     */
    virtual void byeByeResponseCallback() = 0;

    /**
     * Called when a voice session notification is received. Note that this callback only applies
     * to you if you do not always send a PTT short press to us always. If you always send PTT
     * short press to us, you should be able to ignore this call altogether.
     * @param status The status of the voice recongition session.
     */
    virtual void voiceSessionNotificationCallback(VoiceSessionStatus status) = 0;

    /**
     * Called when the source wishes to acquire audio focus.
     * @param request Can be one of AUDIO_FOCUS_GAIN, AUDIO_FOCUS_GAIN_TRANSIENT,
     *        AUDIO_FOCUS_GAIN_TRANSIENT_MAY_DUCK, AUDIO_FOCUS_RELEASE.
     */
    virtual void audioFocusRequestCallback(AudioFocusRequestType request) = 0;

    /**
     * Called when CarConnectedDevicesRequest is received from phone. After
     * taking necessary steps, car side should send CarConnectedDevices
     * message.
     */
    virtual void carConnectedDevicesRequestCallback() = 0;

    /**
     * Called when UserSwitchRequest is received from phone. Car side should
     * initiate a new connection with the specified device.
     * The car should then send a UserSwitchResposne message to the original
     * projection phone, with the status of the connection attempt.
     * If the attempt is successful, car should also send BYEBYE_REQUEST.
     * @param selectedDevice The mobile device selected by user to initiate
     * new AAP connection.
     */
    virtual void userSwitchRequestCallback(
            const ConnectedDeviceStruct& selectedDevice) = 0;

    /**
     * Called when a battery status notification is received from mobile device.
     * @param batteryLevel The current battery percentage of the mobile device.
     * @param timeRemainingS An estimate of how long the battery will last, in seconds.
     * @param batteryCritical Whether the mobile device's battery is at a critical level,
     *        as determined by the mobile device.
     */
    virtual void batteryStatusNotificationCallback(int32_t batteryLevel,
            int32_t timeRemainingS, bool batteryCritical) = 0;

    /**
     * Called when latencies have been above a certain threshold for the last
     * time period. HUIG defines the threshold as 200ms for 1 second, but the
     * MD may change these values.
     */
    virtual void highLatencyCallback(std::vector<int> latencies) {}

    /**
     * Called when latencies have dropped below a certain threshold for the last
     * time period, after highLatencyCallback has been called. HUIG defines the
     * threshold as 200ms for 1 second, but the MD may change these values.
     */
    virtual void latencyBackToNormalCallback(std::vector<int> latencies) {}
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_ICONTROLLER_CALLBACKS_H
