// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H

#include "ChannelManager.h"
#include "ConnectionConfigurationCallbacksAdapter.h"
#include "Controller.h"
#include "IConnectionConfigurationCallbacks.h"
#include "MessageRouter.h"
#include "common.h"
#include "util/MonotonicClock.h"

/**
 * This class is the fundamental building block of the Google Automotive Link protocol.
 * <br>
 * Initializing the receiver library consists of the following steps.
 * <pre>
 *      GalReceiver galReceiver;
 *      // Create an instance of type IControllerCallbacks.
 *      galReceiver.init(controllerCallbacks);
 *      galReceiver.setIdentityInfo(...);
 *      galReceiver.setClientCreds(...);
 *      galReceiver.setDriverPosition(...);
 *      // Repeat this for every endpoint you want.
 *      // Create and start service. Look at the documentation for the following for details.
 *      // AudioSink
 *      // AudioSource
 *      // BluetoothEndpoint
 *      // InputSource
 *      // InstrumentClusterEndpoint
 *      // SensorSource
 *      // RadioEndpoint
 *      // VendorExtension
 *      // VideoSink
 *      galReceiver.registerService()
 *      galReceiver.start();
 * </pre>
 * <br>
 * Under normal operation, you would normally have two threads one that reads the underlying
 * transport and submits data to the GAL receiver and another that reads data from the GAL
 * receiver and submits data to the underlying transport.
 * <br>
 * A typical transport to GAL loop may look like the pseudocode below. Note that this example
 * assumes that you are reading from a buffered transport, it might be necessary to add your
 * own buffering layer if the underlying transport does not support arbitrary sized reads.
 * <pre>
 *      uint8_t header[FRAME_HEADER_MIN_LENGTH];
 *      while (!stop) {
 *          readFromTransport(header, sizeof(header));
 *          int len = galReceiver.getAdditionalBytesToRead(header);
 *          memcpy(buf, header, sizeof(header);
 *          readFromTransport(buf + sizeof(header), len);
 *          mGalReceiver.queueIncoming(buf, len);
 *      }
 * </pre>
 * <br>
 * A typical GAL to transport loop may look like this:
 * <pre>
 *      IoBuffer buf;
 *      while (!stop) {
 *          galReceiver.getEncodedFrame(&buf);
 *          writeToTransport(buf.raw(), buf.size());
 *      }
 * </pre>
 * <br>
 * The shutdown sequence would typically look like this. The most common reason to tear down
 * would be a disconnection event either caused by an unplug or by a callback signalling an
 * unrecoverable error.
 * <pre>
 *      galReceiver.prepareShutdown(); // This will unblock any blocked threads.
 *      // Stop transport threads.
 *      galReceiver.shutdown();
 * </pre>
 */
class GalReceiver {
public:
    GalReceiver() : mController(&mMessageRouter) { }
    /**
     * Call this method to initialize the receiver library. This must be called before any other
     * methods on the library.
     */
    bool init(const shared_ptr<IControllerCallbacks>& controllerCallbacks);
    /**
     * Call this when you want to send out the VERSION_REQUEST message to the phone. All services
     * must be registered before this.
     */
    void start();
    /**
     * Call this method to make sure there are no messages waiting in the
     * control channel queue. This is asynchronous method it will just put a
     * special marker frame into the queue such that subsequent
     * #getEncodedFrame(IoBuffer) will return false once the queue reached that
     * marker.
     */
    void flushControllerChannelQueue();
    /**
     * Call this method to unblock any threads that may be waiting on
     * getEncodedFrame().
     */
    void prepareShutdown();
    /**
     * Call this method to shutdown the receiver. It is okay to call init again once this method
     * has returned.
     */
    void shutdown();
    /**
     * This method should be called when new data is available for processing. The data is copied
     * so it may be freed once this method returns.
     * @param raw The raw data that was received.
     * @param len The size of data in bytes.
     * @return STATUS_SUCCESS if the data was processed successfully, an approriate error code
     *         otherwise.
     */
    int queueIncoming(void* raw, size_t len);
    /**
     * Acquire a buffer that can be passed on to queueIncoming.
     * @return A shared_ptr to an IoBuffer of the requested size.
     */
    shared_ptr<IoBuffer> allocateBuffer(size_t size);
    /**
     * This method should be called when there is new data available for processing. With this
     * method, you transfer ownership of the buffer to the receiver library so you should not
     * use it after calling this method.
     * @param buf The buffer containing the data.
     * @return STATUS_SUCCESS if the data was processed successfully, an approriate error code
     *         otherwise.
     */
    int queueIncoming(const shared_ptr<IoBuffer>& buf);
    /**
     * When data is available on the underlying transport, call this method with the first 4 bytes
     * to determine how many more bytes should be read to reach the end of the frame. The data
     * read along with these 4 bytes should be passed into queueIncoming subsequently.
     * @param buf A buffer with the data that belongs to the header.
     * @return How many more bytes should be read to get the complete message.
     */
    int getAdditionalBytesToRead(unsigned char buf[FRAME_HEADER_MIN_LENGTH]);
    /**
     * Call this method to obtain a frame for transmission. Note that this method blocks until a
     * frame is actually available. The IoBuffer that is passed in will be populated with the
     * frame data.
     * @param buf The io buffer that should be populated with the frame data.
     * @return true on success, false otherwise.
     */
    bool getEncodedFrame(IoBuffer* buf);
    /**
     * Call this function to register a new 'Service' with the receiver. This allows the receiver
     * to advertise it in the ServiceDiscoveryResponse and for the phone to open channels to it.
     * @param endpoint The service to register.
     * @return true on success, false otherwise.
     */
    bool registerService(ProtocolEndpointBase* endpoint);

    /**
     * Call this function to receive callbacks to configure or retrieve
     * parameters of an active TCP connection. This should only be called for
     * wireless connections.
     *
     * Note: This method can only be called once (returning true if it was set
     * successfully. Subsequent calls will return false and noop).
     */
    bool setTransportConfigurationListener(
            shared_ptr<IConnectionConfigurationCallbacks> callback);

    /**
     * Call this function to update a 'Service' that was previously registered.
     * @param endpoint The service to update.
     * @return true on success, false otherwise.
     */
    bool updateService(ProtocolEndpointBase* endpoint);
    /**
     * Call this function to set up identity information of the car. This must be called prior to
     * calling start(). Send empty strings for any fields that you cannot provide.
     * @param make The make of the car.
     * @param model Car model.
     * @param year Production year.
     * @param id A unique identifier for this car. Ideally, there should be a way for the user to
     *           have this id regenerated.
     */
    void setIdentityInfo(const std::string& make, const std::string& model,
            const std::string& year, const std::string& id) {
        mController.setIdentityInfo(make, model, year, id);
    }
    /**
     * Set up information about the head unit. Send empty strings for any fields that you cannot
     * provide.
     * @param huMake The make of the head unit.
     * @param huModel The model of the head unit.
     * @param huSwBuild The software build of the head unit.
     * @param huSwVersion The software version of the head unit.
     */
    void setHeadUnitInfo(const std::string& huMake, const std::string& huModel,
            const std::string& huSwBuild, const std::string& huSwVersion) {
        mController.setHeadUnitInfo(huMake, huModel, huSwBuild, huSwVersion);
    }
    /**
     * Sets the driver position - left/right/center. This influences the layout of the screen.
     * This must be called before start().
     * @param position See the DriverPosition protocol buffer enum for details.
     */
    void setDriverPosition(int position) {
        mController.setDriverPosition(position);
    }
    /**
     * Sets the value sent to the MD to configure its behaviour during projection.
     * This must be called before start().
     * @param sessionConfiguration the bitmask to send to the MD
     */
    void setSessionConfiguration(int sessionConfiguration) {
        mController.setSessionConfiguration(sessionConfiguration);
    }
    /**
     * Sets the string which the MD will display as-is to refer to the native software experience.
     */
    void setDisplayName(const std::string& displayName) {
        mController.setDisplayName(displayName);
    }
    /**
     * Set the credentials used in the ssl handshake. Must be called before calling start().
     * @param rootCert The root of trust to be used while validating a peer certificate.
     * @param clientCert The client certificate to be used in the handshake.
     * @param privKey The private key corresponding to the client certificate.
     * @return true on success, false otherwise.
     */
    bool setClientCreds(const std::string& rootCert,
            const std::string& clientCert, const std::string& privKey) {
        return mController.setClientCreds(rootCert, clientCert, privKey);
    }

    /* Changed by ADIT
     * CR-740: AAuto-SDC: SDC OpenSSL Engine
     */
    /**
     * Set the credentials used in the ssl handshake. Must be called before calling start().
     * Use key id instead of patch to raw file
     * @param rootCert The root of trust to be used while validating a peer certificate.
     * @param clientCert The client certificate to be used in the handshake.
     * @param privKeyId The id of private key corresponding to the client certificate.
     * @param rootKeyId The id of root public key corresponding to the root certificate.
     * @param engineName name of openssl engine to be used
     * @return true on success, false otherwise.
     */
    bool setClientCreds(const std::string& rootCert,
                        const std::string& clientCert,
                        const std::string& privKeyId,
                        const std::string& rootKeyId,
                        const std::string& engineName) {
        return mController.setClientCreds(rootCert, clientCert,
                                          privKeyId, rootKeyId,
                                          engineName);
    }

    /**
     * Request the other end to generate a bug report locally. The other end may
     * choose to ignore this request.
     */
    void sendBugReportRequest() {
      mController.sendPingRequest(
          GalUtil::MonotonicClock::GetCurrentMonoTimeMs(),
          /* bugReport = */ true);
    }
    MessageRouter* messageRouter() { return &mMessageRouter; }
    /**
     * Set the navigation focus. This is used to prevent the case where the user is offered turn
     * by turn directions from both the native navigation application on the head unit and by the
     * connected phone. When the phone requests navigation focus, this method should be used to
     * grant it, or if the user starts up the native navigation application, this method can be
     * used to acquire it. Navigation focus should only be changed as a direct consequence of a
     * user action.
     *
     * <p>Navigation focus must always be set in response to a navigation focus request
     * (IControllerCallbacks::navigationFocusCallback()) from the mobile device (even if there
     * is no change in navigation focus).
     *
     * @param type Can be NAV_FOCUS_NATIVE or NAV_FOCUS_PROJECTED.
     */
    void setNavigationFocus(int type) {
        mController.setNavigationFocus((NavFocusType) type);
    }

    /**
     * Send ByeByeRequest to phone.
     */
    void sendByeByeRequest(int code) {
        mController.sendByeByeRequest((ByeByeReason) code);
    }

    /**
     * Send ByeByeResponse as a response to ByeByeRequest.
     */
    void sendByeByeResponse() {
        mController.sendByeByeResponse();
    }

    /**
     * Send UserSwitchResponse as a response to UserSwitchRequest
     */
    void sendUserSwitchResponse(int code,
            ConnectedDeviceStruct selectedDevice) {
        mController.sendUserSwitchResponse((UserSwitchStatus) code,
                selectedDevice);
    }

    /**
     * Send CarConnectedDevices message to phone.
     * @param connectedDevices Mobile devices connected to car to which the
     *         head unit can initiate an AAP connection.
     * @param unsolicited False when responding to a CarConnectedDevicesRequest;
     *         true when unsolicited.
     * @param finalList False if the HU is still discovering nearby mobile
     *         devices and this is not the final list of connected devices.
     *         True if the HU has finished the device discovery process
     *         and this is the final list.
     */
    void sendCarConnectedDevices(
            const std::vector<ConnectedDeviceStruct>& connectedDevices,
            bool unsolicited, bool finalList) {
        mController.sendCarConnectedDevices(
                connectedDevices, unsolicited, finalList);
    }

    /**
     * Set audio focus.
     */
    void setAudioFocus(AudioFocusStateType focusState, bool unsolicited) {
        mController.setAudioFocus(focusState, unsolicited);
    }
    /**
     * Use this api to set the time that will be used to verify the validity of the certificate
     * presented by the phone. You should only use this if your system does not run wall time.
     * This should be called after init() but before start() if you wish to override the time,
     * you should be able to call it any number of times after without any effect.
     * @param The number of seconds elapsed since Jan 1 1970 UTC 00:00:00.
     */
    void setCertificateVerificationTime(time_t time) {
        mController.setCertificateVerificationTime(time);
    }

    /**
     * Sends call availability status to MD.
     * @param callAvailable true if phone call via HFP becomes available; false
     * if phone call via HFP becomes unavailable due to ongoing call on another
     * device/SIM.
     */
    void setCallAvailability(bool callAvailable) {
        mController.setCallAvailability(callAvailable);
    }

    /**
     * Sets a flag indicating that this AAP session is only to probe whether or not the device
     * supports AAP. The MD will not display any UI, and will indicate whether or not AAP is
     * supported by responding with a byebye request.
     */
    void setProbeOnlySession() {
        mController.setProbeOnlySession();
    }

private:
    MessageRouter mMessageRouter;
    ChannelManager mChannelManager;
    Controller mController;
    shared_ptr<ConnectionConfigurationCallbacksAdapter>
        mConnectionConfigurationAdapter;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_GAL_RECEIVER_H
