// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H

#include "common.h"
#include "IControllerCallbacks.h"
#include "PingService.h"
#include "ProtocolEndpointBase.h"

/**
 * @internal
 * This class manages the internal GAL state machine, authentication and common messaging. You
 * should not need to directly interact with this class.
 */
class Controller : public ProtocolEndpointBase,
        GalService::IPingControllerCallbacks {
public:
    Controller(MessageRouter* router) :
        ProtocolEndpointBase(CONTROLLER_SERVICE_ID, router, false),
        mHealthy(false),
        mDriverPosition(DRIVER_POSITION_UNKNOWN),
        mSessionConfiguration(0),
        mPingService(this),
        mProbeOnly(false) {
        mRouter->registerConnectionConfigurationListener(&mPingService);
    }

    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void start();
    void shutdown() {
        mPingService.shutdown();
        mSslWrapper.shutdown();
    }
    /**
     * @internal
     * Sets information that describes this car. Should be called before calling
     * start because this information is exchanged in the service discovery phase.
     * Send empty strings for any fields that you cannot provide.
     * @param make The car make.
     * @param model The car model.
     * @param year Car model year.
     * @param id Typically an Id that that can identify this car but can also be
     *        reset by the user. By definition, this cannot be the vin, it should
     *        be a separately generated uuid for instance which can be reset on
     *        user request.
     */
    void setIdentityInfo(const std::string& make, const std::string& model,
            const std::string& year, const std::string& id) {
        mMake = make;
        mModel = model;
        mYear = year;
        mId = id;
    }
    /**
     * @internal
     * Set up information about the head unit. Send empty strings for any fields that you cannot provide.
     * @param huMake The make of the head unit.
     * @param huModel The model of the head unit.
     * @param huSwBuild The software build of the head unit.
     * @param huSwVersion The software version of the head unit.
     */
    void setHeadUnitInfo(const std::string& huMake, const std::string& huModel,
            const std::string& huSwBuild, const std::string& huSwVersion) {
        mHuMake = huMake;
        mHuModel = huModel;
        mHuSwBuild = huSwBuild;
        mHuSwVersion = huSwVersion;
    }
    /**
     * @internal
     * Lets the phone know which side the driver sits on.
     * @param position See the DriverPosition proto.
     */
    void setDriverPosition(int position) { mDriverPosition = position; }
    /**
     * @internal
     * Sets the session configuration bitmask
     * @param sessionConfiguration See SessionConfiguration proto
     */
    void setSessionConfiguration(int sessionConfiguration) {
        mSessionConfiguration = sessionConfiguration;
    }
    /**
     * @internal
     * Sets the display name of the native software.
     * @param displayName See SessionConfiguration proto
     */
    void setDisplayName(const std::string& displayName) {
        mDisplayName = displayName;
    }
    void registerCallbacks(
            const shared_ptr<IControllerCallbacks>& controllerCallbacks) {
        mControllerCallbacks = controllerCallbacks;
    }
    void unrecoverableError(MessageStatus err) override {
        mControllerCallbacks->unrecoverableErrorCallback(err);
    }
    /**
     * Ping the other side to check to see if it is alive.
     * @param timestamp The local timestamp. This is repeated by the other side
     *        in the response so it can be used to do interesting things like
     *        computing round trip delays.
     * @param bugReport Should the other side save away a bug report locally. The
     *        other side may choose to ignore this request if it is deemed to be
     *        too high frequency.
     */
    void sendPingRequest(int64_t timestamp, bool bugReport) override;

    bool setClientCreds(const std::string& rootCert,
            const std::string& clientCert, const std::string& privKey);

    /* Changed by ADIT
     * CR-740: AAuto-SDC: SDC OpenSSL Engine
     */
    /**
     * Set the credentials used in the ssl handshake. Must be called before calling start().
     * Use key id instead of patch to raw file
     * @param rootCert The root of trust to be used while validating a peer certificate.
     * @param clientCert The client certificate to be used in the handshake.
     * @param privKeyId The id of private key corresponding to the client certificate.
     * @param rootKeyId The id of root public key corresponding to the root certificate.
     * @param engineName name of openssl engine to be used
     * @return true on success, false otherwise.
     */
    bool setClientCreds(const std::string& rootCert,
                        const std::string& clientCert,
                        const std::string& privKeyId,
                        const std::string& rootKeyId,
                        const std::string& engineName
                        );

    void setNavigationFocus(NavFocusType type);
    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr) { };
    /**
     * Send ByeByeRequest to the other side. After sending this, ByeByeResponse will be coming,
     * and that is the end of the current connection.
     * @param reason reason for finishing.
     */
    void sendByeByeRequest(ByeByeReason reason);
    /**
     * Send ByeByeResponse as a response to ByeByeRequest. This should be sent only after getting
     * ByeByeRequest, which will be notified via byeByeRequestCallback(..).
     */
    void sendByeByeResponse();
    /**
     * Send UserSwitchResponse as a response to UserSwitchRequest. This should be sent only after
     * getting UserSwitchRequest, which will be notified by UserSwitchRequestCallback(..).
     */
    void sendUserSwitchResponse(UserSwitchStatus status,
            ConnectedDeviceStruct selectedDevice);
    /**
     * Set audio focus.
     * @param focusState Can be one of AUDIO_FOCUS_STATE_GAIN, AUDIO_FOCUS_STATE_LOSS,
     *        AUDIO_FOCUS_STATE_LOSS_TRANSIENT.
     * @param unsolicited whether this is a response to AudioFocusRequestNotification (false) or
     *        sink's notification (true).
     */
    void setAudioFocus(AudioFocusStateType focusState, bool unsolicited);
    void setCertificateVerificationTime(time_t time) {
        mSslWrapper.setCertificateVerificationTime(time);
    }
    /**
     * Send CarConnectedDevices message. This could be an unsolicited message
     * in response to UserSwitchRequest.
     */
    void sendCarConnectedDevices(
            const std::vector<ConnectedDeviceStruct> &connectedDevices,
            bool unsolicited, bool finalList);
    /**
     * Set whether phone call via HFP is available.
     * @param call_available Whether phone call via HFP is available.
     */
     void setCallAvailability(bool callAvailable);

     /**
      * Send ServiceDiscoveryUpdate message to update the configuration of a
      * Service. This could be an unsolicited message in response to a settings
      * change.
      */
     void sendServiceDiscoveryUpdate(ProtocolEndpointBase* endpoint);

     void setProbeOnlySession() { mProbeOnly = true; }

     /**
      * Sends a high latency callback to mControllerCallbacks.
      *
      * @param latencies A list of latencies on which this callback is based on.
      */
     void highLatencyCallback(std::vector<int> latencies) override {
         mControllerCallbacks->highLatencyCallback(latencies);
     }

     /**
      * Sends a restored latency callback to mControllerCallbacks.
      *
      * @param latencies A list of latencies on which this callback is based on.
      */
     void latencyBackToNormalCallback(std::vector<int> latencies) override {
         mControllerCallbacks->latencyBackToNormalCallback(latencies);
     }

    private:
     int sendVersionRequest();
     int handleVersionResponse(void* msg, size_t len);
     void handleVersionResponseOptions(const VersionResponseOptions& resp);
     int startAuthentication();
     int handleServiceDiscoveryRequest(const ServiceDiscoveryRequest& req);
     int sendServiceDiscoveryResponse();
     int handlePingRequest(const PingRequest& req);
     int handlePingResponse(const PingResponse& resp);
     void sendAuthResponse(int status);
     void handleUserSwitchRequest(const UserSwitchRequest& req);

     bool mHealthy;
     std::string mMake;
     std::string mModel;
     std::string mYear;
     std::string mId;
     std::string mHuMake;
     std::string mHuModel;
     std::string mHuSwBuild;
     std::string mHuSwVersion;
     int mDriverPosition;
     int mSessionConfiguration;
     std::string mDisplayName;
     shared_ptr<IControllerCallbacks> mControllerCallbacks;
     GalService::PingService mPingService;
     SslWrapper mSslWrapper;
     bool mProbeOnly;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H
