/*
 * fc_dumm_MosquittoCpp_Wrapper.cpp
 *
 *  Created on: Dec 22, 2017
 *      Author: gjd5kor
 */

#include <cstdlib>
#include "mqtt_MosquittoCpp_Wrapper.h"
#include "mosquitto_plugin.h"
#include <mosquitto.h>
#include <stdint.h>

enum mosquitto_msg_direction {
	mosq_md_in = 0,
	mosq_md_out = 1
};

enum mosquitto_msg_state {
	mosq_ms_invalid = 0,
	mosq_ms_publish_qos0 = 1,
	mosq_ms_publish_qos1 = 2,
	mosq_ms_wait_for_puback = 3,
	mosq_ms_publish_qos2 = 4,
	mosq_ms_wait_for_pubrec = 5,
	mosq_ms_resend_pubrel = 6,
	mosq_ms_wait_for_pubrel = 7,
	mosq_ms_resend_pubcomp = 8,
	mosq_ms_wait_for_pubcomp = 9,
	mosq_ms_send_pubrec = 10,
	mosq_ms_queued = 11
};

enum mosquitto_client_state {
	mosq_cs_new = 0,
	mosq_cs_connected = 1,
	mosq_cs_disconnecting = 2,
	mosq_cs_connect_async = 3,
	mosq_cs_connect_pending = 4,
	mosq_cs_connect_srv = 5,
	mosq_cs_disconnect_ws = 6,
	mosq_cs_disconnected = 7,
	mosq_cs_socks5_new = 8,
	mosq_cs_socks5_start = 9,
	mosq_cs_socks5_request = 10,
	mosq_cs_socks5_reply = 11,
	mosq_cs_socks5_auth_ok = 12,
	mosq_cs_socks5_userpass_reply = 13,
	mosq_cs_socks5_send_userpass = 14,
	mosq_cs_expiring = 15,
};

enum _mosquitto_protocol {
	mosq_p_invalid = 0,
	mosq_p_mqtt31 = 1,
	mosq_p_mqtt311 = 2,
	mosq_p_mqtts = 3
};

enum mosquitto__threaded_state {
	mosq_ts_none,		/* No threads in use */
	mosq_ts_self,		/* Threads started by libmosquitto */
	mosq_ts_external	/* Threads started by external code */
};

enum _mosquitto_transport {
	mosq_t_invalid = 0,
	mosq_t_tcp = 1,
	mosq_t_ws = 2,
	mosq_t_sctp = 3
};

struct _mosquitto_packet{
	uint8_t *payload;
	struct _mosquitto_packet *next;
	uint32_t remaining_mult;
	uint32_t remaining_length;
	uint32_t packet_length;
	uint32_t to_process;
	uint32_t pos;
	uint16_t mid;
	uint8_t command;
	int8_t remaining_count;
};

struct mosquitto_message_all{
	struct mosquitto_message_all *next;
	time_t timestamp;
	//enum mosquitto_msg_direction direction;
	enum mosquitto_msg_state state;
	bool dup;
	struct mosquitto_message msg;
};

typedef int mosq_sock_t;

struct mosquitto {
	mosq_sock_t sock;
#ifndef WITH_BROKER
	mosq_sock_t sockpairR, sockpairW;
#endif
#if defined(__GLIBC__) && defined(WITH_ADNS)
	struct gaicb *adns; /* For getaddrinfo_a */
#endif
	enum _mosquitto_protocol protocol;
	char *address;
	char *id;
	char *username;
	char *password;
	uint16_t keepalive;
	uint16_t last_mid;
	enum mosquitto_client_state state;
	time_t last_msg_in;
	time_t next_msg_out;
	time_t ping_t;
	struct _mosquitto_packet in_packet;
	struct _mosquitto_packet *current_out_packet;
	struct _mosquitto_packet *out_packet;
	struct mosquitto_message *will;
#ifdef WITH_TLS
	SSL *ssl;
	SSL_CTX *ssl_ctx;
	char *tls_cafile;
	char *tls_capath;
	char *tls_certfile;
	char *tls_keyfile;
	int (*tls_pw_callback)(char *buf, int size, int rwflag, void *userdata);
	char *tls_version;
	char *tls_ciphers;
	char *tls_psk;
	char *tls_psk_identity;
	int tls_cert_reqs;
	bool tls_insecure;
#endif
	bool want_write;
	bool want_connect;
#if defined(WITH_THREADING) && !defined(WITH_BROKER)
	pthread_mutex_t callback_mutex;
	pthread_mutex_t log_callback_mutex;
	pthread_mutex_t msgtime_mutex;
	pthread_mutex_t out_packet_mutex;
	pthread_mutex_t current_out_packet_mutex;
	pthread_mutex_t state_mutex;
	pthread_mutex_t in_message_mutex;
	pthread_mutex_t out_message_mutex;
	pthread_mutex_t mid_mutex;
	pthread_t thread_id;
#endif
	bool clean_session;
#ifdef WITH_BROKER
	bool is_dropping;
	bool is_bridge;
	struct _mqtt3_bridge *bridge;
	struct mosquitto_client_msg *msgs;
	struct mosquitto_client_msg *last_msg;
	int msg_count;
	int msg_count12;
	struct _mosquitto_acl_user *acl_list;
	struct _mqtt3_listener *listener;
	time_t disconnect_t;
	struct _mosquitto_packet *out_packet_last;
	struct _mosquitto_subhier **subs;
	int sub_count;
	int pollfd_index;
#  ifdef WITH_WEBSOCKETS
#    if defined(LWS_LIBRARY_VERSION_NUMBER)
	struct lws *wsi;
#    else
	struct libwebsocket_context *ws_context;
	struct libwebsocket *wsi;
#    endif
#  endif
	bool ws_want_write;
#else
#  ifdef WITH_SOCKS
	char *socks5_host;
	int socks5_port;
	char *socks5_username;
	char *socks5_password;
#  endif
	void *userdata;
	bool in_callback;
	unsigned int message_retry;
	time_t last_retry_check;
	struct mosquitto_message_all *in_messages;
	struct mosquitto_message_all *in_messages_last;
	struct mosquitto_message_all *out_messages;
	struct mosquitto_message_all *out_messages_last;
	void (*on_connect)(struct mosquitto *, void *userdata, int rc);
	void (*on_disconnect)(struct mosquitto *, void *userdata, int rc);
	void (*on_publish)(struct mosquitto *, void *userdata, int mid);
	void (*on_message)(struct mosquitto *, void *userdata, const struct mosquitto_message *message);
	void (*on_subscribe)(struct mosquitto *, void *userdata, int mid, int qos_count, const int *granted_qos);
	void (*on_unsubscribe)(struct mosquitto *, void *userdata, int mid);
	void (*on_log)(struct mosquitto *, void *userdata, int level, const char *str);
	//void (*on_error)();
	char *host;
	int port;
	int in_queue_len;
	int out_queue_len;
	char *bind_address;
	unsigned int reconnect_delay;
	unsigned int reconnect_delay_max;
	bool reconnect_exponential_backoff;
	char threaded;
	struct _mosquitto_packet *out_packet_last;
	int inflight_messages;
	int max_inflight_messages;
#  ifdef WITH_SRV
	ares_channel achan;
#  endif
#endif

#ifdef WITH_BROKER
	UT_hash_handle hh_id;
	UT_hash_handle hh_sock;
	struct mosquitto *for_free_next;
#endif
};


static void on_connect_wrapper(struct mosquitto *mosq, void *userdata, int rc)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	bool sessionPresent = mosq->in_packet.payload[0];
	m->on_connect_with_flags(rc,sessionPresent);
	m->on_connect(rc);
}

static void on_disconnect_wrapper(struct mosquitto *mosq, void *userdata, int rc)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_disconnect(rc);
}

static void on_publish_wrapper(struct mosquitto *mosq, void *userdata, int mid)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_publish(mid);

}

static void on_message_wrapper(struct mosquitto *mosq, void *userdata, const struct mosquitto_message *message)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_message(message);
}

static void on_subscribe_wrapper(struct mosquitto *mosq, void *userdata, int mid, int qos_count, const int *granted_qos)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_subscribe(mid, qos_count, granted_qos);
}

static void on_unsubscribe_wrapper(struct mosquitto *mosq, void *userdata, int mid)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_unsubscribe(mid);
}


static void on_log_wrapper(struct mosquitto *mosq, void *userdata, int level, const char *str)
{
	(void)mosq;
	class mqtt_MosquittoCppWrapper *m = (class mqtt_MosquittoCppWrapper *)userdata;
	m->on_log(level, str);
}

int lib_version(int *major, int *minor, int *revision)
{
	if(major) *major = LIBMOSQUITTO_MAJOR;
	if(minor) *minor = LIBMOSQUITTO_MINOR;
	if(revision) *revision = LIBMOSQUITTO_REVISION;
	return LIBMOSQUITTO_VERSION_NUMBER;
}

int lib_init()
{
	return mosquitto_lib_init();
}

int lib_cleanup()
{
	return mosquitto_lib_cleanup();
}

const char* connack_string(int connack_code)
{
	return mosquitto_connack_string(connack_code);
}

int sub_topic_tokenise(const char *subtopic, char ***topics, int *count)
{
	return mosquitto_sub_topic_tokenise(subtopic, topics, count);
}

int sub_topic_tokens_free(char ***topics, int count)
{
	return mosquitto_sub_topic_tokens_free(topics, count);
}

int topic_matches_sub(const char *sub, const char *topic, bool *result)
{
	return mosquitto_topic_matches_sub(sub, topic, result);
}

mqtt_MosquittoCppWrapper::mqtt_MosquittoCppWrapper(const char *id, bool clean_session)
{

	m_mosq = mosquitto_new(id, clean_session, this);
	mosquitto_connect_callback_set(m_mosq, on_connect_wrapper);
	mosquitto_disconnect_callback_set(m_mosq, on_disconnect_wrapper);
	mosquitto_publish_callback_set(m_mosq, on_publish_wrapper);
	mosquitto_message_callback_set(m_mosq, on_message_wrapper);
	mosquitto_subscribe_callback_set(m_mosq, on_subscribe_wrapper);
	mosquitto_unsubscribe_callback_set(m_mosq, on_unsubscribe_wrapper);
	mosquitto_log_callback_set(m_mosq, on_log_wrapper);
}

mqtt_MosquittoCppWrapper::~mqtt_MosquittoCppWrapper()
{
	mosquitto_destroy(m_mosq);
}

int mqtt_MosquittoCppWrapper::reinitialise(const char *id, bool clean_session)
{
	int rc;
	rc = mosquitto_reinitialise(m_mosq, id, clean_session, this);

	if(rc == MOSQ_ERR_SUCCESS)
	{
		mosquitto_connect_callback_set(m_mosq, on_connect_wrapper);
		mosquitto_disconnect_callback_set(m_mosq, on_disconnect_wrapper);
		mosquitto_publish_callback_set(m_mosq, on_publish_wrapper);
		mosquitto_message_callback_set(m_mosq, on_message_wrapper);
		mosquitto_subscribe_callback_set(m_mosq, on_subscribe_wrapper);
		mosquitto_unsubscribe_callback_set(m_mosq, on_unsubscribe_wrapper);
		mosquitto_log_callback_set(m_mosq, on_log_wrapper);
	}
	return rc;
}

int mqtt_MosquittoCppWrapper::connect(const char *host, int port, int keepalive)
{
	return mosquitto_connect(m_mosq, host, port, keepalive);
}

int mqtt_MosquittoCppWrapper::connect(const char *host, int port, int keepalive, const char *bind_address)
{
	return mosquitto_connect_bind(m_mosq, host, port, keepalive, bind_address);
}

int mqtt_MosquittoCppWrapper::connect_async(const char *host, int port, int keepalive)
{
	return mosquitto_connect_async(m_mosq, host, port, keepalive);
}

int mqtt_MosquittoCppWrapper::connect_async(const char *host, int port, int keepalive, const char *bind_address)
{
	return mosquitto_connect_bind_async(m_mosq, host, port, keepalive, bind_address);
}

int mqtt_MosquittoCppWrapper::reconnect()
{
	return mosquitto_reconnect(m_mosq);
}

int mqtt_MosquittoCppWrapper::reconnect_async()
{
	return mosquitto_reconnect_async(m_mosq);
}

int mqtt_MosquittoCppWrapper::disconnect()
{
	return mosquitto_disconnect(m_mosq);
}

int mqtt_MosquittoCppWrapper::socket()
{
	return mosquitto_socket(m_mosq);
}

int mqtt_MosquittoCppWrapper::will_set(const char *topic, int payloadlen, const void *payload, int qos, bool retain)
{
	return mosquitto_will_set(m_mosq, topic, payloadlen, payload, qos, retain);
}

int mqtt_MosquittoCppWrapper::will_clear()
{
	return mosquitto_will_clear(m_mosq);
}

int mqtt_MosquittoCppWrapper::username_pw_set(const char *username, const char *password)
{
	return mosquitto_username_pw_set(m_mosq, username, password);
}

int mqtt_MosquittoCppWrapper::publish(int *mid, const char *topic, int payloadlen, const void *payload, int qos, bool retain)
{
	return mosquitto_publish(m_mosq, mid, topic, payloadlen, payload, qos, retain);
}

int mqtt_MosquittoCppWrapper::reconnect_delay_set(unsigned int reconnect_delay, unsigned int reconnect_delay_max, bool reconnect_exponential_backoff)
{
	return mosquitto_reconnect_delay_set(m_mosq, reconnect_delay, reconnect_delay_max, reconnect_exponential_backoff);
}

int mqtt_MosquittoCppWrapper::max_inflight_messages_set(unsigned int max_inflight_messages)
{
	return mosquitto_max_inflight_messages_set(m_mosq, max_inflight_messages);
}

void mqtt_MosquittoCppWrapper::message_retry_set(unsigned int message_retry)
{
	mosquitto_message_retry_set(m_mosq, message_retry);
}

int mqtt_MosquittoCppWrapper::subscribe(int *mid, const char *sub, int qos)
{
	return mosquitto_subscribe(m_mosq, mid, sub, qos);
}

int mqtt_MosquittoCppWrapper::unsubscribe(int *mid, const char *sub)
{
	return mosquitto_unsubscribe(m_mosq, mid, sub);
}

int mqtt_MosquittoCppWrapper::loop(int timeout, int max_packets)
{
	return mosquitto_loop(m_mosq, timeout, max_packets);
}

int mqtt_MosquittoCppWrapper::loop_misc()
{
	return mosquitto_loop_misc(m_mosq);
}

int mqtt_MosquittoCppWrapper::loop_read(int max_packets)
{
	return mosquitto_loop_read(m_mosq, max_packets);
}

int mqtt_MosquittoCppWrapper::loop_write(int max_packets)
{
	return mosquitto_loop_write(m_mosq, max_packets);
}

int mqtt_MosquittoCppWrapper::loop_forever(int timeout, int max_packets)
{
	return mosquitto_loop_forever(m_mosq, timeout, max_packets);
}

int mqtt_MosquittoCppWrapper::loop_start()
{
	return mosquitto_loop_start(m_mosq);
}

int mqtt_MosquittoCppWrapper::loop_stop(bool force)
{
	return mosquitto_loop_stop(m_mosq, force);
}

bool mqtt_MosquittoCppWrapper::want_write()
{
	return mosquitto_want_write(m_mosq);
}

int mqtt_MosquittoCppWrapper::opts_set(enum mosq_opt_t option, void *value)
{
	return mosquitto_opts_set(m_mosq, option, value);
}

int mqtt_MosquittoCppWrapper::threaded_set(bool threaded)
{
	return mosquitto_threaded_set(m_mosq, threaded);
}

void mqtt_MosquittoCppWrapper::user_data_set(void *userdata)
{
	mosquitto_user_data_set(m_mosq, userdata);
}

int mqtt_MosquittoCppWrapper::socks5_set(const char *host, int port, const char *username, const char *password)
{
	return mosquitto_socks5_set(m_mosq, host, port, username, password);
}


int mqtt_MosquittoCppWrapper::tls_set(const char *cafile, const char *capath, const char *certfile, const char *keyfile, int (*pw_callback)(char *buf, int size, int rwflag, void *userdata))
{
	return mosquitto_tls_set(m_mosq, cafile, capath, certfile, keyfile, pw_callback);
}

int mqtt_MosquittoCppWrapper::tls_opts_set(int cert_reqs, const char *tls_version, const char *ciphers)
{
	return mosquitto_tls_opts_set(m_mosq, cert_reqs, tls_version, ciphers);
}

int mqtt_MosquittoCppWrapper::tls_insecure_set(bool value)
{
	return mosquitto_tls_insecure_set(m_mosq, value);
}

int mqtt_MosquittoCppWrapper::tls_psk_set(const char *psk, const char *identity, const char *ciphers)
{
	return mosquitto_tls_psk_set(m_mosq, psk, identity, ciphers);
}
