#pragma once

#include <litehtml/Config.h>

namespace WeakRefInternal {

    static size_t s_ObjectAliveCount = 0;

    class ObjectAlive {
        public:
            ObjectAlive();
            ~ObjectAlive();

            bool IsAlive() const;

            void Destroyed();

            REF_CS& GetCriticalSection();

        private:
            bool m_alive;

            DefineShareable()
    };

    inline
    WeakRefInternal::ObjectAlive::ObjectAlive() :
        m_alive(true)
    {
        ++s_ObjectAliveCount;
    }

    inline
    WeakRefInternal::ObjectAlive::~ObjectAlive()
    {
        --s_ObjectAliveCount;
    }

    inline
    bool WeakRefInternal::ObjectAlive::IsAlive() const
    {
        return m_alive;
    }

    inline
    void WeakRefInternal::ObjectAlive::Destroyed()
    {
        m_alive = false;
    }

    inline
    REF_CS& WeakRefInternal::ObjectAlive::GetCriticalSection()
    {
        return m_criticalSection;
    }

    class ObjectAliveOwner {
        public:
            ObjectAliveOwner();

            REF_CS* GetCriticalSection() const;

            const Ref<ObjectAlive>& Alive() const;

        private:
            Ref<ObjectAlive> m_alive;
    };

    inline
    WeakRefInternal::ObjectAliveOwner::ObjectAliveOwner() :
        m_alive(LITEHTML_NEW(ObjectAlive))
    {

    }

    inline
    REF_CS* WeakRefInternal::ObjectAliveOwner::GetCriticalSection() const
    {
        return m_alive.IsValid() ? &Alive()->GetCriticalSection() : 0;
    }

    inline
    const Ref<WeakRefInternal::ObjectAlive>& WeakRefInternal::ObjectAliveOwner::Alive() const
    {
        return m_alive;
    }
}

template<typename T>
class WeakRef {
    public:
        WeakRef() :
            m_weakRef(0)
        {
        }

        WeakRef(const Ref<T>& r) :
            m_alive((0 != r.m_ref) ? r.m_ref->m_alive.Alive() : Ref<WeakRefInternal::ObjectAlive>()),
            m_weakRef(r.m_ref)
        {
        }

        Ref<T> ToRef() const
        {
            if (m_alive.IsValid()) {
                REF_CS_SCOPE scope(&m_alive->GetCriticalSection());
                if (m_alive->IsAlive()) {
                    return Ref<T>(m_weakRef);
                }
            }
            return Ref<T>();
        }

        bool expired() const
        {
            if (m_alive.IsValid()) {
                REF_CS_SCOPE scope(&m_alive->GetCriticalSection());
                if (m_alive->IsAlive()) {
                    return false;
                }
            }
            return true;
        }

        Ref<T> lock() const
        {
            return ToRef();
        }

    private:
        Ref<WeakRefInternal::ObjectAlive> m_alive;
        T* m_weakRef;
};

template<typename T> class Ref;
template<typename T> class WeakRef;

#define DefineWeakRefShareable() \
    template<typename T> friend class ::Ref; \
    template<typename T> friend class ::WeakRef; \
    \
    void Obtain() \
    { \
        REF_CS_SCOPE scope(m_alive.GetCriticalSection()); \
        ++m_refCounter; \
    } \
    \
    void Release() \
    { \
        Ref<WeakRefInternal::ObjectAlive> alive = m_alive.Alive(); \
        REF_CS_SCOPE scope(m_alive.GetCriticalSection()); \
        if (m_refCounter > 0) { \
            if (--m_refCounter == 0) { \
                if (m_alive.Alive().IsValid()) { \
                    m_alive.Alive()->Destroyed(); \
                } \
                LITEHTML_DELETE(this); \
            } \
        } \
    } \
    \
    const Ref<WeakRefInternal::ObjectAlive>& Alive() const \
    { \
        REF_CS_SCOPE scope(m_alive.GetCriticalSection()); \
        return m_alive.Alive(); \
    } \
    \
    WeakRefInternal::ObjectAliveOwner m_alive; \
    RefInternal::RefCounter m_refCounter;
