/*
 * Copyright (C) 2008-2013 J.Rios <anonbeat@gmail.com>
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 *     Jörn Magens <shuerhaaken@googlemail.com>
 *     Matias De lellis <mati86dl@gmail.com>
 *     Pavel Vasin <rat4vier@gmail.com>
 */


#ifndef TAGINFO_H
#define TAGINFO_H

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <locale.h>
#include <gst/gst.h>
#include <gst/pbutils/pbutils.h>
#include <gst/gsttaglist.h>

#include <tag.h>
#include <fileref.h>
#include <asffile.h>
#include <mp4file.h>
#include <oggfile.h>
#include <xiphcomment.h>
#include <aifffile.h>
#include <wavfile.h>
#include <speexfile.h>

#include <apetag.h>
#include <apefile.h>
#include <id3v2tag.h>
#include <id3v1tag.h>

using namespace TagLib;
using namespace std;



namespace TagInfo {


    enum MediaFileType {
        MEDIA_FILE_TYPE_UNKNOWN,
        MEDIA_FILE_TYPE_AAC,
        MEDIA_FILE_TYPE_AIF,
        MEDIA_FILE_TYPE_APE,
        MEDIA_FILE_TYPE_ASF,
        MEDIA_FILE_TYPE_FLAC,
        MEDIA_FILE_TYPE_M4A,
        MEDIA_FILE_TYPE_M4B,
        MEDIA_FILE_TYPE_M4P,
        MEDIA_FILE_TYPE_MP3,
        MEDIA_FILE_TYPE_MP4,
        MEDIA_FILE_TYPE_MPC,
        MEDIA_FILE_TYPE_OGA,
        MEDIA_FILE_TYPE_OGG,
        MEDIA_FILE_TYPE_TTA,
        MEDIA_FILE_TYPE_WAV,
        MEDIA_FILE_TYPE_WMA,
        MEDIA_FILE_TYPE_WV,
        MEDIA_FILE_TYPE_SPEEX,
        MEDIA_FILE_TYPE_WMV,
        MEDIA_FILE_TYPE_VIDEO,
        MEDIA_FILE_TYPE_IMAGE

    };

    enum ImageType {
        IMAGE_TYPE_UNKNOWN,
        IMAGE_TYPE_JPEG,
        IMAGE_TYPE_PNG,
        IMAGE_TYPE_GIF,
        IMAGE_TYPE_BMP
    };

    enum MediaType
    {
        MEDIA_TYPE_UNKNOWN = 0,
        MEDIA_TYPE_MUSIC_FILE,
        MEDIA_TYPE_PODCAST,
        MEDIA_TYPE_AUDIOBOOK,
        MEDIA_TYPE_VIDEO,
        MEDIA_TYPE_PLAYLIST,
        MEDIA_TYPE_AUDIO_STREAM
    };

    class Info {
    protected :
        TagLib::FileRef * taglib_fileref;
        TagLib::File *    taglib_file;
        TagLib::Tag *     taglib_tag;

        String file_name;

        //Tags
        String title;
        String genre;
        String artist;
        String album_artist;
        String album;
        String composer;
        String comments;
        int    tracknumber;
        int    year;
        bool   is_compilation;
        int disk_number;

        //Audioproperties
        int length_seconds;
        int bitrate;
        int samplerate;
        int channels;

        //Extras
        int playcount;
        int rating;
        StringList track_labels;
        String track_labels_string;
        StringList artist_labels;
        String artist_labels_string;
        StringList album_labels;
        String album_labels_string;

        long long int changedflag;
        bool has_image;
        bool valid;
        bool allCoverTypeSupported;

        //For yomimetadata
        String yomititle; //stores yomi title
        String yomiartist; //store yomi artist
        String yomialbum; //store yomi album
        int changedyomidataflag;

        String encoding_time; //stores encoding time of file

        void set_file_name(const String &filename);
        void read_virtual_tags(TagLib::Tag * tag);
        void write_virtual_tags(TagLib::Tag * tag);
        virtual bool create_file_ref();

    public:

        //General stuff
        String get_file_name(void);

        bool is_valid();

        //Tag access functions
        String get_title(void) const;
        void   set_title(String new_title);

        String get_genre(void) const;
        void   set_genre(String new_genre);

        String get_artist(void) const;
        void   set_artist(String new_artist);

        String get_album_artist(void) const;
        void   set_album_artist(String new_artist);

        String get_album(void) const;
        void   set_album(String new_album);

        String get_composer(void) const;
        void   set_composer(String new_composer);

        String get_comments(void) const;
        void   set_comments(String new_comments);

        int  get_tracknumber() const;
        void set_tracknumber(int new_tracknumber);

        int  get_year() const;
        void set_year(int new_year);

        //Extra
        int  get_playcount() const;
        void set_playcount(int new_playcount);

        /*rating 1 - 5 ;  0 -> not set*/
        int  get_rating() const;
        void set_rating(int new_rating);

        int  get_disk_number() const;
        void set_disk_number(int number);

        bool get_has_image() const;



        //Audio properties access functions
        int get_length_seconds() const;

        int get_bitrate() const;

        int get_channels() const;
        int get_samplerate() const;

        //Labels
        StringList get_track_labels_list() const;
        void set_track_labels_list(const StringList &new_track_labels_list);
        StringList get_artist_labels_list() const;
        void set_artist_labels_list(const StringList &new_artist_labels_list);
        StringList get_album_labels_list() const;
        void set_album_labels_list(const StringList &new_album_labels_list);


        //Misc
        bool get_is_compilation() const;
        void set_is_compilation(bool compilation);


        Info(const String &filename = "", bool create_own_fileref = false);
        virtual ~Info();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);

        virtual String get_videocodec(void) const;
        virtual void   set_videocodec(String new_title);

        virtual String get_audiocodec(void) const;
        virtual void   set_audiocodec(String new_title);

        virtual String get_containerformat(void) const;
        virtual void   set_containerformat(String new_title);

        virtual String get_languagecode(void) const;
        virtual void   set_languagecode(String new_languagecode);

        virtual String get_copyrights(void) const;
        void   set_copyrights(String new_copyrights);

        virtual String get_encoder(void) const;
        virtual void   set_encoder(String new_encoder);

        virtual String get_profile(void) const;
        virtual float get_level() const;

        virtual int get_width() const;
        virtual int get_height() const;
        virtual int get_mpegversion() const;
        virtual unsigned long long int get_duration() const;
        virtual int get_episode() const;
        virtual int get_framerate() const;
        virtual unsigned int get_genrenumber() const;
        virtual bool get_encrypted(void) const;
        virtual String get_image_date_time(void) const;
        virtual String get_image_date_time_Modified() const;
        virtual int get_image_x_dimension() const;
        virtual int get_image_y_dimension() const;
        virtual int get_image_x_resolution() const;
        virtual int get_image_y_resolution() const;
        virtual long int get_image_size() const;
        virtual MediaType get_media_type() const;
        virtual int read(const string path, unsigned int seekPercent, const unsigned int time){}
        virtual int GetThumbnailBuf(unsigned char **imagedata, unsigned int &imagesize,unsigned int &hight, unsigned int &width){}
        /*!
         * Creates and returns a info pointer based on the format .This pointer
         * should be deleted by the user after use.
         *
         * If useGst is set , gtreamer is used as backend to read the metadata
         */
        static Info * create_tag_info(const String &file,bool useGst=false);
        static Info * create_tag_info_for_cover_art(const String &file,bool allCoverType = false,bool useGst=false);

        static Info * create_tag_info_with_format(const String &file, MediaFileType format);
        static Info * create_tag_info_from_mime(const String &filename, const String &mime_type);
        static Info * create_tag_info_for_video_thumbnail();

        //public functions  for yomidata
        void   set_yomititle(const String& yomititle);
        String get_yomititle() const ;
        void   set_yomiartist(const String& yomiartist);
        String get_yomiartist() const ;
        void   set_yomialbum(const String& yomialbum);
        String get_yomialbum() const ;

        //public functions for encoding time
        void set_encodingtime(const String& encodingtime);
        String get_encodingtime() const;
    };

    class Mp3Info : public Info {
    protected :
        ID3v2::Tag * taglib_tagId3v2;
        ID3v1::Tag * taglib_tagId3v1;
        unsigned int genre_number;
        virtual bool create_file_ref();

    public :
        Mp3Info(const String &filename = "");
        ~Mp3Info();

        virtual bool read(void);
        virtual bool write();

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
        virtual unsigned int get_genrenumber(void) const;
        virtual bool set_genrenumber(const unsigned int &genrenumber);
    };


    class FlacInfo : public Info {
    protected :
        Ogg::XiphComment * xiphcomment;
        virtual bool create_file_ref();

    public :
        FlacInfo(const String &filename = "");
        ~FlacInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class OggInfo : public Info {
    protected :
        Ogg::XiphComment * xiphcomment;
        virtual bool create_file_ref();

    public :
        OggInfo(const String &filename = "");
        ~OggInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class SpeexInfo : public Info {
    protected :
        Ogg::XiphComment * xiphcomment;
        virtual bool create_file_ref();

    public :
        SpeexInfo(const String &filename = "");
        ~SpeexInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class Mp4Info : public Info {
    protected :
        TagLib::MP4::Tag * mp4_tag;
        virtual bool create_file_ref();

    public :
        Mp4Info(const String &filename = "");
        ~Mp4Info();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);

        virtual bool get_encrypted(void) const;
    };


    class ApeInfo : public Info {
    protected:
        TagLib::APE::Tag * taglib_apetag;
        virtual bool create_file_ref();

    public:
        ApeInfo(const String &filename = "");
        ~ApeInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);
    };


    class MpcInfo : public Info {
    protected :
        TagLib::APE::Tag * taglib_apetag;
        virtual bool create_file_ref();

    public :
        MpcInfo(const String &filename = "");
        ~MpcInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);
    };


    class WavPackInfo : public Info {
    protected :
        TagLib::APE::Tag * taglib_apetag;
        virtual bool create_file_ref();

    public :
        WavPackInfo(const String &filename = "");
        ~WavPackInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class TrueAudioInfo : public Info {
    protected :
        ID3v2::Tag * taglib_tagId3v2;
        virtual bool create_file_ref();

    public :
        TrueAudioInfo(const String &filename = "");
        ~TrueAudioInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class AiffInfo : public Info {
    protected :
        ID3v2::Tag * taglib_tagId3v2;
        virtual bool create_file_ref();

    public :
        AiffInfo(const String &filename = "");
        ~AiffInfo();

        virtual bool read(void);
        virtual bool write();

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };


    class ASFInfo : public Info {
    protected :
        ASF::Tag * asf_tag;
        virtual bool create_file_ref();

    public :
        ASFInfo(const String &filename = "");
        ~ASFInfo();

        virtual bool read(void);
        virtual bool write(void);

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);

        virtual bool get_encrypted(void) const;
    };


    class WaveInfo : public Info {
    protected :
        ID3v2::Tag * taglib_tagId3v2;
        virtual bool create_file_ref();

    public :
        WaveInfo(const String &filename = "");
        ~WaveInfo();

        virtual bool read(void);
        virtual bool write();

        virtual bool can_handle_images(void) const;
        virtual bool get_image(char*& data, int &data_length, ImageType &image_type) const;
        virtual bool set_image(const char* data, int data_length, ImageType image_type);

        virtual bool can_handle_lyrics(void) const;
        virtual String get_lyrics(void) const;
        virtual bool set_lyrics(const String &lyrics);
    };

    /**
     * The MessageLoopData structure to hold GStreamer specific details
     */
    struct MessageLoopData
    {
        /** Width of the Video */
        uint uWidth;

        /** Height of the Video */
        uint uHeight;

         /** framerate of the file */
        uint uFramerate;

        /** Bitrate of the media file */
        uint uBitrate;

        /** Profile of the media file */
        uint uProfile;

        /** profile data of the media file */
        String  pProfile;

        /**level of the media file */
        float pLevel;

        /** Mpeg version of the media */
        gint uMpegVersion;

   };

    /**
     * The PipelineElements structure to hold GStreamer elements in the pipeline for extraction of media information
     */
    struct PipelineElements
    {
        /** Main pipeline element holder for GStreamer pipeline */
        GstElement *pPipeline;

        /** Source element holder for GStreamer pipeline */
        GstElement *pSource;

        /** Demuxer element holder for GStreamer pipeline */
        GstElement *pDemuxer;

        GstElement *pAudioConvert;
        GstElement *pVideoConvert;

        /** Flag for knowing Audio presence in the Media file */
        gboolean bAudioPresent;

        /** Flag for knowing Video presence in the Media file */
        gboolean bVideoPresent;

    };

    /**
     * The CVideoExtractor class is created by CUSBExtractor for extracting metadata and codec information
     * by GStreamer calls
     */
    class GstInfo: public Info
    {

    public:
        MessageLoopData pipelineData;
        PipelineElements pipelineElts;
    private:
        /*Video tag properties */
        String audiocodec;
        String videocodec;
        unsigned long long  int duration;
        int episode;
        String copyright;
        String encoder;
        String languagecode;
        String container_format;
        bool ishttpsrc;
        // Track number, file name, title, artist, genre, composer, album are used from the existing values from Info
        /*Video properties End () */

    public:
        /**
         *
         * CVideoExtractor constructor
         *
         */
        GstInfo(const String &filename = "");

        /**
         *
         * CVideoExtractor destructor
         *
         */
        ~GstInfo();

        bool read(void);

        /**
         *
         * This function initialize Gstreamer locale
         *
         * @return: MEDIAPLAYER_SUCCESS if succeed
         */
        int Init();

        /**
         *
         * This function extract metadata for audio or video file
         *
         * @return: Error code.
         *          MEDIAPLAYER_SUCCESS: If No Error occurs,
         *          MEDIAPLAYER_ERROR_UNKNOWN_ERROR: Unknown error,
         *          MEDIAPLAYER_ERROR_NULL_POINTER: Null pointer error.
         */
        bool ExtractMetaData();
        void retrieve_stream_info (GstDiscovererStreamInfo * info);
        void read_stream_info (GstDiscovererStreamInfo * info);

        /**
         *
         * This function fetch video codec data to find it's profie
         *
         * @param: GstStructrue pointer
         * @param: pointer to MessageLoopData
         * @return: true: if succeeded to fetch profile data
         *          false: if not succeeded to fetch profile data
         */

        static bool FetchHighProfileDataForVideoStream(GstStructure *f_pCapsStructure, MessageLoopData *f_pMsgLoopData);
        virtual String get_videocodec(void) const;
        virtual String get_audiocodec(void) const;
        virtual String get_containerformat(void) const;
        virtual String get_languagecode(void) const;
        virtual String get_copyrights(void) const;
        virtual String get_encoder(void) const;
        virtual int get_width() const;
        virtual int get_height() const;
        virtual int get_mpegversion() const;
        virtual unsigned long long int get_duration() const;
        virtual int get_episode() const;
        virtual String get_profile(void) const;
        virtual float get_level() const;
        virtual int get_framerate() const;
        virtual MediaType get_media_type()const;

#ifndef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
        virtual void set_duration(guint64 length);
#else
    virtual void set_duration(unsigned long long  int fileduration);
#endif

       // virtual void set_duration(unsigned long long  int fileduration);
        virtual void set_episode(int episode_number);
        virtual void set_bitrate(int bit_rate);
        virtual void set_videocodec(String new_title);
        virtual void set_audiocodec(String new_title);
        virtual void set_containerformat(String new_title);
        virtual void set_languagecode(String new_languagecode);
        virtual void set_copyrights(String new_copyrights);
        virtual void set_encoder(String new_encoder);
        virtual void set_width(int video_width);
        virtual void set_height(int video_height);
        virtual void set_mpegversion(int mpeg_version);
        virtual void set_profile(String profile);
        virtual void set_level(float level);
        virtual void set_framerate(int frame_rate);


        /**
         *
         * This function creates the GStreamer pipeline for media information retrieval
         *
         * @return: Success value
         *
         */
        int CreateFileSrcPipeline();

        /**
         *
         * This function creates the GStreamer pipeline for media information retrieval
         *
         * @return: Success value
         *
         */
        int CreateHttpSrcPipeline();

        /**
         *
         * This function populate metadarecord based on media type.
         *
         * @param: f_rMetaDataRecord reference to metadatarecord
         * @param: f_pTags pointer to GstTagList to extract tag information
         * @param: f_eMediaType media file type
         *
         * @return: Success value
         *
         */
        int FillTag(GstTagList *f_pTags, MediaType f_eMediaType);

        /**
         *
         * This function extract unsigned int64 value by given tag name
         *
         * @param: f_pTagList pointer to GstTagList to retrieve metadata
         * @param: f_cTagName string containing tag name to extract
         * @param: f_uiTagValue unsigned int64 value reference to store extracted tag value
         *
         */
        void GetUInt64TagValue(GstTagList *f_pTagList, const gchar *f_cTagName, guint64& f_uiTagValue);

        /**
         *
         * This function extract unsigned int value by given tag name
         *
         * @param: f_pTagList pointer to GstTagList to retrieve metadata
         * @param: f_cTagName string containing tag name to extract
         * @param: f_uiTagValue unsigned int64 value reference to store extracted tag value
         *
         */
        void GetUIntTagValue(GstTagList *f_pTagList, const gchar *f_cTagName, unsigned int& f_uiTagValue);

        /**
         *
         * This function extract string value by given tag name
         *
         * @param: f_pTagList pointer to GstTagList to retrieve metadata
         * @param: f_cTagName string containing tag name to extract
         * @param: f_rTagValue string value reference to store extracted tag value
         *
         */
        void GetStringTagValue(GstTagList *f_pTagList, const gchar *f_cTagName, String& f_rTagValue);

        /**
         *
         * This function fill video tags in videorecord container
         *
         * @param: f_rMetaDataRecord reference to videometadata record container
         * @param: f_pTags pointer to GstTagList to retrieve metadata
         *
         */
        void PopulateVideoTags(GstTagList *f_pTags);

        /**
         *
         * This function fill audio tags in videorecord container
         *
         * @param: f_rMetaDataRecord reference to videometadata record container
         * @param: f_pTags pointer to GstTagList to retrieve metadata
         *
         */
        void PopulateAudioTags(GstTagList *f_pTags);

        /**
         *
         * This function guess file type whether the same one is audio or video
         *
         * @param: f_cFilePath filepath to check the correct type
         * @param: f_pTagList pointer to GstTagList to retrieve codec information
         * @param: f_eMediaType reference to filetype variable to set
         *
         */
        void GuessFileType(String f_cFilePath, GstTagList *f_pTagList, MediaType& f_eMediaType);

        /**
         *
         * This function checks if file is playable based on Limitations of codec
         *
         * @return: True if playable
         *
         */
        bool CheckFilePlayable();

    };
    /**
     * The VideoThumbnail class is created by VideoThumbnail for extracting Video buffure
     * by GStreamer calls
     */
    class VideoThumbnail : public Info
    {
        typedef enum {
            THUMBNAIL_OK = 0,
            THUMBNAIL_TIMEOUT,
            THUMBNAIL_ERROR
        }tErrorInfo;
        typedef enum {
            THUMBNAIL_JPEG = 26UL,    /**< JPEG image format */
            THUMBNAIL_BMP = 28UL,        /**< BMP image format */
            THUMBNAIL_PNG = 29UL        /**< PNG image format */
        }tImageFormat;

        /*Ctrl Loop*/
        typedef enum{
            INIT_LOOP = 0,
            START_LOOP,
            STOP_LOOP
        }eCtrlMainLoop;
        /* GstPlayFlags flags from playbin2 */
        typedef enum {
            GST_PLAY_FLAG_VIDEO         = (1 << 0),
            GST_PLAY_FLAG_AUDIO         = (1 << 1),
            GST_PLAY_FLAG_TEXT          = (1 << 2),
            GST_PLAY_FLAG_VIS           = (1 << 3),
            GST_PLAY_FLAG_SOFT_VOLUME   = (1 << 4),
            GST_PLAY_FLAG_NATIVE_AUDIO  = (1 << 5),
            GST_PLAY_FLAG_NATIVE_VIDEO  = (1 << 6),
            GST_PLAY_FLAG_DOWNLOAD      = (1 << 7),
            GST_PLAY_FLAG_BUFFERING     = (1 << 8),
            GST_PLAY_FLAG_DEINTERLACE   = (1 << 9)
        } GstPlayFlags;
    public :
        /**
         *
         * VideoThumbnail constructor
         *
         */
        VideoThumbnail();
        /**
         *
         * VideoThumbnail Destroctor
         *
         */
        ~VideoThumbnail();

        /*internal functions*/
        /**
         *
         * Controle main loop for Gst
         *
         * @return: THUMBNAIL_OK if succeed
         */
        void CtrlMainLoop(eCtrlMainLoop type);
        int read(const string path, unsigned int seekPercent, const unsigned int time);
        tErrorInfo error;
        void DeallocateGst();
        int GetThumbnailBuf(unsigned char **imagedata, unsigned int &imagesize,unsigned int &hight, unsigned int &width);

        static gboolean cb_bus_call(GstBus* /*bus*/, GstMessage *msg, gpointer that);
        static gboolean cb_playtime(gpointer *that);
    private:
        GstElement* SetupPlay();
        int SetGstFilename(const string config, GstElement* play);
        GstBuffer* GetPlaybinPad(GstElement *element);
        int SetPlaystate(GstElement* play, GstState state);
        int SetBusError(GstElement* play);
        int StartSeek(GstElement* play,gint64    _time);
        unsigned int GetDuration(GstElement *element);

        /*variables */
        GMainLoop *mLoop;
        GstElement *mPlay;
        GstBuffer *mBuf;
        gint outwidth;
        gint outheight;
    };
    /**
     *
     * Creates sink for pipleline.
     *
     * @param: f_pDemuxerElement Demuxer element
     * @param: f_pPad source pad added
     * @param: f_pData user_data if any
     *
     */
    void OnPadAddedFileSrc(GstElement *f_pDemuxerElement, GstPad *f_pPad, gpointer f_pData);

    void OnPadAddedHttpSrc(GstElement *f_pSrcElement, GstPad *f_pPad, gpointer f_pData);

    /**
     *
     * Adds corresponding demuxer to pipleline.
     *
     * @param: f_pTypeFind Typefind element
     * @param: f_uiProbability probability of the element
     * @param: f_pCaps gstcaps
     * @param: f_pUserData user_data if any
     *
     */
    void AddDeMuxer(GstElement *f_pTypeFind, guint f_uiProbability, GstCaps *f_pCaps, gpointer f_pUserData);

}
#endif
