/*
 * Copyright (C) 2008-2013 J.Rios <anonbeat@gmail.com>
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 *     Jörn Magens <shuerhaaken@googlemail.com>
 *     Matias De lellis <mati86dl@gmail.com>
 *     Pavel Vasin <rat4vier@gmail.com>
 */


#include <algorithm>
#include "taginfo.h"
#include "taginfo_internal.h"



using namespace TagInfo;


Info * Info::create_tag_info_with_format(const String &filename, MediaFileType format) {
    Info * info = NULL;
    switch(format) {
        case  MEDIA_FILE_TYPE_UNKNOWN :
            info = new GstInfo(filename);
            break; // TODO
        case  MEDIA_FILE_TYPE_MP3 :
            info = new Mp3Info(filename);
            break;
        case  MEDIA_FILE_TYPE_FLAC :
            info = new FlacInfo(filename);
            break;
        case  MEDIA_FILE_TYPE_OGG :
        case  MEDIA_FILE_TYPE_OGA :
            info = new OggInfo(filename);
            break;
        case  MEDIA_FILE_TYPE_MP4 :
        case  MEDIA_FILE_TYPE_M4A :
        case  MEDIA_FILE_TYPE_M4B :
        case  MEDIA_FILE_TYPE_M4P :
        case  MEDIA_FILE_TYPE_AAC :
            info = new Mp4Info(filename);
            break;
        case  MEDIA_FILE_TYPE_WMA :
        case  MEDIA_FILE_TYPE_WMV :
        case  MEDIA_FILE_TYPE_ASF :
            info = new ASFInfo(filename);
            break;
        case MEDIA_FILE_TYPE_APE :
            info = new ApeInfo(filename);
            break;
        case MEDIA_FILE_TYPE_WAV :
            info = new WaveInfo(filename);
            break;
        case MEDIA_FILE_TYPE_AIF :
            info = new AiffInfo(filename);
            break;
        case MEDIA_FILE_TYPE_WV :
            info = new WavPackInfo(filename);
            break;
        case MEDIA_FILE_TYPE_TTA :
            info = new TrueAudioInfo(filename);
            break;
        case MEDIA_FILE_TYPE_MPC :
            info = new MpcInfo(filename);
            break;
        case MEDIA_FILE_TYPE_SPEEX :
            info = new SpeexInfo(filename);
            break;
        case MEDIA_FILE_TYPE_VIDEO :
            info = new GstInfo(filename);
            break;
        default :
            break;
    }
    if(!info)
    {
        std::cout<<"Null info for file :" << filename <<std::endl;
        return NULL;
    }
    else if(!info->is_valid()) {
        std::cout << "NOT valid OR unrecognized: " << filename <<std::endl;
        delete info;
        info=NULL;

        if(MEDIA_FILE_TYPE_AAC == format)
            return create_tag_info_with_format(filename,MEDIA_FILE_TYPE_MP3);
    }
    return info;
}

Info * Info::create_tag_info(const String &filename,bool useGst) {
    MediaFileType format = MEDIA_FILE_TYPE_UNKNOWN;
    String fnex = filename.substr(filename.rfind(".") + 1);
    fnex = fnex.upper();

    if(useGst)
    {
        return create_tag_info_with_format(filename,MEDIA_FILE_TYPE_UNKNOWN);
    }
    else
    {
        if(fnex == "MP3")
            format = MEDIA_FILE_TYPE_MP3;
        else if(fnex == "MP4" || fnex == "3GPP" || fnex == "FLV" || fnex == "DIVX" || fnex == "3GP" || fnex == "WMV"
                || fnex == "AVI" || fnex == "M4V" || fnex == "MPG" || fnex == "MPEG" || fnex == "MOV")
            format = MEDIA_FILE_TYPE_VIDEO;
        else if(fnex == "OGG")
            format = MEDIA_FILE_TYPE_OGG;
        else if(fnex == "FLAC")
            format = MEDIA_FILE_TYPE_FLAC;
        else if(fnex == "WMA")
            format = MEDIA_FILE_TYPE_WMA;
        else if(fnex == "OGA")
            format = MEDIA_FILE_TYPE_OGA;
        else if(fnex == "MP4")
            format = MEDIA_FILE_TYPE_MP4;
        else if(fnex == "M4A")
            format = MEDIA_FILE_TYPE_M4A;
        else if(fnex == "M4B")
            format = MEDIA_FILE_TYPE_M4B;
        else if(fnex == "M4P")
            format = MEDIA_FILE_TYPE_M4P;
        else if(fnex == "AAC")
            format = MEDIA_FILE_TYPE_AAC;
        else if(fnex == "ASF")
            format = MEDIA_FILE_TYPE_ASF;
        else if(fnex == "APE")
            format = MEDIA_FILE_TYPE_APE;
        else if(fnex == "WAV")
            format = MEDIA_FILE_TYPE_WAV;
        else if(fnex == "AIF")
            format = MEDIA_FILE_TYPE_AIF;
        else if(fnex == "WV")
            format = MEDIA_FILE_TYPE_WV;
        else if(fnex == "TTA")
            format = MEDIA_FILE_TYPE_TTA;
        else if(fnex == "MPC")
            format = MEDIA_FILE_TYPE_MPC;
        else if(fnex == "SPX")
            format = MEDIA_FILE_TYPE_SPEEX;
        else if(fnex == "WMV")
            format = MEDIA_FILE_TYPE_WMV;
        else if(fnex == "JPEG" || fnex == "JPG" || fnex == "PNG" || fnex == "GIF" || fnex == "TIF" || fnex == "BMP" )
            format = MEDIA_FILE_TYPE_IMAGE;

        return Info::create_tag_info_with_format(filename, format);
    }



}

Info * Info::create_tag_info_for_cover_art(const String &filename,bool allCoverType,bool useGst) {
    MediaFileType format = MEDIA_FILE_TYPE_UNKNOWN;
    String fnex = filename.substr(filename.rfind(".") + 1);
    fnex = fnex.upper();

    if(useGst)
    {
        return create_tag_info_with_format(filename,MEDIA_FILE_TYPE_UNKNOWN);
    }
    else
    {
        if(fnex == "MP3")
            format = MEDIA_FILE_TYPE_MP3;
        else if(fnex == "OGG")
            format = MEDIA_FILE_TYPE_OGG;
        else if(fnex == "FLAC")
            format = MEDIA_FILE_TYPE_FLAC;
        else if(fnex == "WMA")
            format = MEDIA_FILE_TYPE_WMA;
        else if(fnex == "OGA")
            format = MEDIA_FILE_TYPE_OGA;
        else if(fnex == "MP4")
            format = MEDIA_FILE_TYPE_MP4;
        else if(fnex == "M4A")
            format = MEDIA_FILE_TYPE_M4A;
        else if(fnex == "M4B")
            format = MEDIA_FILE_TYPE_M4B;
        else if(fnex == "M4P")
            format = MEDIA_FILE_TYPE_M4P;
        else if(fnex == "AAC")
            format = MEDIA_FILE_TYPE_AAC;
        else if(fnex == "ASF")
            format = MEDIA_FILE_TYPE_ASF;
        else if(fnex == "APE")
            format = MEDIA_FILE_TYPE_APE;
        else if(fnex == "WAV")
            format = MEDIA_FILE_TYPE_WAV;
        else if(fnex == "AIF")
            format = MEDIA_FILE_TYPE_AIF;
        else if(fnex == "WV")
            format = MEDIA_FILE_TYPE_WV;
        else if(fnex == "TTA")
            format = MEDIA_FILE_TYPE_TTA;
        else if(fnex == "MPC")
            format = MEDIA_FILE_TYPE_MPC;
        else if(fnex == "SPX")
            format = MEDIA_FILE_TYPE_SPEEX;
        else if(fnex == "WMV")
            format = MEDIA_FILE_TYPE_WMV;
        else if(fnex == "JPEG" || fnex == "JPG" || fnex == "PNG" || fnex == "GIF" || fnex == "TIF" || fnex == "BMP" )
            format = MEDIA_FILE_TYPE_IMAGE;
        else if(fnex == "3GPP" || fnex == "FLV" || fnex == "DIVX" || fnex == "3GP" || fnex == "AVI" || fnex == "M4V"
                || fnex == "MPG" || fnex == "MPEG" || fnex == "MOV")
            format = MEDIA_FILE_TYPE_VIDEO;

        //return Info::create_tag_info_with_format(filename, format);
        Info * retInfo = Info::create_tag_info_with_format(filename, format);
        if(retInfo && allCoverType)
                   retInfo->allCoverTypeSupported = true;
        return retInfo;
    }
}


Info * Info::create_tag_info_from_mime(const String &filename, const String &mime_type) {
    MediaFileType format = MEDIA_FILE_TYPE_UNKNOWN;
    String mime = mime_type;

    if(mime == "audio/mpeg" || mime == "audio/x-mpegurl")
        format = MEDIA_FILE_TYPE_MP3;
    else if(mime == "audio/x-vorbis+ogg" || mime == "audio/ogg")
        format = MEDIA_FILE_TYPE_OGG;
    else if(mime == "audio/flac" || mime == "audio/x-flac+ogg" || mime == "audio/x-flac")
        format = MEDIA_FILE_TYPE_FLAC;
    else if(mime == "audio/x-ms-wma")
        format = MEDIA_FILE_TYPE_WMA;
    else if(mime == "video/x-ms-wmv" || mime == "video/x-msvideo")
        format = MEDIA_FILE_TYPE_WMV;
    //    else if(mime == "OGA")
    //        format = MEDIA_FILE_TYPE_OGA;
    else if(mime == "audio/mp4" || mime == "video/mp4")
        format = MEDIA_FILE_TYPE_MP4;
    else if(mime == "audio/mp4a-latm")
        format = MEDIA_FILE_TYPE_M4A;
    //    else if(mime == "M4B")
    //        format = MEDIA_FILE_TYPE_M4B;
    //    else if(mime == "M4P")
    //        format = MEDIA_FILE_TYPE_M4P;
    else if(mime == "audio/aac" || mime == "audio/x-aac")
        format = MEDIA_FILE_TYPE_AAC;
    else if(mime == "audio/x-ms-asf" || mime == "video/x-ms-asf")
        format = MEDIA_FILE_TYPE_ASF;
    else if(mime == "audio/ape")
        format = MEDIA_FILE_TYPE_APE;
    else if(mime == "audio/x-wav")
        format = MEDIA_FILE_TYPE_WAV;
    else if(mime == "audio/x-aiff")
        format = MEDIA_FILE_TYPE_AIF;
    else if(mime == "application/x-wavpack" || mime == "audio/wavpack" || mime == "audio/x-wavpack")
        format = MEDIA_FILE_TYPE_WV;
    //    else if(mime == "TTA")
    //        format = MEDIA_FILE_TYPE_TTA;
    else if(mime == "audio/mpc" || mime == "audio/x-musepack")
        format = MEDIA_FILE_TYPE_MPC;
    else if(mime == "audio/x-spx" || mime == "audio/x-speex" || mime == "audio/x-speex+ogg")
        format = MEDIA_FILE_TYPE_SPEEX;
    return Info::create_tag_info_with_format(filename, format);
}

Info * Info::create_tag_info_for_video_thumbnail()
{
    Info * info = NULL;
    info = new VideoThumbnail();

    if(!info)
    {
        std::cout<<"Null info for file :"<<std::endl;
        return NULL;
    }
    return info;
}
// Info

Info::Info(const String &filename, bool create_own_fileref) {

    taglib_fileref = NULL;
    taglib_file    = NULL;
    taglib_tag     = NULL;

    set_file_name(filename);

    tracknumber = 0;
    year = 0;
    length_seconds = 0;
    bitrate = 0;
    rating = -1;
    playcount = 0;
    is_compilation = false;
    has_image = false;
    disk_number = 1;
    encoding_time = "1970-01-01 00:00:00"; //encoding_time is set to epoch time(default) to avoid setting/getting empty values in TDEN frame as accessing empty value(ivalid memory) from file might lead to segmentation fault

    valid = true; //TODO

    changedflag = CHANGED_DATA_NONE;
    changedyomidataflag = CHANGED_YOMIDATA_NONE;

    yomititle = "UNKNOWN";
    yomiartist = "UNKNOWN";
    yomialbum = "UNKNOWN";

    allCoverTypeSupported = false;

    if(create_own_fileref) {
        if(!file_name.isEmpty() && !create_file_ref())
            printf("Error creating file ref! %s\n", filename.toCString(false));
        if(taglib_fileref && !taglib_fileref->isNull()) {
            taglib_file = taglib_fileref->file();
            if(!taglib_file) {
                printf("Cant get file object from '%s'\n", file_name.toCString(false));
                delete taglib_fileref;
                taglib_fileref = NULL;
                taglib_file    = NULL;
                taglib_tag     = NULL;
                valid = false;
            }
            taglib_tag = taglib_file->tag();
            valid = true;
            if(!taglib_tag || taglib_tag) {
                printf("Cant get tag object from '%s'\n", file_name.toCString(false));
                delete taglib_fileref;
                taglib_fileref = NULL;
                taglib_file    = NULL;
                taglib_tag     = NULL;
                valid = false;
            }
        }
        else {
            delete taglib_fileref;
            taglib_fileref = NULL;
            taglib_file    = NULL;
            taglib_tag     = NULL;
            valid = false;
        }
    }
}

Info::~Info() {
    if(taglib_fileref)
        delete taglib_fileref;
    if(taglib_file)
        delete taglib_file;
}

bool Info::is_valid() {
    return valid;
}

void Info::set_file_name(const String &filename) {
    file_name = filename;
}


bool Info::create_file_ref() {
    if(!file_name.isEmpty())
        taglib_fileref = new TagLib::FileRef(file_name.toCString(false),
                true, TagLib::AudioProperties::Fast);
    else
        return false;
    return true;
}


void Info::read_virtual_tags(TagLib::Tag * tag) {
    if(tag) {
        title  = tag->title();
        artist      = tag->artist();
        album       = tag->album();
        genre       = tag->genre();
        comments    = tag->comment();
        tracknumber = tag->track();
        year        = tag->year();
    }
}

bool Info::read(void) {
    AudioProperties * properties;
    if(taglib_tag) {
        read_virtual_tags(taglib_tag);
    }

    if(taglib_file &&
            (properties = taglib_file->audioProperties())) {
        length_seconds = properties->length();
        bitrate        = properties->bitrate();
        samplerate     = properties->sampleRate();
        channels       = properties->channels();
        return true;
    }
    else if(taglib_fileref &&
            (properties = taglib_fileref->audioProperties())) {
        length_seconds = properties->length();
        bitrate        = properties->bitrate();
        samplerate     = properties->sampleRate();
        channels       = properties->channels();
        return true;
    }
    //printf("Problem with Info::read for %s\n", file_name.toCString(false));
    return false;
}

void Info::write_virtual_tags(TagLib::Tag * tag) {
    if(changedflag & CHANGED_TITLE_TAG)
        tag->setTitle(title);
    if(changedflag & CHANGED_ARTIST_TAG)
        tag->setArtist(artist);
    if(changedflag & CHANGED_ALBUM_TAG)
        tag->setAlbum(album);
    if(changedflag & CHANGED_GENRE_TAG)
        tag->setGenre(genre);
    if(changedflag & CHANGED_COMMENT_TAG)
        tag->setComment(comments);
    if(changedflag & CHANGED_TRACKNO_TAG)
        tag->setTrack(tracknumber); // set the id3v1 track
    if(changedflag & CHANGED_YEAR_TAG)
        tag->setYear(year);
}

bool Info::write() {
    if(taglib_tag && changedflag)
        write_virtual_tags(taglib_tag);

    if(!taglib_file->save())
        return false;

    return true;
}

void Info::set_title(String new_title) {
    title = new_title;
    changedflag |= CHANGED_TITLE_TAG;
}

String Info::get_title() const {
    if(String::null != title)
        return title;
    else
        return String();
}

void Info::set_genre(String new_genre) {
    genre = new_genre;
    changedflag |= CHANGED_GENRE_TAG;
}
String Info::get_genre() const {
    if(String::null != genre)
        return genre;
    else
        return String();
}


void Info::set_disk_number(int number) {
    disk_number = number;
    if(disk_number < 0)
        disk_number = 0;
    changedflag |= CHANGED_DATA_DISK_NUM;
}
int Info::get_disk_number() const {
    return disk_number;
}


void Info::set_artist(String new_artist) {
    artist = new_artist;
    changedflag |= CHANGED_ARTIST_TAG;
}
String Info::get_artist() const {
    if(String::null != artist)
        return artist;
    else
        return String();
}

void Info::set_is_compilation(bool compilation) {
    is_compilation = compilation;
    changedflag |= CHANGED_IS_COMPILATION_TAG;
}
bool Info::get_is_compilation() const {
    return is_compilation;
}

void Info::set_album_artist(String new_album_artist) {
    album_artist = new_album_artist;
    changedflag |= CHANGED_DATA_ALBUMARTIST;
}
String Info::get_album_artist() const {
    if(String::null != album_artist)
        return album_artist;
    else
        return String();
}

void Info::set_album(String new_album) {
    album = new_album;
    changedflag |= CHANGED_ALBUM_TAG;
}
String Info::get_album() const {
    if(String::null != album)
        return album;
    else
        return String();
}

void Info::set_composer(String new_composer) {
    composer = new_composer;
    changedflag |= CHANGED_COMPOSER_TAG;
}
String Info::get_composer() const {
    if(String::null != composer)
        return composer;
    else
        return String();
}

void Info::set_comments(String new_comments) {
    comments = new_comments;
    changedflag |= CHANGED_COMMENT_TAG;
}
String Info::get_comments(void) const {
    if(String::null != comments)
        return comments;
    else
        return String();
}

void Info::set_tracknumber(int new_tracknumber) {
    tracknumber = new_tracknumber;
    changedflag |= CHANGED_TRACKNO_TAG;
}
int Info::get_tracknumber() const {
    return tracknumber;
}

void Info::set_year(int new_year) {
    year = new_year;
    changedflag |= CHANGED_YEAR_TAG;
}
int Info::get_year() const {
    return year;
}

int Info::get_channels() const {
    return channels;
}

int Info::get_bitrate() const {
    return bitrate;
}

int Info::get_samplerate() const {
    return samplerate;
}

int Info::get_length_seconds() const {
    return length_seconds;
}

int  Info::get_playcount() const {
    return playcount;
}
void Info::set_playcount(int new_playcount) {
    changedflag |= CHANGED_DATA_PLAYCOUNT;
    playcount = new_playcount;
    if(playcount < 0)
        playcount = 0;
}

int  Info::get_rating() const {
    return rating;
}
void Info::set_rating(int new_rating) {
    changedflag |= CHANGED_DATA_RATING;
    rating = new_rating;
    if(rating < 0)
        rating = 0;
}

void Info::set_track_labels_list(const StringList &new_track_labels_list) {
    track_labels = new_track_labels_list;
    track_labels_string = new_track_labels_list.toString("|");
    changedflag |= CHANGED_TRACK_LABELS;
}
StringList Info::get_track_labels_list() const {
    return track_labels;
}

void Info::set_artist_labels_list(const StringList &new_artist_labels_list) {
    artist_labels = new_artist_labels_list;
    artist_labels_string = new_artist_labels_list.toString("|");
    changedflag |= CHANGED_ARTIST_LABELS;
}
StringList Info::get_artist_labels_list() const {
    return artist_labels;
}

void Info::set_album_labels_list(const StringList &new_album_labels_list) {
    album_labels = new_album_labels_list;
    album_labels_string = new_album_labels_list.toString("|");
    changedflag |= CHANGED_ALBUM_LABELS;
}
StringList Info::get_album_labels_list() const {
    return album_labels;
}

bool Info::get_has_image() const {
    return has_image;
}

bool Info::can_handle_images() const {
    return false;
}

bool Info::get_image(char*& data, int &data_length, ImageType &image_type) const {
    data = NULL;
    data_length = 0;
    image_type = IMAGE_TYPE_UNKNOWN;
    return false;
}

bool Info::set_image(const char* data, int data_length, ImageType image_type) {
    return false;
}


bool Info::can_handle_lyrics() const {
    return false;
}


String Info::get_lyrics() const {
    return String();
}


bool Info::set_lyrics(const String &lyrics) {
    return false;
}

String Info::get_videocodec(void) const{
    return String();
}
void   Info::set_videocodec(String new_title){
}

String Info::get_audiocodec(void) const{
    return String();
}

void   Info::set_audiocodec(String new_title){

}

String Info::get_containerformat(void) const{
    return String();
}
void   Info::set_containerformat(String new_title){

}

String Info::get_languagecode(void) const{
    return String();
}
void   Info::set_languagecode(String new_languagecode){
}

String Info::get_copyrights(void) const{
    return String();
}
void   Info::set_copyrights(String new_copyrights){
}

String Info::get_encoder(void) const{
    return String();
}
void   Info::set_encoder(String new_encoder){
}

int Info::get_width() const{
    return 0;
}

int Info::get_height() const{
    return 0;
}

int Info::get_mpegversion() const{
    return 0;
}

unsigned long long int Info::get_duration() const{
    return 0;
}

int Info::get_episode() const{
    return 0;
}

String Info::get_profile(void) const{
    return String();
}

float Info::get_level() const{
    return 0;
}

int Info::get_framerate() const{
    return 0;
}

unsigned int Info::get_genrenumber() const{
    return 255;
}

bool Info::get_encrypted(void) const{
    return false;
}

String Info::get_image_date_time(void) const{
    return String();
}

String Info::get_image_date_time_Modified(void) const{
    return String();
}

int Info::get_image_x_dimension(void) const{
    return 0;
}

int Info::get_image_y_dimension(void) const{
    return 0;
}

int Info::get_image_x_resolution(void) const{
    return 0;
}

int Info::get_image_y_resolution(void) const{
    return 0;
}

long int Info::get_image_size(void) const{
    return 0;
}

MediaType Info::get_media_type(void)const{
    return MEDIA_TYPE_UNKNOWN;
}


//yomi data setter and getter definition

void Info::set_yomititle(const String& yomi_title)
{
    if((!yomi_title.isEmpty()) && (yomi_title != yomititle))
    {
        yomititle = yomi_title;
        changedyomidataflag |= CHANGED_YOMITITLE_TAG;
    }
}

String Info::get_yomititle() const
{
    return yomititle;
}

void Info::set_yomiartist(const String& yomi_artist)
{
    if((!yomi_artist.isEmpty()) && (yomi_artist != yomiartist))
    {
        yomiartist = yomi_artist;
        changedyomidataflag |= CHANGED_YOMIARTIST_TAG;
    }
}

String Info::get_yomiartist() const
{
    return yomiartist;
}

void Info::set_yomialbum(const String& yomi_album)
{
    if((!yomi_album.isEmpty()) && (yomi_album != yomialbum))
    {
        yomialbum = yomi_album;
        changedyomidataflag |= CHANGED_YOMIALBUM_TAG;
    }
}

String Info::get_yomialbum() const
{
    return yomialbum;
}

void Info::set_encodingtime(const String& encodingtime)
{
    if((!encodingtime.isEmpty()) && (encodingtime != encoding_time))
    {
        encoding_time = encodingtime;
        changedflag |= CHANGED_ENCODINGTIME_TAG;
    }
}

String Info::get_encodingtime() const
{
    return encoding_time;
}
