/*
 * Copyright (C) 2008-2013 J.Rios <anonbeat@gmail.com>
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 *     Jörn Magens <shuerhaaken@googlemail.com>
 *     Matias De lellis <mati86dl@gmail.com>
 *     Pavel Vasin <rat4vier@gmail.com>
 */



#include "taginfo.h"
#include "taginfo_internal.h"



using namespace TagInfo;


FlacInfo::FlacInfo(const String &filename) : Info(filename) {
    if(!file_name.isEmpty() && !create_file_ref())
        printf("Error creating file ref! %s\n", filename.toCString(false));
    if(taglib_file) {
        xiphcomment = ((TagLib::FLAC::File *) taglib_file)->xiphComment();
        if(!xiphcomment || xiphcomment->isEmpty()) { // Use fallback for id3v1 or extended header
            taglib_tag = ((TagLib::FLAC::File *) taglib_file)->tag();
            if(!taglib_tag) {
                printf("Cant get tag object from '%s'\n", file_name.toCString(false));
                valid = false;
            }
        }
    }
    else {
        printf("Cant get xiphcomment from '%s'\n", file_name.toCString(false));
        xiphcomment = NULL;
        valid = false;
    }
}


FlacInfo::~FlacInfo() {
    xiphcomment = NULL;
}


bool FlacInfo::create_file_ref() {
    if(file_name.isEmpty())
        return false;
    taglib_file = new TagLib::FLAC::File(file_name.toCString(false), true, TagLib::AudioProperties::Fast);
    if(taglib_file) {
        return true;
    }
    else {
        printf("TagLib::File creation failed for '%s'\n", file_name.toCString(false));
        return false;
    }
}


bool FlacInfo::read(void) {
    if(Info::read()) {
        if(xiphcomment) {

            if(!taglib_tag)
                read_virtual_tags((TagLib::Tag *)xiphcomment);

            if(xiphcomment->fieldListMap().contains("COMPOSER")) {
                composer = xiphcomment->fieldListMap()["COMPOSER"].front();
            }
            if(xiphcomment->fieldListMap().contains("DISCNUMBER")) {
                disk_number =
                    atol(xiphcomment->fieldListMap()["DISCNUMBER"].front().toCString(false));
                if(disk_number <= 0)
                    disk_number = 1;
            }
            if(xiphcomment->fieldListMap().contains("COMPILATION")) {
                is_compilation = xiphcomment->fieldListMap()["COMPILATION"].front() == String("1");
            }
            if(xiphcomment->fieldListMap().contains("ALBUMARTIST")) {
                album_artist = xiphcomment->fieldListMap()["ALBUMARTIST"].front();
            }
            else if(xiphcomment->fieldListMap().contains("ALBUM ARTIST")) {
                album_artist = xiphcomment->fieldListMap()["ALBUM ARTIST"].front();
            }
            // Rating
            if(xiphcomment->fieldListMap().contains("RATING")) {
                long rat = 0;
                rat = atol(xiphcomment->fieldListMap()["RATING"].front().toCString(true));
                if(rat != 0) {
                    if(rat > 5)
                        rating = popularity_to_rating(rat);
                    else
                        rating = rat;
                }
            }
            if(xiphcomment->fieldListMap().contains("PLAY_COUNTER")) {
                playcount = atol(xiphcomment->fieldListMap()["PLAY_COUNTER"].front().toCString(false));
                if(playcount < 0)
                    playcount = 0;
            }
            // Labels
            if(track_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("TRACK_LABELS")) {
                    track_labels_string = xiphcomment->fieldListMap()["TRACK_LABELS"].front();
                    track_labels = split(track_labels_string, "|");
                }
            }
            if(artist_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("ARTIST_LABELS")) {
                    artist_labels_string = xiphcomment->fieldListMap()["ARTIST_LABELS"].front();
                    artist_labels = split(artist_labels_string, "|");
                }
            }
            if(album_labels.size() == 0) {
                if(xiphcomment->fieldListMap().contains("ALBUM_LABELS")) {
                    album_labels_string = xiphcomment->fieldListMap()["ALBUM_LABELS"].front();
                    album_labels = split(album_labels_string, "|");
                }
            }
            if(xiphcomment->contains("COVERART")) {// TODO
                has_image = true;
            }
            else {
                TagLib::FLAC::File * fi = ((TagLib::FLAC::File *) taglib_file);
                List<TagLib::FLAC::Picture *> plist = fi->pictureList();
                has_image = (plist.size() > 0);
            }
            return true;
        }
    }
    return false;
}



bool FlacInfo::write(void) {
    if(xiphcomment) {
        if(changedflag) {
            if(changedflag & CHANGED_DATA_DISK_NUM)
                xiphcomment->addField("DISCNUMBER", format("%u", disk_number));
            if(changedflag & CHANGED_COMPOSER_TAG)
                xiphcomment->addField("COMPOSER", composer);

            if(changedflag & CHANGED_IS_COMPILATION_TAG) {
                if(is_compilation)
                    xiphcomment->addField("COMPILATION", String("1"));
                else
                    xiphcomment->addField("COMPILATION", String("0"));
            }

            if(changedflag & CHANGED_DATA_ALBUMARTIST)
                xiphcomment->addField("ALBUMARTIST", album_artist);

            if(changedflag & CHANGED_DATA_RATING) {
                xiphcomment->addField("RATING", format("%u", rating_to_popularity(rating)));
            }
            if(changedflag & CHANGED_DATA_PLAYCOUNT) {
                xiphcomment->addField("PLAY_COUNTER", format("%u", playcount));
            }
            // The Labels
            if(changedflag & CHANGED_TRACK_LABELS)
                check_xiph_label_frame(xiphcomment, "TRACK_LABELS", track_labels_string);
            if(changedflag & CHANGED_ARTIST_LABELS)
                check_xiph_label_frame(xiphcomment, "ARTIST_LABELS", artist_labels_string);
            if(changedflag & CHANGED_ALBUM_LABELS)
                check_xiph_label_frame(xiphcomment, "ALBUM_LABELS", album_labels_string);

            write_virtual_tags((TagLib::Tag *)xiphcomment);
        }
    }
    return Info::write();
}


bool FlacInfo::can_handle_images(void) const {
    return true;
}

bool FlacInfo::get_image(char*& data, int &data_length, ImageType &image_type) const {
    data = NULL;
    data_length = 0;
    image_type = IMAGE_TYPE_UNKNOWN;

    bool ret = false;

    if(has_image)
    {
        ret =  get_xiph_comment_cover_art(xiphcomment, data, data_length, image_type);

        // get_xiph_comment_cover_art id returning false for FLAC file even if the file has Albumart.
        // Try to get the Picture list and read the FrontCover from it
        if(!ret && (taglib_file != NULL))
        {
            TagLib::FLAC::File * fi = ((TagLib::FLAC::File *) taglib_file);
            List<FLAC::Picture *> plist = fi->pictureList();

            for(List<TagLib::FLAC::Picture *>::Iterator it = plist.begin(); it != plist.end(); ++it)
            {
                if((*it != NULL) && (TagLib::FLAC::Picture::FrontCover == (*it)->type()))
                {
                    String mimetype = (*it)->mimeType();

                    if(mimetype.find("/jpeg") != -1 || mimetype.find("/jpg") != -1)
                        image_type = IMAGE_TYPE_JPEG;
                    else if(mimetype.find("/png") != -1)
                        image_type = IMAGE_TYPE_PNG;

                    ByteVector Image = (*it)->data();
                    char* ImageData = Image.data();

                    data_length = Image.size();

                    if((data_length > 0) && (NULL != ImageData))
                    {
                        data = new char[data_length];
                        memcpy(data, ImageData, data_length);
                        ret = true;
                    }
                    else
                    {
                        data_length = 0;
                        image_type = IMAGE_TYPE_UNKNOWN;
                    }

                    break;
                }
            }
        }
    }

    return ret;
}

bool FlacInfo::set_image(const char* data, int data_length, ImageType image_type) {
    if(xiphcomment)
        return set_xiph_comment_cover_art(xiphcomment, data, data_length, image_type);

    return false;
}


bool FlacInfo::can_handle_lyrics(void) const {
    return true;
}


String FlacInfo::get_lyrics(void) const {
    return get_xiph_comment_lyrics(xiphcomment);
}


bool FlacInfo::set_lyrics(const String &lyrics) {
    return set_xiph_comment_lyrics(xiphcomment, lyrics);
}


