/***************************************************************************
 *
 * Copyright 2010,2011 BMW Car IT GmbH
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include "scene_provider.h"
#include "ICommandExecutor.h"
#include "CommitCommand.h"
#include "LayerCreateCommand.h"
#include "LayerSetDestinationRectangleCommand.h"
#include "LayerSetSourceRectangleCommand.h"
#include "LayerSetVisibilityCommand.h"
#include "LayerSetOpacityCommand.h"
#include "ScreenSetRenderOrderCommand.h"
#include "appids.h"
#include "SurfaceCreateCommand.h"
#include "SurfaceSetVisibilityCommand.h"
#include "SurfaceSetOpacityCommand.h"
#include "SurfaceSetDimensionCommand.h"
#include "SurfaceSetDestinationRectangleCommand.h"
#include "SurfaceSetSourceRectangleCommand.h"
#include "Configuration.h"
#include "Log.h"
#include <unistd.h>
#include <stdio.h>

AIVISceneProvider::AIVISceneProvider(ICommandExecutor& executor, Configuration& config)
: ISceneProvider(&executor)
, mExecutor(executor)
, mConfiguration(config)
{
}

typedef struct t_layerScene
{
    uint layer;
    bool visibility;
    float opacity;
} layerScene;

static layerScene g_scene1_layers[] =
{
    // for non-pivi Rvc should be on lower display
  #ifndef VARIANT_S_FTR_ENABLE_DUAL_ANIMATION
    { LAYER_RVC          /* 5000 */, false, 1.0},
    { LAYER_RVC_TEE      /* 5003 */, false, 1.0},
    { LAYER_RVC_GRAPHICS /* 5001 */, false, 1.0},
  #endif
    { LAYER_ANIMATION    /* 5002 */, false, 1.0},
};

#ifdef VARIANT_S_FTR_ENABLE_DUAL_ANIMATION
static layerScene g_scene2_layers[] =
{
   // for pivi Rvc should be on Upper Display (Secondary Display)
   { LAYER_RVC          /* 5000 */, false, 1.0},
   { LAYER_RVC_TEE      /* 5003 */, false, 1.0},
   { LAYER_RVC_GRAPHICS /* 5001 */, false, 1.0},
   { LAYER_ANIMATION2   /* 6013 */, false, 1.0}
};
#endif //VARIANT_S_FTR_ENABLE_DUAL_ANIMATION

/* has to be the same for all Scene plugins*/
PluginApi AIVISceneProvider::pluginGetApi() const
{
    return SceneProvider_Api_v1;
}

/* can be the same for all Scene plugins
 * this plugin is not running all the time
 * nobody should use this function
 * but we have to fulfill the plugin interface*/
HealthCondition AIVISceneProvider::pluginGetHealth()
{
    return HealthRunning;
}

/* GM specific implementation, helper function
 * find out the biggest width and biggest height in all screens
 * will be used to create a layer in size which will cover all screens
 * means the application surfaces will not be truncated by the screen size
 * not the layer size*/
uint* AIVISceneProvider::find_max_resolution(uint* scr_ids, int num_ids)
{
    int i;
    uint *max_res = new uint[2];
    uint *resolution = new uint [2];

    max_res[0] = max_res[1] = 0;

    for (i = 0; i < num_ids; ++i)
    {
        resolution = mExecutor.getScreenResolution(scr_ids[i]);

        if (resolution[0] > max_res[0])
            max_res[0] = resolution[0];
        if (resolution[1] > max_res[1])
            max_res[1] = resolution[1];
    }

    delete[] resolution;

    return max_res;
} 

/*Core function of the Scene plugin
 * this will be executed when plugin is loaded*/
bool AIVISceneProvider::delegateScene()
{
    uint num_screens = 0;
	uint current_screen = 0;
    uint *screen_ids = mExecutor.getScreenIDs( &num_screens);
    bool result = true;
    result = CreateSceneLayers(&g_scene1_layers[0], sizeof(g_scene1_layers)/sizeof(t_layerScene), 1);
#ifdef VARIANT_S_FTR_ENABLE_DUAL_ANIMATION
    if (result == true)
        result = CreateSceneLayers(&g_scene2_layers[0], sizeof(g_scene2_layers)/sizeof(t_layerScene), 2);
#endif
	
/*	Commented the code since the screen id is not retrieved in ascending order. Have to check with ADIT - 04-May-2017 - ADY4COB
    for (current_screen =0; current_screen < num_screens; current_screen++)
    {
		switch (current_screen)
		{
			case 0:
			    result = CreateSceneLayers(&g_scene1_layers[0], sizeof(g_scene1_layers)/sizeof(t_layerScene), screen_ids[current_screen]);
			break;
			case 1:
#ifdef VARIANT_S_FTR_ENABLE_DUAL_ANIMATION
			    result = CreateSceneLayers(&g_scene2_layers[0], sizeof(g_scene2_layers)/sizeof(t_layerScene), screen_ids[current_screen]);
#endif
			break;
			default:
			break;
		}
    }
*/	
    if (screen_ids != NULL)
        delete[] screen_ids;

    //To survive an invalid configuration delegateScene() will always indicate success
    if (result == false)
    {
        result = true;
        LOG_INFO("AIVISceneProvider", "SCENE PROVIDER PLUGIN: delegateScene() failure but returning success");
    }
    else
    {
        LOG_INFO("AIVISceneProvider", "SCENE PROVIDER PLUGIN: delegateScene() successful");
    }
	return result;
}

/* all Scene setup has to be defined here*/
bool AIVISceneProvider::CreateSceneLayers(struct t_layerScene stLayerScene[], uint num_layers, uint nScreenID)
{
    bool result = true;
    int i = 0;
    uint *render_order = new uint [num_layers];
    uint* max_resolution = NULL;
    uint num_screens = 0;

    pid_t lm_pid = getpid();
    uint *screen_ids = mExecutor.getScreenIDs( &num_screens);
    max_resolution = find_max_resolution(screen_ids, num_screens);

    if (num_layers > 0)
    {
	 	/* setup inital layer scenery */
		for (i = 0; i < num_layers; i++)
		{
				/*create and configure ilm layer*/
				result &= mExecutor.execute(new LayerCreateCommand(lm_pid,
																max_resolution[0],
																max_resolution[1],
																&(stLayerScene[i].layer)));
				result &= mExecutor.execute(new LayerSetSourceRectangleCommand(lm_pid,
																		stLayerScene[i].layer,
																			0, 0,
																			max_resolution[0],
																			max_resolution[1]));
				result &= mExecutor.execute(new LayerSetDestinationRectangleCommand(lm_pid,
																					stLayerScene[i].layer,
																					0, 0,
																					max_resolution[0],
																					max_resolution[1]));
				result &= mExecutor.execute(new LayerSetOpacityCommand(lm_pid,
																	stLayerScene[i].layer,
																	stLayerScene[i].opacity));
				result &= mExecutor.execute(new LayerSetVisibilityCommand(lm_pid,
																		stLayerScene[i].layer,
																		stLayerScene[i].visibility));
				result &= mExecutor.execute(new CommitCommand(lm_pid));
				render_order[i]=stLayerScene[i].layer;
		}
		/* Finally set the first executed renderorder */
		/*needed a tmp variable because ScreenSetRenderOrderCommand releases the render_order memory*/
		uint *tmp_render_order = new uint [i];
		memcpy(tmp_render_order, render_order, sizeof(uint)*i);

		result &= mExecutor.execute(
				new ScreenSetRenderOrderCommand(lm_pid, nScreenID, tmp_render_order, i));
		result &= mExecutor.execute(new CommitCommand(lm_pid));
    }
    else
        result = false;

    /*cleanup*/
    if (render_order != NULL)
        delete[] render_order;

    if (screen_ids != NULL)
        delete[] screen_ids;

    delete[] max_resolution;
    LOG_INFO("AIVISceneProvider", "SCENE PROVIDER PLUGIN WAS EXECUTED");
    return result;
}

/* should be the name of a class!!!*/
t_ilm_const_string AIVISceneProvider::pluginGetName() const
{
    return "AIVISceneProvider";
}

int AIVISceneProvider::getIterationCounter()
{
    // this plugin is one-shot and is not monitored.
    static int dummyValue = 0;
    return ++dummyValue;
}

/* The name in brakes has to be the same as name of a class!!!
 * otherwise plugin will not be accepted*/
DECLARE_LAYERMANAGEMENT_PLUGIN(AIVISceneProvider)
