/*
 *
 *  Connection Manager
 *
 *  Copyright (C) 2012-2014  BMW Car IT GmbH.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */
/*
 * Extended with SourceIP based routing and MultiService per dbus client support
 * 2017
 * Robert Bosch Car Multimedia, Robert Bosch Strasse 200, Hildesheim, Deutschland
 * Björn Thorwirth, Anderas Achtzehn
 */



#include <errno.h>
#include <string.h>
#include <sys/inotify.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#include <stdlib.h>
#include <stdio.h>
#include <gio.h>

#define CONNMAN_API_SUBJECT_TO_CHANGE
#include <connman/plugin.h>
//#include <connman/log.h>
#include <connman/session.h>
#include <connman/service.h>
#include <connman/dbus.h>
#include <connman/inotify.h>



#include "session_policy_local_defines.h"
#ifdef LOG_DLT
#include <dlt/dlt.h>

// DLT Macro
#define INF_PLUGIN(format)			_INFO_TRACE format

#define _INFO_TRACE(format,args...) do { \
	char log_buff[1024]; \
    char final_string[1024]; \
    snprintf(final_string, sizeof(final_string),"LINE:%d ", __LINE__); \
    snprintf(log_buff, sizeof(log_buff), format,##args); \
    strncat(final_string, log_buff, sizeof(final_string)-1); \
    DLT_LOG(CMP1, DLT_LOG_INFO, DLT_STRING(final_string)); \
} while(0)

#define ERR_PLUGIN(format)			_ERROR_TRACE format

#define _ERROR_TRACE(format,args...) do {                                \
    char log_buff[1024];                                                \
    char final_string[1024];                                            \
    snprintf(final_string, sizeof(final_string),"LINE:%d ", __LINE__);  \
    snprintf(log_buff, sizeof(log_buff), format,##args);                \
    strncat(final_string, log_buff, sizeof(final_string)-1);              \
	  DLT_LOG(CMP1, DLT_LOG_ERROR, DLT_STRING(final_string));              \
  } while(0)


DLT_DECLARE_CONTEXT(CMP1)
// DLT Macro
#define DBG_PLUGIN(format)			_DBG_TRACE format

#define _DBG_TRACE(format,args...) do { \
    char log_buff[1024]; \
    char final_string[1024]; \
    snprintf(final_string, sizeof(final_string),"LINE:%d ", __LINE__); \
    snprintf(log_buff, sizeof(log_buff), format,##args); \
    strncat(final_string, log_buff, sizeof(final_string)-1); \
    DLT_LOG(CMP1, DLT_LOG_DEBUG, DLT_STRING(final_string)); \
} while(0)

#ifndef DLT_APPID_CONNMANPLUGIN
#define DLT_APPID_CONNMANPLUGIN 				"CMPL"

#endif

#else
#define DBG_PLUGIN(format)			DBG format
#define ERR_PLUGIN(format)			DBG format
#define INF_PLUGIN(format)			DBG format
#endif


#define MODE		(S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | \
			S_IXGRP | S_IROTH | S_IXOTH)

//This needs to mach connman version!!! Find exact value in config.h which will be generated during connman build
#define VERSION "1.40"

static DBusConnection *connection;

static GHashTable *file_hash;    /* (filename, policy_file) */
static GHashTable *session_hash; /* (connman_session, policy_config) */
static GHashTable *session_creation_hash; /* (connman_session, policy_creation_data) */
static GHashTable *firewall_context_hash; /* (connman_session, firewall_context ) */
static GHashTable *session_state_hash; /* (connman_session, connman_session_state */
/* Global lookup tables for mapping sessions to policies */
static GHashTable *selinux_hash; /* (lsm context, policy_group) */
static GHashTable *uid_hash;     /* (uid, policy_group) */
static GHashTable *gid_hash;     /* (gid, policy_group) */
static GHashTable *userservice_hash;     /* (userservice, policy_group) */



struct firewall_context {
	GList *rules;
};

static struct policy_group *default_group;


/*
 * A instance of struct policy_file is created per file in
 * POLICYDIR.
 */
struct policy_file {
	/*
	 * A valid file is a keyfile with one ore more groups. All
	 * groups are keept in this list.
	 */
	GSList *groups;
};

struct policy_creation_data {
	char *userservice;
};
/*
* BIG FAT WARNING
* creation_data has been copied from session.c and may be subject to change
*/
struct creation_data {
	DBusMessage *pending;
	struct connman_session *session;

	/* user config */
	enum connman_session_type type;
	GSList *allowed_bearers;
	char *allowed_interface;
	bool source_ip_rule;
	char *context_identifier;
	char *service;
};

struct policy_group {
	char *selinux;
	char *uid;
	char *gid;
	char *userservice;

	GSList *source_ips;
        //string tokens parsed from policy
	GSList *allowed_bearers_names;
        // list of parsed CONNMAN_SERVICE_TYPE enums -> our "specific bearers will be "0"
        //SERVICE_TYPE_UNKNOWN
	GSList *allowed_bearers; /* we maintain a local copy of the allowed bearers list */
	/*
	 * Each policy_group owns a config and is not shared with
	 * sessions. Instead each session copies the valued from this
	 * object.
	 */
	struct connman_session_config *config;

	/* All 'users' of this policy. */
	GSList *sessions;
};

/* A struct policy_config object is created and owned by a session. */
struct policy_config {
	char *selinux;
	char *selinux_context;
	char *uid;
	GSList *gids;

	int sip_forward_id;
	guint service_no;
	/* The policy config owned by the session */
	struct connman_session_config *config;

	/* To which policy belongs this policy_config */
	struct connman_session *session;
	/*
	 * Points to the policy_group when a config has been applied
	 * from a file.
	 */
	struct policy_group *group;
};


struct cb_data {
	void *cb;
	void *user_data;
	void *data;
};

static inline struct cb_data *cb_data_new(void *cb, void *user_data)
{
	struct cb_data *ret;

	ret = g_new0(struct cb_data, 1);
	ret->cb = cb;
	ret->user_data = user_data;

	return ret;
}


static int default_bearers[] = {
	CONNMAN_SERVICE_TYPE_ETHERNET,
	CONNMAN_SERVICE_TYPE_WIFI,
	CONNMAN_SERVICE_TYPE_BLUETOOTH,
	CONNMAN_SERVICE_TYPE_CELLULAR,
	CONNMAN_SERVICE_TYPE_GADGET,
};

static void add_default_bearer_types(GSList **list)
{
	unsigned int i;

	for (i = 0; i < G_N_ELEMENTS(default_bearers); i++)
		*list = g_slist_append(*list,
				GINT_TO_POINTER(default_bearers[i]));
}

static void copy_session_config(struct connman_session_config *dst,
			struct connman_session_config *src)
{
	g_slist_free(dst->allowed_bearers);

	dst->allowed_bearers = g_slist_copy(src->allowed_bearers);
	dst->ecall = src->ecall;
	dst->type = src->type;
	dst->roaming_policy = src->roaming_policy;
	dst->priority = src->priority;
}

static void set_policy(struct policy_config *policy,
			struct policy_group *group)
{
	DBG_PLUGIN(("policy %p group %p\n", policy, group));

	group->sessions = g_slist_prepend(group->sessions, policy);
	policy->group = group;

	copy_session_config(policy->config, group->config);
}

static char *parse_selinux_type(const char *context)
{
	char *ident, **tokens;

	/*
	 * SELinux combines Role-Based Access Control (RBAC), Type
	 * Enforcment (TE) and optionally Multi-Level Security (MLS).
	 *
	 * When SELinux is enabled all processes and files are labeled
	 * with a contex that contains information such as user, role
	 * type (and optionally a level). E.g.
	 *
	 * $ ls -Z
	 * -rwxrwxr-x. wagi wagi unconfined_u:object_r:haifux_exec_t:s0 session_ui.py
	 *
	 * For identifyng application we (ab)using the type
	 * information. In the above example the haifux_exec_t type
	 * will be transfered to haifux_t as defined in the domain
	 * transition and thus we are able to identify the application
	 * as haifux_t.
	 */

	tokens = g_strsplit(context, ":", 0);
	if (g_strv_length(tokens) < 2) {
		g_strfreev(tokens);
		return NULL;
	}

	/* Use the SELinux type as identification token. */
	ident = g_strdup(tokens[2]);

	g_strfreev(tokens);

	return ident;
}

static void cleanup_config(gpointer user_data);

static void finish_create(struct policy_config *policy,
				connman_session_config_func_t cb,
				void *user_data)
{
	struct policy_group *group = NULL;
	struct policy_creation_data *policy_creation_data = NULL;
	char *service = NULL;
	GSList *list;
	struct creation_data *cdata = user_data;
	policy_creation_data = g_hash_table_lookup(session_creation_hash,policy->session);
	service = policy_creation_data->userservice;
	DBG_PLUGIN(("finish_create: %s",service));
	if (policy->selinux) {
		if (!service) {
			group = g_hash_table_lookup(selinux_hash, policy->selinux);
		} else {
			char *long_service = g_strconcat("selinux:",policy->selinux,"/",service,NULL);
			group = g_hash_table_lookup(userservice_hash, long_service);
		}
	}

	if (group) {
		set_policy(policy, group);

		policy->config->id_type = CONNMAN_SESSION_ID_TYPE_LSM;
		policy->config->id = g_strdup(policy->selinux_context);
		goto done;
	}

	if (policy->uid) {
	        DBG_PLUGIN(("finish_create: %s",policy->uid));
		if (!service) {
			group = g_hash_table_lookup(uid_hash, policy->uid);
		} else {

			char *long_service = g_strconcat("uid:",policy->uid,"/",service,NULL);
		        DBG_PLUGIN(("finish_create: long_service %s",long_service));
			group = g_hash_table_lookup(userservice_hash, long_service);


		}
	}

	if (group) {
	        DBG_PLUGIN(("finish_create: found group %s ", group->uid));
		set_policy(policy, group);
		cdata->source_ip_rule = TRUE;
		policy->config->id_type = CONNMAN_SESSION_ID_TYPE_UNKNOWN;
		policy->config->id = g_strdup(policy->uid);
		goto done;
	}

	for (list = policy->gids; list; list = list->next) {
		char *gid = list->data;

		group = g_hash_table_lookup(gid_hash, gid);
		if (!group)
			continue;

		set_policy(policy, group);

		policy->config->id_type = CONNMAN_SESSION_ID_TYPE_GID;
		policy->config->id = g_strdup(gid);
		break;
	}
done:
	if (policy->uid && !group) // this is a hack to stop falling back to unrestrictive default policy of base session API
	{
	  DBG_PLUGIN(("finish_create: failed to load group.fallback to default_group"));
	      set_policy(policy, default_group);
	}

	g_hash_table_replace(session_hash, policy->session, policy);

	(*cb)(policy->session, policy->config, user_data, 0);
}

static void failed_create(struct policy_config *policy,
			connman_session_config_func_t cb,
			void *user_data, int err)
{
	(*cb)(policy->session, NULL, user_data, err);

	cleanup_config(policy);
}

static void selinux_context_reply(const unsigned char *context, void *user_data,
					int err)
{
	struct cb_data *cbd = user_data;
	connman_session_config_func_t cb = cbd->cb;
	struct policy_config *policy = cbd->data;
	char *ident = NULL;

	DBG_PLUGIN(("session %p", policy->session));

	if (err == -EIO) {
		/* No SELinux support, drop back to UID/GID only mode */
		finish_create(policy, cb, cbd->user_data);
		goto done;
	}

	if (err < 0) {
		failed_create(policy, cb, cbd->user_data, err);
		goto done;
	}

	DBG_PLUGIN(("SELinux context %s", context));

	policy->selinux_context = g_strdup((const char *)context);
	ident = parse_selinux_type(policy->selinux_context);
	if (ident)
		policy->selinux = g_strdup(ident);

	finish_create(policy, cb, cbd->user_data);

done:
	g_free(cbd);
	g_free(ident);
}

static void get_uid_reply(unsigned int uid, void *user_data, int err)
{
	struct cb_data *cbd = user_data;
	connman_session_config_func_t cb = cbd->cb;
	struct policy_config *policy = cbd->data;
	const char *owner;
	struct passwd *pwd;
	struct group *grp;
	gid_t *groups = NULL;
	int nrgroups, i;

	DBG_PLUGIN(("get_uid_reply:session %p uid %d", policy->session, uid));

	if (err < 0) {
	        ERR_PLUGIN(("get_uid_reply: uid reply error %i", err));
		// cleanup_config(policy); --> cleanup_config is already triggered in failed_create()
		goto err;
	}

	pwd = getpwuid((uid_t)uid);
	if (!pwd) {
	        ERR_PLUGIN(("get_uid_reply: failed to get pwd err: %i", errno));
		if (errno != 0)
			err = -errno;
		else
			err = -EINVAL;
		goto err;
	}

	policy->uid = g_strdup(pwd->pw_name);

	nrgroups = 0;
	getgrouplist(pwd->pw_name, pwd->pw_gid, NULL, &nrgroups);
	groups = g_try_new0(gid_t, nrgroups);
	if (!groups) {
		err = -ENOMEM;
		goto err;
	}

	err = getgrouplist(pwd->pw_name, pwd->pw_gid, groups, &nrgroups);
	if (err < 0)
		goto err;

	for (i = 0; i < nrgroups; i++) {
		grp = getgrgid(groups[i]);
		if (!grp) {
			if (errno != 0)
				err = -errno;
			else
				err = -EINVAL;
			goto err;
		}

		policy->gids = g_slist_prepend(policy->gids,
					g_strdup(grp->gr_name));
	}
	g_free(groups);

	owner = connman_session_get_owner(policy->session);

	err = connman_dbus_get_selinux_context(connection, owner,
						selinux_context_reply, cbd);
	if (err == 0) {
		/*
		 * We are able to ask for a SELinux context. Let's defer the
		 * creation of the session config until we get the answer
		 * from D-Bus.
		 */
		return;
	}

	finish_create(policy, cb, cbd->user_data);
	g_free(cbd);

	return;

err:
	failed_create(policy, cb, cbd->user_data, err);
	g_free(cbd);
	g_free(groups);
}

static int policy_local_create(struct connman_session *session,
				connman_session_config_func_t cb,
				void *user_data)
{
	struct firewall_context *fw_ctx = NULL;
	struct cb_data *cbd = cb_data_new(cb, user_data);
	struct policy_config *policy = NULL;
	struct creation_data *cdata = NULL;
	struct policy_creation_data *policy_cdata = g_new0(struct policy_creation_data,1);
	const char *owner;
	int err;

	DBG_PLUGIN(("session %p", session));

	policy = g_new0(struct policy_config, 1);
	policy->config = connman_session_create_default_config();
	policy->session = session;
	cdata = (struct creation_data*) user_data;
	policy_cdata->userservice = g_strdup(cdata->service);
	fw_ctx = g_new0(struct firewall_context, 1);
	g_hash_table_replace(firewall_context_hash, session, fw_ctx);
	policy->service_no = G_MAXUINT;
	cbd->data = policy;

	DBG_PLUGIN(("policy_local_create: Service: %s" , policy_cdata->userservice));
	g_hash_table_replace(session_creation_hash, session, policy_cdata);

	owner = connman_session_get_owner(session);

	err = connman_dbus_get_connection_unix_user(connection, owner, get_uid_reply, cbd);
	if (err < 0) {
		connman_error("Could not get UID");
		cleanup_config(policy);
		g_free(cbd);
		g_hash_table_remove(firewall_context_hash,session);
		g_hash_table_remove(session_creation_hash, session);
		g_free(fw_ctx);
		g_free(policy_cdata->userservice);
		g_free(policy_cdata);
		return err;
	}

	return 0;
}

void cleanup_policy_creation_data(struct policy_creation_data *policy_creation_data)
{
	if (policy_creation_data->userservice) {
		g_free(policy_creation_data->userservice);
	}

	g_free(policy_creation_data);
}

static void policy_local_destroy(struct connman_session *session)
{
	struct policy_config *policy;
	struct policy_creation_data *policy_creation_data;
	struct firewall_context *fw_ctx;
	struct creation_data *cdata;
	enum connman_session_state *session_state;

	DBG_PLUGIN(("policy_local_destroy: session %p", session));

	fw_ctx = g_hash_table_lookup(firewall_context_hash, session);
	if (fw_ctx)
	  g_hash_table_remove(firewall_context_hash, session);

	policy = g_hash_table_lookup(session_hash, session);
	if (policy){
	  g_hash_table_remove(session_hash, session);
	}
	policy_creation_data = g_hash_table_lookup(session_creation_hash, session);
	if (policy_creation_data){
	  cleanup_policy_creation_data(policy_creation_data);
	}
	cdata = g_hash_table_lookup(session_creation_hash, session);
	if (cdata){
	  g_hash_table_remove(session_creation_hash, session);
	}
	session_state = g_hash_table_lookup(session_state_hash, session);
	if (session_state){
	  g_hash_table_remove(session_state_hash, session);
	  g_free(session_state);
	}

}

struct connman_service* policy_local_get_service_for_session(struct connman_session *session, GSList *services)
{
	GSList *list, *list2, *service_iter;
	struct policy_config *policy = NULL;
	struct connman_service *service_lookup = NULL;
	guint no = 0;
	char *bearername = NULL;
	enum connman_service_type bearertype, servicetype;
	if (services == NULL){
	  DBG_PLUGIN(("policy_local_get_service_for_session: service list is NULL"));
	  goto err;
	}
	DBG_PLUGIN(("policy_local_get_service_for_session: session %p service list length %i",session, g_slist_length(services)));
	policy = g_hash_table_lookup(session_hash, session);
	if (!policy ){
	  DBG_PLUGIN(("policy_local_get_service_for_session: no policy"));
	  goto err;
	}
	if(!policy->group){
	  DBG_PLUGIN(("policy_local_get_service_for_session: policy has no group"));
	  goto err;
	}

	for (list = policy->group->allowed_bearers, list2 = policy->group->allowed_bearers_names ;
	     list && list2;
	     list= list->next, no++, list2= list2->next) { /* we ignore a modified allowed bearer list from session */
	  servicetype = GPOINTER_TO_INT(list->data);
	  DBG_PLUGIN(("policy_local_get_service_for_session: servicetype %i",servicetype));
	  if (servicetype == CONNMAN_SERVICE_TYPE_UNKNOWN) {
	    DBG_PLUGIN((" servicetype == CONNMAN_SERVICE_TYPE_UNKNOWN"));
	    bearername = (char*) list2->data;
	    if (!bearername){
	      DBG_PLUGIN((" bearername is NULL"));
	      continue;
	      }
	    DBG_PLUGIN((" bearername %s",bearername));
	    service_lookup = connman_service_lookup_from_identifier(bearername);
	    if (!service_lookup){
	      DBG_PLUGIN((" service_loockup failed. continue "));
	      continue;
	    }
	    INF_PLUGIN((" Got service %p", service_lookup));
	    for (service_iter = services; service_iter; service_iter = service_iter->next) {
	      if (service_lookup == (struct connman_service *) service_iter->data) {
		goto done;
	      }
	    }
	  } else {
	    DBG_PLUGIN((" service_type known service_type %i",servicetype));
	    for (service_iter = services; service_iter; service_iter = service_iter->next) {
	      service_lookup = (struct connman_service *) service_iter->data;
	      bearertype = connman_service_get_type(service_lookup);
	      DBG_PLUGIN((" bearertype %i",bearertype));
	      if (bearertype == servicetype) {
		goto done;
	      }
	    }
	  }
	}
 err:
	INF_PLUGIN((" no service found .. return NULL"));
	return (struct connman_service*) NULL;
 done:
	INF_PLUGIN(("Found service %p for session %p", service_lookup, session));
	policy->service_no = no;
	return service_lookup;
}


bool policy_local_service_allowed (struct connman_session *session,
			struct connman_service *service)
{
	/*
	This is called from session.c in case where a service reaches online / ready state.
	return true if:
		- session is offline and service allowed
		- session is online and service has higher prio?
			-> according to our architecture we change bearer only if session is reopened
	false:
		- session is already online with some service
		- service not allowed
	*/
	struct connman_service *service_allowed  = NULL;
	GSList *service_list = NULL;
	enum connman_session_state *session_state;

	if (!(service && session)){
	  DBG_PLUGIN(("service or session NULL: %p %p", service, session));
	  return FALSE;
	}
	session_state = g_hash_table_lookup(session_state_hash, session);
	if ( session_state == NULL )
	  DBG_PLUGIN(("old session state not found session: %p", session));
	else if (*session_state > CONNMAN_SESSION_STATE_DISCONNECTED){
	  DBG_PLUGIN(("session already connected. state %i session %p", *session_state,session));
	  return FALSE;
	} else
	  DBG_PLUGIN(("session is disconnected state %i session %p", *session_state,session ));
	service_list = g_slist_prepend(service_list,service);
	DBG_PLUGIN(("starting lookup for service %p", service));
	service_allowed = policy_local_get_service_for_session(session, service_list);
	if (!service_allowed){
	  DBG_PLUGIN(("got NULL from service lookup"));
	} else {
	  INF_PLUGIN(("got servie %p from lookup ", service_allowed));
	  if (service_allowed == service){
	    INF_PLUGIN(("service allowed "));
	    g_slist_free(service_list);
	    return TRUE;
	  }
	}
	g_slist_free(service_list);
	INF_PLUGIN(("service not allowed"));
	return FALSE;
}



static void del_source_ip_forwarding(struct connman_session *session)
{
	int err;
	struct policy_config *policy;
	char *ipt_cmd;
        unsigned int fwmark;
	GSList *ip_element;
	char *ip;
	DBG_PLUGIN(("Deleting sourceip mark"));
	//struct firewall_context *fw_ctx;
	fwmark = connman_session_firewall_get_fwmark(session);
	DBG_PLUGIN(("Got mark %i",fwmark));
	policy = g_hash_table_lookup(session_hash,session);
	if (!policy){
	  DBG_PLUGIN(("del_source_ip_forwarding: policy for session %p not found", session));
	  return;
	    }
	DBG_PLUGIN(("del_source_ip_forwarding: length of sip %i service no %i", g_slist_length(policy->group->source_ips), policy->service_no));
	if (policy->service_no == G_MAXUINT){
	  DBG_PLUGIN(("no service associated?"));
	  return;
	}
	ip_element = g_slist_nth(policy->group->source_ips, policy->service_no);
	if (ip_element->data == NULL){
	  DBG_PLUGIN(("del_source_ip_forwarding: ip_element->data is NULL. skip del_source_ip_forwarding"));
	  return;
	}
	ip = (char*) ip_element->data;
	DBG_PLUGIN(("del_source_ip_forwarding: ip: %s", ip));
	DBG_PLUGIN(("del_source_ip_forwarding: fwmark %i", fwmark));
	ipt_cmd = g_strdup_printf(IPTABLES_MARK_SIP_RM_SKELETON, ip, fwmark);
	DBG_PLUGIN(("del_source_ip_forwarding: iptables cmd sip rm: %s", ipt_cmd));
	err = system(ipt_cmd);
	g_free(ipt_cmd);
	if (err < 0) {
	  DBG_PLUGIN(("del_source_ip_forwarding: could not disable source IP routing rule err %i", err));
	  return;
	}
}

static void add_source_ip_forwarding(struct connman_session *session)
{
  //int id
	int err;
	char *ip;
	struct policy_config *policy;
	GSList *ip_element;
	unsigned int fwmark;
	//struct firewall_context *fw_ctx;
	char *ipt_cmd;
	DBG_PLUGIN(("add_source_ip_forwarding: Adding sourceip mark"));
	fwmark = connman_session_firewall_get_fwmark(session);
	//fw_ctx = g_hash_table_lookup(firewall_context_hash, session);
	DBG_PLUGIN(("Got mark %i",fwmark));
	policy = g_hash_table_lookup(session_hash, session);
	if(policy == NULL){
	  DBG_PLUGIN(("add_source_ip_forwarding: policy for session not found"));
	  return;
	}
	if(policy->service_no == G_MAXUINT ){
	  DBG_PLUGIN(("add_source_ip_forwarding: no service associated?"));
	  return;
	}
	ip_element = g_slist_nth(policy->group->source_ips, policy->service_no);
	ip = (char*) ip_element->data;
	ipt_cmd = g_strdup_printf(IPTABLES_MARK_SIP_ADD_SKELETON, ip, fwmark);
	DBG_PLUGIN(("iptables cmd sip add: %s", ipt_cmd));
	err = system(ipt_cmd);
	g_free(ipt_cmd);
	if (err < 0)
	{
	  ERR_PLUGIN(("could not add source IP routing rule err %i",err));
		return;
	}
}

void policy_local_update_session_state(struct connman_session *session, enum connman_session_state state)
{
	struct policy_config *pconfig = NULL;
	pconfig = g_hash_table_lookup(session_hash, session);
	enum connman_session_state *old_state = NULL;
	INF_PLUGIN(("policy_local_update_session_state: got session state update for session %p with config %p state %i" ,session, pconfig, state));
	if (!pconfig)
	  DBG_PLUGIN(("policy_local_update_session_state: policy_config for session %p not found", session));
	else {
		if (!pconfig->group)
			DBG_PLUGIN(("policy_local_update_session_state: policy_config->group is NULL session: %p ", session));
		else {
		  old_state = g_hash_table_lookup(session_state_hash, session);
		  if ( old_state == NULL ){
		    DBG_PLUGIN(("policy_local_update_session_state: old session state not found ... adding to hash"));
		    old_state = g_new0(enum connman_session_state, 1);
		  } else
		    DBG_PLUGIN(("policy_local_update_session_state: old_state found %i" , *old_state));
		  *old_state = state;
		  g_hash_table_replace(session_state_hash, session, old_state);
		  if (!pconfig->group->source_ips)
		    DBG_PLUGIN(("policy_local_update_session_state: Session without SIP routing"));
		  else {
		    DBG_PLUGIN(("policy_local_update_session_state: Session with SIP routing"));
		    switch (state) {
		    case CONNMAN_SESSION_STATE_CONNECTED:
		      INF_PLUGIN(("policy_local_update_session_state: Session with SIP routing connected"));
		      del_source_ip_forwarding(session);
		      add_source_ip_forwarding(session);
		      break;
		    case CONNMAN_SESSION_STATE_DISCONNECTED:
		      INF_PLUGIN(("policy_local_update_session_state: Session with SIP routing disconnected"));
		      del_source_ip_forwarding(session);
		      break;
                    default:
                      break;
		    }
		  }
		}
	}
}


static struct connman_session_policy session_policy_local = {
	.name = "BOSCH2.01 session local policy configuration",
	.priority = CONNMAN_SESSION_POLICY_PRIORITY_DEFAULT,
	.create = policy_local_create,
	.destroy = policy_local_destroy,
	.get_service_for_session = policy_local_get_service_for_session,
	.allowed = policy_local_service_allowed,
	.update_session_state = policy_local_update_session_state
};

static int load_keyfile(const char *pathname, GKeyFile **keyfile)
{
	GError *error = NULL;
	int err;
	struct stat st;
	long size;
	
	*keyfile = g_key_file_new();

	if (!g_key_file_load_from_file(*keyfile, pathname, 0, &error))
		goto err;

	stat(pathname, &st);
	size = st.st_size;
    DBG_PLUGIN(("policy file size is %li", size));
	if (size == 0 )
    {
      ERR_PLUGIN(("policy file has zero size"));
      goto err;
    }

	return 0;

err:
	/*
	 * The fancy G_FILE_ERROR_* codes are identical to the native
	 * error codes.
	 */
	err = -error->code;

	ERR_PLUGIN(("Unable to load policy file! %s: %s", pathname, error->message));
	g_clear_error(&error);
	connman_error("Failed to load session policy file");
	g_key_file_free(*keyfile);
	*keyfile = NULL;

	return err;
}

int connman_session_parse_source_ips(const char *token, GSList **list)
{
	*list = g_slist_append(*list, g_strdup(token));

	return 0;
}

static int bearer2service(const char *bearer, enum connman_service_type *type)
{
	if (g_strcmp0(bearer, "ethernet") == 0)
		*type = CONNMAN_SERVICE_TYPE_ETHERNET;
	else if (g_strcmp0(bearer, "wifi") == 0)
		*type = CONNMAN_SERVICE_TYPE_WIFI;
	else if (g_strcmp0(bearer, "gadget") == 0)
		*type = CONNMAN_SERVICE_TYPE_GADGET;
	else if (g_strcmp0(bearer, "bluetooth") == 0)
		*type = CONNMAN_SERVICE_TYPE_BLUETOOTH;
	else if (g_strcmp0(bearer, "cellular") == 0)
		*type = CONNMAN_SERVICE_TYPE_CELLULAR;
	else if (g_strcmp0(bearer, "vpn") == 0)
		*type = CONNMAN_SERVICE_TYPE_VPN;
	else
		*type = CONNMAN_SERVICE_TYPE_UNKNOWN;

	return 0;
}

int parse_bearers(const char *token, GSList **list)
{
	enum connman_service_type bearer;
	int err;

	if (g_strcmp0(token, "") == 0)
		return 0;

	if (g_strcmp0(token, "*") == 0) {
		add_default_bearer_types(list);
		return 0;
	}

	err = bearer2service(token, &bearer);
	if (err < 0)
		return err;

	*list = g_slist_append(*list, GINT_TO_POINTER(bearer));

	return 0;
}

static int load_policy(GKeyFile *keyfile, const char *groupname,
			struct policy_group *group)
{
	struct connman_session_config *config = group->config;
	char *str, **tokens;
	guint i; 
	int err = 0;
	INF_PLUGIN(("load_policy: groupname %s", groupname));
	group->selinux = g_key_file_get_string(keyfile, groupname,
						"selinux", NULL);

	group->gid = g_key_file_get_string(keyfile, groupname,
						"gid", NULL);


	group->uid = g_key_file_get_string(keyfile, groupname,
						"uid", NULL);
	INF_PLUGIN(("load_policy: uid: %s", group->uid));
	if (!group->selinux && !group->gid && !group->uid)
		return -EINVAL;

	group->userservice = g_key_file_get_string(keyfile, groupname,
			"Service", NULL);
	if (group->userservice != NULL) {
		if (group->uid  != NULL)
			group->userservice = g_strconcat("uid:",group->uid,"/",group->userservice, NULL);
		if (group->gid  != NULL)
			group->userservice = g_strconcat("gid:",group->gid,"/",group->userservice, NULL);
		if (group->selinux  != NULL)
			group->userservice = g_strconcat("selinux:",group->selinux,"/",group->userservice, NULL);
		INF_PLUGIN(("load_policy: final userservice : %s", group->userservice));
	}

	config->priority = g_key_file_get_boolean(keyfile, groupname,
						"Priority", NULL);

	str = g_key_file_get_string(keyfile, groupname, "RoamingPolicy",
				NULL);
	if (str) {
		config->roaming_policy = connman_session_parse_roaming_policy(str);
		g_free(str);
	}

	str = g_key_file_get_string(keyfile, groupname, "ConnectionType",
				NULL);
	if (str) {
		config->type = connman_session_parse_connection_type(str);
		g_free(str);
	}

	config->ecall = g_key_file_get_boolean(keyfile, groupname,
						"EmergencyCall", NULL);

	str = g_key_file_get_string(keyfile, groupname, "AllowedBearers",
				NULL);
	if (str) {
	  	INF_PLUGIN(("load_policy: AllowedBearers: %s", str));
		g_slist_free(config->allowed_bearers);
		config->allowed_bearers = NULL;
    g_strstrip(str);
		tokens = g_strsplit(str, " ", 0);

		for (i = 0; tokens[i]; i++) {
			err = parse_bearers(tokens[i],
					&config->allowed_bearers);
			if (err < 0)
				break;
			INF_PLUGIN(("load_policy: appending: %s", g_strdup(tokens[i]) ));
			group->allowed_bearers_names = g_slist_append(group->allowed_bearers_names, g_strdup(tokens[i]));
		}

		g_free(str);
		g_strfreev(tokens);
		DBG_PLUGIN(("load_policy: final length config->allowed_bearers: %u", g_slist_length(config->allowed_bearers)));
		DBG_PLUGIN(("load_policy: final length group->allowed_bearers_names: %u", g_slist_length(group->allowed_bearers_names)));

		group->allowed_bearers = g_slist_copy(config->allowed_bearers);
		DBG_PLUGIN(("load_policy: final length group->allowed_bearers: %u", g_slist_length(group->allowed_bearers)));
	}

	str = g_key_file_get_string(keyfile, groupname, "SourceIPs",
				NULL);
	if (str) {
	        INF_PLUGIN(("load_policy: SourceIPs  %s", str));
		guint noips;
		g_slist_free(group->source_ips);
		group->source_ips = NULL;
    g_strstrip(str);
		tokens = g_strsplit(str, " ", 0);
		noips = g_strv_length(tokens);

		if (noips != 1 && noips != g_slist_length(config->allowed_bearers))
		{
			g_free(str);
			g_strfreev(tokens);
			return -EINVAL;
		}

		for (i = 0; i < g_slist_length(config->allowed_bearers); i++) {
			if (tokens[i]) {
			  DBG_PLUGIN(("load_policy: SourceIPs token  %s",tokens[i] ));
			  err = connman_session_parse_source_ips(tokens[i],
						&group->source_ips);
			} else {
			  DBG_PLUGIN(("load_policy: SourceIPs token  %s", tokens[1] ));
			  err = connman_session_parse_source_ips(tokens[1],
						&group->source_ips);
			}
			if (err < 0)
				break;
		}

	 	g_free(str);
	 	g_strfreev(tokens);
	 }


	DBG_PLUGIN(("group %p selinux %s uid %s gid %s", group, group->selinux,
		group->uid, group->gid));

	return err;
}

static void update_session(struct policy_config *policy)
{
	DBG_PLUGIN(("update session policy %p session %p", policy, policy->session));

	if (!policy->session){
		DBG_PLUGIN(("policy has no associated session"));
		return;
	}
	if (connman_session_config_update(policy->session) < 0){
		DBG_PLUGIN(("return value of connman_session_config_update was negative. destroying session"));
		connman_session_destroy(policy->session);
	}
}

static void set_default_config(gpointer user_data)
{
	struct policy_config *policy = user_data;

	connman_session_set_default_config(policy->config);
	policy->group = NULL;
	update_session(policy);
}

static void cleanup_config(gpointer user_data)
{
	struct policy_config *policy = user_data;

	DBG_PLUGIN(("policy %p group %p", policy, policy->group));

	if (policy->group)
		policy->group->sessions = g_slist_remove(policy->group->sessions, policy);

	g_slist_free(policy->config->allowed_bearers);
	g_free(policy->config->id);
	g_free(policy->config);
	g_free(policy->selinux_context);
	g_free(policy->selinux);
	g_free(policy->uid);
	g_slist_free_full(policy->gids, g_free);
	g_free(policy);
}

static void cleanup_group(gpointer user_data)
{
	struct policy_group *group = user_data;

	DBG_PLUGIN(("group %p", group));

	g_slist_free_full(group->sessions, set_default_config);

	g_slist_free(group->config->allowed_bearers);
	g_free(group->config->id);
	g_free(group->config);
	if (group->selinux)
		g_hash_table_remove(selinux_hash, group->selinux);
	if (group->uid)
		g_hash_table_remove(uid_hash, group->uid);
	if (group->gid)
		g_hash_table_remove(gid_hash, group->gid);
	if (group->userservice)
		g_hash_table_remove(userservice_hash, group->userservice);

	if (group->source_ips)
		g_slist_free(group->source_ips);

	if (group->allowed_bearers_names)
	{
		g_slist_free_full(group->allowed_bearers_names,g_free);
	}
	if (group->allowed_bearers)
		g_slist_free(group->allowed_bearers);

	g_free(group->selinux);
	g_free(group->uid);
	g_free(group->gid);
	g_free(group->userservice);
	g_free(group);
}

static void cleanup_file(gpointer user_data)
{
	struct policy_file *file = user_data;

	DBG_PLUGIN(("file %p", file));

	g_slist_free_full(file->groups, cleanup_group);
	g_free(file);
}

static void recheck_sessions(void)
{
	GHashTableIter iter;
	gpointer value, key;
	struct policy_group *group = NULL;
	GSList *list;

	g_hash_table_iter_init(&iter, session_hash);
	while (g_hash_table_iter_next(&iter, &key, &value)) {
		struct policy_config *policy = value;

		if (policy->group)
			continue;

		if (policy->selinux)
			group = g_hash_table_lookup(selinux_hash,
							policy->selinux);
		if (group) {
			policy->config->id_type = CONNMAN_SESSION_ID_TYPE_LSM;
			g_free(policy->config->id);
			policy->config->id = g_strdup(policy->selinux_context);
			update_session(policy);
			continue;
		}

		group = g_hash_table_lookup(uid_hash, policy->uid);
		if (group) {
			set_policy(policy, group);

			policy->config->id_type = CONNMAN_SESSION_ID_TYPE_UID;
			g_free(policy->config->id);
			policy->config->id = g_strdup(policy->uid);
			update_session(policy);
			continue;
		}

		for (list = policy->gids; list; list = list->next) {
			char *gid = list->data;
			group = g_hash_table_lookup(gid_hash, gid);
			if (group) {
				set_policy(policy, group);

				policy->config->id_type = CONNMAN_SESSION_ID_TYPE_GID;
				g_free(policy->config->id);
				policy->config->id = g_strdup(gid);
				update_session(policy);
			}
		}
	}
}

static int load_file(const char *filename, struct policy_file *file)
{
	GKeyFile *keyfile;
	struct policy_group *group;
	char **groupnames;
	char *pathname;
	int err = 0, i;

	DBG_PLUGIN(("load_file: %s", filename));

	pathname = g_strdup_printf("%s/%s", POLICYDIR, filename);
	err = load_keyfile(pathname, &keyfile);
	g_free(pathname);

	if (err < 0)
		return err;

	groupnames = g_key_file_get_groups(keyfile, NULL);

	for (i = 0; groupnames[i]; i++) {
		group = g_new0(struct policy_group, 1);
		group->config = connman_session_create_default_config();

		err = load_policy(keyfile, groupnames[i], group);
		if (err < 0) {
		  ERR_PLUGIN(("Failed to load policy file: error %i",err));
		  g_free(group->config);
		  g_free(group);
		  break;
		}
		if (group->selinux)
			g_hash_table_replace(selinux_hash, group->selinux, group);

		if (group->uid)
			g_hash_table_replace(uid_hash, group->uid, group);

		if (group->gid)
			g_hash_table_replace(gid_hash, group->gid, group);

		if (group->userservice){
		  DBG_PLUGIN(("load_file: storing group for service %s", group->userservice));
		  g_hash_table_replace(userservice_hash, group->userservice, group);
		}

		file->groups = g_slist_prepend(file->groups, group);
	}

	g_strfreev(groupnames);

	if (err < 0)
		g_slist_free_full(file->groups, cleanup_group);

	g_key_file_free(keyfile);

	return err;
}

static bool is_filename_valid(const char *filename)
{
	if (!filename)
		return false;

	if (filename[0] == '.')
		return false;

	return g_str_has_suffix(filename, ".policy");
}

static int read_policies(void)
{
	GDir *dir;
	const gchar *filename;
	struct policy_file *file;

	DBG_PLUGIN(("Reading policy file : %s", POLICYDIR));

	dir = g_dir_open(POLICYDIR, 0, NULL);
	if (!dir)
		return -EINVAL;

	while ((filename = g_dir_read_name(dir))) {
		if (!is_filename_valid(filename))
			continue;

		file = g_new0(struct policy_file, 1);
		if (load_file(filename, file) < 0) {
			g_free(file);
			continue;
		}

		g_hash_table_replace(file_hash, g_strdup(filename), file);
	}

	g_dir_close(dir);

	return 0;
}


static void notify_handler(struct inotify_event *event,
                                        const char *filename)
{
	struct policy_file *file;

	DBG_PLUGIN(("event %x file %s", event->mask, filename));

	if (event->mask & IN_CREATE)
		return;

	if (!is_filename_valid(filename))
		return;

	/*
	 * load_file() will modify the global selinux/uid/gid hash
	 * tables. We need to remove the old entries first before
	 * else the table points to the wrong entries.
	 */
	g_hash_table_remove(file_hash, filename);

	if (event->mask & (IN_DELETE | IN_MOVED_FROM))
		return;

	if (event->mask & (IN_MOVED_TO | IN_MODIFY)) {
		connman_info("Policy update for '%s'", filename);

		file = g_new0(struct policy_file, 1);
		if (load_file(filename, file) < 0) {
			g_free(file);
			return;
		}

		g_hash_table_replace(file_hash, g_strdup(filename), file);
		recheck_sessions();
	}
}

static int session_policy_local_init(void)
{
	int err;
#ifdef LOG_DLT
  DLT_REGISTER_APP(DLT_APPID_CONNMANPLUGIN, "Connman session registration for Logging");
  /* register context */
  DLT_REGISTER_CONTEXT(CMP1, "CMP1", "CONNMAN sesssion context for Logging");
#endif

	DBG_PLUGIN(("session_policy_local_init"));

	/* If the dir doesn't exist, create it */
	if (!g_file_test(POLICYDIR, G_FILE_TEST_IS_DIR)) {
		if (mkdir(POLICYDIR, MODE) < 0) {
			if (errno != EEXIST)
			{
				ERR_PLUGIN(("failed to create %s", POLICYDIR));
				return -errno;
			}
		}
	}

	connection = connman_dbus_get_connection();
	if (!connection)
		return -EIO;

	file_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
					g_free, cleanup_file);
	session_hash = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						NULL, cleanup_config);
	session_creation_hash = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						NULL, NULL);
	firewall_context_hash = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						NULL, NULL);
	selinux_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
					NULL, NULL);
	uid_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
					NULL, NULL);
	gid_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
					NULL, NULL);
	userservice_hash = g_hash_table_new_full(g_str_hash, g_str_equal,
					NULL, NULL);
	session_state_hash = g_hash_table_new_full(g_direct_hash, g_str_equal,
					NULL, NULL);

	err = connman_inotify_register(POLICYDIR, notify_handler);
	if (err < 0)
		goto err;

	err = connman_session_policy_register(&session_policy_local);
	if (err < 0)
		goto err_notify;

	default_group = g_new0(struct policy_group, 1);
	default_group->config = connman_session_create_default_config();

	g_slist_free(default_group->config->allowed_bearers);
	default_group->config->allowed_bearers = NULL;

	read_policies();

	return 0;

err_notify:

	connman_inotify_unregister(POLICYDIR, notify_handler);

err:
	if (file_hash)
		g_hash_table_destroy(file_hash);

	if (session_hash)
		g_hash_table_destroy(session_hash);

	if (session_creation_hash)
		g_hash_table_destroy(session_creation_hash);

	if (firewall_context_hash)
		g_hash_table_destroy(firewall_context_hash);

	if (selinux_hash)
		g_hash_table_destroy(selinux_hash);

	if (uid_hash)
		g_hash_table_destroy(uid_hash);

	if (gid_hash)
		g_hash_table_destroy(gid_hash);

	if (userservice_hash)
		g_hash_table_destroy(userservice_hash);

	if (session_state_hash)
	  g_hash_table_destroy(session_state_hash);

	connman_session_policy_unregister(&session_policy_local);

	dbus_connection_unref(connection);

	return err;
}

static void session_policy_local_exit(void)
{
	DBG_PLUGIN(("session_policy_local_exit"));

	g_hash_table_destroy(file_hash);
	g_hash_table_destroy(session_hash);
	g_hash_table_destroy(firewall_context_hash);
	g_hash_table_destroy(session_creation_hash);
	g_hash_table_destroy(selinux_hash);
	g_hash_table_destroy(uid_hash);
	g_hash_table_destroy(gid_hash);
	g_hash_table_destroy(userservice_hash);
	g_hash_table_destroy(session_state_hash);

	connman_session_policy_unregister(&session_policy_local);

	dbus_connection_unref(connection);

	connman_inotify_unregister(POLICYDIR, notify_handler);
}

CONNMAN_PLUGIN_DEFINE(session_policy_local,
		"BOSCH session local file policy configuration plugin",
		VERSION, CONNMAN_PLUGIN_PRIORITY_DEFAULT,
		session_policy_local_init, session_policy_local_exit)
