/*
 * aud-dbus-connector.cpp
 *
 *  Created on: Mar 9, 2016
 *      Author: rjk2kor
 */

#include <poll.h>
#include "aud_dbus_connector.h"

#include "controllerplugin_Trace.h"
#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCONTROLLERPLUGIN_DIAGNOSIS
#include "trcGenProj/Header/aud_dbus_connector.cpp.trc.h"
#endif

/**************************************************************************************
 * IMPLEMENTATION OF CLASS : aud_ml_dbus_timer_connector
 **************************************************************************************/
/**
   Constructor of the class aud_ml_dbus_timer_connector

   @param[in] bus_conn Pointer to opaque DBusConnection object
   @param[in] pSockHndl Pointer to Socket Handler of Genivi Audio Manager
   @return None
*/
aud_ml_dbus_timer_connector::aud_ml_dbus_timer_connector(DBusConnection* poBusConn, am::CAmSocketHandler* pSockHndl)
:m_poMainLoop(pSockHndl)
{
  ///Use pointer to dbus connection object to setup timer functions for the provided connection
  if(poBusConn)
  {
    if(!dbus_connection_set_timeout_functions(poBusConn,bAdd, vRemove, vToggle, this, 0))
    {
      ETG_TRACE_ERR(("CMD_AUDIO, !!!! E R R O R >>>> Failed to set timeout functions !!!! "));
    }
  }
}
/**
   Destructor for the class aud_ml_dbus_timer_connector
*/
aud_ml_dbus_timer_connector::~aud_ml_dbus_timer_connector()
{
  m_poMainLoop = NULL;
}
/**
   This function would be called by libdbus to requesting to add a timer to its mainloop and poll it on its behalf

   @param[in] pTimeout Pointer to DBusTimeout Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
   @return True if timer added successfully to mainloop, else False
*/
dbus_bool_t  aud_ml_dbus_timer_connector::bAdd(DBusTimeout *pTimeout, void* pvUserData)
{
  ETG_TRACE_USR4(("Request to add timer : %x",pTimeout));
  aud_ml_dbus_timer_connector* pSelf = static_cast<aud_ml_dbus_timer_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    ///It is a bug to process a timeout request from libdbus when it is not enabled
    if(dbus_timeout_get_enabled(pTimeout))
    {
      ///Creating a dynamically allocated handle
      am::sh_timerHandle_t* ptrHandle = new am::sh_timerHandle_t;
      if(ptrHandle == NULL)
        return FALSE;
      ///Now that timeout is enabled, extact the timeout. Note: Timeout is specified in milliseconds
      int timeout_ms = dbus_timeout_get_interval(pTimeout);

      ///Split the timeout in milliseconds to seconds and nano seconds, as expected by timespec
      timespec ts;
      ts.tv_sec = timeout_ms/1000;//Seconds
      ts.tv_nsec = (timeout_ms%1000)* 1000000;//Nano seconds

      ///Add the timer to main loop, and extract the handle
      pSelf->m_poMainLoop->addTimer(ts,pSelf,*ptrHandle,pTimeout);

      ETG_TRACE_USR4(("Added timeout :%d, with handle : %d",timeout_ms,*ptrHandle));

      ///Save the handle with timeout message
      //Inform lint that custody of the pointer ptrHandle, argument number 2 is taken care by dbus_timeout_set_data function
      //lint -sem(dbus_timeout_set_data,custodial(2))
      dbus_timeout_set_data(pTimeout,ptrHandle,NULL);
      return TRUE;
    }
    else
    {
      ETG_TRACE_ERR(("CMD_AUDIO, !!!! E R R O R >>>> Timeout not enabled !!!! "));

    }
  }
  return TRUE;
}
/**
   This function would be called by libdbus to remove a previously added timer to from mainloop

   @param[in] pTimeout Pointer to DBusTimeout Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
*/
void  aud_ml_dbus_timer_connector::vRemove(DBusTimeout *pTimeout, void* pvUserData)
{
  ETG_TRACE_USR4(("Remove timeout called : %x",pTimeout));
  aud_ml_dbus_timer_connector* pSelf = static_cast<aud_ml_dbus_timer_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    ///Extract the handle, which we previously stored, from timeout message
    am::sh_timerHandle_t* ptrHandle = (am::sh_timerHandle_t*)dbus_timeout_get_data(pTimeout);

    if(ptrHandle)
    {
      //Remove the timer from main loop
      pSelf->m_poMainLoop->removeTimer(*ptrHandle);
      ETG_TRACE_USR4(("Removed timeout with handle : %d",*ptrHandle));

      delete ptrHandle;///Free memory associated with the handle
    }
  }
}
/**
   This function would be called by libdbus to modify a previous timer which was added to mainloop

   @param[in] pTimeout Pointer to DBusTimeout Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
*/
void aud_ml_dbus_timer_connector::vToggle(DBusTimeout *pTimeout, void* pvUserData)
{
  ETG_TRACE_USR4(("Toggle timeout called: %x",pTimeout));
  aud_ml_dbus_timer_connector* pSelf = static_cast<aud_ml_dbus_timer_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    ///Extract the handle, which we previously stored, from timeout message
    am::sh_timerHandle_t* ptrHandle = static_cast<am::sh_timerHandle_t*>(dbus_timeout_get_data(pTimeout));

    if(ptrHandle)
    {
      ///See if the timer has to be enabled or disabled
      if(dbus_timeout_get_enabled(pTimeout))
      {
        ///Now that timeout is enabled, extact the timeout. Note: Timeout is specified in milliseconds
        int timeout_ms = dbus_timeout_get_interval(pTimeout);

        timespec ts;
        ts.tv_sec = timeout_ms/1000;//Seconds
        ts.tv_nsec = (timeout_ms%1000)* 1000000;//Nano seconds

        ///Update the timer values to the mainloop
        pSelf->m_poMainLoop->updateTimer(*ptrHandle,ts);
        ETG_TRACE_USR4(("Updated timer with handle: %d to %d milliseconds",*ptrHandle,timeout_ms));
      }
      else
      {
        ///Timer has to be disabled
        pSelf->m_poMainLoop->stopTimer(*ptrHandle);
        ETG_TRACE_USR4(("Stopped timer with handle: %d ",*ptrHandle));
      }
    }
  }
}
/**
   This function would be called by mainloop to inform that the timer has expired.
   Once the timer is expired, libdbus has to be informed about it using the function dbus_timeout_handle function

   @param[in] handle Handle to the timer object previously created
   @param[in] pvUserData Pointer to userdata provided previously
*/
void aud_ml_dbus_timer_connector::Call(am::sh_timerHandle_t handle, void* pvuserdata)
{
  ETG_TRACE_USR4(("Timer expired with handle: %d",handle));
  if(pvuserdata && m_poMainLoop)
  {
    ///Inform time out to libdbus
    dbus_timeout_handle((DBusTimeout*)pvuserdata);
    ETG_TRACE_USR4(("Timer restarted with handle: %d",handle));
    ///Restart the timer, as libdbus expects the timers to be continuous. So timer would run till libdbus asks us to stop.
    m_poMainLoop->restartTimer(handle);
  }
}
/**************************************************************************************
 * IMPLEMENTATION OF CLASS : aud_ml_dbus_watch_connector
 **************************************************************************************/
/**
   Constructor of the class aud_ml_dbus_watch_connector

   @param[in] ptrConnection Pointer to opaque DBusConnection object
   @param[in] pSockHndl Pointer to Socket Handler of Genivi Audio Manager
   @return None
*/
aud_ml_dbus_watch_connector::aud_ml_dbus_watch_connector(DBusConnection* ptrConnection, am::CAmSocketHandler* pSockHndl)
:m_poMainLoop(pSockHndl)
,m_poConnection(ptrConnection)
,m_oCheck(this,&aud_ml_dbus_watch_connector::bCheck)
,m_oFire(this,&aud_ml_dbus_watch_connector::vFire)
,m_oDispatch(this,&aud_ml_dbus_watch_connector::bDispatch)
{
  ///Use pointer to dbus connection object to setup timer functions for the provided connection
  if(ptrConnection)
  {
    if(!dbus_connection_set_watch_functions(ptrConnection,bAdd, vRemove, vToggle, this, 0))
    {
      ETG_TRACE_FATAL(("!!!! E R R O R >>>> Failed to set watch functions !!!! "));
    }
  }
}
/**
   Destructor for the class aud_ml_dbus_watch_connector
*/
aud_ml_dbus_watch_connector::~aud_ml_dbus_watch_connector()
{
  m_poMainLoop = NULL;
  m_poConnection = NULL;
}
/**
   This function would be called by libdbus to requesting to add a watch to its mainloop and poll it on its behalf

   @param[in] watch Pointer to DBusTimeout Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
   @return True if watch added successfully to mainloop, else False
*/
dbus_bool_t  aud_ml_dbus_watch_connector::bAdd(DBusWatch *watch, void* pvUserData)
{
  ETG_TRACE_USR4(("bAddWatch Requested, with handle: %x",watch));
  if(watch == NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, bAddWatch Requested, Watch pointer is NULL !!!!"));
    return FALSE;
  }
  aud_ml_dbus_watch_connector* pSelf = static_cast<aud_ml_dbus_watch_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    ///Check if the watch is enabled, is not mandatory, as whnen a watch is added it many
    if(dbus_watch_get_enabled(watch))
    {
      int16_t l_events = 0;
      ///It is a bug to process a timeout request from libdbus when it is not enabled
      uint flags = dbus_watch_get_flags(watch);

      am::sh_pollHandle_t* poHandle = new am::sh_pollHandle_t;
      if(poHandle == NULL)
        return FALSE;

      if(flags & DBUS_WATCH_READABLE)
      {
        ETG_TRACE_USR4(("Watch is readable"));
        l_events |=POLLIN;
      }
      if(flags & DBUS_WATCH_WRITABLE)
      {
        ETG_TRACE_USR4(("Watch is writable"));
        l_events |=POLLOUT;
      }
      ETG_TRACE_USR4(("Adding watch with flags > %x",l_events));
      ///Now add the file descriptor to mainloop, with the requested fields
      pSelf->m_poMainLoop->addFDPoll(  dbus_watch_get_unix_fd(watch),//File descriptor to watch
                    l_events,//Event flags
                    NULL,//Pointer to prepare callback
                    &pSelf->m_oFire,//Pointer to Fire callback
                    &pSelf->m_oCheck,//Pointer to check callback
                    &pSelf->m_oDispatch,//Pointer to dispatch callback
                    watch,//Userdata
                    *poHandle);//Handle
      ETG_TRACE_USR4(("bAddWatch, added watch with handle: %d",*poHandle));
      ///Store the main loop handle with libdbus
      //Inform lint that custody of the pointer poHandle, argument number 2 is taken care by dbus_watch_set_data function
      //lint -sem(dbus_watch_set_data,custodial(2))
      dbus_watch_set_data(watch,poHandle,NULL);
      return TRUE;
    }
    else
    {
      ETG_TRACE_USR4(("bAddWatch, watch is not enabled : %x",watch));
      return TRUE;
    }
  }
  return TRUE;
}
/**
   This function would be called by libdbus to requesting to remove the watch previously added to mainloop

   @param[in] watch Pointer to DBusWatch Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
*/
void aud_ml_dbus_watch_connector::vRemove(DBusWatch *watch, void* pvUserData)
{
  ETG_TRACE_USR4(("vRemove Watch Requested : %x",watch));
  aud_ml_dbus_watch_connector* pSelf = static_cast<aud_ml_dbus_watch_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    //Extract the handle information from it
    am::sh_pollHandle_t* poHandle = (am::sh_pollHandle_t*)dbus_watch_get_data(watch);
    if(poHandle)
    {
      ETG_TRACE_USR4(("Removed Watch with handle : %d",*poHandle));
      ///Remove the file descriptor from the poll of mainloop
      pSelf->m_poMainLoop->removeFDPoll(*poHandle);
      delete poHandle;///Free resources allocated to handle
    }
  }

}
/**
   This function would be called by libdbus to requesting to modify the previously added watch to the mainloop

   @param[in] watch Pointer to DBusWatch Object
   @param[in] pvUserData Pointer to userdata provided, which was provided while setting timeout functions using
              dbus_connection_set_timeout_functions()
*/
void aud_ml_dbus_watch_connector::vToggle(DBusWatch* watch, void *pvUserData)
{
  ETG_TRACE_USR4(("vToggle Watch Requested : %x",watch));
  aud_ml_dbus_watch_connector* pSelf = static_cast<aud_ml_dbus_watch_connector*>(pvUserData);
  if(pSelf && pSelf->m_poMainLoop)
  {
    ///Extract the handle information from it
    am::sh_pollHandle_t* poHandle = (am::sh_pollHandle_t*)dbus_watch_get_data(watch);
    if(poHandle)
    {
      int16_t l_events = 0;
      ///It is a bug to process a timeout request from libdbus when it is not enabled
      uint flags = dbus_watch_get_flags(watch);

      ///If the watch is enabled, then get the latest event fields and update it to the mainloop
      if(dbus_watch_get_enabled(watch))
      {
        if(flags & DBUS_WATCH_READABLE)
        {
          l_events |=POLLIN;
        }
        if(flags & DBUS_WATCH_WRITABLE)
        {
          l_events |=POLLOUT;
        }
      }
      pSelf->m_poMainLoop->updateEventFlags(*poHandle,l_events);
    }
    else
    {
      ETG_TRACE_USR4(("vToggle handle is NULL  : %x",watch));
    }
  }
}
/**
   This function would be called by mainloop of genivi audio manager
   This function is called by mainloop to check if conditions that triggered to dispatch still exist.

   @param[in] handle Handle from the mainloop. This is the same handle that we have received while adding watch to mainloop.
   @param[in] pvuserdata Pointer to userdata provided previously
   @return TRUE, if the condition to process data still exists, else FALSE
*/
bool aud_ml_dbus_watch_connector::bCheck(const am::sh_pollHandle_t /*handle*/, void* /*pvuserdata*/)
{
  //std::cout<<"\ndbus connection dispatch status : "<< dbus_connection_get_dispatch_status(m_poConnection);
  //return (dbus_connection_get_dispatch_status(m_poConnection) != DBUS_DISPATCH_DATA_REMAINS);
  return true;
}
/**
   This function would be called by mainloop of genivi audio manager
   This function is called by mainloop when the previously added file descriptor has fired.
   In this function, we are supposed to read the data from the file descriptor. We achieve it by calling dbus_watch_handle function.

   @param[in] l_pfd object of the pollfd structure, which corresponds to the file descriptor of the watch.
   @param[in] handle Handle from the mainloop. This is the same handle that we have received while adding watch to mainloop.
   @param[in] pvuserdata Pointer to userdata provided previously
*/
void aud_ml_dbus_watch_connector::vFire(const pollfd l_pfd, const am::sh_pollHandle_t handle, void* pvuserdata)
{
  ETG_TRACE_USR4(("vFire Watch Fired, Handle : %d",handle));
  if(pvuserdata)
  {
    //Extract the received flags and modify them in such a way that libdbus can understand it
    uint flags = 0;

    if(l_pfd.revents & POLLIN)
    {
      flags |= DBUS_WATCH_READABLE;
    }
    if(l_pfd.revents & POLLOUT)
    {
      flags |= DBUS_WATCH_WRITABLE;
    }
    if(l_pfd.revents & POLLERR)
    {
      flags |= DBUS_WATCH_ERROR;
    }
    if(l_pfd.revents & POLLHUP)
    {
      flags |= DBUS_WATCH_HANGUP;
    }
    DBusWatch* pwatch = (DBusWatch*) pvuserdata;

    ETG_TRACE_USR4(("vFire Triggered for watch : %x",pwatch));

    ///In general, when dbus_watch_handle is called, libdbus reads the raw message from the message queue
    ///Here the socket/unix file descriptor that we are polling got fired. This means that there is some data (or may be an error) on it.
    ///Inform libdbus, so that it can read the data from it (or can handle the error)
    dbus_watch_handle(pwatch,flags);
  }
}
/**
   This function would be called by mainloop of genivi audio manager
   This function is called by mainloop after fire has been called.
   Here we are supposed dispatch the data previously read from the file descriptor (using dbus_watch_handle).
   We achieve the dispatching part by invoking the function dbus_connection_dispatch function.

   @param[in] handle Handle from the mainloop. This is the same handle that we have received while adding watch to mainloop.
   @param[in] pvUserData Pointer to userdata provided previously
   @return TRUE, if there is still some data left to dispatch (i.e we have processed only 1 message, where as the queue consists of 10 messages),
   else FALSE, when we have processed all the messages in the message queue.
*/
bool  aud_ml_dbus_watch_connector::bDispatch(const am::sh_pollHandle_t, void* /*pvUserData*/)
{
  bool bretval = false;
  if(m_poConnection)
  {
    if(dbus_connection_dispatch(m_poConnection) == DBUS_DISPATCH_COMPLETE)
    {
      ///There is nothing left to dispatch
      bretval = false;
    }
    else
    {
      bretval = true;
    }
  }
  ETG_TRACE_USR4(("bDispatch returns : %d",bretval));
  return bretval;
}


