/************************************************************************
 * FILE:         CAmControlCvm.cpp
 * PROJECT:      G3g
 * SW-COMPONENT: Genivi Audio Manager Controller Plugin
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Class to handle System voltage level handling.
 *
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2014 RBEI
 * HISTORY:
 * Date        | Author                 | Modification
 * 26.02.2014  | RBEI/ECV Ranjit Susal  | initial version
 *
 *************************************************************************/

//include to have dlt logs
//Class header
#include "CAmControlCvm.h"
#include "CAmControlSenderBase.h"

#include "controllerplugin_Trace.h"
#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCONTROLLERPLUGIN
#include "trcGenProj/Header/CAmControlCvm.cpp.trc.h"
#endif

using namespace am;

/*******************************************************************************
                  Specific defines for this component
 *******************************************************************************/

#define CAM_CONTROLLERCVM_C_U32_EVENT_MASK_STOP_THREAD   0x01000000

#define CAM_CONTROLLERCVM_C_U32_EVENT_MASK_ALL \
    (DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY | \
       DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE | \
     DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE  | \
        CAM_CONTROLLERCVM_C_U32_EVENT_MASK_STOP_THREAD)

/************************************************************************
 *FUNCTION     : CAmControlCvm
 *DESCRIPTION  : Constructor
 *PARAMETER    : CAmControlSenderBase*
 *RETURN VALUE :
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
CAmControlCvm::CAmControlCvm(CAmControlSenderBase* pControlSenderBase,am::IAmControlReceiverShadow* pControlReceiver) : m_poControlSenderBase(pControlSenderBase)
      ,m_poControlReceiverShadow(pControlReceiver)
      ,m_IODeviceVoltDescr((tS32)OSAL_C_INVALID_HANDLE)
      ,m_EventHandle(OSAL_C_INVALID_HANDLE)
      ,m_ThreadIdCvmEvent(OSAL_ERROR)
      ,m_u32ClientId(0)
      ,m_u32PreviousSystemVoltage(DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING)

{
  ETG_TRACE_USR3(("CAmControlCvm::Entered in constructor CAmControlCvm()"));

  DEV_VOLT_trClientRegistration        rClientRegistration = {0,""};


  //open device
  m_IODeviceVoltDescr = OSAL_IOOpen(OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE);
  if (OSAL_ERROR == m_IODeviceVoltDescr) {
     ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::CAmControlCvm(): Failed to open /dev/volt."));
  } else {
    if (OSAL_s32IOControl(m_IODeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_REGISTER_CLIENT, (intptr_t)&rClientRegistration) == OSAL_OK)  {
      m_u32ClientId = rClientRegistration.u32ClientId;
      if (OSAL_s32EventOpen(rClientRegistration.szNotificationEventName, &m_EventHandle) != OSAL_OK)  {
          ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::CAmControlCvm(): Failed to open handle."));
      }
      vStartVoltageMonitoring();
    }
    else {
        ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::CAmControlCvm(): Failed to register for /dev/volt."));
    }
  }
}

/************************************************************************
 *FUNCTION     : ~CAmControlCvm
 *DESCRIPTION  : Destructor
 *PARAMETER    : CAmControlSenderBase*
 *RETURN VALUE :
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
CAmControlCvm::~CAmControlCvm() {

  ETG_TRACE_USR3(("CAmControlCvm::Entered in destructor CAmControlCvm()"));
  if (m_IODeviceVoltDescr != (tS32)OSAL_C_INVALID_HANDLE) {
    if (OSAL_s32IOControl(m_IODeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_CLIENT, (intptr_t)m_u32ClientId) == OSAL_ERROR)  {
        ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::~CAmControlCvm(): Cannot unregister OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_CLIENT."));
    }

    if ( (m_ThreadIdCvmEvent != OSAL_ERROR) && (m_EventHandle != OSAL_C_INVALID_HANDLE)) {
      //thread is running --> stop now
       ETG_TRACE_USR4(("CAmControlCvm::~CAmControlCvm(): Send event to stop thread"));

      if (OSAL_s32EventPost(m_EventHandle, CAM_CONTROLLERCVM_C_U32_EVENT_MASK_STOP_THREAD, OSAL_EN_EVENTMASK_OR) == OSAL_OK)  {
        ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::~CAmControlCvm(): Cannot post osal event."));
        OSAL_s32ThreadDelete(m_ThreadIdCvmEvent); // delete the thread.
      }
    }
  }

  if(NULL != m_poControlSenderBase)
    m_poControlSenderBase = NULL;
  if(NULL != m_poControlReceiverShadow)
    m_poControlReceiverShadow = NULL;
}

/************************************************************************
 *FUNCTION     : vStartVoltageMonitoring
 *DESCRIPTION  : Cvm worker thread will be created to monitor system
 *         voltage.
 *PARAMETER    :
 *RETURN VALUE :
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vStartVoltageMonitoring()
{
  ETG_TRACE_USR4(("CAmControlCvm::vStartVoltageMonitoring entered"));
  if (m_IODeviceVoltDescr != (tS32)OSAL_C_INVALID_HANDLE) {
    OSAL_trThreadAttribute  rAttr;
    tC8                     szThreadName[] = "CAmControlCvmVolt";

    rAttr.szName = szThreadName;
    rAttr.s32StackSize = 4096;
    rAttr.u32Priority = OSAL_C_U32_THREAD_PRIORITY_HIGHEST;
    rAttr.pfEntry = (OSAL_tpfThreadEntry)vCamCCvmWorkerThread;
    rAttr.pvArg = (tPVoid)this;

    m_ThreadIdCvmEvent = OSAL_ThreadSpawn(&rAttr);

    if ( m_ThreadIdCvmEvent == OSAL_ERROR ) {
        ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::vStartVoltageMonitoring(): Failed to spawn thread!"));
    }
  }
}

/************************************************************************
 *FUNCTION     : vCamCCvmWorkerThread
 *DESCRIPTION  : Register for system voltage notification and wait event
 *               for voltage change notify.
 *PARAMETER    : pointer object of a class
 *RETURN VALUE :
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vCamCCvmWorkerThread(tVoid *pvArg)
{
  ETG_TRACE_USR4(("CAmControlCvm::vCamCCvmWorkerThread entered"));
  tBool             bStopThread            = FALSE;
  OSAL_tEventMask   rEventMaskResult       = 0;

  CAmControlCvm* poThis               = (CAmControlCvm*) pvArg;

  DEV_VOLT_trSystemVoltageRegistration rSystemVoltageRegistration;

  rSystemVoltageRegistration.u32ClientId              = poThis->m_u32ClientId;
  rSystemVoltageRegistration.u32VoltageIndicationMask = (DEV_VOLT_C_U32_BIT_MASK_INDICATE_LOW_VOLTAGE     |
      DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_LOW_VOLTAGE  |
      DEV_VOLT_C_U32_BIT_MASK_INDICATE_HIGH_VOLTAGE          |
      DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_HIGH_VOLTAGE  );

  //Register for System Voltage Notification
  if (OSAL_s32IOControl(poThis->m_IODeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_REGISTER_SYSTEM_VOLTAGE_NOTIFICATION, (intptr_t)&rSystemVoltageRegistration) != OSAL_OK) {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::vCamCCvmWorkerThread(): Failed to register system notification."));
  }

  while (FALSE == bStopThread)  {

    rEventMaskResult = 0;
    if (OSAL_s32EventWait(poThis->m_EventHandle, CAM_CONTROLLERCVM_C_U32_EVENT_MASK_ALL, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &rEventMaskResult) == OSAL_OK) {

      if (OSAL_s32EventPost(poThis->m_EventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND) == OSAL_OK) {  // Clear evaluated event bits
        ETG_TRACE_USR4(("CAmControlCvm::vCamCCvmWorkerThread(): Voltage notification received= %d", (uint32_t)rEventMaskResult));

        if (rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY) {
            ETG_TRACE_USR4(("CAmControlCvm::vCamCCvmWorkerThread(): Voltage level changed -->[VVD] new system event --> DEV_VOLT_C_U32_EVENT_MASK_SYSTEM_VOLTAGE_CHANGED_NOTIFY."));

            poThis->vSystemVoltageNotification();
        }
        if(rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE) //vvd
        {
          ETG_TRACE_USR4(("CAmControlCvm::vCamCCvmWorkerThread(): Voltage level changed permanent --> new system event --> DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE."));
          poThis->vNotifyCvmEvent(GAM_PERMANENT_HIGH_VOLTAGE_START);
        }
        if (rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE) //vvd
        {
            ETG_TRACE_USR4(("CAmControlCvm::vCamCCvmWorkerThread(): Voltage level changed permanent --> new system event --> DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE."));
            poThis->vNotifyCvmEvent(GAM_PERMANENT_CRITICAL_HIGH_VOLTAGE_START);
        }
        if (rEventMaskResult & CAM_CONTROLLERCVM_C_U32_EVENT_MASK_STOP_THREAD) {
          bStopThread = TRUE;
        }
      } else {
        ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::vCamCCvmWorkerThread(): Failed to post osal event!"));
      }
    }
    else  {
      ETG_TRACE_ERR(("CMD_AUDIO, CAmControlCvm::vCamCCvmWorkerThread(): Failed to wait for event!"));
      OSAL_s32ThreadWait(1000);
    }
  }// End of while
}

/************************************************************************
 *FUNCTION     : s32GetSystemVoltageHistory
 *DESCRIPTION  : reference object passed to OSAL_s32IOControl to get
 *               system voltage history.
 *PARAMETER    : DEV_VOLT_trSystemVoltageHistory
 *RETURN VALUE :
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
tS32 CAmControlCvm::s32GetSystemVoltageHistory(DEV_VOLT_trSystemVoltageHistory & rSystemVoltageHistory)
{
  ETG_TRACE_USR4(("CAmControlCvm::s32GetSystemVoltageHistory entered"));
  return OSAL_s32IOControl(m_IODeviceVoltDescr, OSAL_C_S32_IOCTRL_VOLT_GET_SYSTEM_VOLTAGE_HISTORY, (intptr_t)&rSystemVoltageHistory);
}


/************************************************************************
 *FUNCTION     : vSystemVoltageNotification
 *DESCRIPTION  : system voltage notification based on voltage history
 *               and notify the control sender base.
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vSystemVoltageNotification()
{
  ETG_TRACE_USR4(("CAmControlCvm::vSystemVoltageNotification entered"));
  DEV_VOLT_trSystemVoltageHistory rSystemVoltageHistory;

  rSystemVoltageHistory.u32ClientId = m_u32ClientId;

  //Get the system voltage changed history.
  if(OSAL_OK != s32GetSystemVoltageHistory(rSystemVoltageHistory))
  {
    ETG_TRACE_FATAL(("CAmControlCvm::vSystemVoltageNotification() !!!!!! Error detected !!!!!!"));
    return;
  }
  else
  {
    tU32 u32CurrentSystemVoltage = rSystemVoltageHistory.rSystemVoltage.u32CurrentSystemVoltageState;

  ETG_TRACE_USR4(("CAmControlCvm::vSystemVoltageNotification(): Low Voltage Counter= %d, Critical Low Voltage Counter= %d, High Voltage Counter= %d, Critical High Voltage Counter= %d",
    (uint32_t)rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter,
    (uint32_t)rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter,
    (uint32_t)rSystemVoltageHistory.rSystemVoltage.u32HighVoltageCounter,
    (uint32_t)rSystemVoltageHistory.rSystemVoltage.u32CriticalHighVoltageCounter));

  ETG_TRACE_USR4(("CAmControlCvm::vSystemVoltageNotification() Previous CVM state= %d, Current CVM state= %d",
    (uint32_t)m_u32PreviousSystemVoltage,
    (uint32_t)u32CurrentSystemVoltage));

    if (m_u32PreviousSystemVoltage >= DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ) {
      vPrevVoltStateOperating(rSystemVoltageHistory,u32CurrentSystemVoltage);
    }

    if (m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW ) {
      vPrevVoltStateLow(rSystemVoltageHistory,u32CurrentSystemVoltage);
    }

    if (m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW ) {
      vPrevVoltStateCriticalLow(u32CurrentSystemVoltage);
    }

    if (u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING) {
      vCurrVoltStateOperating();
    }

    if (u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH) {
      vCurrVoltStateHigh();
    }

    if (u32CurrentSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH) {
      vCurrVoltStateCriticalHigh();
    }

    m_u32PreviousSystemVoltage = rSystemVoltageHistory.rSystemVoltage.u32CurrentSystemVoltageState;
  }
}

/************************************************************************
 *FUNCTION     : vPrevVoltStateOperating
 *DESCRIPTION  : To notify AudioManager when previous voltage is greater
 *           than or equal to dev volt state operating based on the
 *           voltage history and current voltage, notifies the low
 *           and critical low voltage events.
 *PARAMETER    : const DEV_VOLT_trSystemVoltageHistory &, const tU32 &
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vPrevVoltStateOperating(const DEV_VOLT_trSystemVoltageHistory & rSystemVoltageHistory,const tU32 & u32CurrentSystemVoltage)
{
  ETG_TRACE_USR4(("CAmControlCvm::vPrevVoltStateOperating entered"));

  if (
       ( m_u32PreviousSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING)
    && ( rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0 )
  ){
      vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_END);
  }

  if (rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0) {
    vNotifyCvmEvent(OSALCVM_LOW_VOLTAGE_START);
  }

  if (rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter > 0) {
    vNotifyCvmEvent(OSALCVM_CRITICAL_LOW_VOLTAGE_START);
    if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW) {
      vNotifyCvmEvent(OSALCVM_CRITICAL_LOW_VOLTAGE_END);
    }
  }

  if (rSystemVoltageHistory.rSystemVoltage.u32LowVoltageCounter > 0) {
    if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW) {
      vNotifyCvmEvent(OSALCVM_LOW_VOLTAGE_END);
    }
  }
}

/************************************************************************
 *FUNCTION     : vPrevVoltStateLow
 *DESCRIPTION  : To notify AudioManager when previous voltage is equal to
 *           dev volt state low and based on the voltage history
 *           and current voltage, notifies the low and critical low
 *           voltage events.
 *PARAMETER    : const DEV_VOLT_trSystemVoltageHistory &, const tU32 &
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vPrevVoltStateLow(const DEV_VOLT_trSystemVoltageHistory & rSystemVoltageHistory,const tU32 & u32CurrentSystemVoltage)
{
  ETG_TRACE_USR4(("CAmControlCvm::vPrevVoltStateLow entered"));

  if (rSystemVoltageHistory.rSystemVoltage.u32CriticalLowVoltageCounter > 0) {
    vNotifyCvmEvent(OSALCVM_CRITICAL_LOW_VOLTAGE_START);
    if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW) {
      vNotifyCvmEvent(OSALCVM_CRITICAL_LOW_VOLTAGE_END);
    }
  }
  if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW) {
    vNotifyCvmEvent(OSALCVM_LOW_VOLTAGE_END);
  }
}

/************************************************************************
 *FUNCTION     : vPrevVoltStateCriticalLow
 *DESCRIPTION  : To notify AudioManager when previous voltage is equal to
 *           dev volt state critical low and based on the current
 *           voltage, notifies the low and critical low voltage events.
 *PARAMETER    : const tU32 &
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vPrevVoltStateCriticalLow(const tU32 & u32CurrentSystemVoltage)
{
  ETG_TRACE_USR4(("CAmControlCvm::vPrevVoltStateCriticalLow entered"));

  if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW) {
    vNotifyCvmEvent(OSALCVM_CRITICAL_LOW_VOLTAGE_END);
  }
  if (u32CurrentSystemVoltage > DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW) {
    vNotifyCvmEvent(OSALCVM_LOW_VOLTAGE_END);
  }
}

/************************************************************************
 *FUNCTION     : vCurrVoltStateOperating
 *DESCRIPTION  : To notify AudioManager when current voltage is equal to
 *           dev volt state operating, notifies the high and critical
 *           high voltage events.
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vCurrVoltStateOperating()
{
  ETG_TRACE_USR4(("CAmControlCvm::vCurrVoltStateOperating entered"));

  //if ((m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH )
  //    ||  (m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH ) ) {
  //  vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_END);
  //}

  /**
   * vdu1kor : Fix NCG3D-1658
   **/
  //If transition happens from DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH[5] -> DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING [4]
  if(m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH){
    ETG_TRACE_USR3(("CAmControlCvm::vCurrVoltStateOperating, volatge event generated from HighVoltage[5] -> OperatingVoltage[4]"));
    vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_END);
  }

  //If transition happens from DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH [6] -> DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING [4]
  if(m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH){
    ETG_TRACE_USR3(("CAmControlCvm::vCurrVoltStateOperating, volatge event generated from CriticalHighVoltage[6] -> OperatingVoltage[4]"));
    vNotifyCvmEvent(OSALCVM_CRITICAL_HIGH_VOLTAGE_END);
    vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_END);
  }

}

/************************************************************************
 *FUNCTION     : vCurrVoltStateHigh
 *DESCRIPTION  : To notify AudioManager when current voltage is equal to
 *           dev volt state High, notifies the high and critical
 *           high voltage events.
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vCurrVoltStateHigh()
{
  ETG_TRACE_USR4(("CAmControlCvm::vCurrVoltStateHigh entered"));

  if (m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ) {
    vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_START);
  } else {
    vNotifyCvmEvent(OSALCVM_CRITICAL_HIGH_VOLTAGE_END);
  }
}

/************************************************************************
 *FUNCTION     : vCurrVoltStateCriticalHigh
 *DESCRIPTION  : To notify AudioManager when current voltage is equal to
 *           dev volt state critical High, notifies the high and critical
 *           high voltage events.
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *26.02.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vCurrVoltStateCriticalHigh()
{
 ETG_TRACE_USR4(("CAmControlCvm::vCurrVoltStateCriticalHigh entered"));

  if (m_u32PreviousSystemVoltage == DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING ) {
    vNotifyCvmEvent(OSALCVM_HIGH_VOLTAGE_START);
  }
  vNotifyCvmEvent(OSALCVM_CRITICAL_HIGH_VOLTAGE_START);
}

/************************************************************************
 *FUNCTION     : vNotifyCvmEvent
 *DESCRIPTION  : To notify using thread context.
 *PARAMETER    : void
 *RETURN VALUE : void
 *HISTORY      :
 *07.03.2014   Rev 1.0     RBEI/ECG4 - Ranjit Susal  Initial Revision
 ************************************************************************/
void CAmControlCvm::vNotifyCvmEvent(const tU32 & u32CvmEvent)
{
  ETG_TRACE_USR4(("CAmControlCvm::vNotifyCvmEvent entered"));

  if((NULL != m_poControlReceiverShadow) && (NULL != m_poControlSenderBase)) {
    m_poControlReceiverShadow->vSwitchThreadContext<CAmControlSenderBase,tU32>(m_poControlSenderBase, &CAmControlSenderBase::vHandleCvmEvent,u32CvmEvent);
  } else {
      ETG_TRACE_FATAL(("CAmControlCvm::vNotifyCvmEvent, NULL pointer detected !!!!! "));
  }
}

