/*!************************************************************************
 * FILE :         clAudioSMEngine.cpp
 *
 * SW-COMPONENT:  Audio statemachine
 *
 * DESCRIPTION:
 *                STARTUP:
 *                StraightForward Case is:
 *                - Check in persistent DP for LSM and PSM
 *                  LSM: Last Selected Mode, i.e. the Entertainment Source that was
 *                  played before switching off the system
 *                  PSM: Previous Selected Mode, i.e. the Entertainment Source that was
 *                  played before the LSM
 *                - Check if LSM still available
 *                  If LSM available: start it
 *                  If LSM not available: Check if PSM still available
 *                     If PSM available: start it
 *                     If PSM not available: Start Radio.FM1
 *                Startup Details:
 *                - Before any communication to FC-Audio is initiated, wait for
 *                  AudioService to become available.
 *                - Before the availability of a media source can be checked, wait for
 *                  MediaService to become available and wait, until information about
 *                  removable mediasources is available. (was SD removed from slot while off?)
 *                - Prevent, that the XML framework begins to place API-Calls like:
 *                  Sound.GetCurrentAudioComponent()
 *                  AVDC.GetCurrentSource()
 *                  AVDC.Audio.GetAudioSourceState(...)
 *                  before the mentioned Services are available.
 *                - Before starting LSM, ensure, that media-removed info is evaluated on stack
 *                  Don't try to start e.g. LSM:SD and then immediately stop it, because is was removed
 *                  during off-state.
 *
 *                  DATAPOOL:
 *                  The following DP entries are relevant for Audio Source Switching
 *                  and the corresponding Screen-Changes / visual animations:
 *                  DPAUDIO__PER_ENTSRC:
 *                     Stores the current Audio Component (Radio or Media)
 *                     Used to determine/switch the menu net Radio vs. Media
 *                     Not persistent.
 *                  DPAUDIO__ENT_SRC:
 *                     Stores the current Radio-Band. (AM, FM, DAB)
 *                     Used to determine/switch the screen in Radio-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC__CURSRC:
 *                     Stores the current Media-Source (SD, CD-DA, CDC, etc)
 *                     Used to determine/switch the screen in Media-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC_PREV_AUDIO_SRC and DPAVDC_PCUR_AUDIO_SRC:
 *                     Stores the current and previous (Entertainment-)Audio-Source
 *                     Not used for Screen Changes. No GUI-Data Events.
 *                     Used for the history-stack for entertainment sources in the Audio-handler
 *                     Persistent.
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steilmann
 *
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Version
 *  23.10.07       CM-DI/PJ-VW34 Steinle
 *                 moved Creation of AudioSourceController from Contructor
 *                 to bConfigure.
 * 06.03.2008      CM-DI/PJ-VW34 Steinle
 *                 AudioEngine requests are allowed now also when AudioEngine is not started
 *                 This is posibble due to the Srv-Unavailable-handling in
 *                 the audio-src-implementations
 * 07.03.2008      CM-DI/PJ-VW34 Steinle
 *                 fix for mediaSourceRemoved-Event. If PSM=radio-am
 *                 goto am (not fm). Solved with isEntertainmentsourceAvailable()
 * 08.03.2008      CM-DI/PJ-VW34 Steinle
 *                 introduced persistence for mute-state and separate mute-pin source
 * 17.07.2013      CM-AI/CB-3 R.Volkmer (mailto:R.Volkmer@ITB-Solutions)
 *             initial extraction from HMI and integrate to Genivi AudioManager CtrlPlugIn
 *************************************************************************/

#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

//#define DP_DATAPOOL_ID =0xE880
#define DP_S_IMPORT_INTERFACE_FI
//#include "dp_if.h"

#ifndef VARIANT_S_FTR_ENABLE_AUDIO_STACK_UTEST
#include "dp_e880_if.h"
#else
#include "dp_e880_mock_if.h"
#endif

#define OEM_TYPE_INFINITI     0x16
// Unorganised traces
#ifndef USE_DLT_TRACE
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSMEngine.cpp.trc.h"
#endif
#endif

#define CFC_FI_S_IMPORT_INTERFACE_FI_TYPES
#include "cfc_fi_if.h"


#include "AudioStack/clTogglingSequences.h"
#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"
namespace AudioStack
{

using namespace AudioSource;

tU8                        clAudioSMEngine::m_u8ErrorCode                     = AUDIOSM_OK;
tU8                        clAudioSMEngine::m_u8AudioRouteMgr_SrvState        = 0;//AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
tU8                        clAudioSMEngine::m_u8Runlevel                      = 0;
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Previous = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Active   = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_LSM      = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Entertainment_Last_Tuner = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Current_Active         = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
SourceID                   clAudioSMEngine::m_enSource_Previous_Active        = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);

time_t                     clAudioSMEngine::m_stInitTime                      = time(NULL);


tVoid clAudioSMEngine::Invalidate()
{
   clAudioSourceFactory::Invalidate();
   clFactory_AudioSourceClass::Invalidate();
   clAudioSourceController::getInstance().vInvalidate();
   m_u8ErrorCode = AUDIOSM_OK;
   m_u8AudioRouteMgr_SrvState = 0;//AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
   m_u8Runlevel                      = 0;
   m_enSource_Entertainment_Previous = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_enSource_Entertainment_Active   = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_enSource_Entertainment_LSM      = SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_enSource_Entertainment_Last_Tuner= SourceID(clFactory_AudioSourceClass::GetInvalidSourceClass(),0);
   m_stInitTime                      = time(NULL);
}


/*!************************************************************************
 * METHOD:        constructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      default constructor.
 *
 * PARAMETER:
 *             none
 *
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 *************************************************************************/
clAudioSMEngine::clAudioSMEngine()
{


}



/*!***********************************************************************
 * METHOD:        destructor
 *
 * CLASS:         clAudioSMEngine
 *
 * DESCRIPTION:      destructor.
 *
 * PARAMETER:
 *             none
 * RETURNVALUE:
 *             none
 * HISTORY:
 *  19.09.07       CM-DI/PJ-VW34 Steilmann
 *  Initial Revision
 ************************************************************************/
clAudioSMEngine::~clAudioSMEngine()
{}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_On(SourceID srcID, tU32 u32UserData)
{
  ETG_TRACE_USR3((" Source_On: ignoring SinkID, hardcoded 1, u32UserData= %d",u32UserData));
  return clAudioSMEngine::Source_On(srcID, 1, u32UserData);
}

//new one, no more defaults allowed
clAudioSMEngine::enErrorCode clAudioSMEngine::Source_On(SourceID srcID, tU16 u16SinkID, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR3((" Source_On:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clFactory_AudioSourceClass::isValidSourceClass(srcID.enSourceClass))
   {
      ETG_TRACE_ERR(("CMD_AUDIO, Source_On: SourceClass %d not available by AudioSourceFactory",
            srcID.enSourceClass));
      return AUDIOSM_ERROR;
   }
   //return AudioSourceOn(srcID, FALSE, u32UserData);
   return AudioSourceOn(srcID, u16SinkID, FALSE, u32UserData);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Off(SourceID source, tU32 u32UserData)
{
  ETG_TRACE_USR3((" Source_Off: ignoring SinkID, hardcoded 1, u32UserData= %d",u32UserData));
  return clAudioSMEngine::Source_Off(source, 1, u32UserData);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Off(SourceID source, tU16 u16SinkID, tU32 u32UserData)
{
   if(((m_u8Runlevel < 2))) //
   {
      ETG_TRACE_USR3((" Source_Off:  NOT allowed due to runlevel %d", m_u8Runlevel));
      return AUDIOSM_ERROR;
   }
   if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
   {
      ETG_TRACE_ERR(("CMD_AUDIO, Source_Off: SourceClass %d not available by AudioSourceFactory",
            source.enSourceClass));
      return AUDIOSM_ERROR;
   }
   if(bAudioSourceOff(source, u16SinkID, u32UserData))
   {
    return AUDIOSM_OK;
   }
   else
   {
    return AUDIOSM_ERROR;
   }
}



clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Removed(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("CMD_AUDIO, Source_Removed: SourceClass %d not available by AudioSourceFactory",
               source.enSourceClass));
         return AUDIOSM_ERROR;
      }
      return SourceRemoved(source);
   }
   else
   {
      ETG_TRACE_USR3((" Source_Removed:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}

clAudioSMEngine::enErrorCode clAudioSMEngine::Source_Insert(SourceID source)
{
   // ignore request, as long as runlevel for media not reached
   if(m_u8Runlevel >= 3)
   {
      if(! clFactory_AudioSourceClass::isValidSourceClass(source.enSourceClass))
      {
         ETG_TRACE_ERR(("CMD_AUDIO, Source_Insert: SourceClass %d not available by AudioSourceFactory",
               source.enSourceClass));
         return AUDIOSM_ERROR;
      }
      return SourceInserted(source);
   }
   else
   {
      ETG_TRACE_USR3((" Source_Insert:  Ignored because of runlevel %d", m_u8Runlevel));
   }
   return AUDIOSM_ERROR;
}


tVoid clAudioSMEngine::ResetStack()
{
   clAudioSourceController::getInstance().vResetStack();
}



tVoid clAudioSMEngine::TraceStack()
{
   clAudioSourceController::getInstance().vTraceStack();
}

tVoid clAudioSMEngine::TraceBackupStack()
{
   clAudioSourceController::getInstance().vTraceBackupStack();
}


/*************************************************************************
* METHOD:          vMediaSourceInserted
*
* DESCRIPTION:     A media source has been inserted
*                  So it should be started (regarding priority rules)
*                  This event is currently handled the same way like a
*                  SetSource() request from user.
* PARAMETER:       -u8MediaSource   The inserted media source
*
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceInserted(SourceID srcID)
{
   ETG_TRACE_USR3(("SourceInserted: SourceClass: %d subID %#x(%d)"
         , srcID.enSourceClass
         , srcID.u16SubSource
         , srcID.u16SubSource));

   //daw2hi 24.08.2016 get the default sink and use that
   const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);
   //const sourceClassID invalidSrcClassID = clFactory_AudioSourceClass::GetInvalidSourceClass();

   if(srcClass.getClassID() != clFactory_AudioSourceClass::GetInvalidSourceClass())
   {
      return AudioSourceOn(srcID,srcClass.defaultSink,false,0);
   }
   else
   {
      ETG_TRACE_ERR(("CMD_AUDIO, !!! ERROR srcClass not found for Source %d#%d",srcID.enSourceClass,srcID.u16SubSource));
      return AUDIOSM_NOT_ALLOWED;
      //return AudioSourceOn(srcID,1,false,0);
   }

}
/*************************************************************************
* METHOD:          vMediaSourceRemoved
*
* DESCRIPTION:     If a MediaSource is removed/ejected by the user, we have
*                  to stop it if it is currently playing.
*                  Then we start another AudioSource, regarding the rules:
*                  If the previously played Source was Radio: start Radio
*                  If the previously played Source was Media:
*                     If the Media is still available: start that media
*                     If the Media is not available: start Radio
*
* PARAMETER:       tU32 u32MediaSourceToRemove
*                       the media source that was ejected/removed by the user
* RETURNVALUE:     -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SourceRemoved(SourceID srcID)
{
   ETG_TRACE_USR4(("clAudioSMEngine::SourceRemoved"));
   clAudioSource* pSrc = clAudioSourceFactory::getAudioSource(srcID);
   //Set source to unavailable
   if(pSrc==NULL)
   {
      ETG_TRACE_ERR(("CMD_AUDIO, ERROR SourceRemoved: no Instance for SourceClass: %d subID %#x(%d)"
         , srcID.enSourceClass
         , srcID.u16SubSource
         , srcID.u16SubSource));
      return clAudioSMEngine::AUDIOSM_ERROR;
   }
   if(pSrc->enIsSourceAvailable() != clAudioSource::not_available)
   {
      pSrc->vSourceAvailablilityChange(clAudioSource::not_available,clAudioSource::nomedia);
   }

   /*
    * TUNER_FM source to be activated upon the reception of SOURCE UNAVAILABLE response 
    * for requested source on user profile change
    */
   if(pSrc->bIsSourceProfChg() && clAudioSourceController::bIsSourceOnStack(pSrc->sGetId()))
      pSrc->vSetSwitchtoDefaultSrcOnProfChg(true);

   ETG_TRACE_USR3(("SourceRemoved: SourceClass: %d subID %#x(%d)"
         , srcID.enSourceClass
         , srcID.u16SubSource
         , srcID.u16SubSource));

   /*********************************************************************************************
    * vdu1kor : Remove connection entry from database
    *********************************************************************************************/
   clGeniviAudioSource* pGAS = clGeniviAudioCtrlAdapter::getAudioSource(srcID);

   if(NULL != pGAS){
     clGeniviAudioCtrlAdapter::removeMainConnection(pGAS->u16GetGeniviSourceID());
     ETG_TRACE_USR4(("SourceRemoved: connection entry removed from database"));
   }
   else{
     ETG_TRACE_USR4(("SourceRemoved: clGeniviAudioSource is NULL, Failed to removeMainConnection from DB !!!!"));
   }

   /*
    * Delete source if not attached to stack & diagnosis session is not active (PSARCC30-675/PSARCC21-3180)
    */
     if(clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
     {
     ETG_TRACE_USR4(("SourceRemoved: Diagnosis session active"));
     if(!clAudioSourceController::bIsSourceOnStack(srcID))
     {
       ETG_TRACE_USR4(("SourceRemoved: Source not attached to New/Active Stack"));
       if(!clAudioSourceController::bIsSourceOnBackUpStack(srcID)){
       ETG_TRACE_USR4(("SourceRemoved: Source not attached to New & BackUp Stack, delete it"));
       clAudioSourceFactory::removeAudioSources(srcID);
       return clAudioSMEngine::AUDIOSM_OK;
       }
       else{
       /*
        * During diagnosis session, if source removed which is present on BackUp stack, No need to delete source
        */
       ETG_TRACE_USR4(("SourceRemoved: Source attached to BackUp Stack, No need to delete it"));
       return clAudioSMEngine::AUDIOSM_OK;
       }
     }
     else
     {
       ETG_TRACE_USR4(("SourceRemoved: Source is attached to New/Active Stack"));
     }

     }
     else  // Diagnosis session not active
     {
     /*
      * Only if the source is not attached to Stack we can destruct it
      * it may be added to stack recently and still in Off state because former source
      * is still ramping down
      */

     ETG_TRACE_USR4(("SourceRemoved: Diagnosis session not active"));
     if(!clAudioSourceController::bIsSourceOnStack(srcID))
     {
       ETG_TRACE_USR4(("SourceRemoved: Source not attached to Stack, delete it"));
       clAudioSourceFactory::removeAudioSources(srcID);
       return clAudioSMEngine::AUDIOSM_OK;
     }
   }

  /*
   * vdu1kor - Fix JAC2-4143
   * if active source is removed then source should be changed to m_enSource_Entertainment_Previous
   * otherwise source in stack which is not active is removed hence no need to fallback to previous source
   * as active source is under rampup process
   */
   if(m_enSource_Entertainment_Active !=  srcID){

     if(m_enSource_Entertainment_Active == m_enSource_Current_Active)
     {
       /*
        * Active source must be on stack
        * Ex:
        *   1. Media is on both stack
        *   2. Diagnosis session is ended -> media replaced by FM (m_enSource_Current_Active-FM)
        *   3. On back up stack media source moved to SrcAvailCheck_On
        *   4. FM is not present in current stack
        */
       if(clAudioSourceController::bIsSourceOnStack(m_enSource_Current_Active))
       {
      //Since source removed was not active means active src is still under rampup hence no need to do anything
      ETG_TRACE_USR4(("SourceRemoved: m_enSource_Entertainment_Active, SourceClass: %d subID %d"
         , m_enSource_Entertainment_Active.enSourceClass, m_enSource_Entertainment_Active.u16SubSource));
      ETG_TRACE_USR4(("SourceRemoved: Source removed is not active source, means active src is still  " \
         "under rampup hence no need to do fallback to previous source !!"));
      return AUDIOSM_OK;
       }
     }
     else if((pSrc->getSourceType().name != "entertain"))
     {
       ETG_TRACE_USR4(("SourceRemoved: Source removed is not active Entertainment source perhaps src removed is  " \
           "not entertainment source, no need to do fallback to previous source  !!"));
       /* This handling not required because source component (i.e Phone) will always send
        * SourceRouteRequest(off) if source bacame UnAvailable
        * when source removed is not entertainment source then source should be send to Off state and removed from stack
        * vAudioSourceOff(srcID);
       */
       return AUDIOSM_OK;
     }
   //fix issue Bug 520882
   //removed source is ramping down, no need to fall back , next source is started processing
   else if (pSrc->pclGetState() == SourceStateMachine::clSrcStateFactory::pclCreateRampDownToOff() || pSrc->pclGetState()== SourceStateMachine::clSrcStateFactory::pclCreateOff())
   {
      ETG_TRACE_USR4(("SourceRemoved: Source removed is not active source, means active src is  " \
         "under rampdownToOff hence no need to do fallback to previous source !!"));
          return AUDIOSM_OK;
   }

   }

  return RestorePreviousSource(srcID);
}

clAudioSMEngine::enErrorCode clAudioSMEngine::RestorePreviousSource(SourceID srcID)
{
   ETG_TRACE_USR4(("clAudioSMEngine::RestorePreviousSource: entered"));
   ETG_TRACE_USR4(("clAudioSMEngine::RestorePreviousSource, SourceID.enSourceClass(%d) and SourceID.u16SubSource(%d)", srcID.enSourceClass, srcID.u16SubSource));

   
   tU8 lastSourceClass = 0;
   tU16 u16SubID = 0;
   SourceID lastSource(lastSourceClass, u16SubID);
        
   clAudioSource* pclAudioSource = clAudioSourceFactory::isSourceInstantiated(srcID);
   if(pclAudioSource != NULL)
   {
      ETG_TRACE_USR3(("clAudioSMEngine::RestorePreviousSource, requested source on profile change is still registered, switch to default source"));

      if(true == pclAudioSource->bSwitchtoDefaultSrcOnProfChg())
      {
        ETG_TRACE_USR3(("clAudioSMEngine::RestorePreviousSource, fetching the default source to be activatated post timeout or Unavailable notification when timer is running"));
        lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
        lastSource.u16SubSource = 0;
        
        m_enSource_Entertainment_Previous = lastSource;

        pclAudioSource->vSetSwitchtoDefaultSrcOnProfChg(false);
      }
   }
   
#ifndef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
  SourceID restoreSrcID = m_enSource_Entertainment_Previous;
#else
  SourceID restoreSrcID = m_enSource_Entertainment_Last_Tuner;
#endif

  //No need to create new source object if it's dynamic source vdu1kor SUZUKI-15832
  tBool bCreateNewSrcObj = true;
  const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(restoreSrcID.enSourceClass);

  if(srcClass.getRegistrationMode() == clSourceClass::registerDynamic)
  {
    bCreateNewSrcObj = false;
    ETG_TRACE_USR4(("RestorePreviousSource: m_enSource_Entertainment_Previous is of type dynamic"));
  }

  clAudioSource* prevAudioSrc = NULL;
  if(!bIsSourceAvailable(restoreSrcID,bCreateNewSrcObj))
  {
    sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
    prevAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
    if(prevAudioSrc == NULL)
      prevAudioSrc = clAudioSourceFactory::getAvailableAudioSource();

    if(prevAudioSrc != NULL)
      restoreSrcID = prevAudioSrc->sGetId();
  }
  else
  {
    prevAudioSrc = clAudioSourceFactory::getAudioSource(restoreSrcID);
  }

  if(prevAudioSrc == NULL)
  {
    ETG_TRACE_FATAL(("RestorePreviousSource: THERE IS NO AVAILABLE AUDIO SOURCE. CANNOT RESTORE ANY AUDIO SOURCE !!"))
      return AUDIOSM_ERROR;
  }

  ETG_TRACE_USR3(("RestorePreviousSource: m_enSource_Current_Active, SourceClass: %d subID %d"
      , m_enSource_Current_Active.enSourceClass
      , m_enSource_Current_Active.u16SubSource));

  /*
   * vdu1kor : If active source is not Entertainment source removed, it's background source exchange,
   * For background source exchange, no need to apply push/pop action rules (Fix NCG3D-1538)
   */
  if( m_enSource_Current_Active !=  srcID )
  {
    ETG_TRACE_USR4(("RestorePreviousSource: Source removed is not active source, means active src is other than  " \
        "entertainment source, Background source Exchange is required !!!!"));

    prevAudioSrc->vSetSourceBG(TRUE);
  }
  else
  {
    ETG_TRACE_USR4(("RestorePreviousSource: Source removed is active source, activating previous source, SourceClass: %d subID %d"
        , restoreSrcID.enSourceClass
        , restoreSrcID.u16SubSource));
  }

  prevAudioSrc->vSetPrevSrcRemoved(TRUE);
  ETG_TRACE_USR4(("RestorePreviousSource: activating m_enSource_Entertainment_Previous, SourceClass: %d subID %d"
      , restoreSrcID.enSourceClass
      , restoreSrcID.u16SubSource));

  //AudioSourceOn(restoreSrcID, TRUE);
  // no more with defaults
  AudioSourceOn(restoreSrcID,1, TRUE, 0);

  if(prevAudioSrc->bIsSourceBG())
    prevAudioSrc->vSetSourceBG(FALSE);

  if(prevAudioSrc->bIsPrevSourceRemoved())
    prevAudioSrc->vSetPrevSrcRemoved(FALSE);

  return AUDIOSM_OK;
}

/*************************************************************************
* METHOD:         vHandleSrcActError
*
* DESCRIPTION:    Remove defect Source from AudioStack.
************************************************************************/
tVoid clAudioSMEngine::vHandleSrcActError(SourceID errSrc)
{

//   SourceID prevAudioSrcID = m_enSource_Entertainment_Previous;
//   if(!bIsSourceAvailable(prevAudioSrcID))
//   {
//      sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
//      clAudioSource* pAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
//      if(pAudioSrc == NULL)
//         pAudioSrc = clAudioSourceFactory::getAvailableAudioSource();
//   }

   clAudioSourceController::getInstance().vHandleSrcActError(errSrc);
}

/*************************************************************************
* METHOD:         vAudioSourceOn
*
* DESCRIPTION:    Called on each request of a source change.
*                 This method handles requests for Entertainment-sources
*                 as well as requests for Foreground- and Mix-sources
*
*                 FOREGROUND SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 MIX SOURCES:
*                 - Just forward the request to the AudioSourceController
*                 - No update of Datapool entries
*                 - No update play history
*
*                 ENTERTANMENT SOURCES:
*                 - Immediately and synchronously update the Datapool in order
*                 to initiate the screen change and to start the visual
*                 animation. This operation has to be performed, independent
*                 of the operation on the AudioStack. I.e. we even update the
*                 Datapool.CurrentSource although we still do not know, if the
*                 new requested source is allowed to play.
*                 This is essential for entertainment sources as it might be
*                 required to change screens and menu-nets for entertainment-
*                 source while other (Foreground-)audiosources are active.
*                 In order to store the playhistory for LSM and PSM, the
*                 Datapool entries:
*                    DPAVDC__CUR_AUDIO_SRC
*                    DPAVDC__PREV_AUDIO_SRC
*                 are stored. Thes entries only contain entertainment sources!
*
* PARAMETER:       -
*
* RETURNVALUE:     -
*
************************************************************************/
// no more with default
clAudioSMEngine::enErrorCode clAudioSMEngine::AudioSourceOn(SourceID srcID, tU16 sinkID, tBool bRemoveCurrent, tU32 u32UserData)
{
   clAudioSource* poAudioSource = NULL;
//   clSourceClass::group_t enGroup;
//   clSourceClass::type_t enType;

   // u8AudioSource = static_cast<AudioSources::enAudioSources>(clAudioSourceFactory::mapVirtualToLogicalSource(static_cast<AudioSources::enAudioSources>(u8AudioSource)));
   if(! clFactory_AudioSourceClass::isValidSourceClass(srcID.enSourceClass))
   {
      ETG_TRACE_ERR(("CMD_AUDIO, clAudioSMEngine::vAudioSourceOn,SourceClass %d not available by AudioSourceFactory",
            srcID.enSourceClass));
      return AUDIOSM_ERROR;
   }

   ETG_TRACE_USR2(("vAudioSourceOn: SourceClass: %d subID %d, UsrData 0x%x"
         , (srcID.enSourceClass)
         , (tU16)srcID.u16SubSource
         , (tU32)u32UserData));

   /*
   * vdu1kor (fix : NCG3D-5327)
   * If source is dynamic and source not_available is reported earlier then source object is deleted
   * After source become unAvailable, if sourceRouteRequest() is received then it should be ignored
   * For registerDynamic source no need to create object.
   * For registerStatic(Ex AM,FM) & registerNone(all mute sources), object should be created
   */
   const clSourceClass& SrcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);

   if(SrcClass.getRegistrationMode() == clSourceClass::registerDynamic){
    ETG_TRACE_USR4(("vAudioSourceOn: it's Dynamic source, No need to create object if not found"));
      poAudioSource = clAudioSourceFactory::getAudioSource(srcID,false);
   }
   else{
     ETG_TRACE_USR4(("vAudioSourceOn: it's not Dynamic source"));
     poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
   }

   if(poAudioSource == NULL)
   {
      ETG_TRACE_USR4(("vAudioSourceOn: SourceClass: %d subID %d not available by AudioSourceFactory"
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource));
      return AUDIOSM_ERROR;
   }

   //ToDo: can this be improved

   tBool bSinkAllowed = false;
   std::vector<tU16>::const_iterator it;
   for(it=poAudioSource->getSourceClass().Sinks.begin(); it != poAudioSource->getSourceClass().Sinks.end(); ++it)
   {
      if((*it) == sinkID)
      {
         bSinkAllowed = true;
         break;
      }
   }
   if(bSinkAllowed == false)
   {
      ETG_TRACE_USR4(("vAudioSourceOn: not allowed for Sink %d",sinkID));
      return AUDIOSM_ERROR;
   }


   poAudioSource->vSetCurActSink(sinkID);
//   enGroup = poAudioSource->getSourceClass().getGroupID();
//   enType  = poAudioSource->getSourceClass().getTypeID();

   if (clAudioSourceController::getInstance().enCheckSource(poAudioSource) != clSourceClass::allowed)
   {
      ETG_TRACE_USR4(("SourceClass: %d not allowed!",
            (poAudioSource->sGetId().enSourceClass)));
      return AUDIOSM_ERROR;
   }

   //For available and unknown we will put the source on stack
   if(poAudioSource->enIsSourceAvailable() == clAudioSource::not_available)
   {
      ETG_TRACE_USR4(("ERROR: SourceClass: %d subID %#x(%d): NOT AVAILABLE"
         , (srcID.enSourceClass)
         , (tU16)srcID.u16SubSource
         , (tU16)srcID.u16SubSource));
         return AUDIOSM_ERROR;
   }

   // ******
   // STEP 1
   // ******
   // Update the persistent history-stack of (persistent) Entertainment-Sources
   // Introduced a new configurable parameter "previous". Store previous history if parameter set to "YES".
   // Earlier this was checked by "restorable" parameter.

   /* vdu1kor
    * Only entairtainment source can be cosidered as previous source to be played instead of All sources (fix for JAC2-3489)
  * poAudioSource->getSourceType().previous is always "TRUE" if not defined in source_priorities.xml
    */
   if ((poAudioSource->getSourceType().previous) && (poAudioSource->getSourceType().name == "entertain") && (sinkID == 1) )
   {
      if (srcID != m_enSource_Entertainment_Active)
      {
        bool bDiagActive = clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn();
         if(bDiagActive)
         {
           ETG_TRACE_USR4(("Diagnose is active, keep m_enSource_Entertainment_Previous"));
         }
         else {
       if (bRemoveCurrent)
       {
        // if current bg-source should be removed, don't keep it in history-stack
        sourceClassID srcClass = clFactory_AudioSourceClass::GetInvalidSourceClass();
        m_enSource_Entertainment_Previous = SourceID(srcClass,0);
       }
       else
       {
        // the current bg- source will be saved in the history-stack as the previous source
       clAudioSource* poEntSource =clAudioSourceFactory::isSourceInstantiated(m_enSource_Entertainment_Active);
       if(poEntSource && poEntSource->bIsSourcePrevious())
       {
         m_enSource_Entertainment_Previous = m_enSource_Entertainment_Active;
       }
       }
         }
         m_enSource_Entertainment_Active = srcID;

         ETG_TRACE_USR2(("Entertainment Source update: ACTIVE SourceClass: %d subID %d, PREVIOUS SourceClass: %d subID %d"
         , m_enSource_Entertainment_Active.enSourceClass
         , m_enSource_Entertainment_Active.u16SubSource
         , m_enSource_Entertainment_Previous.enSourceClass
         , m_enSource_Entertainment_Previous.u16SubSource));
      }
   }

   //Check if source is tuner entertainment source then update system property SYP_LAST_TUNER_SOURCE (CMG3G-15357)
   if((clSourceClass::group_tuner == poAudioSource->getSourceClass().getGroupID())
     && ("entertain" == poAudioSource->getSourceType().name) )
   {
    ETG_TRACE_USR4(("vAudioSourceOn: SourceClass: %d subID %d is Tuner Entertainment Source, \
                   Updating property SYP_LAST_TUNER_SOURCE",(srcID.enSourceClass),(tU16)srcID.u16SubSource));

    clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_LAST_TUNER_SOURCE,srcID.enSourceClass);
   }

   if((clSourceClass::group_tuner == poAudioSource->getSourceClass().getGroupID())
     && ("entertain" == poAudioSource->getSourceType().name) )
   {
    ETG_TRACE_USR1(("vAudioSourceOn: SourceClass: %d subID %d is Tuner Entertainment Source, \
                   Updating LAST TUNER SOURCE",(srcID.enSourceClass),(tU16)srcID.u16SubSource));

  m_enSource_Entertainment_Last_Tuner.enSourceClass = srcID.enSourceClass;
  m_enSource_Entertainment_Last_Tuner.u16SubSource = srcID.u16SubSource;
  #ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
    vWriteLastTunerDataPool(m_enSource_Entertainment_Last_Tuner.enSourceClass,m_enSource_Entertainment_Last_Tuner.u16SubSource);
  #endif
   }

   /* Remember the current active source -> not only entertainment source but any source */

   // daw2hi: but only for sink 1
   //resolve issue  NCG3D-113136:ssw3kor, handeled in on_done
   /* if(sinkID == 1) {
      vSetCurrentActiveSource(srcID);
   }*/


   // ******
   // STEP 2
   // ******
   if(sinkID == 1) {
      // daw2hi: check for MUTE_ENTERTAIN and store persistent Mute
      vDPPersistentMute(poAudioSource, PersMuteDP_Set);
      //Store persistently
      if((poAudioSource->getSourceType().restorable) &&
      ((poAudioSource->getRestoreValue() == clSourceClass::profileChange) ||
      (poAudioSource->getRestoreValue() == clSourceClass::startupAndProfileChange)))
      {
         m_enSource_Entertainment_Active = srcID;
         // Update the DP values, which will trigger the screen-changes and animations
         ETG_TRACE_USR4(("vAudioSourceOn(): Step 2: Store Last Source SourceClass: %d subID %d"
               , m_enSource_Entertainment_Active.enSourceClass
               , (tU16)m_enSource_Entertainment_Active.u16SubSource));

         //Check if Diag Session is active (PSARCCB-8911)
         if(false == clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
         {
            ETG_TRACE_USR4(("vAudioSourceOn():Diagnosis session is not active, Store Last Source in datapool !"));
            vWriteDataPool(m_enSource_Entertainment_Active.enSourceClass,m_enSource_Entertainment_Active.u16SubSource);
         }
         else
         {
            ETG_TRACE_USR4(("vAudioSourceOn():Diagnosis session is active, No need to update LSM in datapool !"));
         }

      }
      else {
         // Update the DP values, which will trigger the screen-changes and animations
         ETG_TRACE_USR4(("vAudioSourceOn(): Step 2: Ignored due to SourceClassTypeID %d"
               , poAudioSource->getSourceType().id));
      }
   }
   // ******
   // STEP 3
   // ******
   // Request Source Change on AudioSourceController
   if (!(poAudioSource->getSourceType().stackable))
   {
     /*
      * Fix NCG3D-10801
      * if source is not stackable and already on stack, Need to process the source_change(on) request
      */
     if((clFactory_AudioSourceClass::GetSourceSettings().AllowSrcAlreadyonStack)
     && (clAudioSourceController::getInstance().bIsSourceOnStack(srcID,sinkID)))
     {
       /*
        * To handle below use case, when source change request for Media is received
    * AST:  --------TOP--------
    * AST:     TUNER_FM : off
    * AST:   MEDIA_PLAYER#1 : rampdownToOff
    * AST:  ------BOTTOM-------
        */
       if(clAudioSourceController::getInstance().bIsAnotherEntSrcOnStack(srcID))
       {
         if(clAudioSourceController::getInstance().bSELECT(srcID, sinkID, u32UserData))
           return AUDIOSM_OK;
       }

       //source on stack and there is no other entertainment source in stack
       if(clAudioSourceController::getInstance().bON(srcID, sinkID, u32UserData))
        return AUDIOSM_OK;
     }
     /*
      * if source is not stackable and not present on stack, then source will be replaced with one already present
      */
      if(clAudioSourceController::getInstance().bSELECT(srcID, sinkID, u32UserData))
      {
         return AUDIOSM_OK;
      }
   }
   else {
      if (!(poAudioSource->getSourceType().exclusive))
      {
         if(clAudioSourceController::getInstance().bMIX(srcID, sinkID, u32UserData))
         {
            return AUDIOSM_OK;
         }
      }
      else
      {
         if(clAudioSourceController::getInstance().bON(srcID, sinkID, u32UserData))
         {
            return AUDIOSM_OK;
         }
      }
   }
   return AUDIOSM_ERROR;
}


clAudioSMEngine::enErrorCode clAudioSMEngine::Source_AvailabilityChange(SourceID srcID,
                                                        clAudioSource::enSourceAvailability availability,
                                                        clAudioSource::enSourceAvailabilityReason reason)
{
   ETG_TRACE_USR4(("Source_Availability: SourceClass: %d subID %#x(%d), Availability: %d, Reason: %d"
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, reason)
            ));
   clAudioSource* pAudSrc = NULL;
   if(availability == clAudioSource::not_available)
   {
      //prevent re-creating instances for sources that had become not available
      pAudSrc = clAudioSourceFactory::isSourceInstantiated(srcID);
   }
   else if(availability == clAudioSource::unkonwn)
   {
     /*
      * Handle the 'unknown' source availability status when profile change is initiated by user.
      * In this scenario, the timer is removed and waits indefinetely for source availablity.
      */
      pAudSrc = clAudioSourceFactory::isSourceInstantiated(srcID);
      if( pAudSrc!= NULL)
        {
          ETG_TRACE_USR4(("clAudioSMEngine::Source_AvailabilityChange, removing the timer since source unavailability status is received for profile change request for timerID(%d)"
                           , pAudSrc->getTimerHandle()));
          if(pAudSrc->getTimerHandle() != 0 && (pAudSrc->bIsSourceProfChg() == true || pAudSrc->bIsSourceLSM() == true))
             pAudSrc->vRemoveTimer();

          return clAudioSMEngine::AUDIOSM_OK;
        }
   }else{
      //create/get an Instance
      pAudSrc = clAudioSourceFactory::getAudioSource(srcID);
   }

   if(pAudSrc != NULL)
   {
      if(clAudioSource::available == availability)
      {
         if(pAudSrc->bIsSourceProfChg())
            pAudSrc->vSetSourceProfChg(false);

         if(pAudSrc->bIsSourceLSM())
            pAudSrc->vSetSourceLSM(false);

         //Remove the active timer if applicable, since the source is available now  
         if(pAudSrc->getTimerHandle() != 0)
            pAudSrc->vRemoveTimer();
      }

      //Set the Availability according received
      pAudSrc->vSourceAvailablilityChange(availability, reason);
   }
   return clAudioSMEngine::AUDIOSM_OK;
}


/*************************************************************************
* METHOD:         u8GetNextAvailableEntertainmentSource
*
* DESCRIPTION:    Iterate through the list of entertainment sources,
*                 defined in the specified sequence and find the next available one.
*
* PARAMETER:      u8CurrentAudioSource
*                 pau8ToggleSequence      defines the sequence descriptor to iterate through
*                 u8SizeToggleSequence    defines the size of the sequence descriptor
* RETURNVALUE:    next available ent-source or NONE if no ent-source
*                 available
*
************************************************************************/
SourceID clAudioSMEngine::GetNextAvailableEntertainmentSource(SourceID srcID)
{
   ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Find next Source after SourceClass: %d subID %#x(%d) "
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource
            , (tU16)srcID.u16SubSource));
   SourceID nextSrcID ( 0 ,0);
   clAudioSource* pSrc = clAudioSourceFactory::getAvailableAudioSource(srcID.enSourceClass);

   if(pSrc != NULL)
   {
      ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %d "
                  , (pSrc->sGetId().enSourceClass)
                  , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }else{
      //If we cannot find any available
      while(pSrc == NULL)
      {
         sourceClassID nextSrcClass =
               clFactory_AudioSourceClass::GetDefaultSourceClass();
                              //clTogglingSequences::GetNextSourceClass(srcID.enSourceClass, AudioToggleGroups::RMT_CTRL);
         ETG_TRACE_USR4(("u8GetNextAvailableEntertainmentSource: Search default SourceClass Source: %d "
                           , nextSrcClass));
         pSrc = clAudioSourceFactory::getAvailableAudioSource(nextSrcClass);
      }
      ETG_TRACE_USR3(("u8GetNextAvailableEntertainmentSource: Next Source: %d subID %d "
                        , (pSrc->sGetId().enSourceClass)
                        , (tU16)pSrc->sGetId().u16SubSource));
      return pSrc->sGetId();
   }
}

/*************************************************************************
* METHOD:         vOnAudioServiceStateChanged
*
* DESCRIPTION:    ! Under Construction !
*
*
* PARAMETER:      u8ServiceState
*
* RETURNVALUE:    -
*
************************************************************************/

tVoid clAudioSMEngine::vOnAudioServiceStateChanged(tU16 u16Service, tU8 u8ServiceState)
{

   ETG_TRACE_USR4(("vOnAudioServiceStateChanged Service:%d State:%d",
         u16Service,
         u8ServiceState));
// TODO SERVICE AVAILABILITY
   //
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_NOT_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Not Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_NOT_AVAILABLE;
//      ETG_TRACE_USR1((" RESET AUDIO STACK!"));
//      vResetStack();
//   }
//
//   if((u16Service == CCA_C_U16_SRV_AUDIO_ROUTE_MGR) && (u8ServiceState == AMT_C_U8_SVCSTATE_AVAILABLE))
//   {
//      ETG_TRACE_USR1(("Set m_u8AudioRouteMgr_SrvState : Available"));
//      m_u8AudioRouteMgr_SrvState = AMT_C_U8_SVCSTATE_AVAILABLE;
//   }
}

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStarted(tU8 u8AudioSource) { (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceStopped
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceStopped(tU8 u8AudioSource){ (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceInitStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8AudioSource
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceInitStarted(tU8 u8AudioSource) { (void)u8AudioSource; }

/*************************************************************************
* METHOD:         vAudioSourceStarted
*
* DESCRIPTION:    Callback method from AudioSourceController
*                 Distribute the notification to CMs
*
* PARAMETER:      u8NewEntertainmentSource
*                 bBackground
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vEntertainmentSourceExchanged(tU8 u8NewEntertainmentSource, tBool bBackground) {(void)u8NewEntertainmentSource;(void)bBackground; }

/*************************************************************************
* METHOD:         vAudioSourceOff
*
* DESCRIPTION:    Stop an FG Source
* PARAMETER:      tU8 u8AudioSource    the FG source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vAudioSourceOff(SourceID srcID, tU32 u32UserData)
{
   // we can just use the bAudioSourceOff function and ignore the return value.
   (void)bAudioSourceOff(srcID, u32UserData);
}
/*************************************************************************
* METHOD:         bAudioSourceOff
*
* DESCRIPTION:    Stop an FG Source
* PARAMETER:      tU8 u8AudioSource    the FG source to stop
*
* RETURNVALUE:    bool bSuccess  informs if the SourceOff was succesfull (true) or not (false)
*
************************************************************************/
tBool clAudioSMEngine::bAudioSourceOff(SourceID srcID, tU32 u32UserData)
{
  // use sink 1 if not provided
  return bAudioSourceOff(srcID, 1, u32UserData);
}

tBool clAudioSMEngine::bAudioSourceOff(SourceID srcID, tU16 sinkID, tU32 u32UserData)
{
   tBool bSuccess = true;
   clAudioSource* poAudioSource = NULL;

   ETG_TRACE_USR4(("bAudioSourceOff: SourceClass: %d subID %d"
            , srcID.enSourceClass
            , srcID.u16SubSource));

   /*
   * vdu1kor (fix : NCG3D-5327)
   * If source is dynamic and source not_available is reported earlier then source object is deleted
   * After source become unAvailable, if sourceRouteRequest() is received then it should be ignored
   * For registerDynamic source no need to create object.
   * For registerStatic(Ex AM,FM) & registerNone(all mute sources), object should be created
   */
   const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(srcID.enSourceClass);

   if(srcClass.getRegistrationMode() == clSourceClass::registerDynamic){
     ETG_TRACE_USR3(("bAudioSourceOff: it's Dynamic source, No need to create object if not found"));
     poAudioSource = clAudioSourceFactory::getAudioSource(srcID,false);
   }
   else{
     ETG_TRACE_USR3(("bAudioSourceOff: it's not Dynamic source"));
     poAudioSource = clAudioSourceFactory::getAudioSource(srcID);
   }

   if(poAudioSource == NULL)
   {
      ETG_TRACE_USR3(("bAudioSourceOff: %d subID %d not available by AudioSourceFactory"
            , (srcID.enSourceClass)
            , (tU16)srcID.u16SubSource));
      return false;
   }

   const stSourceType& srcType = poAudioSource->getSourceType();

   poAudioSource->vSetCurActSink(sinkID);
   // Delegate Request to AudioSourceController
   if(!srcType.exclusive)
   {
      clAudioSourceController::getInstance().bMIXOFF(srcID, sinkID, u32UserData);
   }
   else if(srcType.stackable)
   {
    //daw2hi: Remove also PersistentMute
    vDPPersistentMute(poAudioSource, PersMuteDP_Reset);
      clAudioSourceController::getInstance().bOFF(srcID, 0);
   }
   else
   {

      //ToDo: daw2hi 02.08.2016 check if below part shoudl be done only for sink 1

      bSuccess = false;

      if(sinkID == 1)
    {
          clAudioSource* mMuteAudioOff = NULL;
          const clSourceClass* mMuteAudioOffSrcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_AUDIO_OFF");

          clAudioSource* mMuteEntertain = NULL;
          const clSourceClass* mMuteEntertainSrcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");

          if(mMuteAudioOffSrcClass != NULL)
          {
            mMuteAudioOff = clAudioSourceFactory::getAudioSource(mMuteAudioOffSrcClass->SourceClassID,0);
          }

          if(mMuteEntertainSrcClass != NULL)
          {
            mMuteEntertain = clAudioSourceFactory::getAudioSource(mMuteEntertainSrcClass->SourceClassID,0);
          }

      // CMG3G-10573 Command for HMI to request a source change to previous audio source
      #ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
          if(bIsSourceAvailable(m_enSource_Entertainment_Last_Tuner,false))
          {
             if(AUDIOSM_OK == AudioSourceOn(m_enSource_Entertainment_Last_Tuner,1, FALSE, 0))
             {
                bSuccess = true;
             }

          }
      #else
      // Checking previous entertainment source is available ? and current disconnect source ID is not equal to previous entertainment source to avoid multiple disconnect request on same source.
        if(bIsSourceAvailable(m_enSource_Entertainment_Previous,false) && (srcID != m_enSource_Entertainment_Previous))
        {
          clAudioSource* prevEntSrcClass =  clAudioSourceFactory::getAudioSource(m_enSource_Entertainment_Previous,false);

           if((poAudioSource != NULL) && (mMuteAudioOff != NULL) && (mMuteEntertain != NULL) && (prevEntSrcClass != NULL))
           {
              if((poAudioSource->getSourceType().name == "entertain") && (clAudioSourceController::bIsSourceOnStack(mMuteAudioOff->sGetId()) || clAudioSourceController::bIsSourceOnStack(mMuteEntertain->sGetId())) )
              {
                prevEntSrcClass->vSetSourceBG(TRUE);
              }
           }

            if(AUDIOSM_OK == AudioSourceOn(m_enSource_Entertainment_Previous,1, FALSE, 0))
            {
                bSuccess = true;
            }

            if(prevEntSrcClass != NULL)
            {
                if(prevEntSrcClass->bIsSourceBG())
                {
                   prevEntSrcClass->vSetSourceBG(FALSE);
                }
            }
        }
      #endif
        else
        {
            ETG_TRACE_ERR(("CMD_AUDIO, previous Source seems no more available, try to get default source"));
            sourceClassID defClass = clFactory_AudioSourceClass::GetDefaultSourceClass(); //default has currently no subsource ??
            clAudioSource* defAudioSrc = clAudioSourceFactory::getAvailableAudioSource(defClass);
            if(defAudioSrc != NULL)
            {
                SourceID  defSrcID = defAudioSrc->sGetId();

                if((poAudioSource != NULL) && (mMuteAudioOff != NULL) && (mMuteEntertain != NULL))
                {
                    if( (poAudioSource->getSourceType().name == "entertain") && (clAudioSourceController::bIsSourceOnStack(mMuteAudioOff->sGetId()) || clAudioSourceController::bIsSourceOnStack(mMuteEntertain->sGetId())))
                    {
                        defAudioSrc->vSetSourceBG(TRUE);
                    }
                }

                if(AUDIOSM_OK == AudioSourceOn(defSrcID,1, FALSE, 0))
                {
                    bSuccess = true;
                }

                if(defAudioSrc->bIsSourceBG())
                {
                    defAudioSrc->vSetSourceBG(FALSE);
                }
            }
            else
            {
                ETG_TRACE_ERR(("CMD_AUDIO, default source also not available -> rejecting disconnect"));
            }
        }
      }
      else
    {
      bSuccess = false;
      //#### allow disconnect of Entertain for other sink (e.g. RSE) ####
      bSuccess = clAudioSourceController::getInstance().bOFF(srcID, sinkID, u32UserData);
      //#### allow disconnect end ####
    }
   }
   return bSuccess;
}
/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOn
*
* DESCRIPTION:    Activate an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is started with force-option, i.e. a play
*                 request is sent to FC-Audio without any prio check and
*                 without any media-available check!
*                 The normal AudioStack is frozen.
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to start
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOn(tU8 u8AudioSource, tBool bForce)
{
   (void) u8AudioSource;
   (void) bForce;
   ETG_TRACE_ERR(("CMD_AUDIO, vDiagnosisAudioSourceOn: actually not supported"));
}

/*************************************************************************
* METHOD:         vDiagnosisAudioSourceOff
*
* DESCRIPTION:    Stop an AudioSource for Diagnosis/RemoteControl
*                 AudioSource is stopped with force-option, i.e. a stop
*                 request is sent to FC-Audio without any prio check!
* PARAMETER:      tU8 u8AudioSource    the Diagnosis Audio source to stop
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisAudioSourceOff(tU8 u8AudioSource, tBool bForce)
{
  (void) u8AudioSource;
  (void) bForce;
   ETG_TRACE_ERR(("CMD_AUDIO, vDiagnosisAudioSourceOff: actually not supported"));
}

/*************************************************************************
* METHOD:         vDiagnosisOff
*
* DESCRIPTION:    Stop Diagnosis Mode
*                 If a Diagnosis AudioSource is currently active, it is
*                 stopped.
*                 The normal Audio Stack is recovered.
* PARAMETER:
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDiagnosisOff()
{
   ETG_TRACE_ERR(("CMD_AUDIO, vDiagnosisOff: actually not supported"));
}

/*************************************************************************
* METHOD:         bIsSystemFree
*
* DESCRIPTION:    Asks CM-System, if PIN-Code Lock is active
* PARAMETER:
*
* RETURNVALUE:    tBool: True, if PIN-Code is unlocked, False otherwise
*
************************************************************************/
tBool clAudioSMEngine::bIsSystemFree()
{
   ETG_TRACE_USR3(("bIsSystemFree: TRUE"));
   return TRUE;
}

/*************************************************************************
* METHOD:         bIsEntertainmentSourceAvailable
*
* DESCRIPTION:    there are 2 types of Entertainment sources: Radio and Media
*                 Radio sources are available always.
*                 Media sources availability is given by bIsMediaSourceAvailable()
*
* PARAMETER:       u8AudioSource
*
* RETURNVALUE:     true, if the specified Entertainment source is available
*
*
************************************************************************/
tBool clAudioSMEngine::bIsSourceAvailable(SourceID audioSource,tBool bCreateNewSource)
{
   tBool retVal = FALSE;
   clAudioSource* pAudSrc = clAudioSourceFactory::getAudioSource(audioSource,bCreateNewSource);
   if(pAudSrc != NULL)
   {
      if((pAudSrc->enIsSourceAvailable() == clAudioSource::available)
            ||(pAudSrc->enIsSourceAvailable() == clAudioSource::unkonwn))
      {
         retVal = TRUE;
      }
      ETG_TRACE_USR4(("bIsSourceAvailable: SourceClass: %d subID %d, Availability: %d"
                  , audioSource.enSourceClass
                  , audioSource.u16SubSource
                  , ETG_CENUM(clAudioSource::enSourceAvailability, pAudSrc->enIsSourceAvailable())));
   }else{
      ETG_TRACE_USR4(("bIsSourceAvailable: SourceClass: %d subID %d not available "
                  , audioSource.enSourceClass
                  , audioSource.u16SubSource));
   }

   return retVal;
}

/*************************************************************************
* METHOD:         vSetRunlevel
*
* DESCRIPTION:    This is the entrypoint for CM-Startup, which controls
*                 the powerstates and runlevels of the AudioSystem via
*                 this method.
*                 --------------------------------------------------------
*                 Runlevel 0:
*                    - Audio is muted.
*                    - AudioStack is reseted
*                    - The virtual Source: System-Mute is on stack. This
*                    ensures the audio system to remain muted, even if
*                    source-switch requests are applied.
*                    - AudioSource-ON() is allowed. This is required to enable
*                    early Source-Switch requests. The requested sources
*                    are put to the stack but not yet started
*                 --------------------------------------------------------
*                 Runlevel 1:
*                    - This is the runlevel for limited Phone mode
*                    - All Audiosources except phone are muted
*                 --------------------------------------------------------
*                 Runlevel 2:
*                    - This is the runlevel for limited Entertainment audio
*                    - All Radio Sources are enabled
*                    - All Media Sources are still disabled
*                 --------------------------------------------------------
*                 Runlevel 3:
*                    - This is the runlevel for full audio
*                    - All Audio Sources are enabled
*
*
* PARAMETER:      tU8 u8NewRunlevel
*
* RETURNVALUE:    -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::SetRunlevel(tU8 u8NewRunlevel, tBool bColdStart)
{
   clAudioSMEngine::enErrorCode retVal = clAudioSMEngine::AUDIOSM_OK;
   tU8 u8OldRunlevel = m_u8Runlevel;
   //register static AudioSources at first
   if (u8OldRunlevel == 0 && u8NewRunlevel > 0)
   {
      // lets try this first (need this now more earlier)
      (void)clAudioSourceController::getInstance();

      if(!clAudioSourceFactory::Init())
      {
         ETG_TRACE_ERR(("CMD_AUDIO, SetRunlevel: ERROR Initializing AudioSource Factory"));
         retVal = clAudioSMEngine::AUDIOSM_ERROR;
         return retVal;
      }
   }

   m_u8Runlevel = u8NewRunlevel;

   tU8 au8Data[2]={u8OldRunlevel, u8NewRunlevel};
   ETG_TRACE_USR1(("vSetRunlevel: Old Runlevel: %d New Runlevel: %d"
         ,au8Data[0]
         ,au8Data[1]));

   switch (u8NewRunlevel)
   {
   case 0:
      {
         if (u8OldRunlevel > 0)
         {
            // Reset Stack
            vResetStack();
//            AudioSourceOn(SourceID(AudioSources::MUTE_SYSTEM,0));
//            AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 1:
      {
         if (u8OldRunlevel < 1)
         {
            // enable limited phone audio
//            vAudioSourceOff(SourceID(AudioSources::MUTE_PIN, 0));
//            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         else if (u8OldRunlevel > 1)
         {
            // No Entertainment in runlevel 1. Only Phone.
            //vResetEntertainmentsource();
//            AudioSourceOn(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 2:
      {
         if (u8OldRunlevel < 1)
         {
//            vAudioSourceOff(SourceID(AudioSources::MUTE_SYSTEM,0));
         }
         if (u8OldRunlevel < 2)
         {
            // Restore Entertainmentsource (limited to radio) and persistent Pause-state
            retVal = vRestoreEntertainmentsource();
            vRestorePersistentMuteState();
//            vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));
         }
      }
      break;
   case 3:
      {
         if (u8OldRunlevel < 3)
         {
            time(&m_stInitTime);

      clGeniviAudioCtrlAdapter::vSetAmplifierMute(MUTE_NO_AUD_DEV_ON);
      clGeniviAudioCtrlAdapter::vSetMuteOnStartUp(SYP_HMISTARTUP_MUTE_ON,bColdStart);

      ETG_TRACE_USR1(("In runlevel case 3 :Before Pushing welcome sound on startup"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
      tU8 u8VariantInfo = 0;
      if((DP_S32_NO_ERR == DP_s32GetConfigItem("CMVariantCoding", "OEMType", &u8VariantInfo,1)))
      {
          if(0x08 == u8VariantInfo)  //check if OEM type is Renault
          {
      ETG_TRACE_USR1(("After Pushing welcome sound on startup"));
      clGeniviAudioCtrlAdapter::vSetWelcomeMuteOnStartUp(bColdStart);
          }
      }
#endif
            // Restore Entertainmentsource (radio and media)and persistent Pause-state
            retVal = vRestoreEntertainmentsource();
            vRestorePersistentMuteState();
            const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
            if(srcClass != NULL)
            {
              sourceClassID muteClassID = srcClass->getClassID();
              vAudioSourceOff(SourceID(muteClassID,0));
            }
            else
            {
              //Error, no MUTE_ZERO_VOLUME
              ETG_TRACE_ERR(("CMD_AUDIO, AudioSourceClass MUTE_ZERO_VOLUME not found !!!"));
            }
//          vAudioSourceOff(SourceID(AudioSources::MUTE_ZERO_VOLUME,0));
//          vAudioSourceOff(SourceID(AudioSources::MUTE_LIMITED_SYSTEM,0));

      //Get AmpAvailability & remove MUTE_NO_AUD_DEV_ON if Present
      if(true == clGeniviAudioCtrlAdapter::bAudioDeviceAvailability())
      {
        ETG_TRACE_USR1(("SetRunlevel, AudioDeviceAvailability is TRUE, Removing MUTE_NO_AUD_DEV"));
        const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_NO_AUD_DEV");
        if(srcClass != NULL)
        {
         sourceClassID muteClassID = srcClass->getClassID();
                 vAudioSourceOff(SourceID(muteClassID,0));
        }
      }
         }
      }
      break;
   default:
      // invalid runlevel
      break;
   }
   return retVal;
}

tU8 clAudioSMEngine::GetRunlevel()
{
   return m_u8Runlevel;
}

tVoid clAudioSMEngine::ListSources()
{
   clAudioSourceFactory::ListSources();
}


tVoid clAudioSMEngine::vResetStack()
{
   ETG_TRACE_ERR(("CMD_AUDIO, vResetStack called without sink -> using sink 1"));
   clAudioSMEngine::vResetStack(1);
}

tVoid clAudioSMEngine::vResetStack( int sinkID)
{

   clAudioSourceController::getInstance().vResetStack(sinkID);
   // place additionally to a clean stack a Pin-Mute to ensure Pin-lock before
   // XML-Start, because XML locks the PIN-lock too late at startup.
   //TODO check this needed
   //clAudioSourceController::getInstance().bON(SourceID(AudioSources::MUTE_PIN,0), 0);
}

bool clAudioSMEngine::bIsPersistentMuteState()
{
   ETG_TRACE_USR4(("entered bIsPersistentMuteState"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
   dp_tclAudioStackUsrDPAudioStack_Usr_PersistentMuteAudioOff   oPersMute;
   tU8 u8PersMute = 0;
   tS32 ret = oPersMute.s32GetData(u8PersMute);
   ETG_TRACE_USR2(("bIsPersistentMuteState is %d (ret=%x)",(tU16)u8PersMute,ret));
   if(u8PersMute != 0)
   {
     return true;
   }
   return false;
#else
   return false;
#endif
}

/*************************************************************************
* METHOD:         vRestorePersistentMuteState
*
* DESCRIPTION:    Recovers the Persistent MuteState
*                 The Persistent MuteState is the user-requested Mute
*                 which pauses Entertainmentsources. (MFL, HK-Phone, ...)
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vRestorePersistentMuteState()
{
   ETG_TRACE_USR4(("vRestorePersistentMuteState not for all variants used"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA)
   dp_tclAudioStackDPAudioStack_PersistentMute   oPersMute;
   tU8 u8PersMute = 0;
   tS32 ret = oPersMute.s32GetData(u8PersMute);
   ETG_TRACE_USR2(("vRestorePersistentMuteState is %d (ret=%x)",(tU16)u8PersMute,ret));
   if(u8PersMute != 0)
   {
     const clSourceClass* pSrcClass_EntMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_ENTERTAIN");
     if(pSrcClass_EntMute)
     {
       ETG_TRACE_USR4(("vRestorePersistentMuteState: Set MUTE_ENTERTAIN"));
     //AudioSourceOn(SourceID(pSrcClass_EntMute->getClassID(),0));
     // daw2hi: no more with defaults (RSE)
     AudioSourceOn(SourceID(pSrcClass_EntMute->getClassID(),0),1,false, 0);
       //need to inform HMI about the Mute
       clGeniviAudioCtrlAdapter::vSetPersistentMute();
     }
   }
#elif defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
   dp_tclAudioStackUsrDPAudioStack_Usr_PersistentMuteAudioOff   oPersMute;
   tU8 u8PersMute = 0;
   tS32 ret = oPersMute.s32GetData(u8PersMute);
   ETG_TRACE_USR4(("vRestorePersistentMuteState is %d (ret=%x)",(tU16)u8PersMute,ret));
   const clSourceClass* pSrcClass_EntMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_AUDIO_OFF");
   if(u8PersMute != 0)
   {
     if(pSrcClass_EntMute)
     {
       ETG_TRACE_USR1(("vRestorePersistentMuteState: Set MUTE_AUDIO_OFF"));
       AudioSourceOn(SourceID(pSrcClass_EntMute->getClassID(),0),1,false, 0);
       clGeniviAudioCtrlAdapter::vSetPersistentMute();
     }
   }
   else
   {
       if(pSrcClass_EntMute)
        {
          ETG_TRACE_USR1(("vRestorePersistentMuteState: Set MUTE_AUDIO_OFF off"));
          vAudioSourceOff(SourceID(pSrcClass_EntMute->getClassID(),0));
        }
       clGeniviAudioCtrlAdapter::updateSoundPropertyDB(MSP_AUDIO_OFF_MUTE,AM_MSP_AUDIO_OFF_MUTE_OFF);
   }
#endif
}

time_t  clAudioSMEngine::GetTime_Init()
{
   return m_stInitTime;
}

tVoid clAudioSMEngine::vResetTime_Init()
{
  // Reset the reference Time
  time(&m_stInitTime);
  ETG_TRACE_USR1(("Reset Time_Init to %s",ctime(&m_stInitTime)));
}

/*
 * Function to check if user specific data pool should be loaded for Personalization feature
 * Currently Personalization feature is required only for PIVI
 * Checking OEM type is 'Infiniti' then it's PIVI specific target
 */
bool clAudioSMEngine::bLoadUsrProfileDataPool()
{
    ETG_TRACE_USR4(("Entered bLoadUsrProfileDataPool"));
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
    ETG_TRACE_USR4(("bLoadUsrProfileDataPool: Device is IVI"));
    return true;
#else
    return false;
#endif
}

clAudioSMEngine::enErrorCode clAudioSMEngine::vRestoreUserProfileEntsource(tBool bIsInitialProfileChange)
{
   ETG_TRACE_USR4(("clAudioSMEngine::vRestoreUserProfileEntsource, has user initiated profile change for first time(%d)", bIsInitialProfileChange));

   tU8 lastSourceClass = 0;
   tU16 u16SubID = 0;

   //Load last source from data pool
   vReadDataPool(lastSourceClass, u16SubID);
   SourceID lastSource (lastSourceClass, u16SubID);
   clAudioSource* pAudSrc = clAudioSourceFactory::isSourceInstantiated(lastSource);
   
   if(pAudSrc != NULL && !pAudSrc->bIsSourceLSM() && !pAudSrc->bIsSourceProfChg())
   {
      //Process the requested source by user which is AVAILABLE
      vProcessRegisteredUserProfileSrc(pAudSrc, lastSource);
   }
   else
   {
      if(bIsInitialProfileChange == TRUE) //This is the first user profile change when source is unavailable
      {
        /**
         * Process the requested source when user has selected the profile for first time,
         * A timer with corresponding timeout value must be started in this context.
         */
         vProcessUnregisteredUserProfileSrc(lastSource);
      }
     else
      {
         ETG_TRACE_USR3(("vRestoreUserProfileEntsource: Restore Default source since source is unavailable and it's NOT the initial profile change "));
         lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
         lastSource.u16SubSource = 0;
        
         if (!bIsSourceAvailable(lastSource))
         {
            ETG_TRACE_USR4(("vRestoreUserProfileEntsource: Restore Default source due to availability" ));
            lastSource = GetNextAvailableEntertainmentSource(
            SourceID(clFactory_AudioSourceClass::GetDefaultSourceClass(),0));
         }

         ETG_TRACE_USR2(("vRestoreUserProfileEntsource: Adding MainConnection SourceClass: %d subID %d"
                          , lastSource.enSourceClass
                          , lastSource.u16SubSource));
         clAudioSource* pGSrc =  clAudioSourceFactory::getAudioSource(lastSource);
         if(pGSrc == NULL )
         {
            ETG_TRACE_ERR(("CMD_AUDIO, vRestoreUserProfileEntsource: No AudioSource Object found"));
            return AUDIOSM_ERROR;
         }

         AudioSourceOn(lastSource,1,false, 0);
      }
   }
   return AUDIOSM_OK;
}

/*************************************************************************

* METHOD:         vProcessRegisteredUserProfileSrc
*
* DESCRIPTION:    Process the requested source on user profile change when AVAILABLE
* 
* PARAMETER:      Instance of source requested by user
*                 SourceID requested by user

* RETURNVALUE:    tvoid
*
************************************************************************/
tVoid clAudioSMEngine::vProcessRegisteredUserProfileSrc(AudioSource::clAudioSource* poAudioSource, SourceID lastSource)
{
   ETG_TRACE_USR4(("clAudioSMEngine::vProcessRegisteredUserProfileSrc, SourceID.enSourceClass(%d) and SourceID.u16SubSource(%d)", lastSource.enSourceClass, lastSource.u16SubSource));
   
   if((lastSource.enSourceClass == 0)||
      ((poAudioSource != NULL) && (poAudioSource->getSourceType().name != "entertain")))
   {
     ETG_TRACE_USR4(("vProcessRegisteredUserProfileSrc: Restoring to default" ));

     lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
     lastSource.u16SubSource = 0;
   }

   //NCG3D:273838: If Mute_audio_Off is ON in stack do background source change
   poAudioSource = clAudioSourceFactory::isSourceInstantiated(lastSource);

   if(poAudioSource != nullptr)
   {
      if(clAudioSMEngine::bIsMuteActive())
          poAudioSource->vSetSourceBG(true);

      AudioSourceOn(lastSource, 1, false, 0);

      if(poAudioSource->bIsSourceBG())
          poAudioSource->vSetSourceBG(false);
   }
   else
   {
       ETG_TRACE_ERR(("CMD_AUDIO, clAudioSMEngine::vProcessRegisteredUserProfileSrc, poAudioSource is nullptr"));
   }
}


/*************************************************************************
* METHOD:         vProcessUnregisteredUserProfileSrc
*
* DESCRIPTION:    Process the requested source on user profile change when UNAVAILABLE
* 
* PARAMETER:      Instance of source requested by user
*                 SourceID requested by user
* RETURNVALUE:    tvoid
*
************************************************************************/
tVoid clAudioSMEngine::vProcessUnregisteredUserProfileSrc(SourceID lastSource)
{
   ETG_TRACE_USR4(("clAudioSMEngine::vProcessUnregisteredUserProfileSrc, SourceID.enSourceClass(%d) and SourceID.u16SubSource(%d)", lastSource.enSourceClass, lastSource.u16SubSource));

   ETG_TRACE_USR4(("vProcessUnregisteredUserProfileSrc: pushing source to stack as UNKNOWN" ));
   //source is not there yet, we create it with unknown
   //availability
   //source will itself check if timeout occurs or already tiemd out
   //then, if it is in unknown state, it will mark itself as unavailable
   clAudioSource* poAudioSource = clAudioSourceFactory::getAudioSource(lastSource);
   if(poAudioSource != NULL)
   {
      if(poAudioSource->bIsSourceLSM())
         poAudioSource->vSetSourceLSM(false);

      poAudioSource->vSetSourceProfChg(true);
      poAudioSource->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);

      ETG_TRACE_USR4(("clAudioSMEngine::vProcessUnregisteredUserProfileSrc: poAudioSource != nullptr" ));
      if(clAudioSMEngine::bIsMuteActive())
         poAudioSource->vSetSourceBG(true);

      ETG_TRACE_USR4(("clAudioSMEngine::vProcessUnregisteredUserProfileSrc: AudioSourceOn" ));
      AudioSourceOn(lastSource, 1, false, 0);

      if(poAudioSource->bIsSourceBG())
         poAudioSource->vSetSourceBG(false);
   }
   else
   {
      ETG_TRACE_ERR(("CMD_AUDIO, clAudioSMEngine::vProcessUnregisteredUserProfileSrc, poAudioSource is nullptr"));
   }
}

/*************************************************************************
* METHOD:         vRestoreEntertainmentsource
*
* DESCRIPTION:    Recovers the LSM Entertainmentsource
*                 If runlevel is < 3, a Media-LSM will not be started
* PARAMETER:      -
*
* RETURNVALUE:    -
*
************************************************************************/
clAudioSMEngine::enErrorCode clAudioSMEngine::vRestoreEntertainmentsource(bool bUserProfileChanged)
{
   ETG_TRACE_USR4(("vRestoreEntertainmentsource:" ));
   time_t TheTimeIsNow;
   time(&TheTimeIsNow);
   tDouble SecSinceRunlvl3 = difftime(TheTimeIsNow,m_stInitTime);
   ETG_TRACE_USR4(("vRestoreEntertainmentsource: Seconds since Runlevel 3: %f",(etg_tF32)SecSinceRunlvl3 ));

   DP_vCreateDatapool();

   tU8 lastSourceClass = 0;
   tU16 u16SubID = 0;

   //Load last source from data pool
  #ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
    vReadLastTunerDataPool();
  #endif
   vReadDataPool(lastSourceClass,u16SubID);

   ETG_TRACE_USR2(("vRestoreEntertainmentsource: Read from DP SourceClass: %d subID %d", lastSourceClass, u16SubID));

   SourceID lastSource (lastSourceClass,u16SubID);

   m_enSource_Entertainment_LSM = lastSource;

   clAudioSource* pAudSrc = clAudioSourceFactory::isSourceInstantiated(lastSource);
   //Check if last source is already there
   if(bUserProfileChanged == false)
   {
   	if(clAudioSourceFactory::isSourceInstantiated(lastSource) == NULL)
   	{
      	  //source is not there yet, we create it with unkown
          //availability
          //source will itself check if timeout occurs or already tiemd out
          //then, if it is in unkown state, it will mark itself as unavailable
          pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);
          if(pAudSrc != NULL)
          {
             pAudSrc->vSetSourceLSM(true);
             pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
          }
        }
        else
        {
     	  // if we have not a fresh startup (==source exists), we should still do this for dynamic sources
          // we might execute this also for critical low volt and short ACC off/on
     	  pAudSrc = clAudioSourceFactory::getAudioSource(lastSource);
          // we should have found the source and will check now if it is dynamic
          if((pAudSrc!=NULL) && (pAudSrc->getSourceClass().getRegistrationMode() == clSourceClass::registerDynamic))
          {
            pAudSrc->vSetSourceLSM(true);
            pAudSrc->vSourceAvailablilityChange(clAudioSource::unkonwn, clAudioSource::samemedia);
          }
        }
   }

   if((pAudSrc!=NULL) &&
   (bUserProfileChanged == false) &&
   (pAudSrc->getRestoreValue() != clSourceClass::startupAndProfileChange))
   {
       vDefaultSource(lastSource);
       lastSourceClass = lastSource.enSourceClass;
       ETG_TRACE_USR3(("vRestoreEntertainmentsource: available source not restorable on startup, Restoring to default"));
   }

   if((pAudSrc!=NULL) && (pAudSrc->getSourceType().name != "entertain"))
   {
     ETG_TRACE_USR4(("vRestoreEntertainmentsource: Resored source Not Entertainment Source, Restoring to default" ));

   vDefaultSource(lastSource);

     lastSourceClass = lastSource.enSourceClass;
   }
   else
   {
     ETG_TRACE_USR4(("vRestoreEntertainmentsource: Resored source Entertainment Source" ));
   }


   if(lastSourceClass == 0)
   {
      ETG_TRACE_USR3(("vRestoreEntertainmentsource: Restore Default source due to default value of datapool" ));
      vDefaultSource(lastSource);
   }

   if (!bIsSourceAvailable(lastSource))
   {
      ETG_TRACE_USR4(("vRestoreEntertainmentsource: Restore Default source due to availability" ));
      lastSource = GetNextAvailableEntertainmentSource(
            SourceID(clFactory_AudioSourceClass::GetDefaultSourceClass(),0));
//      lastSource.enSourceClass = AudioSources::TUNER_FM;
//      lastSource.u16SubSource = 0;
   }

   ETG_TRACE_USR2(("vRestoreEntertainmentsource: Adding MainConnection SourceClass: %d subID %d"
         , lastSource.enSourceClass
         , lastSource.u16SubSource));
   clAudioSource* pGSrc =  clAudioSourceFactory::getAudioSource(lastSource);
   if(pGSrc == NULL )
   {
      ETG_TRACE_ERR(("CMD_AUDIO, vRestoreEntertainmentsource: No AudioSource Object found"));
      return AUDIOSM_ERROR;
   }

   if(clAudioSMEngine::bIsMuteActive())
        pGSrc->vSetSourceBG(true);

   AudioSourceOn(lastSource,1,false, 0);

   if(pGSrc->bIsSourceBG())
        pGSrc->vSetSourceBG(false);

   return AUDIOSM_OK;
}

tVoid clAudioSMEngine::vDefaultSource(SourceID& lastSource)
{
#ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
  lastSource = m_enSource_Entertainment_Last_Tuner;
#else
    lastSource.enSourceClass = clFactory_AudioSourceClass::GetDefaultSourceClass();
    lastSource.u16SubSource = 0;
#endif

}
/************************************************************************
 * FUNCTION     : u8GetSrcGroupID_Previous
 * DESCRIPTION  : function to get previous source group id
 * PARAMETER    : NIL
 * RETURNVALUE  : clSourceClass::sourceGroupID
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
clSourceClass::sourceGroupID clAudioSMEngine::u8GetSrcGroupID_Previous()
{
   return (clFactory_AudioSourceClass::GetSourceClass(m_enSource_Previous_Active.enSourceClass).getGroupID());
}

/************************************************************************
 * FUNCTION     : u8GetSrcGroupID_Current
 * DESCRIPTION  : function to get current source group id
 * PARAMETER    : NIL
 * RETURNVALUE  : clSourceClass::sourceGroupID
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
clSourceClass::sourceGroupID clAudioSMEngine::u8GetSrcGroupID_Current()
{
  return (clFactory_AudioSourceClass::GetSourceClass(m_enSource_Current_Active.enSourceClass).getGroupID());
}

/************************************************************************
 * FUNCTION     : vSetCurrentActiveSource
 * DESCRIPTION  : function to set current active source and previous active source
 * PARAMETER    : AudioSource::SourceID
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              04.01.16 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vSetCurrentActiveSource(AudioSource::SourceID srcID)
{
  if(m_enSource_Current_Active != srcID)
  {
    m_enSource_Previous_Active = m_enSource_Current_Active;
    m_enSource_Current_Active = srcID;

    ETG_TRACE_USR2(("vSetCurrentActiveSource(): updated m_enSource_Current_Active, SourceClass: %d subID %d", srcID.enSourceClass, srcID.u16SubSource));
    ETG_TRACE_USR2(("vSetCurrentActiveSource(): updated m_enSource_Previous_Active, SourceClass: %d subID %d", m_enSource_Previous_Active.enSourceClass, m_enSource_Previous_Active.u16SubSource));
  }
}

/************************************************************************
 * FUNCTION     : vReadDataPool
 * DESCRIPTION  : function to read Last source from data pool
 * PARAMETER    : tU8& , tU16&
 * RETURNVALUE  : bool
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vReadDataPool(tU8& u8SrcID, tU16& u16SubSrcID)
{
    ETG_TRACE_USR4(("vReadDataPool: entered"));
    if (bLoadUsrProfileDataPool())
    {
        ETG_TRACE_USR2(("vReadDataPool: Reading last source from multi user data pool for Personalization"));

        //Get SourceClass
        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_Class oLastSource_Usr_Class;
        oLastSource_Usr_Class.u8GetData(u8SrcID);

        //Get SubID
        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_SubID oLastSource_Usr_SubID;
        oLastSource_Usr_SubID.s32GetData(u16SubSrcID);
    }
    else
    {
        ETG_TRACE_USR4(("vReadDataPool: Reading last source"));

        //Get SourceClass
        dp_tclAudioStackDPAudioStack_LastSource_Class oLastSource_Class;
        oLastSource_Class.u8GetData(u8SrcID);

        //Get SubID
        dp_tclAudioStackDPAudioStack_LastSource_SubID oLastSource_SubID;
        oLastSource_SubID.s32GetData(u16SubSrcID);
    }

    ETG_TRACE_USR2(("vReadDataPool: Loaded Last source from data pool SourceClass: %d,subID: %d", u8SrcID, u16SubSrcID));
}
#ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
/************************************************************************
 * FUNCTION     : vReadLastTunerDataPool
 * DESCRIPTION  : function to read Last tuner source from data pool
 * PARAMETER    : void
 * RETURNVALUE  : void
 * HISTORY      :
 *              5.01.18 Swati  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vReadLastTunerDataPool()
{
    ETG_TRACE_USR1(("vReadLastTunerDataPool: entered"));
    //Get SourceClass
    dp_tclAudioStackDPAudioStack_LastTunerSource_Class oLastSource_Class;
    oLastSource_Class.u8GetData(m_enSource_Entertainment_Last_Tuner.enSourceClass);

    //Get SubID
    dp_tclAudioStackDPAudioStack_LastTunerSource_SubID oLastSource_SubID;
    oLastSource_SubID.s32GetData(m_enSource_Entertainment_Last_Tuner.u16SubSource);
    ETG_TRACE_USR4(("vReadLastTunerDataPool: Loaded Last source from data pool SourceClass: %d,subID: %d", m_enSource_Entertainment_Last_Tuner.enSourceClass, m_enSource_Entertainment_Last_Tuner.u16SubSource));
}
#endif
/************************************************************************
 * FUNCTION     : vWriteDataPool
 * DESCRIPTION  : function to update Last source into data pool
 * PARAMETER    : tVoid
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vWriteDataPool(bool bUserProfileChanged)
{
    ETG_TRACE_USR4(("vWriteDataPool: entered"));

    vWriteDataPool(m_enSource_Entertainment_Active.enSourceClass,m_enSource_Entertainment_Active.u16SubSource,bUserProfileChanged);
}
#ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
/************************************************************************
 * FUNCTION     : vWriteLastTunerDataPool
 * DESCRIPTION  : function to update Last source into data pool
 * PARAMETER    : tU8& , tU16&
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vWriteLastTunerDataPool(tU8& u8SrcID, tU16& u16SubSrcID, bool bUserProfileChanged)
{
    ETG_TRACE_USR4(("vWriteLastTunerDataPool: entered, SourceClass: %d,subID: %d",u8SrcID,u16SubSrcID));
    dp_tclAudioStackDPAudioStack_LastTunerSource_Class oLastSource_Class;
    oLastSource_Class.vSetData(u8SrcID);

    dp_tclAudioStackDPAudioStack_LastTunerSource_SubID oLastSource_SubID;
    oLastSource_SubID.vSetData(u16SubSrcID);
}
#endif
/************************************************************************
 * FUNCTION     : vWriteDataPool
 * DESCRIPTION  : function to update Last source into data pool
 * PARAMETER    : tU8& , tU16&
 * RETURNVALUE  : tVoid
 * HISTORY      :
 *              12.09.17 Vyankatesh VD  Initial Revision.
 ************************************************************************/
tVoid clAudioSMEngine::vWriteDataPool(tU8& u8SrcID, tU16& u16SubSrcID, bool bUserProfileChanged)
{
    ETG_TRACE_USR2(("vWriteDataPool: entered, SourceClass: %d,subID: %d",u8SrcID,u16SubSrcID));

    if (bLoadUsrProfileDataPool())
    {
        ETG_TRACE_USR3(("vWriteDataPool: Updating last source to multi user data pool for Personalization"));

        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_Class oLastSource_Usr_Class;
        oLastSource_Usr_Class.vSetData(u8SrcID);

        dp_tclAudioStackUsrDPAudioStack_Usr_LastSource_SubID oLastSource_Usr_SubID;
        oLastSource_Usr_SubID.vSetData(u16SubSrcID);
    }
    else if(!bUserProfileChanged)
    {
        ETG_TRACE_USR3(("vWriteDataPool: Updating data pool with last source"));

        dp_tclAudioStackDPAudioStack_LastSource_Class oLastSource_Class;
        oLastSource_Class.vSetData(u8SrcID);

        dp_tclAudioStackDPAudioStack_LastSource_SubID oLastSource_SubID;
        oLastSource_SubID.vSetData(u16SubSrcID);
    }
}

/*************************************************************************
* METHOD:         vDPPersistentMute
*
* DESCRIPTION:    Sets or Resets the persistenet mute based on some conditions
*
* PARAMETER:      poAudioSource: only if it is MUTE_ENTERTAIN, else no action
*                 action: set or reset for PersMute
*
* RETURNVALUE:    -
*
************************************************************************/
tVoid clAudioSMEngine::vDPPersistentMute(AudioSource::clAudioSource* poAudioSource, enPersMuteDPAction action)
{
#ifndef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI
  ETG_TRACE_USR3(("vDPPersistentMute():Entered in vDPPersistentMute"));
  //Check if Diag Session is active
  if(true == clAudioSourceController::getInstance().bIsTopOfBackUpStackMuteAndOn())
  {
    ETG_TRACE_USR4(("vDPPersistentMute():Diagnosis session is active, No need to update Persistent Mute in datapool !"));
    return;
  }

  if(0 == OSAL_s32StringCompare(poAudioSource->pacGetName(),"MUTE_ENTERTAIN"))
  {
    //we store this persistent

    dp_tclAudioStackDPAudioStack_PersistentMute   oPersMute;
    if(action == PersMuteDP_Reset)
    {
      ETG_TRACE_USR3(("Reset Persistent Mute source MUTE_ENTERTAIN"));
      (void)oPersMute.s32SetData(0);
    }
    else
    {
      // to avoid persistency of non user action MUTE_ENTERTAIN during shutdown
      // check if MUTE_SYSTEM is on stack and if active the MUTE_ENTERTAIN was not triggerd by the user
      const clSourceClass* pSrcClass_SysMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_SYSTEM");
      if(pSrcClass_SysMute==NULL)
      {
        ETG_TRACE_ERR(("CMD_AUDIO, Audio Source Class MUTE_SYSTEM not found"));
        return;
      }
      if(clGeniviAudioCtrlAdapter::bIsCurrentActiveAudiosource(SourceID(pSrcClass_SysMute->getClassID(),0)))
      {
        ETG_TRACE_USR4(("MUTE_SYSTEM is active -> MUTE_ENTERTAIN not triggered by user, no persistent storage"));
      }
      else
      {
        ETG_TRACE_USR3(("MUTE_SYSTEM not active > MUTE_ENTERTAIN triggered by user, persistent storage"));
        (void)oPersMute.s32SetData(1);
      }
    }
  }
#else
  if(0 == OSAL_s32StringCompare(poAudioSource->pacGetName(),"MUTE_AUDIO_OFF"))
  {
    //we store MUTE_AUDIO_OFF persistent
    dp_tclAudioStackUsrDPAudioStack_Usr_PersistentMuteAudioOff   oPersMute;
    ETG_TRACE_USR2(("Persistent Mute source MUTE_AUDIO_OFF : %d",action));
    (void)oPersMute.s32SetData(action);
    ETG_TRACE_USR4(("vDPPersistentMute need to update MUTE_AUDIO_OFF"));
    clGeniviAudioCtrlAdapter::updateSoundPropertyDB(MSP_AUDIO_OFF_MUTE,action);
   }
#endif
}

tBool clAudioSMEngine::bIsMuteActive()
{
    //check if mute is active
    const clSourceClass* pSrcClass_MuteAudioOff { clFactory_AudioSourceClass::GetSourceClass("MUTE_AUDIO_OFF") };
    bool bIsMuteActive { false };
    if(pSrcClass_MuteAudioOff)
    {
        clAudioSource* poMuteSource { nullptr };
        clStack* m_pcoAudioStack { clAudioSourceController::getInstance().getStackForSink(1) };

        if(m_pcoAudioStack)
          poMuteSource = m_pcoAudioStack->GetTopSource(pSrcClass_MuteAudioOff->SourceClassID);

        if(poMuteSource)
            bIsMuteActive = true;
    }

    ETG_TRACE_USR4(("clAudioSMEngine::bIsMuteActive MuteStatus %d",bIsMuteActive));
    return bIsMuteActive;
}
}//namespace
