/*!************************************************************************
 * FILE :         clAudioSourceController.cpp
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * The main task of AudioSourceController is to perform source-switches
 * by activating and deactivating AudioSources.
 * It handles Priority checks by applying Stack-Rules, defined in clStackRules.cpp
 * AudioSources are kept in a stack, where the priority queue is realised.
 *
 * The typical straight forward approach for a source-switch is:
 *
 * - Client requests an AudioSource to be played
 * - Controller adds the requested AudioSource to the stack, applying
 *   priority rules etc.
 * - If the new AudioSource is on top of the stack, it has to be started
 * - Before, the previous active source has to be stopped.
 * - So the sequence is:
 *   - Start to mute the current active source and wait until the mute-ramp
 *   for this source has finished.
 *   - Then start the new source
 *
 * Things become a littel bit more complicated, when mix sources have to
 * be added, because the means, that more than on source can be active
 * at the same time.
 *
 * So the advanced approach is:
 * - Client requests an AudioSource to be played
 * - AudioSourceController stopps all sources, which should not be active
 * after applying the StackRules.
 * - Wait until all stopping source have finished the mute-ramp.
 * - AudioSourceController starts all sources, which should be active after
 * applying the StackRules.
 *
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *
 *************************************************************************/
#include <stdlib.h>

#include "AudioStack/AudioSources/clAudioSourceController.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"

#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/cliAudioSourceControllerObserver.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"
#include "include/audiomanagertypes.h"
#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/clStack.h"
#include "AudioStack/clStackRules.h"
#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSourceController.cpp.trc.h"
#endif

namespace AudioStack
{

using namespace SourceStateMachine;
namespace AudioSource
{
//Init static Members
clStack*                          clAudioSourceController::m_pcoAudioStack    = NULL;
clStack*                          clAudioSourceController::m_pcoBackupStack   = NULL;

clIAudioSourceControllerObserver* clAudioSourceController::m_poObserver       = NULL;
clAudioSourceController*          clAudioSourceController::m_poSelf_Singleton = NULL;

std::vector<clStack*> clAudioSourceController::m_allAudioStacks;

tVoid clAudioSourceController::vInvalidate()
{
   // loop through vector and clean up
   std::vector<clStack*>::iterator it;
   for(it=m_allAudioStacks.begin(); it != m_allAudioStacks.end(); ++it)
   {
     delete(*it);
   }
   m_allAudioStacks.clear();
   m_pcoAudioStack    = NULL;
   m_poObserver       = NULL;
   m_poSelf_Singleton = NULL;
   delete this;
}

clAudioSourceController& clAudioSourceController::getInstance()
{
   if(m_poSelf_Singleton == NULL)
   {
      m_poSelf_Singleton = OSAL_NEW clAudioSourceController();
      clFactory_AudioSourceClass::vSetAudioSourceControllerCallback(clAudioSourceController::m_poSelf_Singleton);
   }
   return *m_poSelf_Singleton;
}

/*!************************************************************************
 * METHOD:        constructor
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   constructor
 *                Creates a new AudioStack and initializes it with the
 *                NONE Source. The NONE Source is a permanent dummy background
 *                source which makes most Stack operations easier.
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:      (CM-DI/PJ-VW34)
 *************************************************************************/
clAudioSourceController::clAudioSourceController()
{
   m_poObserver = NULL;
   m_pcoAudioStack= OSAL_NEW clStack(1,"AST  "); //the first one for Sink 1, always present
   m_allAudioStacks.push_back(m_pcoAudioStack);

}

//destructor
clAudioSourceController::~clAudioSourceController()
{
  ETG_TRACE_USR4(("clAudioSourceController destructor called"));
  m_allAudioStacks.clear();
}

/*!************************************************************************
 * METHOD:        bSELECT
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to select the background source
 *                Backround sources (Entertainment Sources) are not stopped/removed.
 *                They are always exchanged/replaced by another one.
 *                Sequence is:
 *                - Stop or Pause current background source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification for source stopped
 *                - Send Broadcast Notification for Exchanged BG-Source
 *                - Start new background source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *
 * PARAMETER:     u8newBgSource
 *                userData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/

clStack* clAudioSourceController::getStackForSink(tU16 sinkID)
{
   //if diag is active we return m_pcoAudioStack
   //ToDo Add Diag support for all stacks
   if(m_pcoBackupStack != 0)
   {
      return m_pcoAudioStack;
   }
  //ToDo find something better instead of -1 (add a Sink to the stack class and search for the matching sinkID)
  //if(m_allAudioStacks.size()<=sinkID) return m_allAudioStacks.at(sinkID-1);
  tU32 u32Size=(tU32)m_allAudioStacks.size(); //VVD_GEN4
  if( (sinkID<=u32Size) && (sinkID>=1) )
  {
    return m_allAudioStacks[sinkID-1];
  }
  else
  {
    return m_pcoAudioStack; // ToDo return form vector for default Sink. Ensure that it is always there
  }
}

tBool clAudioSourceController::bSELECT(SourceID srcID, tU32 u32UserData)
{
  return bSELECT(srcID, 1, u32UserData);
}

tBool clAudioSourceController::bSELECT(SourceID srcID, tU16 sinkID, tU32 u32UserData)
{
  tBool bResult = FALSE;
  clAudioSource* poNewBgSrc = NULL;
  clAudioSource* poUnderlyingBgSrc = NULL;
  clAudioSource* poCurrentActiveSource = NULL;

  poNewBgSrc = clAudioSourceFactory::getAudioSource(srcID);
  if (poNewBgSrc == NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, clAudioSourceController::bSELECT poNewBgSrc is NULL"));
    return FALSE;
  }
  poNewBgSrc->vSetCurActSink(sinkID);

  // Now we fetch the Audio Stack for this sinkID
  m_pcoAudioStack = getStackForSink(sinkID);
  if (m_pcoAudioStack == NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, clAudioSourceController::bSELECT m_pcoAudioStack is NULL"));
    return FALSE;
  }

  m_pcoAudioStack->trace();

  poNewBgSrc->vSetObserver(this);
  // pass userData to AudioSource for use as soon as Source is called.
  // Currently used for Preset-Info only. (AM and FM sources)
  poNewBgSrc->vSetUserData(u32UserData);
  bResult = m_pcoAudioStack->AddSource(*poNewBgSrc);
  m_pcoAudioStack->trace();
  poUnderlyingBgSrc = m_pcoAudioStack->GetUnderlayingSource(*poNewBgSrc);


  // step 1: stop and remove prev. bg-source
  if (poUnderlyingBgSrc != NULL)
  {
    poUnderlyingBgSrc->vSetCurActSink(sinkID);

    // different behaviour for 2 cases:
    // 1: underlying bg-source is active (rampUp, RampDownToPause, RampDownToOff, On)
    // 2:  underlying bg-source is NOT active (Pause, Off)

    if (poUnderlyingBgSrc->bIsActive(sinkID))
    {
      ETG_TRACE_USR3(("bSELECT, poUnderlyingBgSrc->bIsActive() is true "));
      // case 1: underlying bg is active:
      // stop it and send it to background

      // determine the bg-source, which is activated next in order to define the type of mute-ramp
      // and to indicate, which band to bring to Foreground
      poUnderlyingBgSrc->vOff(poNewBgSrc->sGetId(),sinkID);
    }
    else
    {
      ETG_TRACE_USR4(("bSELECT, poUnderlyingBgSrc->bIsActive() is false "));
      // case 2: underlying bg is NOT active:
      if (poUnderlyingBgSrc->pclGetState(sinkID) == clSrcStateFactory::pclCreatePause())
      {
        ETG_TRACE_USR3(("bSELECT, underlying bg is NOT active "));
        // case 2.1: underlying src is in PAUSE
        // send it to Background by changing from pause to off

        // determine the bg-source, which is activated next in order to define the type of mute-ramp
        // and to indicate, which band to bring to Foreground
        poUnderlyingBgSrc->vOff(poNewBgSrc->sGetId(),sinkID);
      }
      else
      {
        ETG_TRACE_USR3(("bSELECT, underlying src is in OFF "));
        // case 2.2: underlying src is in OFF
        // example:
        //         |  newBG    | off
        //         |  BG1      | off
        //         |  BG0      | rampDownToOff
        // Next line added to sent the GAS to Off and remove the connection from DB
        poUnderlyingBgSrc->vOff(poNewBgSrc->sGetId(),sinkID);
        // before it is removed from the AudioStack
        m_pcoAudioStack->RemoveSource(*poUnderlyingBgSrc);
      }
    }
  }
  else
  {
    ETG_TRACE_USR3(("bSELECT, for preset only "));
    // for preset only (user defined action)
    // there is no underlying bg-source. So the requested source might be already on
    // stack in pause state
    if (poNewBgSrc->pclGetState(sinkID) == clSrcStateFactory::pclCreateRampDownToPause()
    || poNewBgSrc->pclGetState(sinkID) == clSrcStateFactory::pclCreatePause())
    {
      ETG_TRACE_USR3(("bSELECT, before poNewBgSrc->vPause "));
      poNewBgSrc->vPause(poNewBgSrc->sGetId(),sinkID);
    }
  }

  // step 2: start this new bg-source if required. Otherwise keep it in Pause
  // start the bg-source, if a) it is top of stack AND b) no other source is active
  poCurrentActiveSource = pcoGetCurrentActiveAudiosource();
  if (poNewBgSrc == pcoGetTopOfStackAudiosource()
      &&
      (poCurrentActiveSource == NULL || poCurrentActiveSource == poNewBgSrc))
  {
    //Check for PushAction WaitFor rule
    if(WaitForSourceNeeded(*poNewBgSrc))
    {
      ETG_TRACE_USR3(("bSELECT, before poNewBgSrc->vWaitFor "));
      poNewBgSrc->vWaitFor(sinkID);
    }else{
      //We can proceed directly
      ETG_TRACE_USR3(("bSELECT, before poNewBgSrc->vOn "));
      if(!bPauseEntSourceOnMixSourceRequest())
        poNewBgSrc->vOn(sinkID);
    }
  }
  // if this new bg src is exchanged silent (an FG-source is on stack) and currently is in rampDownToOff
  // then change it from rampDownToOff to rampDownToPause to keep it in Foreground
  // Don't wait in this case until the prev. bg-source is stopped cause there is no stop in progress
  // example:
  //         |  FG       | off
  //         |  BG1      | off
  //         |  BG0      | rampDownToOff
  // -> SELECT(BG0)
  //         |  FG       | off
  //         |  BG0      | rampDownToOff
  //         |  BG1      | off
  else if (poNewBgSrc->pclGetState(sinkID) == clSrcStateFactory::pclCreateRampDownToOff())
  {
    ETG_TRACE_USR3(("bSELECT, pclGetState is pclCreateRampDownToOff "));
    poNewBgSrc->vPause(poNewBgSrc->sGetId(),sinkID);

    //daw2hi 10.11.2017 do not execute if poCurrentActiveSource is NULL
    if(poCurrentActiveSource != NULL)
    {
      //Now we might Kick the Top FG Src
      //Check if Kick Action is provided
      const std::vector<clSourceClass::actions_t>& pushRules = poNewBgSrc->getSourceClass().getPushActionList();
      for(std::vector<clSourceClass::actions_t>::const_iterator it = pushRules.begin();
          it != pushRules.end();
          ++it)
      {
        if((*it) == ((clSourceClass::actions_t)((poCurrentActiveSource->sGetId().enSourceClass) | clSourceClass::/*actions_t::*/action_abort)))
        {
          ETG_TRACE_ERR(("CMD_AUDIO,  NOT IMPLEMENTED EXCEPTION"));
        }
      }
    }
    else
    {
      ETG_TRACE_ERR(("CMD_AUDIO, bSELECT, pclGetState is pclCreateRampDownToOff but poCurrentActiveSource == NULL"));
    }
  }else if( NULL  != pcoGetCurrentActiveAudiosource())
  {
    /* case 1 : SYS_MUTE (State=on) -> this is active
     *          -------------------
     *          MEDIA/FM (State=off)
     *
     * case 2 : NEW BG (State = off)
     *          -------------------
     *          OLD BG (State = RampDownToOff) -> this is active
     * */
    // Check if active source is on top of new bg source
    clAudioSource* poActSrc = pcoGetCurrentActiveAudiosource();
    //clAudioSource* poUnderlyingSrc = poGetUnder
    poUnderlyingBgSrc = m_pcoAudioStack->GetUnderlayingSource(*poActSrc);
    while((poUnderlyingBgSrc != NULL)&&(poUnderlyingBgSrc->sGetId() != poNewBgSrc->sGetId()))
    {
      poUnderlyingBgSrc = m_pcoAudioStack->GetUnderlayingSource(*poUnderlyingBgSrc);
    }
    //case 1: if underlying src == new src
    //case 2: if underlying src == null
    if(poUnderlyingBgSrc != NULL)
    {
      if(poActSrc->getSourceType().exclusive)
      {
        ETG_TRACE_USR3(("bSELECT: Send new source classID %d subID%d to pause, because active classID %d subID %d"
            , poNewBgSrc->sGetId().enSourceClass
            , poNewBgSrc->sGetId().u16SubSource
            , (tU16)poActSrc->sGetId().enSourceClass
            , (tU16)poActSrc->sGetId().u16SubSource));
        poNewBgSrc->vPause(poNewBgSrc->sGetId(),sinkID);
      }
    }else{
      ETG_TRACE_USR3(("bSELECT: Send wait that source classID %d subID%d is gone"
          , (tU16)poActSrc->sGetId().enSourceClass
          , (tU16)poActSrc->sGetId().u16SubSource));
    }
  }
  else{
    ETG_TRACE_USR4(("bSELECT, else "));
  }

  // trace
  m_pcoAudioStack->trace();

  return bResult;
}


/*!************************************************************************
 * METHOD:        bON
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to activate a Foreground Source
 *                A Foreground Source is added to the stack. The positon in
 *                the stack depends on the priority rules.
 *                By default an FG source does not replace another source.
 *                But keep in mind, that the stack rules might remove other
 *                Sources automatically.
 *                Sequence is:
 *                - Stop or Pause the sources below in the stack (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification for sources, which have been stopped
 *                - Start this source, if it is the top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vMuteStack()
 *
 *
 * PARAMETER:     u8newFgSource
 *                userData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 * CHANGES:
 *    22.01.2008 J. Steinle: Before: Mute-Demute-Phase was started only
 *               if  the new FG source was in mute-state. This was done
 *               to prevent mute in case that the source was already active.
 *               In case of toggling On-Off-Requests, an On-Request could be
 *               lost because source was in ramp-down-state and not in
 *               mute-state. Solution: Start mute-demute-ramp always.
 *               To prevent a mute in case that the source is already active
 *               the mute-ramp begins on the underlying source now.
 *************************************************************************/
tBool clAudioSourceController::bON(SourceID srcID, tU32 u32UserData)
{
  ETG_TRACE_USR4(("bON using default Sink 1"));
  return bON(srcID, 1, u32UserData);
}
tBool clAudioSourceController::bON(SourceID srcID, tU16 sinkID, tU32 u32UserData )
{
  tBool bResult = FALSE;
  clAudioSource* poNewFgSrc = NULL;
  clAudioSource* poPrevActiveSrc = NULL;

  // Now we fetch the Audio Stack for this sinkID
  m_pcoAudioStack = getStackForSink(sinkID);

  poNewFgSrc = clAudioSourceFactory::getAudioSource(srcID);
  poPrevActiveSrc = pcoGetCurrentActiveAudiosource();

  //daw2hi set the sink context
  // but only if the pointers are valid
  if((poNewFgSrc==NULL) || (m_pcoAudioStack == NULL))
  {
    ETG_TRACE_FATAL(("poNewFgSrc == NULL || m_pcoAudioStack == NULL (%p,%p)",poNewFgSrc,m_pcoAudioStack));
    return FALSE;
  }

  if(poPrevActiveSrc!=NULL)
  poPrevActiveSrc->vSetCurActSink(sinkID);
  poNewFgSrc->vSetCurActSink(sinkID);

  //trace("bON - newSrc: %s", poNewFgSrc->pacGetName());
  //      vTrace(AUD_SM_SRC_ON,poNewFgSrc->u8GetId());
  ETG_TRACE_USR3(("bON SourceClass: %d subID %d"
      , poNewFgSrc->sGetId().enSourceClass
      , poNewFgSrc->sGetId().u16SubSource));
  // push new source on stack
  poNewFgSrc->vSetObserver(this);
  poNewFgSrc->vSetUserData(u32UserData);
  bResult = m_pcoAudioStack->AddSource(*poNewFgSrc);

  if (bResult)
  {
    // different behavior for 2 cases:
    // 1: the new FG source is the top of stack source
    // 2: the new FG source is NOT the top of stack source

    if (pcoGetTopOfStackAudiosource() == poNewFgSrc)
    {
      // case 1: the new FG source is top of stack source
      // start a mute-phase for the currently active source
      if (poPrevActiveSrc!= NULL && poPrevActiveSrc != poNewFgSrc)
      {
        // case 1.1: Another source is currently active. So start mute-phase for it.
        // only send pause to active source if not already in ramp down or off or pause
        // note: forbidden to send pause while a source is going to off, because
        // this would change the target state of that source
        if (poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUp()
        || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpInit()
        || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateSrcAvailableCheck_On()
        || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpRequesting()
        || poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateOn())
        {
          // reset user data
          poPrevActiveSrc->vSetUserData(0);
          poPrevActiveSrc->vPause( poNewFgSrc->sGetId());
          //poPrevActiveSrc->vOff(poNewFgSrc->u8GetId());
        }
      }
      else
      {
        // case 1.2: Either NO source is active (can occur at startup) or this FG is already active
        // send On-request to cancel a ramp-down for this new FG source
        // Check for PushAction WaitFor rule
        if(WaitForSourceNeeded(*poNewFgSrc))
        {
          poNewFgSrc->vWaitFor();
        }else{
          //We can proceed directly
          poNewFgSrc->vOn();
        }
      }
    }
    else
    {
      // case 2: the new FG source is NOT the top of stack source
      // request for an FG-source which is not on top of stack, but somewhere below/between
      // example:
      //         |  FG0    | on                     |  FG0    | on
      //         |  newFG  | rampDownToOff     ->   |  newFG  | rampDownToPause
      //         |  BG     | pause                  |  BG     | pause
      if (poNewFgSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff())
      {
        poNewFgSrc->vSetUserData(0);
        poNewFgSrc->vPause(poNewFgSrc->sGetId());
      }
    }
  }
  
  if (poPrevActiveSrc!= NULL && poPrevActiveSrc->pclGetState() == clSrcStateFactory::pclCreateSrcAvailableCheck_Pause())
  {
    if(WaitForSourceNeeded(*poNewFgSrc))
    {
          poNewFgSrc->vWaitFor();
    }else{
          //We can proceed directly
          poNewFgSrc->vOn();
    }
  }
  // trace
  m_pcoAudioStack->trace();

  return bResult;
}

/*!************************************************************************
 * METHOD:        bMIX
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to activate a Mix Source
 *                A Mix Source is added to the stack. The position in
 *                the stack depends on the priority rules.
 *                By default a Mix source does not replace another source.
 *                But keep in mind, that the stack rules might remove other
 *                Sources automatically.
 *                In case of Mix souces there is no need to mute other
 *                existing sources.
 *                Sequence is:
 *                - Start this source, if it is the top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vOn()
 *
 * PARAMETER:     u8newMixSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bMIX(SourceID srcID, tU32 u32UserData)
{
  ETG_TRACE_USR4(("bMIX using default Sink 1"));
  return bMIX(srcID, 1, u32UserData);
}

tBool clAudioSourceController::bMIX(SourceID srcID, tU16 sinkID, tU32 u32UserData)
{
  tBool bResult = FALSE;
  clAudioSource* poNewMixSrc = NULL;

  poNewMixSrc = clAudioSourceFactory::getAudioSource(srcID);
  if((poNewMixSrc == NULL) || (m_pcoAudioStack == NULL))
  {
    ETG_TRACE_ERR(("CMD_AUDIO, poNewMixSrc (%p)== NULL and/or m_pcoAudioStack (%p) == NULL",poNewMixSrc,m_pcoAudioStack));
    return FALSE;
  }

  //daw2hi set the sink context
  poNewMixSrc->vSetCurActSink(sinkID);
  // Now we fetch the Audio Stack for this sinkID
  m_pcoAudioStack = getStackForSink(sinkID);

  ETG_TRACE_USR2(("bMIX SourceClass: %d subID %d"
      , poNewMixSrc->sGetId().enSourceClass
      , poNewMixSrc->sGetId().u16SubSource));
  // push new source on stack
  poNewMixSrc->vSetObserver(this);
  poNewMixSrc->vSetUserData(u32UserData);
  bResult = m_pcoAudioStack->AddSource(*poNewMixSrc);
  if (bResult)
  {
    //Check for PushAction WaitFor rule
    if (WaitForSourceNeeded(*poNewMixSrc))
    {
      clAudioSource* SourceWaitingFor = pcoGetSourceWaitingFor(*poNewMixSrc);
      if(SourceWaitingFor != nullptr && SourceWaitingFor->pclGetState() == clSrcStateFactory::pclCreatePause())
      {
        poNewMixSrc->vOn();
      }
      else
        poNewMixSrc->vWaitFor();
    }
    else
    {
      /*
       * If there is  already another mix source active then this need to be paused first
       */
      clAudioSource* poPrevActiveMixSrc = pcoGetTopOfStackMixsource();
      clAudioSource* poAnotherMixSrc = pcoGetSecondMixSource();
      clAudioSource* poTopOfStackSrc = pcoGetTopOfStack();

      //If Source requested is top mix source
      //if (pcoGetTopOfStack() == poNewMixSrc)
      if((poPrevActiveMixSrc != NULL) && (poNewMixSrc == poPrevActiveMixSrc))
    {

        if(poNewMixSrc!=NULL)
        {
          ETG_TRACE_USR3(("bMIX, Requested mix source is top of stack mix source : SourceClass: %d subID %d"
            , poNewMixSrc->sGetId().enSourceClass
            , poNewMixSrc->sGetId().u16SubSource));
        }
        //If there is one more mix source on stack below requested mix source (top of stack)
        if((poAnotherMixSrc != NULL) && (poAnotherMixSrc != poNewMixSrc))
        {
          ETG_TRACE_USR3(("bMIX, Already another mix source is active, SourceClass: %d subID %d"
              , poAnotherMixSrc->sGetId().enSourceClass
              , poAnotherMixSrc->sGetId().u16SubSource));
       poNewMixSrc->vOn(); // start the new mix source

          /*
           *  Use Case when New Mix is requested when
           *  Previous mix source is in rampupRequesting
           *  OR when Previous Mix is already playing (ON)
           *
           *  Precondition : New mix is allowed over Previous mix source & no wait_off / abort rule present
           *
           *  AST  :  --------TOP--------
           *  AST  :      NAVI_SPEECH : rampupInit / rampupRequesting / on
           *  AST  :   MEDIA_PLAYER#1 :              on
           *  AST  :  ------BOTTOM-------
           *
           */

          /*if(((poAnotherMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpInit())
              ||(poAnotherMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampUpRequesting()))
              ||(poAnotherMixSrc->pclGetState() == clSrcStateFactory::pclCreateOn())
              ||(poAnotherMixSrc->pclGetState() == clSrcStateFactory::pclCreateOff()))
          {
            // reset user data
            poAnotherMixSrc->vSetUserData(0);
            poAnotherMixSrc->vPause( poNewMixSrc->sGetId());
          }*/
        }
        else //there is no other mix source
        {
          ETG_TRACE_USR3(("bMIX, There is no other mix source is active !!"));
          //We can proceed directly
          poNewMixSrc->vOn();
        }
      }
      else /* If requested Mix source is not top of stack */
      {
        /*
         * Case 1: the new Mix source is NOT the top of stack source
         * request for an mix-source which is not on top of stack, but somewhere below/between
         * example:
         *        |  Mix1              | on                     |  FG0                | on
         *        |  Mix2              | rampDownToOff     ->   |  newFG              | rampDownToPause
         *        |  Entertainment_Source        | on                   |  Entertainment_Source       | pause
         *
         * Case 2: SourceOn(Navi) requested
         *
         *       rule1> VPA_TTS allowed over NAVI & abort(Navi)
         *     rule2> Navi not allowed over VPA
         * example:
         *     AST  :  --------TOP--------
         *     AST  :  VPA_TTS        :wait_for
         *     AST  :  NAVI_SPEECH    :rampdownToOff
         *     AST  :  MEDIA_PLAYER#1 :on
         *     AST  :  ------BOTTOM-------
         */

        if (((poNewMixSrc->pclGetState() == clSrcStateFactory::pclCreateOff())
            || (poNewMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff()))
            && (poPrevActiveMixSrc != NULL) && (poPrevActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateWaitFor()))
        {
          ETG_TRACE_USR3(("bMIX, Already another mix source is active, SourceClass: %d subID %d"
              , poPrevActiveMixSrc->sGetId().enSourceClass
              , poPrevActiveMixSrc->sGetId().u16SubSource));

          poNewMixSrc->vSetUserData(0);
          poNewMixSrc->vPause(poNewMixSrc->sGetId());
        }
        else if((poTopOfStackSrc != NULL) && (poTopOfStackSrc->getSourceType().exclusive))
        {
          ETG_TRACE_USR2(("bMIX, Already No mix source is active, top of Stack is  SourceClass: %d subID %d"
              , poTopOfStackSrc->sGetId().enSourceClass
              , poTopOfStackSrc->sGetId().u16SubSource));

          poNewMixSrc->vOn();
        }
      }
    }
    // a mix source can directly be started. The mute-demute-phases are skipped.
    // No Crossover and no BG-Exchange-Triggers are sent
  }
  // trace
  //m_pcoAudioStack->trace();

  return bResult;
}

tBool clAudioSourceController::WaitForSourceNeeded(clAudioSource &src)
{
  //1. Check if Source defines a WaitFor rule
  const std::vector<clSourceClass::actions_t>& pushActions = src.getSourceClass().getPushActionList();
  ETG_TRACE_USR2(("WaitForSourceNeeded: check %d actions of source classID %d subID %d "
        , pushActions.size()
        , src.sGetId().enSourceClass
        , src.sGetId().u16SubSource));

  for(std::vector<clSourceClass::actions_t>::const_iterator it = pushActions.begin();
        it != pushActions.end();
        ++it)
  {
      //switch(action & clSourceClass::/*actions_t::*/mask)  // daw2hi: this seems wrong
      switch((*it) & clSourceClass::/*actions_t::*/mask)
      {
      case clSourceClass::/*actions_t::*/action_waitFor:
      {
         sourceClassID tgtClassID = static_cast<sourceClassID>((*it) &  clSourceClass::/*actions_t::*/idxMask);
         ETG_TRACE_USR4(("WaitForSourceNeeded: WaitFor rule found: target source class %d"
               , tgtClassID));

          if (NULL != m_pcoAudioStack)
          {
            clAudioSource* pTargetSrc =  m_pcoAudioStack->GetTopSource(tgtClassID);
                //(static_cast<AudioSources::enAudioSources>(action & clStackRules::/*actions_t::*/idxMask));
        if(pTargetSrc!=NULL && pTargetSrc->bIsActive())
        {
        ETG_TRACE_USR3(("WaitFor rule found: SourceClass: %d subID %d WAITS FOR SourceClass: %d subID %d"
        , src.sGetId().enSourceClass
        , src.sGetId().u16SubSource
        , pTargetSrc->sGetId().enSourceClass
        , pTargetSrc->sGetId().u16SubSource));
        //1.1.1 WaitFor Source, return false
        return TRUE;
				}
				else
				{
					if(pTargetSrc != NULL)
					{
						if(src.getSourceType().name == "mix")
						{
							std::string traceTxt(src.pacGetName());
							traceTxt = traceTxt + std::string(" is of type mix and has wait_for ");
							traceTxt = traceTxt + std::string(pTargetSrc->pacGetName());
							ETG_TRACE_USR3(("Source %s",traceTxt.c_str()));
							return true;
						}
						if(std::string(src.pacGetName()) == "MIC_PERMANENT")
						{
							ETG_TRACE_USR3(("MIC_PERMANENT will wait independent of the state of the wait_for source"));
							return true;
						}

						if(pTargetSrc->bIsPause())
						{
							ETG_TRACE_USR3(("WaitFor rule found: SourceClass: %d subID %d WAITS FOR SourceClass: %d subID %d in Pause state"
									, src.sGetId().enSourceClass
									, src.sGetId().u16SubSource
									, pTargetSrc->sGetId().enSourceClass
									, pTargetSrc->sGetId().u16SubSource));
							return TRUE;
						}
          ETG_TRACE_USR4(("WaitFor rule found: Found NOT active SourceClass: %d subID %d"
            , pTargetSrc->sGetId().enSourceClass
            , pTargetSrc->sGetId().u16SubSource));
         }else{
          ETG_TRACE_USR4(("WaitFor rule found: No conflicting SourceClass active"));
         }
        }
          }
      }
      break;
      default:
        //do nothing
      break;
      }
  }
  //No WaitFor rule or no corresponding active Source
  return FALSE;
}



tBool clAudioSourceController::ActionPauseForSourceNeeded(clAudioSource &src)
{
  //1. Check if Source defines a WaitFor rule
  const std::vector<clSourceClass::actions_t>& pushActions = src.getSourceClass().getPushActionList();
  ETG_TRACE_USR2(("ActionPauseForSourceNeeded: check %d actions of source classID %d subID %d "
        , pushActions.size()
        , src.sGetId().enSourceClass
        , src.sGetId().u16SubSource));

  for(std::vector<clSourceClass::actions_t>::const_iterator it = pushActions.begin();
        it != pushActions.end();
        ++it)
  {

      switch((*it) & clSourceClass::/*actions_t::*/mask)
      {
      case clSourceClass::/*actions_t::*/action_pause:
      {
         sourceClassID tgtClassID = static_cast<sourceClassID>((*it) &  clSourceClass::/*actions_t::*/idxMask);
         ETG_TRACE_USR4(("action_pause: action_pause found: target source class %d"
               , tgtClassID));

          if (NULL != m_pcoAudioStack)
          {
            clAudioSource* pTargetSrc =  m_pcoAudioStack->GetTopSource(tgtClassID);
                //(static_cast<AudioSources::enAudioSources>(action & clStackRules::/*actions_t::*/idxMask));
        if(pTargetSrc!=NULL)
        {
        ETG_TRACE_USR3(("ActionPauseForSourceNeeded: SourceClass: %d subID %d Action Pause SourceClass: %d subID %d"
        , src.sGetId().enSourceClass
        , src.sGetId().u16SubSource
        , pTargetSrc->sGetId().enSourceClass
        , pTargetSrc->sGetId().u16SubSource));
        if((pTargetSrc->pclGetState() != clSrcStateFactory::pclCreateRampDownToPause()) && (pTargetSrc->pclGetState() != clSrcStateFactory::pclCreatePause()))
          pTargetSrc->vPause(pTargetSrc->sGetId());
        return TRUE;
				}
          }
      }
      break;
      default:
        //do nothing
      break;
      }
  }
  //No WaitFor rule or no corresponding active Source
  return FALSE;
}





/*!************************************************************************
 * METHOD:        bOFF
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to deativate an FG Source
 *                Sequence is:
 *                - Stop or Pause this source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification (Source stopped)
 *                - Start the new top-stack-source (demute-ramp)
 *                - Wait for asynchronous confirmation of start(demute-ramp finished)
 *                - Send Broadcast Notification for source started
 *                This method initiates this sequence by calling vMuteStack()
 *
 *
 * PARAMETER:     u8FgSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bOFF(SourceID srcID, tU32 u32UserData)
{
   ETG_TRACE_USR4(("bOFF using default Sink 1"));
   return bOFF(srcID, 1, u32UserData);
}

tBool clAudioSourceController::bOFF(SourceID srcID, tU16 sinkID, tU32 u32UserData)
{
   (void) u32UserData;
   tBool bResult = FALSE;
   clAudioSource* poFgSrc = NULL;
   clAudioSource* poNextSource = NULL;

   // Now we fetch the Audio Stack for this sinkID
   m_pcoAudioStack = getStackForSink(sinkID);


   poFgSrc = clAudioSourceFactory::getAudioSource(srcID);

   if (NULL != poFgSrc && m_pcoAudioStack != NULL)
   {
      //trace("bOFF(%s)", poFgSrc->pacGetName());
     ETG_TRACE_USR2(("bOFF SourceClass: %d subID %d"
            , poFgSrc->sGetId().enSourceClass
            , poFgSrc->sGetId().u16SubSource));

      // different behaviour for 2 cases:
      // 1: the FG source is OFF (it's on stack but not yet started)
      // 2: the FG source is NOT off (pause, rampUp, rampDownToPause, rampDownToOff, on)

      if (poFgSrc->pclGetState() == clSrcStateFactory::pclCreateOff())
      {
         // case 1: FG is in OFF state
         // if this FG is the TopSource than cancel the RampDownToPause of the underlying src
         // Don't cancel a RampDownToOff of the underlying src
         m_pcoAudioStack->RemoveSource(*poFgSrc);
         poNextSource = pcoGetTopOfStackAudiosource();

         if (poNextSource != NULL
             && (poNextSource->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause()|| 
			 poNextSource->pclGetState() == clSrcStateFactory::pclCreateSrcAvailableCheck_Pause()))
         {
            poNextSource->vOn();
         }
      }
      else
      {
         // case 2: FG is NOT in OFF state
         // in this case stop it and proceed later asynchronously on off_done()
         // find the next source to start and specify it in the Off-Command
         // Skip virtual sources (e.g. mute-sources) because FC-Audio needs a real source to set
         poNextSource = pcoGetTopOfStackAudiosource();
         while(poNextSource != NULL
            && (poNextSource == poFgSrc // the stopped src itself is topOfStack, so find the next below
               //|| poNextSource->getGroup() == clStackRules::groupInternal
               )) // the found src is not a real src, so find the next below
         {
            poNextSource = m_pcoAudioStack->GetUnderlayingSource(*poNextSource);
         }
         if (poNextSource != NULL)
         {
            ETG_TRACE_USR4(("Call vOFF of SourceClass: %d subID %d, PNS %d"
                        , poFgSrc->sGetId().enSourceClass
                        , poFgSrc->sGetId().u16SubSource
                        , poNextSource->sGetId().enSourceClass));
            poFgSrc->vOff(poNextSource->sGetId());
         }else{
            //Maybe we do not have any source
            poFgSrc->vOff();
         }

      }

      m_pcoAudioStack->trace();
      bResult = TRUE;
   }
   else
   {
      ETG_TRACE_ERR(("CMD_AUDIO, NULL == poFgSrc or m_pcoAudioStack == NULL"));
      bResult = FALSE;
   }
   return bResult;
}

/*!************************************************************************
 * METHOD:        bMIXOFF
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Use this Interface Method to deativate a Mix source
 *                Sequence is:
 *                - Stop this source (mute-ramp)
 *                - Wait for asynchronous confirmation of stop(mute-ramp finished)
 *                - Send Broadcast Notification (Source stopped)
 *                This method initiates this sequence by calling vOff()
 *
 * PARAMETER:     u8MixSource
 *                u32UserData          reserved
 * RETURNVALUE:   bSucess
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSourceController::bMIXOFF(SourceID mixSrc, tU32 u32UserData)
{
  // use Sink 1 if not provided
  return bMIXOFF(mixSrc, 1, u32UserData);
}

tBool clAudioSourceController::bMIXOFF(SourceID mixSrc, tU16 u16SinkID, tU32 u32UserData)
{
   tBool bResult = FALSE;
   clAudioSource* poMixSrc = NULL;

   poMixSrc = clAudioSourceFactory::getAudioSource(mixSrc);

   if (NULL != poMixSrc && m_pcoAudioStack != NULL)
   {
      //trace("bMIXOFF(%s)", poMixSrc->pacGetName());
    ETG_TRACE_USR2(("bMIXOFF SourceClass: %d subID %d"
            , poMixSrc->sGetId().enSourceClass
            , poMixSrc->sGetId().u16SubSource));

      // a mix source can directly be stopped. The mute-demute-phases are skipped.
      // No other sources have to be started after this stop.
      // No Crossover and no BG-Exchange-Triggers are sent

      if (u32UserData & BITMASK_FORCE_OFF) // force OFF
      {
        ETG_TRACE_USR3(("bMIXOFF(): SourceClass: %d subID %d"
            , mixSrc.enSourceClass
            , mixSrc.u16SubSource));
        poMixSrc->vResetPlayCmdCounter();
      }
      poMixSrc->vOff(poMixSrc->sGetId(), u16SinkID);

      //m_pcoAudioStack->trace();
      bResult = TRUE;
   }
   else
   {
      ETG_TRACE_ERR(("CMD_AUDIO, Null Pointer for Mix source or AudioStack"));
      bResult = FALSE;
   }
  return bResult;
}


/*!************************************************************************
 * METHOD:        vCrossOver
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   The trigger for animation sync
 *
 *
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vCrossOver()
{
   //notify observers
   //trace("--- CROSSOVER TRIGGER ---");
   ETG_TRACE_USR4(("vCrossOver"));
}




/*!************************************************************************
 * METHOD:        vOff_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the mute-ramp was finished.
 *                AudioSources send this acknowledge also, if no mute-ramp
 *                was performed. E.g. when an off() is requested while the
 *                AudioSource is already in off-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be sopped completely, so send the stop-notification now.
 *
 * PARAMETER:     src   The AudioSource which has finished the mute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vOff_done(clAudioSource* src)
{
  clAudioSource* topOfStackSrc = NULL;
  clAudioSource* currentActiveSrc = NULL;
  clAudioSource* topMixSource = NULL;

  if (src != NULL && m_pcoAudioStack != NULL)
  {
      // Now we fetch the Audio Stack for this sinkID
      m_pcoAudioStack = getStackForSink(src->u16GetCurActSink());
    // stopped sources are removed from the stack
    m_pcoAudioStack->RemoveSource(*src);

    ETG_TRACE_USR2(("vOff_done: SourceClass: %d subID %d"
        , src->sGetId().enSourceClass
        , src->sGetId().u16SubSource));
    
    
    //Remove the timer since source will be in OFF state and removed from the stack.
    if(src->getSourceType().name == "entertain" && src->getTimerHandle() != 0 && (src->bIsSourceProfChg() == true || src->bIsSourceLSM() == true))
       src->vRemoveTimer();
    

    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceStopped(src->sGetId());
    }

    /*
     * vdu1kor -> update sink state to "DeMuted" after MUTE_ENTERTAIN(OFF)[PSARCCB-6614]
     *  |MUTE_ENTERTAIN : on    |
     *  |TUNER_TA_FM    : pause |
     *  |TUNER_FM       : pause |
     */
    if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_ENTERTAIN"))
    {
      //MCAN Bose Amplifier need to be DeMuted, once MUTE_ENT is removed
      clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_UNMUTED);
      ETG_TRACE_USR3(("vOff_done, MUTE_ENTERTAIN source removed, calling changeSinkMuteStateDB->DeMuted"));
      clGeniviAudioCtrlAdapter::changeSinkMuteStateDB(MS_UNMUTED,1);

      /* Update System Property 'SYP_USER_MUTE_STATUS' with value 'SYP_USER_MUTE_STATUS_OFF' (NCG3D-21845) */
      clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_USER_MUTE_STATUS,SYP_USER_MUTE_STATUS_OFF);
    }
#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
   if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_AUDIO_OFF"))
    {
      //MCAN Bose Amplifier need to be DeMuted, once MUTE_ENT is removed
      clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_UNMUTED);

      // NCG3D-263574 - to avoid sending an AUDIO_ON update to HMI during shutdown when MUTE_ZERO_VOLUME is present on audio stack
      // check if MUTE_ZERO_VOLUME is present on audio stack and if active, then don't update AUDIO_ON to GENIVI Audio Manger
      const clSourceClass* pSrcClass_MuteZeroVolume = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");

      if(pSrcClass_MuteZeroVolume != NULL && bIsSourceOnStack(SourceID(pSrcClass_MuteZeroVolume->getClassID(),0)))
      {
         ETG_TRACE_USR4(("MUTE_ZERO_VOLUME is present on audio stack -> restrict AUDIO_ON update to HMI"));
      }
      else
      {
         ETG_TRACE_USR3(("vOff_done, MUTE_AUDIO_OFF source removed, calling changeSinkMuteStateDB->DeMuted"));
         clGeniviAudioCtrlAdapter::changeSinkMuteStateDB(MS_UNMUTED,1);

         /* Update Sound Property 'MSP_AUDIO_OFF_MUTE' with value 'SYP_USER_MUTE_STATUS_OFF' (NCG3D-21845) */
         clGeniviAudioCtrlAdapter::updateSoundPropertyDB(MSP_AUDIO_OFF_MUTE,AM_MSP_AUDIO_OFF_MUTE_OFF);
      }
    }
#endif
    topOfStackSrc = pcoGetTopOfStackAudiosource();
    currentActiveSrc = pcoGetCurrentActiveAudiosource();

    if (!src->getSourceType().stackable)
    {
      //find source of same type on the stack
      clAudioSource * currentBGsrc = m_pcoAudioStack->GetTopSource_Type(src->getSourceType().id);
      //clAudioSource* currentBGsrc = m_pcoAudioStack->GetCurrentBgSource();

      if(currentBGsrc!=NULL)//VVD
      {
        if(currentBGsrc == topOfStackSrc){
          ETG_TRACE_USR3(("vOff_done, FOREGROUND exchange of entertainment source"));
        }
        else{
          ETG_TRACE_USR3(("vOff_done, BACKGROUND exchange of entertainment source"));
        }
      }

      if (currentBGsrc != NULL && m_poObserver != NULL)
      {
        //if it is top exclusive send them to on
        if (currentBGsrc == topOfStackSrc)
        {
          //foreground exchange
          ETG_TRACE_USR3(("vOff_done: FOREGROUND exchange of entertainment source: SourceClass: %d subID %d"
              , currentBGsrc->sGetId().enSourceClass
              , currentBGsrc->sGetId().u16SubSource));
          m_poObserver->vEntertainmentSourceExchanged(currentBGsrc->sGetId(), FALSE);
        }
        else
        {
          // if there is other exclusive on top move to pause
          //background exchange
          // bring new bg src from off to pause
          currentBGsrc->vPause(currentBGsrc->sGetId());
          ETG_TRACE_USR3(("vOff_done: BACKGROUND exchange of entertainment SourceClass: %d subID %d"
              , currentBGsrc->sGetId().enSourceClass
              , currentBGsrc->sGetId().u16SubSource));
          m_poObserver->vEntertainmentSourceExchanged(currentBGsrc->sGetId(), TRUE);
        }
      }
    }

    //Take care of Mix-Sources that got rule WaitFor()
    //to wait for this Source
    topMixSource = pcoGetTopOfStackMixsource();
    if (topMixSource != NULL)
    {
      ETG_TRACE_USR3(("vOff_done,topMixSource is not NULL"));
      if ((WaitForSourceNeeded(*topMixSource)))
      {
        ETG_TRACE_USR3(("vOff_done,vWaitFor"));
        topMixSource->vWaitFor();
      }
      else
      {
        ETG_TRACE_USR4(("vOff_done: topMixSource SourceClass: %d subID %d"
               , topMixSource->sGetId().enSourceClass
               , topMixSource->sGetId().u16SubSource));

         //ETG_TRACE_USR4(("topMixSource is not NULL, WaitForSourceNeeded not required , vOff_done, No need to call vOn"));
         /*
         * In below case if response for MEDIA_PLAYER for off is received
         * then Navi was also starting (NCG3D-24225)
         *
         * AST:  --------TOP--------
         * AST:      NAVI_SPEECH :   rampdownToOff
         * AST:         TUNER_FM :             off
         * AST:   MEDIA_PLAYER#1 :   rampdownToOff
         * AST:  ------BOTTOM-------
         *
         * AST:  --------TOP--------
         * AST:      NAVI_SPEECH : rampupRequesting
         * AST:         TUNER_FM : rampupRequesting
         * AST:  ------BOTTOM-------
         *
         */
         //topMixSource->vOn();

        if(src->getSourceType().exclusive)
        {
       ETG_TRACE_USR3(("vOff_done: Called for Source is not MIX source, SourceClass: %d subID %d"
           , src->sGetId().enSourceClass
           , src->sGetId().u16SubSource));

       if ( topMixSource->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause()
         || topMixSource->pclGetState() == clSrcStateFactory::pclCreatePause()
         || topMixSource->pclGetState() == clSrcStateFactory::pclCreateRampUpRequesting()
         || topMixSource->pclGetState() == clSrcStateFactory::pclCreateOn()
         || topMixSource->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff())    /* (NCG3D-24225) */
       {
         ETG_TRACE_USR4(("vOff_done: topMixSource is not NULL, WaitForSourceNeeded not required , vOff_done, No need to call vOn"));
       }
       else
       {
         ETG_TRACE_USR3(("vOff_done: Starting topMixSource SourceClass: %d subID: %d"
             , topMixSource->sGetId().enSourceClass
             , topMixSource->sGetId().u16SubSource));
         topMixSource->vOn();
       }
        }
        else
        {
          /*
           * Use Case : If SourceOn(Navi_Speech) is requested & Navi is allowed over Mix3 & Abort(Mix3) is also present
           *        In this case Navi should start (NewMixRequested_Allowed_WaitOff_PrevMix_Mix1RampDownToOff_SrcOn_Mix1_AbortRule)
           *
           * AST  :  --------TOP--------
        * AST  :   MIX3         : wait_for
        * AST  :   NAVI_SPEECH    : rampdownToOff
        * AST  :   MEDIA_PLAYER#1 : on
        * AST  :  ------BOTTOM-------
           *
           */
       ETG_TRACE_USR4(("vOff_done: Called for Source is MIX source, Starting another Mix, SourceClass: %d subID: %d"
           , topMixSource->sGetId().enSourceClass
           , topMixSource->sGetId().u16SubSource));
       topMixSource->vOn();
        }
      }
    }

    if (currentActiveSrc == NULL && topOfStackSrc != NULL)
    {
      ETG_TRACE_USR4(("vOff_done,currentActiveSrc is NULL and topOfStackSrc is not NULL"));

      clAudioSource *pcoAudioSourceIterator = NULL;

      if(NULL != m_pcoAudioStack && topOfStackSrc != NULL && (topOfStackSrc->getSourceType().name == "entertain"))
      {
    	 pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

    	while(pcoAudioSourceIterator)
    	     {

    		if((pcoAudioSourceIterator->getSourceType().exclusive == false))
    		   {
    		        //found what I want
    			 if(ActionPauseForSourceNeeded(*pcoAudioSourceIterator))
    			   {
    		             break;
    			   }
    		    }


    	       pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    	     }
      }


      if (WaitForSourceNeeded(*topOfStackSrc))
      {
        ETG_TRACE_USR4(("vOff_done,vWaitFor"));
        topOfStackSrc->vWaitFor();
      }
      else
      {
        // daw2hi: check first if we have another Entertainment below of this in Pause state.
        // Needs to be removed first
        clAudioSource*  pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*topOfStackSrc);
        ETG_TRACE_USR4(("We have an underlying src, p = %p",(void*)pcoAudioSource));

        //daw2hi: 19.06.2014 (This was added for SUZUKI-11613)
        if(!topOfStackSrc->getSourceType().stackable)
        {
          // if top source is an entertain, check if any other entertain exists below. That needs to be removed first, before top of stack can start
          while(pcoAudioSource)
          {
            // ToDo: check if this condition is enough to remove it or do we need
            // to do it only for specific states of this source?
            if( (!pcoAudioSource->getSourceType().stackable) )
            {
              ETG_TRACE_USR4(("We have %s below",pcoAudioSource->pacGetName()));
              ETG_TRACE_USR4(("in state %s -> remove first",pcoAudioSource->pacGetState()));
        // Found by PSARCC21-3904 Reset issue. Pointer on the AudioSource has to be removed from Stack first before the AudioSource gets destroyed.
        m_pcoAudioStack->RemoveSource(*pcoAudioSource);
        // end
              pcoAudioSource->vOff();
            }
            pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
          }
        }

        if(!bPauseEntSourceOnMixSourceRequest())
        {
          //We can proceed directly
          ETG_TRACE_USR4(("vOff_done,vOn"));
          //22.07.2016 daw2hi we need to give the sink for this source (on which stack do we want to apply the on)
          topOfStackSrc->vSetCurActSink((tU16)m_pcoAudioStack->GetSinkID());
          topOfStackSrc->vOn((tU16)m_pcoAudioStack->GetSinkID());
        }

      }
      }
      m_pcoAudioStack->trace();

      /*
      *  If there is active source (source other than mix & not in off/pause/wait_off)
      *  which is top of stack,then source need to be started if it's entertainment source
      *
      *  Ex:
      *  Initial state : TA_FM(on) / MEDIA_PLAYER#1(pause)
      *  Media source removed : TA_FM(RampdownToOff) / FM(off) / MEDIA_PLAYER#1(pause)
      *               TA_FM(RampdownToOff) / FM(RampdownToPause)
      *  On receiving response for TA_FM(off) : FM(rampdownToPause)
      *
      */
      if((currentActiveSrc != NULL)&&(currentActiveSrc == topOfStackSrc))
      {
        //check if source is entertainment source
        if(!topOfStackSrc->getSourceType().stackable){

          //if source not already started
          if((topOfStackSrc->pclGetState() != clSrcStateFactory::pclCreateOn())
              && (topOfStackSrc->pclGetState() != clSrcStateFactory::pclCreateRampUpRequesting()))
          {
              ETG_TRACE_USR4(("vOff_done,top of stack is active source,calling vOn"));

              ETG_TRACE_USR4(("vOff_done: Starting topOfStackSrc SourceClass: %d subID %d"
               , topOfStackSrc->sGetId().enSourceClass
               , topOfStackSrc->sGetId().u16SubSource));
              if(false == WaitForSourceNeeded(*topOfStackSrc) && bPauseEntSourceOnMixSourceRequest()  == FALSE)
                     topOfStackSrc->vOn();
          }
          else{
            /*
             * NAVI(ON)/FM(ON) -> NAVI(off) requested and response is received for NAVI(off)
             */
            ETG_TRACE_USR4(("vOff_done,top of stack is active source, already source is ON "));
          }
        }
        else{
          ETG_TRACE_USR4(("vOff_done,top of stack is active source, topOfStackSrc is NOT Entertainment source "));
        }
      }

    //After source is finished/stopped, load volume context for next possible source
  if(clFactory_AudioSourceClass::bIsMCanBoseAmplifierConnected())
  {
  if( src!= NULL && clSourceClass::group_mute != src->getSourceClass().getGroupID())
  {
    ETG_TRACE_USR4(("vOff_done vLoadVolumeContext"));
    clGeniviAudioCtrlAdapter::vLoadVolumeContext(src);
  }
  }
  else
    clGeniviAudioCtrlAdapter::vLoadVolumeContext(src);
   }
}

tBool clAudioSourceController::bPauseEntSourceOnMixSourceRequest()
{
  ETG_TRACE_USR4(("bPauseEntSourceOnMixSourceRequest entered"));
  clAudioSource *pcoAudioSourceIterator = NULL;

  if(m_pcoAudioStack)
    pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

  while(pcoAudioSourceIterator != NULL)
  {

    if((pcoAudioSourceIterator->getSourceType().exclusive == false) && (ActionPauseForSourceNeeded(*pcoAudioSourceIterator) ))
    {
      ETG_TRACE_USR4(("found pause rule for %s",pcoAudioSourceIterator->pacGetName()));
      return TRUE;
    }
    else
    {
      pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }
  }
  return FALSE;
}

/*!************************************************************************
 * METHOD:        vPause_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the mute-ramp was finished.
 *                AudioSources send this acknowledge also, if no mute-ramp
 *                was performed. E.g. when an pause() is requested while the
 *                AudioSource is already in pause-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be stopped completely, so send the stop-notification now.
 *
 * PARAMETER:     src   The AudioSource which has finished the mute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vPause_done(clAudioSource* src)
{
  clAudioSource* topOfStackSrc = NULL;
  clAudioSource* currentActiveSrc = NULL;
  clAudioSource* topOfStackMixSrc = NULL;

  if(src != NULL)
  {
    // paused sources are NOT removed from the stack
    // trace("---Notification: AudioSource Stopped: %s", src->pacGetName());

    ETG_TRACE_USR2(("vPause_done: SourceClass: %d subID %d"
            , src->sGetId().enSourceClass
            , src->sGetId().u16SubSource));
    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceStopped(src->sGetId());
    }

    /*
     * In below use case : MCAN Bose Amplifier need to be Demute
     *  |TUNER_TA_FM    : on        |
     *  |MUTE_ENTERTAIN : pause     |
     *  |TUNER_FM       : pause     |
     */
    if (0 == OSAL_s32StringCompare(src->pacGetName(), "MUTE_ENTERTAIN"))
    {
        ETG_TRACE_USR3(("vPause_done, MUTE_ENTERTAIN source Paused"));
        clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_UNMUTED);
    }

  if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_AUDIO_OFF"))
    {
    ETG_TRACE_USR3(("vPause_done, MUTE_AUDIO_OFF source Paused"));
      //MCAN Bose Amplifier need to be DeMuted, once MUTE_ENT is removed
      clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_UNMUTED);
    }

    //daw2hi 22.07.2016
    // Now we fetch the Audio Stack for this sinkID
    m_pcoAudioStack = getStackForSink(src->u16GetCurActSink());


    topOfStackSrc    = pcoGetTopOfStackAudiosource();
    currentActiveSrc = pcoGetCurrentActiveAudiosource();
    topOfStackMixSrc = pcoGetTopOfStackMixsource();

    if (currentActiveSrc == NULL && topOfStackSrc != NULL && topOfStackSrc!= src)
    {
      //Check for PushAction WaitFor rule
      if (WaitForSourceNeeded(*topOfStackSrc))
      {
          clAudioSource* SourceWaitingFor = pcoGetSourceWaitingFor(*topOfStackSrc);
          if(SourceWaitingFor != NULL && SourceWaitingFor->bIsPause())
          {
            ETG_TRACE_USR3(("vPause_done, Starting next source Paused"));
            topOfStackSrc->vOn();
          }
          else
          {
            topOfStackSrc->vWaitFor();
          }
      }
      else
      {
        //We can proceed directly
           topOfStackSrc->vOn();
      }
    }
    if((topOfStackMixSrc != NULL)
    		 //&& (!src->getSourceType().exclusive)
			 && (topOfStackMixSrc != src))	//if pause is for this source we should not start again if it is also the topMixSrc
    {
      //If Mix source is paused
      ETG_TRACE_USR4(("vPause_done: Mix source is Paused, SourceClass: %d subID %d"
          , src->sGetId().enSourceClass
          , src->sGetId().u16SubSource));

      ETG_TRACE_USR4(("vPause_done: Starting topOfStackMixSrc SourceClass: %d subID %d"
          , topOfStackMixSrc->sGetId().enSourceClass
          , topOfStackMixSrc->sGetId().u16SubSource));

        //We can proceed directly
      if(topOfStackMixSrc->pclGetState() == clSrcStateFactory::pclCreateOff() || topOfStackMixSrc->pclGetState() == clSrcStateFactory::pclCreateWaitFor())//if in off/wait_for state then proceed directly
      {
    	if(false == WaitForSourceNeeded(*topOfStackMixSrc))
    	{
    		topOfStackMixSrc->vOn();
    	}
    	else
    	{
    		ETG_TRACE_USR4(("we have a wait_for and do not start topOfStackMixSrc %s",topOfStackMixSrc->pacGetName()));
        clAudioSource* SourceWaitingFor = pcoGetSourceWaitingFor(*topOfStackMixSrc);
        if(SourceWaitingFor == src)
        {
          ETG_TRACE_USR3(("vPause_done, Starting next source Paused"));
          topOfStackMixSrc->vOn();
        }
        else
          topOfStackMixSrc->vWaitFor();
    	}

      }
    }
    if (NULL != m_pcoAudioStack)
      m_pcoAudioStack->trace();

    if(m_pcoBackupStack != NULL)
    {
      ETG_TRACE_USR4(("Trace the Backup"));
      m_pcoBackupStack->trace();
      ETG_TRACE_USR4(("Trace the Backup end"));
    }
  }
}

clAudioSource* clAudioSourceController::pcoGetSourceWaitingFor(clAudioSource &src)
{
  //1. Check if Source defines a WaitFor rule
  clAudioSource* pTargetSrc = NULL;
  const std::vector<clSourceClass::actions_t>& pushActions = src.getSourceClass().getPushActionList();
  ETG_TRACE_USR4(("WaitForSource: check %d actions of source classID %d subID %d "
        , pushActions.size()
        , src.sGetId().enSourceClass
        , src.sGetId().u16SubSource));
  for(std::vector<clSourceClass::actions_t>::const_iterator it = pushActions.begin();
        it != pushActions.end();
        ++it)
  {
      switch((*it) & clSourceClass::/*actions_t::*/mask)
      {
      case clSourceClass::/*actions_t::*/action_waitFor:
      {
        sourceClassID tgtClassID = static_cast<sourceClassID>((*it) &  clSourceClass::/*actions_t::*/idxMask);
        ETG_TRACE_USR4(("WaitForSource: WaitFor rule found: target source class %d"
               , tgtClassID));
        if (NULL != m_pcoAudioStack)
        {
          pTargetSrc =  m_pcoAudioStack->GetTopSource(tgtClassID);
          if(pTargetSrc!=NULL)
          {
             ETG_TRACE_USR4(("WaitFor rule found: SourceClass: %d subID %d WAITS FOR SourceClass: %d subID %d"
            , src.sGetId().enSourceClass
            , src.sGetId().u16SubSource
            , pTargetSrc->sGetId().enSourceClass
            , pTargetSrc->sGetId().u16SubSource));
            return pTargetSrc;
          }
          else
          {
            ETG_TRACE_USR4(("WaitFor rule found: No conflicting SourceClass active"));
          }
        }
       }
      break;
      default:
        //do nothing
      break;
      }
  }
  //No WaitFor rule or no corresponding active Source
  return pTargetSrc;
}

/*!************************************************************************
 * METHOD:        vOn_done
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the demute-ramp was finished.
 *                AudioSources send this acknowledge also, if no demute-ramp
 *                was performed. E.g. when an on() is requested while the
 *                AudioSource is already in on-state.
 *
 *                When this confirmation arrives, the source is considered
 *                to be started completely, so send the start-notification now.
 *
 *
 * PARAMETER:     src   The AudioSource which has finished the demute-ramp
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vOn_done(clAudioSource* src)
{
  //TODO check if we need to prevent NONE source
  //if (src != NULL && src->sGetId().enSourceClass != AudioSources::NONE)
  if (src != NULL)
  {
    //trace("---Notification: AudioSource Started: %s", src->pacGetName());
    ETG_TRACE_USR2(("vOn_done: SourceClass: %d subID %d"
            , src->sGetId().enSourceClass
            , src->sGetId().u16SubSource));
    if (m_poObserver != NULL)
    {
      m_poObserver->vAudioSourceStarted(src->sGetId());
    }
    
   /*
  * After MUTE_DIAGNOSIS source became on, systemProeprty for RemoteCtrl should be updated (NCG3D-5857)
  */
    if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_DIAGNOSIS"))
    {
      ETG_TRACE_USR3(("vOn_done, MUTE_DIAGNOSIS source started, Diagnosis session started"));
      clGeniviAudioCtrlAdapter::vRemoteControlUpdateDB(SYP_DIAG_REMOTE_CONTROL_ON);
    vCreateBackupStack();
    }
  else
    {
      ETG_TRACE_USR3(("vOn_done, Diagnosis session not started"));
    }

  /*
   * vdu1kor -> update sink state to "muted" after MUTE_ENTERTAIN(ON)[CAFI06-2180]
   * Use Case :  to handle below special use case, when Mute_Ent requested first and even before request is processed
   *         request for MUTE_AUDIO_OFF is received. In this use case when MUTE_ENTERTAIN is send to ON,
   *        sink state update is sent as "muted"
   *  |MUTE_AUDIO_OFF : on  |
   *  |MUTE_ENTERTAIN : off   |
   *  |MEDIA_PLAYER#4 : pause |
   *
   */
    if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_ENTERTAIN"))
    {
    /*
    * In below use case : MCAN Bose Amplifier need to be Muted, once TA is over
    *  |TUNER_TA_FM    : on        |
    *  |MUTE_ENTERTAIN : pause     |
    *  |TUNER_FM       : pause     |
    */
    if (0 == OSAL_s32StringCompare(src->pacGetName(), "MUTE_ENTERTAIN"))
    {
       ETG_TRACE_USR3(("vOn_done, MUTE_ENTERTAIN source started"));
       clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_MUTED);
    }
    if(NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource()){
      ETG_TRACE_USR3(("vOn_done, MUTE_ENTERTAIN source started, No Mix Source active, calling changeSinkMuteStateDB->Muted"));
      clGeniviAudioCtrlAdapter::changeSinkMuteStateDB(MS_MUTED,1);
    }
    else{
      ETG_TRACE_USR3(("vOn_done, MUTE_ENTERTAIN source started, Mix Source active, No need to send changeSinkMuteStateDB->Muted"));
    }

      /* Update System Property 'SYP_USER_MUTE_STATUS' with value 'SYP_USER_MUTE_STATUS_OFF' (NCG3D-21845) */
      clGeniviAudioCtrlAdapter::updateSystemPropertyDB(SYP_USER_MUTE_STATUS,SYP_USER_MUTE_STATUS_ON);
    }
    if(m_pcoAudioStack != NULL)
    {
      if(m_pcoAudioStack->bCheckStackStable() == true && (NULL != clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource()))
      {
        ETG_TRACE_USR3(("Stack is stable, vStackStableTimer() called"));
        m_pcoAudioStack->vStackStableTimer();
      }
    }

#if defined(VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_RN_AIVI)
  if(0 == OSAL_s32StringCompare(src->pacGetName(),"MUTE_AUDIO_OFF"))
    {
      //MCAN Bose Amplifier need to be DeMuted, once MUTE_ENT is removed
    clGeniviAudioCtrlAdapter::vSetEntertainmentMute(MS_MUTED);
    if(NULL == clGeniviAudioCtrlAdapter::pcoGetTopOfStackMixsource())
    {
      ETG_TRACE_USR3(("vOn_done, MUTE_ENTERTAIN source started, No Mix Source active, calling changeSinkMuteStateDB->Muted"));
      clGeniviAudioCtrlAdapter::changeSinkMuteStateDB(MS_MUTED,1);
    }
    else
    {
      ETG_TRACE_USR3(("vOn_done, MUTE_ENTERTAIN source started, Mix Source active, No need to send changeSinkMuteStateDB->Muted"));
      }
      /* Update Sound Property 'MSP_AUDIO_OFF_MUTE' with value 'SYP_USER_MUTE_STATUS_OFF' */
      clGeniviAudioCtrlAdapter::updateSoundPropertyDB(MSP_AUDIO_OFF_MUTE,AM_MSP_AUDIO_OFF_MUTE_ON);
    }
#endif
  }

#ifdef PAUSE_MIX_SOURCES
  tBool bTopOfSrc = FALSE;
  clAudioSource* topOfStackSrc = pcoGetTopOfStackAudiosource();
  clAudioSource* poActiveMixSrc = pcoGetTopOfStackMixsource(src);

  if( poActiveMixSrc != NULL)
  {
    // play a mix source which is higher than the new FgSrc
    if(poActiveMixSrc != NULL)
    {
      if (poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateOff()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreatePause()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToOff()
      || poActiveMixSrc->pclGetState() == clSrcStateFactory::pclCreateRampDownToPause())
      {
        //vTrace((tU16)TRACE_AUDSM_SRC_MIXON_AFTER_BG_ONDONE,poActiveMixSrc->u8GetId());

        ETG_TRACE_USR2(("vOn_done: Starting Mix Source SourceClass: %d subID %d"
                , poActiveMixSrc->sGetId().enSourceClass
                , poActiveMixSrc->sGetId().u16SubSource));

        poActiveMixSrc->vOn();
      }
    }
  }
#endif

  if (m_pcoAudioStack != NULL)
  {
    m_pcoAudioStack->trace();
  }
}

am_Error_e clAudioSourceController::vCreateBackupStack()
{
  am_Error_e ret = E_UNKNOWN;
  if(m_pcoBackupStack != NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, Remote Session already active"));
    return E_ALREADY_EXISTS;
  }

  // create a new DiagStack and assign it to
  ETG_TRACE_USR2(("clAudioSourceController::vCreateBackupStack"));
  if (NULL != m_pcoAudioStack)
  {
    // keep it as Backup
    m_pcoBackupStack = m_pcoAudioStack;

    // create a new Stack for RemoteControl
    m_pcoAudioStack = OSAL_NEW clDiagStack(1,"DIAG ");

    //We could play around now with the new stack

    //20.06.2017 for PSA: they want the Entertainment source on the Diag Stack
    //search for the entertain source below of MUTE_DIAGNOSIS
    clAudioSource* pSrc = pcoGetTopOfStackEntertainmentAudiosource(m_pcoBackupStack);
    if(pSrc != NULL)
    {
       //add also the MUTE_DIAGNOSIS
       const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_DIAGNOSIS");
       if(srcClass != NULL)
       {
          clAudioSource* pMute = clAudioSourceFactory::getAudioSource(srcClass->SourceClassID,0);
          if(pMute != NULL)
          {
             //bON(pMute->sGetId(),1,0);
             ETG_TRACE_USR2(("Entertainment on BackUp is %s",pSrc->pacGetName()));
             tBool bResult = FALSE;
             bResult = m_pcoAudioStack->AddSource(*pSrc);
             if(bResult)
                pSrc->vPause(pSrc->sGetId());
          }
          else
          {
             ETG_TRACE_ERR(("CMD_AUDIO, Could not get clAudioSource* for MUTE_DIAGNOSIS"));
          }
       }
       else
       {
          ETG_TRACE_ERR(("CMD_AUDIO, GetSourceClass(MUTE_DIAGNOSIS) returned NULL"));
       }
    }
    else
    {
       ETG_TRACE_ERR(("CMD_AUDIO, pcoGetTopOfStackEntertainmentAudiosource returned NULL"));
    }
    //end 20.06.2017 / 25.08.2017 (improved solution)
    // We could take care and forward replies from source MUTE_DIAGNOSIS to the backup stack
    ret = E_OK;
  }
  return ret;
}

/*!************************************************************************
 * METHOD:        vInit_started
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Callback from AudioSource
 *                The AudioSource indicates, that the init-state has been
 *                entered.
 *
 *                When this confirmation arrives, the source is considered
 *                to be about to pepare for play-request
 *
 * PARAMETER:     src   The concerned AudioSource
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vInit_started(clAudioSource* src)
{
   //TODO Check if NONE source check is needed
   //if (src != NULL && src->sGetId().enSourceClass != AudioSources::NONE)
   if (src != NULL)
   {
      ETG_TRACE_USR2(("vInit_started: SourceClass: %d subID %d"
            , src->sGetId().enSourceClass
            , src->sGetId().u16SubSource));

      clAudioSMEngine::vSetCurrentActiveSource(src->sGetId());

      if (m_poObserver != NULL)
      {
         ETG_TRACE_USR2(("vInit_started: call vAudioSourceInitStarted"));
         m_poObserver->vAudioSourceInitStarted(src->sGetId());
      }
   }

   if (m_pcoAudioStack != NULL)
   {
      m_pcoAudioStack->trace();
   }
}


/*!************************************************************************
 * METHOD:        vResetStack
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   Interface method to clean up the stack the "hard way"
 *                The Stack is completely cleaned up. All active sources are
 *                stopped and removed from the stack.
 *
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSourceController::vResetStack()
{
   ETG_TRACE_USR2(("vResetStack called without Sink -> using sink1"));
   vResetStack(1);
}

tVoid clAudioSourceController::vResetStack(int sinkID)
{
   m_pcoAudioStack = getStackForSink((tU16)sinkID);
   if(sinkID>1)
   {
      // ToDo: see how we can clean this stacks without muting main sink
      //sink disappeared, no need for silence or special care here
//      clAudioSource *pcoAudioSource = m_pcoAudioStack->GetTopSource();
//      while(pcoAudioSource)
//      {
//         pcoAudioSource->vOff(pcoAudioSource->sGetId(),sinkID);
//         pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
//      }
      m_pcoAudioStack->vResetStack();
      return;
   }


   if (NULL != m_pcoAudioStack)
   {
      ETG_TRACE_USR2(("vResetStack"));
      clAudioSource *pcoAudioSource = m_pcoAudioStack->GetTopSource();

      // ensure silence during stack-reset
      const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_ZERO_VOLUME");
      if(srcClass == NULL)
      {
         ETG_TRACE_FATAL(("NO SOURCE CLASS: MUTE_ZERO_VOLUME FOUND"));
         return;
      }
      sourceClassID muteClassID = srcClass->getClassID();
      bON(SourceID(muteClassID,0), 0);

      while(pcoAudioSource)
      {
         // prevent the system-mute to be stopped, because this could demute the audio !
         if (pcoAudioSource->sGetId().enSourceClass != muteClassID)
         {
            pcoAudioSource->vOff(pcoAudioSource->sGetId());
         }
         pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
      }
      m_pcoAudioStack->vResetStack();
      clAudioSourceFactory::resetAudioSources();
      //bSELECT( SourceID(AudioSources::NONE, 0));
      //bON(     SourceID(AudioSources::MUTE_LIMITED_SYSTEM, 0));
      bON(     SourceID(muteClassID, 0));
   }
}

tVoid clAudioSourceController::vTraceStack()
{
   ETG_TRACE_USR4(("Trace Stack entered"));
   if((m_pcoAudioStack != NULL) && (m_pcoBackupStack != NULL))
   {
     //DIAG Stack was not part of the array below so handle separately.
     m_pcoAudioStack->trace();
   }
   unsigned int stackID;
   for(stackID=0;stackID<m_allAudioStacks.size();stackID++)
   {
      clStack* pclStack = m_allAudioStacks[stackID];
      if(pclStack != NULL)
      {
         ETG_TRACE_USR4(("Trace Stack for Sink %d",pclStack->GetSinkID()));
         pclStack->trace();
      }
      else
      {
         ETG_TRACE_ERR(("CMD_AUDIO, ERROR: Found NULL Pointer in m_allAudioStacks[%d]",stackID));
      }
   }

}

tVoid clAudioSourceController::vTraceBackupStack() const
{
   if (m_pcoBackupStack != NULL)
   {
   ETG_TRACE_USR4(("PRINTING BACKUP STACK"));
     m_pcoBackupStack->trace();
   }
   else
   {
     ETG_TRACE_ERR(("CMD_AUDIO, No Remote session active, nothing to print"));
   }
}

tBool clAudioSourceController::bIsTopOfBackUpStackMuteAndOn()
{
  clAudioSource *pcoAudioSourceIterator = NULL;

  if(m_pcoBackupStack != NULL)
  {
    pcoAudioSourceIterator = m_pcoBackupStack->GetTopSource();
    if(pcoAudioSourceIterator == NULL)
      return false;
    //what have we found on top  of backup?
    const tChar* pSrcName = pcoAudioSourceIterator->pacGetName();
    if(pSrcName == NULL)
      return false;

    if(0 == OSAL_s32StringCompare(pSrcName,"MUTE_DIAGNOSIS"))
    {
      ETG_TRACE_USR3(("BACKUP TOP IS MUTE_DIAGNOSIS"));
      if(AudioStates::ON != pcoAudioSourceIterator->pclGetState()->enGetStateID())
      {
        ETG_TRACE_USR3(("MUTE_DIAGNOSIS not yet in ON -> backup"));
        // we return the backup
        return false;
      }
      else
      {
        ETG_TRACE_USR3(("MUTE_DIAGNOSIS is in ON"));
        //Fix for PSARCCB-3229 - paj5kor - updating property only when MUTE_DIAG is ON
        clGeniviAudioCtrlAdapter::vRemoteControlUpdateDB(SYP_DIAG_REMOTE_CONTROL_ON);
        return true;
      }
    }
    else
    {
      ETG_TRACE_USR3(("BACKUP TOP IS %s",pSrcName));
    }
  }
  return false;
}

clAudioSource* clAudioSourceController::pcoGetTopOfStack()
{
  if (NULL == m_pcoAudioStack) //vdu1kor : to fix issue SUZUKI-16997
  {
  ETG_TRACE_ERR(("CMD_AUDIO, pcoGetTopOfStack: m_pcoAudioStack is NULL"));
  return NULL;
  }
  clAudioSource *pcoAudioSourceIterator(m_pcoAudioStack->GetTopSource());
  return pcoAudioSourceIterator;
}
//returns AudioSource which is top of stack and Entertainment sources
clAudioSource* clAudioSourceController::pcoGetTopOfStackEntertainmentAudiosource(clStack* pStack)
{
  if(pStack != NULL)
  {
    clAudioSource *pcoAudioSourceIterator(pStack->GetTopSource());
    while(pcoAudioSourceIterator)
    {
      if(   (pcoAudioSourceIterator->getSourceType().stackable == false)
         && (pcoAudioSourceIterator->getSourceType().exclusive == true)
         && (pcoAudioSourceIterator->getSourceClass().SourceRegistration != clSourceClass::registerNone) )
      {
        //found what I want
        break;
      }

      pcoAudioSourceIterator = pStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }

    return pcoAudioSourceIterator;
  }
  return NULL;

}
//returns AudioSource which is top of stack, except Mix sources
clAudioSource* clAudioSourceController::pcoGetTopOfStackAudiosource(clStack* pStack)
{
  if(pStack != NULL)
  {
    // check if we reached the MUTE_DIAGNOSIS on state on the backup
    // if not -> this is still ongoing and we operate with the backup
      clAudioSource *pcoAudioSourceIterator(pStack->GetTopSource());
    while(pcoAudioSourceIterator && !(pcoAudioSourceIterator->getSourceType().exclusive))
    {
      pcoAudioSourceIterator = pStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }
    return pcoAudioSourceIterator;
  }
  return NULL;

}
//returns AudioSource which is top of stack, except Mix sources
clAudioSource* clAudioSourceController::pcoGetTopOfStackAudiosource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   (void) pcoAudioSourceIterator;

   //daw2hi: new for remote
   if( (m_pcoBackupStack != NULL) && (false==bIsTopOfBackUpStackMuteAndOn()) )
   {
    // check if we reached the MUTE_DIAGNOSIS on state on the backup
    // if not -> this is still ongoing and we operate with the backup
//    pcoAudioSourceIterator = m_pcoBackupStack->GetTopSource();
//    while(pcoAudioSourceIterator && !(pcoAudioSourceIterator->getSourceType().exclusive))
//    {
//      pcoAudioSourceIterator = m_pcoBackupStack->GetUnderlayingSource(*pcoAudioSourceIterator);
//    }
//    return pcoAudioSourceIterator;
    return pcoGetTopOfStackAudiosource(m_pcoBackupStack);
   }
   // daw2hi end

//   if (NULL != m_pcoAudioStack)
//   {
//      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();
//
//      while(pcoAudioSourceIterator && !(pcoAudioSourceIterator->getSourceType().exclusive))
//      {
//            pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
//      }
//   }
//   return pcoAudioSourceIterator;
   return pcoGetTopOfStackAudiosource(m_pcoAudioStack);
}

/**
 * @brief returns the AudioSource on stack which is not Mix AND NOT in Pause/Off
 * there is always exactly one result
 */
clAudioSource* clAudioSourceController::pcoGetCurrentActiveAudiosource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;

   //daw2hi: new for remote
   if( (m_pcoBackupStack != NULL) && (false==bIsTopOfBackUpStackMuteAndOn()) )
   {
    pcoAudioSourceIterator = m_pcoBackupStack->GetTopSource();

    while(pcoAudioSourceIterator && (!(pcoAudioSourceIterator->getSourceType().exclusive)
     || !(pcoAudioSourceIterator->bIsActive())) )
    {
      pcoAudioSourceIterator = m_pcoBackupStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }
    return pcoAudioSourceIterator;
   }
   // daw2hi end

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator && (!(pcoAudioSourceIterator->getSourceType().exclusive)
         || !(pcoAudioSourceIterator->bIsActive())) )
      {
            pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return pcoAudioSourceIterator;
}

/**
 * @brief returns the true/false if source is current active audio source
 *
 */
tBool clAudioSourceController::bIsCurrentActiveAudiosource(SourceID srcID)
{
  bool bResult = false;
  clAudioSource *pcoAudioSourceIterator = pcoGetCurrentActiveAudiosource();
  if((NULL != pcoAudioSourceIterator) &&
        (srcID == pcoAudioSourceIterator->sGetId()))
  {
    bResult = true;
  }
  return bResult;
}

/**
 * @brief returns the AudioSource on stack which is Mix AND NOT in Pause/Off
 * there is always exactly one result
 * @return NULL if there is no Mix Source active, else Pointer to active MixSource
 */
clAudioSource* clAudioSourceController::pcoGetCurrentMixSource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator && (!(pcoAudioSourceIterator->getSourceType().exclusive)
         || !(pcoAudioSourceIterator->bIsActive())) )
      {
         pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return pcoAudioSourceIterator;
}

/**
 * @brief most top MixAudioSource of the Stack
 * @return NULL if there is no Mix source on stack
 */
clAudioSource* clAudioSourceController::pcoGetTopOfStackMixsource()
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();
      // daw2hi: continue only if it is exclusive and leave when first none-exclusive is found

      while(pcoAudioSourceIterator && ((pcoAudioSourceIterator->getSourceType().exclusive)))
      {
          pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }

      if(pcoAudioSourceIterator && (!(pcoAudioSourceIterator->getSourceType().exclusive)))
      {
         return pcoAudioSourceIterator;
      }
   }
   return NULL;
}

/**
 * @brief Second most top MixAudioSource of the Stack
 * @return NULL if there is no Second Mix source on stack
 */
clAudioSource* clAudioSourceController::pcoGetSecondMixSource()
{
  //top Mix source
  clAudioSource *pcoTopMixSrc = pcoGetTopOfStackMixsource();

  clAudioSource *pcoAudioSourceIterator = NULL;
  if (NULL != m_pcoAudioStack)
  {
    pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

    // vdu1kor: continue only if it is exclusive or first mix source and leave when second none-exclusive is found
    while(pcoAudioSourceIterator &&
        ((((pcoAudioSourceIterator->getSourceType().exclusive))) || (pcoAudioSourceIterator->sGetId() == pcoTopMixSrc->sGetId())))
    {
      pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }

    if(pcoAudioSourceIterator && (!(pcoAudioSourceIterator->getSourceType().exclusive)))
    {
      return pcoAudioSourceIterator;
    }
  }
  return NULL;
}

/************************************************************************
 *FUNCTION     : pcoGetPossibleNextSource
 *DESCRIPTION  : function to get source below current active source
 *PARAMETER    : NIL
 *RETURN VALUE : clAudioSource*
 *HISTORY         :
 *04.01.2016   Rev 1.0     RBEI/ECV2 - Vyankatesh VD  Initial Revision
 ************************************************************************/
clAudioSource* clAudioSourceController::pcoGetPossibleNextSource()
{
  clAudioSource* pActiveSource = pcoGetCurrentActiveAudiosource();
  if(NULL == pActiveSource)
    return NULL;

  clAudioSource *pcoAudioSourceNext = NULL;
  if (NULL != m_pcoAudioStack)
  {
    pcoAudioSourceNext = m_pcoAudioStack->GetUnderlayingSource(*pActiveSource);
  }
  return pcoAudioSourceNext;
}

tBool clAudioSourceController::GetSinkIDsForSource(SourceID srcID, std::vector<int> &sinkIDs)
{

   tBool ret = false;
   //std::vector<int> sinkIDs;
   for(std::vector<clStack*>::iterator it=m_allAudioStacks.begin(); it!=m_allAudioStacks.end(); ++it)
   {
      ETG_TRACE_USR4(("bIsSourceOnStack for SinkID %d",(*it)->GetSinkID()));
      if(bIsSourceOnStack(srcID,tU16((*it)->GetSinkID())))
      {
         ETG_TRACE_USR4(("bIsSourceOnStack yes "));
         sinkIDs.push_back((*it)->GetSinkID());
         ret = true;
      }
      else
      {
         ETG_TRACE_USR4(("bIsSourceOnStack no "));
      }
   }
   return ret;
}
/************************************************************************
 *FUNCTION     : pcoGetFirstNonMuteSrcFromStack
 *DESCRIPTION  : function to get first non mute source from stack
 *PARAMETER    : NIL
 *RETURN VALUE : clAudioSource*
 *HISTORY         :
 *30.06.2016   Rev 1.0     RBEI/ECO12 - Vyankatesh VD  Initial Revision
 ************************************************************************/
clAudioSource* clAudioSourceController::pcoGetFirstNonMuteSrcFromStack()
{
  clAudioSource *pcoAudioSourceIterator = NULL;
  if (NULL != m_pcoAudioStack)
  {
    pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();
    while(pcoAudioSourceIterator != NULL)
    {
      if(clSourceClass::group_mute != pcoAudioSourceIterator->getSourceClass().getGroupID())
      {
        return pcoAudioSourceIterator;
      }
      pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
    }
  }
  return NULL;
}

tBool clAudioSourceController::bIsSourceOnStack(SourceID srcID)
{
   // 20.07.2016 daw2hi: we should check if srcID is on "any" stack
  /* for(std::vector<clStack*>::iterator it=m_allAudioStacks.begin(); it!=m_allAudioStacks.end(); ++it)
   {
      ETG_TRACE_USR4(("bIsSourceOnStack for SinkID %d",(*it)->GetSinkID()));
      if(bIsSourceOnStack(srcID,tU16((*it)->GetSinkID())))
      {
         ETG_TRACE_USR4(("bIsSourceOnStack yes "));
      }
      else
      {
         ETG_TRACE_USR4(("bIsSourceOnStack no "));
      }
   }*/

  ETG_TRACE_USR4(("bIsSourceOnStack use SinkID 1"));
  return bIsSourceOnStack(srcID, 1);
}

tBool clAudioSourceController::bIsSourceOnStack(SourceID srcID, tU16 SinkID)
{

   clStack* pStack = clAudioSourceController::getInstance().getStackForSink(SinkID);

   clAudioSource *pcoAudioSourceIterator = NULL;
   if (NULL != pStack)
   {
      pcoAudioSourceIterator = pStack->GetTopSource();
      while(pcoAudioSourceIterator != NULL)
      {
         if(pcoAudioSourceIterator->sGetId() == srcID)
         {
            return TRUE;
         }
         pcoAudioSourceIterator = pStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return FALSE;
}

tBool clAudioSourceController::bIsSourceOnBackUpStack(SourceID srcID)
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   if (NULL != m_pcoBackupStack)
   {
      pcoAudioSourceIterator = m_pcoBackupStack->GetTopSource();
      while(pcoAudioSourceIterator != NULL)
      {
         if(pcoAudioSourceIterator->sGetId() == srcID)
         {
            return TRUE;
         }
         pcoAudioSourceIterator = m_pcoBackupStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return FALSE;
}

tBool clAudioSourceController::bIsAnotherEntSrcOnStack(SourceID srcID)
{
   if(clFactory_AudioSourceClass::GetType(srcID).stackable)
   {
     ETG_TRACE_ERR(("CMD_AUDIO, bIsAnotherEntSrcOnStack: passes source is stackable"));
     return FALSE;
   }
   clAudioSource *pcoAudioSourceIterator = NULL;
   if (NULL != m_pcoAudioStack)
   {
     pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();
     while(pcoAudioSourceIterator != NULL)
     {
     SourceID newSrcID(pcoAudioSourceIterator->sGetId());

     if((newSrcID != srcID) && (!clFactory_AudioSourceClass::GetType(newSrcID).stackable) )
     {
         ETG_TRACE_USR4(("bIsAnotherEntSrcOnStack: there is another Entertainment src on stack,SrcId:%d,SubId:%d",newSrcID.enSourceClass,newSrcID.u16SubSource ));
       return TRUE;
     }
     pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
     }
   }
   return FALSE;
}

/**
 * @briefCopies pointer AudioSources that are actually on Stack
 * to given List.
 * @param sourceList
 * @return
 */
tVoid clAudioSourceController::vGetStackList(std::list<  clAudioSource * > &sourceList)
{
   if(NULL == m_pcoAudioStack)
   {
     ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: m_pcoAudioStack is NULL"));
     return;
   }
   //Get TopSource
   clAudioSource *pcoAudioSource = m_pcoAudioStack->GetTopSource();
   if(pcoAudioSource != NULL)
   {
      ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Top SourceClass: %d subID %d"
         , pcoAudioSource->sGetId().enSourceClass
         , (tU16)pcoAudioSource->sGetId().u16SubSource));
   }else{
      ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: TopSource is NULL"));
   }
   //Add every source of the Stack
   while(pcoAudioSource)
   {
      //VVD lint fix
    //if(pcoAudioSource != NULL)
      //{

      ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Push SourceClass: %d subID %d"
         , pcoAudioSource->sGetId().enSourceClass
         , (tU16)pcoAudioSource->sGetId().u16SubSource));

      //}
      sourceList.push_back(pcoAudioSource);
      pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
   }
}


/*!************************************************************************
 * METHOD:        bIsAudioMuted
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   specifies, if the audio is currently muted
 *                This method return true, if (and only if) one of the virtual
 *                mute sources: [SYSTEM_MUTE, PIN_MUTE, ENTERTAINMENT_MUTE
 *                LIMITED_SYSTEM_MUTE]
 *                is on top of stack.
 *                This method does not take care of temp. Phone-,
 *                Cluster- and Diagnosis-Mute
 *                If a virtual Mute-Source is on top of stack, this method
 *                returns TRUE, no matter if the AudioSource, which was active
 *                before, is already completely muted. (Mute-Ramp running)
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
//tBool clAudioSourceController::bIsAudioMuted()
//{
//   // check for top of stack audio source (excluding MIX sources)
//   clAudioSource *pcoAudioSource = pcoGetTopOfStackAudiosource();
//   if (pcoAudioSource != NULL)
//   {
//      if(pcoAudioSource->getGroup() == clStackRules::group_t::groupInternal)
//      {
//         return TRUE;
//      }
//   }
//   return FALSE;
//}

/*!************************************************************************
 * METHOD:        bIsEntertainmentSourcePaused
 * CLASS:         clAudioSourceController
 * DESCRIPTION:
 * PARAMETER:
 * RETURNVALUE:
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
//tBool clAudioSourceController::bIsEntertainmentSourcePaused()
//{
//   tBool bResult = FALSE;
//
//   // check for top of stack audio source (excluding MIX sources)
//   clAudioSource *pcoAudioSource = pcoGetTopOfStackAudiosource();
//   if (pcoAudioSource != NULL)
//   {
//      if (pcoAudioSource->getType() != clStackRules::typeBg)
//      {
//         // there is a None-Entertainment-Source on top of stack, so
//         // the entertainment source is muted
//         bResult = TRUE;
//      }
//   }
//   return bResult;
//}



#if 0
/*!************************************************************************
 * METHOD:        u8GetActiveAudioComponent
 * CLASS:         clAudioSourceController
 * DESCRIPTION:   returns the current audiocomponent
 *                IMPORTANT NOTE:
 *                1) This method returns the asynchronous component!
 *                This means, that during a running source switch (mute-demute-ramp)
 *                the currently active source is returned, taking care of mute-phases.
 *                OBSOLETE:
 *                1) This method returns the synchronous component!
 *                This means, that during a running source switch (mute-demute-ramp)
 *                already the new source is returned, even if the previous
 *                source is still in mute-ramp.
 *                END OBSOLETE
 *                2) This method refers to valid audio components only.
 *                This means, that if there are virtual sources  on top of the
 *                stack (e.g. System-Mute, Entertainment-Mute, etc), they
 *                will be ignored by this method!
 *
 *
 * PARAMETER:
 * RETURNVALUE:   [AudioComponent::NAVI, AudioComponent::PHONE,
 *                 AudioComponent::RADIO_ANNOUNCEMENT, AudioComponent::AUDIO]
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/

tU8 clAudioSourceController::u8GetActiveAudioComponent()
{
   clAudioSource *pcoAudioSourceIterator = NULL;
   tU8 u8ActiveAudioComponent = AudioComponent::AC_AUDIO;
   tU8 u8CurrentAudioSource = AudioSources::NONE;

   if (NULL != m_pcoAudioStack)
   {
      pcoAudioSourceIterator = m_pcoAudioStack->GetTopSource();

      while(pcoAudioSourceIterator)
      {
         u8CurrentAudioSource = pcoAudioSourceIterator->u8GetId();
         /* exclude rampDown Phases if you require synchronous behaviour
         if (pcoAudioSourceIterator->pclGetState() != clSrcStateFactory::pclCreateRampDownToPause()
            && pcoAudioSourceIterator->pclGetState() != clSrcStateFactory::pclCreateRampDownToOff())
            */
         {
            if (u8CurrentAudioSource == AudioSources::NAVISPEECH )
            {
               u8ActiveAudioComponent = AudioComponent::AC_NAVI;
               break;
            }
            else if (u8CurrentAudioSource == AudioSources::TELEPHONE)
            {
               u8ActiveAudioComponent = AudioComponent::AC_PHONE;
               break;
            }
            else if (u8CurrentAudioSource == AudioSources::RADIO_TA)
            {
               u8ActiveAudioComponent = AudioComponent::AC_RADIO_ANNOUNCEMENT;
               break;
            }
            else if (pcoAudioSourceIterator->getType() == clStackRules::typeBg)
            {
               u8ActiveAudioComponent = AudioComponent::AC_AUDIO;
               break;
            }
         }

         // iterator points to a source which is not of a valid AudioComponent type, so continues
         // iterations going down the stack
         pcoAudioSourceIterator = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSourceIterator);
      }
   }
   return u8ActiveAudioComponent;
}
#endif

/**
 * Check if given source is allowed to Enter Stack
 * @param poAudioSource stack is checked against this source
 * @return
 */
clSourceClass::allow_t clAudioSourceController::enCheckSource(clAudioSource* poAudioSource)
{
  //VVD lint fix
   clSourceClass::allow_t allow = clSourceClass::disallowed;
   if(m_pcoAudioStack)
      if( m_pcoAudioStack->AllowSource(*poAudioSource) != NULL)
         allow = clSourceClass::allowed;

   //else
   //   return clSourceClass::disallowed; // only for lint, case will never happen

   return allow;
}

/**
 * Retrieves most top exclusive Audiosource ID
 * @return if the stack is empty NONE is returned
 */
//SourceID clAudioSourceController::u8GetTopOfStackAudiosource()
//{
//   clAudioSource* poTopStackSrc = NULL;
//   SourceID  sTopStackSrc (AudioSources::NONE,0);
//
//   poTopStackSrc = pcoGetTopOfStackAudiosource();
//   if(poTopStackSrc != NULL)
//   {
//      sTopStackSrc =  poTopStackSrc->sGetId();
//   }
//   return sTopStackSrc;
//}

tVoid clAudioSourceController::vHandleSrcActError(SourceID errSrc)
{
   //Get errSrc Obj
   clAudioSource* pErrSrc = clAudioSourceFactory::getAudioSource(errSrc);

   //Iterate through list and find Source in RampUp/Down state
   if(pErrSrc != NULL)
   {
      clSrcState* state = pErrSrc->pclGetState();
      (void) state;
      ETG_TRACE_ERR(("CMD_AUDIO, vHandleSrcActError:  SourceClass: %d subID %d, State: %d"
            , errSrc.enSourceClass
            , errSrc.u16SubSource
            , ETG_CENUM(AudioStates::enAudioStates,   pErrSrc->pclGetState()->enGetStateID())));
      //Reset Source
      ETG_TRACE_ERR(("CMD_AUDIO, vHandleSrcActError: Reset source"));
      //CRQ-1300254 [Update of error memory when source change timeout occurs for gen3]
      ETG_TRACE_ERRMEM(("CMD_AUDIO, vHandleSrcActError: Src name %s",pErrSrc->pacGetName()));
      pErrSrc->vReset();
      // Off_Done is not called implicitly when reset the source
      pErrSrc->vNotifyOffDone();
   }
}

// daw2hi: Remote Ctrl
am_Error_e clAudioSourceController::vRemoteCtrlStart()
{
  am_Error_e ret = E_UNKNOWN;
  if(m_pcoBackupStack != NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, Remote Session already active"));
    return E_ALREADY_EXISTS;
  }

  // create a new DiagStack and assign it to
  ETG_TRACE_USR2(("clAudioSourceController::vRemoteCtrlStart"));
  if (NULL != m_pcoAudioStack)
  {
    //ToDo: Put a mute on the "old"stack
    const clSourceClass* srcClass = clFactory_AudioSourceClass::GetSourceClass("MUTE_DIAGNOSIS");

    if(srcClass == NULL)
    {
       ETG_TRACE_FATAL(("NO SOURCE CLASS: MUTE_DIAGNOSIS FOUND"));
       return E_NON_EXISTENT;
    }
    sourceClassID muteClassID = srcClass->getClassID();
    SourceID srcID(muteClassID,0);

    if(clAudioSourceController::getInstance().bON(srcID))
      ret = E_OK;

    clAudioSourceController::getInstance().vTraceStack();
    ETG_TRACE_USR2(("clAudioSourceController::vRemoteCtrlStart backup Stack now !!"));
  }
  return ret;
}

am_Error_e clAudioSourceController::vRemoteCtrlEnd()
{
  //ToDo: All sources present on the DiagStack and not on the Backup
  // have to be sent to off
  //Bring sources modified during remote to the previous state (always Pause?)

  am_Error_e ret = E_UNKNOWN;
  if(m_pcoBackupStack == NULL)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, No Remote Session active"));
    return E_NON_EXISTENT;
  }

  std::list<clAudioSource *> sourceListBackup;
  std::list<clAudioSource *>::iterator sourceListBackupIter;

  //Get TopSource on Backup
  clAudioSource *pcoAudioSource = m_pcoBackupStack->GetTopSource();
  if(/*m_pcoBackupStack*/pcoAudioSource != NULL) /** Fix SIGSEGV (PSARCCB-6421) **/
  {
    ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Top SourceClass: %d subID %d"
        , pcoAudioSource->sGetId().enSourceClass
        , (tU16)pcoAudioSource->sGetId().u16SubSource));
  }else{
    ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: TopSource is NULL"));
  }

  ETG_TRACE_USR3(("Building BackupStackList"));

  //Add every source of the BackupStack
  while(pcoAudioSource)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Push SourceClass: %d subID %d"
        , pcoAudioSource->sGetId().enSourceClass
        , (tU16)pcoAudioSource->sGetId().u16SubSource));

    pcoAudioSource->vPrint();
    sourceListBackup.push_back(pcoAudioSource);
    pcoAudioSource = m_pcoBackupStack->GetUnderlayingSource(*pcoAudioSource);
  }
  ETG_TRACE_USR3(("Building BackupStackList done\n\n"));

  //Get TopSource
  if(m_pcoAudioStack != NULL)
  {
    pcoAudioSource = m_pcoAudioStack->GetTopSource();
    if(pcoAudioSource != NULL)
    {
      ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Top SourceClass: %d subID %d"
          , pcoAudioSource->sGetId().enSourceClass
          , (tU16)pcoAudioSource->sGetId().u16SubSource));
    }else{
      ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: TopSource is NULL"));
    }
  }
  else
    ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: m_pcoAudioStack is NULL"));

  ETG_TRACE_ERR(("CMD_AUDIO, running the while to check..."))
  // check for each source if also avail on backup
  while(pcoAudioSource)
  {
    ETG_TRACE_ERR(("CMD_AUDIO, vGetStackList: Push SourceClass: %d subID %d"
        , pcoAudioSource->sGetId().enSourceClass
        , (tU16)pcoAudioSource->sGetId().u16SubSource));

    ETG_TRACE_ERR(("CMD_AUDIO, enter the for loop ..."));
    tBool bFound=false;
    // Now check if we have this on the backup:
    for(sourceListBackupIter=sourceListBackup.begin();
        sourceListBackupIter!=sourceListBackup.end();
        sourceListBackupIter++)
    {
      if(pcoAudioSource == *sourceListBackupIter)
      {
        bFound=true;
        ETG_TRACE_ERR(("CMD_AUDIO, found source on both: %s",
            (*sourceListBackupIter)->pacGetName()));
        ETG_TRACE_ERR(("CMD_AUDIO, in state: %s",
            (*sourceListBackupIter)->pacGetState()));
        break;
      }
      else
      {
        ETG_TRACE_ERR(("CMD_AUDIO, skip source: %s",(*sourceListBackupIter)->pacGetName()));
        ETG_TRACE_ERR(("CMD_AUDIO, in state: %s",(*sourceListBackupIter)->pacGetState()));
      }
    }
    if(bFound==true)
    {
      // if source is on current and on backup
      // ToDo check if in Pause and if not, send backup to pause
      (*sourceListBackupIter)->vPause(pcoAudioSource->sGetId());
      // should be the same as this, cause they are equal
      //pcoAudioSource->vPause(pcoAudioSource->sGetId());
    }
    else
    {
      // send the one on current stack and not present on backup to off
      pcoAudioSource->vOff(pcoAudioSource->sGetId());
    }
    pcoAudioSource = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
  }


  // Delete the Remote Stack and restore the Backup
  ETG_TRACE_USR3(("clAudioSourceController::vRemoteCtrlEnd"));
  OSAL_DELETE m_pcoAudioStack;
  m_pcoAudioStack = m_pcoBackupStack;
  m_pcoBackupStack = NULL;

  //Now finally remove the MUTE_DIAGNOSE
  pcoAudioSource = m_pcoAudioStack->GetTopSource();

  if(pcoAudioSource != NULL)
  {
    clAudioSource *pcoAudioBelowTop = m_pcoAudioStack->GetUnderlayingSource(*pcoAudioSource);
    if(pcoAudioBelowTop != NULL)
    {
      ETG_TRACE_USR3(("sending %s to Pause",pcoAudioBelowTop->pacGetName()));
      pcoAudioBelowTop->vPause(pcoAudioBelowTop->sGetId());
    }

    //We should have DIAGNOSE_MTE but we can also check it
    if(clAudioSourceController::getInstance().bOFF(pcoAudioSource->sGetId()))
      ret = E_OK;
  }
  else
  {
    ETG_TRACE_ERR(("CMD_AUDIO, vRemoteCtrlEnd: pcoAudioSource is NULL"));
  }

  return ret;
}

tVoid clAudioSourceController::vAddSinkClass(const am_Sink_s& sinkClass)
{
   if(sinkClass.sinkClassID > 1) //to keep backward compatibility with old xml assume first stack already created
   {
      ETG_TRACE_USR3(("vAddSinkClass -> create a stack for sinkClassID %d, name %s",sinkClass.sinkClassID, sinkClass.name.c_str()));
      m_allAudioStacks.push_back(OSAL_NEW clStack(sinkClass.sinkClassID, sinkClass.name.c_str()));
   }
   else
   {
      ETG_TRACE_USR3(("no stack creation for default sink sinkClassID %d, name %s",sinkClass.sinkClassID, sinkClass.name.c_str()));
      ETG_TRACE_USR4(("assume this was done already (backward compatibility with old xml"));
   }
}

tVoid clAudioSourceController::vRemoveSinkClass(tU16 sinkClassID)
{
   ETG_TRACE_ERR(("CMD_AUDIO, vRemoveSinkClass ID %d -> delete a stack ??",sinkClassID));
}
} }//namespace
