/*!************************************************************************
 * FILE :         clAudioSource.cpp
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/

#include "string.h"

#include <stdlib.h>
#include <time.h>
#include <pthread.h>

#include "AudioStack/clAudioSMEngine.h"
#include "AudioStack/AudioSources/clAudioSource.h"
#include "AudioStack/SMT/clSrcStateFactory.h"
#include "AudioStack/SMT/clSrcState.h"

#include "AudioStack/clGeniviAudioCtrlAdapter.h"
#include "AudioStack/AudioSources/clAudioSourceFactory.h"
#include "AudioStack/AudioSources/clFactory_AudioSourceClass.h"

#ifndef USE_DLT_TRACE
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_COMP_AUDIOSTACK
#include "trcGenProj/Header/clAudioSource.cpp.trc.h"
#endif

#define WELCOME_SOUND_TIMEOUT 25
namespace AudioStack
{
using namespace SourceStateMachine;
namespace AudioSource
{

CAmSocketHandler* clAudioSource::m_pSocketHandler       = NULL;

/**
 * Public constructor for instanziating a source that won't do
 * external communications. Could be used as mute source
 */
clAudioSource::clAudioSource(SourceID enSrcId) :
       m_pclState(NULL)
       , m_pacName(NULL)
       , m_SrcId(enSrcId)
       , m_possibleNextSource(enSrcId)
       , m_u32UserData(0)
       , m_bWelcomeSound(false)
       , m_u32PlayCmdCounter(0)
      // , m_StackRule(enSrcId)
       , m_pclAudioSourceObserver(NULL)
       , m_availability(clAudioSource::not_available)
       , m_availabilityReason(clAudioSource::nomedia)
       , m_handle(0)

{
   std::string str("UNDEF");
   const clSourceClass& srcClass = clFactory_AudioSourceClass::GetSourceClass(enSrcId.enSourceClass);
   if(  (srcClass.getRegistrationMode() == clSourceClass::registerNone)
      ||(srcClass.getRegistrationMode() == clSourceClass::registerStatic))
//   if((m_StackRule.getRegistration() == clStackRules::registerNone)
//         || (m_StackRule.getRegistration() == clStackRules::registerStatic))
   {
      str = srcClass.getName();
      m_availability = clAudioSource::available;
      m_availabilityReason = clAudioSource::samemedia;
   }

   char *cstr = new char[str.length() + 1];

   //VVD lint fix
   if(cstr != NULL)
  strcpy(cstr, str.c_str());

   m_pacName = cstr;

   m_curActSinkID = srcClass.defaultSink;  //daw2hi: 24.8.2016 better to use default sink, cause it could be also other than 1

   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);

   // put them in the map
   sinkStateMap[1]=m_pclState;

   m_bSourceIsLSM = false;
   m_bSourceIsProfChg = false;
   m_bSwitchtoDefaultSrcOnProfChg = false;

   m_bBackgroundSrcExchange = false;
   m_bPrevSrcRemoved = false;

   ETG_TRACE_USR4(("Constructor clAudioSource(SourceID enSrcId) %d, %d, p=%x  %s",
       (tU16)m_SrcId.enSourceClass,m_SrcId.u16SubSource,(void *)this,m_pacName));
}

/**
 * \brief
 * Constructor
 *
 * \param pacName
 * name represents the audio source using a string
 *
 * \param u8Id
 * u8Id represents the audio source
 *
 * Constructor
 */
clAudioSource::clAudioSource(const tChar *pacName, SourceID SrcId) :
             m_pclState(NULL)
             , m_pacName(NULL)
             , m_SrcId(SrcId)
             , m_possibleNextSource(SrcId)
             , m_u32UserData(0)
             , m_u32PlayCmdCounter(0)
            // , m_StackRule(SrcId)
             , m_pclAudioSourceObserver(NULL)
             , m_availability(clAudioSource::not_available)
             , m_availabilityReason(clAudioSource::newmedia)
       , m_handle(0)
{

   m_pacName = NULL;

   m_curActSinkID = 1;

   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);

   // put them in the map
   sinkStateMap[1]=m_pclState;

   m_bSourceIsLSM = false;
   m_bSourceIsProfChg = false;
   m_bSwitchtoDefaultSrcOnProfChg = false;
   m_bBackgroundSrcExchange = false;
   m_bPrevSrcRemoved = false;

   //VVD lint fix
   if(NULL == pacName)
        return;

   std::string str(pacName);
      char *cstr = new char[str.length() + 1];

      //VVD lint fix
   if(cstr != NULL)
    strcpy(cstr, str.c_str());

   m_pacName = cstr;

   ETG_TRACE_USR3(("Constructor clAudioSource(const tChar *pacName, SourceID SrcId) %d, %d, p=%x  %s",
       (tU16)m_SrcId.enSourceClass, m_SrcId.u16SubSource, (void *)this, m_pacName));
}
void clAudioSource::addSink(int sinkID)
{
   // if this sink is not yet added, add it now
   // check if sink is in sinkStateMap
   // sink -> state
   // std::map<tU16,SourceStateMachine::clSrcState*> sinkStateMap;
   std::map<tU16,SourceStateMachine::clSrcState*>::iterator it;
   it = sinkStateMap.find((tU16)sinkID);
   if(it==sinkStateMap.end())
   {
      //not yet present,we add it now
      SourceStateMachine::clSrcState* pclState = clSrcStateFactory::pclCreateOff();
      pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);

      sinkStateMap[(tU16)sinkID] = pclState;
   }
}
tVoid clAudioSource::vMW_CCAResponse(tS32 extSourceId, tU16 subSrc, enSourceActivity activity)
{
  (void) extSourceId;
  (void) subSrc;
  (void) activity;
   ETG_TRACE_ERR(("CMD_AUDIO, ERROR: vMW_CCAResponse called for base class %d, subID %d",
         m_SrcId.enSourceClass, (tU16)m_SrcId.u16SubSource));
}
tVoid clAudioSource::vMW_CCAResponse(tS32 extSourceId, tU16 subSrc, enSourceActivity activity, tU16 sinkID)
{
  (void) extSourceId;
  (void) subSrc;
  (void) activity;
  (void) sinkID;
   ETG_TRACE_ERR(("CMD_AUDIO, ERROR: vMW_CCAResponse called for base class %d, subID %d, sinkID %d",
         m_SrcId.enSourceClass, (tU16)m_SrcId.u16SubSource, sinkID));
}
const clSourceClass& clAudioSource::getSourceClass()
{
   return clFactory_AudioSourceClass::GetSourceClass(sGetId().enSourceClass);
}

const stSourceType& clAudioSource::getSourceType()
{
   return clFactory_AudioSourceClass::GetType(sGetId());
}

clSourceClass::restore_t clAudioSource::getRestoreValue()
{
   const clSourceClass& SrcClass = clFactory_AudioSourceClass::GetSourceClass(sGetId().enSourceClass);
   return SrcClass.Restore;
}

bool clAudioSource::bIsSourcePrevious()
{
   const clSourceClass& SrcClass = clFactory_AudioSourceClass::GetSourceClass(sGetId().enSourceClass);
   return SrcClass.Previous;
}

tVoid clAudioSource::Initialize(const tChar *pacName, SourceID srcId)
{
  (void) pacName;
  (void) srcId;

   /* check the name and set the stack rules */
   /// OBSOLETE DUE TO STACKRULE CONSTRUCTOR
   /// REQUIRES SOURCEID AND SETS CORRECTLY
   /// WITHIN INIT LIST
//   if (!m_StackRule.setRules(this))
//   {
//      ETG_TRACE_ERRMEM(("CMD_AUDIO, Could not find Rule for SourceClass: %d",
//            ETG_CENUM(AudioSources::enAudioSources,static_cast<AudioSources::enAudioSources>(srcId.enSourceClass))));
//      return;
//   }
   // init the state machine
}

/**
* /brief
* Destructor
*/
clAudioSource::~clAudioSource()
{
  ETG_TRACE_USR4(("Destructor clAudioSource() p=%x",(void*)this));

  if(m_handle != 0)
  {
    if(m_pSocketHandler != NULL)
    {
      //Stop and remove the timer from mainloop
      am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
      ETG_TRACE_USR4(("Destructor clAudioSource() after removeTimer,m_handle=%d,error=%d",m_handle,(tU16)ret));
      m_pSocketHandler = NULL;
    }
  }
  m_pclAudioSourceObserver = NULL;
  OSAL_DELETE[] m_pacName;
  m_pclState = NULL;
   sinkStateMap.clear(); //ToDO check if we need to erase/free anything from elements

  m_bSourceIsLSM = false;
  m_bSourceIsProfChg = false;
  m_bSwitchtoDefaultSrcOnProfChg = false;
  m_bBackgroundSrcExchange = false;
  m_bPrevSrcRemoved = false;
  m_curActSinkID = 0;

}

void clAudioSource::vSetSourceBG(tBool bBGsrc)
{
  ETG_TRACE_USR3(("clAudioSource::vSetSourceBG"));
  m_bBackgroundSrcExchange = bBGsrc;
}

tVoid clAudioSource::vSetPrevSrcRemoved(tBool bPrevSrcRemoved)
{
  ETG_TRACE_USR3(("clAudioSource::vSetPrevSrcRemoved"));
  m_bPrevSrcRemoved = bPrevSrcRemoved;
}
tBool clAudioSource::bIsSourceBG()
{
  return m_bBackgroundSrcExchange;
}
tBool clAudioSource::bIsPrevSourceRemoved()
{
  return m_bPrevSrcRemoved;
}

/**
* /brief
* vReset() sets all internal members to an initial state
*/
void clAudioSource::vSetSourceLSM(tBool bLSM)
{
  // ToDo: Any LSM for RSE ??
  if(bLSM == true)
  {
    ETG_TRACE_USR3(("clAudioSource::vSetSourceLSM true"));
    m_bSourceIsLSM = true;
  }
  else
  {
    ETG_TRACE_USR3(("clAudioSource::vSetSourceLSM false"));
    m_bSourceIsLSM = false;
  }
}

tBool clAudioSource::bIsSourceLSM()
{
  ETG_TRACE_USR4(("clAudioSource::bIsSourceLSM(%d)", m_bSourceIsLSM));
  return m_bSourceIsLSM;
}

/**
* /brief
* vReset() sets all internal members to an initial state
*/
void clAudioSource::vSetSourceProfChg(tBool bSrcProfChg)
{
    ETG_TRACE_USR4(("clAudioSource::vSetSourceProfChg, bSrcProfChg(%d)", bSrcProfChg));
    m_bSourceIsProfChg = bSrcProfChg;
}

tBool clAudioSource::bIsSourceProfChg()
{
   ETG_TRACE_USR4(("clAudioSource::bIsSourceProfChg(%d)", m_bSourceIsProfChg));
   return m_bSourceIsProfChg;
}

void clAudioSource::vSetSwitchtoDefaultSrcOnProfChg(tBool bSwitchtoDefaultSrc)
{
    ETG_TRACE_USR4(("clAudioSource::vSetSwitchtoDefaultSrcOnProfChg, bSwitchtoDefaultSrc(%d)", bSwitchtoDefaultSrc));
    m_bSwitchtoDefaultSrcOnProfChg = bSwitchtoDefaultSrc;
}

tBool clAudioSource::bSwitchtoDefaultSrcOnProfChg()
{
   ETG_TRACE_USR4(("clAudioSource::bSwitchtoDefaultSrcOnProfChg(%d)", m_bSwitchtoDefaultSrcOnProfChg));
   return m_bSwitchtoDefaultSrcOnProfChg;
}

void clAudioSource::vReset()
{
   ETG_TRACE_USR4(("clAudioSource::vReset"));
   // init the state machine
   m_pclState = clSrcStateFactory::pclCreateOff();
   m_pclState->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);
   //should go through whole map and reset every entry
   std::map<tU16,SourceStateMachine::clSrcState*>::iterator it;
   for(it=sinkStateMap.begin();it!=sinkStateMap.end();++it)
   {
      it->second = clSrcStateFactory::pclCreateOff();
      (it->second)->vEnter(this, 0, clSrcState::ENTRYCONDITION_INITIAL);
   }

   m_u32UserData = 0;
   m_possibleNextSource = SourceID(0, 0);
   m_u32PlayCmdCounter = 0;
   m_bSourceIsLSM = false;
   m_bSourceIsProfChg = false;
   m_bSwitchtoDefaultSrcOnProfChg = false;
   m_bBackgroundSrcExchange = false;
   m_bPrevSrcRemoved = false;

   /* Stop and remove the timer from mainloop as all sources will be deleted and removed from stack to avoid SIGSEGV (SUZUKI-22771) */
   if(m_handle != 0)
   {
    if(m_pSocketHandler != NULL)
    {
      //Stop and remove the timer from mainloop
      am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
    ETG_TRACE_USR4(("Destructor clAudioSource() after removeTimer,m_handle=%d,error=%d",m_handle,(tU16)ret));
      ETG_TRACE_USR4(("clAudioSource::vReset() called, Stop and remove the timer from mainloop, p=%x, m_handle=%d, error=%d",(void*)this,m_handle,(tU16)ret));
      m_pSocketHandler = NULL;
    }
  m_handle = 0;
   }
}

void clAudioSource::vPrint()
{
   ETG_TRACE_USR4(("%s"
      , m_pacName));
   ETG_TRACE_USR4(("  | SourceClass: 0x%02x(%03d), SubID 0x%02x(%03d),"
      , m_SrcId.enSourceClass
      , m_SrcId.enSourceClass
      , m_SrcId.u16SubSource
      , m_SrcId.u16SubSource));
   ETG_TRACE_USR4(("  | MidwID 0x%02x(%03d)"
      , clFactory_AudioSourceClass::GetSourceClass(m_SrcId.enSourceClass).getExtID()
      , clFactory_AudioSourceClass::GetSourceClass(m_SrcId.enSourceClass).getExtID()));
   //daw2hi: 8.8.2016 print always (only) for sink 1
   const tChar* pStateName = pacGetState(1);
   ETG_TRACE_USR4(("  | State: %d (%s)"
         , ETG_CENUM(AudioStates::enAudioStates, pclGetState(1)->m_enStateID),pStateName));


   ETG_TRACE_USR4(("  | Availability: %d"
      , ETG_CENUM(clAudioSource::enSourceAvailability, m_availability)));
   ETG_TRACE_USR4(("                "));
}

tBool clAudioSource::bCCACallback(tU32 u32msgcode, tU32 u32msgparam1, tU32 msgparam2)
{
  (void) u32msgcode;
  (void) u32msgparam1;
  (void) msgparam2;
   ETG_TRACE_USR4(("clAudioSource::bCCACallback: Default Implementation"));
   return TRUE;
}

/**
* /brief
* vSets member Variable to given value.
* This is used for RadioPresets or
* NaviAnnouncement Index (ACR FI, ChannelRequest Index)
*/
tVoid clAudioSource::vSetUserData(tU32 u32UserData)
{
   ETG_TRACE_USR4(("vSetUserData:  Data %d SourceClass: %d subID %d"
         , u32UserData
         , sGetId().enSourceClass
         , (tU16)sGetId().u16SubSource));

   m_u32UserData = u32UserData;
}

// ToDo: temp handling
tVoid clAudioSource::vSetCurActSink(tU16 u16SinkID)
{
   ETG_TRACE_USR4(("vSetCurActSink:  SinkID=%d"
         , u16SinkID));

   m_curActSinkID = u16SinkID;
}

tU16 clAudioSource::u16GetCurActSink()
{
   return m_curActSinkID;
}
/**
 * \brief
 * allowOnTop guide the request to the audio stack object.
 *
 * \param src
 * pointer to the audio source to be checked.
 *
 * \returns
 * allowed, disallowed or kicked
 */
//clStackRules::allow_t clAudioSource::allowOnTop(clAudioSource* src)
//{
//   if(src == NULL)
//   {
//      return clStackRules::disallowed;
//   }
//  return m_StackRule.allowOnTop(src->m_SrcId.enSourceClass);
//}

/**
 * \brief
 * pacGetState
 *
 * \returns
 * returns the name of the current state of the source
 */
const tChar *clAudioSource::pacGetState()
{
  return pacGetState(m_curActSinkID);
}
const tChar *clAudioSource::pacGetState(tU16 u16SinkID)
{
  const tChar* retVal = "not initialized";

  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      retVal = (*it).second->m_pacName;
    }
  }

  return retVal;
}

/**
 * \brief
 * u8GetId
 *
 * \returns
 * return the id of the audio source
 */
SourceID clAudioSource::sGetId()
{
   return m_SrcId;
}

am::sh_timerHandle_t clAudioSource::getTimerHandle()
{
   return m_handle;
}


/**
 * \brief
 * u8GetNameIndex guides the request to the audio stack
 *
 * \returns
 * returns the index of the audio source
 */
//tU8 clAudioSource::u8GetNameIndex()
//{
//  return m_StackRule.getSrcNameIndex();
//}

/**
 * \brief
 * getPushAction guides the request to the audio stack
 *
 * \param index
 * index of the audio source to be checked
 *
 * \returns
 * returns the push actions
 */
//clStackRules::actions_t clAudioSource::getPushAction(tU8 indx)
//{
//  return m_StackRule.getPushAction(indx);
//}

/**
 * \brief
 * getPopAction guides the request to the audio stack
 *
 * \param index
 * index of the audio source to be checked
 *
 * \returns
 * returns the pop actions
 */
//clStackRules::actions_t clAudioSource::getPopAction(tU8 indx)
//{
//  return m_StackRule.getPopAction(indx);
//}

/**
 * \brief
 * Write brief comment for getType here.
 *
 * \returns
 * returns the type of this audio source (foreground, background, mix, etc.)
 */
//clStackRules::type_t clAudioSource::getType()
//{
//  return m_StackRule.getType();
//}

/**
 * \brief
 * Write brief comment for getGroup here.
 *
 * \returns
 * returns the group of this audio source (radio, media, virtual, none)
 */
//clStackRules::group_t clAudioSource::getGroup()
//{
//  return m_StackRule.getGroup();
//}
//
//tS32 clAudioSource::getExternalId()
//{
//   return m_StackRule.getExternalID();
//}


tVoid clAudioSource::vWaitFor()
{
  vWaitFor(m_curActSinkID);
}



tVoid clAudioSource::vWaitFor(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      (*it).second->vMsg_WaitFor(this, (tU32)0);
    }
  }
}

/*!************************************************************************
 * METHOD:        vOn
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vOn()
{
  // forward with m_curActSinkID. To be removed later
  vOn(m_curActSinkID);
}

tVoid clAudioSource::vOn(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    (*it).second->vMsg_On(this, (tU32)0);
  }
}

/*!************************************************************************
 * METHOD:        vOff
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vOff(SourceID possibleNextSource)
{
  vOff(possibleNextSource, m_curActSinkID);
}

tVoid clAudioSource::vOff(SourceID possibleNextSource, tU16 u16SinkID)
{
  // ToDo need this as array for RSE
  m_possibleNextSource = possibleNextSource;

  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second != NULL && bPreCheckStateChange2Off())
    {
      (*it).second->vMsg_Off(this, (tU32)0);
    }
  }
}

tVoid clAudioSource::vOff()
{
  vOff(m_curActSinkID);
}
tVoid clAudioSource::vOff(tU16 u16SinkID)
{

  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second != NULL && bPreCheckStateChange2Off())
    {
      (*it).second->vMsg_Off(this, (tU32)0);
    }
  }
}

/*!************************************************************************
 * METHOD:        vPause
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *                Delegate to StateMachine
 *
 * PARAMETER:     callback    reference to AudioSourceController for callback
 *                u32UserData reserved
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vPause(SourceID u8PossibleNextSource)
{
  vPause(u8PossibleNextSource,m_curActSinkID);
}
tVoid clAudioSource::vPause(SourceID u8PossibleNextSource, tU16 u16SinkID)
{
  m_possibleNextSource = u8PossibleNextSource;

  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      (*it).second->vMsg_Pause(this, (tU32)0);
    }
  }
}

/*!************************************************************************
 * METHOD:        bIsActive
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE, if AudioSource is in state: rampUp, On or rampDown
 *                FALSE, if AudioSource is in state: pause or off
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSource::bIsActive()
{
  return bIsActive(m_curActSinkID);
}

tBool clAudioSource::bIsActive(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if ((*it).second  == clSrcStateFactory::pclCreateOff()
    || (*it).second   == clSrcStateFactory::pclCreatePause()
    || (*it).second   == clSrcStateFactory::pclCreateWaitFor())
    {
      return FALSE;
    }
  }
  return TRUE;
}

tBool clAudioSource::bIsPause()
{
  return bIsPause(m_curActSinkID);
}

tBool clAudioSource::bIsPause(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second  == clSrcStateFactory::pclCreatePause())
    {
      return TRUE;
    }
  }
  return FALSE;
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyOnDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vOn_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyOffDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vOff_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyOnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyPauseDone()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vPause_done(this);
   }
}

/*!************************************************************************
 * METHOD:        vNotifyInitStarted
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                Notify AudioSourceController
 *                Event is given, when Audiosource enters the Init-State
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vNotifyInitStarted()
{
   if (m_pclAudioSourceObserver)
   {
      m_pclAudioSourceObserver->vInit_started(this);
   }
}

/*!************************************************************************
 * METHOD:        vMW_On
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_On()
{
   vMW_RequestConfirmed();
   vMW_OnDone();
}

/*!************************************************************************
 * METHOD:        vMW_Init
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Init()
{
   vMW_InitDone();
}

/*!************************************************************************
 * METHOD:        bIsSourceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Checks if source is ready to get activated,
 *                e.g. CD is inserted, IPod is authenticated...
 *
 * PARAMETER:
 * RETURNVALUE:   TRUE if Player is ready
 *                FALSE if Player is not ready yet
 *
 * AUTHOR:        R.Volkmer
 *************************************************************************/
tVoid clAudioSource::vSourceAvailablilityChange(enSourceAvailability availability,
      enSourceAvailabilityReason availabilityReason)
{
   ETG_TRACE_USR2(("vSourceAvailablilityChange: SourceClass: %d subID %#x(%d) Available: %d Reason: %d"
            , m_SrcId.enSourceClass
            , (tU16)m_SrcId.u16SubSource
            , (tU16)m_SrcId.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, availabilityReason)));

   if((availability == clAudioSource::not_available)
      && (m_availability != clAudioSource::not_available))
   {
      m_availability = availability;
      m_availabilityReason = availabilityReason;
    //CMG3G-6592 source removal performed based on availability reason
      /*clAudioSMEngine::Source_Removed(m_SrcId);
      return;*/
    vSrcChgUnAvailabilityReason();
    return;
   }
   if((m_availability != clAudioSource::available)
      && (availability == clAudioSource::available)
      && (availabilityReason == clAudioSource::newmedia))
   {
      ETG_TRACE_USR3(("vSourceAvailablilityChange: Trigger AutoPlay SourceClass: %d subID %#x(%d) Available: %d Reason: %d"
            , m_SrcId.enSourceClass
            , (tU16)m_SrcId.u16SubSource
            , (tU16)m_SrcId.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, availability)
            , ETG_CENUM(clAudioSource::enSourceAvailabilityReason, availabilityReason)));
      m_availability = availability;
      m_availabilityReason = availabilityReason;
      if((clFactory_AudioSourceClass::GetAutoPlayMode(sGetId()) == clSourceClass::autoplay_avail)
      && (clGeniviAudioCtrlAdapter::bIsAutoplayEnabled()))
      {
         clAudioSMEngine::Source_Insert(m_SrcId);
      }else{
         ETG_TRACE_USR3(("vSourceAvailablilityChange: Autoplay is not configured for this SourceClass"));
         //ToDo use the map
         std::map<tU16,SourceStateMachine::clSrcState*>::iterator it;
         it = sinkStateMap.find(m_curActSinkID);
         if(it!=sinkStateMap.end())
         {
            // found it
            if((*it).second != NULL)
            {
               ETG_TRACE_USR3(("call vMsg_SrcAvailabilityChg for sinkID %d",m_curActSinkID));
               (*it).second->vMsg_SrcAvailabilityChg(this, m_u32UserData);
            }
            else
            {
               ETG_TRACE_ERR(("CMD_AUDIO, Could not find sinkID %d",m_curActSinkID));
            }
         }

      }
      return;
   }
  // if(((availability == clAudioSource::unkonwn) && (m_availability != clAudioSource::unkonwn))||(m_availabilityReason != availabilityReason))
   if(((availability == clAudioSource::unkonwn) && (m_availability != clAudioSource::unkonwn))
       || (m_availabilityReason != availabilityReason)
       || ((availability == clAudioSource::unkonwn)))
   {
       m_availability = availability;
       m_availabilityReason = availabilityReason;
      //TODO
      //ONLY start timer on startup and LSM restore,
      //because parsing may take loong time.
      //When finding some playable file while parsing the availability
      //changes from unknown to playable. But this may be even after minutes...
       ETG_TRACE_USR4(("AudioSource in state (%d) %s",
       this->pclGetState()->m_enStateID, this->pclGetState()->m_pacName));
       vWatchdogStart_Availability();

   }
   m_availability = availability;
   m_availabilityReason = availabilityReason;

   std::map<tU16,SourceStateMachine::clSrcState*>::iterator it;
   it = sinkStateMap.find(m_curActSinkID);
   if(it!=sinkStateMap.end())
   {
      // found it
      if((*it).second != NULL)
      {
         ETG_TRACE_USR3(("call vMsg_SrcAvailabilityChg for sinkID %d",m_curActSinkID));
         (*it).second->vMsg_SrcAvailabilityChg(this, m_u32UserData);
      }
      else
      {
         ETG_TRACE_ERR(("CMD_AUDIO, Could not find sinkID %d",m_curActSinkID));
      }
   }
}

tVoid clAudioSource::vSourceAvailablilityChange()
{
  vSourceAvailablilityChange(m_curActSinkID);
}
tVoid clAudioSource::vSourceAvailablilityChange(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second != NULL)
    {
      (*it).second->vMsg_SrcAvailabilityChg(this, m_u32UserData);
    }
  }
}
/*!************************************************************************
 * METHOD:        vSrcChgUnAvailabilityReason
 * CLASS:         clAudioSource
 * DESCRIPTION:   Ckecks availability reason of requested source and
 *          perform action accordingly.
 *
 * PARAMETER:
 * RETURNVALUE:   tVoid
 *
 * AUTHOR:        Parusharam J
 *************************************************************************/
tVoid clAudioSource::vSrcChgUnAvailabilityReason()
{
   ETG_TRACE_USR3(("vSourceAvailablilityChange: SourceClass: %d subID %#x(%d)"
        , m_SrcId.enSourceClass
        , (tU16)m_SrcId.u16SubSource
        , (tU16)m_SrcId.u16SubSource));
  switch(m_availabilityReason)
  {
  case clAudioSource::temperature:
  case clAudioSource::overcurrent:
  case clAudioSource::error:
    {
      clAudioSMEngine::Source_Removed(m_SrcId);
    }
    break;
  case clAudioSource::voltage:
  case clAudioSource::internaldisconnect:
    {
      ETG_TRACE_USR2(("vSourceAvailablilityChange: Keep source on stack, don't remove it"));
    }
    break;
  default:
    {
      clAudioSMEngine::Source_Removed(m_SrcId);
      ETG_TRACE_USR2(("vSourceAvailablilityChange: Wrong availability reason for non-available source"));
    }
  }
}
/*!************************************************************************
 * METHOD:        enIsSourceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Checks if source is ready to get activated,
 *                e.g. CD is inserted, IPod is authenticated...
 *
 * PARAMETER:
 * RETURNVALUE:   TRUE if Player is ready
 *                FALSE if Player is not ready yet
 *
 * AUTHOR:        R.Volkmer
 *************************************************************************/
clAudioSource::enSourceAvailability clAudioSource::enIsSourceAvailable()
{
   ETG_TRACE_USR4(("enIsSourceAvailable: SourceClass: %d subID %d Available: %d"
            , m_SrcId.enSourceClass
            , (tU16)m_SrcId.u16SubSource
            , ETG_CENUM(clAudioSource::enSourceAvailability, m_availability)));
   return m_availability;
}

/*!************************************************************************
 * METHOD:        vMW_Off
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Off(SourceID )
{
   vMW_OffDone();
}

/*!************************************************************************
 * METHOD:        vMW_Pause
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from internal StateMachine
 *                This is the default implementation
 *                The default impl. just confirms the operation.
 *                Overwrite this method to establish MW communication
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_Pause(SourceID )
{
   vMW_PauseDone();
}

/*!************************************************************************
 * METHOD:        vMW_OnDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_OnDone()
{
  vMW_OnDone(m_curActSinkID);
}

tVoid clAudioSource::vMW_OnDone(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second != NULL)
    {
      ETG_TRACE_USR2(("vMW_OnDone Source: %d, SubSource %d, State: %d, Sink: %d"
          ,m_SrcId.enSourceClass
          ,m_SrcId.u16SubSource
          ,ETG_CENUM(AudioStates::enAudioStates, (*it).second->enGetStateID())
          ,m_curActSinkID
      ));
      (*it).second->vMsg_OnDone(this, 0);
    }
  }
}

/*!************************************************************************
 * METHOD:        vMW_OffDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_OffDone()
{
  vMW_OffDone(m_curActSinkID);
}

tVoid clAudioSource::vMW_OffDone(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      ETG_TRACE_USR2(("vMW_OffDone Source: %d, SubSource %d, State: %d, Sink: %d"
          ,m_SrcId.enSourceClass
          ,m_SrcId.u16SubSource
          ,ETG_CENUM(AudioStates::enAudioStates, (*it).second->enGetStateID())
          ,m_curActSinkID
      ));

      (*it).second->vMsg_OffDone(this, 0);
    }


    if(  (clFactory_AudioSourceClass::GetRegistrationMode(sGetId()) ==clSourceClass::registerDynamic)
        &&(m_availability == clAudioSource::not_available))
    {
      ETG_TRACE_USR2(("Remove Source: Source: %d, SubSource %d, State: %d"
          ,m_SrcId.enSourceClass
          ,m_SrcId.u16SubSource
          ,ETG_CENUM(AudioStates::enAudioStates, (*it).second->enGetStateID())
      ));

         //daw2hi: 21.07.2016 only if source is off on all stacks
         tBool bAllOff = true;
         std::map<tU16,SourceStateMachine::clSrcState*>::iterator itSrcState;
         for(itSrcState=sinkStateMap.begin();itSrcState!=sinkStateMap.end();++itSrcState)
         {
            //if( (itSrcState->second != NULL) && (*(itSrcState->second) !=AudioSource::AudioStates::OFF))
            if((itSrcState->first != 0) && (itSrcState->second != NULL) && ((itSrcState->second)->m_enStateID != AudioSource::AudioStates::OFF))
            {
               bAllOff=false;
               break;
            }
         }
         if(bAllOff)
         {
            clAudioSourceFactory::removeAudioSources(m_SrcId);
            ETG_TRACE_USR2(("Remove Source: done"));
         }
         else
         {
            ETG_TRACE_USR2(("Remove Source: not possible, not OFF on all stacks"));
         }
       //end
       //clAudioSourceFactory::removeAudioSources(m_SrcId);
    }
  }
}

/*!************************************************************************
 * METHOD:        vMW_InitDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_InitDone()
{
  vMW_InitDone(m_curActSinkID);
}

tVoid clAudioSource::vMW_InitDone(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      ETG_TRACE_USR2(("vMW_InitDone Source: %d, SubSource %d, State: %d, Sink: %d"
          ,m_SrcId.enSourceClass
          ,m_SrcId.u16SubSource
          ,ETG_CENUM(AudioStates::enAudioStates, (*it).second->enGetStateID())
          ,m_curActSinkID
      ));
      (*it).second->vMsg_InitDone(this, 0);
    }
  }
}
#if 0
tVoid clAudioSource::vMW_InitDone()
{
  SourceStateMachine::clSrcState* clActSrcState;
  if(m_curActSinkID<=1) clActSrcState = m_pclState;
  else clActSrcState = m_pclStateRSE;

  if(NULL != clActSrcState)
  {
    ETG_TRACE_USR2(("vMW_InitDone Source: %d, SubSource %d, State: %d, Sink: %d"
        ,m_SrcId.enSourceClass
        ,m_SrcId.u16SubSource
        ,ETG_CENUM(AudioStates::enAudioStates, clActSrcState->enGetStateID())
        ,m_curActSinkID
    ));
    clActSrcState->vMsg_InitDone(this, 0);
  }
}
#endif
/*!************************************************************************
 * METHOD:        vMW_RequestConfirmed
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_RequestConfirmed()
{
  vMW_RequestConfirmed(m_curActSinkID);
}

tVoid clAudioSource::vMW_RequestConfirmed(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      (*it).second->vMsg_RequestConfirmed(this, 0);
    }
  }
}

/*!************************************************************************
 * METHOD:        vMW_PauseDone
 * CLASS:         clAudioSource
 * DESCRIPTION:   Confirmation from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_PauseDone()
{
  return vMW_PauseDone(m_curActSinkID);
}
tVoid clAudioSource::vMW_PauseDone(tU16 u16SinkID)
{

  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
         ETG_TRACE_USR2(("vMW_PauseDone Source: %d, SubSource %d, State: %d, Sink: %d"
             ,m_SrcId.enSourceClass
             ,m_SrcId.u16SubSource
             ,ETG_CENUM(AudioStates::enAudioStates, (*it).second->enGetStateID())
             ,m_curActSinkID
             ));
         (*it).second->vMsg_PauseDone(this, 0);
    }
  }
}

/*!************************************************************************
 * METHOD:        bIsOff
 * CLASS:         clAudioSource
 * DESCRIPTION:   Request from AudioSourceController
 *
 *
 * PARAMETER:
 *
 * RETURNVALUE:   TRUE, if AudioSource is in state: Off
 *                FALSE, otherwise
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tBool clAudioSource::bIsOff()
{
  return bIsOff(m_curActSinkID);
}
tBool clAudioSource::bIsOff(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if((*it).second == clSrcStateFactory::pclCreateOff())
    {
      return TRUE;
    }
    return FALSE;
  }
  return FALSE;
}

/*!************************************************************************
 * METHOD:        pclGetState
 * CLASS:         clAudioSource
 * DESCRIPTION:   simple getter for prop
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
clSrcState* clAudioSource::pclGetState()
{
   ETG_TRACE_USR4(("Warning: clAudioSource::pclGetState() called without sinkID, using Sink 1"));
   return pclGetState(m_curActSinkID);
}

clSrcState* clAudioSource::pclGetState(tU16 u16SinkID)
{
//  ETG_TRACE_USR2(("AudioSrc: %s, pclGetState ",this->pacGetName()));
//  ETG_TRACE_USR2(("for SinkID %d ",u16SinkID));
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    //ETG_TRACE_USR2(("returns SrcState %s",((*it).second)->m_pacName));
    return (*it).second;
  }
  //daw2hi ToDo: check if this causes additional Nullpointer checks
  return NULL;
}

void clAudioSource::pclSetState(clSrcState* pclState)
{
  return pclSetState(pclState,m_curActSinkID);
}

void clAudioSource::pclSetState(clSrcState* pclState, tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);
   ETG_TRACE_USR4(("AudioSrc %s",this->m_pacName));
  if(it!=sinkStateMap.end())
  {
      ETG_TRACE_USR4(("pclSetState Sink %d state %s",u16SinkID,m_pclState->m_pacName));
    (*it).second = pclState;
  }
   else
   {
      ETG_TRACE_ERR(("CMD_AUDIO, pclSetState Sink %d  not found !!! (state %s)",u16SinkID,m_pclState->m_pacName));
   }
}

/*!************************************************************************
 * METHOD:        vMW_ServiceUnavailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Notification from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_ServiceUnavailable()
{
  vMW_ServiceUnavailable(m_curActSinkID);
}


tVoid clAudioSource::vMW_ServiceUnavailable(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      (*it).second->vMsg_ServiceUnavailable(this, 0);
    }
  }
}

/*!************************************************************************
 * METHOD:        vMW_ServiceAvailable
 * CLASS:         clAudioSource
 * DESCRIPTION:   Notification from derived subclasses
 *                Delegate this confirmation to the internal StateMachine
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW34)
 *************************************************************************/
tVoid clAudioSource::vMW_ServiceAvailable()
{
  vMW_ServiceAvailable(m_curActSinkID);
}

tVoid clAudioSource::vMW_ServiceAvailable(tU16 u16SinkID)
{
  std::map<tU16,SourceStateMachine::clSrcState*>::iterator it = sinkStateMap.find(u16SinkID);

  if(it!=sinkStateMap.end())
  {
    if(NULL != (*it).second)
    {
      (*it).second->vMsg_ServiceAvailable(this, 0);
    }
  }
}

/*!************************************************************************
 * METHOD:        bPreCheckStateChange2Off
 * CLASS:         clAudioSource
 * DESCRIPTION:   check the cmd counter for play commands. this has to be 0
 *                after decrease, what means, that there is no channel request
 *                open any longer, so the channel can be closed now. Increasing
 *                the counter has to be done by each source itself, which needs
 *                this counter (currently only navi is doing so)
 *
 * PARAMETER:
 * RETURNVALUE:
 *
 * AUTHOR:        (CM-DI/PJ-VW37)
 *************************************************************************/
tBool clAudioSource::bPreCheckStateChange2Off()
{
   if (m_u32PlayCmdCounter != 0)
   {
      m_u32PlayCmdCounter--;

      if(m_u32PlayCmdCounter == 0)
         return TRUE;
      else
      {
         ETG_TRACE_USR4(("bPreCheckStateChange2Off(): m_u32PlayCmdCounter: %d", m_u32PlayCmdCounter));
         return FALSE;
      }
   }
   else
      return TRUE; // default when source does not use the cmd counter
}

tVoid clAudioSource::vWelcomeSoundTimerStart()
{
  if(m_handle==0)
  {
    if(bStartTimer(WELCOME_SOUND_TIMEOUT))
    {
    m_bWelcomeSound = true;
      ETG_TRACE_USR4(("vWatchdogStart_Availability, Timer started"));
    }
  }
}
tVoid clAudioSource::vWelcomeSoundTimerStop()
{
  if(m_pSocketHandler != NULL && m_handle && m_bWelcomeSound == true)
    {
        am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
    m_bWelcomeSound = false ;
        ETG_TRACE_USR4(("vWatchdogStart_Availability Timer removed,m_handle=%d,error=%d",m_handle,(tU16)ret));
    }
}

tVoid clAudioSource::vWatchdogStart_Availability()
{
  ETG_TRACE_USR2(("vStartWatchdog_Availability(), starting timer for Src=%d, SubId=%d, p=%x",
      (tU16)m_SrcId.enSourceClass, m_SrcId.u16SubSource, (void*)this));

  if(m_bSourceIsLSM == false && m_bSourceIsProfChg == false)
  {
    ETG_TRACE_USR4(("vStartWatchdog_Availability() Src is not LSM return"));
    return;
  }

  if(getSourceClass().getAvailabilityTimeout() == 0)
  {
    ETG_TRACE_USR4(("vStartWatchdog_Availability(), timeout not specified for this source, no need to start timer!"));
    m_bSourceIsLSM = false;
    m_bSourceIsProfChg = false;
    return;
  }

  //Start timer here
  if(m_handle==0)
  {
    if(bStartTimer(getSourceClass().getAvailabilityTimeout()))
    {
      ETG_TRACE_USR4(("vWatchdogStart_Availability, Timer started"));
    }
  }
  else
  {
    ETG_TRACE_USR4(("vWatchdogStart_Availability,Timer already running, stopping timer, handle=%d",m_handle));
    if(m_pSocketHandler != NULL)
    {
        //Stop and remove the timer from mainloop
        //VVD lint fix
    am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
        ETG_TRACE_USR4(("vWatchdogStart_Availability Timer removed,m_handle=%d,error=%d",m_handle,(tU16)ret));
        m_pSocketHandler = NULL;
        m_handle = 0;
        //m_bSourceIsLSM = false;

        if(bStartTimer(getSourceClass().getAvailabilityTimeout()))
        {
           ETG_TRACE_USR4(("vWatchdogStart_Availability, Restarting the timer with handle(%d) because user initiated the profile change for source", m_handle));
           if(true == bIsSourceLSM())
              vSetSourceLSM(false);
        }
    }
    else
    {
      ETG_TRACE_ERR(("CMD_AUDIO, vWatchdogStart_Availability, could not stop the timer m_pSocketHandler is NULL"));
    }
  }
}

//This function is going to be our timer callback
void clAudioSource::Call(const am::sh_timerHandle_t handle, void* /*userData*/)
{
  ETG_TRACE_USR2(("clAudioSource:Call() timer expired,p=%x, handle=%d",(void*)this,handle));
  if(handle == m_handle)//This check is not needed in reality
  {
       if(m_pSocketHandler)
       {
            am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
            ETG_TRACE_USR3(("clAudioSource::Call timer expired after removeTimer,m_handle=%d,error=%d",m_handle,(tU16)ret));
       }
       else
       {
            ETG_TRACE_ERR(("CMD_AUDIO, m_pSocketHandler nullptr"));
       }

      m_handle = 0;

      if(m_bWelcomeSound)
      {
        m_bWelcomeSound = false ;
          const clSourceClass* pSrcClass_RemoveWelcomeMute = clFactory_AudioSourceClass::GetSourceClass("MUTE_WELCOMESOUND_STARTUP");

        //CID 4626437
        if(pSrcClass_RemoveWelcomeMute != NULL)
        {
            SourceID Src = SourceID(pSrcClass_RemoveWelcomeMute->getClassID(),0);
            clGeniviAudioCtrlAdapter::RequestSourceOff(Src);
        }

        return;
      }

    /*
     * reset flag as availability time out happened (PSARCC30-1542)
     */
    if(true == bIsSourceLSM() && clAudioSourceController::bIsSourceOnStack(this->sGetId()))
    {
      ETG_TRACE_USR3(("clAudioSource:Call() timer expired, reseting flag m_bSourceIsLSM to false"));
      vSetSourceLSM(false);
    }

    if(true == bIsSourceProfChg() && clAudioSourceController::bIsSourceOnStack(this->sGetId()))
    {
      ETG_TRACE_USR3(("clAudioSource:Call() timer expired, reseting flag m_bSourceIsProfChg to false"));
      vSetSourceProfChg(false);
      vSetSwitchtoDefaultSrcOnProfChg(true);
    }

    vSourceAvailablilityChange();//Trigger source availability change
  }
  else
  {
    ETG_TRACE_ERR(("CMD_AUDIO, Trigger received based on invalid handle... ignoring "));
  }
}
/**
* Helper function to start timer
*/
bool clAudioSource::bStartTimer(uint32_t sec)
{
  timespec start;
  start.tv_sec = sec;
  start.tv_nsec = 0;

  IAmControlReceiverShadow* pReceiveShadow = IAmControlReceiverShadow::getInstance();
  if(pReceiveShadow != NULL)
  {
    m_pSocketHandler = pReceiveShadow->mpCAmSocketHandler;
  }

  if(m_pSocketHandler != NULL)
  {
    if(E_OK == m_pSocketHandler->addTimer(start, this, m_handle,NULL))
    {
      ETG_TRACE_USR4(("Src=%d, SubId=%d, Timer (%ds) started, p=%x, timer handle=%d",
          m_SrcId.enSourceClass, m_SrcId.u16SubSource, sec, (void*)this, m_handle));
      return true;
    }
    else
    {
      ETG_TRACE_ERR(("CMD_AUDIO, Timer could not be added for p=%x",(void*)this));
    }
  }
  else
  {
    ETG_TRACE_ERR(("CMD_AUDIO, Pointer to socket handler is NULL"));
  }
  return false;
}

tVoid clAudioSource::vRemoveTimer()
{
  ETG_TRACE_USR2(("clAudioSource::vRemoveTimer, p=%x, handle=%d",(void*)this, m_handle));
  if(m_handle != 0)//This check is not needed in reality
  {
       if(m_pSocketHandler)
       {
            am_Error_e ret = m_pSocketHandler->removeTimer(m_handle);
            ETG_TRACE_USR3(("clAudioSource::vRemoveTimer, removing source since state of src is ON,m_handle=%d", m_handle));
            (void)ret;
       }
       else
       {
            ETG_TRACE_ERR(("CMD_AUDIO, m_pSocketHandler nullptr"));
       }

      m_handle = 0;
  }
  else
  {
    ETG_TRACE_ERR(("CMD_AUDIO, Trigger received based on invalid handle... ignoring "));
  }
}

}} //namespace


/* EOF */
