/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */

#ifndef CONTROLSENDER_H_
#define CONTROLSENDER_H_

#include <vector>
#include <list>
#include <map>
#include <algorithm>

class amcontrol_clienthandler_fc_audiomanager;
class amcontrol_clienthandler_LCM;
class amcontrol_clienthandler_fc_audioroutemanager;
class CAmControlCvm;
class aud_dbus_lcm_handler;

#include "IAmControl.h"
#include "IAmControlReceiverShadow.h"
#include "aud_diaglib_dbus.h"
#include "aud_dbus_connector.h"
#include "CAmCommandLineSingleton.h"
#include "include/audiomanagertypes.h"
#include "LcmDBusProxy/aud_dbus_lcm_handler.h"
#include "common.h"

#define SINK_ID_ADR3 1



typedef std::map<am_CustomMainSoundPropertyType_t, int16_t> MapMainSoundProperty_t;
typedef std::map<am_CustomMainSoundPropertyType_t, int16_t>::iterator MapSoundPropertyIterator_t;

typedef std::map<am_CustomSystemPropertyType_t, int16_t> MapSystemProperty_t;
typedef std::map<am_CustomSystemPropertyType_t, int16_t>::iterator MapSystemPropertyIterator_t;

typedef std::map<am_sinkID_t, MapMainSoundProperty_t> MapMainSinkSoundProperty_t;
typedef std::map<am_sinkID_t, MapMainSoundProperty_t>::iterator  MapMainSinkSoundPropertyIterator_t;

typedef std::map<am_sourceID_t, MapMainSoundProperty_t> MapMainSourceSoundProperty_t;
typedef std::map<am_sourceID_t, MapMainSoundProperty_t>::iterator  MapMainSourceSoundPropertyIterator_t;

class CAmControlSenderBase: public IAmControlSend
{
public:
  CAmControlSenderBase();
  virtual ~CAmControlSenderBase();
  am_Error_e startupController(IAmControlReceive* controlreceiveinterface);
  void setControllerReady();
  void setControllerRundown(const int16_t signal);
  am_Error_e hookUserConnectionRequest(const am_sourceID_t sourceID, const am_sinkID_t sinkID, am_mainConnectionID_t& mainConnectionID);
  am_Error_e hookUserDisconnectionRequest(const am_mainConnectionID_t connectionID);
  am_Error_e hookUserSetMainSinkSoundProperty(const am_sinkID_t sinkID, const am_MainSoundProperty_s& soundProperty);
  am_Error_e hookUserSetMainSourceSoundProperty(const am_sourceID_t sourceID, const am_MainSoundProperty_s& soundProperty);
  am_Error_e hookUserSetSystemProperty(const am_SystemProperty_s& property);
  am_Error_e hookUserVolumeChange(const am_sinkID_t SinkID, const am_mainVolume_t newVolume);
  am_Error_e hookUserVolumeStep(const am_sinkID_t SinkID, const int16_t increment);
  am_Error_e hookUserSetSinkMuteState(const am_sinkID_t sinkID, const am_MuteState_e muteState);
  am_Error_e hookSystemRegisterDomain(const am_Domain_s& domainData, am_domainID_t& domainID);
  am_Error_e hookSystemDeregisterDomain(const am_domainID_t domainID);
  void hookSystemDomainRegistrationComplete(const am_domainID_t domainID);
  am_Error_e hookSystemRegisterSink(const am_Sink_s& sinkData, am_sinkID_t& sinkID);
  am_Error_e hookSystemDeregisterSink(const am_sinkID_t sinkID);
  am_Error_e hookSystemRegisterSource(const am_Source_s& sourceData, am_sourceID_t& sourceID);
  am_Error_e hookSystemDeregisterSource(const am_sourceID_t sourceID);
  am_Error_e hookSystemRegisterGateway(const am_Gateway_s& gatewayData, am_gatewayID_t& gatewayID);
  am_Error_e hookSystemRegisterConverter(const am_Converter_s& converterData, am_converterID_t& converterID);
  am_Error_e hookSystemDeregisterConverter(const am_converterID_t converterID);
  am_Error_e hookSystemDeregisterGateway(const am_gatewayID_t gatewayID);
  am_Error_e hookSystemRegisterCrossfader(const am_Crossfader_s& crossfaderData, am_crossfaderID_t& crossfaderID);
  am_Error_e hookSystemDeregisterCrossfader(const am_crossfaderID_t crossfaderID);
  void hookSystemSinkVolumeTick(const am_Handle_s handle, const am_sinkID_t sinkID, const am_volume_t volume);
  void hookSystemSourceVolumeTick(const am_Handle_s handle, const am_sourceID_t sourceID, const am_volume_t volume);
  void hookSystemInterruptStateChange(const am_sourceID_t sourceID, const am_InterruptState_e interruptState);
  void hookSystemSinkAvailablityStateChange(const am_sinkID_t sinkID, const am_Availability_s& availability);
  void hookSystemSourceAvailablityStateChange(const am_sourceID_t sourceID, const am_Availability_s& availability);
  void hookSystemDomainStateChange(const am_domainID_t domainID, const am_DomainState_e state);
  void hookSystemReceiveEarlyData(const std::vector<am_EarlyData_s>& data);
  void hookSystemSpeedChange(const am_speed_t speed);
  void hookSystemTimingInformationChanged(const am_mainConnectionID_t mainConnectionID, const am_timeSync_t time);
  void cbAckConnect(const am_Handle_s handle, const am_Error_e errorID);
  void cbAckDisconnect(const am_Handle_s handle, const am_Error_e errorID);
  void cbAckCrossFade(const am_Handle_s handle, const am_HotSink_e hostsink, const am_Error_e error);
  void cbAckSetSinkVolumeChange(const am_Handle_s handle, const am_volume_t volume, const am_Error_e error);
  void cbAckSetSourceVolumeChange(const am_Handle_s handle, const am_volume_t voulme, const am_Error_e error);
  void cbAckSetSourceState(const am_Handle_s handle, const am_Error_e error);
  void cbAckSetSourceSoundProperties(const am_Handle_s handle, const am_Error_e error);
  void cbAckSetSourceSoundProperty(const am_Handle_s handle, const am_Error_e error);
  void cbAckSetSinkSoundProperties(const am_Handle_s handle, const am_Error_e error);
  void cbAckSetSinkSoundProperty(const am_Handle_s handle, const am_Error_e error);
  am_Error_e getConnectionFormatChoice(const am_sourceID_t sourceID, const am_sinkID_t sinkID, const am_Route_s listRoute, const std::vector<am_CustomConnectionFormat_t> listPossibleConnectionFormats, std::vector<am_CustomConnectionFormat_t>& listPrioConnectionFormats);
  void getInterfaceVersion(std::string& version) const;
  void confirmCommandReady(const am_Error_e error);
  void confirmRoutingReady(const am_Error_e error);
  void confirmCommandRundown(const am_Error_e error);
  void confirmRoutingRundown(const am_Error_e error);

    am_Error_e hookSystemUpdateSink(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties) ;
    am_Error_e hookSystemUpdateSource(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID, const std::vector<am_SoundProperty_s>& listSoundProperties, const std::vector<am_CustomConnectionFormat_t>& listConnectionFormats, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties) ;
    am_Error_e hookSystemUpdateGateway(const am_gatewayID_t gatewayID, const std::vector<am_CustomConnectionFormat_t>& listSourceConnectionFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkConnectionFormats, const std::vector<bool>& convertionMatrix) ;
  am_Error_e hookSystemUpdateConverter(const am_converterID_t converterID, const std::vector<am_CustomConnectionFormat_t>& listSourceConnectionFormats, const std::vector<am_CustomConnectionFormat_t>& listSinkConnectionFormats, const std::vector<bool>& convertionMatrix);
  void cbAckSetVolumes(const am_Handle_s handle, const std::vector<am_Volumes_s>& listVolumes, const am_Error_e error) ;
  void cbAckSetSinkNotificationConfiguration(const am_Handle_s handle, const am_Error_e error) ;
  void cbAckSetSourceNotificationConfiguration(const am_Handle_s handle, const am_Error_e error) ;
  void hookSinkNotificationDataChanged(const am_sinkID_t sinkID, const am_NotificationPayload_s& payload) ;
  void hookSourceNotificationDataChanged(const am_sourceID_t sourceID, const am_NotificationPayload_s& payload) ;
  am_Error_e hookUserSetMainSinkNotificationConfiguration(const am_sinkID_t sinkID, const am_NotificationConfiguration_s& notificationConfiguration) ;
  am_Error_e hookUserSetMainSourceNotificationConfiguration(const am_sourceID_t sourceID, const am_NotificationConfiguration_s& notificationConfiguration) ;

    void hookSystemSingleTimingInformationChanged(const am_connectionID_t connectionID, const am_timeSync_t time);

  am_ServiceAvailability_e m_eGAM_ServiceAvailability;
  void InitStackManager();

  void response_startup();
  void response_shutdown();

  /* Map to Maintain main sound properties to avoid sending updates if values are unchanged */
  MapMainSinkSoundProperty_t m_MapMainSinkSoundProperty;
  MapMainSourceSoundProperty_t m_MapMainSourceSoundProperty;
  MapSystemProperty_t m_MapSystemProperty;

  void vLoadVolumeContext();


private:
  /**********************************Data Members*****************************************************/

  am_domainID_t g_domainId;
  std::map<uint16_t, am::am_Source_s> m_aSourceMap;
  std::map<uint16_t, am::am_Sink_s> m_aSinkMap;

  CAmSocketHandler* m_pSocketHandler;
  IAmControlReceive * mControlReceiveInterface;
  TCLAP::ValueArg<std::string> mCommandLineArg;
  IAmControlReceiverShadow * mControlReceiveShadowInterface;

  amcontrol_clienthandler_fc_audioroutemanager* mClientHandlerAudioRouteMgr;
  amcontrol_clienthandler_fc_audiomanager* mClientHandlerFCAudioMgr;
  amcontrol_clienthandler_LCM *mClientHandlerLCM;
  aud_dbus_lcm_handler *m_pDbusLcmHandler;
  CAmControlCvm* mClientControlCvm;



    struct rHandleStatus
  {
    bool status;
    am_Handle_s handle;
  };

    struct rMainConnectionSet
  {

  am_sourceID_t sourceID;
    am_sinkID_t sinkID;
    am_mainConnectionID_t mainConnectionID;
  };
    struct rMainVolumeSet
  {
    am_sinkID_t sinkID;
    am_Handle_s handle;
    am_mainVolume_t mainVolume;
    am_MuteState_e muteState;
  };

  struct rMainSourceSoundPropertySet
  {
    am_sourceID_t sourceID;
    am_Handle_s handle;
    am_MainSoundProperty_s mainSoundProperty;
  };

  struct rMainSinkSoundPropertySet
  {
    am_sinkID_t sinkID;
    am_Handle_s handle;
    am_MainSoundProperty_s mainSoundProperty;
  };

  class findHandle
  {
        rHandleStatus mHandle;
  public:
        explicit findHandle(rHandleStatus handle) :
    mHandle(handle)
    {
    }
        bool operator()(const rHandleStatus& handle) const
    {
      return (handle.handle.handle == mHandle.handle.handle);
    }
  };

    struct rCheckHandle
  {

        rHandleStatus mHandleStatus;
        explicit rCheckHandle(const rHandleStatus& value) :
                mHandleStatus(value)
    {
    }

        bool operator()(const rHandleStatus &value)
    {
      return !value.status;
    }
  };

    struct rCheckMainConnectionID
  {
    am_MainConnection_s mMainConnection;
        explicit rCheckMainConnectionID(const am_MainConnection_s& mainConnection) :
                                                                    mMainConnection(mainConnection)
    {
    }
    bool operator()(const am_MainConnection_s& mainConnection)
    {
      if (mMainConnection.mainConnectionID == mainConnection.mainConnectionID)
        return true;
      return false;
    }
  };
  enum cs_stateflow_e
  {
    SF_NONE,
    SF_CONNECT,
    SF_NAVI,
    SF_TA
  };

  enum cs_connectSf_e
  {
    SFC_RAMP_DOWN,
    SFC_SOURCE_STATE_OFF,
    SFC_DISCONNECT,
    SFC_CONNECT,
    SFC_SOURCE_STATE_ON,
    SFC_RAMP_UP,
    SFC_FINISHED
  };

  struct cs_connectData_s
  {
    am_mainConnectionID_t currentMainConnection;
    am_mainConnectionID_t newMainConnection;
    am_sourceID_t oldSourceID;
    am_sinkID_t sinkID;
    am_sourceID_t sourceID;
  };

  cs_connectSf_e mConnectSf;
  cs_connectData_s mConnectData;
  cs_stateflow_e mStateflow;

  bool m_bIsPendingConnectRequest;
  rMainConnectionSet mPendingConnectionRequest;
  tenUserRequest m_RequestUnderProcess;
  std::vector<rMainVolumeSet> mListOpenVolumeChanges;

  std::map<am_CustomMainSoundPropertyType_t, rMainSourceSoundPropertySet > m_MainSourceSoundPropertyChangeRequest;
  std::map<am_CustomMainSoundPropertyType_t, rMainSinkSoundPropertySet > m_MainSinkSoundPropertyChangeRequest;

  //Call Back Handler Function Prototype
  typedef void (CAmControlSenderBase::*SinkSoundPropertyCallBackHandler)(rMainSinkSoundPropertySet &);
  //Map of Sound property type and Handler functions
  typedef  std::map<am_CustomMainSoundPropertyType_t, SinkSoundPropertyCallBackHandler>  m_MainSinkSoundPropertyFunctionMap_t;
  const m_MainSinkSoundPropertyFunctionMap_t m_MainSinkSoundPropertySetHandlers;

  ///This pointer will hold the object of diaglib dbus
  aud_diaglib_handler m_pdiaglibhandler;


  /*******************************Methods********************************************************/

  void callStateFlowHandler();
  void callConnectHandler();
  void callConnectPendingRequest();

  void connect(am_sourceID_t sourceID, am_sinkID_t sinkID, am_mainConnectionID_t mainConnectionID) const;
  void disconnect(am_mainConnectionID_t connectionID) const;

  am_Error_e enRegisterDomain();
  am_Error_e enRegisterSources();
  am_Error_e enRegisterSinks();
  am_Error_e DeRegistersAll();

  CAmControlSenderBase::m_MainSinkSoundPropertyFunctionMap_t InitSinkSoundPropertySetHandlers();

  void vHandleSinkSoundProperty_Bass(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_MidRange(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_Treble(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_Loudness(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_Balance(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_Fader(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_DynaCompression(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_EqualizerPreset(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SpeedDepVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SoundOptimization(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_BeepLevel(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ChildrenSleepMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_GeometryMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SurroundSetting(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeEntertainment(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumePhone(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeONVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeTASetup(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeBeep(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_PDCAttenuationLevel(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_BTAudioINLevel(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_PDCActive(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeLock(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_AuxInLevel(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SystemMute(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeNavigation(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeSDS(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeRingtone(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_VolumeSPI_INFO(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_AudioOffMute(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_AudioMode(rMainSinkSoundPropertySet &);

  void vHandleSinkSoundProperty_Center(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SubWoofer(rMainSinkSoundPropertySet &);

  void vHandleSinkSoundProperty_VolumeSource(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RoomEffect(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_FMEnhancerMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ArkamysAutoGainControl(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_AmpAvailabilityStatus(rMainSinkSoundPropertySet &);

  void vHandleSinkSoundProperty_ESESoundType(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ESEVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RSoundActive(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RSoundOnOff(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RSoundSoundType(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RSoundVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ArkamysAudioMode(rMainSinkSoundPropertySet &);

  void vHandleSinkSoundProperty_UPA_SoundType(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ParkAssistanceVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ADAS_Setting_Sound(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ADAS_UPA_Mute(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ADAS_LDW_UserVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_EntStartupVolume(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_EntMinimumStartupVolume(rMainSinkSoundPropertySet &);  //minimum volume limit
  void vHandleSinkSoundProperty_AudioPilotMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_CenterPointMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SurroundVol(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_DriverSoundMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_RearSoundMode(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ANC_ASC(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ANC_Setting(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ASC_Setting(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ESE_Setting(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_AudioPilotIntensity_Setting(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_SRSoundNotificationState(rMainSinkSoundPropertySet &);
  void vHandleSinkSoundProperty_ADAS_LDW(rMainSinkSoundPropertySet &);

  // function called for project specific behaviour
  am_Error_e hookUserVolumeStep_IVI(const int16_t increment) ;
  am_Error_e hookUserVolumeStep_Other_Projects(const int16_t increment) ;
  am_Error_e hookUserVolumeStep_G3G(const int16_t increment) ;

public :
  am_SystemAudioState_e m_eAudioState;
  //To handle Voltage specific events
  void vHandleCvmEvent(tU32 u32CvmEvent);

  //To update sink sound property into Database
  void vUpdateSinkSoundPropertyDB(am_CustomMainSoundPropertyType_t etype,int16_t ivalue);

  //To update source sound property into Database
  void vUpdateSourceSoundPropertyDB(const am_sourceID_t sourceID,am_CustomMainSoundPropertyType_t etype,int16_t ivalue);

  //To update system property into Database
  void vUpdateSystemPropertyDB(am_CustomSystemPropertyType_t etype,int16_t ivalue,bool bUpdateIfPropNotChanged=false);

  //To check if update DB is required or not
  bool bIsSinkSoundPropertyChanged(const am_sinkID_t sinkID,const am_CustomMainSoundPropertyType_t etype,const int16_t ivalue);
  bool bIsSourceSoundPropertyChanged(const am_sourceID_t sourceID,const am_CustomMainSoundPropertyType_t etype,const int16_t ivalue);
  bool bIsSystemPropertyChanged(const am_CustomSystemPropertyType_t etype,const int16_t ivalue,bool bNoUpdate=false);
  void vHandleLCMState();

};

#endif /* CONTROLSENDER_H_ */
