/*!************************************************************************
 * FILE :      clAudioSMEngine.h
 *
 * SW-COMPONENT:  Audio statemachine
 *
 * DESCRIPTION:
 *                STARTUP:
 *                StraightForward Case is:
 *                - Check in persistent DP for LSM and PSM
 *                  LSM: Last Selected Mode, i.e. the Entertainment Source that was
 *                  played before switching off the system
 *                  PSM: Previous Selected Mode, i.e. the Entertainment Source that was
 *                  played before the LSM
 *                - Check if LSM still available
 *                  If LSM available: start it
 *                  If LSM not available: Check if PSM still available
 *                     If PSM available: start it
 *                     If PSM not available: Start Radio.FM1
 *                Startup Details:
 *                - Before any communication to FC-Audio is initiated, wait for
 *                  AudioService to become available.
 *                - Before the availability of a media source can be checked, wait for
 *                  MediaService to become available and wait, until information about
 *                  removable mediasources is available. (was SD removed from slot while off?)
 *                - Prevent, that the XML framework begins to place API-Calls like:
 *                  Sound.GetCurrentAudioComponent()
 *                  AVDC.GetCurrentSource()
 *                  AVDC.Audio.GetAudioSourceState(...)
 *                  before the mentioned Services are available.
 *                - Before starting LSM, ensure, that media-removed info is evaluated on stack
 *                  Don't try to start e.g. LSM:SD and then immediately stop it, because is was removed
 *                  during off-state.
 *
 *                  DATAPOOL:
 *                  The following DP entries are relevant for Audio Source Switching
 *                  and the corresponding Screen-Changes / visual animations:
 *                  DPAUDIO__PER_ENTSRC:
 *                     Stores the current Audio Component (Radio or Media)
 *                     Used to determine/switch the menu net Radio vs. Media
 *                     Not persistent.
 *                  DPAUDIO__ENT_SRC:
 *                     Stores the current Radio-Band. (AM, FM, DAB)
 *                     Used to determine/switch the screen in Radio-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC__CURSRC:
 *                     Stores the current Media-Source (SD, CD-DA, CDC, etc)
 *                     Used to determine/switch the screen in Media-Menu
 *                     In addition this value is required to maintain the history
 *                     Persistent.
 *                  DPAVDC_PREV_AUDIO_SRC and DPAVDC_PCUR_AUDIO_SRC:
 *                     Stores the current and previous (Entertainment-)Audio-Source
 *                     Not used for Screen Changes. No GUI-Data Events.
 *                     Used for the history-stack for entertainment sources in the Audio-handler
 *                     Persistent.
 *
 * AUTHOR:      CM-DI/PJ-VW34 Steilmann
 *
 * COPYRIGHT:    (c) 2006 Blaupunkt Werke
 * HISTORY:
 *  19.09.07      CM-DI/PJ-VW34 Steilmann
 *  Initial Version
 *  23.10.07       CM-DI/PJ-VW34 Steinle
 *                 moved Creation of AudioSourceController from Contructor
 *                 to bConfigure.
 *
 *************************************************************************/

#ifndef CLAUDIOSMENGINEH
#define CLAUDIOSMENGINEH

/*!************************************************************************
 * OSAL interfaces
 !************************************************************************/
#include "InterfaceAudioStack.h"
#include "include/audiomanagertypes.h"
/*!************************************************************************
 * private interfaces
 !************************************************************************/

#include "AudioStack/AudioSources/clAudioSourceController.h"
//#include "Audio_SM/clDiagAudioSourceController.h"
#include "AudioStack/AudioSources/cliAudioSourceControllerObserver.h"
#include "AudioStack/clTogglingSequences.h"

#include <time.h>

#ifndef _DO_NOT_USE_BOSCH_OSAL_
//#define SYSTEM_S_IMPORT_INTERFACE_MAP
//#define SYSTEM_S_IMPORT_INTERFACE_ALGORITHM
//#include <stl_pif.h>
#endif

namespace AudioStack
{


using namespace AudioSource;


// using data pool



/**
 * @brief
 * This is the main public Interface for interacting with AudioStack
 */
class clAudioSMEngine : public AudioSource::clIAudioSourceControllerObserver
{

friend class clAudioSMEngine_dbg;

public:

   typedef enum{
     PersMuteDP_Reset,
     PersMuteDP_Set
   } enPersMuteDPAction;

   typedef enum{
   AUDIOSM_OK,
   AUDIOSM_NOT_ALLOWED,
   AUDIOSM_KICKED,
   AUDIOSM_ERROR
   } enErrorCode;

   /**
    * Free all resources and cleanup.
    * Afterwards Initialization is needed
    */
   static tVoid Invalidate();

   /**
    * Notify availability change
    * @param srcID source of concern
    * @param availability actual state of availability
    * @param reason for the notification
    */
   static clAudioSMEngine::enErrorCode Source_AvailabilityChange(AudioSource::SourceID srcID,
         AudioSource::clAudioSource::enSourceAvailability availability,
         AudioSource::clAudioSource::enSourceAvailabilityReason reason);

    /**
      * Deactivate Source
      * This is actually working only on Foreground source
      * @param source deactivated source
      * @param optional flags (tbd)
      */
    static clAudioSMEngine::enErrorCode Source_Off(AudioSource::SourceID srcID, tU32 flags = 0);
    static clAudioSMEngine::enErrorCode Source_Off(AudioSource::SourceID srcID, tU16 u16SinkID, tU32 flags = 0);
    /**
      * Request source activation
      * @param source requested source
      * @param u32UserData optional userData
      */
    static clAudioSMEngine::enErrorCode Source_On(AudioSource::SourceID srcID, tU32 u32UserData=0);
    static clAudioSMEngine::enErrorCode Source_On(AudioSource::SourceID srcID, tU16 u16SinkID, tU32 u32UserData);
    static clAudioSMEngine::enErrorCode Source_Removed(AudioSource::SourceID source);
    static clAudioSMEngine::enErrorCode Source_Insert(AudioSource::SourceID source);

   /** overwrite the abstract interface clIAudioSourceControllerObserver */
   virtual tVoid vAudioSourceStarted(tU8 u8AudioSource);
   /** overwrite the abstract interface clIAudioSourceControllerObserver */
   virtual tVoid vAudioSourceStopped(tU8 u8AudioSource);
   /** overwrite the abstract interface clIAudioSourceControllerObserver */
   virtual tVoid vAudioSourceInitStarted(tU8 u8AudioSource);
   /** overwrite the abstract interface clIAudioSourceControllerObserver */
   virtual tVoid vEntertainmentSourceExchanged(tU8 u8NewEntertainmentSource, tBool bBackground);

   static tVoid vHandleSrcActError(AudioSource::SourceID errSrc);
   static tVoid ResetStack();
   static tVoid TraceStack();

   static tVoid TraceBackupStack();

   static clAudioSMEngine::enErrorCode SetRunlevel(tU8 u8NewRunlevel, tBool bStartUp=FALSE);

   //No need to create new source object if it's dynamic source vdu1kor SUZUKI-15832
   static tBool bIsSourceAvailable(AudioSource::SourceID u8AudioSource,tBool bCreateNewSource=true);

   static tU8   GetRunlevel();
   static tVoid ListSources();
   static time_t  GetTime_Init();
   static tVoid  vResetTime_Init();

   static tVoid vSetCurrentActiveSource(AudioSource::SourceID srcID);

   /* Current active source, not only entertainment source, but all source including logical,announcement sources */
   static AudioSource::SourceID m_enSource_Current_Active;

   /* Previous active source, not only entertainment source, but all sources including logical,announcement sources */
   static AudioSource::SourceID m_enSource_Previous_Active;

   static clSourceClass::sourceGroupID u8GetSrcGroupID_Current();
   static clSourceClass::sourceGroupID u8GetSrcGroupID_Previous();

private:
   /** constructor */
    clAudioSMEngine();
    virtual ~clAudioSMEngine();

    static clAudioSMEngine::enErrorCode SourceRemoved(AudioSource::SourceID source);

   //to be called when a AudioService changes its state
   static tVoid vOnAudioServiceStateChanged(tU16 u16Service, tU8 u8ServiceState);
   static clAudioSMEngine::enErrorCode AudioSourceOn(AudioSource::SourceID srcID, tU16 sinkID, tBool bForceStop, tU32 u32UserData);
   static tVoid vAudioSourceOff(AudioSource::SourceID srcID, tU32 u32UserData = 0);
   static tBool bAudioSourceOff(AudioSource::SourceID srcID, tU32 u32UserData = 0);
   static tBool bAudioSourceOff(AudioSource::SourceID srcID, tU16 sinkID, tU32 u32UserData = 0);

   static tVoid vDiagnosisAudioSourceOn(tU8 u8AudioSource, tBool bForce);
   static tVoid vDiagnosisAudioSourceOff(tU8 u8AudioSource, tBool bForce);
   static tVoid vDiagnosisOff();

   static AudioSource::SourceID GetNextAvailableEntertainmentSource(AudioSource::SourceID srcID);
   static clAudioSMEngine::enErrorCode SourceInserted(AudioSource::SourceID source);

   static tVoid vResetStack();

public:
   static tVoid vResetStack(int sinkID);
   static bool bIsPersistentMuteState();

   static tVoid vReadDataPool(tU8& srcID, tU16& subSrcID);
#ifdef VARIANT_S_FTR_ENABLE_CONTROLLERPLUGIN_PSA
   static tVoid vReadLastTunerDataPool();
   static tVoid vWriteLastTunerDataPool(tU8& u8SrcID, tU16& u16SubSrcID, bool bUserProfileChanged = false);
#endif
   static tVoid vWriteDataPool(bool bUserProfileChanged = false);
   static tVoid vWriteDataPool(tU8& srcID, tU16& subSrcID, bool bUserProfileChanged = false);
   static tVoid vRestorePersistentMuteState();
   static enErrorCode vRestoreEntertainmentsource(bool bUserProfileChanged=false);
   static clAudioSMEngine::enErrorCode vRestoreUserProfileEntsource(tBool bIsInitialProfileChange);
   static tVoid vDefaultSource(SourceID& Src);
private:
   static bool bLoadUsrProfileDataPool();
   static tBool bIsSystemFree();
   static tVoid vDPPersistentMute(AudioSource::clAudioSource* poAudioSource, enPersMuteDPAction action);

   /** LSM counter start**/
   static time_t   m_stInitTime;

   /** member stores the current runlevel 0(lowest)-3(highest) */
   static tU8      m_u8Runlevel;

   static tU8      m_u8AudioRouteMgr_SrvState;
   // Do we need this also for additional sinks ?
   static AudioSource::SourceID m_enSource_Entertainment_Previous;
   static AudioSource::SourceID m_enSource_Entertainment_Active;
   static AudioSource::SourceID m_enSource_Entertainment_LSM;
   static AudioSource::SourceID m_enSource_Entertainment_Last_Tuner;

   static clAudioSMEngine::enErrorCode RestorePreviousSource(AudioSource::SourceID srcID);

   static tU8      m_u8ErrorCode;

   static tVoid vStart_LSM_Timer(tU16 seconds);

   static tVoid vSetErrorCode(tU8 u8ErrorCode) {m_u8ErrorCode = u8ErrorCode;};
   static tU8 u8GetErrorCode()                 {return m_u8ErrorCode;};
   static tBool bIsMuteActive();

   static tVoid vProcessRegisteredUserProfileSrc(AudioSource::clAudioSource* poAudioSource, SourceID lastSource);
   static tVoid vProcessUnregisteredUserProfileSrc(SourceID lastSource);

};

}

#endif //CLAUDIOSMENGINEH



