/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */


#include "CAmDbusMessageHandler.h"
#include "audiomanagerconfig.h"
#include <cstdlib>
#include <cassert>
#include <vector>
#include "CAmCommandSenderDbus.h"
//#include "CAmDltWrapper.h"
#include "CAmLoggerDlt.h"

#include "commandplugin_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCOMMANDPLUGIN_DBUS_MSG
#include "trcGenProj/Header/CAmDbusMessageHandler.cpp.trc.h"
#endif

namespace am
{

CAmDbusMessageHandler::CAmDbusMessageHandler() :
        mDBusMessageIter(), //
        mDBusError(), //
        mSerial(0), //
        mErrorName(""), //
        mErrorMsg(""), //
        mpDBusMessage(NULL), //
        mpReveiveMessage(NULL), //
        mpDBusConnection(NULL)
{
  ET_TRACE_OPEN;
  ETG_TRACE_USR1(("CAmDbusMessageHandler::DBusMessageHandler constructed"));
}

CAmDbusMessageHandler::~CAmDbusMessageHandler()
{
  ETG_TRACE_USR1(("CAmDbusMessageHandler::DBusMessageHandler destructed"));
  mpDBusConnection = NULL;
  mpDBusMessage = NULL;
  mpReveiveMessage = NULL;
   ET_TRACE_CLOSE;
}

void CAmDbusMessageHandler::initReceive(DBusMessage* msg)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::initReceive called"))

  assert(msg!=NULL);
    mpReveiveMessage = msg;
    mErrorMsg.clear();
    mErrorName.clear();

    if (!dbus_message_iter_init(msg, &mDBusMessageIter))
    {
    ETG_TRACE_FATAL(("CAmDbusMessageHandler::initReceive DBus Message has no arguments!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBUS Message has no arguments!";
    }
}

void CAmDbusMessageHandler::initReceive_NoArgs(DBusMessage* msg)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::initReceive_NoArgs called"));
  assert(msg!=NULL);
    mpReveiveMessage = msg;
    mErrorMsg.clear();
    mErrorName.clear();

    if (dbus_message_iter_init(msg, &mDBusMessageIter))
    {
    ETG_TRACE_FATAL(("CAmDbusMessageHandler::initReceive_NoArgs invoked on a DBus Message with arguments! fix it !!!"));
    }
}


void CAmDbusMessageHandler::initReply(DBusMessage* msg)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::initReply called"));
  assert(msg!=NULL);
    mpDBusMessage = dbus_message_new_method_return(msg);
    if (mpDBusMessage == NULL)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::initReply Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
    dbus_message_iter_init_append(mpDBusMessage, &mDBusMessageIter);
}

void CAmDbusMessageHandler::initSignal(std::string path, std::string signalName)
{
  ETG_TRACE_USR4(("CAmDbusMessageHandler::initSignal called"));
  if(path.empty()||signalName.empty())
    assert(0);//So much to satisfy lint
    std::string completePath = std::string(DBUS_SERVICE_OBJECT_PATH) + "/" + path;
    std::string completeName = std::string(DBUS_SERVICE_PREFIX) + "." + path;
    mpDBusMessage = dbus_message_new_signal(completePath.c_str(), completeName.c_str(), signalName.c_str());

    if (mpDBusMessage == NULL)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::initSignal Cannot allocate DBus message!"));
    }
    dbus_message_iter_init_append(mpDBusMessage, &mDBusMessageIter);
}

void CAmDbusMessageHandler::sendMessage()
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::sendMessage called"));
  assert(mpDBusConnection!=NULL);
    if (mpReveiveMessage != 0)
    {
        mSerial = dbus_message_get_serial(mpReveiveMessage);
    }
    else
    {
        mSerial = 1;
    }
    if (!mErrorName.empty())
    {
        mpDBusMessage = dbus_message_new_error(mpReveiveMessage, mErrorName.c_str(), mErrorMsg.c_str());
    }
    if (!dbus_connection_send(mpDBusConnection, mpDBusMessage, &mSerial))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::sendMessage cannot send message!"));
    }
    dbus_connection_flush(mpDBusConnection);
    dbus_message_unref(mpDBusMessage);
    mpDBusMessage = NULL;
    mpReveiveMessage = NULL;
}

char* CAmDbusMessageHandler::getString()
{

  char* param = NULL;

    if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getString DBUS handler argument is no String!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no string";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &param);
        dbus_message_iter_next(&mDBusMessageIter);
        ETG_TRACE_USR4(("CAmDbusMessageHandler::getString called, param = %s", param))
    }
    return (param);
}

dbus_bool_t CAmDbusMessageHandler::getBool()
{
  dbus_bool_t boolparam = false;

    if (DBUS_TYPE_BOOLEAN != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getBool DBUS handler argument is no bool!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no bool";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &boolparam);
        dbus_message_iter_next(&mDBusMessageIter);
        ETG_TRACE_USR4(("CAmDbusMessageHandler::getBool called, boolparam = %u", boolparam))
    }
    return (boolparam);
}

char CAmDbusMessageHandler::getByte()
{
  char param(0);

    if (DBUS_TYPE_BYTE != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getByte DBUS handler argument is no byte!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no byte";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &param);
        dbus_message_iter_next(&mDBusMessageIter);
        ETG_TRACE_USR4(("CAmDbusMessageHandler::getByte called, param = %d", param))
    }
    return (param);
}

dbus_uint16_t CAmDbusMessageHandler::getUInt()
{
  dbus_uint16_t param(0);

#ifdef GLIB_DBUS_TYPES_TOLERANT
    if (DBUS_TYPE_UINT16 != dbus_message_iter_get_arg_type(&mDBusMessageIter) && DBUS_TYPE_UINT32 != dbus_message_iter_get_arg_type(&mDBusMessageIter))
#else
    if (DBUS_TYPE_UINT16 != dbus_message_iter_get_arg_type(&mDBusMessageIter))
#endif
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getUInt DBUS handler argument is no uint16_t!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no uint16_t";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &param);
        dbus_message_iter_next(&mDBusMessageIter);

        ETG_TRACE_USR4(("CAmDbusMessageHandler::getUInt called, param = %u", param))
    }
    return (param);
}

dbus_int16_t CAmDbusMessageHandler::getInt()
{
  dbus_int16_t param(0);

#ifdef GLIB_DBUS_TYPES_TOLERANT
    if (DBUS_TYPE_INT16 != dbus_message_iter_get_arg_type(&mDBusMessageIter) && DBUS_TYPE_INT32 != dbus_message_iter_get_arg_type(&mDBusMessageIter))
#else
    if (DBUS_TYPE_INT16 != dbus_message_iter_get_arg_type(&mDBusMessageIter))
#endif
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getInt DBUS handler argument is no int16_t!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no int16_t";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &param);
        dbus_message_iter_next(&mDBusMessageIter);

        ETG_TRACE_USR4(("CAmDbusMessageHandler::getInt called, param = %d", param))
    }
    return (param);
}

double CAmDbusMessageHandler::getDouble()
{
  double param(0);
    if (DBUS_TYPE_DOUBLE != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getDouble DBUS handler argument is no double!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no double";
    }
    else
    {
        dbus_message_iter_get_basic(&mDBusMessageIter, &param);
        dbus_message_iter_next(&mDBusMessageIter);

        ETG_TRACE_USR4(("CAmDbusMessageHandler::getDouble called, param = %f", (etg_tF32)param))
    }
    return (param);
}

void CAmDbusMessageHandler::getProperty(dbus_int16_t & type, dbus_int16_t & value)
{
  DBusMessageIter arrayIter;
    if (DBUS_TYPE_STRUCT != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getProperty DBUS handler argument is no array!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no array";
    }
    else
    {
        dbus_message_iter_recurse(&mDBusMessageIter, &arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &type);
        dbus_message_iter_next(&arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &value);
        dbus_message_iter_next(&mDBusMessageIter);

    ETG_TRACE_USR4(("CAmDbusMessageHandler::getProperty called"));
    }
}

void CAmDbusMessageHandler::append(bool toAppend)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(bool) called, toAppend = %u", toAppend))

  dbus_bool_t mybool = toAppend;
    if (!dbus_message_iter_append_basic(&mDBusMessageIter, DBUS_TYPE_BOOLEAN, &mybool))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(double toAppend)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(double) called, toAppend = %f", (etg_tF32)toAppend))

  if (!dbus_message_iter_append_basic(&mDBusMessageIter, DBUS_TYPE_DOUBLE, &toAppend))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(char toAppend)
{
     ETG_TRACE_USR4(("CAmDbusMessageHandler::append(char) called, toAppend = %d", toAppend))

  if (!dbus_message_iter_append_basic(&mDBusMessageIter, DBUS_TYPE_BYTE, &toAppend))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(dbus_int16_t toAppend)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(dbus_int16_t) called, toAppend = %d", toAppend))

  if (!dbus_message_iter_append_basic(&mDBusMessageIter, DBUS_TYPE_INT16, &toAppend))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(dbus_uint16_t toAppend)
{
  ETG_TRACE_USR4(("CAmDbusMessageHandler::append(dbus_uint16_t) called, toAppend = %u", toAppend));

  if (!dbus_message_iter_append_basic(&mDBusMessageIter, DBUS_TYPE_UINT16, &toAppend))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::setDBusConnection(DBusConnection *& connection)
{
    ETG_TRACE_USR1(("CAmDbusMessageHandler::setDBusConnection called"));
  assert(connection!=NULL);
    mpDBusConnection = connection;
}

void CAmDbusMessageHandler::append(const am::am_SinkType_s& sinkType)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_SinkType_s) called {availability = %u, reason = %u, volume = %u, mutestate = %u} for sink %u(%s)"
          , sinkType.availability.availability, sinkType.availability.availabilityReason
          , sinkType.volume, sinkType.muteState, sinkType.sinkID, sinkType.name.c_str()))

  DBusMessageIter structIter;
    DBusMessageIter structAvailIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &sinkType.sinkID);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &sinkType.name);

    success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_STRUCT, NULL, &structAvailIter);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &sinkType.availability.availability);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &sinkType.availability.availabilityReason);
    success = success && dbus_message_iter_close_container(&structIter, &structAvailIter);

    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &sinkType.volume);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &sinkType.muteState);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &sinkType.sinkClassID);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_SourceType_s & sourceType)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_SourceType_s) called, {availability = %u, reason = %u} for source %u(%s)"
          , sourceType.availability.availability, sourceType.availability.availabilityReason
          , sourceType.sourceID, sourceType.name.c_str()))

  DBusMessageIter structIter;
    DBusMessageIter structAvailIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &sourceType.sourceID);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &sourceType.name);

    success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_STRUCT, NULL, &structAvailIter);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &sourceType.availability.availability);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &sourceType.availability.availabilityReason);
    success = success && dbus_message_iter_close_container(&structIter, &structAvailIter);

    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &sourceType.sourceClassID);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_MainSoundProperty_s mainSoundProperty)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_MainSoundProperty_s) called, type = %u, value = %d"
          , mainSoundProperty.type, mainSoundProperty.value))

  DBusMessageIter structIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &mainSoundProperty.type);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &mainSoundProperty.value);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_Availability_s & availability)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_Availability_s) called, availability = %u, reason = %u"
          , availability.availability, availability.availabilityReason))

  DBusMessageIter structAvailIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structAvailIter);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &availability.availability);
    success = success && dbus_message_iter_append_basic(&structAvailIter, DBUS_TYPE_INT16, &availability.availabilityReason);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structAvailIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_SystemProperty_s & SystemProperty)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_SystemProperty_s) called, type = %u, value = %d"
          , SystemProperty.type, SystemProperty.value))

  DBusMessageIter structIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &SystemProperty.type);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &SystemProperty.value);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
void CAmDbusMessageHandler::append(const am::am_MainConnectionType_s& newMainConnections)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_MainConnectionType_s) called, ID = %u (%u --> %u), state = %u, delay = %u"
          , newMainConnections.mainConnectionID, newMainConnections.sourceID, newMainConnections.sinkID
          , newMainConnections.connectionState, newMainConnections.delay))

  DBusMessageIter structIter;
    dbus_bool_t success = true;

  success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
  success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &newMainConnections.mainConnectionID);
  success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &newMainConnections.sourceID);
  success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &newMainConnections.sinkID);
  success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &newMainConnections.delay);
  success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &newMainConnections.connectionState);
  success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

  if (!success)
  {
  ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
    mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
    mErrorMsg = "Cannot create reply!";
  }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_MainConnectionType_s>& listMainConnections)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_MainConnectionType_s>) called, size = %u"
          , listMainConnections.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    std::vector<am::am_MainConnectionType_s>::const_iterator listIterator = listMainConnections.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(qqqnn)", &arrayIter);
    for (; listIterator < listMainConnections.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->mainConnectionID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sourceID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sinkID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->delay);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->connectionState);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\tID = %u (%u --> %u), state = %u, delay = %u"
          , listIterator->mainConnectionID, listIterator->sourceID, listIterator->sinkID
          , listIterator->connectionState, listIterator->delay))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_SinkType_s> & listMainSinks)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_SinkType_s>) called, size = %u"
          , listMainSinks.size()))
  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    DBusMessageIter availIter;
    std::vector<am::am_SinkType_s>::const_iterator listIterator = listMainSinks.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(qs(nn)nnq)", &arrayIter);
    for (; listIterator < listMainSinks.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sinkID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &listIterator->name);
        success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_STRUCT, NULL, &availIter);
        success = success && dbus_message_iter_append_basic(&availIter, DBUS_TYPE_INT16, &listIterator->availability.availability);
        success = success && dbus_message_iter_append_basic(&availIter, DBUS_TYPE_INT16, &listIterator->availability.availabilityReason);
        success = success && dbus_message_iter_close_container(&structIter, &availIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->volume);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->muteState);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sinkClassID);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\t{availability = %u, reason = %u, mute-state = %u, volume = %u} for sink %u (%s)"
          , listIterator->availability.availability, listIterator->availability.availabilityReason, listIterator->muteState
          , listIterator->volume, listIterator->sinkID, listIterator->name.c_str()))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_SourceType_s> & listMainSources)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_SourceType_s>) called, size = %u", listMainSources.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    DBusMessageIter availIter;
    std::vector<am::am_SourceType_s>::const_iterator listIterator = listMainSources.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(qs(nn)q)", &arrayIter);
    for (; listIterator < listMainSources.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sourceID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &listIterator->name);
        success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_STRUCT, NULL, &availIter);
        success = success && dbus_message_iter_append_basic(&availIter, DBUS_TYPE_INT16, &listIterator->availability.availability);
        success = success && dbus_message_iter_append_basic(&availIter, DBUS_TYPE_INT16, &listIterator->availability.availabilityReason);
        success = success && dbus_message_iter_close_container(&structIter, &availIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sourceClassID);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\t{availability = %u, reason = %u} for source type = %u (%s)"
          , listIterator->availability.availability, listIterator->availability.availabilityReason
          , listIterator->sourceID, listIterator->name.c_str()))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_MainSoundProperty_s> & listMainSoundProperties)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_MainSoundProperty_s>) called, size = %u"
          , listMainSoundProperties.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    std::vector<am::am_MainSoundProperty_s>::const_iterator listIterator = listMainSoundProperties.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(nn)", &arrayIter);
    for (; listIterator < listMainSoundProperties.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->type);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->value);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\ttype = %u, value = %d"
          , listIterator->type, listIterator->value))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_SourceClass_s> & listSourceClasses)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_SourceClass_s>) called, size = %u"
          , listSourceClasses.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    DBusMessageIter propIter;
    DBusMessageIter innerIter;
    std::vector<am::am_SourceClass_s>::const_iterator listIterator = listSourceClasses.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(qsa(nn))", &arrayIter);
    for (; listIterator < listSourceClasses.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sourceClassID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &listIterator->name);
        success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_ARRAY, "(nn)", &innerIter);

        std::vector<am::am_ClassProperty_s>::const_iterator listInnerIterator = listIterator->listClassProperties.begin();
        for (; listInnerIterator < listIterator->listClassProperties.end(); ++listInnerIterator)
        {
            success = success && dbus_message_iter_open_container(&innerIter, DBUS_TYPE_STRUCT, NULL, &propIter);
            success = success && dbus_message_iter_append_basic(&propIter, DBUS_TYPE_INT16, &listInnerIterator->classProperty);
            success = success && dbus_message_iter_append_basic(&propIter, DBUS_TYPE_INT16, &listInnerIterator->value);
            success = success && dbus_message_iter_close_container(&innerIter, &propIter);
        }
        success = success && dbus_message_iter_close_container(&structIter, &innerIter);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\t %u properties for source class %u (%s)"
          , listIterator->listClassProperties.size(), listIterator->sourceClassID, listIterator->name.c_str()))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_SinkClass_s> & listSinkClasses)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_SinkClass_s>) called with %u sink-classes"
          , listSinkClasses.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    DBusMessageIter propIter;
    DBusMessageIter innerIter;
    std::vector<am::am_SinkClass_s>::const_iterator listIterator = listSinkClasses.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(qsa(nn))", &arrayIter);
    for (; listIterator < listSinkClasses.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_UINT16, &listIterator->sinkClassID);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_STRING, &listIterator->name);
        success = success && dbus_message_iter_open_container(&structIter, DBUS_TYPE_ARRAY, "(nn)", &innerIter);

        std::vector<am::am_ClassProperty_s>::const_iterator listInnerIterator = listIterator->listClassProperties.begin();
        for (; listInnerIterator < listIterator->listClassProperties.end(); ++listInnerIterator)
        {
            success = success && dbus_message_iter_open_container(&innerIter, DBUS_TYPE_STRUCT, NULL, &propIter);
            success = success && dbus_message_iter_append_basic(&propIter, DBUS_TYPE_INT16, &listInnerIterator->classProperty);
            success = success && dbus_message_iter_append_basic(&propIter, DBUS_TYPE_INT16, &listInnerIterator->value);
            success = success && dbus_message_iter_close_container(&innerIter, &propIter);
        }
        success = success && dbus_message_iter_close_container(&structIter, &innerIter);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\t%u properties for sink class %u (%s)"
          , listIterator->listClassProperties.size(), listIterator->sinkClassID, listIterator->name.c_str()))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_SystemProperty_s> & listSystemProperties)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_SystemProperty_s>) called with %u properties"
          , listSystemProperties.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    std::vector<am::am_SystemProperty_s>::const_iterator listIterator = listSystemProperties.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(nn)", &arrayIter);
    for (; listIterator < listSystemProperties.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->type);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->value);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\ttype = %u, value = %d", listIterator->type, listIterator->value))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::getNotificationConfiguration(dbus_int16_t& type, dbus_int16_t& status, dbus_int16_t& parameter)
{
    ETG_TRACE_USR2(("CAmDbusMessageHandler::getNotificationConfiguration called"))

  DBusMessageIter arrayIter;
    if (DBUS_TYPE_STRUCT != dbus_message_iter_get_arg_type(&mDBusMessageIter))
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::getProperty DBUS handler argument is no array!"));
        mErrorName = std::string(DBUS_ERROR_INVALID_ARGS);
        mErrorMsg = "DBus argument is no array";
    }
    else
    {
        dbus_message_iter_recurse(&mDBusMessageIter, &arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &type);
        dbus_message_iter_next(&arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &status);
        dbus_message_iter_next(&arrayIter);
        dbus_message_iter_get_basic(&arrayIter, &parameter);
        dbus_message_iter_next(&mDBusMessageIter);
    }
}

void CAmDbusMessageHandler::append(const std::vector<am::am_NotificationConfiguration_s>& listNotifications)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(vector<am::am_NotificationConfiguration_s>) called for %u notifications"
          , listNotifications.size()))

  DBusMessageIter arrayIter;
    DBusMessageIter structIter;
    std::vector<am::am_NotificationConfiguration_s>::const_iterator listIterator = listNotifications.begin();
    dbus_bool_t success = true;

    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_ARRAY, "(nnn)", &arrayIter);
    for (; listIterator < listNotifications.end(); ++listIterator)
    {
        success = success && dbus_message_iter_open_container(&arrayIter, DBUS_TYPE_STRUCT, NULL, &structIter);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->type);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->status);
        success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &listIterator->parameter);
        success = success && dbus_message_iter_close_container(&arrayIter, &structIter);

        ETG_TRACE_USR4(("\t\t\ttype = %u, status = %d, parameter = %d"
              , listIterator->type, listIterator->status, listIterator->parameter))
    }
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &arrayIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_NotificationPayload_s& notificationPayload)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_NotificationPayload_s) called with type = %u, value = %d"
          , notificationPayload.type, notificationPayload.value))

  DBusMessageIter structIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &notificationPayload.type);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &notificationPayload.value);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}

void CAmDbusMessageHandler::append(const am::am_NotificationConfiguration_s& notificationConfiguration)
{
    ETG_TRACE_USR4(("CAmDbusMessageHandler::append(am_NotificationConfiguration_s) called with type = %u, status = %d, parameter = %d"
           , notificationConfiguration.type, notificationConfiguration.status, notificationConfiguration.parameter))

  DBusMessageIter structIter;
    dbus_bool_t success = true;
    success = success && dbus_message_iter_open_container(&mDBusMessageIter, DBUS_TYPE_STRUCT, NULL, &structIter);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &notificationConfiguration.type);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &notificationConfiguration.status);
    success = success && dbus_message_iter_append_basic(&structIter, DBUS_TYPE_INT16, &notificationConfiguration.parameter);
    success = success && dbus_message_iter_close_container(&mDBusMessageIter, &structIter);

    if (!success)
    {
    ETG_TRACE_ERR(("CMD_AUDIO, CAmDbusMessageHandler::append Cannot allocate DBus message!"));
        mErrorName = std::string(DBUS_ERROR_NO_MEMORY);
        mErrorMsg = "Cannot create reply!";
    }
}
}//Namespace end
