/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */

#include "audiomanagertypes.h"
#include "CAmCommandSenderDbus.h"
#include <algorithm>
#include <string>
#include <vector>
#include <cassert>
#include <set>
#include "CAmDbusMessageHandler.h"
//#include "CAmDltWrapper.h"
#include "CAmDbusWrapper.h"
#include "CAmLoggerDlt.h"

#include "commandplugin_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AMCOMMANDPLUGIN
#include "trcGenProj/Header/CAmCommandSenderDbus.cpp.trc.h"
#endif

using namespace am;

/**
 * factory for plugin loading
 */
extern "C" IAmCommandSend* g3g_amcommandplugin_soFactory()
{
    return (new CAmCommandSenderDbus());
}

/**
 * destroy instance of commandSendInterface
 */
extern "C" void destroyRoutingPluginInterfaceDbus(IAmCommandSend* commandSendInterface)
{
    delete commandSendInterface;
}

CAmCommandSenderDbus::CAmCommandSenderDbus() :
        mCAmDbusMessageHandler(), //
        mIAmCommandReceiverShadow(), //
        mpCAmDbusWrapper(NULL), //
        mpIAmCommandReceive(NULL), //
        mReady(false)
{
  ET_TRACE_OPEN;

  ETG_TRACE_USR1(("CAmCommandSenderDbus::DbusCommandSender constructor called"));
}

CAmCommandSenderDbus::~CAmCommandSenderDbus()
{
  ETG_TRACE_USR1(("CAmCommandSenderDbus::DbusCommandSender destructed"));

  mpCAmDbusWrapper = NULL;
  mpIAmCommandReceive = NULL;
  ET_TRACE_CLOSE;
}

am_Error_e CAmCommandSenderDbus::startupInterface(IAmCommandReceive* commandreceiveinterface)
{
  ETG_TRACE_USR1(("CAmCommandSenderDbus::startupInterface called"));

    mpIAmCommandReceive = commandreceiveinterface;
    mIAmCommandReceiverShadow.setCommandReceiver(mpIAmCommandReceive);
    mpIAmCommandReceive->getDBusConnectionWrapper(mpCAmDbusWrapper);
    assert(mpCAmDbusWrapper!=NULL);
    DBusConnection * connection;
    mpCAmDbusWrapper->getDBusConnection(connection);
    assert(connection!=NULL);
    mCAmDbusMessageHandler.setDBusConnection(connection);
    return (E_OK);
}

void CAmCommandSenderDbus::setCommandReady(const uint16_t handle)
{
  vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);
    //todo:implement handle handling

  ETG_TRACE_USR1(("CAmCommandSenderDbus::setCommandReady called %d",handle));
    mReady = true;
    mpIAmCommandReceive->confirmCommandReady(handle, E_OK);//modified
}

void CAmCommandSenderDbus::setCommandRundown(const uint16_t handle)
{
  vGAM_CMD_PLUGIN_NULL_POINTER_CHECK(mpIAmCommandReceive);

  ETG_TRACE_USR1(("CAmCommandSenderDbus::setCommandRundown called , handle:%d", handle));
    mReady = false;
    mpIAmCommandReceive->confirmCommandRundown(handle, E_OK);//modified
    /**
     * todo: implement DbusCommandSender::cbCommunicationRundown()
     */
}

void CAmCommandSenderDbus::cbNewMainConnection(const am_MainConnectionType_s& mainConnection)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbNewMainConnection called, mainConnection ID = %u, sourceID = %u, sinkID = %u, delay = %d, connectionState = %u"
             , mainConnection.mainConnectionID, mainConnection.sourceID, mainConnection.sinkID, mainConnection.delay, (uint16_t)mainConnection.connectionState));

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("NewMainConnection"));
        mCAmDbusMessageHandler.append(mainConnection);
        mCAmDbusMessageHandler.sendMessage();

        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("NumberOfMainConnectionsChanged"));
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbNewMainConnection - NOT Ready - could not send signal for new connection %u (%u --> %u)"
             , mainConnection.mainConnectionID, mainConnection.sourceID, mainConnection.sinkID))
}

void CAmCommandSenderDbus::cbRemovedMainConnection(const am_mainConnectionID_t mainConnection)
{

    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbRemovedMainConnection called, mainConnection:%u",mainConnection));

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("RemovedMainConnection"));
        mCAmDbusMessageHandler.append(mainConnection);
        mCAmDbusMessageHandler.sendMessage();

        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("NumberOfMainConnectionsChanged"));
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbRemovedMainConnection - NOT Ready - could not send signal for mainConnection %u"
             , mainConnection));
}

void CAmCommandSenderDbus::cbNewSink(const am_SinkType_s& sink)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbNewSink called, SinkType = {sinkID:%u,volume:%d,mutestate:%u,sinkClassID:%u,Availability{availability:%u,availabilityreason:%u},name:%s}"
        , sink.sinkID,sink.volume,(uint16_t)sink.muteState,sink.sinkClassID,(uint16_t)sink.availability.availability,(uint16_t)sink.availability.availabilityReason,sink.name.c_str()))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), "SinkAdded");
        mCAmDbusMessageHandler.append(sink);

        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbNewSink - NOT Ready - could not send signal SinkAdded for sink %u (%s)"
             , sink.sinkID, sink.name.c_str()))
}

void CAmCommandSenderDbus::cbRemovedSink(const am_sinkID_t sinkID)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbRemovedSink called, sinkID:%u",sinkID));

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), "SinkRemoved");
        mCAmDbusMessageHandler.append(sinkID);

        mCAmDbusMessageHandler.sendMessage();
    }
    else
        ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbRemovedSink - NOT Ready - could not send signal SinkRemoved for sinkID:%u"
              , sinkID))
}

void CAmCommandSenderDbus::cbNewSource(const am_SourceType_s& source)
{

    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbNewSource called: sourceID:%u, Availability = %u, Reason = %u, sourceClassID = %u, name = %s"
          , source.sourceID, (uint16_t)source.availability.availability, (uint16_t)source.availability.availabilityReason, source.sourceClassID, source.name.c_str()))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), "SourceAdded");
        mCAmDbusMessageHandler.append(source);

        mCAmDbusMessageHandler.sendMessage();
    }
    else
        ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbNewSource - NOT Ready - could not send signal SourceAdded for source %u (%s)"
              , source.sourceID, source.name.c_str()))
}

void am::CAmCommandSenderDbus::cbRemovedSource(const am_sourceID_t source)
{
  ETG_TRACE_USR2(("CAmCommandSenderDbus::cbRemovedSource called, sourceID:%u",source))

  if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), "SourceRemoved");
        mCAmDbusMessageHandler.append(source);

        mCAmDbusMessageHandler.sendMessage();
    }
  else
      ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbRemovedSource - NOT Ready - could not send signal SourceRemoved"))
}

void CAmCommandSenderDbus::cbNumberOfSinkClassesChanged()
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbNumberOfSinkClassesChanged called"))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("NumberOfSinkClassesChanged"));
        mCAmDbusMessageHandler.sendMessage();
    }
    else
        ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbNumberOfSinkClassesChanged - NOT Ready"
             " - could not send signal NumberOfSinkClassesChanged"))
}

void CAmCommandSenderDbus::cbNumberOfSourceClassesChanged()
{
    ETG_TRACE_USR3(("CAmCommandSenderDbus::cbNumberOfSourceClassesChanged called"))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("NumberOfSourceClassesChanged"));
        mCAmDbusMessageHandler.sendMessage();
    }
    else
        ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbNumberOfSourceClassesChanged"
             " - NOT Ready - could not send signal NumberOfSourceClassesChanged"))
}

void CAmCommandSenderDbus::cbMainConnectionStateChanged(const am_mainConnectionID_t connectionID, const am_ConnectionState_e connectionState)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbMainConnectionStateChanged called, connectionID= %d, connectionState= %d"
          , connectionID, connectionState))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("MainConnectionStateChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) connectionID);
        mCAmDbusMessageHandler.append((dbus_int16_t) connectionState);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbMainConnectionStateChanged - NOT Ready"
             " - could not send signal MainConnectionStateChanged, connectionID= %d, connectionState= %d"
             , connectionID, connectionState))
}

void CAmCommandSenderDbus::cbMainSinkSoundPropertyChanged(const am_sinkID_t sinkID, const am_MainSoundProperty_s & soundProperty)
{
    ETG_TRACE_USR3(("CAmCommandSenderDbus::cbMainSinkSoundPropertyChanged called, sinkID= %d, SoundProperty type= %d, value= %d"
        , sinkID, soundProperty.type, soundProperty.value))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("MainSinkSoundPropertyChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sinkID);
        mCAmDbusMessageHandler.append(soundProperty);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbMainSinkSoundPropertyChanged - NOT Ready"
             " - could not send signal MainSinkSoundPropertyChanged, sinkID= %d, SoundProperty type= %d, value= %d"
             , sinkID, soundProperty.type, soundProperty.value))
}

void CAmCommandSenderDbus::cbMainSourceSoundPropertyChanged(const am_sourceID_t sourceID, const am_MainSoundProperty_s & SoundProperty)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbMainSourceSoundPropertyChanged called, sourceID= %d, SoundProperty type= %d, value= %d"
        , sourceID, SoundProperty.type, SoundProperty.value))
    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("MainSourceSoundPropertyChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sourceID);
        mCAmDbusMessageHandler.append(SoundProperty);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbMainSourceSoundPropertyChanged - NOT Ready"
             " - could not send signal MainSourceSoundPropertyChanged, sourceID= %d, SoundProperty type= %d, value= %d"
             , sourceID, SoundProperty.type, SoundProperty.value))
}

void CAmCommandSenderDbus::cbSinkAvailabilityChanged(const am_sinkID_t sinkID, const am_Availability_s & availability)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSinkAvailabilityChanged called, sinkID= %d, availability= %d, reason= %d"
          , sinkID, availability.availability, availability.availabilityReason))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SinkAvailabilityChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sinkID);
        mCAmDbusMessageHandler.append(availability);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSinkAvailabilityChanged - NOT Ready"
             " - could not send signal SinkAvailabilityChanged, sinkID= %d, availability= %d, reason= %d"
             , sinkID, availability.availability, availability.availabilityReason))
}

void CAmCommandSenderDbus::cbSourceAvailabilityChanged(const am_sourceID_t sourceID, const am_Availability_s & availability)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSourceAvailabilityChanged called, sourceID= %d, availability= %d, reason= %d}"
          , sourceID, availability.availability, availability.availabilityReason))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SourceAvailabilityChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sourceID);
        mCAmDbusMessageHandler.append(availability);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSourceAvailabilityChanged - NOT Ready"
             " - could not send signal SourceAvailabilityChanged, sourceID= %d, availability= %d, reason= %d}"
             , sourceID, availability.availability, availability.availabilityReason))
}

void CAmCommandSenderDbus::cbVolumeChanged(const am_sinkID_t sinkID, const am_mainVolume_t volume)
{
     ETG_TRACE_USR2(("CAmCommandSenderDbus::cbVolumeChanged called, sinkID= %d, volume= %d"
           , sinkID, volume))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("VolumeChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sinkID);
        mCAmDbusMessageHandler.append((dbus_int16_t) volume);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbVolumeChanged - NOT Ready"
             " - could not send signal VolumeChanged, sinkID= %d, volume= %d", sinkID, volume))
}

void CAmCommandSenderDbus::cbSinkMuteStateChanged(const am_sinkID_t sinkID, const am_MuteState_e muteState)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSinkMuteStateChanged called, sinkID= %d, muteState= %d"
          , sinkID, muteState))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SinkMuteStateChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) sinkID);
        mCAmDbusMessageHandler.append((dbus_int16_t) muteState);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSinkMuteStateChanged - NOT Ready"
             " - could not send signal SinkMuteStateChanged, sinkID= %d, muteState= %d"
             , sinkID, muteState))
}

void CAmCommandSenderDbus::cbSystemPropertyChanged(const am_SystemProperty_s & SystemProperty)
{
  ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSystemPropertyChanged called, SystemProperty {type= %d, value= %d}"
        , SystemProperty.type, SystemProperty.value))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SystemPropertyChanged"));
        mCAmDbusMessageHandler.append(SystemProperty);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSystemPropertyChanged - NOT Ready"
             " - could not send signal SystemPropertyChanged, {type= %d, value= %d}"
             , SystemProperty.type, SystemProperty.value))
}

void am::CAmCommandSenderDbus::cbTimingInformationChanged(const am_mainConnectionID_t mainConnectionID, const am_timeSync_t sectime)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbTimingInformationChanged called, mainConnectionID= %d, time= %d"
        , mainConnectionID, sectime))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("TimingInformationChanged"));
        mCAmDbusMessageHandler.append((dbus_uint16_t) mainConnectionID);
        mCAmDbusMessageHandler.append((dbus_int16_t) sectime);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbTimingInformationChanged - NOT Ready"
             " - could not send signal TimingInformationChanged, mainConnectionID= %d, time= %d"
             , mainConnectionID, sectime))
}

void CAmCommandSenderDbus::getInterfaceVersion(std::string & version) const
{
    ETG_TRACE_USR4(("CAmCommandSenderDbus::getInterfaceVersion called"))

  version = CommandVersion;
}

void am::CAmCommandSenderDbus::cbSinkUpdated(const am_sinkID_t sinkID, const am_sinkClass_t sinkClassID, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSinkUpdated called, sinkID= %d"
          , sinkID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SinkUpdated"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sinkID));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sinkClassID));
        mCAmDbusMessageHandler.append(listMainSoundProperties);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSinkUpdated - NOT Ready"
             " - could not send signal SinkUpdated, sinkID= %d", sinkID))
}

void am::CAmCommandSenderDbus::cbSourceUpdated(const am_sourceID_t sourceID, const am_sourceClass_t sourceClassID, const std::vector<am_MainSoundProperty_s>& listMainSoundProperties)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSourceUpdated called, sourceID= %d"
          , sourceID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SourceUpdated"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sourceID));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sourceClassID));
        mCAmDbusMessageHandler.append(listMainSoundProperties);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSourceUpdated - NOT Ready"
             " - could not send signal SourceUpdated, sourceID= %d", sourceID))
}

void am::CAmCommandSenderDbus::cbSinkNotification(const am_sinkID_t sinkID, const am_NotificationPayload_s& notification)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSinkNotification called, sinkID= %d"
          , sinkID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SinkNotification"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sinkID));
        mCAmDbusMessageHandler.append(notification);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSinkNotification - NOT Ready"
             " - could not send signal SinkNotification, sinkID= %d", sinkID))
}

void am::CAmCommandSenderDbus::cbSourceNotification(const am_sourceID_t sourceID, const am_NotificationPayload_s& notification)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSourceNotification called, sourceID= %d"
          , sourceID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SourceNotification"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sourceID));
        mCAmDbusMessageHandler.append(notification);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSourceNotification - NOT Ready"
             " - could not send signal SourceNotification, sourceID= %d", sourceID))
}

void am::CAmCommandSenderDbus::cbMainSinkNotificationConfigurationChanged(const am_sinkID_t sinkID, const am_NotificationConfiguration_s& mainNotificationConfiguration)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSinkMainNotificationConfigurationChanged called, sinkID= %d", sinkID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SinkMainNotificationConfigurationChanged"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sinkID));
        mCAmDbusMessageHandler.append(mainNotificationConfiguration);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSinkMainNotificationConfigurationChanged - NOT Ready"
             " - could not send signal SinkMainNotificationConfigurationChanged, sinkID= %d", sinkID))
}

void am::CAmCommandSenderDbus::cbMainSourceNotificationConfigurationChanged(const am_sourceID_t sourceID, const am_NotificationConfiguration_s& mainNotificationConfiguration)
{
    ETG_TRACE_USR2(("CAmCommandSenderDbus::cbSourceMainNotificationConfigurationChanged called, sourceID= %d", sourceID))

    if (mReady)
    {
        mCAmDbusMessageHandler.initSignal(std::string(MY_NODE), std::string("SourceMainNotificationConfigurationChanged"));
        mCAmDbusMessageHandler.append(static_cast<dbus_uint16_t>(sourceID));
        mCAmDbusMessageHandler.append(mainNotificationConfiguration);
        mCAmDbusMessageHandler.sendMessage();
    }
    else
       ETG_TRACE_ERR(("CMD_AUDIO, CAmCommandSenderDbus::cbSourceMainNotificationConfigurationChanged - NOT Ready"
             " - could not send signal SourceMainNotificationConfigurationChanged, sourceID= %d", sourceID))
}
