/**
 *  Copyright (c) 2012 BMW
 *
 *  \author Christian Mueller, christian.ei.mueller@bmw.de BMW 2011,2012
 *
 *  \copyright
 *  Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation files (the "Software"), to deal in the Software without restriction,
 *  including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so,
 *  subject to the following conditions:
 *  The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 *  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR
 *  THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *  For further information see http://www.genivi.org/.
 */

/*
 * DO NOT REMOVE ABOVE COPYRIGHT - Ranjit
 */
#ifndef _DBUSMESSAGEHANDLER_H_
#define _DBUSMESSAGEHANDLER_H_

#include <dbus/dbus.h>
#include <vector>
#include <sstream>
#include <string>
#include <list>
#include "g3g_dbusbindingdatatypes.h"

/**
 * handles DBus Messages, is used to extract & append parameters and send messages
 */

class g3g_dbusmessage;

class g3g_dbusmessagehandler
{
private:
	g3g_dbusmessagehandler();
public:
	g3g_dbusmessagehandler(DBusConnection* connection, std::string ifname, std::string objpath );
	g3g_dbusmessagehandler(DBusConnection* connection, std::string nodename, std::string ifname, std::string objpath );
    ~g3g_dbusmessagehandler();

    /**
     * sends out the signal
     */
    void sendSignal(g3g_dbusmessage* signal);

    /**
     * sends out method reply
     */
    void sendMethodReply(g3g_dbusmessage* method_call, g3g_dbusmessage* method_reply);

    /**
     * sends out method and expects no reply
     */
    void sendMethodCallNoReply(g3g_dbusmessage* method_call);

    /**
     * sends out the method call .... blocking call
     * @return Return value of the method in response
     */
    g3g_dbusmessage* sendMethodCall(g3g_dbusmessage* method_call);

    /**
     *  is called to initiate the reply to a message
     * @param msg pointer to the message the reply is for
     */
    g3g_dbusmessage* CreateReply(g3g_dbusmessage* method_call) const;

    /**
     * inits a signal to be sent via dbus
     * parameters can be added before sending the signal
     * @param signalName the signal name
     */
    g3g_dbusmessage* CreateSignal(std::string signalName) const;

    /**
     * inits a method call to be sent via dbus
     * parameters can be added before sending the method call
     * @param MethodName the method name
     */
    g3g_dbusmessage* CreateMethodCall(std::string MethodName) const;

    /**
     * inits a signal to be sent via dbus
     * parameters can be added before sending the signal
     * @param signalName the signal name
     */
    g3g_dbusmessage* CreateSignal_Basic(std::string interfacename, std::string objectpath,std::string signalName) const;

    /**
     * inits a method call to be sent via dbus
     * parameters can be added before sending the method call
     * @param MethodName the method name
     */
    g3g_dbusmessage* CreateMethodCall_Basic(std::string busname, std::string objectpath, std::string interfacename,std::string MethodName) const;


private:

    DBusConnection* mpDBusConnection;
    std::string msInterfaceName;
    std::string msPath;
    std::string msNodeName;
};

class g3g_dbusmessage
{
private:

    DBusError mDBusError;
    dbus_uint32_t mSerial;
    std::string mErrorName;
    std::string mErrorMsg;
    DBusMessage* mpDBusMessage;

    std::vector<DBusMessageIter*> mMessageIterList;
    void vClearMessageIterList();

    g3g_dbusmessage();

public:
    g3g_dbusmessage(DBusMessage* , bool bWriteMode = false, dbus_uint32_t reply_serial = 1);
	~g3g_dbusmessage();

    /**
     * the get functions return a value from the received dbus message
     * @return
     */
    dbus_uint16_t 	getUInt16();
    dbus_int16_t 	getInt16();
    dbus_uint32_t 	getUInt32();
    dbus_int32_t 	getInt32();
    dbus_uint64_t 	getUInt64();
    dbus_int64_t 	getInt64();
    dbus_bool_t 	getBool();
    char 			getByte();
    double 			getDouble();
    char* 			getString();
    void 			getProperty(dbus_int16_t& type, dbus_int16_t& value);

    /**
     * the overloaded append function appends different datatypes to the dbusmessage
     */
	void append					(uint8_t toAppend);
    void append					(dbus_int16_t toAppend);
    void append					(dbus_uint16_t toAppend);
    void append					(dbus_uint32_t toAppend);
    void append					(dbus_int32_t  toAppend);
    void append					(dbus_uint64_t toAppend);
    void append					(dbus_int64_t  toAppend);
    void append					(char toAppend);
    void append					(bool toAppend);
    void append					(double toAppend);
    void append					(const char* toAppend);
	void append				    (std::string& toAppend);

    /**
     * Container related functions
     */

    /*
     * Use the below functions for Read Operations
     */
    bool bRecurseSubContainer();
    bool bCloseSubContainer_ReadFinished();
    bool bSubContainerEnd();

    /*
     * Use the below function for write operations
     */
    bool bCreateSubContainer(T_e8_ContainerType eType, const char* signature=NULL);
    bool bCloseSubContainer_WriteFinished();

    /*
     * Helper functions
     */
    DBusMessage* getDbusMsgPtr()
    {
    	return mpDBusMessage;
    }
    dbus_uint32_t getSerial()
    {
    	return mSerial;
    }
    std::string getErrorName()
    {
    	return mErrorName;
    }
    std::string getErrorMessage()
    {
    	return mErrorMsg;
    }
    std::string getMemberName()
    {
    	std::string retval;
    	const char* s = dbus_message_get_member(mpDBusMessage);
    	if(NULL != s)
    	{
    		retval.assign(s);
    	}
    	return retval;
    }
    std::string getSenderUniqueName()
    {
    	std::string s;
    	const char *st = dbus_message_get_sender(mpDBusMessage);
    	if(st != NULL)
    	{
    		s = st;
    	}
    	return s;
    }
    std::string getSenderPath()
    {
    	std::string s;
    	const char *st = dbus_message_get_path(mpDBusMessage);
    	if(st != NULL)
    	{
    		s = st;
    	}
    	return s;
    }
    std::string getSenderInterface()
    {
    	std::string s;
    	const char *st = dbus_message_get_interface(mpDBusMessage);
    	if(st != NULL)
    	{
    		s = st;
    	}
    	return s;
    }
    std::string getSignature()
    {
    	std::string s;
    	const char* st= dbus_message_get_signature(mpDBusMessage);
    	if(st != NULL)
    	{
    		s = st;
    	}
    	return s;
    }
    void FreeDBusMessage()
    {
    	dbus_message_unref(mpDBusMessage);
    }

    friend class g3g_dbusmessagehandler;
};

#endif // _DBUSMESSAGEWRAPPER_H_
