/***************************************************************************
 * Copyright                                                               *
 *                                                                         *
 *     ESCRYPT GmbH - Embedded Security       ESCRYPT Inc.                 *
 *     Zentrum fuer IT-Sicherheit             315 E Eisenhower Parkway     *
 *     Lise-Meitner-Allee 4                   Suite 214                    *
 *     44801 Bochum                           Ann Arbor, MI 48108          *
 *     Germany                                USA                          *
 *                                                                         *
 *     http://www.escrypt.com                                              *
 *     info"at"escrypt.com                                                 *
 *                                                                         *
 * All Rights reserved                                                     *
 ***************************************************************************/

/***************************************************************************/
/*!
 \file        crypto.c

 \brief       Crypto wrapper

 \attention   Good random numbers are essential for key generation!

 $Rev: 812 $
 */
/***************************************************************************
 $Author: mlange $
 $Date: 2017-08-25 15:34:36 +0200 (Fr, 25. Aug 2017) $

 History:
 11-Feb-2013 DV ------------------------ Creation ---------------------------

 $ENDOFHISTORY$
 ****************************************************************************/

// TODO: Replace random number generation!
/***************************************************************************
 * 1. INCLUDES                                                             *
 ***************************************************************************/

/* ASN.1 structure headers */
#include "../../cmp/asn1-2005-multi-file-os-asn1c/PKIMessage.h"
#include "PrivateKeyInfo.h"

/* common headers */
#include "../../ecm/inc/ecm.h"
#include <esc_utils.h>
#include <esc_debug.h>
#include <esc_asn1_utils.h>

#include "base64.h"

/* CycurLIB headers */
#include <pkcs1_v15.h>
#include <sha_1.h>
#include <sha_256.h>


/* Crypto headers */
#include "RSAPublicKey.h"
#include "RSAPrivateKey.h"
#include "../inc/crypto.h"

/* ECM header */
#include "../../cmp/inc/asn1_utils.h"
#include <cmp.h>
//#include <ecm.h>
#include "../../ecm/inc/ecm_utils.h"


#ifdef UNIT_TESTING
#include "ecmUnitTests.h"
#endif


/***************************************************************************
 * 2. DEFINES                                                              *
 ***************************************************************************/

#ifndef EscRsa_ENABLE_GENRSA
#	error "configure CycurLIB to enable RSA key generation EscRsa_ENABLE_GENRSA"
#endif

#if EscRsa_KEY_BITS != 2048U
#	error "wrong RSA key size (need 2048 bit)"
#endif

/***************************************************************************
 * 3. DECLARATIONS                                                         *
 ***************************************************************************/
#ifdef DEBUG_APP
unsigned int dbgMask = 0xFF;
#endif

static BOOL
EscCrypto_ToBytesBE( UINT8 dest[ EscRsa_KEY_BYTES ], const EscRsa_FieldElementT* const source, UINT32 length);

static BOOL
EscCrypto_GetSeed( UINT8 seed[ EscRandom_RND_LENGTH ] );

UINT32 getKeyPairSize(UINT32 keySize);

/***************************************************************************
 * 4. IMPLEMENTATION OF FUNCTIONS                                          *
 ***************************************************************************/

/***************************************************************
 * Converts a word array into a byte array (big endian)        *
 ***************************************************************/

static BOOL EscCrypto_ToBytesBE( UINT8* dest, const EscRsa_FieldElementT* const source, UINT32 length ) {
    UINT16 i;
    EscRsa_HWORD v;
    BOOL hasFailed = TRUE;

    if ( dest != NULL && source != NULL ) {
        for ( i = 0U; i < length; i++ ) {
            v = source->words[ i >> ( EscRsa_WORD_SIZE / 2U ) ];
            v >>= ( (UINT16)( ( i & ( EscRsa_WORD_SIZE - 1U ) ) << 3 ) ) & 0xFFU;
            dest[ ( length - 1U ) - i ] = (UINT8)v;
        }
        hasFailed = FALSE;
    }
    return hasFailed;
}

static BOOL EscCrypto_ToUINT32( UINT32* const dest, const void* const vsource, UINT32 len ) {
    UINT16 i;
    BOOL hasFailed = TRUE;
    EscRsa_FieldElementT* source = (EscRsa_FieldElementT*)vsource;

    if ( dest != NULL && source != NULL ) {
        hasFailed = FALSE;
        for ( i = 1U; i < ( len / EscRsa_WORD_SIZE ); i++ ) {
            hasFailed |= (  source->words[ i ] != 0 );
        }
        *dest = (UINT32)source->words[ 0 ];
    }
    return hasFailed;
}


static BOOL EscCrypto_FromBytesBE( void* const dest, const UINT8* source, UINT32 keySize ) {

    if ( dest != NULL && source != NULL ) {
    	// EscRsaFe_FromBytesBE() does not care about the actual size of the Key.
    	// It just converts a byte array into a big endian word array until the given length.
    	// --> Therefore, we do not need to use dedicated functions for each Keysize!
    	// Furthermore, we can just cast the destination to an EscRsa_FieldElementT,
    	// as the FieldElements are all basically just a struct with a UINT32 Array
        EscRsaFe_FromBytesBE( (EscRsa_FieldElementT*) dest, source, keySize/8 );
        return FALSE;
    }
    return TRUE;
}

/***************************************************************
 * Creates some values which can be used for the RND seed      *
 ***************************************************************/
static BOOL EscCrypto_GetSeed( UINT8 seed[ EscRandom_RND_LENGTH ] ) {
    BOOL hasFailed = TRUE;
    /* for the HASH_DRBG implementation a nonce is required
     * -> acquired from /dev/random (entropy is acquired
     * from /dev/random as well) */
    FILE *fp = fopen( RANDOM_SOURCE, "r" );
    if ( fp != NULL ) {
        if ( EscRandom_RND_LENGTH == fread( seed, 1, EscRandom_RND_LENGTH, fp ) ) {
            hasFailed = FALSE;
        }
        fclose( fp );
    }
    return hasFailed;
}

BOOL EscCrypto_InitRandom( EscRandom_ContextT* randCtx ) {
    UINT8 seed[ EscRandom_RND_LENGTH ];
    BOOL hasFailed = TRUE;

    if ( randCtx != NULL ) {
        /* get seed data (used as NONCE for Hash DRBG) */
        hasFailed = EscCrypto_GetSeed( seed );
        /* next init step (Hash DRBG entropy is acquired within this function */
        hasFailed |= EscRandom_Init( randCtx, seed );
    }
    return hasFailed;
}

BOOL EscCrypto_GenerateKeyPair (
    EscRandom_ContextT* randCtx,
    void* const keyPair,
    const UINT32 pubKey,
    UINT8 **keyBlob,
    UINT32 *keyBlobSize,
    UINT32 keySize )
{
    BOOL hasFailed = TRUE;
    if (randCtx != NULL && keyPair != NULL) {

            dbgPrint( DBG_INFO, "generating key pair..\n");
            switch (keySize) {
            case 1024: {
                do{
                    hasFailed = EscRsa1024_GenerateKeyPair(randCtx,
                            (EscCrypto1024_RsaKeyPairT*) keyPair, pubKey);
                } while (hasFailed == TRUE);
                break;
            }
            case 2048: {
                do{
                    hasFailed = EscRsa_GenerateKeyPair(randCtx,
                            (EscCrypto_RsaKeyPairT*) keyPair, pubKey);
                } while (hasFailed == TRUE);
                break;
            }
            case 4096: {
                do{
                    hasFailed = EscRsa4096_GenerateKeyPair(randCtx,
                            (EscCrypto4096_RsaKeyPairT*) keyPair, pubKey);
                } while (hasFailed == TRUE);
                break;
            }
            default: {
                dbgPrint( DBG_INFO,
                        "wrong keySize for generating key pair..\n");
                break;
            }
            }
    }

    if (hasFailed != FALSE && keyPair != NULL) {
        switch (keySize) {
        case 1024: {
            memset(keyPair, 0, sizeof(EscRsa1024_KeyPairT));
            break;
        }
        case 2048: {
            memset(keyPair, 0, sizeof(EscRsa_KeyPairT));
            break;
        }
        case 4096: {
            memset(keyPair, 0, sizeof(EscRsa4096_KeyPairT));
            break;
        }
        default: {
            dbgPrint( DBG_INFO, "wrong keySize..\n");
            break;
        }
        }

    }
    return hasFailed;
}

BOOL
EscCrypto_EncodePublicKey(
    const void* const keyPair,
    UINT8* const pubKeyData,
    UINT32* const pubKeyDataSize,
    UINT32 keySize )
{
    BOOL hasFailed = TRUE;
    RSAPublicKey_t rsaPubKey = { };

    if (keyPair != NULL && pubKeyData != NULL && pubKeyDataSize != NULL
            && *pubKeyDataSize > keySize / 8) {
        UINT32 tempSize = (keySize / 8U + 1);
        UINT8* temp = malloc(tempSize);
        temp[0] = 0U;

        switch (keySize) {
        case 1024: {
            EscRsa1024_KeyPairT* ckeyPair = (EscRsa1024_KeyPairT*) keyPair;
            hasFailed = EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->n, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize, &rsaPubKey.modulus);

            hasFailed |= EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->e, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize,
                    &rsaPubKey.publicExponent);
        }
            break;

        case 2048: {
            EscRsa_KeyPairT* ckeyPair = (EscRsa_KeyPairT*) keyPair;
            hasFailed = EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->n, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize, &rsaPubKey.modulus);

            hasFailed |= EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->e, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize,
                    &rsaPubKey.publicExponent);
        }
            break;

        case 4096: {
            EscRsa4096_KeyPairT* ckeyPair = (EscRsa4096_KeyPairT*) keyPair;
            hasFailed = EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->n, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize, &rsaPubKey.modulus);

            hasFailed |= EscCrypto_ToBytesBE(&temp[1],
                    (EscRsa_FieldElementT*) &ckeyPair->e, keySize / 8U);
            hasFailed |= setBigInteger(temp, tempSize,
                    &rsaPubKey.publicExponent);
        }
            break;
        }

        free(temp);

        if (hasFailed == FALSE) {
            dbgPrint( DBG_INFO, "Encode public key\n");
            hasFailed |= asn1EncodeToBuffer(&rsaPubKey, &asn_DEF_RSAPublicKey,
                    pubKeyData, pubKeyDataSize, FALSE);
        }
    }

    if (hasFailed != FALSE && pubKeyData != NULL && pubKeyDataSize != NULL) {
        memset(pubKeyData, 0, *pubKeyDataSize);
    }

    ASN_STRUCT_FREE_CONTENTS_ONLY(asn_DEF_RSAPublicKey, &rsaPubKey);

    return hasFailed;
}


static BOOL rsaKey2privKey( RSAPrivateKey_t* rsaPrivKey, const void* const vkeyPair, UINT32 keySize) {
    UINT8* temp = NULL;
    UINT32 tempSize =  keySize/8 +1;
    BOOL failed = FALSE;

    temp = malloc(tempSize);

    if(!temp)
    	return TRUE;

    switch(keySize){

    case 1024:{

    	EscRsa1024_KeyPairT* keyPair = (EscRsa1024_KeyPairT*) vkeyPair;
        temp[ 0 ] = 0U;

        failed = ( asn_long2INTEGER( &rsaPrivKey->version, 0 ) != 0 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->n, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->modulus );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->e, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->d, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->p, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->q, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->dmp1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*)&keyPair->dmq1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->iqmp, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
    }
    	break;

    case 2048:
    {

    	EscRsa_KeyPairT* keyPair = (EscRsa_KeyPairT*) vkeyPair;
        temp[ 0 ] = 0U;

        failed = ( asn_long2INTEGER( &rsaPrivKey->version, 0 ) != 0 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->n, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->modulus );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->e, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->d, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->p, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->q, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->dmp1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*)&keyPair->dmq1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->iqmp, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
    }
    	break;

    case 4096:
    {

    	EscRsa4096_KeyPairT* keyPair = (EscRsa4096_KeyPairT*) vkeyPair;
        temp[ 0 ] = 0U;

        failed = ( asn_long2INTEGER( &rsaPrivKey->version, 0 ) != 0 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->n, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->modulus );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->e, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->d, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->p, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->q, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->prime2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->dmp1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*)&keyPair->dmq1, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );

        failed |= EscCrypto_ToBytesBE( &temp[ 1 ], (EscRsa_FieldElementT*) &keyPair->iqmp, keySize/8 );
        failed |= setBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
    }
    	break;

    default:
        dbgPrint(DBG_ERR, "Not supported Key Size while decoding Private Key");
        failed = TRUE;
        break;
    }

    escFreePtr( (UINT8 **) &temp );

    return failed;
}

BOOL EscCrypto_EncodePrivateKey(
    const void* const keyPair,
    UINT8* const privKeyData,
    UINT32* const privKeyDataSize,
	UINT32 keySize)
{
    BOOL hasFailed = TRUE;
    RSAPrivateKey_t rsaPrivKey = {};

    if ( keyPair != NULL &&
    		privKeyData != NULL &&
    		privKeyDataSize != NULL &&
    		*privKeyDataSize > ( 4 * keySize/8 ) ) {

    	hasFailed = rsaKey2privKey(&rsaPrivKey, keyPair, keySize);

        if ( hasFailed == FALSE ) {
            hasFailed |= asn1EncodeToBuffer( &rsaPrivKey, &asn_DEF_RSAPrivateKey,
            		privKeyData, privKeyDataSize, FALSE );
        }
    }

    if ( hasFailed != FALSE && privKeyData != NULL && privKeyDataSize != NULL ) {
        memset( privKeyData, 0, *privKeyDataSize );
    }

    ASN_STRUCT_FREE_CONTENTS_ONLY( asn_DEF_RSAPrivateKey, &rsaPrivKey );

    return hasFailed;
}

BOOL
EscCrypto_GetPubKeyModSize (
    UINT8 *const srcKeyData,
    UINT32 const srcKeySize,
    UINT32 *keyModSize)
{
    BOOL failed = FALSE;
    RSAPublicKey_t *rsaPubKey = NULL;

    dbgIn ( __func__ );

    failed = alloc( (void **) &rsaPubKey, sizeof ( RSAPublicKey_t ));

    ifNotFailed {
        failed = asn1Decode (
            srcKeyData, srcKeySize, &asn_DEF_RSAPublicKey, rsaPubKey, FALSE );
    }

    ifNotFailed {
        *keyModSize = rsaPubKey->modulus.size - 1;
    } else {
        *keyModSize = 0;
    }

    asnFreeStruct( (void **) &rsaPubKey, asn_DEF_RSAPublicKey );

    dbgOut ( __func__ );

    return ( failed );
}


BOOL EscCrypto_DecodePublicKey(
    void* const vkeyPair,
    UINT8* const pubKeyData,
    const UINT32 pubKeyDataSize,
	UINT32 keySize)
{
	UINT32 tempLen = keySize/8 + 1;
    UINT8* temp = malloc(tempLen);
    BOOL hasFailed = TRUE;
    RSAPublicKey_t rsaPubKey = {};

    if ( vkeyPair != NULL && pubKeyData != NULL && pubKeyDataSize > keySize / 8 ) {
        hasFailed = asn1Decode( pubKeyData, pubKeyDataSize, &asn_DEF_RSAPublicKey, &rsaPubKey, FALSE );

        if ( hasFailed == FALSE ) {
        	switch (keySize){
        	case 1024:{
        		EscRsa1024_KeyPairT* keyPair = (EscRsa1024_KeyPairT*) vkeyPair;
                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.modulus );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->n, &temp[ 1 ], keySize );

                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.publicExponent );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->e, &temp[ 1 ], keySize );
        	}
        		break;

        	case 2048:{
        		EscRsa_KeyPairT* keyPair = (EscRsa_KeyPairT*) vkeyPair;
                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.modulus );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->n, &temp[ 1 ], keySize );

                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.publicExponent );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->e, &temp[ 1 ], keySize );
        	}
        		break;

        	case 4096:{
        		EscRsa4096_KeyPairT* keyPair = (EscRsa4096_KeyPairT*) vkeyPair;
                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.modulus );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->n, &temp[ 1 ], keySize );

                hasFailed |= getBigInteger( temp, tempLen, &rsaPubKey.publicExponent );
                hasFailed |= EscCrypto_FromBytesBE( (EscRsa_FieldElementT*)&keyPair->e, &temp[ 1 ], keySize );
        	}
        		break;

        	}

        }
    }

    if ( hasFailed != FALSE && vkeyPair != NULL ) {
        memset( vkeyPair, 0, getKeyPairSize(keySize) );
    }

    free (temp);
    ASN_STRUCT_FREE_CONTENTS_ONLY( asn_DEF_RSAPublicKey, &rsaPubKey );

    return hasFailed;
}

static BOOL privKey2rsaKey(const RSAPrivateKey_t* const rsaPrivKey, void* vkeyPair, UINT32 keySize) {
    UINT8* temp = NULL;
    UINT32 tempSize = 0;
    long temp_long;
    BOOL failed = FALSE;

//    EscRsa_KeyPairT* keyPair = (EscRsa_KeyPairT*) vkeyPair;

    switch(keySize){

    case 1024:{
        tempSize = keySize/8 + 1;
    	temp = malloc(tempSize);


    	EscRsa1024_KeyPairT* keyPair = (EscRsa1024_KeyPairT*) vkeyPair;
        failed |= ( asn_INTEGER2long( &rsaPrivKey->version, &temp_long ) != 0 );
        failed |= ( temp_long != 0 );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->modulus );
        failed |= EscCrypto_FromBytesBE( &keyPair->n, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->e, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->d, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->p, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->q, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmp1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmq1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
        failed |= EscCrypto_FromBytesBE( &keyPair->iqmp, &temp[ 1 ], keySize );
    }
    	break;

    case 2048:
    {
    	temp = malloc(EscRsa_KEY_BYTES + 1);
    	tempSize = EscRsa_KEY_BYTES + 1;

    	EscRsa_KeyPairT* keyPair = (EscRsa_KeyPairT*) vkeyPair;
        failed |= ( asn_INTEGER2long( &rsaPrivKey->version, &temp_long ) != 0 );
        failed |= ( temp_long != 0 );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->modulus );
        failed |= EscCrypto_FromBytesBE( &keyPair->n, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->e, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->d, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->p, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->q, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmp1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmq1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
        failed |= EscCrypto_FromBytesBE( &keyPair->iqmp, &temp[ 1 ], keySize );
    }
    	break;

    case 4096:
    {
    	temp = malloc(EscRsa4096_KEY_BYTES + 1);
    	tempSize = EscRsa4096_KEY_BYTES + 1;

    	EscRsa4096_KeyPairT* keyPair = (EscRsa4096_KeyPairT*) vkeyPair;
        failed |= ( asn_INTEGER2long( &rsaPrivKey->version, &temp_long ) != 0 );
        failed |= ( temp_long != 0 );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->modulus );
        failed |= EscCrypto_FromBytesBE( &keyPair->n, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->publicExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->e, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->privateExponent );
        failed |= EscCrypto_FromBytesBE( &keyPair->d, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->p, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->prime2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->q, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent1 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmp1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->exponent2 );
        failed |= EscCrypto_FromBytesBE( &keyPair->dmq1, &temp[ 1 ], keySize );

        failed |= getBigInteger( temp, tempSize, &rsaPrivKey->coefficient );
        failed |= EscCrypto_FromBytesBE( &keyPair->iqmp, &temp[ 1 ], keySize );
    }
    	break;

    default:
    	dbgPrint(DBG_ERR, "Not supported Key Size while decoding Private Key");
    	return TRUE;
    }

    if(!temp)
    	return TRUE;

    free (temp);

    return failed;
}

BOOL EscCrypto_DecodePrivateKey(
    void* const keyPair,
    const UINT8* const privKeyData,
    const UINT32 privKeyDataSize,
	UINT32 keySize)
{
    if(!keyPair){
        return TRUE;
    }

    if( !privKeyData || privKeyDataSize < 1){
    	switch(keySize){
    	case 1024:
    		memset( keyPair, 0, sizeof(EscRsa1024_KeyPairT) );
    		break;
    	case 2048:
    		memset( keyPair, 0, sizeof(EscRsa_KeyPairT));
    		break;
    	case 4096:
    		memset( keyPair, 0, sizeof(EscRsa4096_KeyPairT));
    		break;
    	}
    	return TRUE;
    }

    // check Data for correct keysize
    // KeySize: 2x keysize[bit] (modulus, private exp)
    // 			+ 5x keysize/2[bit] (prime1, prime2,exp1,exp2,coeff) + public exp + padding
    if ( privKeyDataSize < ( 4U * ( keySize / 8U ) ) ||
    		privKeyDataSize > ( 6U * ( keySize / 8U ) ) ){
    	dbgPrint(DBG_ERR, "Stored Device Key (%d bytes) does not match configured Key Size\n", privKeyDataSize);
    	switch(keySize){
    	case 1024:
    		memset( keyPair, 0, sizeof(EscRsa1024_KeyPairT) );
    		break;
    	case 2048:
    		memset( keyPair, 0, sizeof(EscRsa_KeyPairT));
    		break;
    	case 4096:
    		memset( keyPair, 0, sizeof(EscRsa4096_KeyPairT));
    		break;
    	}
    	return TRUE;
    }

    // Decode keyData
    dbgPrint ( DBG_INFO, "Decode key file\n" );

    BOOL failed = FALSE;
    RSAPrivateKey_t* rsaPrivKey = NULL;
    failed = alloc ( (void **)&rsaPrivKey, sizeof ( RSAPrivateKey_t ));

    if(asn1Decode( privKeyData, privKeyDataSize, &asn_DEF_RSAPrivateKey, rsaPrivKey, FALSE )){
    	return TRUE;
    }

    failed = privKey2rsaKey(rsaPrivKey, keyPair, keySize);

    if ( failed && keyPair != NULL ) {
    	switch(keySize){
    	case 1024:
    		memset( keyPair, 0, sizeof(EscRsa1024_KeyPairT) );
    		break;
    	case 2048:
    		memset( keyPair, 0, sizeof(EscRsa_KeyPairT));
    		break;
    	case 4096:
    		memset( keyPair, 0, sizeof(EscRsa4096_KeyPairT));
    		break;
    	}
    }

    asnFreeStruct( (void **) &rsaPrivKey, asn_DEF_RSAPrivateKey );

    return failed;
}

UINT32 getKeyPairSize(UINT32 keySize){
	switch (keySize){
	case 1024:
		return sizeof(EscRsa1024_KeyPairT);
		break;

	case 2048:
		return sizeof(EscRsa_KeyPairT);
		break;

	case 4096:
		return sizeof(EscRsa4096_KeyPairT);
		break;

	default:
		return 0;
	}
}

BOOL EscCrypto_StorePrivateKey2(
    SDC_PARAMS_s *sdc,
    const char* const file,
    const UINT32 containerid,
    const UINT32 formatid,
    const void* const keyPair,
    UINT32 keySize)
{
    UINT32 privKeyDataSize = getKeyPairSize(keySize);
    UINT8* privKeyData = malloc(privKeyDataSize);

    BOOL hasFailed = FALSE;
    char *pem_encoded = NULL;
    char *result_with_linebreak = NULL;
    char *test = NULL;
    size_t s = 0;

    if (file != NULL && keyPair != NULL) {
        hasFailed = EscCrypto_EncodePrivateKey(keyPair, privKeyData,
                &privKeyDataSize, keySize);

        /* PKCS#8 container is used */
        if (containerid == ID_FORMAT_CONTAINER_PKCS8) {
            PrivateKeyInfo_t p8;
            memset(&p8, 0x0, sizeof(PrivateKeyInfo_t));
            UINT8 *p1Data = NULL;
            UINT32 p1DataLen = privKeyDataSize;
            UINT8 *d = NULL;
            hasFailed = alloc((void**) &p1Data, privKeyDataSize );
            if(!hasFailed){
                memcpy(p1Data, privKeyData, privKeyDataSize);
                hasFailed = asn1U322Integer(&p8.version, 0);
                hasFailed |= setObjIdentifier(&p8.privateKeyAlgorithm.algorithm,
                        oid_rsaEnc);

                if (!hasFailed) {
                    p8.privateKey.size = p1DataLen;
                    p8.privateKey.buf = p1Data;

                    memset(privKeyData, 0x0, privKeyDataSize);
                    privKeyDataSize = 0;
                    hasFailed = asn1Encode(&p8, &asn_DEF_PrivateKeyInfo, &d,
                            &privKeyDataSize, FALSE);

                    memcpy(privKeyData, d, privKeyDataSize);
                }

                if (hasFailed) {
                    escFreePtr( (UINT8 **)&p1Data);
                }
            }

            asnFreeStructContent(&p8, asn_DEF_PrivateKeyInfo);
            /* p1Data is already freed by the call above! */
            escFreePtr((UINT8 **) &d);
        }

        if (!hasFailed) {
            /* PEM format */
            if (formatid == ID_FORMAT_CONTAINER_PEM) {
                char *start_tag;
                char *end_tag;
                if (containerid == ID_FORMAT_CONTAINER_PKCS1) {
                    start_tag = PEM_KEY_START_TAG
                    ;
                    end_tag = PEM_KEY_END_TAG
                    ;
                } else if (containerid == ID_FORMAT_CONTAINER_PKCS8) {
                    start_tag = PEM_KEY_PKCS8_START_TAG
                    ;
                    end_tag = PEM_KEY_PKCS8_END_TAG
                    ;
                }

                dbgPrint( DBG_INFO,
                        "Starting convert binary privKeyData to Base64 PEM. \n");
                UINT32 outsize = 0;
                test = escBase64Encode(privKeyData, privKeyDataSize, &outsize);

                if (!test || outsize < 1) {
                    hasFailed = TRUE;
                } else {
                    char result[outsize + 1];
                    snprintf(result, outsize + 1, "%s", test);
                    result_with_linebreak = ecmAddPemNl(
                            (unsigned char *) result);

                    s = strlen(start_tag) + strlen(result_with_linebreak) + strlen(end_tag) + 1;
                    hasFailed = escCheckSizetForOv( s );

                    if ( FALSE == hasFailed ) {

                        pem_encoded = malloc( s );
                        if ( NULL == pem_encoded ) {
                            hasFailed = TRUE;
                        }
                    }

                    if ( FALSE == hasFailed ) {
                        memcpy(pem_encoded, start_tag, strlen(start_tag));
                        memcpy(pem_encoded + strlen(start_tag), result_with_linebreak, strlen(result_with_linebreak));
                        memcpy( pem_encoded + strlen(start_tag) + strlen(result_with_linebreak), end_tag, strlen(end_tag) + 1);
                    }

                    if (hasFailed == FALSE) {
                        hasFailed = writeByteStreamToFile(
                            sdc, file, (UINT8*) pem_encoded, strlen(pem_encoded));
                    }
                }
            }

            /* Key format is DER. */
            else if (formatid == ID_FORMAT_CONTAINER_DER) {
                if (hasFailed == FALSE) {
                    hasFailed = writeByteStreamToFile(
                        sdc, file, privKeyData, privKeyDataSize );
                }
            } else {
                hasFailed = TRUE;
            }
        }
    } else {
        hasFailed = TRUE;
    }

    escFreePtr((UINT8 **) &privKeyData);
    escFreePtr((UINT8 **) &pem_encoded);
    escFreePtr((UINT8 **) &result_with_linebreak);
    escFreePtr((UINT8 **) &test);

    return hasFailed;
}

BOOL EscCrypto_LoadPrivateKey( SDC_PARAMS_s *sdc, const char* const file, void* const keyPair,
        int keySize)
{
    if( !keyPair || keySize < 1024 || !file)
        return TRUE;

    // loadFile from Disk
    UINT8* byteStream = NULL;
    UINT32 streamSize = 0;
    if (loadFromFileToBuffer( sdc, file, &byteStream, &streamSize) || !byteStream
            || streamSize < 1 )
        return TRUE;

    // find Format - if PEM convert first to DER
    if (checkForPEM((char*) byteStream)) {
        if (checkForPEMRSAKey((char*) byteStream)) {
            UINT8* der_Data = NULL;
            UINT32 der_DataSize = 0;
            if (decodePEM(byteStream, streamSize, &der_Data, &der_DataSize)
                    || !der_Data || der_DataSize < 1) {
                free(byteStream);
                escFreePtr( (UINT8 **)&der_Data );
                dbgPrint(DBG_ERR, "Could not decode PEM Data");
                return TRUE;
            }
            free(byteStream);
            byteStream = der_Data;
            streamSize = der_DataSize;
        } else if (checkForPEMPKCS8Key((char*) byteStream)) {
            UINT8* der_Data = NULL;
            UINT32 der_DataSize = 0;
            if (decodePEM(byteStream, streamSize, &der_Data, &der_DataSize)
                    || !der_Data || der_DataSize < 1) {
                free(byteStream);
                escFreePtr( (UINT8 **)&der_Data );
                dbgPrint(DBG_ERR, "Could not decode PEM Data");
                return TRUE;
            }
            //extract RSA Private Key from PKCS#8 Key Container
            UINT8* rsaData = NULL;
            UINT32 rsaLength = 0;
            if (extractPrivKeyFromPKCS8(der_Data, der_DataSize, &rsaData,
                    &rsaLength) || !rsaData || rsaLength < 1) {
                free(byteStream);
                free(der_Data);
                dbgPrint(DBG_ERR,
                        "Could not extract RSA Private Key from PKCS#8 Container");
                return TRUE;
            }
            free(der_Data);
            free(byteStream);
            byteStream = rsaData;
            streamSize = rsaLength;
        } else {
            free(byteStream);
            dbgPrint(DBG_ERR,
                    "PEM encoded file is neither RSA nor PKCS#8 encoded");
            return TRUE;
        }
    } else {
        // DER encoded Data
        // In case of DER encoded File we do not know if PKCS#1 or PKCS#8
        // give PKCS#8 a try
        UINT8* rsaData = NULL;
        UINT32 rsaLength = 0;
        if (!extractPrivKeyFromPKCS8(byteStream, streamSize, &rsaData, &rsaLength)) {
            // extracting was successful
            free(byteStream);
            byteStream = rsaData;
            streamSize = rsaLength;
        }
        // else: Do nothing here as the byteStream is most likely already in PKCS#1 Format
    }
    // At this Point byteStream contains DER encoded PKCS#1 Data
    BOOL failed = FALSE;
    failed = EscCrypto_DecodePrivateKey(keyPair, byteStream, streamSize,
            keySize);

    // Clean up
    free(byteStream);
    byteStream = 0;

    return failed;

}

/* if pkcs#8 container is used, extract pkcs#1 privatekey */
BOOL extractPrivKeyFromPKCS8(const UINT8* pkcs8Data, UINT32 pkcs8Size,
        UINT8** rsaPrivKeyData, UINT32* rsaPrivKeySize) {

    if (!pkcs8Data || pkcs8Size < 1) {
        *rsaPrivKeyData = NULL;
        *rsaPrivKeySize = 0;
        return TRUE;
    }

    BOOL failed = FALSE;

    // Decode asn1 PrivateKeyInfo
    PrivateKeyInfo_t* privKey = NULL;
    failed = alloc((void **) &privKey, sizeof(PrivateKeyInfo_t));
    if (asn1Decode(pkcs8Data, pkcs8Size, &asn_DEF_PrivateKeyInfo, privKey,
            FALSE)) {
        asnFreeStruct((void **) &privKey, asn_DEF_PrivateKeyInfo);
        *rsaPrivKeyData = NULL;
        *rsaPrivKeySize = 0;
        return failed = TRUE;
    }

    *rsaPrivKeySize = privKey->privateKey.size;
    *rsaPrivKeyData = malloc(*rsaPrivKeySize);
    if (!*rsaPrivKeyData)
        failed = TRUE;

    ifNotFailed {
        memset(*rsaPrivKeyData, 0x0, sizeof(*rsaPrivKeySize));
        memcpy(*rsaPrivKeyData, privKey->privateKey.buf, *rsaPrivKeySize);
    }

    asnFreeStruct((void **) &privKey, asn_DEF_PrivateKeyInfo);
//	asnFreeStructContent( &privKey, asn_DEF_PrivateKeyInfo );

    return failed;
}


BOOL EscCrypto_SignMessage(
    const void* const keyPair,
    const UINT8 message[ ],
    const UINT32 messageLen,
    UINT8* signature,
    const HASH_t hashType,
    UINT32 keySize)
{
    BOOL failed = FALSE;
    UINT8 digest[ EscSha256_DIGEST_LEN ];

    switch(keySize)
    {
    case 1024:{
        EscPkcs1V15_SignParamsT1024 signer;
        EscPkcs1V15_PrivKeyT1024 privKey;


        if ( hashType == hash_sha1 ) {
            failed = EscSha1_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA1;
        } else if ( hashType == hash_sha256 ) {
            failed = EscSha256_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA256;
        } else {
            failed = TRUE;
        }

        EscRsa1024_KeyPairT* ckeyPair = (EscRsa1024_KeyPairT*) keyPair;
        failed |= EscCrypto_ToBytesBE(privKey.modulus,
                (EscRsa_FieldElementT*) &ckeyPair->n, keySize / 8);
        failed |= EscCrypto_ToBytesBE(privKey.privExp,
                (EscRsa_FieldElementT*) &ckeyPair->d, keySize / 8);
        if ( failed == FALSE ) {
            signer.hashDigest = digest;
            signer.privKey = &privKey;
            signer.signature = signature;
            failed |= EscPkcs1V15_Sign1024( &signer );
        }
    }
    break;

    case 2048:
    {
        EscPkcs1V15_SignParamsT signer;
        EscPkcs1V15_PrivKeyT privKey;


        if ( hashType == hash_sha1 ) {
            failed = EscSha1_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA1;
        } else if ( hashType == hash_sha256 ) {
            failed = EscSha256_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA256;
        } else {
            failed = TRUE;
        }
        EscRsa_KeyPairT* ckeyPair = (EscRsa_KeyPairT*) keyPair;
        failed |= EscCrypto_ToBytesBE( privKey.modulus, &ckeyPair->n, keySize/8 );
        failed |= EscCrypto_ToBytesBE( privKey.privExp, &ckeyPair->d, keySize/8 );
        if ( failed == FALSE ) {
            signer.hashDigest = digest;
            signer.privKey = &privKey;
            signer.signature = signature;
            failed |= EscPkcs1V15_Sign( &signer );
        }
    }
        break;

    case 4096:
    {
        EscPkcs1V15_SignParamsT4096 signer;
        EscPkcs1V15_PrivKeyT4096 privKey;


        if ( hashType == hash_sha1 ) {
            failed = EscSha1_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA1;
        } else if ( hashType == hash_sha256 ) {
            failed = EscSha256_Calc( message, messageLen, digest );
            signer.digestType = EscPkcs1V15_DIGEST_TYPE_SHA256;
        } else {
            failed = TRUE;
        }
        EscRsa4096_KeyPairT* ckeyPair = (EscRsa4096_KeyPairT*) keyPair;
        failed |= EscCrypto_ToBytesBE( privKey.modulus, (EscRsa_FieldElementT*)&ckeyPair->n, keySize/8 );
        failed |= EscCrypto_ToBytesBE( privKey.privExp, (EscRsa_FieldElementT*)&ckeyPair->d, keySize/8 );
        if ( failed == FALSE ) {
            signer.hashDigest = digest;
            signer.privKey = &privKey;
            signer.signature = signature;
            failed |= EscPkcs1V15_Sign4096( &signer );
        }
    }
        break;

    }

    return failed;
}

BOOL
EscCrypto_VerifyMessage(
    const void* const vkeyPair,
    const UINT8* message,
    const UINT32 messageLen,
    const UINT8* signature,
    const HASH_t hashType,
    UINT32 keySize)
{
    EscPkcs1V15_VerifyParamsT verifier;
    UINT8 digest[ EscSha256_DIGEST_LEN ];
    UINT8 modulus[ keySize/8 ];

    BOOL hasFailed = FALSE;
    memset ( digest, 0x0, sizeof ( digest ));

    if ( hashType == hash_sha1 ) {
        hasFailed = EscSha1_Calc( message, messageLen, digest );
        verifier.digestType = EscPkcs1V15_DIGEST_TYPE_SHA1;
    } else if ( hashType == hash_sha256 ) {
        hasFailed = EscSha256_Calc( message, messageLen, digest );
        verifier.digestType = EscPkcs1V15_DIGEST_TYPE_SHA256;
    } else {
        hasFailed = TRUE;
    }


    if ( hasFailed == FALSE ) {
        switch(keySize){
        case 1024:{
            EscRsa1024_KeyPairT* keyPair = (EscRsa1024_KeyPairT*) vkeyPair;
            hasFailed |= EscCrypto_ToBytesBE(modulus,
                    (EscRsa_FieldElementT*) &keyPair->n, keySize / 8);
            hasFailed |= EscCrypto_ToUINT32(&verifier.pubExp, &keyPair->e,
                    keySize / 8);

            if ( hasFailed == FALSE ) {
                verifier.modulus = modulus;
                verifier.hashDigest = digest;
                verifier.signature = signature;
                hasFailed = EscPkcs1V15_Verify1024( &verifier );
            }
        }
            break;

        case 2048:{
            EscRsa_KeyPairT* keyPair = (EscRsa_KeyPairT*) vkeyPair;
            hasFailed |= EscCrypto_ToBytesBE( modulus, &keyPair->n, keySize/8 );
            hasFailed |= EscCrypto_ToUINT32( &verifier.pubExp, &keyPair->e, keySize/8 );

            if ( hasFailed == FALSE ) {
                verifier.modulus = modulus;
                verifier.hashDigest = digest;
                verifier.signature = signature;
                hasFailed = EscPkcs1V15_Verify( &verifier );
            }
        }

            break;

        case 4096:{
            EscRsa4096_KeyPairT* keyPair = (EscRsa4096_KeyPairT*) vkeyPair;
            hasFailed |= EscCrypto_ToBytesBE(modulus,
                    (EscRsa_FieldElementT*) &keyPair->n, keySize / 8);
            hasFailed |= EscCrypto_ToUINT32(&verifier.pubExp, &keyPair->e,
                    keySize / 8);

            if ( hasFailed == FALSE ) {
                verifier.modulus = modulus;
                verifier.hashDigest = digest;
                verifier.signature = signature;
                hasFailed = EscPkcs1V15_Verify4096( &verifier );
            }
        }

            break;

        default:
            hasFailed = TRUE;
            break;
        }
    }

    return hasFailed;
}

BOOL
EscCrypto_StorePrivateKey( const char* const file, const EscCrypto_RsaKeyPairT* const keyPair ) {
    UINT8 privKeyData[ 8U * EscRsa_KEY_BYTES ];
    UINT32 privKeyDataSize = sizeof( privKeyData );
    BOOL hasFailed = TRUE;

    if ( file != NULL && keyPair != NULL ) {
        hasFailed = EscCrypto_EncodePrivateKey( keyPair, privKeyData, &privKeyDataSize, 2048 );

        if ( hasFailed == FALSE ) {
            FILE *fp = fopen( file, "w" );
            if ( fp != NULL ) {
                const UINT32 len = privKeyDataSize;
                if ( len != fwrite( privKeyData, 1, len, fp ) ) {
                    hasFailed = TRUE;
                }
            } else {
                hasFailed = TRUE;
            }
            if ( fp ) {
                fclose( fp );
            }
        }
    }
    return hasFailed;
}
