#pragma once
#include "attributes.h"
#include <string>
#include "css_margins.h"

namespace litehtml
{
    class property_value
    {
    public:
        tstring m_value;
        bool            m_important;

        property_value()
        {
            m_important = false;
        }
        property_value(const tchar_t* val, bool imp)
        {
            m_important = imp;
            m_value     = val;
        }
        property_value(const property_value& val)
        {
            m_value     = val.m_value;
            m_important = val.m_important;
        }

        property_value& operator=(const property_value& val)
        {
            m_value     = val.m_value;
            m_important = val.m_important;
            return *this;
        }
    };

    typedef std::map<tstring, property_value>   props_map;

    class style;
    class element;

    struct PropValue {
        typedef Ref<PropValue> ptr;

        enum Name {
            _Undefined_ = -1,
            Display,
            MarginTop,
            MarginBottom,
            MarginLeft,
            MarginRight,
            PaddingTop,
            PaddingBottom,
            PaddingLeft,
            PaddingRight,
            Width,
            Height,
            Position,
            Color,
            BackgroundColor,
            TextAlign,
            TextOverflow,
            LineHeight,
            MinWidth,
            MaxWidth,
            MinHeight,
            MaxHeight,
            TextIndent,
            FontWeight,
            FontSize,
            Overflow,
            WhiteSpace,
            VerticalAlign,
            Left,
            Top,
            Right,
            Bottom
        };

        bool m_important;

        virtual ~PropValue() {}

        DefineShareable()
    };

    struct DisplayPropValue : public PropValue {
        style_display m_value;
    };

    struct LengthPropValue : public PropValue {
        css_length m_value;
    };

    struct PositionPropValue : public PropValue {
        element_position m_value;
    };

    struct ColorPropValue : public PropValue {
        web_color m_value;
    };

    struct TextAlignPropValue : public PropValue {
        text_align m_value;
    };

    #define text_overflow_strings   _t("clip;ellipsis;string;initial;inherit")
    enum text_overflow {
        text_overflow_clip,
        text_overflow_ellipsis,
        text_overflow_string,
        text_overflow_initial,
        text_overflow_inherit
    };

    struct TextOverflowPropValue : public PropValue {
        text_overflow m_value;
    };

    struct OverflowPropValue : public PropValue {
        overflow m_value;
    };

    struct WhiteSpacePropValue : public PropValue {
        white_space m_value;
    };

    struct VerticalAlignPropValue : public PropValue {
        vertical_align m_value;
    };

    typedef std::map<PropValue::Name, PropValue::ptr> PropertyMap;

    /*struct PropValueSetter {
        typedef PropValue* (PropertySetter::*Handler)(const PropValue*, const tchar_t*);

        PropValue::Name m_name;
        Handler m_handler;
    };*/

    class PropertySetter {
        public:
            static PropertySetter& GetInstance();

            bool Parse(style& st, const tchar_t* name, const tchar_t* val, bool important);

        private:
            struct Handler {
                typedef PropValue* (PropertySetter::*HandlerFn)(style&, const tchar_t*);

                Handler() : m_handlerFn(0), m_name(PropValue::_Undefined_) {}
                Handler(PropValue::Name name, HandlerFn fn) : m_handlerFn(fn), m_name(name) {}

                HandlerFn m_handlerFn;
                PropValue::Name m_name;
            };
            typedef std::map<tstring, Handler> List;
            List m_list;

            PropertySetter();

            PropValue* SetDisplay(style& st, const tchar_t* val);
            PropValue* SetLength(style& st, const tchar_t* val);
            PropValue* SetPosition(style& st, const tchar_t* val);
            PropValue* SetColor(style& st, const tchar_t* val);
            PropValue* SetMargin(style& st, const tchar_t* val);
            PropValue* SetMargins(style& st, const tchar_t* val);
            PropValue* SetPaddings(style& st, const tchar_t* val);
            PropValue* SetTextAlign(style& st, const tchar_t* val);
            PropValue* SetTextOverflow(style& st, const tchar_t* val);
            PropValue* SetOverflow(style& st, const tchar_t* val);
            PropValue* SetWhiteSpace(style& st, const tchar_t* val);
            PropValue* SetVerticalAlign(style& st, const tchar_t* val);

            void SetMarginsPaddings(style& st, const tchar_t* val,
                                    PropValue::Name nameTop, PropValue::Name nameRight, PropValue::Name nameBottom, PropValue::Name nameLeft);
    };

    class style
    {
    public:
        typedef Ref<style>      ptr;
        typedef std::vector<style::ptr>     vector;
    private:
        props_map           m_properties;
        PropertyMap m_propertyMap;
        static string_map   m_valid_values;
        static string_map init_map();
    public:
        style();
        style(const style& val);
        virtual ~style();

        void operator=(const style& val)
        {
            m_properties = val.m_properties;
            m_propertyMap = val.m_propertyMap;
        }

        void add(const tchar_t* txt, const tchar_t* baseurl)
        {
            parse(txt, baseurl);
        }

        void add_property(const tchar_t* name, const tchar_t* val, const tchar_t* baseurl, bool important);

        const tchar_t* get_property(const tchar_t* name) const
        {
            if(name)
            {
                props_map::const_iterator f = m_properties.find(name);
                if(f != m_properties.end())
                {
                    return f->second.m_value.c_str();
                }
            }
            return 0;
        }

        const PropValue* GetProperty(PropValue::Name name) const
        {
            PropertyMap::const_iterator f = m_propertyMap.find(name);
            if (f != m_propertyMap.end()) {
                return &(*(f->second));
            }
            return 0;
        }

        void combine(const litehtml::style& src);
        void clear()
        {
            m_properties.clear();
        }

    private:
        friend class PropertySetter;

        void parse_property(const tstring& txt, const tchar_t* baseurl);
        void parse(const tchar_t* txt, const tchar_t* baseurl);
        void parse_short_border(const tstring& prefix, const tstring& val, bool important);
        void parse_short_background(const tstring& val, const tchar_t* baseurl, bool important);
        void parse_short_font(const tstring& val, bool important);
        void add_parsed_property(const tstring& name, const tstring& val, bool important);
        void AddParsedProperty(PropValue::Name name, PropValue* value, bool important);
        void remove_property(const tstring& name, bool important);

        DefineShareable()
    };
}
