#include <syslog.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <pwd.h>
#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>

#include "include/ConnManProxy.h"
#include "include/ConnManProxyManager.h"
#include "include/ConnManProxySession.h"
#include "include/DBus.h"

DLT_DECLARE_CONTEXT(CMP1)

static ConnManProxyManagerManager *manager = NULL;      // towards clients
static ConnManProxyManagerManager *connman = NULL;      // towards connman
static DBusDBus *dbusdaemon = NULL;                     // towards clients' dbus daemon
static GDBusConnection *session_connection = NULL;      // for communicating with clients
static GDBusConnection *system_connection = NULL;       // for communicating with connman
static gboolean connman_up = FALSE;
static guint owner_id = 0;
static guint connmanwatch_id = 0;
static GSList *client_sessions = NULL;
static GMainLoop *loop = NULL;
static gboolean shutdown_initiated = FALSE;
static GHashTable *userservice_hash = NULL;


struct policy_file {
	/*
	 * A valid file is a keyfile with one ore more groups. All
	 * groups are keept in this list.
	 */
	GSList *groups;
};


struct policy_group {
  GHashTable *bearer_source_ip_mapping;
  GSList *allowed_bearers_names; /* we maintain a local copy of the allowed bearers list */
  char *selinux;
  char *uid;
  char *gid;
  char *userservice;
  GSList *source_ips;
};


static void cleanup_group(gpointer user_data)
{
	struct policy_group *group = user_data;

	DBG_TRACE((CMP1,"group %p", group));

	if (group->userservice)
		g_hash_table_remove(userservice_hash, group->userservice);

	if (group->source_ips)
		g_slist_free(group->source_ips);

	if (group->allowed_bearers_names)
	{
		g_slist_free_full(group->allowed_bearers_names,g_free);
	}
	g_free(group->selinux);
	g_free(group->uid);
	g_free(group->gid);
	g_free(group->userservice);
	g_free(group);
}

dbus_cb_parameters * dbus_cb_param_new(GObject * client_proxy, GDBusMethodInvocation * invocation,  struct client_session * csession, void *user_data ){
  dbus_cb_parameters * cb_data = g_new(struct dbus_cb_parameters, 1);
  cb_data->client_proxy = client_proxy;
  cb_data->invocation = invocation;
  cb_data->csession = csession;
  cb_data->user_data = user_data;
  return cb_data;
}


int parse_source_ips(const char *token, GSList **list)
{
	*list = g_slist_append(*list, g_strdup(token));

	return 0;
}


static int load_keyfile(const char *pathname, GKeyFile **keyfile)
{
	GError *error = NULL;
	int err;

	*keyfile = g_key_file_new();

	if (!g_key_file_load_from_file(*keyfile, pathname, 0, &error))
		goto err;

	return 0;

err:
	/*
	 * The fancy G_FILE_ERROR_* codes are identical to the native
	 * error codes.
	 */
	err = -error->code;

	INF_TRACE((CMP1,"Unable to load %s: %s", pathname, error->message));
	g_clear_error(&error);

	g_key_file_free(*keyfile);
	*keyfile = NULL;

	return err;
}


static int load_policy(GKeyFile *keyfile, const char *groupname,
			struct policy_group *group)
{
	char *str, **tokens;
	int i, err = 0;

	group->selinux = g_key_file_get_string(keyfile, groupname,
						"selinux", NULL);

	group->gid = g_key_file_get_string(keyfile, groupname,
						"gid", NULL);

	group->uid = g_key_file_get_string(keyfile, groupname,
						"uid", NULL);

	if (!group->selinux && !group->gid && !group->uid)
		return -1;

	group->userservice = g_key_file_get_string(keyfile, groupname,
			"Service", NULL);
	if (group->userservice != NULL) {
		if (group->uid  != NULL)
			group->userservice = g_strconcat("uid:",group->uid,"/",group->userservice, NULL);
		if ((group->selinux  != NULL) ||  (group->gid  != NULL))
		      return -1;
	}
	str = g_key_file_get_string(keyfile, groupname, "AllowedBearers",
				NULL);
	if (str) {
		g_slist_free(group->allowed_bearers_names);
		group->allowed_bearers_names = NULL;
		tokens = g_strsplit(str, " ", 0);

		for (i = 0; tokens[i]; i++) {
		       group->allowed_bearers_names = g_slist_append(group->allowed_bearers_names, g_strdup(tokens[i]));
		}

		g_free(str);
		g_strfreev(tokens);
	}

	str = g_key_file_get_string(keyfile, groupname, "SourceIPs",
				NULL);
	if (str) {
		int noips;
		g_slist_free(group->source_ips);
		group->source_ips = NULL;
		tokens = g_strsplit(str, " ", 0);
		noips = g_strv_length(tokens);

		if (noips != 1 && noips != g_slist_length(group->allowed_bearers_names))
			return -1;

		for (i = 0; i < g_slist_length(group->allowed_bearers_names); i++) {
			if (tokens[i]) {
				err = parse_source_ips(tokens[i],
						&group->source_ips);
			} else {
				err = parse_source_ips(tokens[1],
						&group->source_ips);
			}
			if (err < 0)
				break;
		}

	 	g_free(str);
	 	g_strfreev(tokens);
	 }


	INF_TRACE((CMP1,"group %p selinux %s uid %s gid %s", group, group->selinux,
	     group->uid, group->gid));

	return err;
}

/*static int load_policy_file()
{
  GKeyFile *keyfile;
	struct policy_group *group;
	 char **groupnames;

	 int err = 0, i;
	 err = load_keyfile(POLICY_FILE, &keyfile);
	 if (err < 0)
	 	return err;

	 groupnames = g_key_file_get_groups(keyfile, NULL);

	 for (i = 0; groupnames[i]; i++) {
	 	group = g_new0(struct policy_group, 1);
	 	err = load_policy(keyfile, groupnames[i], group);
		if (group->userservice)
			g_hash_table_replace(userservice_hash, group->userservice, group);
	}

	g_strfreev(groupnames);
	g_key_file_free(keyfile);

	return 1;
}*/


gboolean
check_caller_credentials(GDBusMethodInvocation * invocation,
                         struct client_session *csession)
{
  guint uid;
  GError *error = NULL;

  INF_TRACE((CMP1,"[session %p] checking caller credentials for with expected UID %d",
       csession, csession->uid));

  dbus_dbus_call_get_connection_unix_user_sync(dbusdaemon,
                                               g_dbus_method_invocation_get_sender
                                               (invocation), &uid, NULL,
                                               &error);

  if (error) {
    INF_TRACE
      ((CMP1,"[session %p] error getting credentials. Error code: %d Error message: %s",
	csession, error->code, error->message));
    g_error_free(error);
    return FALSE;
  }
  if (csession->uid != uid) {
    INF_TRACE
      ((CMP1,"[session %p] credentials mismatch. UID session: %d  UID caller: %d",
	csession, csession->uid, uid));
    return FALSE;
  }

  INF_TRACE((CMP1,"[session %p] credentials okay", csession, invocation, csession->uid,
       uid));
  return TRUE;
}

gboolean setup_notification_client(struct client_session * csession)
{
  GError *error = NULL;

  INF_TRACE((CMP1,"[session %p] setting up notification client for notifier %s",
       csession, csession->notifier));

  if (csession->notification_client) {
    DBG_TRACE((CMP1,"[session %p] notification client for client already exists!",
	 csession));
    return FALSE;
  }

  csession->notification_client =
    conn_man_proxy_notification_notification_proxy_new_sync
    (session_connection,
     G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES |
     G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS, csession->sender,
     csession->notifier, NULL, &error);
  g_dbus_proxy_set_default_timeout((GDBusProxy*) csession->notification_client,CONNMANPROXY_IDDBUS_TIMEOUT);

  if (error) {
    ERR_TRACE
      ((CMP1,"[session %p] error setting up notification client. Error code: %d Error message: %s ",
	csession, error->code, error->message));
    g_error_free(error);
    return FALSE;
  }

  INF_TRACE((CMP1,"[session %p] notification client %p ready", csession,
       csession->notification_client));
  return TRUE;
}




void free_client_session(struct client_session *csession)
{
  if(!csession)
    {
      ERR_TRACE((CMP1,"free_client_session called with NULL pointer"));
      return;
    }
  INF_TRACE((CMP1,"[session %p] freeing session ", csession));
  if (csession->notification_server) {
    DBG_TRACE((CMP1,"[session %p] freeing notification_server", csession));
    g_dbus_interface_skeleton_unexport_from_connection
      (G_DBUS_INTERFACE_SKELETON(csession->notification_server),
       system_connection);
    g_object_unref(csession->notification_server);
  }
  if (csession->session_server) {
      DBG_TRACE((CMP1,"[session %p] freeing session_server", csession));
    g_dbus_interface_skeleton_unexport_from_connection
      (G_DBUS_INTERFACE_SKELETON(csession->session_server), session_connection);
    g_object_unref(csession->session_server);
  }
  if (csession->notification_client)
    DBG_TRACE((CMP1,"[session %p] freeing notification_client", csession));
    g_object_unref(csession->notification_client);

  if (csession->session_client) {
    DBG_TRACE((CMP1,"[session %p] freeing session_client", csession));
    if (csession->connman_initiated_closing)
      DBG_TRACE((CMP1,"[session %p] session destroy already initiated", csession));
    else
      {
	DBG_TRACE((CMP1,"[session %p] destroying session on native bus", csession));
	conn_man_proxy_session_session_call_destroy_sync(csession->session_client,
							 NULL, NULL);
	g_object_unref(csession->session_client);
      }
  }
  if (csession->watchid)
    {
      DBG_TRACE((CMP1,"[session %p] unwatching connman", csession));
      g_bus_unwatch_name(csession->watchid);
    }
  DBG_TRACE((CMP1,"[session %p] freeing pointers", csession));
  g_free(csession->notifier);
  g_free(csession->new_notifier);
  g_free(csession->sender);
  g_free(csession->session);
  g_free(csession);
  DBG_TRACE((CMP1,"[session %p] freed", csession));
}

void close_session(struct client_session *csession)
{
  gint session_active = -1;
  DBG_TRACE((CMP1,"[session %p] closing session", csession));
  session_active = g_slist_index(client_sessions, csession);
  if (session_active < 0)
     DBG_TRACE((CMP1,"[session %p] already closing session", csession));
  else {
    client_sessions = g_slist_remove(client_sessions, csession);
    free_client_session(csession);
    DBG_TRACE((CMP1,"[session %p] closed session", csession));
  }
}


void
client_vanished_handler(GDBusConnection * connection,
                        const gchar * name, gpointer user_data)
{
  struct client_session *csession = (struct client_session *) user_data;

  INF_TRACE((CMP1,"[session %p] client %s has vanished from session bus", csession,
       csession->sender));

  close_session(csession);
}


gchar *get_random_string(guint len)
{
  gchar *s;

  s = g_malloc0(len + 1);
  for (int i = 0; i < len; i++) {
    s[i] = g_random_int_range(65, 91);
  }

  return s;
}


static void release_msg_handler_cb(GObject *notification_client, GAsyncResult *res, void *user_data ){
  gboolean ret = FALSE;
  dbus_cb_parameters *cb_data  = (dbus_cb_parameters *) user_data;
  GError *error = NULL;
  struct client_session *csession = cb_data->csession;

  ret = conn_man_proxy_notification_notification_call_release_finish((ConnManProxyNotificationNotification *) notification_client, res, &error);
  if(!res){
    ERR_TRACE((CMP1, "[session %p] failed to forward release: %d %s", cb_data->csession, error->code, error->message));
    g_dbus_method_invocation_return_error(cb_data->invocation,  CONNMANPROXY_ERROR,
                                          CONNMANPROXY_NOTIFICATION_CLIENT_RELEASE_FAILED,
                                          "%s",
                                          "Release messasge forward failed.");
    g_free(error);
    goto err;
  }
  csession->connman_initiated_closing = TRUE;
  close_session(csession);
  DBG_TRACE((CMP1,"[session %p] release msg forwarded", csession));
 err:
  g_free(cb_data);
}


gboolean
release_msg_handler(ConnManProxyNotificationNotification * object,
                    GDBusMethodInvocation * invocation, gpointer * user_data)
{
  gboolean ret = TRUE;
  GError *error = NULL;
  client_session *csession = (client_session *) user_data;
  dbus_cb_parameters * cb_data = NULL;
  DBG_TRACE((CMP1,"[session %p] release msg received", csession));

  if (!csession->notification_client) {
    ret = setup_notification_client(csession);
  }

  if (!ret) {
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_NOTIFICATION_CLIENT_SETUP_FAILED,
                                          "%s",
                                          "Notification client setup failed.");
    return TRUE;
  }
  cb_data = dbus_cb_param_new((GObject*) object, invocation,csession, NULL);
  conn_man_proxy_notification_notification_call_release(csession->notification_client,
                                                              NULL, release_msg_handler_cb, cb_data);
  return TRUE;
}

static void update_msg_call_update_cb(GObject  *notification_proxy, GAsyncResult *res, gpointer user_data){

   gboolean success = FALSE;
   GError *error = NULL;
   struct dbus_cb_parameters *cb_data = (struct dbus_cb_parameters* ) user_data;
   struct client_session *csession = cb_data->csession;

   success =  conn_man_proxy_notification_notification_call_update_finish((ConnManProxyNotificationNotification *) notification_proxy, res, &error);
   if (success){
     INF_TRACE((CMP1,"[session %p] forwarding update message success", csession));
     conn_man_proxy_notification_notification_complete_update((ConnManProxyNotificationNotification *) cb_data->client_proxy, cb_data->invocation);
   }
   else {
     if (!error)
       ERR_TRACE((CMP1,"[session %p] forwarding update message failed. No error message", csession));
     else {
       ERR_TRACE((CMP1,"[session %p] error forwarding update. error code: %d error message: %s",
                  csession, error->code, error->message));
       g_error_free(error);
     }
     //INF_TRACE((CMP1,"[session %p] will close session",csession));
     //close_session(csession);
   }
   g_free(cb_data);
}


gboolean
update_msg_handler(ConnManProxyNotificationNotification * object,
                   GDBusMethodInvocation * invocation,
                   GVariant * arg_settings, gpointer * user_data)
{
  GError *error = NULL;
  gboolean ret = TRUE;
  struct client_session * csession = (struct client_session *) user_data;
  struct dbus_cb_parameters * cb_data = NULL;
  INF_TRACE((CMP1,"[session %p] update received", csession));

  if (!csession->notification_client) {
    INF_TRACE((CMP1,"sessions notification client is NULL .. creating"));
    ret = setup_notification_client(csession);
  }

  if (!ret) {
    ERR_TRACE((CMP1,"[session %p] error setting up notification client", csession));
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_NOTIFICATION_CLIENT_SETUP_FAILED,
                                          "%s",
                                          "Notification client setup failed.");
    return FALSE;
  }
  cb_data = dbus_cb_param_new((GObject* ) object, invocation, csession, NULL);
  conn_man_proxy_notification_notification_call_update(csession->notification_client, arg_settings,
							 NULL, update_msg_call_update_cb, cb_data);

  return TRUE;

}

static void change_msg_handler_cb(GObject *session_proxy, GAsyncResult * res, void *user_data){
  GError *error = NULL;
  gboolean ret = FALSE;
  dbus_cb_parameters *cb_data = (dbus_cb_parameters * ) user_data;

  ret =  conn_man_proxy_session_session_call_change_finish((ConnManProxySessionSession * ) session_proxy, res, &error);
  if (!ret){
    if (error) {
      ERR_TRACE
        ((CMP1,"[session %p] error changing session settings. error code: %d error message: %s",
          cb_data->csession, error->code, error->message));
      g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                            error->code, "%s", error->message);
      g_free(error);
      goto err;
    }

    ERR_TRACE((CMP1,"[session %p] error changing session",  cb_data->csession));
  }

    conn_man_proxy_session_session_complete_change((ConnManProxySessionSession * ) session_proxy, cb_data->invocation);
    INF_TRACE((CMP1,"[session %p] change forwarded", cb_data->csession));
 err:
    g_free(cb_data);

}


gboolean
change_msg_handler(ConnManProxySessionSession * object,
                   GDBusMethodInvocation * invocation,
                   const gchar * arg_name,
                   GVariant * arg_value, gpointer * user_data)
{
  GError *error = NULL;
  gboolean ret;
  client_session *csession = (client_session *) user_data;
  dbus_cb_parameters *cb_data = NULL;
  INF_TRACE((CMP1,"[session %p] change received", csession));

  if (!check_caller_credentials(invocation, csession)) {
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_CREDENTIALS_MISMATCH,
                                          "%s",
                                          "credentials mismatch/error for session");
    return TRUE;
  }
  cb_data = dbus_cb_param_new((GObject*) object, invocation, csession, NULL);
  conn_man_proxy_session_session_call_change(csession->session_client,
                                                    arg_name, arg_value,
                                               NULL, change_msg_handler_cb, cb_data);
  return TRUE;
}

static void connect_msg_handler_cb(GObject * object, GAsyncResult * res, void * user_data){
  GError *error = NULL;
  gboolean success = FALSE;
  struct  dbus_cb_parameters *cb_data = (struct dbus_cb_parameters *) user_data;

  success = conn_man_proxy_session_session_call_connect_finish(cb_data->csession->session_client, res, &error);
  if (!success){
    if (error) {
      ERR_TRACE
        ((CMP1,"[session %p] error connecting session. error code: %d error message: %s",
          cb_data->csession, error->code, error->message));
      g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                            error->code, "%s", error->message);
      g_error_free(error);
    } else {
      DBG_TRACE((CMP1,"[session %p] connect forwarded", cb_data->csession));
      conn_man_proxy_session_session_complete_connect((ConnManProxySessionSession *) cb_data->client_proxy, cb_data->invocation);
    }
    g_free(cb_data);
  }
}

gboolean
connect_msg_handler(ConnManProxySessionSession * object,
                    GDBusMethodInvocation * invocation, gpointer * user_data)
{
  struct dbus_cb_parameters * cb_data = NULL;
  struct client_session *csession = (struct client_session *) user_data;
  INF_TRACE((CMP1,"[session %p] connect received", csession));

  if (!check_caller_credentials(invocation, csession)) {
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_CREDENTIALS_MISMATCH,
                                          "%s",
                                          "credentials mismatch/error for session");
    return FALSE;
  }
  cb_data = dbus_cb_param_new((GObject *) object, invocation, csession, NULL);
  conn_man_proxy_session_session_call_connect(csession->session_client,
                                                NULL,
                                                connect_msg_handler_cb,
                                                cb_data);

    return TRUE;
  }

static void destroy_msg_handler_cb(GObject *object, GAsyncResult *res, void *user_data){
  GError *error = NULL;
  struct dbus_cb_parameters * cb_data = (struct dbus_cb_parameters*) user_data;
  if (error) {
    ERR_TRACE
      ((CMP1,"[session %p] error forwarding destroy. error code: %d error message: %s",
        cb_data->csession, error->code, error->message));
    g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                          error->code, "%s", error->message);
    g_error_free(error);
    close_session(cb_data->csession);
    return;
  }
  conn_man_proxy_session_session_complete_destroy((ConnManProxySessionSession *) cb_data->client_proxy, cb_data->invocation);
  DBG_TRACE((CMP1,"[session %p] destroy forwarded", cb_data->csession));
  return ;
}


gboolean
destroy_msg_handler(ConnManProxySessionSession * object,
                    GDBusMethodInvocation * invocation, gpointer * user_data)
{
  GError *error = NULL;
  struct client_session *csession = (struct client_session *) user_data;
  struct dbus_cb_parameters *cb_data= dbus_cb_param_new((GObject*) object, invocation,csession, NULL);
  INF_TRACE((CMP1,"[session %p] destroy received", csession));

  if (!check_caller_credentials(invocation, csession)) {
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_CREDENTIALS_MISMATCH,
                                          "%s",
                                          "credentials mismatch for session");
    return TRUE;
  }

  conn_man_proxy_session_session_call_destroy(csession->session_client,
                                              NULL,
                                              destroy_msg_handler_cb,
                                              cb_data);
  return TRUE;
}

gboolean
disconnect_msg_handler(ConnManProxySessionSession * object,
                       GDBusMethodInvocation * invocation, gpointer * user_data)
{
  GError *error = NULL;
  gboolean ret;
  struct client_session *csession = (struct client_session *) user_data;

  INF_TRACE((CMP1,"[session %p] disconnect received", csession));

  if (!check_caller_credentials(invocation, csession)) {
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_CREDENTIALS_MISMATCH,
                                          "%s",
                                          "credentials mismatch/error for session");
    return TRUE;
  }

  ret =
    conn_man_proxy_session_session_call_disconnect_sync(csession->session_client, NULL, &error);

  if (error) {
    ERR_TRACE
      ((CMP1,"[session %p] error forwarding disconnect. error code: %d error message: %s",
	csession, error->code, error->message));
    g_dbus_method_invocation_return_error(invocation, error->domain,
                                          error->code, "%s", error->message);
    g_error_free(error);
    return TRUE;
  }

  conn_man_proxy_session_session_complete_disconnect(object, invocation);
  DBG_TRACE((CMP1,"[session %p] disconnect forwarded", csession));
  return ret;
}


gboolean
destroy_session_msg_handler(ConnManProxyManagerManager * object,
                            GDBusMethodInvocation * invocation,
                            const gchar * arg_session)
{
  GError *error = NULL;
  guint uid;
  GSList *iter = NULL;
  guint found_sessions = 0;
  guint it = 0;

  INF_TRACE((CMP1,"[invocation %p] destroy session request received from sender %s",
       invocation, g_dbus_method_invocation_get_sender(invocation)));

  dbus_dbus_call_get_connection_unix_user_sync(dbusdaemon,
                                               g_dbus_method_invocation_get_sender
                                               (invocation), &uid, NULL,
                                               &error);

  if (error) {
    DBG_TRACE((CMP1,"[invocation %p] error getting user information", invocation));
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_USER_RESOLUTION_FAILED,
                                          "User resolution failed for %s!",
                                          g_dbus_method_invocation_get_sender
                                          (invocation));
    return TRUE;
  }

  for (iter = client_sessions; iter != NULL; iter = g_slist_next(iter)) {
    struct client_session *ctmp;
    ctmp = iter->data;
    if ((ctmp->uid == uid)
        && (g_strcmp0(ctmp->session, arg_session) == 0)) {
      DBG_TRACE((CMP1,"[invocation %p] found session %p", invocation, iter->data));
      found_sessions++;
      free_client_session(ctmp);
      iter->data = NULL;
    }
  }
  for (it = 0; it < found_sessions; it++) {
    client_sessions = g_slist_remove(client_sessions, NULL);
  }

  if (found_sessions == 0) {
    ERR_TRACE((CMP1,"[invocation %p] error getting session information for sender %s",
	 invocation, g_dbus_method_invocation_get_sender(invocation)));
    g_dbus_method_invocation_return_error(invocation, CONNMANPROXY_ERROR,
                                          CONNMANPROXY_SESSION_RESOLUTION_FAILED,
                                          "session resolution failed for %s!",
                                          g_dbus_method_invocation_get_sender
                                          (invocation));
    return TRUE;
  }

  conn_man_proxy_manager_manager_complete_destroy_session(object, invocation);
  DBG_TRACE((CMP1,"[invocation %p] destroy session request finished", invocation));
  return TRUE;
}

gboolean
services_changed_msg_handler(ConnManProxyManagerManager * object,
                             GVariant * arg_changed,
                             const gchar * const *arg_removed)
{
  INF_TRACE((CMP1,"services changed message received"));

  conn_man_proxy_manager_manager_emit_services_changed(manager,
                                                       arg_changed,
                                                       arg_removed);

  return TRUE;
}

GVariant* transform_contextidentifier(struct client_session *csession, GVariant *arg_settings, const gchar *username){

  gchar *service = NULL;
  gchar *service_final = NULL;
  GVariant *item = NULL;
  GVariantIter *iter = NULL;
  GVariantBuilder *builder = NULL;
  gboolean service_tag_found = FALSE;
  GVariant *pass_on_settings = NULL;

  service = g_strconcat(username, ":", NULL);
  DBG_TRACE((CMP1, "[session %p] Service id: %s", csession, service));
  iter = g_variant_iter_new(arg_settings);
  builder = g_variant_builder_new(g_variant_get_type(arg_settings));

  while ((item = g_variant_iter_next_value(iter))) {
    GVariantIter *iter2 = g_variant_iter_new(item);
    GVariant *key, *value, *value_variant;
    const gchar *key_string;

    key = g_variant_iter_next_value(iter2);
    value = g_variant_iter_next_value(iter2);

    key_string = g_variant_get_string(key, NULL);
    value_variant = g_variant_get_variant(value);

    if ( ((g_strcmp0(key_string, "Service") == 0) || (g_strcmp0(key_string, "ContextIdentifier") == 0))
        &&
        (g_variant_type_is_subtype_of
         (g_variant_get_type(value_variant), G_VARIANT_TYPE_STRING))) {
      GVariant *var_s, *var;
      const gchar *service_original = NULL;
      service_original = g_variant_get_string(value_variant, NULL);
	  DBG_TRACE((CMP1, "[session %p] Service orginal id: \"%s\"", csession, service_original));
      service_final = g_strconcat(service, service_original, NULL);

      var_s =
        g_variant_new(g_variant_get_type_string(value_variant), service_final);
      var = g_variant_new(g_variant_get_type_string(item), &CONTEXT_IDENTIFIER , var_s);
      g_variant_builder_add_value(builder, var);
	  service_tag_found = TRUE;
	  INF_TRACE((CMP1,"[session %p] Service id: %s", csession, service_final));
    } else {
      g_variant_builder_add_value(builder, item);
    }
	g_variant_unref(item);
	g_variant_unref(key);
	g_variant_unref(value);
	g_variant_unref(value_variant);
	g_variant_iter_free(iter2);
  }

	if (service_tag_found == FALSE){
	  GVariant *var_s, *var;
      INF_TRACE((CMP1, "[session %p] No Service / ContextIdentifier found. Final service %s", csession, service));
      var_s = g_variant_new("s", service);
      var = g_variant_new("{sv}", &CONTEXT_IDENTIFIER , var_s);
      g_variant_builder_add_value(builder, var);

	}
	DBG_TRACE((CMP1,"ContextID: %s", &CONTEXT_IDENTIFIER));
  pass_on_settings = g_variant_builder_end(builder);
  g_free(service);
  g_free(service_final);
  g_variant_iter_free(iter);
  g_variant_builder_unref(builder);
  return pass_on_settings;

}


static void create_session_msg_handler_cb(GObject * object,
                                   GAsyncResult * res,
                                   void * user_data){
  gboolean success = FALSE;
  GError *error = NULL;
  struct dbus_cb_parameters * cb_data = NULL;
  gchar *session = NULL;
  ConnManProxySessionSession *session_server = NULL;
  cb_data = (struct dbus_cb_parameters *) user_data;
  struct client_session * csession = cb_data->csession;
  DBG_TRACE((CMP1,"[session %p] create_session_msg_handler_cb", csession));
  success = conn_man_proxy_manager_manager_call_create_session_finish (
                                                                       (ConnManProxyManagerManager *) object,
                                                                       &session,
                                                                       res,
                                                                       &error);
  if(!success){
    if (error) {
      ERR_TRACE
        ((CMP1,"[session %p] error calling create session on connman error code: %d error message: %s",
          csession, error->code, error->message));
      g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                            error->code, "%s", error->message);
      g_error_free(error);
    } else {
      ERR_TRACE
        ((CMP1,"[session %p] error calling create session on connman error code: %d error message: unknown",
          csession));
    }
    goto err;
  }

  csession->session = g_strdup(session);

  csession->watchid = g_bus_watch_name_on_connection(session_connection,
                                                              csession->sender,
                                                              G_BUS_NAME_WATCHER_FLAGS_NONE,
                                                              NULL,
                                                              client_vanished_handler,
                                                              csession, NULL);

  session_server = conn_man_proxy_session_session_skeleton_new();
  csession->session_server = session_server;
  g_signal_connect(session_server,
                   "handle-destroy", G_CALLBACK(destroy_msg_handler), csession);
  g_signal_connect(session_server,
                   "handle-connect", G_CALLBACK(connect_msg_handler), csession);
  g_signal_connect(session_server,
                   "handle-disconnect",
                   G_CALLBACK(disconnect_msg_handler), csession);
  g_signal_connect(session_server,
                   "handle-change", G_CALLBACK(change_msg_handler), csession);
  if (!g_dbus_interface_skeleton_export
      (G_DBUS_INTERFACE_SKELETON(session_server), session_connection,
       csession->session, &error)) {
    ERR_TRACE
      ((CMP1,"[session %p] error setting up session interface error code: %d error message: %s",
        csession, error->code, error->message));
    g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                          error->code, "%s", error->message);
    g_error_free(error);
    goto err;
  }
  csession->session_client =
    conn_man_proxy_session_session_proxy_new_sync(system_connection,
                                                  G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES
                                                  |
                                                  G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS,
                                                  CONNMAN_BUS_NAME,
                                                  csession->session, NULL,
                                                  &error);
  g_dbus_proxy_set_default_timeout((GDBusProxy*) csession->session_client, CONNMANPROXY_SYSTEMBUS_TIMEOUT);

  if (error) {
    ERR_TRACE
      ((CMP1,"[session %p] error connecting session interface error code: %d error message: %s",
        csession, error->code, error->message));
    g_dbus_method_invocation_return_error(cb_data->invocation, error->domain,
                                          error->code, "%s", error->message);
    g_error_free(error);
    goto err;
  }
  conn_man_proxy_manager_manager_complete_create_session((ConnManProxyManagerManager *) cb_data->client_proxy, cb_data->invocation,
                                                         csession->session);
  client_sessions = g_slist_prepend(client_sessions, csession);
  INF_TRACE((CMP1,"[session %p] finished creation of session", csession));
  goto done;

 err:
  if (csession) {
    free_client_session(csession);
  }
 done:
  g_free(session);
  g_free(cb_data);
}


gboolean create_session_msg_handler(ConnManProxyManagerManager * object,
                           GDBusMethodInvocation * invocation,
                           GVariant * arg_settings,
                           const gchar * arg_notifier, gpointer * user_data)
{
  struct client_session *csession = NULL;
  struct dbus_cb_parameters *cbparam = NULL;
  gchar *username = NULL;
  guint uid;
  GError *error = NULL;
  struct passwd *pwd = NULL;
  gchar *new_notifier = NULL;
  gchar *notifier_rand_part = NULL;
  GVariant *pass_on_settings = NULL;
  ConnManProxyNotificationNotification *notification_server = NULL;

  gchar *sender = NULL;

  if (invocation == NULL)
  {
	INF_TRACE((CMP1,"[invocation] failed to get sender for invocation. propably invocator has vanished "));
	goto err;

  }
  sender = g_strdup(g_dbus_method_invocation_get_sender(invocation));
  if (sender == NULL) {
    INF_TRACE((CMP1,"[invocation] failed to get sender for invocation. propably invocator has vanished "));
    goto err;
  }
  INF_TRACE((CMP1,"[invocation %p] create session request received from sender %s",
       invocation, sender));
  if (arg_notifier == NULL){
	  INF_TRACE((CMP1,"[invocation %p] notifier is NULL", invocation));
	  goto err;
  }
  INF_TRACE((CMP1,"[invocation %p] notifier path %s", invocation ,arg_notifier));
  if (!g_str_has_prefix(arg_notifier,"/")){
	  INF_TRACE((CMP1,"[invocation %p] notifier is not dbus path %s", invocation, arg_notifier));
	  goto err;
  }
  csession = g_new0(struct client_session, 1);

  csession->sender = sender;
  csession->notifier = g_strdup(arg_notifier);
  csession->connman_initiated_closing = FALSE;


  dbus_dbus_call_get_connection_unix_user_sync(dbusdaemon,
                                               csession->sender,
                                               &uid, NULL, &error);

  if (error) {
    ERR_TRACE((CMP1,"[invocation %p] error getting user information", invocation));
    g_dbus_method_invocation_return_error(invocation,
                                          CONNMANPROXY_ERROR,
                                          CONNMANPROXY_USER_RESOLUTION_FAILED,
                                          "User resolution failed for %s!",
                                          csession->sender);
    goto err;
  }
  csession->uid = uid;

  pwd = getpwuid(uid);
  username = g_strdup(pwd->pw_name);
  pass_on_settings = transform_contextidentifier(csession, arg_settings, username);
  notifier_rand_part = get_random_string(8);
  new_notifier = g_strconcat("/agw/",
                             pwd->pw_name, "/", notifier_rand_part, NULL);

  DBG_TRACE
    ((CMP1,"[session %p] Notification interface object path requested by client: %s",
      csession, arg_notifier));

  csession->new_notifier = g_strdup(new_notifier);
  DBG_TRACE
    ((CMP1,"[session %p] Notification interface exported object path: %s",
      csession, new_notifier));
  notification_server = conn_man_proxy_notification_notification_skeleton_new();
  csession->notification_server = notification_server;

  g_signal_connect(notification_server,
                   "handle-release", G_CALLBACK(release_msg_handler), csession);
  g_signal_connect(notification_server,
                   "handle-update", G_CALLBACK(update_msg_handler), csession);

  if (!g_dbus_interface_skeleton_export
      (G_DBUS_INTERFACE_SKELETON(notification_server), system_connection,
       csession->new_notifier, &error)) {
    ERR_TRACE
      ((CMP1,"[session %p] error setting up notification interface error code: %d error message: %s",
	csession, error->code, error->message));
    g_dbus_method_invocation_return_error(invocation, error->domain,
                                          error->code, "%s", error->message);
    g_error_free(error);
    goto err;
  }
  cbparam = dbus_cb_param_new((GObject*) object, invocation, csession, NULL);
 conn_man_proxy_manager_manager_call_create_session (connman,
                                                     pass_on_settings,
                                                     csession->new_notifier,
                                                     NULL,
                                                     create_session_msg_handler_cb,
                                                     cbparam);
 goto done;
err:
  if (csession) {
    free_client_session(csession);
  }
done:
  g_free(username);
  g_free(notifier_rand_part);
  g_free(new_notifier);

  return TRUE;
}

static void
name_acquired_handler(GDBusConnection * connection,
                      const gchar * name, gpointer user_data)
{
  DBG_TRACE((CMP1,"name acquired"));
}

static void
name_lost_handler(GDBusConnection * connection,
                  const gchar * name, gpointer user_data)
{
  INF_TRACE((CMP1,"lost connman server name on session bus (or never got it). quitting."));
  g_main_loop_quit(loop);
}

void
connman_appeared_handler(GDBusConnection * connection,
                         const gchar * name,
                         const gchar * name_owner, gpointer user_data)
{
  GError *error = NULL;
  INF_TRACE
    ((CMP1,"native connman is available. activating interface, request to own connman name on session bus"));
  connman_up = TRUE;

  manager = conn_man_proxy_manager_manager_skeleton_new();

  g_signal_connect(manager,
                   "handle-create-session",
                   G_CALLBACK(create_session_msg_handler), NULL);
  g_signal_connect(manager,
                   "handle-destroy-session",
                   G_CALLBACK(destroy_session_msg_handler), NULL);

  if (!g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(manager),
                                        session_connection, "/", &error)) {
    DBG_TRACE
      ((CMP1,"error setting up connman service at session bus. Error code: %d Error message: %s",
	error->code, error->message));
    g_main_loop_quit(loop);
    return;
  }

  owner_id = g_bus_own_name_on_connection(session_connection,
                                          CONNMANPROXY_BUS_NAME,
                                          G_BUS_NAME_OWNER_FLAGS_NONE,
                                          name_acquired_handler,
                                          name_lost_handler, NULL, NULL);
}

void close_all_client_sessions()
{
  GSList *iter;

  for (iter = client_sessions; iter != NULL; iter = g_slist_next(iter)) {
    struct client_session *csession;
    csession = iter->data;
    free_client_session(csession);
  }
  g_slist_free(client_sessions);
  client_sessions = NULL;
}

void
connman_vanished_handler(GDBusConnection * connection,
                         const gchar * name, gpointer user_data)
{
  INF_TRACE((CMP1,"connman vanished"));

  close_all_client_sessions();

  connman_up = FALSE;
  if (owner_id) {
    g_bus_unown_name(owner_id);
    owner_id = 0;
  }
  if (manager) {
    g_dbus_interface_skeleton_unexport_from_connection
      (G_DBUS_INTERFACE_SKELETON(manager), session_connection);
    g_object_unref(manager);
    manager = NULL;
  }
}

void signal_handler(int signal_no)
{
  DBG_TRACE((CMP1,"signal handler"));
  if (signal_no == SIGTERM) {
    DBG_TRACE((CMP1,"SIGTERM received"));
    if (loop)
      g_main_loop_quit(loop);
    shutdown_initiated = TRUE;
  }
}

int main(int argc, char *argv[]){
  DLT_REGISTER_APP(DLT_APPID_ISO_CONNMANPROXY, "ConnmanProxy registration for Logging");
  /* register context */
  DLT_REGISTER_CONTEXT(CMP1, "CMP1", "CONNMANPROXY context for Logging");
  INF_TRACE((CMP1,"Starting initialization of ConnManProxy"));
  GError *error = NULL;

  if (signal(SIGTERM, signal_handler) == SIG_ERR) {
    DBG_TRACE((CMP1,"Cannot register to SIGTERM signal.\n"));
    return 1;
  }
  //DBG_TRACE((CMP1,"Starting to read policies from file %s", POLICY_FILE));

  INF_TRACE((CMP1,"Connecting system bus"));
  system_connection = g_bus_get_sync(G_BUS_TYPE_SYSTEM, NULL, &error);
  if (error) {
    INF_TRACE((CMP1,"error occurred for connecting system bus. code: %d message: %s",
	 error->code, error->message));
    goto err;
  }

  INF_TRACE((CMP1,"Connecting session bus"));
  session_connection = g_bus_get_sync(G_BUS_TYPE_SESSION, NULL, &error);
  if (error) {
    INF_TRACE((CMP1,"error occurred for connecting session bus. code: %d message: %s",
	 error->code, error->message));
    goto err;
  }

  DBG_TRACE((CMP1,"Starting watch on connman well-known bus name"));
  connmanwatch_id = g_bus_watch_name_on_connection(system_connection,
                                                   "net.connman",
                                                   G_BUS_NAME_WATCHER_FLAGS_NONE,
                                                   connman_appeared_handler,
                                                   connman_vanished_handler,
                                                   NULL, NULL);

  DBG_TRACE((CMP1,"Setting up dbus daemon proxy"));
  dbusdaemon = dbus_dbus_proxy_new_sync(session_connection,
                                        G_DBUS_PROXY_FLAGS_DO_NOT_LOAD_PROPERTIES
                                        |
                                        G_DBUS_PROXY_FLAGS_DO_NOT_CONNECT_SIGNALS,
                                        "org.freedesktop.DBus",
                                        "/org/freedesktop/DBus", NULL, &error);

  if (error) {
    INF_TRACE((CMP1,"error occurred for connecting session bus. code: %d message: %s",
	 error->code, error->message));
    goto err;
  }

  DBG_TRACE((CMP1,"Setting up connman gdbus proxy"));
  connman = conn_man_proxy_manager_manager_proxy_new_sync(system_connection,
                                                          G_DBUS_PROXY_FLAGS_NONE,
                                                          CONNMAN_BUS_NAME,
                                                          "/", NULL, &error);
  g_dbus_proxy_set_default_timeout((GDBusProxy*) connman, CONNMANPROXY_SYSTEMBUS_TIMEOUT);
  if (error) {
    INF_TRACE
      ((CMP1,"error occurred setting up connman proxy on system bus. code: %d message: %s",
	error->code, error->message));
    goto err;
  }

  g_signal_connect(connman,
                   "services-changed",
                   G_CALLBACK(services_changed_msg_handler), NULL);


  if (!shutdown_initiated) {
    loop = g_main_loop_new(NULL, FALSE);
    g_main_loop_run(loop);
  }

  INF_TRACE((CMP1,"Shutting down regularly"));
  goto done;

err:
  INF_TRACE((CMP1,"Shutting down irregularly"));
  if (error)
    g_error_free(error);
done:

  close_all_client_sessions();

  if (connmanwatch_id)
    g_bus_unwatch_name(connmanwatch_id);

  if (owner_id)
    g_bus_unown_name(owner_id);


  g_object_unref(dbusdaemon);
  g_object_unref(connman);
  if (manager)
    g_dbus_interface_skeleton_unexport_from_connection
      (G_DBUS_INTERFACE_SKELETON(manager), session_connection);
  if (session_connection)
    g_dbus_connection_close_sync(session_connection, NULL, NULL);
  if (system_connection)
    g_dbus_connection_close_sync(system_connection, NULL, NULL);
  return 0;
}
