/******************************************************************
*FILE: guestproxy-main.c 
*SW-COMPONENT: ai_isoconnectivity
*DESCRIPTION: This file contains the main function of guestproxy
*COPYRIGHT: © 2018 Robert Bosch GmbH
*
*The reproduction, distribution and utilization of this file as
*well as the communication of its contents to others without express
*authorization is prohibited. Offenders will be held liable for the
*payment of damages. All rights reserved in the event of the grant
*of a patent, utility model or design.
******************************************************************/

//#include "config.h"

#include "guestproxy-bluez-client-handler.h"
#include "constants.h"
#include "logs-internal.h"
#include <getopt.h>
#include <glib/gstdio.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/signalfd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

static GMainLoop *main_loop = NULL;
static unsigned int terminated = 0;
static gboolean option_version = 0;
DLT_DECLARE_CONTEXT(guestproxy_context);

static gboolean
signal_handler (GIOChannel *channel,
                GIOCondition cond,
                gpointer data)
{
  struct signalfd_siginfo si;
  ssize_t result;
  int fd;

  if (cond & (G_IO_NVAL | G_IO_ERR | G_IO_HUP))
    return FALSE;

  fd = g_io_channel_unix_get_fd (channel);

  result = read (fd, &si, sizeof (struct signalfd_siginfo));
  if (result != sizeof (struct signalfd_siginfo))
    return FALSE;

  switch (si.ssi_signo)
    {
      case SIGINT:
      case SIGABRT:
      case SIGSEGV:
      case SIGTERM:
      if (terminated == 0)
       {
	 DLT_LOG(guestproxy_context,DLT_LOG_INFO, DLT_STRING("Terminating"));
         DLT_UNREGISTER_CONTEXT(guestproxy_context);
         DLT_UNREGISTER_APP();

         g_unlink (PEER_TO_PEER_UNIX_PATH);
         g_main_loop_quit (main_loop);
       }

      terminated = 1;
      break;
      default:
       DLT_LOG(guestproxy_context,DLT_LOG_INFO, DLT_STRING("Defualt signal handler"));
      break;
    }

  return TRUE;
}

static guint
setup_signalfd (void)
{
  GIOChannel *channel;
  guint source;
  sigset_t mask;
  int fd;

  sigemptyset (&mask);
  sigaddset (&mask, SIGINT);
  sigaddset (&mask, SIGTERM);
  sigaddset (&mask, SIGABRT);
  sigaddset (&mask, SIGSEGV);

  if (sigprocmask (SIG_BLOCK, &mask, NULL) < 0)
    {
      DLT_LOG(guestproxy_context,DLT_LOG_ERROR, DLT_STRING("Failed to set signal mask"));
      return 0;
    }

  fd = signalfd (-1, &mask, 0);
  if (fd < 0)
    {
      DLT_LOG(guestproxy_context,DLT_LOG_ERROR, DLT_STRING("Failed to create signal descriptor"));
      return 0;
    }
  channel = g_io_channel_unix_new (fd);

  g_io_channel_set_close_on_unref (channel, TRUE);
  g_io_channel_set_encoding (channel, NULL, NULL);
  g_io_channel_set_buffered (channel, FALSE);

  source = g_io_add_watch (channel,
                           G_IO_IN | G_IO_HUP | G_IO_ERR | G_IO_NVAL,
                           signal_handler, NULL);

  g_io_channel_unref (channel);

  return source;
}

static GOptionEntry options[] = {
  { "version", 'v', 0, G_OPTION_ARG_NONE, &option_version,
    "Show version information and exit", NULL },
  { NULL },
};

int
main (int argc,
      char *argv[])
{

  DLT_REGISTER_APP(DLT_APPID_IDC_GUESTPROXY,"DLT Logging in Guestproxy");

  DLT_REGISTER_CONTEXT(guestproxy_context,"GPCO","context for guestproxy");

  DLT_LOG(guestproxy_context,DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING("entered"));

  guint signal;
  GOptionContext *context;
  g_autoptr(GError) error = NULL;
  g_autoptr (GuestproxyBluezClientHandler) guestproxy_bluez_client_handler = NULL;
  g_unlink(PEER_TO_PEER_UNIX_PATH);

  context = g_option_context_new (NULL);
  g_option_context_add_main_entries (context, options, NULL);

  if (!g_option_context_parse (context, &argc, &argv, &error))
    {
      if (error)
        {
          DLT_LOG(guestproxy_context,DLT_LOG_ERROR, DLT_STRING(error->message));
          g_error_free (error);
        }
      else
        DLT_LOG(guestproxy_context,DLT_LOG_ERROR, DLT_STRING("An unknown error occurred"));
      exit (1);
    }
  g_option_context_free (context);

  if (option_version)
    {
      DLT_LOG(guestproxy_context,DLT_LOG_INFO, DLT_STRING("Version:"), DLT_STRING(VERSION)); 
      DLT_UNREGISTER_CONTEXT(guestproxy_context);
      DLT_UNREGISTER_APP();
      exit (0);
    }

  umask (0077);
  main_loop = g_main_loop_new (NULL, FALSE);
  signal = setup_signalfd ();

  guestproxy_bluez_client_handler = guestproxy_bluez_client_handler_get_object ();
  g_main_loop_run (main_loop);

  g_source_remove (signal);
 
  DLT_UNREGISTER_CONTEXT(guestproxy_context);
  DLT_UNREGISTER_APP();
  g_main_loop_unref (main_loop);
  return 0;
}
