/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/examples/osganimate/osganimate.cpp
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/examples/osgthirdpersonview/osgthirdpersonview.cpp
Copied from https://github.com/gwaldron/osgearth/blob/osgearth-2.1.1/src/osgEarth/HeightFieldUtils.cpp
Copied from https://github.com/xarray/osgRecipes/blob/master/cookbook/common/CommonFunctions.cpp
Copied from https://github.com/xarray/osgRecipes/blob/master/cookbook/chapter6/ch06_06/depth_of_field.cpp
List of changes:
1. Added function to create a blank (white) texture
*/

#include <osg/Geometry>
#include <osg/Geode>
#include <osg/MatrixTransform>
#include <osg/Texture2D>
#include <osg/PolygonMode>
#include <osg/ImageUtils>
#include "oss/Normals.h"
#include "oss/OSSUtils.h"

#define BLURTEXSIZE 1024

using namespace osg;

namespace mapengine {
namespace oss {

const double radiusPolar = 6356752.3142;

bool calcClusterCullingValues(const osg::Vec3d& vertex,
   const osg::Vec3d& center,
   const float verticalScale,
   float& min_dot_product,
   float& max_cluster_culling_height,
   float& max_cluster_culling_radius)
{
   osg::Vec3 dv = vertex - center;
   float d = sqrtf(dv.x()*dv.x() + dv.y()*dv.y() + dv.z()*dv.z());
   float theta = acosf(static_cast<float>(radiusPolar/ (radiusPolar + fabsf(verticalScale))) );
   float phi = 2.0F * asinf(static_cast<float>(d*0.5F/radiusPolar)); // d/globe_radius;
   float beta = theta+phi;
   float cutoff = osg::PI_2 - 0.1;

   if (phi<cutoff && beta<cutoff)
   {
      float local_dot_product = -sinf(theta + phi);
      float local_m = radiusPolar*(1.0/ cosf(theta+phi) - 1.0);
      float local_radius = static_cast<float>(radiusPolar * tanf(beta)); // beta*globe_radius;
      min_dot_product = osg::minimum(min_dot_product, local_dot_product);
      max_cluster_culling_height = osg::maximum(max_cluster_culling_height,local_m);      
      max_cluster_culling_radius = osg::maximum(max_cluster_culling_radius,local_radius);
   }
   else
   {
      // wrap around tile so better turn of callback
      return false;
   }

   return true;
}

osg::Node* createFrustum(osg::Camera* camera)
{
   if (!camera) return NULL;

   // projection and model view matrices
   osg::Matrixd proj = camera->getProjectionMatrix();
   osg::Matrixd mv = camera->getViewMatrix();

   // get near and far from the projection matrix
   const double nearPlane = proj(3, 2) / (proj(2, 2) - 1.0);
   const double farPlane = proj(3, 2) / (1.0 + proj(2, 2));

   // get the sides of the near plane
   const double nLeft = nearPlane * (proj(2, 0) - 1.0) / proj(0, 0);
   const double nRight = nearPlane * (1.0 + proj(2, 0)) / proj(0, 0);
   const double nTop = nearPlane * (1.0 + proj(2, 1)) / proj(1, 1);
   const double nBottom = nearPlane * (proj(2, 1) - 1.0) / proj(1, 1);

   // get the sides of the far plane
   const double fLeft = farPlane * (proj(2, 0) - 1.0) / proj(0, 0);
   const double fRight = farPlane * (1.0 + proj(2, 0)) / proj(0, 0);
   const double fTop = farPlane * (1.0 + proj(2, 1)) / proj(1, 1);
   const double fBottom = farPlane * (proj(2, 1) - 1.0) / proj(1, 1);

   osg::MatrixTransform* mt = 0;
   // our vertex array needs only 9 vertices (origin + 8 corners)
   osg::ref_ptr<osg::Vec3dArray> v = new osg::Vec3dArray(9); 
   if(v)
   {
      (*v)[0].set(0., 0., 0.);
      (*v)[1].set(nLeft, nBottom, -nearPlane);
      (*v)[2].set(nRight, nBottom, -nearPlane);
      (*v)[3].set(nRight, nTop, -nearPlane);
      (*v)[4].set(nLeft, nTop, -nearPlane);
      (*v)[5].set(fLeft, fBottom, -farPlane);
      (*v)[6].set(fRight, fBottom, -farPlane);
      (*v)[7].set(fRight, fTop, -farPlane);
      (*v)[8].set(fLeft, fTop, -farPlane);

      osg::ref_ptr<osg::Geometry> geom = new osg::Geometry;
      if(geom)
      {
         geom->setVertexArray(v);
         osg::ref_ptr<osg::Vec4Array> c = new osg::Vec4Array;
         if(c)
         {
            c->push_back(osg::Vec4(1.0F, 0.0F, 0.0F, 1.0F));
         }
         geom->setColorArray(c);
         geom->setColorBinding(osg::Geometry::BIND_OVERALL);

         GLushort idxLines[8] = { 0, 5, 0, 6, 0, 7, 0, 8 };
         GLushort idxLoops0[4] = { 1, 2, 3, 4 };
         GLushort idxLoops1[4] = { 5, 6, 7, 8 };
         geom->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::LINES, 8, idxLines));
         geom->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::LINE_LOOP, 4, idxLoops0));
         geom->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::LINE_LOOP, 4, idxLoops1));
         osg::ref_ptr<osg::Geode> geode = new osg::Geode;
         if(geode)
         {
            geode->addDrawable(geom);
            geode->getOrCreateStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF | osg::StateAttribute::PROTECTED);
            
            // create parent MatrixTransform to transform the
            // view volume by the inverse ModelView matrix
            mt = new osg::MatrixTransform;
            if(mt)
            {
               mt->setMatrix(osg::Matrixd::inverse(mv));
               mt->addChild(geode);
            }
         }
      }
   }

   return mt;
}

osg::AnimationPath* createAnimationPath(const osg::Vec3& center, float radius, double looptime)
{
   // set up the animation path 
   osg::AnimationPath* animationPath = new osg::AnimationPath;
   if (animationPath)
   {
      animationPath->setLoopMode(osg::AnimationPath::LOOP);

      unsigned int numSamples = 40;
      float yaw = 0.0f;
      float yaw_delta = 2.0f*osg::PI/((float)numSamples-1.0f);
      float roll = osg::inDegrees(30.0f);

      double animTime = 0.0f;
      double time_delta = looptime/(double)numSamples;
      for(unsigned int i=0; i<numSamples; ++i)
      {
         osg::Vec3 position(center+osg::Vec3(sinf(yaw)*radius,cosf(yaw)*radius,0.0f));
         osg::Quat rotation(osg::Quat(static_cast<float>(roll),osg::Vec3(0.0F,1.0F,0.0F))*osg::Quat(-(yaw+osg::inDegrees(90.0f)),osg::Vec3(0.0F,0.0F,1.0F)));

         animationPath->insert(animTime,osg::AnimationPath::ControlPoint(position, rotation));

         yaw += yaw_delta;
         animTime += time_delta;
      }
   }

   return animationPath;
}

void collectViewerStats(osgViewer::CompositeViewer& compositionViewer, std::ostream& stream)
{
   if (compositionViewer.getViewerStats())
   {
      typedef std::vector<osg::Stats*> StatsList;
      StatsList statsList;
      statsList.push_back(compositionViewer.getViewerStats());

      osgViewer::ViewerBase::Contexts contexts;
      compositionViewer.getContexts(contexts);
      for (osgViewer::ViewerBase::Contexts::iterator gcitr = contexts.begin(); gcitr != contexts.end(); ++gcitr)
      {
         osg::GraphicsContext::Cameras& cameras = (*gcitr)->getCameras();
         for (osg::GraphicsContext::Cameras::iterator itr = cameras.begin(); itr != cameras.end(); ++itr)
         {
            if ((*itr)->getStats())
            {
               statsList.push_back((*itr)->getStats());
            }
         }
      }

      for (StatsList::iterator itr = statsList.begin(); itr != statsList.end(); ++itr)
      {
         if (itr == statsList.begin()) (*itr)->report(stream, compositionViewer.getViewerStats()->getLatestFrameNumber() - 1);
         else (*itr)->report(stream, compositionViewer.getViewerStats()->getLatestFrameNumber() - 1, "    ");
      }
   }

   // Databasepager stats
   osgViewer::ViewerBase::Scenes scenes;
   compositionViewer.getScenes(scenes);
   for (osgViewer::ViewerBase::Scenes::iterator itr = scenes.begin(); itr != scenes.end(); ++itr)
   {
      osgViewer::Scene* scene = *itr;
      osgDB::DatabasePager* dp = scene->getDatabasePager();
      if (dp && dp->isRunning())
      {
         double value = dp->getAverageTimeToMergeTiles();
         stream << "DatabasePager time to merge new tiles - average: " << value << std::endl;
         value = dp->getMinimumTimeToMergeTile();
         stream << "min: " << value << std::endl;
         value = dp->getMaximumTimeToMergeTile();
         stream << "max: " << value << std::endl;
         value = dp->getFileRequestListSize();
         stream << "requests: " << value << std::endl;
         value = dp->getDataToCompileListSize();
         stream << "tocompile: " << value << std::endl;
      }
   }
}

osg::Camera* createRTTCamera(osg::Camera::BufferComponent buffer, osg::Texture* tex, bool isAbsolute, int renderOrderNum, int multisampleSamples)
{
   osg::ref_ptr<osg::Camera> camera = new osg::Camera;

   camera->setClearColor(osg::Vec4());
   camera->setClearMask(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
   camera->setRenderTargetImplementation(osg::Camera::FRAME_BUFFER_OBJECT);
   camera->setRenderOrder(osg::Camera::PRE_RENDER, renderOrderNum);
   if (tex)
   {
      camera->setViewport(0, 0, tex->getTextureWidth(), tex->getTextureHeight());
      camera->attach(buffer, tex,0, 0, false, multisampleSamples, 0);

      // TODO: support multisampling
      //camera->attach(buffer, tex, 0, 4);
   }

   if (isAbsolute)
   {
      camera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
      camera->setProjectionMatrix(osg::Matrix::ortho2D(0.0, 1.0, 0.0, 1.0));
      camera->setViewMatrix(osg::Matrix::identity());
      camera->addChild(createScreenQuad(osg::Vec3(), .0f, 1.0f));
   }

   return camera.release();
}

osg::Camera* createRTTCamera(osg::Camera::BufferComponent buffer, osg::Image* image, bool isAbsolute, int renderOrderNum, int multisampleSamples)
{
   osg::ref_ptr<osg::Camera> camera = new osg::Camera;

   camera->setClearColor(osg::Vec4());
   camera->setClearMask(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
   camera->setRenderTargetImplementation(osg::Camera::FRAME_BUFFER_OBJECT);
   camera->setRenderOrder(osg::Camera::PRE_RENDER, renderOrderNum);
   if (image)
   {
      //tex->setFilter(osg::Texture2D::MIN_FILTER, osg::Texture2D::LINEAR);
      //tex->setFilter(osg::Texture2D::MAG_FILTER, osg::Texture2D::LINEAR);
      camera->setViewport(0, 0, image->s(), image->t());
      camera->attach(buffer, image, multisampleSamples);

      // TODO: support multisampling
      //camera->attach(buffer, tex, 0, 4);
   }

   if (isAbsolute)
   {
      camera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
      camera->setProjectionMatrix(osg::Matrix::ortho2D(0.0, 1.0, 0.0, 1.0));
      camera->setViewMatrix(osg::Matrix::identity());
      camera->addChild(createScreenQuad(osg::Vec3(), .0f, 1.0f));
   }

   return camera.release();
}

osg::Geode* createScreenQuad(const osg::Vec3& corner, float width, float height, float scale)
{
   osg::Geometry* geom = osg::createTexturedQuadGeometry(corner, osg::Vec3(width,0.0f,0.0f), osg::Vec3(0.0f,height,0.0f), 0.0f, 0.0f, width*scale, height*scale);
   osg::ref_ptr<osg::Geode> quad = new osg::Geode;
   quad->addDrawable(geom);

   int values = osg::StateAttribute::OFF|osg::StateAttribute::PROTECTED;
   quad->getOrCreateStateSet()->setAttribute(new osg::PolygonMode(osg::PolygonMode::FRONT_AND_BACK, osg::PolygonMode::FILL), values);
   quad->getOrCreateStateSet()->setMode(GL_LIGHTING, values );

   return quad.release();
}

osg::Camera* createHUDCamera(double left, double right, double bottom, double top)
{
   osg::ref_ptr<osg::Camera> camera = new osg::Camera;

   camera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
   camera->setClearMask(GL_DEPTH_BUFFER_BIT);
   camera->setRenderOrder(osg::Camera::POST_RENDER);
   camera->setAllowEventFocus(false);
   camera->setProjectionMatrix(osg::Matrix::ortho2D(left, right, bottom, top));
   camera->getOrCreateStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

   return camera.release();
}

osg::Texture* createRenderTexture(int width, int height, bool depth)
{
   osg::Texture2D* texture2D = new osg::Texture2D;

   if (texture2D)
   {
      texture2D->setTextureSize(width, height);
      texture2D->setResizeNonPowerOfTwoHint(false);
      texture2D->setFilter(osg::Texture2D::MIN_FILTER,osg::Texture2D::LINEAR);
      texture2D->setFilter(osg::Texture2D::MAG_FILTER,osg::Texture2D::LINEAR);
      texture2D->setWrap(osg::Texture2D::WRAP_S,osg::Texture2D::CLAMP_TO_BORDER);
      texture2D->setWrap(osg::Texture2D::WRAP_T,osg::Texture2D::CLAMP_TO_BORDER);
      texture2D->setBorderColor(osg::Vec4(1.0f,1.0f,1.0f,1.0f));

      if (!depth)
      {
         texture2D->setInternalFormat(GL_RGBA16F_ARB);
         texture2D->setSourceFormat(GL_RGBA);
         texture2D->setSourceType(GL_FLOAT);
      }
      else
      {
         texture2D->setInternalFormat(GL_DEPTH_COMPONENT);
      }
   }
   return texture2D;
}

osg::Group* addNormals(osg::Group* group)
{
   if (!group) return NULL;

   const osg::BoundingSphere& bs = group->getBound();
   float scale = bs.radius() * 0.05f; // default is 5% of bounding-sphere radius
   group->addChild(new VertexNormals(group, scale));

   return group;
}

static const char* vertSource = {
   "void main(void)\n"
   "{\n"
   "   gl_Position = ftransform();\n"
   "   gl_TexCoord[0] = gl_MultiTexCoord0;\n"
   "}\n"
};

static const char* blurFragSource = {
   "uniform sampler2D inputTex;\n"
   "uniform vec2 blurDir;\n"
   "void main(void)\n"
   "{\n"
   "   vec2 uv = gl_TexCoord[0].st;\n"
   "   vec2 offset = vec2(0.004*blurDir.x, 0.003*blurDir.y);\n"
   "   vec4 color = texture2D(inputTex, uv) * 0.3;\n"
   "   color += texture2D(inputTex, uv - offset*3.0) * 0.05;\n"
   "   color += texture2D(inputTex, uv - offset*2.0) * 0.1;\n"
   "   color += texture2D(inputTex, uv - offset) * 0.2;\n"
   "   color += texture2D(inputTex, uv + offset) * 0.2;\n"
   "   color += texture2D(inputTex, uv + offset*2.0) * 0.1;\n"
   "   color += texture2D(inputTex, uv + offset*3.0) * 0.05;\n"
   "   gl_FragColor = color;\n"
   "}\n"
};

static const char* combineFragSource = {
   "uniform sampler2D sceneTex;\n"
   "uniform sampler2D blurTex;\n"
   "uniform sampler2D depthTex;\n"
   "uniform float focalDistance;\n"
   "uniform float focalRange;\n"

   "float getBlurFromLinearDepth(vec2 uv)\n"
   "{\n"
   "   float z = texture2D(depthTex, uv).x;\n"
   "   float zCenter = texture2D(depthTex, vec2(0.5,0.5)).x;\n"
   "   //z = 2.0 * 10001.0 / (10001.0 - z * 9999.0) - 1.0;\n"  // Considering the default znear/zfar
   "   //return clamp((z - focalDistance)/focalRange, 0.0, 1.0);\n"
   "   //return clamp((1.0 - zCenter) * z, 0.0, 1.0);\n"
   "   return uv.y;\n"
   "}\n"

   "void main(void)\n"
   "{\n"
   "   vec2 uv = gl_TexCoord[0].st;\n"
   "   vec4 fullColor = texture2D(sceneTex, uv);\n"
   "   vec4 blurColor = texture2D(blurTex, uv);\n"
   "   float blurValue = getBlurFromLinearDepth(uv);\n"
   "   gl_FragColor = fullColor + blurValue * (blurColor - fullColor);\n"
   "}\n"
};

osg::Node* addDOF(osg::Node* scene)
{
   // The first pass: color
   RTTPair pass0_color = createColorInput(scene);

   // The first pass: depth
   RTTPair pass0_depth = createDepthInput(scene);

   // The horizonal blur pass
   RTTPair pass1 = createBlurPass(pass0_color.second, osg::Vec2(1.0f, 0.0f));

   // The vertical blur pass
   RTTPair pass2 = createBlurPass(pass1.second, osg::Vec2(0.0f, 1.0f));

   // The final pass
   osg::ref_ptr<osg::Camera> hudCamera = createHUDCamera(0.0, 1.0, 0.0, 1.0);
   hudCamera->addChild(createScreenQuad(osg::Vec3(), 1.0f, 1.0f));

   osg::ref_ptr<osg::Program> finalProg = new osg::Program;
   finalProg->addShader(new osg::Shader(osg::Shader::VERTEX, vertSource));
   finalProg->addShader(new osg::Shader(osg::Shader::FRAGMENT, combineFragSource));

   osg::StateSet* stateset = hudCamera->getOrCreateStateSet();
   stateset->setTextureAttributeAndModes(0, pass0_color.second, osg::StateAttribute::OFF);
   stateset->setTextureAttributeAndModes(1, pass2.second, osg::StateAttribute::OFF);
   stateset->setTextureAttributeAndModes(2, pass0_depth.second, osg::StateAttribute::OFF);
   stateset->setAttributeAndModes(finalProg.get());
   stateset->addUniform(new osg::Uniform("sceneTex", 0));
   stateset->addUniform(new osg::Uniform("blurTex", 1));
   stateset->addUniform(new osg::Uniform("depthTex", 2));
   stateset->addUniform(new osg::Uniform("focalDistance", 100.0f));
   stateset->addUniform(new osg::Uniform("focalRange", 200.0f));

   // Build the scene graph
   osg::ref_ptr<osg::Group> root = new osg::Group;
   if (root.valid())
   {
      root->addChild(pass0_color.first);
      root->addChild(pass0_depth.first);
      root->addChild(pass1.first);
      root->addChild(pass2.first);
      root->addChild(hudCamera.get());
   }

   return root;
}

osg::Geometry* createEllipsoidGeometry(const osg::EllipsoidModel* ellipsoid, double outerRadius, unsigned int numLatGroups = 1, unsigned int numOfLongGroups = 1, bool genTexCoords = false )
{
   double hae = outerRadius - ellipsoid->getRadiusEquator();

   unsigned int latSegments = 50;
   unsigned lonSegments = 2 * latSegments;

   // no of geometries = nLatSegmentGroup * nLongSegmentGroup
   unsigned int nLatSegmentGroup = numLatGroups;
   unsigned int nLongSegmentGroup = numOfLongGroups;

   unsigned int nLatSubSegments = latSegments/nLatSegmentGroup;
   unsigned int nLastLatSubSeg = latSegments - (nLatSubSegments * (nLatSegmentGroup-1) ); //let the last segment have the remaining segments

   unsigned int nLongSubSegments = lonSegments/nLongSegmentGroup;
   unsigned int nLastLongSubSeg = lonSegments - (nLongSubSegments * (nLongSegmentGroup-1) ); //let the last segment have the remaining segments

   double segmentSize = 180.0/(double)latSegments; // degrees

   double latValue = -90.0;
   unsigned int prevLatSize = 0;
   unsigned int prevLongSize = 0;
   osg::Geometry* geom = new osg::Geometry();
   for(unsigned int i = 0; i < nLatSegmentGroup; ++i)
   {
      unsigned int latSize;
      if(nLatSegmentGroup > 1) { latSize = (i != nLatSegmentGroup - 1) ? ( (i != 0) ? nLatSubSegments + 1 : nLatSubSegments) : nLastLatSubSeg + 2; }
      else { latSize = latSegments + 1; }
      double longValue = -180.0;
      for(unsigned int j = 0; j < nLongSegmentGroup; ++j)
      {
         unsigned int longSize;

         if(nLongSegmentGroup > 1) { longSize = (j != nLongSegmentGroup - 1) ? ( (j != 0) ? nLongSubSegments + 1 : nLongSubSegments) : nLongSubSegments + 2; }
         else { longSize = lonSegments + 1; }

         geom->setUseVertexBufferObjects(true);

         unsigned int arraySize = latSize * longSize; 

         osg::Vec3Array* verts = new osg::Vec3Array();
         verts->reserve(arraySize);

         osg::Vec2Array* texCoords = 0;
         osg::Vec3Array* normals = 0;
         if (genTexCoords)
         {
            texCoords = new osg::Vec2Array();
            texCoords->reserve(arraySize);
            geom->setTexCoordArray( 0, texCoords );

            normals = new osg::Vec3Array();
            normals->reserve(arraySize);
            geom->setNormalArray( normals );
            geom->setNormalBinding(osg::Geometry::BIND_PER_VERTEX );
         }
         osg::DrawElementsUShort* el = new osg::DrawElementsUShort( GL_TRIANGLES );
         el->reserve( (latSize-1) * (longSize-1) * 6 );
         for( unsigned int y = 0; y < latSize; ++y )
         {  
            // save the segment size
            double lat = latValue + segmentSize * (double)y;
            for( unsigned x = 0; x < longSize; ++x )
            {
               double lon = longValue + segmentSize * (double)x;
               double gx, gy, gz;
               ellipsoid->convertLatLongHeightToXYZ( osg::DegreesToRadians(lat), osg::DegreesToRadians(lon), hae, gx, gy, gz );
               verts->push_back( osg::Vec3(gx, gy, gz) );

               if (genTexCoords)
               {
                  double s = static_cast<double>(x + prevLongSize) / static_cast<double>(lonSegments);
                  double t = static_cast<double>(y + prevLatSize) / static_cast<double>(latSegments);
                  texCoords->push_back( osg::Vec2(s, t) );
               }

               if (normals)
               {
                  osg::Vec3 normal( gx, gy, gz);
                  normal.normalize();
                  normals->push_back( normal );
               }

               if ( (y < latSize-1) && (x < longSize-1) )
               {
                  int x_plus_1 = x+1; //x < lonSegments-1 ? x+1 : 0;
                  int y_plus_1 = y+1;
                  el->push_back( y*(longSize) + x );
                  el->push_back( y_plus_1*(longSize) + x );
                  el->push_back( y*(longSize) + x_plus_1 );
                  el->push_back( y*(longSize) + x_plus_1 );
                  el->push_back( y_plus_1*(longSize) + x );
                  el->push_back( y_plus_1*(longSize) + x_plus_1 );
               }
            }
         }

         geom->setVertexArray( verts );
         geom->addPrimitiveSet( el );

         longValue += segmentSize * (double)(longSize - 1);
         prevLongSize += longSize - 1;
      }
      latValue += segmentSize * (double)(latSize - 1);
      prevLatSize += latSize - 1;
   }

   return geom;
}

RTTPair createColorInput(osg::Node* scene)
{
   osg::ref_ptr<osg::Texture2D> tex2D = new osg::Texture2D;
   tex2D->setTextureSize(BLURTEXSIZE, BLURTEXSIZE);
   tex2D->setInternalFormat(GL_RGBA);

   osg::ref_ptr<osg::Camera> camera = createRTTCamera(osg::Camera::COLOR_BUFFER, tex2D.get());
   camera->addChild(scene);

   return RTTPair(camera.release(), tex2D.get());
}

RTTPair createDepthInput(osg::Node* scene)
{
   osg::ref_ptr<osg::Texture2D> tex2D = new osg::Texture2D;
   tex2D->setTextureSize(BLURTEXSIZE, BLURTEXSIZE);
   tex2D->setInternalFormat(GL_DEPTH_COMPONENT24);
   tex2D->setSourceFormat(GL_DEPTH_COMPONENT);
   tex2D->setSourceType(GL_FLOAT);

   osg::ref_ptr<osg::Camera> camera = createRTTCamera(osg::Camera::DEPTH_BUFFER, tex2D.get());
   camera->addChild(scene);

   return RTTPair(camera.release(), tex2D.get());
}

RTTPair createBlurPass(osg::Texture* inputTex, const osg::Vec2& dir)
{
   osg::ref_ptr<osg::Texture2D> tex2D = new osg::Texture2D;
   tex2D->setTextureSize(BLURTEXSIZE, BLURTEXSIZE);
   tex2D->setInternalFormat(GL_RGBA);
   osg::ref_ptr<osg::Camera> camera = createRTTCamera(osg::Camera::COLOR_BUFFER, tex2D.get(), true);

   osg::ref_ptr<osg::Program> blurProg = new osg::Program;
   blurProg->addShader(new osg::Shader(osg::Shader::VERTEX, vertSource));
   blurProg->addShader(new osg::Shader(osg::Shader::FRAGMENT, blurFragSource));

   osg::StateSet* ss = camera->getOrCreateStateSet();
   ss->setTextureAttributeAndModes(0, inputTex, osg::StateAttribute::OFF);
   ss->setAttributeAndModes(blurProg.get(), osg::StateAttribute::ON|osg::StateAttribute::OVERRIDE);
   ss->addUniform(new osg::Uniform("sceneTex", 0));
   ss->addUniform(new osg::Uniform("blurDir", dir));

   osg::ref_ptr<osg::Geode> blurGeode = createScreenQuad(Vec3(), 1.0, 1.0);
   camera->addChild(blurGeode);

   return RTTPair(camera.release(), tex2D.get());
}

osg::Texture2D* createBlankTexture(unsigned int textureWidth, unsigned int textureHeight, osg::Vec4 color)
{
   osg::ref_ptr<osg::Texture2D> texture;
   osg::ref_ptr<osg::Image> image = new osg::Image;
   if (image.valid())
   {
      image->allocateImage(textureWidth, textureHeight, 1, GL_RGB, GL_UNSIGNED_BYTE, 1);
      clearImageToColor(image, color);
      texture = new osg::Texture2D(image);
      if (texture.valid())
      {
         texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
         texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);
      }
   }

   return texture.release();
}

}
}
