/*!
*******************************************************************************
* \file              spi_tclWBLClientHandler.cpp
* \brief             WBL Client handler class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Bluetooth Settings Client handler class
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                           | Modifications
 17.11.2016 |  Unmukt Jain (RBEI/ECP2)          | Initial Version
\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include "spi_tclWBLClientHandler.h"
#include <regex>

#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_WIFI
      #include "trcGenProj/Header/spi_tclWBLClientHandler.cpp.trc.h"
   #endif
#endif

#define WBL_SERVICE_NAME "org.bosch.Wbl"
#define CONFLICT_MANAGEMENT_OBJECT_PATH "/org/bosch/wbl/conflictmanagement"
#define CONFLICT_MANAGEMENT_INTERFACE_NAME "org.bosch.wbl.conflictmanagement"
static const char scoWifisetupObjectPath[] = "/org/bosch/wbl/wlan1/wifisetup/AP";
static const char scoWifisetupInterfaceName[] = "org.bosch.wbl.wifisetup";
static const t_U32 scou32DBUSTimeout = 121000;

static const t_String scoszKeySSID                    = "SSID";
static const t_String scoszKeyPassphrase              = "Passphrase";
static const t_String scoszKeyInterface               = "Interface";
static const t_String scoszKeyMode                    = "Mode";
static const t_String scoszKeyType                    = "Type";
static const t_String scoszKeyCurrentOperatingChannel = "CurrentOperatingChannel";
static const t_String scoszKeyFrequency               = "Frequency";
static const t_String scoszKeySecurity                = "Security";
static const t_String scoszKeyVisible                 = "Visible";
static const t_String scoszKeyPowered                 = "PowerState";
static const t_String scoszStationInfo                = "AssociatedStation";

static const gchar scActivateRestrictions[] = "ACTIVATE";
static const gchar scDeActivateRestritcions[] = "DEACTIVATE";

#define RELEASE_MAP_VALUE(m,key,type) \
	delete static_cast<type *>(m[key]); \
	m[key] = OSAL_NULL;

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::spi_tclWBLClientHandler()
 ***************************************************************************/
spi_tclWBLClientHandler::spi_tclWBLClientHandler(): m_poMainLoop(NULL),
         m_poOrgBoschWblConflictmanagementProxy(NULL),
         m_poConflictManagementPropertiesProxy(NULL),
		 m_poWifisetupProxy(NULL)
{
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::~spi_tclWBLClientHandler()
 ***************************************************************************/
spi_tclWBLClientHandler::~spi_tclWBLClientHandler()
{
   //! Stop the Main loop
   g_main_loop_quit(m_poMainLoop);

   //! Release the memory of m_poMainLoop.
   g_main_loop_unref(m_poMainLoop);
   m_poMainLoop = NULL;
   m_poOrgBoschWblConflictmanagementProxy = NULL;
   m_poConflictManagementPropertiesProxy = NULL;
   m_poWifisetupProxy = NULL;
   m_vecStationsInfo.clear();
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vExecute()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vExecute()
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vExecute entered"));
   m_poMainLoop = g_main_loop_new(OSAL_NULL, FALSE);
   SPI_NORMAL_ASSERT(NULL == m_poMainLoop)
   guint watcher_id = g_bus_watch_name ( G_BUS_TYPE_SYSTEM ,           // dbus type: system or session
            WBL_SERVICE_NAME,				 // dbus service name to watch
            G_BUS_NAME_WATCHER_FLAGS_NONE, // client, NO flag set
            vDbusNameAppearedCb,           // dbus_name_appeared cb
            vDbusNameVanishedCb,           // dbus_name_vanished cb
            (void*)this,                   // user data to be passed
            NULL);                         // user data free func
   if(0 == watcher_id)
   {
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vExecute g_bus_watch_name() failed!"));
   }
   if(OSAL_NULL != m_poMainLoop)
   {
      g_main_loop_run(m_poMainLoop);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vDbusNameAppearedCb()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vDbusNameAppearedCb(GDBusConnection *connection, const gchar *name,
		                                           const gchar *name_owner, gpointer pUserData)
{
    ETG_TRACE_USR1(("spi_tclWBLClientHandler::vDbusNameAppearedCb entered"));
    SPI_INTENTIONALLY_UNUSED(name_owner)
    org_bosch_wbl_conflictmanagement_proxy_new(connection,
                                               G_DBUS_PROXY_FLAGS_NONE,
                                               name,
                                               CONFLICT_MANAGEMENT_OBJECT_PATH,
                                               NULL,
                                               vConfilctManagementProxyCb,
                                               pUserData);

    org_freedesktop_dbus_properties_proxy_new( connection,
                                               G_DBUS_PROXY_FLAGS_NONE,
                                               name,
                                               CONFLICT_MANAGEMENT_OBJECT_PATH,
                                               NULL,
                                               vConfilctManagementPropertiesProxyCb,
                                               pUserData);

    org_bosch_wbl_wifisetup_proxy_new(connection,G_DBUS_PROXY_FLAGS_NONE,name,scoWifisetupObjectPath,NULL,
    		vWifisetupProxyCb,pUserData);
			
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vDbusNameVanishedCb()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vDbusNameVanishedCb(GDBusConnection *connection, const gchar *name, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vDbusNameVanishedCb entered"));
   SPI_INTENTIONALLY_UNUSED(connection)
   SPI_INTENTIONALLY_UNUSED(name)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
   if(NULL != poWBLClientHandler)
   {
      poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy = OSAL_NULL;
      poWBLClientHandler->m_poConflictManagementPropertiesProxy = OSAL_NULL;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vConfilctManagementProxyCb()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vConfilctManagementProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfilctManagementProxyCb entered"));
   SPI_INTENTIONALLY_UNUSED(poSourceObject)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);

   if(OSAL_NULL != poWBLClientHandler && OSAL_NULL == poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy)
   {
      GError *poGError = NULL;
      poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy = org_bosch_wbl_conflictmanagement_proxy_new_finish(poResult,&poGError);
      if(NULL == poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy)
      {
         ETG_TRACE_USR1(("spi_tclWBLClientHandler: Failed to create Conflict management Proxy!!\n"));
         if(NULL != poGError)
         {
            ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfilctManagementProxyCb error message = %s",poGError->message));
         }
      }
      else
      {
         OrgBoschWblConflictmanagementProxy *clientProxy = ORG_BOSCH_WBL_CONFLICTMANAGEMENT_PROXY(poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy);
         if(clientProxy)
         {
             ETG_TRACE_USR2(("spi_tclWBLClientHandler::vConfilctManagementProxyCb set the default timeout %d ms!!",scou32DBUSTimeout));
             g_dbus_proxy_set_default_timeout(&clientProxy->parent_instance,scou32DBUSTimeout);
         }
         t_U32 u32Ret = static_cast <t_U32> (g_signal_connect(poWBLClientHandler->m_poOrgBoschWblConflictmanagementProxy,
                  "properties-changed", G_CALLBACK(vOnConflictManagementPropertiesChanged),pUserData));
         ETG_TRACE_USR2(("spi_tclWBLClientHandler::vConfilctManagementProxyCb properties-changed signal registration status = %d",u32Ret));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vWifisetupProxyCb
 ***************************************************************************/
 t_Void spi_tclWBLClientHandler::vWifisetupProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
 {
	 ETG_TRACE_USR1(("spi_tclWBLClientHandler::vWifisetupProxyCb entered"));

	 SPI_INTENTIONALLY_UNUSED(poSourceObject)
	   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
	   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
	   if((NULL != poWBLClientHandler) && (NULL == poWBLClientHandler->m_poWifisetupProxy))
	   {
	      GError *poGError = NULL;
	      poWBLClientHandler->m_poWifisetupProxy = org_bosch_wbl_wifisetup_proxy_new_finish(poResult,&poGError);
	      if(NULL == poWBLClientHandler->m_poWifisetupProxy)
	      {
	         ETG_TRACE_USR1(("spi_tclWBLClientHandler: Failed to create vWifisetupProxyCb Proxy!!"));
	         if(NULL != poGError)
	         {
	            ETG_TRACE_USR1(("spi_tclWBLClientHandler::vWifisetupProxyCb error message = %s",poGError->message));
	            g_error_free (poGError);
	         }
	      }
	      else
	      {
	    	  t_U32 u32RetVal = static_cast<t_U32> (g_signal_connect(poWBLClientHandler->m_poWifisetupProxy,"properties-changed",
	    			  G_CALLBACK(vOnWiFiSetUpPropertiesChanged),pUserData));

	          ETG_TRACE_USR2(("spi_tclWBLClientHandler::vWifisetupProxyCb: properties-changed signal registration status = %d",u32RetVal));
	      }
	   }
 }

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb
***************************************************************************/
t_Void spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb(GObject *object, GAsyncResult *res,gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb entered"));

   GVariant *result = NULL;
   GError *error = NULL;

   gboolean bResult = org_freedesktop_dbus_properties_call_get_finish((OrgFreedesktopDBusProperties*) object, &result, res, &error);

   if(false == bResult)
   {
      ETG_TRACE_ERR(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb entered:Error in finshing"));
      if(NULL != error)
      {
         ETG_TRACE_ERR(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb entered:Error in finshing:Error message:%s",error->message));
      }
   }//if(false == bResult)
   else
   {
      spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);

      if ((NULL != poWBLClientHandler) && (NULL != result) )
      {
         /*
		 Example update:
		 array [                                                   <--- result points to this outside arrays. If there is only one array then the number of childs will be one.
		        struct{                                            <--- when we peep in, we may have multiple structures. Which are pointed by gVariant. it has 2 childs in the ex. Child1 of gVariant
					object path "xyz"                              <--- Child1 of gVariant is accessed through gVariant2 and it contains 2childs. One is object path and the other is config info.
					array[                                         <--- inside Child1 of gVariant,child 2 of gVariant2 is the actual configguration
					      dict entry abc
						  dict entry def
					]
				}
				struct{                                            <---  Child2 of gVariant = gVariant2
					object path "abc"                              <---  child 1 of gVarinat2 is object path
					array[                                         <---  child 2 of gVarinat2 is actual configuration.
					      dict entry abc
						  dict entry def
						  string "APConfig"
						  variant array[
						          dict entry SSID
								  dict entry PassPhrase
						  ]
					]
				}
		 ]
          */
         std::map<std::string, void *> mapAPConfig;
         t_U8 u8NoOfChildren = static_cast<t_U8>(g_variant_n_children(result));
         ETG_TRACE_ERR(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb NoOfChildern of result:%d",u8NoOfChildren));
         //Iterate through the result received. Ideally it contains only one child, which points to both STA & AP mode configurations.
         for(t_U8 u8index = 0; u8index < u8NoOfChildren; u8index++)
         {
            GVariant *gVariant = g_variant_get_child_value(result,u8index);   
            if(NULL != gVariant)	
            {
               t_U8 u8NoOfChildrenInVariant = static_cast<t_U8>(g_variant_n_children(gVariant));
               ETG_TRACE_ERR(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb NoOfChildern of gVariant:%d",
			   u8NoOfChildrenInVariant));

               //Iterate through the childs to find out whether AP configuration is available in the ActiveWiFiSetUps. 
               //At a time we may have both STA & AP configuration details with the current requirements and hardware capability.
               for(t_U8 u8VarInd=0;u8VarInd<u8NoOfChildrenInVariant;u8VarInd++)
               {
                  ETG_TRACE_USR1(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb gVariant index:%d type:%s",
				  u8VarInd,g_variant_get_type_string(g_variant_get_child_value(gVariant,u8VarInd))));

                  GVariant *gVariant2 = g_variant_get_child_value(gVariant,u8VarInd);
                  if(NULL != gVariant2)
                  {
                     t_U8 u8NoOfChildrenInVariant2 = static_cast<t_U8>(g_variant_n_children(gVariant));
                     ETG_TRACE_ERR(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb NoOfChildern of gVariant2:%d",
					 u8NoOfChildrenInVariant2));

                     //Iterate through the childs to find out which is the AP configuration, if available. 
                     //If the object path is matching then retrieve the complete AP configuration and notifyinternal classes.
                     for(t_U8 u8VarInd2=0;u8VarInd2<u8NoOfChildrenInVariant2;u8VarInd2++)
                     {
                        ETG_TRACE_USR1(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb gVariant2 index:%d type:%s",
						u8VarInd2,g_variant_get_type_string(g_variant_get_child_value(gVariant2,u8VarInd2))));

                        const gchar * czObjectPath = g_variant_get_string(g_variant_get_child_value(gVariant2,u8VarInd2),NULL);
                        if(NULL != czObjectPath)
                        {
                           ETG_TRACE_USR1(("spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb: ObjectPath:%s",czObjectPath));
                           poWBLClientHandler->m_szObjectPath = czObjectPath;
                           if( (poWBLClientHandler->m_szObjectPath.find("/AP") != std::string::npos) && ( u8NoOfChildrenInVariant2 > (u8VarInd2+1)))
                           {
                              poWBLClientHandler->vParseDbusDict(g_variant_get_child_value(gVariant2,u8VarInd2+1),mapAPConfig); //Parse the dictionary from tuple
                              poWBLClientHandler->vStoreWiFiConfig(mapAPConfig);
                           }//if(poWBLClientHandler->m_szObjectPath.find("/AP") != std::string::npos)
                        }//if(NULL != czObjectPath)
                        else
                        {
                           ETG_TRACE_ERR(("[ERR]:spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb: czObjectPath is NULL or current element is not an object path"));
                        }//else if(NULL != czObjectPath)
                     }//for(t_U8 u8VarInd2=0;u8VarInd2<u8NoOfChildrenInVariant2;u8VarInd2++)
                     g_variant_unref(gVariant2);
                  }//if(NULL != gVariant2)
               }//for(t_U8 u8VarInd=0;u8VarInd<u8NoOfChildrenInVariant;u8VarInd++)
               g_variant_unref(gVariant);
            }//if(NULL != gVariant)
         }//for(t_U8 u8index = 0; u8index < u8NoOfChildren; u8index++)
         g_variant_unref(result);
      }// if ( (!result)&&(NULL != poWBLClientHandler) )
      else
      {
         ETG_TRACE_ERR(("spi_tclWBLClientHandler::poWBLClientHandler or result is NULL"));
      }//else  if ( (!error)&&(NULL != poWBLClientHandler) )
   }// else if(false == bResult)
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb entered"));
   SPI_INTENTIONALLY_UNUSED(poSourceObject)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
   if(OSAL_NULL != poWBLClientHandler && OSAL_NULL == poWBLClientHandler->m_poConflictManagementPropertiesProxy)
   {
      GError *poGError = OSAL_NULL;
      poWBLClientHandler->m_poConflictManagementPropertiesProxy = org_freedesktop_dbus_properties_proxy_new_finish(poResult,&poGError);
      if(OSAL_NULL == poWBLClientHandler->m_poConflictManagementPropertiesProxy)
      {
         ETG_TRACE_USR1(("spi_tclWBLClientHandler: Failed to create Conflict management properties Proxy!!\n"));
         if(OSAL_NULL != poGError)
         {
            ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb error message = %s",poGError->message));
         }
      }
      else
      {
         t_U32 u32Ret = static_cast <t_U32> (g_signal_connect(poWBLClientHandler->m_poConflictManagementPropertiesProxy,
                  "properties-changed", G_CALLBACK(vOnFreeDesktopPropertiesChanged),pUserData));
         ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb properties-changed signal registration status = %d",u32Ret));

         org_freedesktop_dbus_properties_call_get(poWBLClientHandler->m_poConflictManagementPropertiesProxy, "org.bosch.wbl.conflictmanagement", "ActiveWiFiSetups",
                  NULL, vGetActiveWiFiSetUpsCb, pUserData);
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vRegisterCallbacks(trWiFiCallbacks rWiFiCallbacks)
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vRegisterCallbacks(trWiFiCallbacks rWiFiCallbacks)
{
	m_rWiFiCallbacks = rWiFiCallbacks;
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vConfigureWiFi(tenDeviceCategory enDevCat)
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vConfigureWiFi(tenDeviceCategory enDevCat)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfigureWiFi entered with enDevCat = %d",ETG_ENUM(DEVICE_CATEGORY,enDevCat)));
   switch(enDevCat)
   {
      case e8DEV_TYPE_DIPO:
         if(OSAL_NULL != m_poOrgBoschWblConflictmanagementProxy)
         {
            ETG_TRACE_USR1(("spi_tclWBLClientHandler::vConfigureWiFi, need to call PrepareSetup for CPW"));
            GVariantBuilder builder;
            GVariant *dict,*apconfig;

            g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));
            g_variant_builder_add(&builder, "{sv}", "Type", g_variant_new_string("CarPlayWireless"));
            g_variant_builder_add(&builder, "{sv}", "StationsToBeReserved", g_variant_new_uint16(1));
            apconfig = g_variant_builder_end(&builder);

            g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));
            g_variant_builder_add(&builder, "{sv}", "Mode", g_variant_new_string("AP"));
            g_variant_builder_add(&builder, "{sv}", "APConfig", apconfig);
            dict = g_variant_builder_end(&builder);

            org_bosch_wbl_conflictmanagement_call_prepare_setup(m_poOrgBoschWblConflictmanagementProxy,
                     dict,
                     NULL,
                     vPrepareSetupCb,
                     (void*)this);
            g_variant_unref(dict);
            g_variant_unref(apconfig);
         }
         break;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vPrepareSetupCb()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vPrepareSetupCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vPrepareSetupCb entered"));
   SPI_INTENTIONALLY_UNUSED(poSourceObject)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
   GError *poGError = OSAL_NULL;
   gboolean bResult = org_bosch_wbl_conflictmanagement_call_prepare_setup_finish((OrgBoschWblConflictmanagement*)poSourceObject,
            poResult,
            &poGError);
   if(bResult)
   {
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vPrepareSetupCb PrepareSetup operation completed successfully"));

      //Do a get for ActiveWiFiSetUps update for the fail safe.
      org_freedesktop_dbus_properties_call_get(poWBLClientHandler->m_poConflictManagementPropertiesProxy, "org.bosch.wbl.conflictmanagement", "ActiveWiFiSetups",
               NULL, vGetActiveWiFiSetUpsCb, pUserData);
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vPrepareSetupCb: Sent a get request for active Wi-Fi Setups"));
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vPrepareSetupCb PrepareSetup operation failed"));
      if(OSAL_NULL != poGError)
      {
         ETG_TRACE_USR1(("spi_tclWBLClientHandler::vPrepareSetupCb error message = %s",poGError->message));
      }
   }

   if(OSAL_NULL != poWBLClientHandler && OSAL_NULL != poWBLClientHandler->m_rWiFiCallbacks.fvOnConfigureWiFiResult)
   {
      poWBLClientHandler->m_rWiFiCallbacks.fvOnConfigureWiFiResult(bResult);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vOnConflictManagementPropertiesChanged
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vOnConflictManagementPropertiesChanged(OrgBoschWblConflictmanagement *object, const gchar *arg_interface_name,
         GVariant *arg_changed_properties, const gchar *const *arg_invalidated_properties,
         gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vOnConflictManagementPropertiesChanged entered"));
   SPI_INTENTIONALLY_UNUSED(object)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
   if(OSAL_NULL != poWBLClientHandler)
   {
      poWBLClientHandler->vHandlePropertiesChanged(arg_interface_name,arg_changed_properties,arg_invalidated_properties);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vOnFreeDesktopPropertiesChanged
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vOnFreeDesktopPropertiesChanged(OrgFreedesktopDBusProperties *object, const gchar *arg_interface_name,
         GVariant *arg_changed_properties, const gchar *const *arg_invalidated_properties,
         gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vOnFreeDesktopPropertiesChanged entered"));
   SPI_INTENTIONALLY_UNUSED(object)
   spi_tclWBLClientHandler* poWBLClientHandler = static_cast<spi_tclWBLClientHandler *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poWBLClientHandler);
   if(OSAL_NULL != poWBLClientHandler)
   {
      poWBLClientHandler->vHandlePropertiesChanged(arg_interface_name,arg_changed_properties,arg_invalidated_properties);
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vOnWiFiSetUpPropertiesChanged
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vOnWiFiSetUpPropertiesChanged(
		OrgBoschWblConflictmanagement *object,
		const gchar *arg_interface_name,
		GVariant *arg_changed_properties,
		const gchar * const *arg_invalidated_properties,
		gpointer pUserData)
{
	ETG_TRACE_USR1(("spi_tclWBLClientHandler::vOnWiFiSetUpPropertiesChanged() entered"));
	SPI_INTENTIONALLY_UNUSED(object)

	if (NULL != arg_interface_name)
	{
		ETG_TRACE_USR1(("spi_tclWBLClientHandler::vOnWiFiSetUpPropertiesChanged():arg_interface_name: %s", arg_interface_name));
	}
	//Updates are not processed currently since it is not needed. CarPlay Wireless session proceeds without checking whether restrictions could be applied.
	//There is no hard dependency with enabling restrictions as of now.
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vHandlePropertiesChanged
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vHandlePropertiesChanged(const gchar *arg_interface_name,
         GVariant *arg_changed_properties,
         const gchar *const *arg_invalidated_properties)
{
   if(NULL != arg_interface_name)
   {
       ETG_TRACE_USR1(("spi_tclWBLClientHandler::vHandlePropertiesChanged entered with arg_interface_name: %s",arg_interface_name));
   }
   SPI_INTENTIONALLY_UNUSED(arg_invalidated_properties)
   GVariantIter iter;
   GVariant *child;
   gchar *key;
   g_variant_iter_init (&iter, arg_changed_properties);
   while (g_variant_iter_next (&iter, "{sv}", &key, &child))
   {
       ETG_TRACE_USR1(("spi_tclWBLClientHandler key: %s",key));
       if( strcmp(key,"ActiveWiFiSetups") == 0 )
       {
           std::map<std::string, void *> mapAPConfig;
           t_U8 u8NoOfChildren = static_cast<t_U8>(g_variant_n_children(child));
           for(t_U8 u8index = 0; u8index < u8NoOfChildren; ++u8index)
           {
               GVariant *structure = g_variant_get_child_value(child,u8index);
               m_szObjectPath = (g_variant_get_string(g_variant_get_child_value(structure,0),NULL));
               ETG_TRACE_USR1(("spi_tclWBLClientHandler::vHandlePropertiesChanged m_szObjectPath = %s",m_szObjectPath.c_str()));
               if(m_szObjectPath.find("/AP") != std::string::npos)
               {
                   vParseDbusDict(g_variant_get_child_value(structure,1),mapAPConfig); //Parse the dictionary from tuple
                   vStoreWiFiConfig(mapAPConfig);
                   g_variant_unref(structure);
               }
           }
       }
       g_variant_unref(child);
       g_free(key);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vParseDbusDict
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vParseDbusDict(GVariant *poDict, std::map<std::string, void*> &mapAPConfig)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vParseDbusDict entered"));
   
   GVariantIter iter;
   GVariant *child;
   gchar *key;
   g_variant_iter_init (&iter, poDict);
   while (g_variant_iter_next(&iter, "{sv}", &key, &child))
   {
      if(g_variant_type_equal(g_variant_get_type(child), G_VARIANT_TYPE_BYTESTRING))
      {
		 gsize size;
         gconstpointer cvByteArray = g_variant_get_fixed_array(child,&size,sizeof(gchar));

         if ( (NULL != cvByteArray) && (size<256) )
         {
            ETG_TRACE_USR4(("vParseDbusDict:Received size of the string:%d and the value:%s",size,cvByteArray));
			char ssid[256];
			memset(ssid,0,256);
			strncpy(ssid,static_cast<const char*>(cvByteArray),size);
			ssid[size] = '\0';
			
			std::string *val = new std::string();
            if( (NULL != val) && (NULL != key) )
            {
               std::string tmp(ssid);
               ETG_TRACE_USR4(("vParseDbusDict:SSID:%s",tmp.c_str()));
               ETG_TRACE_USR4(("vParseDbusDict:Key:%s",key));
               *val = tmp;
               mapAPConfig.insert(std::make_pair<std::string,void*>(key,val));
            }//if( (NULL != val) && (NULL != key) )
		 }//if(NULL != cvByteArray)
      }
      else if(g_variant_is_container(child))
      {
		 ETG_TRACE_USR1(("vParseDbusDict: Another variant detected. Parse it"));
		 t_Bool bFurtherProcessReq = true;
		 t_U8 u8NoOfChildren = static_cast<t_U8>(g_variant_n_children(child));
         ETG_TRACE_USR4(("vParseDbusDict: Another variant-No.Of childs:%d",u8NoOfChildren));
         for(t_U8 u8Index=0;u8Index<u8NoOfChildren;u8Index++)
         {
		   ETG_TRACE_USR1(("vParseDbusDict: Another variant-index:%d type:%s",
				  u8Index,g_variant_get_type_string(g_variant_get_child_value(child,u8Index))));
				  
		   GVariant *gVariant = g_variant_get_child_value(child,u8Index);
		   if(NULL != gVariant)
		   {
		       t_U8 u8NoOfChildren2 = static_cast<t_U8>(g_variant_n_children(gVariant)); 
               ETG_TRACE_USR4(("vParseDbusDict: Another variant-No.Of childs2:%d",u8NoOfChildren2));			   
			   for(t_U8 u8Index2=0;u8Index2<u8NoOfChildren2;u8Index2++)
			   {
		          const gchar * cszKey = g_variant_get_string(g_variant_get_child_value(gVariant,u8Index2),NULL);
		          if(NULL != cszKey)
		          {
			         ETG_TRACE_USR2(("vParseDbusDict:Value:%s",cszKey));
					 const char* czAssociatedStations = "/AssociatedStation_";
					 std::string szKey = cszKey;
					 
					 if(szKey.find(czAssociatedStations) != std::string::npos)
                     {
						trStationInfo rStationInfo;
                        if(true == bParseStationDict(g_variant_get_child_value(gVariant,u8Index2+1),rStationInfo)) //Parse the dictionary from tuple
						{
							m_vecStationsInfo.push_back(rStationInfo);
						}
                     }//if(szKey.find(czAssociatedStations) != std::string::npos)
		          }
			   }
		   }
         }
		 vParseDbusDict(child, mapAPConfig);
      }
      else if(g_variant_type_equal(g_variant_get_type(child),G_VARIANT_TYPE_STRING))
      {
         std::string *val = new std::string();
         if( (NULL != val) && (NULL != key) )
         {
			ETG_TRACE_USR4(("vParseDbusDict:Key:%s",key));	
            gsize size;
            const gchar* czValue = g_variant_get_string(child,&size);
            if(NULL != czValue)
            {
			   std::string tmp(g_variant_get_string(child,NULL));
               ETG_TRACE_USR4(("vParseDbusDict:Value:%s",czValue));	  
               *val = czValue;
               mapAPConfig.insert(std::make_pair<std::string,void*>(key,val));
            }
         }
      }
      else if(g_variant_type_equal(g_variant_get_type(child),G_VARIANT_TYPE_INT32))
      {
         gint32 *val = new gint32();
         if( (NULL != val) && (NULL != key) )
         {
            *val = g_variant_get_int32(child);
            ETG_TRACE_USR4(("vParseDbusDict:Integer Value:%d, Key:%s",*val,key));
            mapAPConfig.insert(std::make_pair<std::string,void*>(key,val));
         }
      }
      else if(g_variant_type_equal(g_variant_get_type(child),G_VARIANT_TYPE_UINT32))
      {
         gint32 *val = new gint32();
         if( (NULL != val) && (NULL != key) )
         {
            *val = g_variant_get_uint32(child);
            ETG_TRACE_USR4(("vParseDbusDict:Unsigned Integer Value:%d, Key:%s",*val,key));
            mapAPConfig.insert(std::make_pair<std::string,void*>(key,val));
         }
      }
      else if(g_variant_type_equal(g_variant_get_type(child), G_VARIANT_TYPE_BOOLEAN))
      {
         gboolean *val = new gboolean();
         if( (NULL != val) && (NULL != key) )
         {
            *val = g_variant_get_boolean(child);
            ETG_TRACE_USR4(("vParseDbusDict:Boolen Value:%d, Key:%s",*val,key));
            mapAPConfig.insert(std::make_pair<std::string,void*>(key,val));
         }
      }
      g_variant_unref(child);
      g_free(key);
   }
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vParseDbusDict exit"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vStoreWiFiConfig
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vStoreWiFiConfig(std::map<std::string, void*> &mapAPconfig)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vStoreWiFiConfig entered"));

   trWiFiConfig rWiFiConfig;
   if(mapAPconfig.find(scoszKeySSID) != mapAPconfig.end())
   {
      rWiFiConfig.szSSID = *(static_cast<std::string *>(mapAPconfig[scoszKeySSID]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeySSID,std::string)
   }
   if(mapAPconfig.find(scoszKeyPassphrase) != mapAPconfig.end())
   {
      rWiFiConfig.szPassphrase = *(static_cast<std::string *>(mapAPconfig[scoszKeyPassphrase]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyPassphrase,std::string)
   }
   if(mapAPconfig.find(scoszKeyInterface) != mapAPconfig.end())
   {
      rWiFiConfig.szInterface = *(static_cast<std::string *>(mapAPconfig[scoszKeyInterface]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyInterface,std::string)
   }
   if(mapAPconfig.find(scoszKeyMode) != mapAPconfig.end())
   {
      rWiFiConfig.szMode = *(static_cast<std::string *>(mapAPconfig[scoszKeyMode]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyMode,std::string)
   }
   if(mapAPconfig.find(scoszKeyType) != mapAPconfig.end())
   {
      rWiFiConfig.szType = *(static_cast<std::string *>(mapAPconfig[scoszKeyType]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyType,std::string)
   }
   if(mapAPconfig.find(scoszKeyCurrentOperatingChannel) != mapAPconfig.end())
   {
      rWiFiConfig.u32CurrentOperatingChannel = *(static_cast<t_U32 *>(mapAPconfig[scoszKeyCurrentOperatingChannel]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyCurrentOperatingChannel,t_U32)
   }
   if(mapAPconfig.find(scoszKeyFrequency) != mapAPconfig.end())
   {
      t_String freq = *(static_cast<std::string *>(mapAPconfig[scoszKeyFrequency]));
      if("5 GHz" == freq)
         rWiFiConfig.enFrequency = tenWifiFrequency::e8_5_dot_0_GHz;
      else if("2.4 GHz" == freq)
         rWiFiConfig.enFrequency = tenWifiFrequency::e8_2_dot_4_GHz;
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyFrequency,std::string)
   }
   if(mapAPconfig.find(scoszKeySecurity) != mapAPconfig.end())
   {
      t_String security = *(static_cast<std::string *>(mapAPconfig[scoszKeySecurity]));
      if("wpa2-psk" == security)
         rWiFiConfig.enSecurity = tenWifiSecurityType::e8_WPA2;
      else if("wpa-psk" == security)
         rWiFiConfig.enSecurity = tenWifiSecurityType::e8_WPA;
      else if("wep" == security)
         rWiFiConfig.enSecurity = tenWifiSecurityType::e8_WEP;
      else if("open" == security)
         rWiFiConfig.enSecurity = tenWifiSecurityType::e8_NONE;
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeySecurity,std::string)
   }
   if(mapAPconfig.find(scoszKeyVisible) != mapAPconfig.end())
   {
      rWiFiConfig.bVisible = *(static_cast<bool *>(mapAPconfig[scoszKeyVisible]));
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyVisible,bool)
   }
   if(mapAPconfig.find(scoszKeyPowered) != mapAPconfig.end())
   {
      t_String szPowerState = *(static_cast<std::string *>(mapAPconfig[scoszKeyPowered]));
      ETG_TRACE_USR4(("spi_tclWBLClientHandler::vStoreWiFiConfig:PowerState:%s",szPowerState.c_str()));

      if("WIFI_STATE_POWERED_ON" == szPowerState)
      {
         rWiFiConfig.bPowered = true;
      }
      else
      {
         //All intermediate power states are treated as Wi-Fi OFF. Intermediate states includes "WIFI_STATE_POWERING_ON" ,
         //"WIFI_STATE_POWERING_OFF". And the final state is "WIFI_STATE_POWERED_OFF" is equivalent to Power OFF.
         //Other states WIFI_STATE_DEVICE_AVAILABLE", "WIFI_STATE_DEVICE_UNAVAILABLE" or not relevant for AP.
         rWiFiConfig.bPowered = false;
      }
      RELEASE_MAP_VALUE(mapAPconfig,scoszKeyPowered,std::string)
   }

      
   if(NULL != m_rWiFiCallbacks.fvOnWiFiConfig)
   {
      m_rWiFiCallbacks.fvOnWiFiConfig(rWiFiConfig,m_vecStationsInfo);
	  m_vecStationsInfo.clear();
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclWBLClientHandler::vSetWiFiLimitationMode()
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vSetWiFiLimitationMode(tenDeviceCategory enDevCategory,t_Bool bActivate)
{

	ETG_TRACE_USR1(("[FUNC]spi_tclWBLClientHandler::vSetWiFiLimitationMode:DeviceCategory:%d, Activate:%d",
			ETG_ENUM(DEVICE_CATEGORY,enDevCategory),ETG_ENUM(BOOL,bActivate)));

	  switch(enDevCategory)
	   {
	      case e8DEV_TYPE_DIPO:
		  {
	         if(NULL != m_poWifisetupProxy)
	         {
	            ETG_TRACE_USR1(("spi_tclWBLClientHandler::vSetWiFiLimitationMode"));
	            
				//Create technology gVariant object : definition in introspection file:<arg name="Technology" type="a{sv}" direction="in"/>

                /*
                 * dict array[
                 * 			   String "SPITechnology"
                 *             variant dict array [
                 *             				string "Technology"
                 *                          variant string "CarPlayWireless"
                 *                          ]
                 *              ]
                 * string "restrict"
                 */
				GVariantBuilder gvTechnologyBuilder;
	            GVariant *Technology;
				g_variant_builder_init(&gvTechnologyBuilder,G_VARIANT_TYPE("a{sv}"));
	            g_variant_builder_add(&gvTechnologyBuilder, "{sv}", "Technology", g_variant_new_string("CarPlayWireless"));
	            Technology = g_variant_builder_end(&gvTechnologyBuilder);

	            GVariantBuilder gvSPITechnologyBuilder;
	            GVariant *SPITechnology;
	            g_variant_builder_init(&gvSPITechnologyBuilder,G_VARIANT_TYPE("a{sv}"));
	            g_variant_builder_add(&gvSPITechnologyBuilder, "{sv}", "SPITechnology", Technology);
	            SPITechnology = g_variant_builder_end(&gvSPITechnologyBuilder);
				
				//Second objects is gChar* to set the restrictions: definition in introspection file:<arg name="restrict" type="s" direction="in"/>
				//Send the request WBL to enable/disable feature restrictions 
				org_bosch_wbl_wifisetup_call_set_restrictions (m_poWifisetupProxy,
						                                       SPITechnology,
															   (true==bActivate)?scActivateRestrictions:scDeActivateRestritcions,
															   NULL,
															   vSetRestrictionCb,
															   (void*)this);

	            g_variant_unref(SPITechnology);
	            g_variant_unref(Technology);
	         }
		  }
	      break;
	      default:
	      {
	    	  ETG_TRACE_USR1(("[FUNC]spi_tclWBLClientHandler::vSetWiFiLimitationMode:Invalid request"));
	      }
	   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::vSetRestrictionCb
 ***************************************************************************/
t_Void spi_tclWBLClientHandler::vSetRestrictionCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclWBLClientHandler::vSetRestrictionCb entered"));

   GError *poGError = OSAL_NULL;
   gboolean bResult = org_bosch_wbl_wifisetup_call_set_restrictions_finish((OrgBoschWblWifisetup*)poSourceObject,poResult,&poGError);
   
   if(true == bResult)
   {
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vSetRestrictionCb:operation completed successfully"));
   }
   else
   {
      ETG_TRACE_USR1(("spi_tclWBLClientHandler::vSetRestrictionCb: operation failed"));
      if(OSAL_NULL != poGError)
      {
         ETG_TRACE_USR1(("spi_tclWBLClientHandler::vSetRestrictionCb:error message = %s",poGError->message));
      }
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclWBLClientHandler::bParseStationDict
 ***************************************************************************/
t_Bool spi_tclWBLClientHandler::bParseStationDict(GVariant *gVariant,trStationInfo& rfrStationInfo)
{
   ETG_TRACE_USR1(("bParseStationDict"));
   t_Bool bRet=false;

   //Get the no. of childern of the variant provided.
   t_U8 u8NoOfChildren = static_cast<t_U8>(g_variant_n_children(gVariant));
   ETG_TRACE_USR4(("bParseStationDict:No.Of childs to gVariant:%d",u8NoOfChildren));
   for(t_U8 u8Index=0;u8Index<u8NoOfChildren;u8Index++)
   {
      ETG_TRACE_USR1(("bParseStationDict: Another gvariant-index:%d type:%s",
               u8Index,g_variant_get_type_string(g_variant_get_child_value(gVariant,u8Index))));

      GVariant *gVariant2 = g_variant_get_child_value(gVariant,u8Index);
      if(NULL != gVariant2)
      {
         GVariantIter iter;
         gchar *key;
         GVariant *child;
         g_variant_iter_init (&iter,gVariant2);
         while (g_variant_iter_next(&iter, "{sv}", &key, &child))
         {
            if(NULL != key)
            {
               ETG_TRACE_USR2(("bParseStationDict:Key:%s",key));
               ETG_TRACE_USR1(("bParseStationDict: Data Type of the Key value:%s",g_variant_get_type_string(child)));
               if(g_variant_type_equal(g_variant_get_type(child),G_VARIANT_TYPE_STRING))
               {
                  gsize size;
                  const gchar* czValue = g_variant_get_string(child,&size);
                  if(NULL != czValue)
                  {
                     ETG_TRACE_USR2(("bParseStationDict:Value:%s",czValue));
                     const char* szIPAddress = "IPAddress";
                     const char* szHostName = "HostName";
                     const char* szMacAddress = "MacAddress";
                     if(0 == strncmp(szIPAddress,key,strlen(szIPAddress)))
                     {
                        bRet = true; // Return TRUE - STA info is available
                        rfrStationInfo.szIPAddress = czValue;
                     }
                     else if(0 == strncmp(szHostName,key,strlen(szHostName)))
                     {
                        bRet = true; // Return TRUE - STA info is available
                        rfrStationInfo.szDeviceName = czValue;
                     }
                     else if(0 == strncmp(szMacAddress,key,strlen(szMacAddress)))
                     {
                        bRet = true; // Return TRUE - STA info is available
                        //Removing the unwanted characters like : from the given string before dispatcahing the message
                        t_String szWifiMacAddress = czValue;
                        std::regex rPattern(":");
                        szWifiMacAddress = std::regex_replace(czValue, rPattern , "");
                        rfrStationInfo.szMacAddress = szWifiMacAddress;
                     }
                  }  //if(NULL != czValue)
               }//if(g_variant_type
               else if(g_variant_is_container(child))
               {
                  ETG_TRACE_USR2(("bParseStationDict:container type. This value is not expected inside a STA info"));
               }
               else
               {
                  ETG_TRACE_USR4(("bParseStationDict:Else case.This value is not expected inside a STA info"));
               }
            }//if(NULL != key)
            g_variant_unref(child);
            g_free(key);
         }//while (g_var

      }//if(NULL != gVaria
   }//for(t_U8 u8Index=0
   ETG_TRACE_USR1(("bParseStationDict exit"));
return bRet;
}

//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
