/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgUtil/Optimizer.cpp
List of changes:
1. Support to replace a single texture in an atlas.
*/

#include <osgDB/Registry>
#include <osgDB/ReaderWriter>
#include "TextureBatchingAtlas.h"

using namespace mapengine::oss;

bool me_TextureBatchingStrategyAtlas::addImage( unsigned int imageId, const std::vector<unsigned char>& imageData, const TextureParameters& textureParam, unsigned int frameNumber )
{
   bool objectAdded = false;
   OpenThreads::ScopedLock<OpenThreads::ReentrantMutex> lock(_mutex);
   
   for(SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
   {  
      AtlasSource* currentSource = static_cast<AtlasSource*>( sitr->get() );  //lint !e1774
      if( currentSource->getID()== imageId  )
      {  
         currentSource->setFrameNumber(frameNumber);
         objectAdded = true;
         break;
      }
   }

   if( !objectAdded)
   {
      if( isImagePNG(imageData))
      {
         //support for only png image
         osgDB::ReaderWriter* reader = 0;
         reader = osgDB::Registry::instance()->getReaderWriterForExtension("png");
         Streambuf sb( reinterpret_cast<char*>(const_cast<unsigned char*>(&imageData[0])), imageData.size());
         std::istream is(&sb);
         osgDB::ReaderWriter::ReadResult result = reader->readImage(is);

         //osg::Texture2D* texture = new osg::Texture2D( result.takeImage() ); // take not get!

         osg::Texture2D* texture; 
         osg::Image* image = result.takeImage(); // take not get!

         if(image->getPixelFormat() != GL_RGBA)
         {
            texture = new osg::Texture2D(mapengine::oss::convertToRGBA(image)); //lint -esym(429,texture)
         }
         else 
         {
            texture = new osg::Texture2D(image);
         }
         if(texture && textureParam.disableMipmap)
         {
            texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
         }
         osg::ref_ptr<osg::Array> texCoords;
         objectAdded = (0 != addTextureObject(imageId, texture, frameNumber, textureParam, texCoords));  //atlas width = atlas height
      }
   }
   return objectAdded;
}

bool me_TextureBatchingStrategyAtlas::isTexturePresent( unsigned int textureID,
                                                        unsigned int atlasID,
                                                        osg::ref_ptr<osg::Texture>& batchedTexture,
                                                        osg::ref_ptr<osg::Array>& texCoords)
{
   bool textureAlreadyExists = false;
   batchedTexture = 0;
   texCoords = 0;
   AtlasSource* currentSource = 0;
   //iterate through the source list of the given atlasID to see if the texture with the given textureID exists
   {
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_mutex);
      for(AtlasList::iterator aitr = _atlasList.begin(); (aitr != _atlasList.end() )&&(!textureAlreadyExists); ++aitr)
      {
         if(static_cast<WallpaperAtlas*>(aitr->get())->getID() == atlasID )//lint !e1774
         {
            for(SourceList::iterator sitr = aitr->get()->_sourceList.begin(); (sitr != aitr->get()->_sourceList.end() ) &&(!textureAlreadyExists); ++sitr)
            {  
               currentSource = static_cast<AtlasSource*>( sitr->get() );//lint !e1774
               if( currentSource->getID()== textureID  )
               {  
                  textureAlreadyExists = true;
                  texCoords = currentSource->getTextureCoords();
                  batchedTexture = currentSource->_atlas->_texture;
               }
            }
         }
         //keep searching other atlases as the atlas id is not necessarily unique
      }
   }
   return textureAlreadyExists;
}

void me_TextureBatchingStrategyAtlas::addDefaultsToTexCoordArray( osg::Array* tcArray ) const
{
   osg::Vec2Array* tcv2 = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   tcv2->push_back(osg::Vec2(1.0F, 1.0F));
   tcv2->push_back(osg::Vec2(0.0F, 1.0F));
   tcv2->push_back(osg::Vec2(0.0F, 0.0F));
   tcv2->push_back(osg::Vec2(1.0F, 0.0F));
}

void me_TextureBatchingStrategyAtlas::replaceEndWithDefaultsForTexCoordArray(osg::Array* tcArray) const
{
   osg::Vec2Array* texCoordArray = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   if(texCoordArray)
   {
      for(unsigned int i=0; i < 4 ; ++i)
      {
         texCoordArray->pop_back();
      }

      addDefaultsToTexCoordArray(tcArray);
   }
}

osg::Array* me_TextureBatchingStrategyAtlas::createTexCoordArray() const
{
    return new osg::Vec2Array;
}

void me_TextureBatchingStrategyAtlas::addToTexCoordArray( osg::Array* tcArray, osg::ref_ptr<osg::Array> texCoordsToAdd ) const
{
   osg::Vec2Array* tcv2 = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   osg::Vec2Array* tcToAddv2 = static_cast<osg::Vec2Array*>(texCoordsToAdd.get()); //lint !e1774
   tcv2->insert(tcv2->end(), tcToAddv2->begin(), tcToAddv2->begin() + 4 );
}

void me_TextureBatchingStrategyAtlas::clearTexCoordArray( osg::Array* tcArray ) const
{
   static_cast<osg::Vec2Array*>(tcArray)->clear(); //lint !e1774
}

bool me_TextureBatchingStrategyAtlas::isTexturePresent(unsigned int textureID,
                                                       unsigned int atlasID,
                                                       osg::ref_ptr<osg::Texture>& batchedTexture,
                                                       osg::ref_ptr<osg::Array>& texCoords,
                                                       unsigned int frameNumber)
{
   bool textureAlreadyExists = false;
   batchedTexture = 0;
   texCoords = 0;
   AtlasSource* currentSource = 0;
   //iterate through the source list of the given atlasID to see if the texture with the given textureID exists
   {
      OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_mutex);
      for(AtlasList::iterator aitr = _atlasList.begin(); (aitr != _atlasList.end() )&&(!textureAlreadyExists); ++aitr)
      {
         if(static_cast<WallpaperAtlas*>(aitr->get())->getID() == atlasID )//lint !e1774
         {
            for(SourceList::iterator sitr = aitr->get()->_sourceList.begin(); (sitr != aitr->get()->_sourceList.end() ) &&(!textureAlreadyExists); ++sitr)
            {  
               currentSource = static_cast<AtlasSource*>( sitr->get() );//lint !e1774
               if( currentSource->getID()== textureID  )
               {  
                  textureAlreadyExists = true;
                  texCoords = currentSource->getTextureCoords();
                  batchedTexture = currentSource->_atlas->_texture;
                  currentSource->setFrameNumber(frameNumber);
               }
            }
         }
         //keep searching other atlases as the atlas id is not necessarily unique
      }
   }
   return textureAlreadyExists;
}

me_TextureBatchingStrategyAtlas::AtlasSource* me_TextureBatchingStrategyAtlas::addSourceImage( const osg::Image* image, unsigned int textureID )
{
   me_TextureBatchingStrategyAtlas::AtlasSource* newSource = 0;
   if (!getSource(image))
   {
      newSource = new AtlasSource(image, textureID);
      _sourceList.push_back(newSource);
   }
   return newSource;
}

me_TextureBatchingStrategyAtlas::AtlasSource* me_TextureBatchingStrategyAtlas::addSourceTexture( const osg::Texture2D* texture, unsigned int textureID )
{
   me_TextureBatchingStrategyAtlas::AtlasSource* newSource = 0;
   if (!getSource(texture)) 
   {
      newSource = new AtlasSource(texture, textureID);
      _sourceList.push_back(newSource);
   }
   return newSource;
}

osg::Texture* me_TextureBatchingStrategyAtlas::addTextureObject( unsigned int textureID, osg::Texture2D* inputTexture, unsigned int frameNumber,
                                                                const TextureParameters& textureParam, osg::ref_ptr<osg::Array>& texCoords)
{
   addSourceTexture(inputTexture, textureID);
   return addSource(textureID, frameNumber, textureParam, texCoords);
}//lint !e429

void me_TextureBatchingStrategyAtlas::storeTextureCoordinates(osg::ref_ptr<AtlasSource> source)
{
   if(source.valid())
   {
      osg::Matrix textureMatrix;
      {
         OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_mutex);
         textureMatrix  = getTextureMatrix(source->_texture);
      }
      osg::ref_ptr<osg::Vec2Array> texCoords = new osg::Vec2Array; 
      if(texCoords.valid())
      {
         texCoords->push_back(osg::Vec2(1.0f,1.0f));
         texCoords->push_back(osg::Vec2(0.0f,1.0f));
         texCoords->push_back(osg::Vec2(0.0f,0.0f));
         texCoords->push_back(osg::Vec2(1.0f,0.0f));

         //compute the actual texture co-ordinates
         for(unsigned int i = 0; i < 4 ; i++)
         {
            (*texCoords)[i].x() = ((*texCoords)[i].x()  * textureMatrix(0,0)) + ( (*texCoords)[i].y()  * textureMatrix(1,0) ) + textureMatrix(3,0);
            (*texCoords)[i].y() = ((*texCoords)[i].x()  * textureMatrix(0,1)) + ( (*texCoords)[i].y()  * textureMatrix(1,1) ) + textureMatrix(3,1);
         }

         {
            OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_mutex);
            source->setTextureCoords(texCoords);
         }
      }
   }
}

bool me_TextureBatchingStrategyAtlas::rebuildAtlas(osg::ref_ptr<AtlasSource> source, osg::ref_ptr<WallpaperAtlas> atlas, unsigned int frameNumber)
{
   // since this method is only called in locked state we remove the mutex here to avoid recursive mutex deadlocks

   //few parts of the code copied from osgUtil::Optimizer::TextureAtlasBuilder::buildAtlas()
   bool srcReplaced = false;
   AtlasList::iterator aitr = _atlasList.end();
   //get the source and atlas iterator
   //need the iterator for completeRow(aitr - _atlasList.begin()) provided by osg
   {
      for(AtlasList::iterator itr = _atlasList.begin(); itr != _atlasList.end(); ++itr)
      {
         if( (*itr) == atlas)
         {
            aitr = itr;
            break;
         }
      }
   }
   bool addedSourceToAtlas = false;
   if (!source->_atlas && source->suitableForAtlas(atlas->_maximumAtlasWidth, atlas->_maximumAtlasHeight,_margin))
   {
      //check where does the source fit in the atlas
      if(atlas->_image) 
      {
         if( atlas->_image->getPixelFormat() == source->_image->getPixelFormat() &&  atlas->_image->getPacking() == source->_image->getPacking() )
         {
            OSG_INFO<<"checking source "<<source->_image->getFileName()<<" to see it it'll fit in atlas "<<atlas<<std::endl;
            Atlas::FitsIn fitsIn = atlas->doesSourceFit(source);
            if (fitsIn == Atlas::FITS_IN_CURRENT_ROW)
            {
               addedSourceToAtlas = true;
               {
                  // addSource() in the atlas is what determines the position of the source image in the wallpaper atlas
                  atlas->addSource(source); // Add in the currentRow.

                  atlas->copyOnlyThisSource(source);
                  source->setFrameNumber(frameNumber);
               }
            }
            else if(fitsIn == Atlas::IN_NEXT_ROW)
            {
               //need the atlas iterator only for this function
               completeRow(aitr - _atlasList.begin()); //Fill Empty spaces
               addedSourceToAtlas = true;
               //if(!source->_atlas) // prevents multiple copies in _atlasList->sourceList
               {
                  // addSource() in the atlas is what determines the position of the source image in the wallpaper atlas
                  atlas->addSource(source); // Add the source in the new row.
                  atlas->copyOnlyThisSource(source);
                  source->setFrameNumber(frameNumber);
               }
            }
            else
            {  
               std::sort(atlas->_sourceList.begin(), atlas->_sourceList.end(), CompareFrame());

               //compare with current frame and then decide whether to replace or create a new atlas 
               // if the source was used within in the last 100 frames, then don't replace. Create a new atlas
               if(100 > (static_cast<AtlasSource*>((atlas->_sourceList[0].get()))->getFrameNumber()- frameNumber) )  //lint !e1774
               {
                  atlas->setAtlasFull(true);
               }
               else
               {
                  for( unsigned int i = 0; i < atlas->_sourceList.size(); i++)
                  {  
                     srcReplaced = replaceSource(atlas->_sourceList[i], source);

                     if(srcReplaced)
                     {                      
                        source->setFrameNumber(frameNumber);
                        addedSourceToAtlas = true;
                        break;      //break for loop
                     }
                  }
                  //if( !_srcReplaced)
                  //completeRow(aitr - _atlasList.begin()); //Fill Empty spaces before creating a new atlas.
               }
            }
         }
         else
         {
            addedSourceToAtlas = false;
         }
      }
      //allocate memory for the atlas->_image
      else
      {         
         {  
            //addSource()in the atlas is what determines the position of the source image in the wallpaper atlas
            //atlas->_images gets its properties from the first source added to it         
            atlas->addSource(source);
         }
         source->setFrameNumber(frameNumber);

         //allocate memory for the atlas image
         GLenum pixelFormat = (atlas)->_image->getPixelFormat();
         GLenum dataType = (atlas)->_image->getDataType();
         GLenum packing = (atlas)->_image->getPacking();
         (atlas)->_image->allocateImage( atlas->_maximumAtlasWidth, atlas->_maximumAtlasHeight ,1, pixelFormat, dataType,packing);
         {
            //clear memory
            unsigned int size = (atlas)->_image->getTotalSizeInBytes();
            unsigned char* str = (atlas)->_image->data();
            memset(str, 0, size);
         }

         atlas->copyOnlyThisSource(source);
         addedSourceToAtlas = true;
      }
   }
   return addedSourceToAtlas;
}

bool me_TextureBatchingStrategyAtlas::replaceSource( Source* oldSrc , Source* newSrc )
{
   //replace source will be successful only if the old texture and the new texture are of the same size

   //check with max width, height and margin of source
   Atlas* atlas = oldSrc->_atlas;
   if( (oldSrc->_image->s() == newSrc->_image->s() && oldSrc->_image->t() == newSrc->_image->t() ))
   { 
      newSrc->_atlas = oldSrc->_atlas;
      newSrc->_x = oldSrc->_x;
      newSrc->_y = oldSrc->_y;
      atlas->_sourceList.push_back(newSrc);
      atlas->copyOnlyThisSource(newSrc);

      //remove the oldSrc from the source lists
      for(SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
      {
         if( oldSrc == (*sitr) )
         {
            _sourceList.erase(sitr);
            break;
         }
      }

      for(SourceList::iterator sitr = atlas->_sourceList.begin(); sitr != atlas->_sourceList.end(); ++sitr)
      {
         if( oldSrc == (*sitr) )
         {
            atlas->_sourceList.erase(sitr);
            break;
         }
      }
      return true;
   }
   return false;
}

void me_TextureBatchingStrategyAtlas::WallpaperAtlas::copyOnlyThisSource(Source* source)
{
   { //code copied from osgUtil::Optimizer::TextureAtlasBuilder::Atlas::copySources() with minor changes

      Atlas* atlas = source->_atlas;
      if (atlas == this)
      {
         OSG_INFO<<"Copying image "<<source->_image->getFileName()<<" to "<<source->_x<<" ,"<<source->_y<<std::endl;
         OSG_INFO<<"        image size "<<source->_image->s()<<","<<source->_image->t()<<std::endl;

         const osg::Image* sourceImage = source->_image.get();
         osg::Image* atlasImage = atlas->_image.get();
         //assert(sourceImage->getPacking() == atlasImage->getPacking()); //Test if packings are equals.
         unsigned int rowSize = sourceImage->getRowSizeInBytes();
         unsigned int pixelSizeInBits = sourceImage->getPixelSizeInBits();
         unsigned int pixelSizeInBytes = pixelSizeInBits/8;
         unsigned int marginSizeInBytes = pixelSizeInBytes*_margin;

         //assert(atlas->_width  == static_cast<int>(atlasImage->s()));
         //assert(atlas->_height == static_cast<int>(atlasImage->t()));
         //assert(source->_x + static_cast<int>(source->_image->s())+_margin <= static_cast<int>(atlas->_image->s()));        // "+_margin" and not "+2*_margin" because _x already takes the margin into account
         //assert(source->_y + static_cast<int>(source->_image->t())+_margin <= static_cast<int>(atlas->_image->t()));
         //assert(source->_x >= _margin);
         //assert(source->_y >= _margin);
         int x = source->_x;
         int y = source->_y;

         int t;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            const unsigned char* sourcePtr = sourceImage->data(0, t);
            memcpy(destPtr, sourcePtr, rowSize);
         }

         // copy top row margin
         y = source->_y + sourceImage->t();
         int m;
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            memset(destPtr, 0, rowSize);
         }
         // copy bottom row margin
         y = source->_y-1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(x, y);
            memset(destPtr, 0, rowSize);
         }


         // copy left column margin
         y = source->_y;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            x = source->_x-1;
            for(m=0; m<_margin; ++m, --x)
            {
               unsigned char* destPtr = atlasImage->data(x, y);
               memset(destPtr, 0, pixelSizeInBytes);
            }
         }            

         // copy right column margin
         y = source->_y;
         for(t=0; t<sourceImage->t(); ++t, ++y)
         {
            x = source->_x + sourceImage->s();
            for(m=0; m<_margin; ++m, ++x)
            {
               unsigned char* destPtr = atlasImage->data(x, y);
               memset(destPtr, 0, pixelSizeInBytes);
            }
         }            

         // copy top left corner margin
         y = source->_y + sourceImage->t();
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x - _margin, y);
            memset(destPtr, 0, marginSizeInBytes);
         }

         // copy top right corner margin
         y = source->_y + sourceImage->t();
         for(m=0; m<_margin; ++m, ++y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x + sourceImage->s(), y);
            memset(destPtr, 0, marginSizeInBytes);
         }

         // copy bottom left corner margin
         y = source->_y - 1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x - _margin, y);
            memset(destPtr, 0, marginSizeInBytes);
         }

         // copy bottom right corner margin
         y = source->_y - 1;
         for(m=0; m<_margin; ++m, --y)
         {
            unsigned char* destPtr = atlasImage->data(source->_x + sourceImage->s(), y);
            memset(destPtr, 0, marginSizeInBytes);
         }
      }
   }
   _image->dirty();
}

osg::Texture* me_TextureBatchingStrategyAtlas::addTextureObject( unsigned int textureID,
                                                                 osg::Image* inputImage,
                                                                 unsigned int frameNumber,
                                                                 const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
                                                                 osg::ref_ptr<osg::Array>& texCoords )
{
   osg::Texture* batchedTexture = 0;
   osg::Texture2D* sourceTexture = new osg::Texture2D(inputImage);
   if (sourceTexture)
   {
      // set the desired attribs to the source texture as OSG Optimizer sets the same to the atlas
      sourceTexture->setWrap(osg::Texture::WRAP_R, textureParam.wrapMode);
      sourceTexture->setWrap(osg::Texture::WRAP_S, textureParam.wrapMode);
      sourceTexture->setWrap(osg::Texture::WRAP_T, textureParam.wrapMode);
      sourceTexture->setFilter(osg::Texture::MIN_FILTER, textureParam.disableMipmap ? osg::Texture::LINEAR : osg::Texture::LINEAR_MIPMAP_LINEAR);
      sourceTexture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
      addSourceTexture(sourceTexture, textureID);
      sourceTexture = 0; //lint !e423
      batchedTexture =  addSource(textureID, frameNumber, textureParam, texCoords);
   }
   return batchedTexture;
}

osg::Texture* me_TextureBatchingStrategyAtlas::replaceTextureObject( unsigned int textureID,
                                                                     osg::Texture2D* inputTexture,
                                                                     unsigned int frameNumber,
                                                                     const TextureParameters& textureParam,
                                                                     osg::ref_ptr<osg::Array>& texCoords )
{
   osg::Texture* atlasTexture = 0;
   bool atlasPresent = false;
   osg::ref_ptr<AtlasSource> existingSource = NULL;
   osg::ref_ptr<WallpaperAtlas> atlas = NULL;
    osg::ref_ptr<AtlasSource> newSource;

   //check if the atlas with the given atlasID already exists
   for(AtlasList::iterator aitr = _atlasList.begin(); aitr != _atlasList.end(); ++aitr)
   {
      WallpaperAtlas* currentAtlas= static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
      if(currentAtlas->getID() == textureParam.atlasID /*&& !(osg::Texture::REPEAT == textureParam.wrapMode) */)
      {
         atlas = static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
         atlasPresent = true;
         break;
      }
   }

   if( atlasPresent && atlas.valid())
   {
      //a valid atlas with the given atlasID is present
      existingSource = atlas->getExistingSource(textureID);
      if(existingSource.valid())
      {
         newSource = addSourceTexture(inputTexture, textureID);
         newSource->setFrameNumber(frameNumber);
         replaceSource(existingSource, newSource);
         storeTextureCoordinates(newSource);
         texCoords = newSource->getTextureCoords();
         atlasTexture = atlas->_texture;
      }
   }

   if (!atlasTexture)
   {
      // The texture doesn't exist yet, so create it
      atlasTexture = addTextureObject(textureID, inputTexture, frameNumber, textureParam, texCoords);
   }
   
   return atlasTexture;
}

osg::Texture* me_TextureBatchingStrategyAtlas::replaceTextureObject( unsigned int textureID,
                                                                     osg::Image* inputImage, 
                                                                     unsigned int frameNumber, 
                                                                     const me_TextureBatchingStrategyBase::TextureParameters& textureParam, 
                                                                     osg::ref_ptr<osg::Array>& texCoords )
{
   osg::Texture* batchedTexture = 0;
   osg::Texture2D* sourceTexture = new osg::Texture2D(inputImage);
   if (sourceTexture)
   {
      // set the desired attribs to the source texture as OSG Optimizer sets the same to the atlas
      sourceTexture->setWrap(osg::Texture::WRAP_R, textureParam.wrapMode);
      sourceTexture->setWrap(osg::Texture::WRAP_S, textureParam.wrapMode);
      sourceTexture->setWrap(osg::Texture::WRAP_T, textureParam.wrapMode);
      sourceTexture->setFilter(osg::Texture::MIN_FILTER, textureParam.disableMipmap ? osg::Texture::LINEAR : osg::Texture::LINEAR_MIPMAP_LINEAR);
      sourceTexture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
      batchedTexture = replaceTextureObject(textureID, sourceTexture, frameNumber, textureParam, texCoords);
      sourceTexture = 0; //lint !e423
   }
   return batchedTexture;
}

void me_TextureBatchingStrategyAtlas::removeAllTextures( const osg::State* state, unsigned int batchID )
{
   OpenThreads::ScopedLock<OpenThreads::Mutex> lock(_mutex);
   AtlasList::iterator aitr = _atlasList.begin();
   while(aitr != _atlasList.end())
   {      
      WallpaperAtlas* atlas = static_cast<WallpaperAtlas*>(aitr->get()); //lint !e1774
      ++aitr;
      if(atlas->getID() == batchID)
      {
         if(atlas->_texture->areAllTextureObjectsLoaded())
         {
            atlas->_texture->releaseTextureObject(state->getContextID(), atlas->_texture->getTextureObject(state->getContextID()));
         }
         _sourceList.erase(remove_if(_sourceList.begin(), _sourceList.end(), SourceErase(atlas)), _sourceList.end());
         aitr = _atlasList.erase(--aitr);
      }
   }   
   _atlasMap.erase(batchID);
}

osg::Texture* me_TextureBatchingStrategyAtlas::createSingleTexture( osg::Image* image ) const
{
   return new osg::Texture2D(image);
}

osg::Texture* me_TextureBatchingStrategyAtlas::addSource(unsigned int textureID,
                                                         unsigned int frameNumber,
                                                         const me_TextureBatchingStrategyBase::TextureParameters& textureParam,
                                                         osg::ref_ptr<osg::Array>& texCoords)
{
   bool atlasPresent = false;
   osg::ref_ptr<AtlasSource> curSource = NULL;
   bool atlasReady;
   osg::ref_ptr<WallpaperAtlas> atlas = NULL;
   unsigned int returnTextureID = 0;

   //check if the atlas with the given atlasID already exists
   for(AtlasList::iterator aitr = _atlasList.begin(); aitr != _atlasList.end(); ++aitr)
   {
      WallpaperAtlas* currentAtlas= static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
      if(currentAtlas->getID() == textureParam.atlasID /*&& !(osg::Texture::REPEAT == textureParam.wrapMode) */)
      {
         atlas = static_cast<WallpaperAtlas*>( aitr->get() );//lint !e1774
         atlasPresent = true;
         break;
      }
   }

   if( atlasPresent && atlas.valid())
   { //a valid atlas with the given atlasID is present
      // go through the sourcelist and find the source
      for(SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
      {
         if( static_cast<AtlasSource*>(sitr->get())->getID() == textureID  && !(*sitr)->_atlas)//lint !e1774
         {
            curSource = static_cast<AtlasSource*>(sitr->get());//lint !e1774
            break;
         }
      }
      if(curSource.valid())
      {
         returnTextureID = textureID;
         atlasReady = rebuildAtlas(curSource, atlas, frameNumber ); 
         if(!atlasReady)
         {          
            {
               unsigned int i = 1;
               while(!atlasReady && (i < _atlasMap[textureParam.atlasID].size()) )        
               {
                  WallpaperAtlas* localAtlas = _atlasMap[textureParam.atlasID][i];
                  //check if re-caching is possible
                  atlasReady = rebuildAtlas(curSource, localAtlas, frameNumber );
                  ++i;
               }
            }
            //if the source could not be put in any of the existing atlas, create a new one
            if(!atlasReady)
            {            
               //std::cout<< "given atlas is full, creating new one"<<std::endl;
               osg::ref_ptr<WallpaperAtlas> texAtlas = new WallpaperAtlas(textureParam.atlasWidth, textureParam.atlasHeight, textureParam.textureMargin); //width, height, margin
               if(texAtlas.valid())
               {
                  _atlasList.push_back(texAtlas);
                  texAtlas->setID(textureParam.atlasID);
                  texAtlas->setSubID(atlas->getSubID() + 1);
                  _atlasMap[textureParam.atlasID].push_back(texAtlas);
                  atlasReady = rebuildAtlas(curSource, texAtlas, frameNumber);
               }
            }
         }
      }
   }
   else
   {
      //if the source was already added, iterate through the _sourceList to find it
      for(SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
      {
         if(static_cast<AtlasSource*>(sitr->get())->getID() == textureID ) //lint !e1774
         {
            curSource = static_cast<AtlasSource*>(sitr->get()); //lint !e1774
            break;
         }
      }

      //if the texture is used in repeat mode, then the size of atlas == size of the texture
      if(curSource)
      {
         returnTextureID = textureID;
         //          if(osg::Texture::REPEAT == textureParam.wrapMode)
         //          {
         //             textureParam.atlasWidth = curSource->_image->s();
         //             textureParam.atlasHeight = curSource->_image->t();
         //          }
         atlas = new WallpaperAtlas(textureParam.atlasWidth, textureParam.atlasHeight, textureParam.textureMargin);  //height, width, margin
         if(atlas.valid())
         {
            _atlasList.push_back(atlas);
            atlas->setID(textureParam.atlasID);
            atlas->setSubID(0);
            _atlasMap[textureParam.atlasID].push_back(atlas);
            rebuildAtlas(curSource, atlas, frameNumber);
         }
      }
   }
   storeTextureCoordinates(curSource);
   texCoords = curSource->getTextureCoords();
   return atlas->_texture;
}

void me_TextureBatchingStrategyAtlas::setTexCoordPointer( osg::State& state, osg::Array* tcArray, unsigned int vertOffset )
{
   osg::Vec2Array* tcv2 = static_cast<osg::Vec2Array*>(tcArray); //lint !e1774
   if(!tcv2->empty()) state.setTexCoordPointer( 0, 2, GL_FLOAT, 0, (&(tcv2->front())+ vertOffset) );
}

me_TextureBatchingStrategyAtlas::AtlasSource* me_TextureBatchingStrategyAtlas::WallpaperAtlas::getExistingSource( unsigned int textureID )
{
   me_TextureBatchingStrategyAtlas::AtlasSource* existingSource = 0;
   for(Optimizer::TextureAtlasBuilder::SourceList::iterator sitr = _sourceList.begin(); sitr != _sourceList.end(); ++sitr)
   {
      me_TextureBatchingStrategyAtlas::AtlasSource* currSource = static_cast<me_TextureBatchingStrategyAtlas::AtlasSource*>(sitr->get()); //lint !e1774
      if( currSource->getID() == textureID)//lint !e1774
      {
         existingSource = currSource;//lint !e1774
         break;
      }
   }
   return existingSource;
}
