
/*
 * ecnr-object.c
 *
 *
 * *************************************************
 * qwa-object.c
 *
 * Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 *
 *************************************************
 * Author : Patrick Rey
 * Date: 27.08.2013
 * description: porting from qwa-alsa.c to G3g
 */

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>


#include "sse.h"
#include "sse_err.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
//#include "ecnr-common-defs.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"
#include "ecnr-audio-rts-handler.h"
#include "ecnr-audio-recording.h"
#include "ecnr-datapool-access.h"
#include "ecnr-engine-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-ioctrl.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR
#include "trcGenProj/Header/ecnr-object.c.trc.h"
#endif

#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

/*
 * Add the ecnr-introspection.h header file to the list of library header files
 * maintained by lint.  It isn't really a library header file, but it is an
 * automatically generated file that we can't modify, so it should be treated
 * like a library header file.
 */
/*lint +libh("ecnr-introspection.h") */


#include "ecnr-introspection.h"

/*
 * Wait for up to one second for the audio thread to respond to a start or
 * stop command.  This typically takes far less than a second since the audio
 * thread is run at a high real-time scheduling priority, but it can take
 * a significant amount of time if console debugging diagnostics are enabled
 * due to the time spent writing to the console.  It can also take an extra
 * long time to enable streaming the first time because some code from shared
 * libraries may have to be read from storage.
 */

#define T_BUG



struct ecnr_signal
{
  const gchar *signal_name;
  EcnrSignal signal_enum;
  guint signal_id;
};



/*//prio2 lint fix Info 715
#define ECNR_SIGNAL(signal,signal_name, signal_id) \
  { \
    .signal_enum = sname \
    .signal_name = signal_name, \
    .signal_id = signal_id, \
  }


#define ECNR_SIGNAL_DEF(name) \
    ECNR_SIGNAL(name, #name,0)
*/


static  struct ecnr_signal   my_signal[E_SIGNAL_COUNT];
static EcnrObject*       my_qobj;

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_frame_info(EcnrObject * qobj,
           guint * FrameCounter,
           gint * VoicePresent,
           gint * AboveBackgroundNoise,
           gint * PercentClipping,
           gint * InstantaneousSNR,
           gint * SmoothVNR,
           gint * ValidOutputCountdown, GError ** error)
{
    gboolean ret = TRUE;


    /*
    * Suppress lint Info 715 "Symbol not referenced" message regarding
    * the formal parameter 'qobj'.
    */
    /*lint -efunc(715,ecnr_object_ecnr_frame_info) */

    pr_message("no supported by ecnr Engine \n");

    /*due to Info 715: prio2:*/
    *FrameCounter = 0;
    *VoicePresent = 0;
    *AboveBackgroundNoise = 0;
    *PercentClipping = 0;
    *InstantaneousSNR = 0;
    *SmoothVNR = 0;
    *ValidOutputCountdown = 0;
    /*due to Info 715: prio2:*/

    ret = report_ecnr_error(ECNR_ERR_OBJECT_FI_NOT_SUPPORTED, "ecnr_object_ecnr_alsa_set_wide_band", error);

    return ret;
}




#ifndef T_BUG
G_DEFINE_TYPE(EcnrObject, ecnr_object, G_TYPE_OBJECT)
#else
static void     ecnr_object_init       (EcnrObject      *self);
static void     ecnr_object_class_init (EcnrObjectClass *klass);
static gpointer ecnr_object_parent_class = NULL;


/*******************************************************************************
*
* FUNCTION: ecnr_object_class_intern_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void     ecnr_object_class_intern_init (gpointer klass)
{
  pr_debug("ecnr_object_class_intern_init \n");
  ecnr_object_parent_class = g_type_class_peek_parent (klass);
  ecnr_object_class_init ((EcnrObjectClass*) klass);
}



/*******************************************************************************
*
* FUNCTION: ecnr_object_get_type
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

GType ecnr_object_get_type (void)
{
    static volatile gsize g_define_type_id__volatile = 0;

    pr_debug("ecnr_object_get_type ENTERED \n");

    if (g_once_init_enter (&g_define_type_id__volatile))
    {
        pr_debug("g_once_init_enter ENTERED \n");
        GType g_define_type_id =
        g_type_register_static_simple (G_TYPE_OBJECT,
                                       g_intern_static_string ("EcnrObject"),
                                       sizeof (EcnrObjectClass),
                                       (GClassInitFunc) ecnr_object_class_intern_init,
                                       sizeof (EcnrObject),
                                       (GInstanceInitFunc) ecnr_object_init,
                                       (GTypeFlags)0);


        pr_debug("the type id is %lu",g_define_type_id);

        g_once_init_leave (&g_define_type_id__volatile, g_define_type_id);
    }
    pr_debug("ecnr_object_get_type EXIT \n");
    return g_define_type_id__volatile;
}

#endif


/*******************************************************************************
*
* FUNCTION: ecnr_object_install_signal_table
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_install_signal_table
(
    GObjectClass * gobj_class
)
{
    int i = 0;

    pr_debug("ENTERED");

    const gchar* signalNames[E_SIGNAL_COUNT] =
    {
    SIGNAL_ECNR_ASR_MODE_ENABLE,
    SIGNAL_ECNR_AUDIO_STATUS,
    SIGNAL_ECNR_MICRO_LEVEL_STATUS
    };

    /* Create the signals in one loop, since they all are similar
     (except for the names). */
    for (i = 0; i < (int)E_SIGNAL_COUNT; i++)
    {
        guint signalId;

        /* Most of the time you will encounter the following code without
         * comments. This is why all the parameters are documented
         * directly below.
         */

        if( i == (int)E_SIGNAL_ECNR_ASR_MODE_ENABLE)
        {
            signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                1,
                                G_TYPE_INT                       /* GType(s),   <arg name="StatusAsrMode"          type="i"    direction="out"/> */
                );

            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
            my_signal[E_SIGNAL_ECNR_ASR_MODE_ENABLE].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }
        else if( i == (int)E_SIGNAL_ECNR_AUDIO_STATUS)
        {
            signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                1,
                                G_TYPE_INT                       /* GType(s),   <arg name="ecnrAudioStatus"          type="i"    direction="out"/> */
                );

            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
            my_signal[E_SIGNAL_ECNR_AUDIO_STATUS].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }
        else if( i == (int)E_SIGNAL_ECNR_MICRO_LEVEL_STATUS)
        {
                                signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                1,
                                G_TYPE_INT                       /* GType(s),   <arg name="ecnrMicrolevelStatus"          type="u"    direction="out"/> */
            );

            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
             my_signal[E_SIGNAL_ECNR_MICRO_LEVEL_STATUS].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }
        else
        {
            pr_debug("signal not supported\n");
        }
    }
}


//#define D_TEST_PARAMETER_SET  /* Enable only if want to debug engine parameter set */

/*******************************************************************************
*
* FUNCTION: ecnr_object_get_property
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_get_property
(
    GObject*    gobj,
    guint       prop_id,
    GValue*     value,
    GParamSpec* pspec
)
{
    gint status ;
    const struct ecnrParameter *p;
    guint table_size           = ecnr_service_get_param_table_size();
    const struct ecnrParameter *table   = ecnr_service_get_param_table();

    if(!table)
    {
        pr_debug("parameter table NULL");
        return;
    }

    if(!table_size)
    {
        pr_debug("parameter table size(%d)",table_size);
        return;
    }

    p = table + prop_id;


    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_object_get_property]: ENTERED"));


    if (prop_id < 1 || prop_id > table_size)
    {
        G_OBJECT_WARN_INVALID_PROPERTY_ID(gobj, prop_id, pspec);
        return;
    }

    status = ecnr_service_fi_get_param((guint16)p->ecnr_data_id, p->ecnr_type, value);


    if(status)
        pr_debug("Get of property(%d) fails with err(%d)", prop_id, status);

#ifdef D_TEST_PARAMETER_SET
    /* test */
    GArray*     array = NULL;
    guint i = 0;
    array = g_value_get_boxed (value);
    guchar *data_read = &g_array_index(array, guchar, 0);
    for (i = 0; i < array->len; i++)
    {
        pr_debug("array[%d]:%d\n", i, *data_read++);
    }
#endif

    pr_message("EXIT");

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_set_property
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_set_property
(
    GObject*    qobj,
    guint       prop_id,
    const GValue*    value,
    GParamSpec* pspec
)
{
    int status  = ECNR_ERR_OK;

    const struct ecnrParameter *p;
    guint table_size           = ecnr_service_get_param_table_size();
    const struct ecnrParameter *table   = ecnr_service_get_param_table();

    if(!table)
    {
        pr_debug("parameter table NULL");
        return;
    }

    if(!table_size)
    {
        pr_debug("parameter table size(%d)",table_size);
        return;
    }

    p = table + prop_id;


    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_object_get_property]: ENTERED"));


    if (prop_id < 1 || prop_id > table_size)
    {
        G_OBJECT_WARN_INVALID_PROPERTY_ID(qobj, prop_id, pspec);
        return;
    }

    pr_message("Property_id -> %d , DataType -> %d ",(guint32)p->ecnr_data_id,(guint)p->ecnr_type);

    status=ecnr_service_fi_set_param(p->ecnr_data_id,p->ecnr_type, value);

    pr_message("EXIT");
    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_install_property_table
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_install_property_table
(
    GObjectClass*               gobj_class
)
{

    guint prop_id;
    GParamSpec *pspec;
    const struct ecnrParameter *p;
    const struct ecnrParameter *table   = ecnr_service_get_param_table();
    guint table_size           = ecnr_service_get_param_table_size();

    pr_debug("ENTERED");

    if(!table)
    {
        pr_debug("parameter table NULL");
        return;
    }
    if(!table_size)
    {
        pr_debug("parameter table size(%d)",table_size);
        return;
    }

    pr_debug("%d properties", table_size - 1);


    for (prop_id = 1, p = table + 1; prop_id < table_size; prop_id++, p++)
    {
        pr_debug("installing property %d: %s", prop_id, p->ecnr_name);
        switch (p->parm_type)
        {
            case G_TYPE_INT:
            {
                gint pmin, pmax, pdefault;

                if (p->ecnr_type < ECNR_TYPE_I32)
                {
                    pmin = G_MINSHORT;
                    pmax = G_MAXSHORT;
                }
                else
                {
                    pmin = G_MININT;
                    pmax = G_MAXINT;
                }
                pdefault = 0;
                pspec = g_param_spec_int(p->prop_name, p->ecnr_name, p->ecnr_name, pmin, pmax, pdefault, p->flags);

                pr_debug("p->prop_name(%s),p->apl_name(%s), pmin(%d), pmax(%d), p->flags(%d), pspec(%p)", p->prop_name,
                                                                                               p->ecnr_name,
                                                                                               pmin,
                                                                                               pmax,
                                                                                               p->flags,
                                                                                               pspec);


                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            case G_TYPE_UINT:
            {
                guint pmin, pmax, pdefault;

                if (p->ecnr_type < ECNR_TYPE_U32)
                    pmax = G_MAXUSHORT;
                else
                    pmax = G_MAXUINT;
                pmin = 0;
                pdefault = 0;
                pspec = g_param_spec_uint(p->prop_name, p->ecnr_name, p->ecnr_name, pmin, pmax, pdefault, p->flags);
                pr_debug("p->prop_name(%s),p->apl_name(%s), pmin(%d), pmax(%d), p->flags(%d), pspec(%p)", p->prop_name,
                                                                                               p->ecnr_name,
                                                                                               pmin,
                                                                                               pmax,
                                                                                               p->flags,
                                                                                               pspec);
                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            case G_TYPE_BOXED:
            {
                GType etype;

                if (p->element_type == G_TYPE_INT)
                    etype = DBUS_TYPE_G_INT_ARRAY;
                else
                    etype = DBUS_TYPE_G_UINT_ARRAY;
                pspec = g_param_spec_boxed(p->prop_name, p->ecnr_name, p->ecnr_name, etype, p->flags);
                pr_debug("p->prop_name(%s),p->apl_name(%s), etype(%lu),p->flags(%d), pspec(%p)", p->prop_name,
                                                                                               p->ecnr_name,
                                                                                               etype,
                                                                                               p->flags,
                                                                                               pspec);
                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            default:
                pr_error("Unsupported GType %lu in property table", p->parm_type);
                break;
        }
    }

    pr_debug("EXIT");

    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_finalize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_finalize(GObject * gobj)
{
    pr_message("");

    ecnr_service_destroy();

    (G_OBJECT_CLASS(ecnr_object_parent_class)->finalize) (gobj);

  my_qobj = NULL;

}

/*******************************************************************************
*
* FUNCTION: ecnr_object_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_init(EcnrObject * qobj)
{
    pr_debug("ecnr_object_init \n");
    ecnr_service_init();

    my_qobj = qobj;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_class_init
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_object_class_init(EcnrObjectClass * qobj_class)
{
  GObjectClass *gobj_class = G_OBJECT_CLASS(qobj_class);

  pr_debug("");

  dbus_g_object_type_install_info(ECNR_TYPE_OBJECT,
          &dbus_glib_ecnr_object_object_info);

  gobj_class->finalize = ecnr_object_finalize;
  gobj_class->get_property = ecnr_object_get_property;
  gobj_class->set_property = ecnr_object_set_property;
  ecnr_object_install_signal_table(gobj_class);
  ecnr_object_install_property_table(gobj_class);

}



/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_version
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_version(EcnrObject * qobj, GArray ** version,
            gchar ** version_string,
            gchar ** version_comment, GError ** error)
{
    int status = (int)ECNR_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_version]: ENTERED"));

    status =  ecnr_service_fi_get_version(version, version_string, version_comment);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_version", error);

    pr_message("EXIT (%d)" , status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_version]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_status_message
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_status_message
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gchar ** status_message,
    GError ** error
)
{
    int status = ECNR_ERR_OK;
    gboolean ret;
    gboolean state = TRUE;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_status_message]: ENTERED"));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_get_status_message", error);
        return ret;
    }
    else
    {
        *status_message = g_malloc0(SSE_MAX_STRING_SIZE + 1);
        if(*status_message)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
          //TODO: get status message from QWA library
          status = ECNR_ERR_NO_RESULT;
#else
            status = ecnr_sse_get_Status_Message((char*) *status_message);
#endif
        else
            status = ECNR_ERR_NULL_POINTER;

    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_status_message", error);
    if (!ret)
    {
        if(*status_message)
            g_free(*status_message);
        *status_message = NULL;
    }
    else
        pr_debug("%s", *status_message);

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
  return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_initialize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_initialize(EcnrObject * qobj, guchar ecnrAppId, gint ecnrConfigurationId, GError ** error)
{
    int status = (int)ECNR_ERR_OK;
    gboolean ret;

    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_initialize]: ENTERED with AppId[%d], CfgId[%d]", ecnrAppId, ecnrConfigurationId));


    /* call service interface */
    status = ecnr_service_fi_initialize(ecnrAppId, ecnrConfigurationId);


    ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize", error);

    pr_message("EXIT with error code -> %d", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_initialize]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_initialize_sync
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_initialize_sync
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
    gchar** InDevices,
    gchar** OutDevices,
    GArray** DeviceProperties,
    GError ** error
)
{
    int         status = (int)ECNR_ERR_OK;
    gboolean    ret = TRUE;

    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId);
    ETG_TRACE_USR3(("[ecnr_object_ecnr_initialize_sync]: ENTERED"));

    /* create new ecnr instance*/
    status = ecnr_service_instance_create_new(ecnrAppId);
    if(status != SSE_ERR_OK)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        status = ecnr_service_do_action_initialize(ecnrConfigurationId);
    }

    /* try to remove instance */
    if(status != ECNR_ERR_OK)
        (void)ecnr_service_instance_remove(ecnrAppId);
    else
    {
        #ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_initialize();
        #endif

        if(ecnr_service_get_pass_through_mode_default_state())
            ecnr_object_ecnr_set_pass_through_mode(qobj, ECNR_APPID_TEST, TRUE, error);

#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE))  && !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
        if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
        {
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
        }
#else
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
#endif

       /***************************************************
        *   return configuration data to the caller
        */

        gboolean isSyncDataMatching = FALSE;

        /* transfer device properties list */
        *DeviceProperties = ecnr_configuration_get_device_properties(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

        /* transfer input device's list */
        *InDevices = ecnr_configuration_get_in_devices(ecnrAppId, ecnrConfigurationId, &isSyncDataMatching);

        /* transfer output device's list */
        *OutDevices = ecnr_configuration_get_out_devices(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

        if(isSyncDataMatching)
            ecnr_configuration_sync_data_clear();

        /*start audio thread */
        //(void)ecnr_service_audio_create();

    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_initialize_sync", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_initialize_sync]: EXIT"));
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_reset
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_reset(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret = TRUE;
    int status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;

    pr_message("ENTERED with appid(%d)\n", ecnrAppId );

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_reset", error);
        return ret;
    }
    else
        status = ecnr_engine_handler_reset();

    ret = report_ecnr_error(status, "ecnr_object_ecnr_reset", error);

    pr_message("EXIT with error (%d)", status );

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_destroy
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_destroy(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret = TRUE;
    int status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;


    pr_message("ENTERED with appid(%d)\n", ecnrAppId );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_destroy]: ENTERED with AppId[%d]", ecnrAppId));


    /* call service interface */
    status = ecnr_service_fi_destroy(ecnrAppId);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_destroy", error);

    pr_message("EXIT with err %d", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_destroy]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_audio_ext
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_audio_ext
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    guchar StartMode,
    GError ** error
)
{
    gboolean    ret;
    int         status = (int)SSE_ERR_OK;


    pr_message("ENTERED with appid(%d) with StartMode(%d)\n", ecnrAppId, StartMode);
    ETG_TRACE_USR3(("[ecnr_object_ecnr_start_audio_ext]: ENTERED with AppId[%d] and StartMode[%d]", ecnrAppId, StartMode));

    /* call service function */
    status = ecnr_service_fi_start_audio(ecnrAppId, StartMode);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_start_audio_ext", error);

    pr_message("EXIT with err %d", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    ETG_TRACE_USR3(("[ecnr_object_ecnr_start_audio_ext]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_audio
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean    ret;
    int         status = (int)SSE_ERR_OK;


    pr_message("ENTERED with appid(%d)\n", ecnrAppId);
    ETG_TRACE_USR3(("[ecnr_object_ecnr_start_audio]: ENTERED with AppId[%d]", ecnrAppId));

    /* call service function */
    status = ecnr_service_fi_start_audio(ecnrAppId, START_MODE_WAIT_STREAMING_RESULT);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_start_audio", error);

    pr_message("EXIT with err %d", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    ETG_TRACE_USR3(("[ecnr_object_ecnr_start_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    (void*)qobj;
    return ret;
}
/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_stop_audio
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_stop_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    gboolean ret;
    gboolean stopped, state;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED with appid(%d)\n", ecnrAppId );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_stop_audio]: ENTERED with AppId[%d]", ecnrAppId));

    status = ecnr_service_fi_stop_audio(ecnrAppId);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_stop_audio", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_stop_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_mic_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_mic_clip_stats
(
    EcnrObject * qobj,
    gint Channel,
    guint * ClipFrames,
    guint * ClipCount,
    gint * MaxAmp,
    gint * MinAmp,
    GError ** error
)
{
    gboolean ret;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_mic_clip_stats]: ENTERED "));

    status = ecnr_service_fi_mic_clip_stats(Channel, ClipFrames, ClipCount, MaxAmp, MinAmp);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_mic_clip_stats", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_mic_clip_stats]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_ref_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_ref_clip_stats
(
    EcnrObject * qobj,
    guint * ClipFrames,
    guint * ClipCount,
    gint * MaxAmp,
    gint * MinAmp,
    GError ** error
)
{
    gboolean ret;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_ref_clip_stats]: ENTERED "));

    status = ecnr_service_fi_ref_clip_stats(ClipFrames, ClipCount, MaxAmp, MinAmp);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_ref_clip_stats", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_ref_clip_stats]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}




/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_recv_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_recv_clip_stats
(
    EcnrObject * qobj,
    guint * ClipFrames,
    guint * ClipCount,
    gint * MaxAmp,
    gint * MinAmp,
    GError ** error
)
{
    gboolean ret;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_recv_clip_stats]: ENTERED "));

    status = ecnr_service_fi_recv_clip_stats(ClipFrames, ClipCount, MaxAmp, MinAmp);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_recv_clip_stats", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_recv_clip_stats]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_mic_out_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_mic_out_clip_stats
(
    EcnrObject * qobj,
    guint * ClipFrames,
    guint * ClipCount,
    gint * MaxAmp,
    gint * MinAmp,
    GError ** error
)
{
    gboolean ret;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_mic_out_clip_stats]: ENTERED "));

    status = ecnr_service_fi_mic_out_clip_stats(ClipFrames, ClipCount, MaxAmp, MinAmp);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_mic_out_clip_stats", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_mic_out_clip_stats]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_recv_out_clip_stats
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_recv_out_clip_stats
(
    EcnrObject * qobj,
    guint * ClipFrames,
    guint * ClipCount,
    gint * MaxAmp,
    gint * MinAmp,
    GError ** error
)
{
    gboolean ret;
    int status = (int)SSE_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_recv_out_clip_stats]: ENTERED "));

    status = ecnr_service_fi_recv_out_clip_stats(ClipFrames, ClipCount, MaxAmp, MinAmp);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_recv_out_clip_stats", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_recv_out_clip_stats]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_configuration
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_configuration
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
    GError ** error
)
{
    int status = ECNR_ERR_OK;
    gboolean ret;


    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_configuration]: ENTERED with AppId[%d], CfgId[%d]", ecnrAppId, ecnrConfigurationId));

    status = ecnr_service_fi_set_configuration(ecnrAppId, ecnrConfigurationId);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration", error);

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_configuration]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_configuration_sync
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_configuration_sync
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    gint ecnrConfigurationId,
  gchar** InDevices,
  gchar** OutDevices,
    GArray** DeviceProperties,
    GError ** error
)
{
  int             status = ECNR_ERR_OK;
  gboolean        ret;
  gboolean        state = TRUE;
  gboolean        CreateNewSession = TRUE; //FALSE;
  gboolean        RetSyncInfo = FALSE;
  struct ecnr_instance_object* my_inst;

    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration_sync", error);
        return ret;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        /* check whether streaming is currently active */
        if(ecnr_alsa_get_audio_run_state())
        {
            pr_warning("audio streaming is active, ecnrStopAutio should be called before\n");
            status = ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING;
        }
        else
        {
            /* retrieve current instance */
            my_inst = ecnr_service_instance_active_get();

            /* check whether config Id can be configured  */
            if(my_inst)
            {
                if (my_inst->ecnr_is_initialized)
                {
                    RetSyncInfo = TRUE;

                    if(my_inst->ecnr_current_config_id == ecnrConfigurationId)
                    {
                        pr_warning("sse already initialized with Config %d", ecnrConfigurationId);
                        CreateNewSession = FALSE;
                        status = 0;
                    }
                    else
                        (void)ecnr_service_do_action_destroy();
                }
            }

            if (CreateNewSession)
            {
                status = ecnr_service_do_action_initialize(ecnrConfigurationId);

                /* try to remove instance */
                if(status != ECNR_ERR_OK)
                    (void)ecnr_service_instance_remove(ecnrAppId);

                else
                {
                    if(ecnr_service_get_pass_through_mode_default_state())
                        ecnr_object_ecnr_set_pass_through_mode(qobj, ECNR_APPID_TEST, TRUE, error);

#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE))  && !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
        if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
        {
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
        }
#else
          ecnr_object_ecnr_set_send_gain_delta(qobj,ecnrAppId,ecnr_service_get_mic_gain_level(),error);
#endif

                    /*start audio thread */
                    //(void)ecnr_service_audio_create();
                }
            }


            if(RetSyncInfo)
            {
                /***************************************************
                 *   return configuration data to the caller
                 */

                gboolean isSyncDataMatching = FALSE;

                /* transfer device properties list */
                *DeviceProperties = ecnr_configuration_get_device_properties(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

                /* transfer input device's list */
                *InDevices = ecnr_configuration_get_in_devices(ecnrAppId, ecnrConfigurationId, &isSyncDataMatching);

                /* transfer output device's list */
                *OutDevices = ecnr_configuration_get_out_devices(ecnrAppId,ecnrConfigurationId, &isSyncDataMatching);

                if(isSyncDataMatching)
                    ecnr_configuration_sync_data_clear();

            }
        }
    }

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_configuration_sync", error);

    pr_message("EXIT with error (%d)", status );
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_ecnr_configuration
*
* DESCRIPTION: This function retrives the current active dataset name(opeth file name)
*          by reading configuration data file and
*          current active AppID, ConfigID, EcnrID and CRC of the of the dataset.
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_ecnr_configuration
(
    EcnrObject * qobj,
    gchar ** DatasetName,
    GArray ** EcnrProperties,
    GError ** error
)
{
    int        status = ECNR_ERR_OK;
    gboolean    ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_ecnr_configuration]: ENTERED"));


    status = ecnr_service_fi_get_ecnr_configuration (DatasetName, EcnrProperties);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_ecnr_configuration", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}



/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_debug_data
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_debug_data(EcnrObject * qobj, guchar ecnrDebugMode, GError ** error)
{
    int         status  = ECNR_ERR_OK;
    gboolean    ret     = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_start_debug_data]: ENTERED"));

    status = ecnr_service_fi_start_debug_data(ecnrDebugMode);
    ret = report_ecnr_error(status, "ecnr_sse_debug_interface_start_session", error);

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_end_debug_data
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

gboolean ecnr_object_ecnr_end_debug_data(EcnrObject * qobj,guchar ecnrDebugMode, GError ** error)
{
    int                             status = ECNR_ERR_OK;
    gboolean                        ret = TRUE;;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");

    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        ret = report_ecnr_error(ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE, "ecnr_object_ecnr_end_debug_data", error);
        return ret;
    }

    status = ECNR_ERR_OBJECT_FI_NOT_SUPPORTED;
    ret = report_ecnr_error(status, "ecnr_object_ecnr_end_debug_data", error);

    pr_debug("EXIT");

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_send_mute_switch
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_send_mute_switch(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrSendMuteSwitch, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_send_mute_switch]: called with AppId[%d], MuteState[%d]", ecnrAppId, ecnrSendMuteSwitch));

    status = ecnr_service_fi_set_send_mute_switch(ecnrAppId, ecnrSendMuteSwitch);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_mute_switch", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_send_gain_delta
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_send_gain_delta(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrLevel, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_send_gain_delta]: ENTERED with AppId[%d], level[%d]", ecnrAppId, ecnrLevel));

    status = ecnr_service_fi_set_send_gain_delta(ecnrAppId, ecnrLevel);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_send_gain_delta", error);

    pr_message("EXIT with error (%d)", status );

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_mic_gain_level
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_mic_gain(EcnrObject * qobj, gint16 MicGainLevel, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_mic_gain]: called with level[%d]", MicGainLevel));

    status = ecnr_service_fi_set_mic_gain(MicGainLevel);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_mic_gain", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_pass_through_mode
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_pass_through_mode(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrPassThroughMode, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;

    status = ecnr_service_fi_set_pass_through_mode(ecnrAppId, ecnrPassThroughMode);

    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_pass_through_mode", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_nr_floor
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_nr_comb_floor(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrNRCombFloor, GError ** error)
{
    int         status = ECNR_ERR_OK;
    gboolean    ret     = TRUE;

    pr_message("ENTERED\n");

    status = ecnr_service_fi_set_nr_comb_floor(ecnrAppId, ecnrNRCombFloor);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_nr_comb_floor", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_echo_cancellation
*
* DESCRIPTION: This function enable/disable the  EC- echo cancellation feature
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_echo_cancellation(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrECState, GError ** error)
{
    int         status  = ECNR_ERR_OK;
    gboolean    ret     = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_echo_cancellation]: called with AppId[%d], AECstate[%d]", ecnrAppId, EcnrECState));

    status = ecnr_service_fi_set_eecho_cancellation(ecnrAppId, EcnrECState);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_echo_cancellation", error);

    pr_message("EXIT with error (%d)", status );

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_set_noise_reduction
*
* DESCRIPTION: This function enable/disable the  NR- noise reduction feature
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_set_noise_reduction(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrNRState, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    int iswitch = (int)qwaOff;
#else
    int lmode = (int)sseOff;
#endif

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_set_noise_reduction]: called with AppId[%d], NRstate[%d]", ecnrAppId, EcnrNRState));

    status = ecnr_service_fi_set_noise_reduction(ecnrAppId, EcnrNRState);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_set_noise_reduction", error);

    pr_message("EXIT with error (%d)", status );

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_device
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_device
(
    EcnrObject * qobj,
    guchar ecnrAppId,
    guint AlsaDeviceSelect,
    gchar * AlsaDeviceName,
    GError ** error
)
{
    int         status  = ECNR_ERR_OK;
    gboolean    ret     = TRUE;


    pr_message("ENTERED App(%d) set device(%s) to channel(%d)\n", ecnrAppId, AlsaDeviceName,AlsaDeviceSelect);
    ETG_TRACE_USR3(("[ecnr_object_ecnr_alsa_set_device]: called with AppId[%d], DevType[%d], DevName[%s]", ecnrAppId, AlsaDeviceSelect, AlsaDeviceName));

    status = ecnr_service_fi_alsa_set_device (ecnrAppId, AlsaDeviceSelect, AlsaDeviceName);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device", error);

    pr_message("EXIT with error (%d)", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_device_properties
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
extern  gboolean ecnr_object_ecnr_alsa_set_device_properties
(
    EcnrObject * qobj,
    guchar  ecnrAppId,
    guint   AlsaDeviceSelect,
    guchar   AlsaDeviceProperty,
    guint  PropertyValue,
    GError ** error
)
{
    int status = ECNR_ERR_OK;;
    gboolean ret = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_alsa_set_device_properties]: ENTERED"));


    status = ecnr_service_fi_alsa_set_device_properties(ecnrAppId, AlsaDeviceSelect, AlsaDeviceProperty, PropertyValue);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_device_properties", error);

    pr_message("EXIT with error (%d)", status );

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}
/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_thread_priority
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_thread_priority(EcnrObject * qobj, guchar ecnrAppId, gint priority, GError ** error)
{
    int         status  = ECNR_ERR_OK;
    gboolean    ret     = TRUE;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_alsa_set_thread_priority]: ENTERED"));

    status = ecnr_service_fi_alsa_set_thread_priority(ecnrAppId, priority);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_thread_priority", error);

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_alsa_set_wide_band
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_alsa_set_wide_band(EcnrObject * qobj, guchar ecnrAppId, GError ** error)
{
    int status = ECNR_ERR_OK;
    gboolean ret = TRUE;
    gboolean state = TRUE;

    pr_debug("ENTERED");

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_wide_band", error);
        return ret;
    }

    status = ECNR_ERR_OBJECT_FI_NOT_SUPPORTED;

    ret = report_ecnr_error(status, "ecnr_object_ecnr_alsa_set_wide_band", error);

    pr_debug("EXIT");

    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    return ret;
}



extern GMainLoop *ecnr_main_loop;

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_terminate
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_terminate(EcnrObject * qobj, GError ** error)
{
    pr_message("");

    if(ecnr_main_loop)
        g_main_loop_quit(ecnr_main_loop);
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    (void*)qobj;
    (void*)error;
    return TRUE;
}

/******************************************************************************
 * FUNCTION     : ecnr_object_emit_signal_ecnr_micro_level_status
 * DESCRIPTION  : emits the microphone levels via DBUS signal
 * PARAMETER    : EcnrMicroLevel -> Microphone levels to be emitted
 * RETURN VALUE : returns error code on failure of emit signal else 0
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
int ecnr_object_emit_signal_ecnr_micro_level_status(guint32 EcnrMicroLevel)
{
    int     err = ECNR_ERR_OK;
    guint   signalId;
    pr_debug("ENTERED\n");

    signalId =  my_signal[E_SIGNAL_ECNR_MICRO_LEVEL_STATUS].signal_id;

    if(my_qobj)
    {
        g_signal_emit( my_qobj, signalId, 0, EcnrMicroLevel);

        /* This is the simplest way of emit signals. */
        pr_message("emit microphone level -> %u\n",EcnrMicroLevel);
        ETG_TRACE_USR4(("[ecnr_object_emit_signal_ecnr_micro_level_status]: emit microphone level = %d\n",EcnrMicroLevel));
    }
    else
    {
        err = ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
        pr_critical("!! microphone level signal emit failed with error(%d)\n",err);
    }

    pr_debug("EXIT\n");
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_emit_signal_ecnr_asr_mode_enable
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_object_emit_signal_ecnr_asr_mode_enable(gint EcnrAsrMode)
{
    int             err = ECNR_ERR_OK;
    guint           signalId;

    pr_debug("ENTERED");

    signalId =  my_signal[E_SIGNAL_ECNR_ASR_MODE_ENABLE].signal_id;

    /* This is the simplest way of emitting signals. */
    if(my_qobj)
    {
        g_signal_emit( my_qobj, signalId, 0, EcnrAsrMode);
        pr_debug("emit signal(%d) with EcnrAsrMode(%d)\n", signalId, EcnrAsrMode );
    }


    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_emit_signal_ecnr_asr_mode_enable
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_object_emit_signal_ecnr_audio_status(gint audiostatus)
{
    int             err = ECNR_ERR_OK;
    guint           signalId;

    pr_debug("ENTERED");

    signalId =  my_signal[E_SIGNAL_ECNR_AUDIO_STATUS].signal_id;

    /* This is the simplest way of emitting signals. */
    if(my_qobj)
    {
        g_signal_emit( my_qobj, signalId, 0, audiostatus);
        pr_debug("emit signal(%d) with audiostatus(%d)\n", signalId, audiostatus );
    }


    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_engine_parameter
*
* DESCRIPTION: This function initialize the  Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_get_engine_parameter
(
    EcnrObject*     qobj,
    guint           param_id,
    GArray**        data,
    GError**        error
)
{
    int         status = (int)ECNR_ERR_OK;
    gboolean    ret;


    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_engine_parameter]: ENTERED"));

    status = ecnr_service_fi_get_engine_param(param_id, data);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_get_engine_parameter", error);

    pr_message("EXIT");

    (void*)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_start_itc_test
*
* DESCRIPTION: This function initialize the  Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_object_ecnr_start_itc_test
(
    EcnrObject * qobj,
    gint32 ITCValue,
    gint32* rspITCValue,
    guchar* TestResult,
    gint32* TestErrorCode,
    GError ** error
)
{
    int         status = (int)ECNR_ERR_OK;
    gboolean    ret;

    pr_message("ENTERED: ECNR ITC value received -> %x",ITCValue );

    status =  ecnr_service_fi_start_itctest(ITCValue,rspITCValue,TestResult,TestErrorCode);
    ret = report_ecnr_error(status, "ecnr_object_ecnr_start_itc_test", error);

    pr_message("EXIT");

    (void*)qobj;
    return ret;
}
