#include "taginfo.h"
#include "taginfo_internal.h"

using namespace TagInfo;

#include <string>
#include <glib.h>

#ifndef GST_TAG_SHOW_EPISODE_NUMBER
#define GST_TAG_SHOW_EPISODE_NUMBER "show-episode-number"
#endif

/*
 *  @todo :
 *  - test use cases with  TAGINFO_USE_GST_DISCOVERER
 *  - remove legacy implementation with pipeline once new approach is released
 *  - remove different pipelines used with http src or filesrc handling as GstDiscoverer can handle both internally
 *  - extend new tags retrieval (like album etc) only if required by any taginfo user
 *  - Audio or Video detection can be simplified with availability of respective audio/video stream in topology
 *  - remove DEBUG switches and add ETG traces for error cases
 *  - test iapps usecases and APIs (e.g media type detection , remote uri properties etc)
*/


/*enable to use GstDiscoverer APIs to retrieve tags/properties*/
#define GST_DISCOVERER_TIMEOUT_SECS 2
#define GST_DISCOVERER_HTTP_TIMEOUT_SECS 30


//#define DEBUG
/* macro can be enabled for debugging purpose to view error logs in console*/
#ifdef  ERROR_LOG
#define ERROR_LOG

#define ERROR(msg) \
        fprintf(stderr, "Error : %s:%d ::%s () \t", __FILE__, __LINE__,__FUNCTION__);\
        fprintf(stderr,"%s\n",msg);

#else
#define ERROR(msg)
#endif


GstInfo::GstInfo(const String &filename):Info(filename) {

    /* Initialising the video tag properties */
    audiocodec=String::null;
    videocodec=String::null;
    duration = 0;
    episode = 0;
    copyright=String::null;
    encoder=String::null;
    languagecode=String::null;
    container_format=String::null;

    pipelineElts.pPipeline = NULL;
    pipelineElts.pSource = NULL;
    pipelineElts.pDemuxer = NULL;
    pipelineElts.bAudioPresent=false;
    pipelineElts.bVideoPresent=false;
    pipelineElts.pAudioConvert=NULL;
    pipelineElts.pVideoConvert=NULL;

    pipelineData.uBitrate=0;
    pipelineData.uFramerate=0;
    pipelineData.uHeight=0;
    pipelineData.uWidth=0;
    pipelineData.uProfile=0;
    pipelineData.pProfile=String::null;
    pipelineData.pLevel=0;
    pipelineData.uMpegVersion=0;
    ishttpsrc=false;
    error_code = TAGINFO_DISCOVERER_ERROR;
}

GstInfo::~GstInfo()
{
}

bool GstInfo::read(void)
{
    bool  result= true;
    if(!Init())
    {
        if (!ExtractMetaData())
        {
            ERROR(("failed to extract video"));
            result=false;
        }
    }
    else
    {
        ERROR(("Video extractor init failed"));
        result=false;
    }
    return result;
}

int GstInfo::Init()
{    
    int result=0;
    GError* vGerror = NULL;
    gboolean isInitialized=FALSE;

    isInitialized = gst_init_check(NULL, NULL,&vGerror);
    if(FALSE == isInitialized)
    {
        ERROR(("Gstreamer initialization failed"));
        result=-1;
        g_error_free(vGerror);
    }
#ifndef TAGINFO_USE_GST_DISCOVERER
    else
    {
        char *url_scheme= NULL;
        url_scheme=g_uri_parse_scheme(file_name.toCString(false));
        if(url_scheme)
        {
            if(!strcmp(url_scheme,"http"))
            {
                if (CreateHttpSrcPipeline())
                {
                    ERROR(("Pipeline  creation failed"));
                    result=-1;
                }
                ishttpsrc=true;
            }
            g_free(url_scheme);
        }
        else
        {
            if (CreateFileSrcPipeline())
            {
                ERROR(("Pipeline  creation failed"));
                result=-1;
            }
        }

    }
#endif
    return result;
}

#ifndef TAGINFO_USE_GST_DISCOVERER

int GstInfo::CreateFileSrcPipeline()
{
    int l_iResult = 0;

    GstElement *l_pTypeFind = NULL;

    if (NULL != pipelineElts.pPipeline)
    {
        gst_object_unref(pipelineElts.pPipeline);
        pipelineElts.pPipeline = NULL;
    }

    pipelineElts.pPipeline = gst_pipeline_new("Video MetaData Extractor");

    if (NULL == pipelineElts.pPipeline)
    {
        ERROR(("CreatePipeline : Pipeline Creation failed"));
        l_iResult = -1;
    }

    pipelineElts.pSource = gst_element_factory_make("filesrc", "source");

    if (!GST_IS_ELEMENT(pipelineElts.pSource))
    {
        ERROR(("CreatePipeline : Source Element is not created"));
        l_iResult = -1;
    }

    l_pTypeFind = gst_element_factory_make("typefind", "typefind");

    if (!GST_IS_ELEMENT(l_pTypeFind))
    {
        ERROR(("CreatePipeline : typefind is not created"));
        l_iResult = -1;
    }
    gst_bin_add_many(GST_BIN (pipelineElts.pPipeline), pipelineElts.pSource, l_pTypeFind, NULL);
    gst_element_link(pipelineElts.pSource, l_pTypeFind);
    g_signal_connect(l_pTypeFind, "have-type", G_CALLBACK(&TagInfo::AddDeMuxer), (void *)this);
    return l_iResult;
}




int GstInfo::CreateHttpSrcPipeline()
{
    int l_iResult = 0;


    GstElement *l_pfakeSink = NULL;


    if (NULL != pipelineElts.pPipeline)
    {
        gst_object_unref(pipelineElts.pPipeline);
        pipelineElts.pPipeline = NULL;
    }

    pipelineElts.pPipeline = gst_pipeline_new("Video MetaData Extractor");

    if (NULL == pipelineElts.pPipeline)
    {
        ERROR(("CreatePipeline : Pipeline Creation failed"));
        l_iResult = -1;
    }

    pipelineElts.pSource = gst_element_factory_make("uridecodebin", "source");

    if (!GST_IS_ELEMENT(pipelineElts.pSource))
    {
        ERROR(("CreatePipeline : Source Element is not created"));
        l_iResult = -1;
    }

    pipelineElts.pAudioConvert = gst_element_factory_make("audioconvert","audio_convert");

    if (!GST_IS_ELEMENT(pipelineElts.pAudioConvert))
    {
        ERROR(("CreatePipeline : audio convert Element is not created"));
        l_iResult = -1;
    }

    pipelineElts.pVideoConvert = gst_element_factory_make("ffmpegcolorspace","video_convert");

    if (!GST_IS_ELEMENT(pipelineElts.pVideoConvert))
    {
        ERROR(("CreatePipeline : video convert Element is not created"));
        l_iResult = -1;
    }

    l_pfakeSink = gst_element_factory_make("fakesink","sink");

    if (!GST_IS_ELEMENT(l_pfakeSink))
    {
        ERROR(("CreatePipeline :fake sink Element is not created"));
        l_iResult = -1;
    }

    gst_bin_add_many(GST_BIN (pipelineElts.pPipeline), pipelineElts.pSource, pipelineElts.pAudioConvert, pipelineElts.pVideoConvert, l_pfakeSink,NULL);
    gst_element_link(pipelineElts.pAudioConvert, l_pfakeSink);
    gst_element_link(pipelineElts.pVideoConvert, l_pfakeSink);
    g_signal_connect (pipelineElts.pSource,"pad-added", G_CALLBACK (OnPadAddedHttpSrc), (void *)this);
    return l_iResult;
}

void TagInfo::AddDeMuxer(GstElement *f_pTypeFind, guint f_uiProbability, GstCaps *f_pCaps, gpointer f_pUserData)
{
    GstStructure *l_pCapsData = NULL;
    const gchar *l_pMimeType= NULL;
    GstInfo *l_pGstInfoPtr = (GstInfo *) f_pUserData;

    if(l_pGstInfoPtr == NULL)
    {
        ERROR(("AddDeMuxer: No user data present"));
        return;
    }

    l_pCapsData = gst_caps_get_structure(f_pCaps, 0);
    if(l_pCapsData)
    {
        l_pMimeType = gst_structure_get_name(l_pCapsData);
        if (!l_pGstInfoPtr->pipelineElts.pDemuxer)
        {
            if (!strcmp(l_pMimeType, "video/quicktime") || !strcmp(l_pMimeType, "video/mj2") || !strcmp(l_pMimeType, "audio/x-m4a")|| !strcmp(l_pMimeType, "application/x-3gp"))
            {
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("qtdemux", "demuxer");
            }
            else if (!strcmp(l_pMimeType, "video/x-msvideo"))
            {
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("avidemux", "demuxer");
            }
            else if (!strcmp(l_pMimeType, "video/x-ms-asf"))
            {
#ifdef TARGET_BUILD
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("aiurdemux", "demuxer");
#else
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("asfdemux", "demuxer");
#endif
            }
            else if (!strcmp(l_pMimeType, "video/x-flv"))
            {
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("flvdemux", "demuxer");
            }
            else if (!strcmp(l_pMimeType, "application/ogg"))
            {
                l_pGstInfoPtr->pipelineElts.pDemuxer = gst_element_factory_make("oggdemux", "demuxer");
            }
            else
            {
                gint l_iVersion = 0;
                if ((!strcmp(l_pMimeType, "video/mpegts")) || (!strcmp(l_pMimeType, "video/mpeg")) || (!strcmp(l_pMimeType, "audio/mpeg")))
                {
                    gst_structure_get_int(l_pCapsData, "mpegversion", &l_iVersion);
                    l_pGstInfoPtr->set_mpegversion(l_iVersion);

                }
            }

            if(NULL != l_pGstInfoPtr->pipelineElts.pDemuxer)
            {
                /* AddDeMuxer:Got demuxer */
                g_signal_connect(l_pGstInfoPtr->pipelineElts.pDemuxer, "pad-added", G_CALLBACK (&OnPadAddedFileSrc), f_pUserData);
                gst_element_set_state(l_pGstInfoPtr->pipelineElts.pDemuxer, GST_STATE_PAUSED);
                gst_bin_add(GST_BIN(l_pGstInfoPtr->pipelineElts.pPipeline), l_pGstInfoPtr->pipelineElts.pDemuxer);
                gst_element_link(f_pTypeFind, l_pGstInfoPtr->pipelineElts.pDemuxer);
                gst_element_set_state(l_pGstInfoPtr->pipelineElts.pPipeline, GST_STATE_PLAYING);
            }
            else
            {
                /* Demuxer not present */
                ERROR(("Demuxer not present for the given file format"));
                GstElement *l_pFakeSink;
                l_pFakeSink = gst_element_factory_make("fakesink", "fakesink");
                gst_element_set_state(l_pFakeSink, GST_STATE_PAUSED);
                gst_bin_add(GST_BIN(l_pGstInfoPtr->pipelineElts.pPipeline), l_pFakeSink);
                gst_element_link(f_pTypeFind, l_pFakeSink);
                gst_element_set_state(l_pGstInfoPtr->pipelineElts.pPipeline, GST_STATE_PLAYING);
                return;
            }

        }
        else
        {
            /* Demuxer added already */
        }
    }
}

/**
 * Callback OnPadAdded.
 */
void TagInfo::OnPadAddedFileSrc(GstElement *f_pDemuxerElement, GstPad *f_pPad, gpointer f_pUserData)
{
    GstElement *l_pFakeSink;
    gchar *l_pPadName = NULL ;
    gchar *l_pMimeType = NULL;
    GstPad *l_pSinkPad = NULL;
    GstCaps *l_pCaps = NULL;
    GstStructure *l_pCapsStructure = NULL;
    gint l_iVersion = 0;
    gint l_width=0,l_height=0;
    gint l_frameratenum=0,l_framerateden=0;

    GstInfo *l_pGstInfoPtr = (GstInfo *)f_pUserData;

    if(NULL == l_pGstInfoPtr)
    {
        ERROR(("No user date present on pad added"));
        return;
    }

    l_pPadName = gst_pad_get_name(f_pPad);
#ifdef ENABLE_GSTREAMER_1_0
    l_pCaps = gst_pad_query_caps(f_pPad , NULL);
#else
    l_pCaps = gst_pad_get_caps(f_pPad);
#endif
    if(l_pCaps)
    {
        l_pCapsStructure = gst_caps_get_structure(l_pCaps, 0);

        if(g_str_has_prefix (l_pPadName,"video"))
        {
            const gchar *l_pLevel=NULL ;
            const gchar *l_pProfile=NULL ;

            l_pMimeType = (gchar *) gst_structure_get_name(l_pCapsStructure);
            if (!strcmp(l_pMimeType, "video/mpeg"))
            {
                gst_structure_get_int(l_pCapsStructure, "mpegversion", &l_iVersion);
                l_pGstInfoPtr->set_mpegversion(l_iVersion);
            }
            if(!GstInfo::FetchHighProfileDataForVideoStream(l_pCapsStructure, &l_pGstInfoPtr->pipelineData))
            {
                l_pProfile=gst_structure_get_string(l_pCapsStructure,"profile");
                if(NULL != l_pProfile )
                {
                    String l_profile(l_pProfile);
                    l_pGstInfoPtr->set_profile(l_profile);
                }
            }

            l_pLevel=(gchar*)gst_structure_get_string(l_pCapsStructure,"level");
            if(NULL != l_pLevel)
            {
                l_pGstInfoPtr->set_level((float)atof((const char*)l_pLevel));
             
            }

            gst_structure_get_int(l_pCapsStructure, "width", &l_width);
            if(l_width)
            {
                l_pGstInfoPtr->set_width((int)l_width);
            }
            gst_structure_get_int(l_pCapsStructure, "height",&l_height);
            if(l_height)
            {
                l_pGstInfoPtr->set_height((int)l_height);
            }

            gst_structure_get_fraction(l_pCapsStructure, "framerate", &l_frameratenum,&l_framerateden);
            if(l_frameratenum && l_framerateden)
            {
                l_pGstInfoPtr->set_framerate((int)(l_frameratenum)/(l_framerateden));
            }

            l_pFakeSink = gst_element_factory_make("fakesink", "fakevideosink");
            l_pGstInfoPtr->pipelineElts.bVideoPresent = true;
        }
        else if (g_str_has_prefix (l_pPadName,"audio"))
        {
            l_pFakeSink = gst_element_factory_make("fakesink", "fakeaudiosink");
            l_pGstInfoPtr->pipelineElts.bAudioPresent = true;
        }
        else
        {
            /* do nothing */
        }
        gst_bin_add(GST_BIN(l_pGstInfoPtr->pipelineElts.pPipeline),l_pFakeSink);
        gst_element_set_state(l_pFakeSink, GST_STATE_PLAYING);
        l_pSinkPad = gst_element_get_static_pad(l_pFakeSink, "sink");
        gst_pad_link(f_pPad, l_pSinkPad);
        gst_object_unref(l_pSinkPad);
        gst_caps_unref(l_pCaps);
    }

    if(l_pPadName)
        g_free(l_pPadName);
}

void TagInfo::OnPadAddedHttpSrc(GstElement *f_pSrcElement, GstPad *f_pPad, gpointer f_pUserData)
{
    GstCaps *l_pPadCaps =NULL;
    GstStructure *l_pPadStructure =NULL;
    const gchar *l_pPadType =NULL;
    GstInfo *l_pGstInfoPtr = (GstInfo *) f_pUserData;

    if(NULL == l_pGstInfoPtr)
    {
        ERROR(("No user date present on pad added"));
        return;
    }

    GstPad *sink_pad_audio = gst_element_get_static_pad (l_pGstInfoPtr->pipelineElts.pAudioConvert,"sink");
    GstPad *sink_pad_video = gst_element_get_static_pad (l_pGstInfoPtr->pipelineElts.pVideoConvert,"sink");

    /* Check the new pad's type */
#ifdef ENABLE_GSTREAMER_1_0
    l_pPadCaps = gst_pad_query_caps (f_pPad , NULL);
#else
     l_pPadCaps = gst_pad_get_caps (f_pPad);
#endif
    
    l_pPadStructure = gst_caps_get_structure (l_pPadCaps,0);
    l_pPadType = gst_structure_get_name (l_pPadStructure);

    if(g_str_has_prefix (l_pPadType,"audio/x-raw")) {
        gst_pad_link (f_pPad, sink_pad_audio);
        l_pGstInfoPtr->pipelineElts.bAudioPresent=true;
    }
    else if(g_str_has_prefix (l_pPadType,"video/x-raw")) {
        gst_pad_link (f_pPad, sink_pad_video);
        l_pGstInfoPtr->pipelineElts.bVideoPresent=true;
    }
    else
    {
        /* no pads found */
    }

    /* Unreference the new pad's caps, if we got them */
    if(l_pPadCaps !=NULL)
    {
        gst_caps_unref (l_pPadCaps);
    }

    /* Unreference the sink pad */
    gst_object_unref (sink_pad_audio);
    gst_object_unref (sink_pad_video);
}
#endif

#ifdef TAGINFO_USE_GST_DISCOVERER
#ifdef DEBUG
//only for gstreamer taglist debug
static void print_tag_foreach (const GstTagList * tags, const gchar * tag,gpointer user_data)
{
    GValue val = { 0, };
    gchar *str;

    gst_tag_list_copy_value (&val, tags, tag);

    if (G_VALUE_HOLDS_STRING (&val))
    {
        str = g_value_dup_string (&val);
    }
    // ?? GST_TYPE_SAMPLE is not supported with currently used gstreamer version 0.10
    //else if(G_VALUE_TYPE (&val) == GST_TYPE_SAMPLE)
    //{
    //     str = strdup("sample");
    // }
    else
    {
        str = gst_value_serialize (&val);
    }

    printf ("Tag:%s -> %s\n", gst_tag_get_nick (tag), str);
    g_free (str);

    g_value_unset (&val);
}
#endif


void GstInfo::retrieve_stream_info (GstDiscovererStreamInfo * info)
{
#ifdef DEBUG
    GstCaps *caps = gst_discoverer_stream_info_get_caps (info);

    if (caps)
    {
        gchar *desc = NULL;
        if (gst_caps_is_fixed (caps) )
            desc = gst_pb_utils_get_codec_description (caps);
        else
            desc = gst_caps_to_string (caps);
        gst_caps_unref (caps);

        if (desc)
        {
            g_print ("StreamType (%s) ->  %s \n",gst_discoverer_stream_info_get_stream_type_nick (info), desc);
            g_free (desc);
        }
    }
#endif

    GstTagList* taglist = NULL;
    taglist = (GstTagList*)gst_discoverer_stream_info_get_tags(info);
    /*retrieve only tag which are of interest for a particular stream */
    if(GST_IS_DISCOVERER_CONTAINER_INFO(info))
    {
#ifdef DEBUG
        /*get taglist for the detected stream (container or video or audio)*/
        if(taglist)
        {
            gst_tag_list_foreach (taglist, print_tag_foreach, NULL);
        }
#endif
    }
    else if (GST_IS_DISCOVERER_VIDEO_INFO (info))
    {
        GstDiscovererVideoInfo *video_info = (GstDiscovererVideoInfo *) info;

        /*retrieve miscellanaius video properties like profile and level*/
        //const GstStructure *misc = gst_discoverer_stream_info_get_misc (info);
        /*gst_discoverer_stream_info_get_misc has failed to retrieve profile and level info*/

        /*retrieve other video properties like profile and level via its capabilities structure*/

        if(taglist)
        {
            PopulateVideoTags(taglist);
        }

        if(!ishttpsrc)
        {
            GstCaps *caps = gst_discoverer_stream_info_get_caps (info);
            if(caps)
            {
                const GstStructure *capsStruct = gst_caps_get_structure(caps, 0);
                if(capsStruct)
                {
#ifdef DEBUG
                    printf("Video caps name : %s \n",gst_structure_get_name(capsStruct));
                    printf("Video caps profile : %s \n",gst_structure_get_string(capsStruct,"profile"));
                    printf("Video caps level : %s \n",gst_structure_get_string(capsStruct,"level"));
#endif

                    const char* profile = gst_structure_get_string(capsStruct,"profile");
                    if(NULL != profile )
                    {
                        String mProfile(profile);
                        set_profile(mProfile);
                    }

                    const char *level =  gst_structure_get_string(capsStruct,"level");
                    if(NULL != level)
                    {
                        set_level((float)atof(level));
                    }
                }
            }

            /*extract and save the properties of interest*/
            set_width(gst_discoverer_video_info_get_width (video_info));
            set_height(gst_discoverer_video_info_get_height (video_info));

            guint framerate_denom = gst_discoverer_video_info_get_framerate_denom (video_info);
            if(0 != framerate_denom) {
                set_framerate(gst_discoverer_video_info_get_framerate_num (video_info)/framerate_denom);
            } else {
                set_framerate(0);
            }
            set_bitrate(gst_discoverer_video_info_get_bitrate (video_info));

        }
        pipelineElts.bVideoPresent = true;

#ifdef DEBUG
        printf("Video Width: %u\n",gst_discoverer_video_info_get_width (video_info));
        printf("Video Height: %u\n",gst_discoverer_video_info_get_height (video_info));
        printf("Video Frame rate: %u/%u\n",gst_discoverer_video_info_get_framerate_num (video_info),gst_discoverer_video_info_get_framerate_denom (video_info));
        printf("Video Bitrate: %u\n",gst_discoverer_video_info_get_bitrate (video_info));

        printf("Video Max bitrate: %u\n",gst_discoverer_video_info_get_max_bitrate (video_info));
        printf("Video Depth: %u\n",gst_discoverer_video_info_get_depth (video_info));
        printf("Video Pixel aspect ratio: %u/%u\n",gst_discoverer_video_info_get_par_num (video_info),gst_discoverer_video_info_get_par_denom (video_info));
        printf("Video Interlaced: %s\n",gst_discoverer_video_info_is_interlaced (video_info) ? "true" : "false");
        if(taglist)
        {
            gst_tag_list_foreach (taglist, print_tag_foreach, NULL);
        }
#endif
    }
    else if (GST_IS_DISCOVERER_AUDIO_INFO (info))
    {
        //GstDiscovererAudioInfo *audio_info = (GstDiscovererAudioInfo *) info;
        //set_bitrate(gst_discoverer_audio_info_get_bitrate (audio_info));
        /*extract and save tags of interest from audio stream*/
        if(taglist)
        {
            PopulateAudioTags(taglist);
        }
        pipelineElts.bAudioPresent = true;

#ifdef DEBUG
        if(taglist)
        {
            gst_tag_list_foreach (taglist, print_tag_foreach, NULL);
        }
        GstDiscovererAudioInfo *audio_info = (GstDiscovererAudioInfo *) info;

        const gchar *ctmp = gst_discoverer_audio_info_get_language (audio_info);
        printf("Language: %s\n",ctmp ? ctmp : "<unknown>");
        printf("Channels: %u\n",gst_discoverer_audio_info_get_channels (audio_info));
        printf("Depth: %u\n",gst_discoverer_audio_info_get_depth (audio_info));
        printf("Sample rate: %u\n",gst_discoverer_audio_info_get_sample_rate (audio_info));
        printf("Bitrate: %u\n",gst_discoverer_audio_info_get_bitrate (audio_info));
        printf("Max bitrate: %u\n",gst_discoverer_audio_info_get_max_bitrate (audio_info));
#endif
    }
}

void GstInfo::read_stream_info (GstDiscovererStreamInfo * info)
{
    if (!info)
        return;

    /*read the current stream*/
    retrieve_stream_info (info);

    /*fetch next sibling stream*/
    GstDiscovererStreamInfo *next = gst_discoverer_stream_info_get_next (info);
    if (next)
    {
        /*fetch the details of composite stream*/
#ifdef DEBUG
        printf("\n Got sibling stream\n");
#endif
        read_stream_info (next);
        gst_discoverer_stream_info_unref (next);
    }
    else if (GST_IS_DISCOVERER_CONTAINER_INFO (info))
    {
        /*get child streams information*/
        GList *tmp, *streams;
        streams =  gst_discoverer_container_info_get_streams (GST_DISCOVERER_CONTAINER_INFO(info));
        for (tmp = streams; tmp; tmp = tmp->next)
        {
#ifdef DEBUG
            printf("\n Got child stream\n");
#endif
            GstDiscovererStreamInfo *tmpinf = (GstDiscovererStreamInfo *) tmp->data;
            read_stream_info (tmpinf);
        }
        gst_discoverer_stream_info_list_free (streams);
    }
}

bool GstInfo::ExtractMetaData()
{
    GError *err = NULL;
    char *url_scheme= NULL;
    gchar *uri = NULL;
    GstDiscoverer *discoverer = NULL;

    url_scheme=g_uri_parse_scheme(file_name.toCString(false));
    if(url_scheme)
    {
        if(!strcmp(url_scheme,"http"))
        {
            uri = (gchar*) file_name.toCString(false);
            ishttpsrc = true;
            discoverer = gst_discoverer_new (GST_DISCOVERER_HTTP_TIMEOUT_SECS * GST_SECOND, &err);
            if (discoverer == NULL)
            {
#ifdef DEBUG
                g_print ("Error initializing: %s\n", err->message);
#endif
                g_error_free (err);
                return false;
            }
        }
        g_free(url_scheme);
    }
    else
    {
        uri = g_filename_to_uri (file_name.toCString(false), NULL, NULL); //todo : avoid this for remote uris: use g_uri_parse_scheme
        discoverer = gst_discoverer_new (GST_DISCOVERER_TIMEOUT_SECS * GST_SECOND, &err);
        if (discoverer == NULL)
        {
    #ifdef DEBUG
            g_print ("Error initializing: %s\n", err->message);
    #endif
            g_error_free (err);
            return false;
        }
    }

    if(uri != NULL)
    {
#ifdef DEBUG
    printf("\n Taginfo:Entered URI : %s\n",uri);
#endif
        GstDiscovererInfo *info = gst_discoverer_discover_uri (discoverer,uri,&err);
        if (info == NULL)
        {
            g_error_free (err);
            return false;
        }
        else
        {
            GstDiscovererResult discoveryResult = gst_discoverer_info_get_result (info);

            switch (discoveryResult)
            {
                case GST_DISCOVERER_OK:
                    /*Ok*/
                    break;
                case GST_DISCOVERER_TIMEOUT:
                {
                    printf ("GstDiscoverer timed out while discovering the info\n");
                    error_code = TAGINFO_DISCOVERER_TIMEOUT;
                    break;
                }
                case GST_DISCOVERER_BUSY:
                {
                    printf ("GstDiscoverer is busy \n");
                    error_code = TAGINFO_DISCOVERER_BUSY;
                    break;
                }

                case GST_DISCOVERER_URI_INVALID:
                {
                    printf("input uri is not valid\n");
                    error_code = TAGINFO_DISCOVERER_URI_INVALID;
                    break;
                }
                case GST_DISCOVERER_ERROR:
                {
                    printf("Error discovering:  %s\n", err->message);
                    error_code = TAGINFO_DISCOVERER_ERROR;
                    break;
                }

                case GST_DISCOVERER_MISSING_PLUGINS:
                {
                    printf ("Missing plugins\n");
                    error_code = TAGINFO_DISCOVERER_MISSING_PLUGINS;
                    break;
                }
            }

            if(discoveryResult != GST_DISCOVERER_OK)
            {
                gst_discoverer_info_unref (info);
                /*failed to retrieve the information due to one of the reason specified by GstDiscovererResult*/
                return false;
            }

            /*retrieve and set duration*/
            set_duration(gst_discoverer_info_get_duration(info));

            /*go deep and extract all available streams in the tree and its properties */
            GstDiscovererStreamInfo *sinfo;
            if ((sinfo = gst_discoverer_info_get_stream_info (info)))
            {
                /*reads the stream tree like container -> audio,video,subtitle etc*/
                read_stream_info(sinfo);
                gst_discoverer_stream_info_unref (sinfo);
            }

#ifdef DEBUG

            /*retrieve overall taglist*/
            GstTagList *tags = NULL;
            if ((tags = (GstTagList *)gst_discoverer_info_get_tags (info)))
            {
                gst_tag_list_foreach (tags, print_tag_foreach, NULL);
            }
#endif

            gst_discoverer_info_unref (info);
            if(ishttpsrc)
            {
                if((true == audiocodec.isNull()) || (true == videocodec.isNull()))
                {
                    error_code = TAGINFO_DISCOVERER_NO_CODEC;
                    return false;
                }
            }
            return true;
        }
    }
    return false;
}

void GstInfo::PopulateVideoTags(GstTagList *f_pTags)
{
    if(NULL != f_pTags)
    {
        /* video name */
        String l_cVideoName = String::null;
        GetStringTagValue(f_pTags, GST_TAG_TITLE, l_cVideoName);
        if( true != l_cVideoName.isNull())
            set_title(l_cVideoName);

        /* video codec */
        String l_cVideoCodec = String::null;
        GetStringTagValue(f_pTags, GST_TAG_VIDEO_CODEC, l_cVideoCodec);
        if( true != l_cVideoCodec.isNull())
            set_videocodec(l_cVideoCodec);

        /* audio codec */
        String l_cAudioCodec = String::null;
        GetStringTagValue(f_pTags, GST_TAG_AUDIO_CODEC, l_cAudioCodec);
        if( true != l_cAudioCodec.isNull())
            set_audiocodec(l_cAudioCodec);

        /* container format */
        String l_cContainerFormat = String::null;
        GetStringTagValue(f_pTags, GST_TAG_CONTAINER_FORMAT, l_cContainerFormat);
        if( true != l_cContainerFormat.isNull())
            set_containerformat(l_cContainerFormat);

        /* file duration */
        guint64 l_cFileDuration = 0;
        if(duration == 0)
        {
            GetUInt64TagValue(f_pTags, GST_TAG_DURATION,(guint64&)l_cFileDuration);
            if(l_cFileDuration)
                set_duration((guint64)l_cFileDuration);

        }
        /* bitrate */
        unsigned int l_uBitRate = 0;
        GetUIntTagValue(f_pTags, GST_TAG_BITRATE, l_uBitRate);
        if(0 != l_uBitRate)
            set_bitrate(l_uBitRate);
    }
}




void GstInfo::PopulateAudioTags(GstTagList *f_pTags)
{

    String l_cTitle = "";
 
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT
    guint64 l_uiDuration = 0;
#else
    unsigned long long int  l_uiDuration = 0;
#endif

    //Title
    GetStringTagValue(f_pTags, GST_TAG_TITLE, l_cTitle);
    if( true != l_cTitle.isNull())
        set_title(l_cTitle);

    //Total Playtime
    if(duration == 0)
    {
        GetUInt64TagValue(f_pTags, GST_TAG_DURATION, l_uiDuration);
        if(l_uiDuration)
            duration = l_uiDuration;
    }

    /* audio codec */
    String l_cAudioCodec = String::null;
    GetStringTagValue(f_pTags, GST_TAG_AUDIO_CODEC, l_cAudioCodec);
    if( true != l_cAudioCodec.isNull())
        set_audiocodec(l_cAudioCodec);
}
#endif

#ifndef TAGINFO_USE_GST_DISCOVERER
// legacy way of extraction : remove once GstDiscoverer usage above is validated
bool GstInfo::ExtractMetaData()
{
    GstStateChangeReturn l_gstSReturn;
    GstTagList *mergelist=NULL;
    gboolean bError = false;
    bool result=false;

    String l_sFileNameTrimmed(file_name.stripWhiteSpace());


    if(ishttpsrc)
        g_object_set(pipelineElts.pSource, "uri", l_sFileNameTrimmed.toCString(false), NULL);
    else
    {
        g_object_set(pipelineElts.pSource, "location", l_sFileNameTrimmed.toCString(false), NULL);
    }

    l_gstSReturn = gst_element_set_state(GST_ELEMENT (pipelineElts.pPipeline), GST_STATE_PLAYING);
    if (GST_STATE_CHANGE_SUCCESS != l_gstSReturn)
    {
        ERROR(("ExtractMetaData : State change failed. Aborting"));
        return false;
    }

    while(l_gstSReturn)
    {
        GstMessage *l_pMsg=NULL;
        if(ishttpsrc)
            l_pMsg = gst_bus_timed_pop_filtered(GST_ELEMENT_BUS (pipelineElts.pPipeline), GST_SECOND * 20,(GstMessageType) (GST_MESSAGE_ERROR | GST_MESSAGE_ASYNC_DONE ));
        else
            l_pMsg = gst_bus_timed_pop_filtered(GST_ELEMENT_BUS (pipelineElts.pPipeline), GST_SECOND * 2,(GstMessageType) ( GST_MESSAGE_ASYNC_DONE | GST_MESSAGE_TAG | GST_MESSAGE_ERROR));

        if (!l_pMsg)
        {
            ERROR(("gst_bus_timed_pop_filtered returned NULL"));
            bError = true;
            break;
        }
        else if (GST_MESSAGE_TYPE (l_pMsg) == GST_MESSAGE_ASYNC_DONE) /* error or async_done */
        {
            gst_message_unref(l_pMsg);
            break;
        }
        else if (GST_MESSAGE_TYPE (l_pMsg) == GST_MESSAGE_ERROR)
        {
            bError = true;
            gst_message_unref(l_pMsg);
            break;
        }
        else
        {
            GstTagList *l_pNewTags;
            gst_message_parse_tag(l_pMsg, &l_pNewTags);
            if (mergelist)
            {
                mergelist = gst_tag_list_merge(mergelist, l_pNewTags, GST_TAG_MERGE_KEEP);
                gst_tag_list_free(l_pNewTags);
            }
            else
            {
                mergelist = l_pNewTags;
            }
            gst_message_unref(l_pMsg);
        }
    }

    if(ishttpsrc)
    {
        gst_element_set_state(GST_ELEMENT (pipelineElts.pPipeline), GST_STATE_NULL);
        gst_object_unref(GST_OBJECT (pipelineElts.pPipeline));
        ishttpsrc=false;
        return true;
    }

    if ((bError)|| (NULL == mergelist))
    {
        ERROR(("Unable to read metadata"));
        gst_element_set_state(GST_ELEMENT (pipelineElts.pPipeline), GST_STATE_NULL);
        gst_object_unref(GST_OBJECT (pipelineElts.pPipeline));
        return true;
    }
    else
    {
        TagInfo::MediaType f_eMediaType ;
        GuessFileType(l_sFileNameTrimmed, mergelist, f_eMediaType);
        if(0 == FillTag(mergelist, f_eMediaType))
        {
            gst_tag_list_free(mergelist);
            mergelist = NULL;
            result=true;
        }
    }
    l_gstSReturn = gst_element_set_state(GST_ELEMENT (pipelineElts.pPipeline), GST_STATE_NULL);

    gst_object_unref (GST_OBJECT (pipelineElts.pPipeline));
    return result;
}

void GstInfo::PopulateVideoTags(GstTagList *f_pTags)
{

    if(NULL != f_pTags)
    {
        /* video name */
        String l_cVideoName = String::null;
        GetStringTagValue(f_pTags, GST_TAG_TITLE, l_cVideoName);
        if( true != l_cVideoName.isNull())
            set_title(l_cVideoName);

        /* video codec */
        String l_cVideoCodec = String::null;
        GetStringTagValue(f_pTags, GST_TAG_VIDEO_CODEC, l_cVideoCodec);
        if( true != l_cVideoCodec.isNull())
            set_videocodec(l_cVideoCodec);

        /* audio codec */
        String l_cAudioCodec = String::null;
        GetStringTagValue(f_pTags, GST_TAG_AUDIO_CODEC, l_cAudioCodec);
        if( true != l_cAudioCodec.isNull())
            set_audiocodec(l_cAudioCodec);

        /* container format */
        String l_cContainerFormat = String::null;
        GetStringTagValue(f_pTags, GST_TAG_CONTAINER_FORMAT, l_cContainerFormat);
        if( true != l_cContainerFormat.isNull())
            set_containerformat(l_cContainerFormat);

        /* file duration */
        gint64 l_cFileDuration = 0;

        GetUInt64TagValue(f_pTags, GST_TAG_DURATION,(guint64&)l_cFileDuration);
        if(0 == l_cFileDuration)
        {
            GstFormat gst_format=GST_FORMAT_TIME;
#ifdef ENABLE_GSTREAMER_1_0
		    gst_element_query_duration(pipelineElts.pPipeline, gst_format, &l_cFileDuration);
#else
            gst_element_query_duration(pipelineElts.pPipeline, &gst_format, &l_cFileDuration);
#endif
        }
        set_duration((gint64)l_cFileDuration);

        /* Episode Number */
        unsigned int l_uEpisodeNumber = 0;
        GetUIntTagValue(f_pTags, GST_TAG_SHOW_EPISODE_NUMBER, l_uEpisodeNumber);
        if( 0 != l_uEpisodeNumber)
            set_episode(l_uEpisodeNumber);

        /* bitrate */
        unsigned int l_uBitRate = 0;
        GetUIntTagValue(f_pTags, GST_TAG_BITRATE, l_uBitRate);
        if(0 != l_uBitRate)
            set_bitrate(l_uBitRate);

        /* track_number */
        unsigned int l_uTrackNumber = 0;
        GetUIntTagValue(f_pTags, GST_TAG_TRACK_NUMBER, l_uTrackNumber);
        if(0 != l_uTrackNumber)
            set_tracknumber(l_uTrackNumber);

        /* composer */
        String l_cComposer = String::null;
        GetStringTagValue(f_pTags, GST_TAG_COMPOSER, l_cComposer);
        if( true != l_cComposer.isNull())
            set_composer(l_cComposer);

        /* artist */
        String l_cArtist = String::null;
        GetStringTagValue(f_pTags, GST_TAG_ARTIST, l_cArtist);
        if( true != l_cArtist.isNull())
            set_artist(l_cArtist);

        /* album */
        String l_cAlbum = String::null;
        GetStringTagValue(f_pTags, GST_TAG_ALBUM, l_cAlbum);
        if( true != l_cAlbum.isNull())
            set_album(l_cAlbum);

        /* genre */
        String l_cGenre = String::null;
        GetStringTagValue(f_pTags, GST_TAG_GENRE, l_cGenre);
        if( true != l_cGenre.isNull())
            set_genre(l_cGenre);

        /* language code */
        String l_cLanguageCode = String::null;
        GetStringTagValue(f_pTags, GST_TAG_LANGUAGE_CODE, l_cLanguageCode);
        if( true != l_cLanguageCode.isNull())
            set_languagecode(l_cLanguageCode);

        /* copyrigths */
        String l_cCopyrights = String::null;
        GetStringTagValue(f_pTags, GST_TAG_COPYRIGHT, l_cCopyrights);
        if( true != l_cCopyrights.isNull())
            set_copyrights(l_cCopyrights);

        /* encoder */
        String l_cEncoder = String::null;
        GetStringTagValue(f_pTags, GST_TAG_ENCODER, l_cEncoder);
        if( true != l_cEncoder.isNull())
            set_encoder(l_cEncoder);
    }
}

void GstInfo::PopulateAudioTags(GstTagList *f_pTags)
{
    String l_cArtist = "";
    String l_cGenre = "";
    String l_cTitle = "";
    String l_cAlbum = "";
    String l_cComposer = "";
    unsigned int l_uiValue = 0;
    guint64  l_uiDuration = 0;

    //Title
    GetStringTagValue(f_pTags, GST_TAG_TITLE, l_cTitle);
    if( true != l_cTitle.isNull())
        set_title(l_cTitle);

    //Artist
    GetStringTagValue(f_pTags, GST_TAG_ARTIST, l_cArtist);
    if( true != l_cArtist.isNull())
        set_artist(l_cArtist);

    //Genre
    GetStringTagValue(f_pTags, GST_TAG_GENRE, l_cGenre);
    if( true != l_cGenre.isNull())
        set_genre(l_cGenre);

    //Album
    GetStringTagValue(f_pTags, GST_TAG_ALBUM, l_cAlbum);
    if( true != l_cAlbum.isNull())
        set_album(l_cAlbum);

    //Composer
    GetStringTagValue(f_pTags, GST_TAG_COMPOSER, l_cComposer);
    if( true != l_cComposer.isNull())
        set_composer(l_cComposer);

    //Tracknumber
    GetUIntTagValue(f_pTags, GST_TAG_TRACK_NUMBER, l_uiValue);
    if(l_uiValue)
        set_tracknumber(l_uiValue);

#ifndef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT // thoemel: for 64 bit compiler and libtag 1.8
    //Total Playtime
    GetUInt64TagValue(f_pTags, GST_TAG_DURATION, l_uiDuration);
    if(l_uiDuration)
        duration = l_uiDuration;
#endif
}



void GstInfo::GuessFileType(String f_cFilePath, GstTagList *f_pTagList, MediaType& f_eMediaType)
{
    String l_cName = "";
    String l_cFileExtension = f_cFilePath.substr(f_cFilePath.rfind(".") + 1);

    if (l_cFileExtension == "mp4" || l_cFileExtension == "3gpp" || l_cFileExtension == "flv" || l_cFileExtension == "divx"
            || l_cFileExtension == "3gp" || l_cFileExtension == "wmv" || l_cFileExtension == "avi" || l_cFileExtension == "m4v"
                    || l_cFileExtension == "mpg" || l_cFileExtension == "mpeg" || l_cFileExtension == "mov" )
    {
        if (!pipelineElts.bVideoPresent)
        {            
            f_eMediaType = MEDIA_TYPE_MUSIC_FILE;
        }
        else
        {           
            f_eMediaType = MEDIA_TYPE_VIDEO;
        }
    }    
}

/**
 *
 * This function fetch video codec data to find it's profie data
 *
 * @param: GstStructrue pointer
 * @param: pointer to MessageLoopData
 * @return: true: if succeeded to fetch profile data
 *          false: if not succeeded to fetch profile data
 */
bool GstInfo::FetchHighProfileDataForVideoStream(GstStructure *f_pCapsStructure, MessageLoopData *f_pMsgLoopData)
{
    const GValue *l_pValue, *l_pAvcc;
    const gchar *l_pStreamformat = NULL;
    const guint8 *l_pdata;
    GstBuffer *l_pbuf = NULL;

    bool l_bFunctionalityCheck = false;

    if (f_pCapsStructure != NULL)
    {
        if (gst_structure_has_name(f_pCapsStructure, "video/x-h264"))
        {
            l_pValue = gst_structure_get_value(f_pCapsStructure, "stream-format");
            if ((NULL != l_pValue) && G_VALUE_HOLDS_STRING (l_pValue))
            {
                l_pStreamformat = g_value_get_string(l_pValue);
                if (NULL != l_pStreamformat)
                {
                    if (0 == strcmp(l_pStreamformat, "avc"))
                    {
                        l_pAvcc = gst_structure_get_value(f_pCapsStructure, "codec_data");
                        if (l_pAvcc != NULL)
                        {
                            if(GST_VALUE_HOLDS_BUFFER (l_pAvcc))
                            {
                                l_pbuf = gst_value_get_buffer (l_pAvcc);
                                if (l_pbuf != NULL)
                                {
#ifdef ENABLE_GSTREAMER_1_0
									unsigned int l_size = gst_buffer_get_size(l_pbuf);
									if(l_size >= 2)
									{
									  /* l_pdata ;= GST_BUFFER_DATA (l_pbuf);
                                        f_pMsgLoopData->uProfile = l_pdata[1];*/ //TODO to replace this GST_BUFFER_DATA
                                        l_bFunctionalityCheck = true;
									}
#else 
                                    unsigned int l_size = GST_BUFFER_SIZE(l_pbuf);
                                    if(l_size >= 2)
                                    {
                                        l_pdata = GST_BUFFER_DATA (l_pbuf);
                                        f_pMsgLoopData->uProfile = l_pdata[1];
                                        l_bFunctionalityCheck = true;
                                    }
#endif
                                    else
                                    {
                                        /* FetchHighProfileDataForVideoStream: Codec_data buffer size is less than to 2 */
                                    }
                                }
                                else
                                {
                                    /* FetchHighProfileDataForVideoStream: Codec_data buffer fetch fail */
                                }
                            }
                            else
                            {
                                /* FetchHighProfileDataForVideoStream: Codec data dose not hold buffer */
                            }
                        }
                        else
                        {
                            /* FetchHighProfileDataForVideoStream: l_pAvcc is NULL, Codec data missing or not available */
                        }
                    }
                    else
                    {
                        /* FetchHighProfileDataForVideoStream: Stream format is not AVC \n */
                    }
                }
                else
                {
                    /* FetchHighProfileDataForVideoStream: Stream format is NULL */
                }
            }
            else
            {
                /* FetchHighProfileDataForVideoStream: stream-format field is missing or not available */
            }
        }
        else
        {
            /* FetchHighProfileDataForVideoStream: structure name is not x-h264 */
        }
    }
    else
    {
        /* FetchHighProfileDataForVideoStream: f_pCapsStructure structure is NULL */
    }

    return l_bFunctionalityCheck;
}
#endif


int GstInfo::FillTag(GstTagList *f_pTags, MediaType f_eMediaType)
{
    int result=0;
    if (NULL != f_pTags)
    {
        switch (f_eMediaType)
        {
            case MEDIA_TYPE_VIDEO:
            {
                PopulateVideoTags(f_pTags);
                break;
            }
            case MEDIA_TYPE_MUSIC_FILE:
            {
                PopulateAudioTags(f_pTags);
                break;
            }
            default:
                break;
        }
    }
    else
    {
        ERROR(("Tag list is empty and no tags recorded "));
        result=-1;
    }

    return result;
}

void GstInfo::GetStringTagValue(GstTagList *f_pTagList, const gchar *f_cTagName, String& f_rTagValue)
{
    gchar *l_cTagValue = NULL;
    gboolean l_bReturn = gst_tag_list_get_string(f_pTagList, f_cTagName, &l_cTagValue);

    if (NULL != l_cTagValue)
    {
        if (l_bReturn && l_cTagValue[0] != '\0')
        {
            f_rTagValue=format("%s",l_cTagValue);
        }
        g_free(l_cTagValue);
    }
}

void GstInfo::GetUIntTagValue(GstTagList *f_pTagList, const gchar *f_cTagName, unsigned int& f_uiTagValue)
{
    guint l_uiTagValue = 0;

    gst_tag_list_get_uint(f_pTagList, f_cTagName, &l_uiTagValue);
    f_uiTagValue = l_uiTagValue;
}

void GstInfo::GetUInt64TagValue(GstTagList *f_pTagList, const gchar *f_cTagName,guint64& f_uiTagValue)
{
    guint64 l_uiTagValue = 0;

    gst_tag_list_get_uint64(f_pTagList, f_cTagName, &l_uiTagValue);
    f_uiTagValue = l_uiTagValue;
}

#ifndef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT // thoemel: for 64 bit compiler and libtag 1.8
void GstInfo::set_duration(guint64 length) {
#else
    void GstInfo::set_duration(long long unsigned int length) {
#endif
    duration = length;
    changedflag |= CHANGED_DURATION_TAG;
}

void GstInfo::set_episode(int episode_number) {
    episode = episode_number;
    changedflag |= CHANGED_EPISODE_TAG;
}

void GstInfo::set_bitrate(int bit_rate) {
    bitrate = bit_rate;
    changedflag |= CHANGED_BITRATE_TAG;
}

void GstInfo::set_videocodec(String codec) {
    videocodec = codec;
    changedflag |= CHANGED_VIDEOCODEC_TAG;
}

String GstInfo::get_videocodec() const {
    return videocodec;
}


void GstInfo::set_audiocodec(String codec) {
    audiocodec = codec;
    changedflag |= CHANGED_AUDIOCODEC_TAG;
}

String GstInfo::get_audiocodec() const {
    return audiocodec;
}


void GstInfo::set_containerformat(String format) {
    container_format = format;
    changedflag |= CHANGED_CONTAINERFORMAT_TAG;
}

String GstInfo::get_containerformat() const {
    return container_format;
}

void GstInfo::set_languagecode(String language_code) {
    languagecode = language_code;
    changedflag |= CHANGED_LANGUAGECODE_TAG;
}

String GstInfo::get_languagecode() const {
    return languagecode;
}

void GstInfo::set_copyrights(String file_copyright) {
    copyright = file_copyright;
    changedflag |= CHANGED_COPYRIGHT_TAG;
}

String GstInfo::get_copyrights() const {
    return copyright;
}

void GstInfo::set_encoder(String file_encoder) {
    encoder = file_encoder;
    changedflag |= CHANGED_ENCODER_TAG;
}

String GstInfo::get_encoder() const {
    return encoder;
}

int GstInfo::get_width() const{
    return pipelineData.uWidth;
}

void GstInfo::set_width(int video_width) {
    pipelineData.uWidth = video_width;
    changedflag |= CHANGED_WIDTH_TAG;
}

int GstInfo::get_height() const{
    return pipelineData.uHeight;
}

void GstInfo::set_height(int video_height) {
    pipelineData.uHeight = video_height;
    changedflag |= CHANGED_HEIGHT_TAG;
}

int GstInfo::get_mpegversion() const{
    return pipelineData.uMpegVersion;
}

void GstInfo::set_mpegversion(int mpeg_version) {
    pipelineData.uMpegVersion = mpeg_version;
    changedflag |= CHANGED_MPEGVERSION_TAG;
}

unsigned long long int GstInfo::get_duration() const{
    return duration;
}

int GstInfo::get_episode() const{
    return episode;
}

void GstInfo::set_profile(String profile_info) {
    pipelineData.pProfile = profile_info;
    changedflag |= CHANGED_PROFILE_TAG;
}

void GstInfo::set_level(float level_info) {
    pipelineData.pLevel = level_info;
    changedflag |= CHANGED_LEVEL_TAG;
}

String GstInfo::get_profile() const {
    return pipelineData.pProfile;
}

float GstInfo::get_level() const{
    return pipelineData.pLevel;
}

int GstInfo::get_framerate() const{
    return pipelineData.uFramerate;
}

void GstInfo::set_framerate(int frame_rate) {
    pipelineData.uFramerate = frame_rate;
    //changedflag |= CHANGED_FRAMERATE_TAG;
}

MediaType GstInfo::get_media_type()const{
    if(pipelineElts.bVideoPresent)
        return MEDIA_TYPE_VIDEO;
    else if(pipelineElts.bAudioPresent)
        return MEDIA_TYPE_MUSIC_FILE;
    else
        return MEDIA_TYPE_UNKNOWN;
}

ErrorCode GstInfo::get_error_code(void)const{
    return error_code;
}
