/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
- Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/src/osgViewer/StatsHandler.cpp
- Implementation of some APIs to enable display of dynamically changing strings as stats in addition to integers and floating point values for debugging purposes.
- Implementation of some additional APIs to group stats based on whether they change or remain consistent across multiple views.
*/

#include "oss/stats/CustomStatsHandlerBase.h"

using namespace osgGA;
using namespace osgViewer;
using namespace mapengine::oss;

me_CustomStatsHandlerBase::me_CustomStatsHandlerBase()
: _userStatsVisibility(NO_USER_STATS)
, _userDepStatsVisibility(NO_DEPENDENT_STATS_TRACE)
, _userIndStatsVisibility(NO_INDEPENDENT_STATS_TRACE)
, _customStatsWidth(1000.0f)
, _customStatsHeight(625.0f)
, _customCharacterSize(13.0f)
, _backgroundColor(0.0f, 0.0f, 0.0f, 0.3f)
, _backgroundMargin(5.0f)
, _backgroundSpacing(3.0f)
, _staticTextColor(1.0f, 1.0f, 0.0f, 1.0f)
, _dynamicTextColor(1.0f, 1.0f, 1.0f, 1.0f)
, _updateStatsFlag(false)
, _enumToStringMaps(USER_STATS_COUNT)

{
   _customCamera = new osg::Camera;

   if(_customCamera.valid())
   {
      _customCamera->setRenderer(new Renderer(_customCamera.get()));
      _customCamera->setProjectionResizePolicy(osg::Camera::FIXED);
   }
}

bool me_CustomStatsHandlerBase::handle(const osgGA::GUIEventAdapter& ea, osgGA::GUIActionAdapter& aa)
{   
   osgViewer::View* myview = dynamic_cast<osgViewer::View*>(&aa);
   if (!myview) return false;
   osgViewer::ViewerBase* viewer = myview->getViewerBase();

   if (ea.getHandled()) return false;

   switch(ea.getEventType())
   {
   case(osgGA::GUIEventAdapter::KEYUP):
      {
         if (ea.getKey() == KEY_CUSTOM_STATS)
         {
            if (viewer->getViewerStats())
            {
               if (!_initialized)
               {
                  customSetUpHUDCamera(myview);
                  customSetUpScene(myview, 0, false);
               }
               else
               {
                  customSetUpScene(myview, 0, true);
               }
               if(ea.getKey() == KEY_CUSTOM_STATS)
               {
                  switch(_userStatsVisibility)
                  {
                  case NO_USER_STATS : 
                     _userStatsVisibility = VIEW_INDEPENDENT_STATS_VISIBLE;
                     break;
                  case VIEW_INDEPENDENT_STATS_VISIBLE :
                     _userStatsVisibility = ALL_MAP_STATS_VISIBLE;
                     break;
                  case ALL_MAP_STATS_VISIBLE :
                  default:
                     _userStatsVisibility = NO_USER_STATS;
                     break;
                  }

                  std::vector<osgViewer::View*> views;
                  viewer->getViews(views);
                  std::vector<osgViewer::View*>::iterator it;

                  switch(_userStatsVisibility)
                  {

                  case(NO_USER_STATS):
                     {
                        osg::Stats* stats = myview->getStats();
                        if (stats)
                        {
                           stats->collectStats("user_viewDep_stats", false);
                        }

                        bool flag = false;

                        for (it = views.begin(); it != views.end(); ++it)
                        {
                           if ((*it)->getStats()->collectStats("user_viewDep_stats")) flag = true;
                        }  

                        if(!flag)
                        {
                           viewer->getViewerStats()->collectStats("user_viewInd_stats", false); 
                           _updateStatsFlag = false;
                        }

                        _customCamera->setNodeMask(0x0);
                        _customSwitch->setAllChildrenOff();
                        break;
                     }

                  case(VIEW_INDEPENDENT_STATS_VISIBLE):
                     {
                        _updateStatsFlag = true;
                        viewer->getViewerStats()->collectStats("user_viewInd_stats", true);
                        _customCamera->setNodeMask(0xffffffff);
                        _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, true);
                        break;
                     }

                  case(ALL_MAP_STATS_VISIBLE):
                     {
                        osg::Stats* stats = myview->getStats();
                        if (stats)
                        {
                           stats->collectStats("user_viewDep_stats", true);
                        }
                        _customCamera->setNodeMask(0xffffffff);
                        _customSwitch->setAllChildrenOn();
                        break;
                     }
                  default:
                     break;
                  }
                  aa.requestRedraw();
               }
            }
         }
      else if (ea.getKey() == KEY_DEPENDENT_STATS)
      {
         if (viewer->getViewerStats())
         {
            if (!_initialized)
            {
               customSetUpHUDCamera(myview);
               customSetUpScene(myview, KEY_DEPENDENT_STATS, false);
            }
            else
            {
               customSetUpScene(myview, KEY_DEPENDENT_STATS, true);
            }
            std::vector<osgViewer::View*> views;
            viewer->getViews(views);
            std::vector<osgViewer::View*>::iterator it;

            switch(_userDepStatsVisibility)
            {
            case NO_DEPENDENT_STATS_TRACE : 
               _userDepStatsVisibility = VIEW_DEPENDENT_STATS_TRACE;
               break;
            case VIEW_DEPENDENT_STATS_TRACE :
               _userDepStatsVisibility = NO_DEPENDENT_STATS_TRACE;
               break;
            default:
               _userDepStatsVisibility = NO_DEPENDENT_STATS_TRACE;
               break;
            }
            switch(_userDepStatsVisibility)
            {
            case NO_DEPENDENT_STATS_TRACE:
               {
                  _updateStatsFlag = false;
                  _userStatsVisibility = NO_USER_STATS;
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", false);
                  }
                  _customSwitch->setAllChildrenOff();
                  _customCamera->setNodeMask(0x0);
                  break;
               }
            case VIEW_DEPENDENT_STATS_TRACE:
               {
                  _updateStatsFlag = true;
                  viewer->getViewerStats()->collectStats("user_viewInd_stats", false); 
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", true);
                  }
                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, false);
                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, true);
                  _customCamera->setNodeMask(0xffffffff);
                  break;
               }
            default:
               break;
            }
               aa.requestRedraw();
         }
      }
      else if(ea.getKey() == KEY_INDEPENDENT_STATS)
      {
         if (viewer->getViewerStats())
         {
            if (!_initialized)
            {
               customSetUpHUDCamera(myview);
               customSetUpScene(myview, KEY_INDEPENDENT_STATS, false);
            }
            else
            {
               customSetUpScene(myview, KEY_INDEPENDENT_STATS, true);
            }

            std::vector<osgViewer::View*> views;
            viewer->getViews(views);
            std::vector<osgViewer::View*>::iterator it;

            switch(_userIndStatsVisibility)
            {
            case NO_INDEPENDENT_STATS_TRACE : 
               _userIndStatsVisibility = VIEW_INDEPENDENT_STATS_TRACE;
               break;
            case VIEW_INDEPENDENT_STATS_TRACE :
               _userIndStatsVisibility = NO_INDEPENDENT_STATS_TRACE;
               break;
            default:
               _userIndStatsVisibility = NO_INDEPENDENT_STATS_TRACE;
               break;
            }
            switch(_userIndStatsVisibility)
            {
            case NO_INDEPENDENT_STATS_TRACE:
               {
                  _updateStatsFlag = false;
                  _userStatsVisibility = NO_USER_STATS;
                  viewer->getViewerStats()->collectStats("user_viewInd_stats", false); 
                  _customSwitch->setAllChildrenOff();
                  _customCamera->setNodeMask(0x0);
                  break;
               }
            case VIEW_INDEPENDENT_STATS_TRACE:
               {
                  _updateStatsFlag = true;
                  osg::Stats* stats = myview->getStats();
                  if (stats)
                  {
                     stats->collectStats("user_viewDep_stats", false);
                  }
                  viewer->getViewerStats()->collectStats("user_viewInd_stats", true);
                  _customSwitch->setValue(USER_VIEW_INDEPENDENT_STATS, true);
                  _customSwitch->setValue(USER_VIEW_DEPENDENT_STATS, false);
                  _customCamera->setNodeMask(0xffffffff);
                  break;
               }
            default:
               break;
            }
            aa.requestRedraw();
         }
      }
      return true;
      }
      break;

   case(osgGA::GUIEventAdapter::RESIZE):
      customSetWindowSize(ea.getWindowWidth(), ea.getWindowHeight());
      break;

   default: break;
   }
   return false;
}

void me_CustomStatsHandlerBase::customSetWindowSize(int width, int height)
{
   if (width <= 0 || height <= 0)
      return;

   _customCamera->setViewport(0, 0, width, height);
   if (fabs(height*_customStatsWidth) <= fabs(width*_customStatsHeight))
   {
      _customCamera->setProjectionMatrix(osg::Matrix::ortho2D(0.0,(double)(width*_customStatsHeight/height), 0.0, (double)(_customStatsHeight)));
   }
   else
   {
      _customCamera->setProjectionMatrix(osg::Matrix::ortho2D(0.0,(double)(_customStatsWidth),(double)(_customStatsHeight-height*_customStatsWidth/width),(double)(_customStatsHeight)));
   }
}

void me_CustomStatsHandlerBase::customSetUpHUDCamera(osgViewer::View* viewer)
{
   _customCamera->setGraphicsContext(viewer->getCamera()->getGraphicsContext());
   _customCamera->setRenderOrder(osg::Camera::POST_RENDER, 10);
   _customCamera->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
   _customCamera->setViewMatrix(osg::Matrix::identity());

   customSetWindowSize(viewer->getCamera()->getGraphicsContext()->getTraits()->width, viewer->getCamera()->getGraphicsContext()->getTraits()->height);

   _customCamera->setClearMask(0);
   _customCamera->setAllowEventFocus(false);
   _customCamera->setRenderer(new Renderer(_customCamera.get()));

   _initialized = true;
}
