/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/include/osgDB/DatabasePager
List of changes:
1. Support several different kinds of requests instead of just 'file' and 'http' requests.
2. Checking if a specific request-type is supported by the DBPager thread looking for requests to execute.
3. Support deletion of objects in a specific type of DBPager thread, instead of only using one particular thread.
4. Support different cache limits in cache-management.
5. Disable incremental-compile of GL objects completely.
*/

#ifndef MAPENGINE_TYPES_DATABASEPAGER_
#define MAPENGINE_TYPES_DATABASEPAGER_ 1

#include <osgDB/DatabasePager>
#include "oss/dbpager/TileCacheManagerBase.h"
#include "oss/dbpager/DatabasePagerConfig.h"
#include "oss/Export.h"

#define DB_PAGER_THREAD_CONFIG "DBPagerThreadConfig"

namespace mapengine {
namespace oss {

namespace DatabasePagerMemoryStates {
   enum States
   {
      MEMORYSTATE_INVALID = 0,
      MEMORYSTATE_NORMAL,
      MEMORYSTATE_HIGH,
      MEMORYSTATE_CRITICAL,
      COUNT
   };
}

class MAPENGINE_OSS_EXPORT me_DatabasePager : public osgDB::DatabasePager
{
public:

   class ThreadConfig : public osg::Referenced
   {
   public:
      ThreadConfig()
      {
         memset(_numberOfExclusiveThreads, 0, sizeof(_numberOfExclusiveThreads));
         memset(_totalNumberOfThreads, 0, sizeof(_totalNumberOfThreads));
      }
      unsigned int                               _numberOfExclusiveThreads[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
      unsigned int                               _totalNumberOfThreads[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
   protected:
      virtual ~ThreadConfig(){}
   };

   me_DatabasePager();

   /////////////////////////////////////////////////////////////////////////////////////////////////////////////////
   //
   //  SortFileRequestFunctor
   //
   struct SortFileRequestFunctorPerFrame
   {
      bool operator() (const osg::ref_ptr<DatabasePager::DatabaseRequest>& lhs,const osg::ref_ptr<DatabasePager::DatabaseRequest>& rhs) const
      {
         if (lhs->_timestampLastRequest>rhs->_timestampLastRequest) return true;
         else if (lhs->_timestampLastRequest<rhs->_timestampLastRequest) return false;
         else return (lhs->_priorityLastRequest>rhs->_priorityLastRequest);
      }
   };

   class MAPENGINE_OSS_EXPORT me_DatabaseThread : public osgDB::DatabasePager::DatabaseThread
   {
   public:
      me_DatabaseThread(DatabasePager* pager, Mode mode, const std::string& name, const std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads);
      virtual void run();
      virtual int cancel();

      std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType> _supportedWorkloads;
      unsigned int _supportedWorkloadsFlag;
   protected:
      virtual ~me_DatabaseThread(){}
   };

   struct MAPENGINE_OSS_EXPORT me_DatabaseRequest : public osgDB::DatabasePager::DatabaseRequest
   {
      me_DatabaseRequest(mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const std::string& fileName,
         const osg::FrameStamp* frameStamp, float prio);

      virtual bool isPerFrameRequest() const
      {
         bool perFrame = false;
         switch(_workloadType)
         {
         case mapengine::oss::DatabasePagerConfig::SYNCHRONOUS_PER_FRAME_REQUEST:
         case mapengine::oss::DatabasePagerConfig::HTTP_PER_FRAME_REQUEST:
            perFrame = true;
            break;
         default:
            break;
         }
         return perFrame;
      }

      unsigned int getWorkloadFlag() const { return (1 << _workloadType); }
      static unsigned int getFlag(std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType> workloads);

      mapengine::oss::DatabasePagerConfig::ThreadWorkloadType _workloadType;

   protected:
      virtual ~me_DatabaseRequest(){}
   };

   friend class me_DatabaseThread;

   /** Modify original addDatabaseThread() to create our own customDatabaseThreads */
   virtual unsigned int addDatabaseThread(DatabaseThread::Mode mode, const std::string& name);

   virtual void requestData(const std::string& fileName, osg::NodePath& nodePath,
                            float priority, const osg::FrameStamp* framestamp,
                            osg::ref_ptr<osg::Referenced>& databaseRequest,
                            mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType,
                            const osg::Referenced* options);
   
   /** Modify original requestNodeFile() to be able to post "http request" directly into the proper queue */
   virtual void requestNodeFile(const std::string& fileName, osg::NodePath& nodePath,
                                float priority, const osg::FrameStamp* framestamp,
                                osg::ref_ptr<osg::Referenced>& databaseRequest,
                                const osg::Referenced* options);

   virtual unsigned int getFileRequestListSize() const;

   /** Cancel the database pager thread(s).*/
   virtual int cancel();

   /** Set whether the database pager thread should be paused or not.*/
   /** This should've been a virtual function in the base class. But we have to over-ride it anyway.*/
   void setDatabasePagerThreadPause(bool pause);

   /** Iterate through the active PagedLOD nodes children removing
          * children which havn't been visited since specified expiryTime.
          * note, should be only be called from the update thread. */
   virtual void removeExpiredSubgraphs(const osg::FrameStamp &frameStamp);


   /** Modify original updateSceneGraph() to inject removeExpiredSubgraphsFromTileCaches() method */
   virtual void updateSceneGraph(const osg::FrameStamp& frameStamp);
   
   /** Remove unused tiles in our own tile caches */
   void removeExpiredSubgraphsFromTileCaches(const osg::FrameStamp &frameStamp, 
      const unsigned int maxMemToFreeUp, const unsigned int maxFrameAge,
      const unsigned int maxMemLimit);
   
   void addObserverToTileCache(const osg::ref_ptr<me_TileCacheManagerBase>& tileCacheManager);

   void addToChildrenToDeleteList(osgDB::DatabasePager::ObjectList& objectsToDeleteList, 
      mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType) const;

   void setPurgeFlag();

   virtual void setMemoryState(DatabasePagerMemoryStates::States memoryState);   
  
   std::vector<osg::observer_ptr<me_TileCacheManagerBase> >& getTileCacheManagers(){return _tileCacheManagers;}  
  
   std::map< unsigned int, osg::ref_ptr<DatabasePager::ReadQueue> > _requestQueues;

   unsigned int getFrameNumberOfLastRequestReset() const { return _frameNumberOfLastRequestReset; }

   void setFrameNumberOfLastRequestReset(unsigned int val) { _frameNumberOfLastRequestReset = val; }

   unsigned int getNumOfViews() const { return _numOfViews; }
   void setNumOfViews(unsigned int val) { _numOfViews = val; }

   unsigned int getMaxMemLimitPerView() const { return _maxMemLimitPerView; }
   void setMaxMemLimitPerView(unsigned int val) { _maxMemLimitPerView = val; }

   unsigned int getTargetDesiredMaximumNumberOfPagedLOD() const { return _targetDesiredMaximumNumberOfPagedLOD; }
   void setTargetDesiredMaximumNumberOfPagedLOD(unsigned int val) 
   {
      _targetDesiredMaximumNumberOfPagedLOD = val;
      osgDB::DatabasePager::setTargetMaximumNumberOfPageLOD(_targetDesiredMaximumNumberOfPagedLOD);
   }

   unsigned int getCacheLimit() const;

protected:
   unsigned int                               _numberOfExclusiveThreadsCreated[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];
   unsigned int                               _totalNumberOfThreadsCreated[mapengine::oss::DatabasePagerConfig::WORKLOAD_TYPE_COUNT];



   virtual ~me_DatabasePager();

   virtual void takeFirstRequest(osg::ref_ptr<DatabasePager::ReadQueue>& requestQueue, 
      osg::ref_ptr<DatabaseRequest>& databaseRequest, const DatabaseThread* dbPagerThread) const;

   bool isWorkloadCurrentlyAllowed(mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const unsigned int supportedWorkloadsFlag) const;

   bool determineMemoryLimits(unsigned int& maxMemToFreeUp, unsigned int& maxFrameAgeForTiles,
      unsigned int& maxMemLimit);

   osg::ref_ptr<DatabasePager::ReadQueue> getRequestQueueByFlag(unsigned int workloadsFlag) const;

   virtual bool isRequestObsolete(const DatabaseRequest* databaseRequest) const;

   void getWorkloadsToBeSupported(std::vector<mapengine::oss::DatabasePagerConfig::ThreadWorkloadType>& supportedWorkloads);

   void createRequestQueues();

   unsigned int _frameNumberOfLastRequestReset;

   bool _purgeFlag;

   std::vector<osg::observer_ptr<me_TileCacheManagerBase> > _tileCacheManagers; 
   
   DatabasePagerMemoryStates::States _memoryState;

   unsigned int _frameNumberOfLastMemoryStateCheck;

   unsigned int _maxMemLimitPerView;

   unsigned int _numOfViews;

   unsigned int _targetDesiredMaximumNumberOfPagedLOD;
};

inline void me_DatabasePager::setPurgeFlag()
{
   _purgeFlag = true;
}

inline bool me_DatabasePager::isWorkloadCurrentlyAllowed( mapengine::oss::DatabasePagerConfig::ThreadWorkloadType workloadType, const unsigned int supportedWorkloadsFlag ) const
{
   return ((1 << workloadType) & supportedWorkloadsFlag) ? true : false;
}

}
}

#endif //MAPENGINE_TYPES_DATABASEPAGER_
