/* -*-c++-*-
mapengine_oss

Copyright (C) 1998-2006 Robert Osfield
Copyright (C) 2015 Robert Bosch Car Multimedia GmbH

This library is open source and may be redistributed and/or modified under
the terms of the OpenSceneGraph Public License (OSGPL) version 0.0 or
(at your option) any later version.  The full license is in LICENSE file
included with this distribution, and on the openscenegraph.org website.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
OpenSceneGraph Public License for more details.

History:
History:
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/examples/osganimate/osganimate.cpp
Copied from https://github.com/openscenegraph/osg/blob/OpenSceneGraph-3.2.0/examples/osgthirdpersonview/osgthirdpersonview.cpp
Copied from https://github.com/gwaldron/osgearth/blob/osgearth-2.1.1/src/osgEarth/HeightFieldUtils.cpp
Copied from https://github.com/xarray/osgRecipes/blob/master/cookbook/common/CommonFunctions.cpp
Copied from https://github.com/xarray/osgRecipes/blob/master/cookbook/chapter6/ch06_06/depth_of_field.cpp
List of changes:
1. Added function to create a blank (white) texture
*/

#ifndef MAPENGINE_OSS_OSGUTILS_
#define MAPENGINE_OSS_OSGUTILS_ 1

#include <osg/Camera>
#include <osg/AnimationPath>
#include <osgViewer/CompositeViewer>
#include "oss/Export.h"

namespace mapengine {
namespace oss {

/**
* Helper to calculate the values needed for the cluster culling callback
* @param vertex The current vertex in world space to process
* @param center The center point in world space
* @param min_dot_product The new calculated min_dot_product
* @param max_cluster_culling_height The new calculated max_cluster_culling_height
* @param max_cluster_culling_radius The new calculated max_cluster_culling_radius
* @return Returns true as long as a cluster culling callback should be used for the tile
*/  
MAPENGINE_OSS_EXPORT bool calcClusterCullingValues(const osg::Vec3d& vertex,
   const osg::Vec3d& center,
   const float verticalScale,
   float& min_dot_product,
   float& max_cluster_culling_height,
   float& max_cluster_culling_radius);

/**
* Creates a visible frustum for the given camera
* @param camera The given camera
* @return The created frustum node
*/  
MAPENGINE_OSS_EXPORT osg::Node* createFrustum(osg::Camera* camera);

/**
* Creates a simple animation path for prototyping
* @param center The center point for the animation
* @param radius The radius for the animation
* @param looptime The duration for the animation
* @return The created animation path
*/
MAPENGINE_OSS_EXPORT osg::AnimationPath* createAnimationPath(const osg::Vec3& center, float radius, double looptime);

/**
* Collect all viewer stats and stream them into the provided stream
* @param compositionViewer The viewer to analyze
* @param stringStream The stream to write the data in
*/
MAPENGINE_OSS_EXPORT void collectViewerStats(osgViewer::CompositeViewer& compositionViewer, std::ostream& stream);

/**
* Creates a camera suitable for RTT
* @param buffer The buffer component to use for the camera
* @param texture The texture to bind
* @param abs Use absolute coordinate system or not
* @param renderOrderNum Render Order Number of the RTT Camera
* @param multisampleSamples No of Multisample Samples for FBO
* @return The created camera
*/  
MAPENGINE_OSS_EXPORT osg::Camera* createRTTCamera(osg::Camera::BufferComponent buffer, osg::Texture* tex, bool isAbsolute=false, int renderOrderNum=0, int multisampleSamples=0);

/**
* Creates a camera suitable for RTT
* @param buffer The buffer component to use for the camera
* @param image The image to attach to the camera
* @param abs Use absolute coordinate system or not
* @param renderOrderNum Render Order Number of the RTT Camera
* @param multisampleSamples No of Multisample Samples for FBO
* @return The created camera
*/  
MAPENGINE_OSS_EXPORT osg::Camera* createRTTCamera(osg::Camera::BufferComponent buffer, osg::Image* image, bool isAbsolute=false, int renderOrderNum = 0, int multisampleSamples=0);

/**
* Creates a screen quad suitable for RTT
* @param corner the corner of the quad
* @param width The given width
* @param width The given height
* @param width The given scale
* @return The created geode
*/  
MAPENGINE_OSS_EXPORT osg::Geode* createScreenQuad(const osg::Vec3& corner, float width, float height, float scale=1.0f);

/**
* Creates a camera suitable as a HUD display
* @param width The left anchor point
* @param width The right anchor point
* @param width The bottom anchor point
* @param width The top anchor point
* @return The created camera
*/  
MAPENGINE_OSS_EXPORT osg::Camera* createHUDCamera(double left, double right, double bottom, double top);

/**
* Creates a texture suitable for RTT
* @param width The given width
* @param width The given height
* @param depth Use texture as a depth component
* @return The created texture
*/  
MAPENGINE_OSS_EXPORT osg::Texture* createRenderTexture(int width, int height, bool depth);

/**
* Adds visible normals to the given node
* @param node The given node
* @return The created group containing the normals and the node
*/  
MAPENGINE_OSS_EXPORT osg::Group* addNormals(osg::Group* group);

/**
* Adds depth of field effect to the given node
* @return The created node for the effect
*/  
MAPENGINE_OSS_EXPORT osg::Node* addDOF(osg::Node* scene);

/**
* Creates an ellipsoid base geometry
*/  
MAPENGINE_OSS_EXPORT osg::Geometry* createEllipsoidGeometry(const osg::EllipsoidModel* ellipsoid, double outerRadius, unsigned int numLatGroups , unsigned int numOfLongGroups, bool genTexCoords);

/**
* Creates a blank texture
* param textureWidth The given width
* param textureHeight The given height
* @return The created blank texture for the effect
*/  
MAPENGINE_OSS_EXPORT osg::Texture2D* createBlankTexture(unsigned int textureWidth, unsigned int textureHeight, osg::Vec4 color = osg::Vec4());

/**
* Typedef for pairing camera and texture
*/  
typedef std::pair< osg::ref_ptr<osg::Camera>, osg::ref_ptr<osg::Texture> > RTTPair;
RTTPair createColorInput(osg::Node* scene);
RTTPair createDepthInput(osg::Node* scene);
MAPENGINE_OSS_EXPORT RTTPair createBlurPass(osg::Texture* inputTex, const osg::Vec2& dir);

}
}

#endif // MAPENGINE_OSS_OSGUTILS_
