#include "html.h"
#include "box.h"
#include "html_tag.h"
#include <document.h>

// TODO: Find more generic trace switch for litehtml
//#define ENABLE_LITEHTML_DEBUG_TRACE

litehtml::box_type litehtml::block_box::get_type()
{
    return box_block;
}

int litehtml::block_box::height()
{
    return m_element->height();
}

int litehtml::block_box::width()
{
    return m_element->width();
}

void litehtml::block_box::add_element(const element::ptr &el)
{
    m_element = el;
    el->m_box = this;
}

void litehtml::block_box::finish(bool last_box)
{
    if(!m_element) return;
    m_element->apply_relative_shift(m_box_right - m_box_left);
}

bool litehtml::block_box::can_hold(const element::ptr &el, white_space ws)
{
    if(m_element || el->is_inline_box())
    {
        return false;
    }
    return true;
}

bool litehtml::block_box::is_empty()
{
    if(m_element)
    {
        return false;
    }
    return true;
}

int litehtml::block_box::baseline()
{
    if(m_element)
    {
        return m_element->get_base_line();
    }
    return 0;
}

void litehtml::block_box::get_elements( elements_vector& els )
{
    els.push_back(m_element);
}

int litehtml::block_box::top_margin()
{
    if(m_element && m_element->collapse_top_margin())
    {
        return m_element->m_margins.top;
    }
    return 0;
}

int litehtml::block_box::bottom_margin()
{
    if(m_element && m_element->collapse_bottom_margin())
    {
        return m_element->m_margins.bottom;
    }
    return 0;
}

void litehtml::block_box::y_shift( int shift )
{
    m_box_top += shift;
    if(m_element)
    {
        m_element->m_pos.y += shift;
    }
}

void litehtml::block_box::new_width( int left, int right, elements_vector& els )
{

}

//////////////////////////////////////////////////////////////////////////

litehtml::box_type litehtml::line_box::get_type()
{
    return box_line;
}

int litehtml::line_box::height()
{
    return m_height;
}

int litehtml::line_box::width()
{
    return m_width;
}

void litehtml::line_box::add_element(const element::ptr &el)
{
    el->m_skip  = false;
    el->m_box   = 0;
    bool add    = true;
    if( (m_items.empty() && el->is_white_space()) || el->is_break() )
    {
        el->m_skip = true;
    } else if(el->is_white_space())
    {
        if (have_last_space())
        {
            add = false;
            el->m_skip = true;
        }
    }

    if(add)
    {
        el->m_box = this;
        m_items.push_back(el);

        if(!el->m_skip)
        {
            int el_shift_left   = el->get_inline_shift_left();
            int el_shift_right  = el->get_inline_shift_right();

            el->m_pos.x = m_box_left + m_width + el_shift_left + el->content_margins_left();
            el->m_pos.y = m_box_top + el->content_margins_top();

            position rect;
            el->m_doc->container()->get_client_rect(rect);

            int width = el->width() + el_shift_left + el_shift_right;
            if ((el->m_pos.x + width > rect.width) && (IsLastLine(*el))) {
                m_truncationNeeded = true;
            }

            m_width += width;

        }
    }
}

void litehtml::line_box::finish(bool last_box)
{
    bool isEmpty = is_empty();
    if( isEmpty || (!isEmpty && last_box && is_break_only()) )
    {
        m_height = 0;
        return;
    }

    bool useBaselineOffset = false;
    int baselineOffset = -1;
    elements_vector::const_iterator it(m_items.begin());
    if (it != m_items.end())
    {
        // if a baseline offset is defined for the document, it only affects the first line!
        baselineOffset = (*it)->get_document()->GetBaselineOffest();
        useBaselineOffset = (baselineOffset >= 0) && (m_lineIndex == 0);
    }

    for(elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend(); i++)
    {
        if((*i)->is_white_space() || (*i)->is_break())
        {
            if(!(*i)->m_skip)
            {
                (*i)->m_skip = true;
                m_width -= (*i)->width();
            }
        }
        else
        {
            break;
        }
    }

    LayoutDirection baseLayoutDir = Undefined;
    LayoutDirection lastElLayoutDir = Neutral;
    int line_width = (m_box_right - m_box_left);
    int left_border = 0;
    int right_border = line_width;
    int blockWidth = -1;
    int blockOffset = -1;

    document* doc = 0;
    if (m_items.begin() != m_items.end()) {
        const element::ptr& firstChunk = (*m_items.begin());
        doc = firstChunk->get_document();
        baseLayoutDir = firstChunk->GetBaseLayoutDirection();
        if (baseLayoutDir == Neutral) {
            baseLayoutDir = firstChunk->GetLayoutDirection();
            element::ptr parent = firstChunk->parent();
            // in case of RTL:
            // If the layout direction is taken from first chunk, it has to be provided to all next lines.
            // Storing it to the parent solves the problem.
            if (baseLayoutDir == RightToLeft && parent.IsValid()) {
                parent->SetBaseLayoutDirection(baseLayoutDir);
            }
        }
    }

    bool cultureDependentAlignment = true;
    bool rightToLeftCulture = false;
    if (doc != 0) {
        document_container* container = doc->container();
        if (container != 0) {
            cultureDependentAlignment = container->IsCultureDependentAlignment();
            rightToLeftCulture = container->IsRightToLeftCulture();
        }
    }

    text_align textAlign = m_text_align;
    if ((!rightToLeftCulture && (baseLayoutDir == RightToLeft)) ||
        (cultureDependentAlignment && rightToLeftCulture && (baseLayoutDir != RightToLeft)) ||
        (!cultureDependentAlignment && rightToLeftCulture && (baseLayoutDir == RightToLeft))) {
        if (textAlign == text_align_left) {
            textAlign = text_align_right;
        }
        else if (textAlign == text_align_right) {
            textAlign = text_align_left;
        }
    }

    int base_line = m_font_metrics.base_line();
    int line_height = 0;

    int add_x = 0;
    if (!m_truncationNeeded)
    {
        switch (textAlign)
        {
            case text_align_right:
                add_x = (m_box_right - m_box_left) - m_width;
                break;
            case text_align_center:
                add_x = ((m_box_right - m_box_left) - m_width) / 2;
                break;
            default:
                add_x = 0;
        }
    }

    m_height = 0;
    if (useBaselineOffset)
    {
        line_height = baselineOffset + m_font_metrics.descent;
    } else {
        if(m_height)
        {
            base_line += (line_height - m_height) / 2;
        }
    }

    // find line box baseline and line-height
    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        const element::ptr& el = (*it);

        if (el->get_display() == display_inline_text)
        {
            font_metrics fm;
            el->get_font(&fm);
            base_line = MAX(base_line, fm.base_line());
            line_height = MAX(line_height, el->line_height());
            m_height = MAX(m_height, fm.height);
        }
        else if (el->get_display() == display_inline_block && is_break_only())
        {
            line_height = MAX(line_height, el->line_height());
        }
        el->m_pos.x += add_x;

#ifdef ENABLE_LITEHTML_DEBUG_TRACE
        litehtml::tstring elText;
        litehtml::el_text* el_text = dynamic_cast<litehtml::el_text*>(&(*el));
        if (el_text != 0)
        {
            el_text->get_text(elText);
        }
        LayoutDirection lytDir = el->GetLayoutDirection();
        printf("->> <%s>: %i %s / %s '%s'\n", el->get_tagName(), el->m_pos.x,
        (Undefined == baseLayoutDir ? "!Undef!" : (Neutral == baseLayoutDir ? "Neutral" : (RightToLeft == baseLayoutDir ? "<-- RtL" : "LtR -->"))),
        (Undefined == lytDir ? "!Undef!" : (Neutral == lytDir ? "Neutral" : (RightToLeft == lytDir ? "<-- RtL" : "LtR -->"))),
        (el_text != 0 ? elText.c_str() : "-X-"));
#endif

        LayoutDirection layoutDir = el->GetLayoutDirection();
        if (el->IsLayoutDirNeutral()) {
            layoutDir = (Neutral == lastElLayoutDir ? baseLayoutDir : lastElLayoutDir);
        } else {
            lastElLayoutDir = layoutDir;
        }

        if (RightToLeft == baseLayoutDir) {
            // BiDi text in right to left context

            if (RightToLeft == layoutDir)
            {
                // mirror the elements of a line horizontally according to their direction.
                // mirror RtL elements within the given boundaries
                el->m_pos.x = (right_border - (el->m_pos.x + el->width()));
                blockWidth = -1;
            }
            else
            {
                // process all LtR elements in this RtL context and place them
                // to the left of the last RtL block (or at the right area border):
                // shift_x = (mirred_x1 - x1) - total_width
                // where mirrored_x1 is X pos of the first element mirrored as above
                //       and x1 is the regular LtR X pos of the first element
                //       and total_width is the width of all elements inc. spaces.
                // and for each of these elements: m_pos.x += shift_x

                // process all subsequent LtR (and neutral) boxes as one block
                // 1. determine the total width of that LtR block
                if (blockWidth < 0) {
                    blockWidth = 0;
                    blockOffset = 0;
                    for (elements_vector::iterator itLtr = it; itLtr != m_items.end() && ((*itLtr)->GetLayoutDirection() != RightToLeft); ++itLtr)
                    {
                        blockWidth += (*itLtr)->width();
                    }
                    // 2. calculate the distance for translating all elements of the LtR block once
                    blockOffset = right_border - (2 * el->m_pos.x + blockWidth);
                }
                // 3. move all elements of that LtR block
                el->m_pos.x += blockOffset;
            }
        }
        else
        {
            // BiDi text in left to right context
            if (RightToLeft == layoutDir)
            {
                // process all subsequent RtL (and neutral) boxes as one block
                // 1. determine the total width of that RtL block
                if (blockOffset < 0) {
                    blockOffset = el->m_pos.x;
                    for (elements_vector::iterator itRtl = it; itRtl != m_items.end() && ((*itRtl)->GetLayoutDirection() != LeftToRight); ++itRtl)
                    {
                       right_border = (*itRtl)->m_pos.x + (*itRtl)->width();
                    }
                 }
                 // 2. mirror the elements of block horizontally within the block boundaries
                 el->m_pos.x = ((blockOffset + right_border) - (el->m_pos.x + el->width()));
             } else {
             blockOffset = -1;
             right_border = line_width;
          }
       }
    }

    m_height = line_height;

    int y1  = 0;
    int y2  = m_height;

    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        if(el->get_display() == display_inline_text)
        {
            font_metrics fm;
            el->get_font(&fm);
            el->m_pos.y = m_height - base_line - fm.ascent;
        } else
        {
            switch(el->get_vertical_align())
            {
            case va_super:
            case va_sub:
            case va_baseline:
                el->m_pos.y = m_height - base_line - el->height() + el->get_base_line() + el->content_margins_top();
                break;
            case va_top:
                el->m_pos.y = y1 + el->content_margins_top();
                break;
            case va_text_top:
                el->m_pos.y = m_height - base_line - m_font_metrics.ascent + el->content_margins_top();
                break;
            case va_middle:
                el->m_pos.y = m_height - base_line - m_font_metrics.x_height / 2 - el->height() / 2 + el->content_margins_top();
                break;
            case va_bottom:
                el->m_pos.y = y2 - el->height() + el->content_margins_top();
                break;
            case va_text_bottom:
                el->m_pos.y = m_height - base_line + m_font_metrics.descent - el->height() + el->content_margins_top();
                break;
            }
            y1 = MIN(y1, el->top());
            y2 = MAX(y2, el->bottom());
        }
    }

    css_offsets offsets;

    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        if (!useBaselineOffset)
        {
            el->m_pos.y -= y1;
            el->m_pos.y += m_box_top;
        }
        if(el->get_display() != display_inline_text)
        {
            switch(el->get_vertical_align())
            {
            case va_top:
                el->m_pos.y = m_box_top + el->content_margins_top();
                break;
            case va_bottom:
                el->m_pos.y = m_box_top + (y2 - y1) - el->height() + el->content_margins_top();
                break;
            case va_baseline:
                //TODO: process vertical align "baseline"
                break;
            case va_middle:
                //TODO: process vertical align "middle"
                break;
            case va_sub:
                //TODO: process vertical align "sub"
                break;
            case va_super:
                //TODO: process vertical align "super"
                break;
            case va_text_bottom:
                //TODO: process vertical align "text-bottom"
                break;
            case va_text_top:
                //TODO: process vertical align "text-top"
                break;
            }
        }

        el->apply_relative_shift(m_box_right - m_box_left);
    }
    if (!useBaselineOffset)
    {
        m_height = y2 - y1;
    }
    m_baseline = (base_line - y1) - (m_height - line_height);
}

bool litehtml::line_box::can_hold(const element::ptr &el, white_space ws)
{
    if(!el->is_inline_box()) return false;

    if (!m_items.empty() && m_items.back()->is_break())
    {
        return false;
    }

    if(el->is_break())
    {
        return true;
    }

    int maxNumberOfLines = el->get_document()->GetMaxNumberOfLines();
    if ((maxNumberOfLines > 0) && (GetLineIndex() >= maxNumberOfLines - 1))
    {
        return true;
    }

    if(ws == white_space_nowrap || ws == white_space_pre)
    {
        return true;
    }

    if(m_box_left + m_width + el->width() + el->get_inline_shift_left() + el->get_inline_shift_right() > m_box_right)
    {
        return false;
    }

    return true;
}

bool litehtml::line_box::have_last_space()
{
    bool ret = false;
    for (elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend() && !ret; i++)
    {
        if((*i)->is_white_space() || (*i)->is_break())
        {
            ret = true;
        } else
        {
            break;
        }
    }
    return ret;
}

bool litehtml::line_box::is_empty()
{
    if(m_items.empty()) return true;
    for (elements_vector::reverse_iterator i = m_items.rbegin(); i != m_items.rend(); i++)
    {
        if(!(*i)->m_skip || (*i)->is_break())
        {
            return false;
        }
    }
    return true;
}

int litehtml::line_box::baseline()
{
    return m_baseline;
}

void litehtml::line_box::get_elements( elements_vector& els )
{
    els.insert(els.begin(), m_items.begin(), m_items.end());
}

int litehtml::line_box::top_margin()
{
    return 0;
}

int litehtml::line_box::bottom_margin()
{
    return 0;
}

void litehtml::line_box::y_shift( int shift )
{
    m_box_top += shift;
    for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
    {
        element::ptr& el = (*it);
        el->m_pos.y += shift;
    }
}

bool litehtml::line_box::is_break_only()
{
    if(m_items.empty()) return true;

    if(m_items.front()->is_break())
    {
        for (elements_vector::iterator it = m_items.begin(); it != m_items.end(); ++it)
        {
            element::ptr& el = (*it);
            if(!el->m_skip)
            {
                return false;
            }
        }
        return true;
    }
    return false;
}

void litehtml::line_box::new_width(int left, int right, elements_vector& els)
{
    int add = left - m_box_left;
    if (add) {
        m_box_left = left;
        m_box_right = right;
        m_width = 0;
        elements_vector::iterator remove_begin = m_items.end();
        for (elements_vector::iterator i = m_items.begin() + 1; i != m_items.end(); i++) {
            element::ptr el = (*i);

            if (!el->m_skip) {
                if (m_box_left + m_width + el->width() + el->get_inline_shift_right() + el->get_inline_shift_left() > m_box_right) {
                    remove_begin = i;
                    break;
                }
                else {
                    el->m_pos.x += add;
                    m_width += el->width() + el->get_inline_shift_right() + el->get_inline_shift_left();
                }
            }
        }
        if (remove_begin != m_items.end()) {
            els.insert(els.begin(), remove_begin, m_items.end());
            m_items.erase(remove_begin, m_items.end());

            for (elements_vector::iterator it = els.begin(); it != els.end(); ++it) {
                element::ptr& el = (*it);
                el->m_box = 0;
            }
        }
    }
}

bool litehtml::line_box::IsLastLine(const element& elem) const
{
    int maxNumberOfLines = elem.get_document()->GetMaxNumberOfLines();

    return ((maxNumberOfLines > 0) && (GetLineIndex() == maxNumberOfLines - 1));
}

