/*!*************************************************************************
 * FILE :			ArabicStringUtil.h
 *
 * DESCRIPTION:	This module provides methods for processing arabic text 
 *                to make it usable in applications which don't support Arabic.
 *
 *                Parts of the core encoder functionality (ReshapeText) was 
 *                adapted from Mohammad (ohammadhi@gmail.com). His
 *                c# code is under MIT and GPL license (Version: 1.0.4)
 *
 *                I took parts of the implementation from the modules 
 *                Arabix.cs and TextUtils.cs and adapted this for the 
 *                ReshapeText method.
 *
 *                You will find the original c# source code here:
 *                  http://sourceforge.net/projects/tadween/files/
 *
 * AUTHOR:			Helmut.Frericks@de.bosch.com
 *
 * COPYRIGHT:     (c) 2005 Robert Bosch GmbH, Hildesheim
 *
 * HISTROY:
 *
 *  28.02.11	    CM-AI / PJ-VW37 Helmut Frericks (fr83hi)  Initial Version
 ***************************************************************************/
/***************************************************************************
 *                                                                      
 *  Version: 1.0.4 - 03.07.2009                                         
 *  Copyright (c) 2009 Mohammad Arabix (mohammadhi at gmail d0t com)    
 *  Dual licensed under the MIT and GPL licenses.                       
 *                                                                     
 ***************************************************************************/

/*   
   Call example for a single line Textlabel:
      const tUTF8char* pubOutString = cArabicStringUtil(pubInString,true,0xF88B));
      const tUTF8char* pubOutString = cArabicStringUtil(pubInString,true));
*/

/*   
   Call example for a multiline Textfield:
   
   cArabicStringUtil sTextOut;

   bool bMirrorStringsLocal = true;
   if (sTextOut.ProcessArabic(pubInString, false) == FALSE)
      bMirrorStringsLocal = false;
   if(bMirrorStringsLocal)
      for( all substrings of (pubInString) )
         cArabicStringUtil::TextUtils::MirrorUTF8String(strSubstring);
*/      



#ifndef __ARABICSTRINGUTIL_H_
#define __ARABICSTRINGUTIL_H_

#define no__TIME_MESSURE_ARABIC

#include <string.h>

#define _UNICODE_(_x_) (_x_)

#define tUTF8char tU8                   // UCS2; note, not a typedef so that is compatible with all "unsigned char" types
typedef unsigned short  tUTF16char;	    // wchar_t in windows

/// class provides methods for processing arabic text to make it usable 
/// in applications which don't support Arabic.
class cArabicStringUtil
{
 public:
   /// ctor
   cArabicStringUtil();
   cArabicStringUtil(const tUTF8char *utf8Text, bool bMirrorBuffer /* normaly mirror */,tUTF16char cursorGlyph=0);
   /// dtor. This one deletes all contained elements.
   ~cArabicStringUtil();
   /// Main Method to convert Arabic Strings
   /// Param
   ///   utf8Text      : Input string 
   ///   bMirrorBuffer : For singleline strings; if true, string will be mirrored  
   tBool ProcessArabic(const tUTF8char *utf8Text, bool bMirrorBuffer /* normaly mirror */);
   tVoid SetCursor(tUTF16char cursorGlyph);
   /// result operator
   operator const tUTF8char *() const;      
   /// returns the result string    
   const unsigned char *get() const { return _strResultAlloc ? _strResultAlloc : _strResultConst; }      
   /// returns the length of the result string    
   int length() const;

   #ifdef __TIME_MESSURE_ARABIC
   void PrintMeasurements ();
   #endif   

 private:
   /// Mirror the non arabic chars like latin text and also arabic numbers 
   int MirrorNonArabic (tUTF16char * data, unsigned utf16CharLen) const;
   int ConvertArabicDigits (tUTF16char * data, unsigned utf16CharLen) const;
   /// deletes all allocated memory and resets all members
   tVoid Clear(); 
   const tUTF8char  * _strResultConst;       // resulting string in UTF8
   tUTF8char        * _strResultAlloc;       // resulting string in UTF8
   tUTF16char         _cursorGlyph;
   tUTF16char       * _szUTF16Work;          // working memory for UTF16 
   int              _szUTF16WorkSize; 
   int              _szUTF16MaxChars; 

 public:
   /// service functions
   class TextUtils
   {
    public:
      /// check for arabic characters inside the string
      static tBool ContainsArabic(const tUTF8char *utf8Text);
      /// swaps all characters inside a UTF8 string
      static tVoid MirrorUTF8String(tUTF8char * data);
      /// swaps all characters inside a UTF16 string
      static tVoid MirrorUTF16String(tUTF16char * data);
      // Returns the mirror symbol of the specified character if it exists.
      static tUTF16char MirrorBrackets ( tUTF16char ch );
      /// Checks if the specified character is an alphanumeric character (RegEx: [A-Za-z0-9]).
      static bool IsAlphaNumeric ( tUTF16char ch )
      {
         return (ch >= 'A' && ch <= 'Z') || (ch >= 'a' && ch <= 'z') || (ch >= '0' && ch <= '9');
      }
      /// Checks if the specified character is a numeric character (RegEx: [0-9]).
      static bool IsLatinNumeric ( tUTF16char ch )
      {
         return(ch >= '0' && ch <= '9');
      }
      #ifdef PROJECTBUILD_NISSAN_LCN2
      /// Checks if the specified character is an MonSpaced digit (U+F830-F839)
      ///  ( only available and used in VW/Nissan programm )
      static bool IsNumericMonoSpaced ( tUTF16char ch )
      {
          return (ch >= _UNICODE_(0xF830) && ch <= _UNICODE_(0xF839));
      }
      #endif
      /// Checks if the specified character is a symbol character.
      static bool IsSymbol ( tUTF16char ch )
      {
         return (ch > ' ' && ch < '0')      // !"#$%&()*+,-./
            || (ch > '9' && ch < 'A')        // :;<=>?@
            || (ch > 'Z' && ch < 'a')        // [\]^_' 
            || (ch > 'z' && ch <= 0xBF)      // {}~ ....
            || (ch == _UNICODE_(0x2013))                // Whrung  
            || (ch == _UNICODE_(0x20A2)) 
            || (ch == _UNICODE_(0x20A4)) 
            || (ch == _UNICODE_(0x20AC)) 
            || (ch == _UNICODE_(0x20B1)) 
            || (ch == _UNICODE_(0x20B4)); 
      }
      /// Checks if the specified character is a symbol character.
      static bool IsSeperator ( tUTF16char ch )
      {
         return (ch >= ',' || ch < ':' || ch < '.');      
      }
      /// Checks if the specified character is Arabic or not;
      static bool IsArabicChar ( tUTF16char ch )
      {
         return  ((ch >= _UNICODE_(0x0600) && ch <= _UNICODE_(0x06FF)) 
            || (ch >= _UNICODE_(0x0750) && ch <= _UNICODE_(0x077F)) 
            || (ch >= _UNICODE_(0xFB50) && ch <= _UNICODE_(0xFDFF)) 
            || (ch >= _UNICODE_(0xFE70) && ch <= _UNICODE_(0xFEFF)) ); 
      }
   };
};




#endif
