#include "taginfo.h"
#include "taginfo_internal.h"

using namespace TagInfo;

#include <string>
#include <glib.h>
#include <gio/gio.h>
#include <syslog.h>
#define GST_DISCOVERER_TIMEOUT_SECS 20
#define GST_DISCOVERER_HTTP_TIMEOUT_SECS 30
#define GST_STATE_CHANGE_TIMEOUT 5

//#define ERROR_LOG
//#define DEBUG
/* macro can be enabled for debugging purpose to view error logs in console*/
#ifdef  ERROR_LOG
#define ERROR_LOG

#define ERROR(msg) \
        syslog(LOG_ERR, "Error : %s:%d ::%s () \t", __FILE__, __LINE__,__FUNCTION__);\
        syslog(LOG_ERR,"%s\n",msg);

#else
#define ERROR(msg)
#endif

VideoThumbnail::VideoThumbnail()
{
    ERROR(("VideoThumbnail Constructor"));
    mLoop = NULL;
    mPlay = NULL;
    mBuf = NULL;
}
VideoThumbnail::~VideoThumbnail()
{
    ERROR(("VideoThumbnail Destroctor"));
    mLoop = NULL;
#ifdef USE_PIXBUF
    mPixbuf = NULL;
#endif
    DeallocateGst();
}

int VideoThumbnail::read(string path, unsigned int seekPercent,unsigned int time=GST_DISCOVERER_TIMEOUT_SECS)
{
    int ret = 0;
    unsigned int duration = 0;

    ERROR(("VideoThumbnail read"));
    ERROR((path.c_str()));
    g_timeout_add_seconds(time , (GSourceFunc)VideoThumbnail::cb_playtime, this); // start playtime
    /*Create palybin Element*/
    mPlay = SetupPlay();
    if(mPlay)
    {
        ret = SetGstFilename(path,mPlay); // Set Uri to Element
        if(!ret)
        {
            ret = SetBusError(mPlay); // Set Bus Error
            ret = SetPlaystate(mPlay, GST_STATE_PAUSED);// Change State to pause
            if(ret == GST_STATE_CHANGE_SUCCESS)
            {
                if(seekPercent)
                {
                    if(seekPercent > 100) // Handle Error input
                    {
                        seekPercent = 50; // 50% of the total video duration
                    }
                    /*Calculate the duration to seek*/
                    duration = ((GetDuration(mPlay)* seekPercent)/100) ;
                    syslog(LOG_INFO,"VideoThumbnail duration %d",duration);
                    if(duration)
                    {
                        StartSeek(mPlay,  duration* 1000); // Seek to calculated duration
                    }
                }
                if(GetPlaybinPad(mPlay)) // get the frame buffer
                {
                    ret = THUMBNAIL_OK;
                }
                else
                {
                    ERROR(("GetPlaybinPad Buf failed"));
                    ret = THUMBNAIL_ERROR;// return error
                }
            }
            else
            {
                ret = THUMBNAIL_ERROR;// return error
            }
        }
        else
        {
            ret = THUMBNAIL_ERROR; // return error
        }
        /*Remove element*/
        if(mPlay)
        {
            SetPlaystate(mPlay, GST_STATE_NULL);
            gst_object_unref(mPlay);
            mPlay = NULL;
        }
    }
    else
    {
        ret = THUMBNAIL_ERROR;// return error
    }
    return ret;
}

GstElement * VideoThumbnail::SetupPlay()
{
    ERROR(("VideoThumbnail SetupPlay"));
    int ret = 0;
    GstElement *Play = NULL;
    GstElement *audio_sink, *video_sink;    
    gst_init(NULL,NULL);

#ifdef DEBUG_VideoThumbnail    
    gst_debug_set_active(true);
#endif

#if defined (ENABLE_GSTREAMER_1_0)
    Play = gst_element_factory_make ("playbin", "play");    
    gst_debug_set_threshold_for_name ("playbin", GST_LEVEL_COUNT); 
#else    
    Play = gst_element_factory_make ("playbin2", "play");    
    gst_debug_set_threshold_for_name ("playbin2", GST_LEVEL_COUNT); 
#endif

    if(Play)
    {
        audio_sink = gst_element_factory_make ("fakesink", "audio-fake-sink");
        video_sink = gst_element_factory_make ("fakesink", "video-fake-sink");
        if(audio_sink && video_sink)
        {
            g_object_set (video_sink, "sync", TRUE, NULL);
            g_object_set (Play,
                    "audio-sink", audio_sink,
                    "video-sink", video_sink,
                    "flags", GST_PLAY_FLAG_VIDEO | GST_PLAY_FLAG_AUDIO,
                    NULL);
        }
        else
        {
            ret = -1;
        }
    }
    else
    {
        ret = -2;
    }
    ERROR(("VideoThumbnail SetupPlay %d", ret));
    return Play;    
}

int VideoThumbnail::SetGstFilename(const string config,GstElement* play)
{
    ERROR(("VideoThumbnail SetGstFilename"));
    int ret = THUMBNAIL_OK;
    GFile *file;
    char *uri;    
    file = g_file_new_for_path(config.c_str());
    if(file)
    {
        uri = g_file_get_uri (file);
        if(uri)
        {                        
            g_object_set(play, "uri", uri, NULL);
            g_object_unref (file);
            g_free (uri);
        }
        else 
            ret = THUMBNAIL_ERROR;
    }
    else
    {
        ret = THUMBNAIL_ERROR;        
    }    
    return ret;
}
GstBuffer* VideoThumbnail::GetPlaybinPad(GstElement *element)
{
    int res = 0;

    ERROR(("Inside GetPlaybinPad "));

    GstStructure *strct = NULL;
    GstCaps *to_caps = NULL;

    g_return_val_if_fail (element != NULL, 0);
    g_return_val_if_fail (GST_IS_ELEMENT (element), 0);

#if defined (ENABLE_GSTREAMER_1_0) 
	GstSample *sample = NULL;
    to_caps = gst_caps_new_simple ("video/x-raw",
            "format", G_TYPE_STRING, "RGB",
            "pixel-aspect-ratio", GST_TYPE_FRACTION, 1, 1,
            NULL);
    g_signal_emit_by_name (element, "convert-sample", to_caps, &sample );
    gst_caps_unref (to_caps);
    if(sample)
    {
        to_caps = gst_sample_get_caps (sample);
        if(to_caps)
        {
            strct = gst_caps_get_structure (to_caps, 0);
            res = gst_structure_get_int (strct, "width", &outwidth);
            res |= gst_structure_get_int (strct, "height", &outheight);
            if (!res)
            {
                ERROR((" could not get snapshot width and Hight dimension"));
                gst_element_set_state (element, GST_STATE_NULL);
                gst_object_unref (element);
            }
        }
        mBuf = gst_sample_get_buffer (sample);
        gst_sample_unref (sample);
    }
#else 
    /* our desired output format (RGB24) */
    to_caps = gst_caps_new_simple ("video/x-raw-rgb",
            "bpp", G_TYPE_INT, 24,
            "depth", G_TYPE_INT, 24,
            /* Note: we don't ask for a specific width/height here, so that
             * videoscale can adjust dimensions from a non-1/1 pixel aspect
             * ratio to a 1/1 pixel-aspect-ratio. We also don't ask for a
             * specific framerate, because the input framerate won't
             * necessarily match the output framerate if there's a deinterlacer
             * in the pipeline. */
            "pixel-aspect-ratio", GST_TYPE_FRACTION, 1, 1,
            "endianness", G_TYPE_INT, G_BIG_ENDIAN,
            "red_mask", G_TYPE_INT, 0xff0000,
            "green_mask", G_TYPE_INT, 0x00ff00,
            "blue_mask", G_TYPE_INT, 0x0000ff,
            NULL);

    /* get frame */

    if(to_caps)
    {
        g_signal_emit_by_name (element, "convert-frame", to_caps, &mBuf);
        gst_caps_unref (to_caps);
        if (!mBuf) {
#ifdef DEBUG_VideoThumbnail
            GST_DEBUG ("Could not take screenshot: %s",
                    "failed to retrieve or convert video frame");
            g_warning ("Could not take screenshot: %s",
                    "failed to retrieve or convert video frame");
#endif //DEBUG_VideoThumbnail
            ERROR(("failed to retrieve or convert video frame"))
        }
    }
    ERROR(("After GetPlaybinPad GetCaps"));
    if(mBuf)
    {
        strct = gst_caps_get_structure (GST_BUFFER_CAPS (mBuf), 0);
        ERROR(("After GetPlaybinPad GetCaps struct"));
        /*Use for PixBuf*/
        res =  gst_structure_get_int (strct, "width", &outwidth);
        res |= gst_structure_get_int (strct, "height", &outheight);
        if (!res)
        {
            ERROR((" could not get snapshot width and Hight dimension"));
            gst_element_set_state (element, GST_STATE_NULL);
            gst_object_unref (element);
        }
    }
#endif
    ERROR(("Leaving GetPlaybinPad "));
    return mBuf;
}
int VideoThumbnail::SetPlaystate(GstElement* play,GstState state)
{
    GstState state_pending;
    GstBus *bus;
    GstMessageType events;
    GstStateChangeReturn msg;

    ERROR(("Checking element SetUpPlay  state:: "));

    msg =  gst_element_set_state(play, state);
    GstState l_gstState;
    GstState l_gstPendingState;
    ERROR(("NewPlaySlot: Waiting 5 sec for complete state transition by Getting state"));
    msg = gst_element_get_state(play,&l_gstState,&l_gstPendingState,GST_SECOND * GST_STATE_CHANGE_TIMEOUT);
    syslog(LOG_INFO,"NewPlaySlot: %d - %d ", l_gstState, l_gstPendingState);
    switch(msg)
    {
        case GST_STATE_CHANGE_FAILURE:
            ERROR(("GST_STATE_CHANGE_FAILURE"));
            break;
        case GST_STATE_CHANGE_SUCCESS:
            ERROR(("GST_STATE_CHANGE_SUCCESS"));
            break;
        case GST_STATE_CHANGE_ASYNC:
            ERROR(("GST_STATE_CHANGE_ASYNC"));;
            break;
        case GST_STATE_CHANGE_NO_PREROLL:
            ERROR(("GST_STATE_CHANGE_NO_PREROLL"));
            break;
    }
    syslog(LOG_INFO,"SetPlaystate msg %d", msg);
    return msg;
}
int VideoThumbnail::SetBusError(GstElement* play)
{
    int res = THUMBNAIL_OK;
    /* -------------------------------------------------------------------- */

    GstBus *bus = gst_pipeline_get_bus(GST_PIPELINE(play));
    if(bus)
    {     
        gst_bus_add_watch(bus, VideoThumbnail::cb_bus_call, this);
#if defined(ENABLE_GSTREAMER_1_0)
        // gst_bus_set_sync_handler(bus, 0, 0 )
        // is modified to gst_bus_set_sync_handler(bus, 0, 0 , NULL)
        gst_bus_set_sync_handler(bus, 0, 0 , NULL);
#else
        gst_bus_set_sync_handler(bus, 0, 0);
#endif

        gst_object_unref(bus);

        /* -------------------------------------------------------------------- */
    }
    return res;
}

gboolean VideoThumbnail::cb_bus_call(GstBus* /*bus*/, GstMessage *msg, gpointer that) 
{
    GstMessageType const type = GST_MESSAGE_TYPE(msg);
    bool bStopLoop = false;
    ERROR(("codec_av:cb_bus_call:type:%d", gst_message_type_get_name(type)));
    VideoThumbnail *oThumb = (VideoThumbnail*)that;

    switch(type) {
        case GST_MESSAGE_EOS: {
            ERROR(("codec_av:cb_bus_call:EOS"));
            /*if(oThumb)
                oThumb->CtrlMainLoop(STOP_LOOP);*/
            break;
        }
        case GST_MESSAGE_DURATION: {
            break;
        case GST_MESSAGE_STATE_CHANGED:
        {
            GstState state0, state1;
            gst_message_parse_state_changed(msg, &state0, &state1, 0);

            ERROR(("codec_av:cb_bus_call:  state:%d  -> %d",
                    gst_element_state_get_name(state0), gst_element_state_get_name(state1)));
            break;
        }
        case GST_MESSAGE_STREAM_STATUS: {
            GstStreamStatusType status;
            gst_message_parse_stream_status(msg, &status, 0);

            ERROR(("codec_av:cb_bus_call:%s ", GST_OBJECT_NAME(msg->src)));
            break;
        }
        case GST_MESSAGE_TAG: {

            ERROR(("codec_av:cb_bus_call:%s", GST_OBJECT_NAME(msg->src)));

            break;
        }
        case GST_MESSAGE_INFO: {

            GError *info0 = 0;
            gchar  *txt   = 0;
            gst_message_parse_info(msg, &info0, &txt);
            gchar  *msg1  = gst_error_get_message(info0->domain, info0->code);

            ERROR(("codec_av:cb_bus_call:INFO:element:name:%s", GST_OBJECT_NAME(msg->src)));
            ERROR(("codec_av:cb_bus_call:msg0:%s", info0->message));
            ERROR(("codec_av:cb_bus_call:msg1:%s", (msg1 ? msg1 : "none")));
            ERROR(("codec_av:cb_bus_call:msg2:%s", (txt  ? txt  : "none")));

            g_error_free(info0);
            g_free(msg1);
            g_free(txt);
            break;
        }
        case GST_MESSAGE_WARNING: {
            GError *warntxt = 0;
            gchar  *txt  = 0;
            gst_message_parse_warning(msg, &warntxt, &txt);
            gchar  *msg1 = gst_error_get_message(warntxt->domain, warntxt->code);

            ERROR(("codec_av:cb_bus_call:WARN:element:name:%s", GST_OBJECT_NAME(msg->src)));
            ERROR(("codec_av:cb_bus_call:msg0:%s", warntxt->message));
            ERROR(("codec_av:cb_bus_call:msg1:%s", (msg1 ? msg1 : "none")));
            ERROR(("codec_av:cb_bus_call:msg2:%s", (txt  ?  txt : "none")));

            g_error_free(warntxt);
            g_free(msg1);
            g_free(txt);
            break;
        }
        case GST_MESSAGE_ERROR: {

            GError *error = 0;
            gchar  *txt   = 0;
            gst_message_parse_error(msg, &error, &txt);
            gchar  *msg1  = gst_error_get_message(error->domain, error->code);

            string ename = GST_OBJECT_NAME(msg->src);
            int    ecode = error->code;

            ERROR(("codec_av:cb_bus_call:ERR:element:name:%s", ename.c_str()));
            ERROR(("codec_av:cb_bus_call:msg0:%s", error->message));
            //  ETG_TRACE_USR1(("codec_av:cb_bus_call:msg1:%s", (msg1 ? msg1 : "none"));
            //  ETG_TRACE_USR1(("codec_av:cb_bus_call:msg2:%s", (txt  ? txt  : "none"));

            if(GST_RESOURCE_ERROR            == error->domain &&
                    (GST_RESOURCE_ERROR_BUSY       == error->code ||
                            GST_RESOURCE_ERROR_OPEN_WRITE == error->code)) {


                ERROR(("codec_av:cb_bus_call:REASON_DEVICE_ERROR"));
            }
            if(GST_RESOURCE_ERROR           == error->domain &&
                    GST_RESOURCE_ERROR_NOT_FOUND == error->code) {
                ERROR(("codec_av:cb_bus_call:REASON_URL_ERROR"));
            }
            if(GST_STREAM_ERROR == error->domain) {
                if(GST_STREAM_ERROR_FAILED == error->code) {
                    ERROR(("codec_av:cb_bus_call:REASON_PIPELINE_ERROR"));
                } else if((GST_STREAM_ERROR_TYPE_NOT_FOUND == error->code) || (GST_STREAM_ERROR_WRONG_TYPE == error->code)){

                    ERROR(("codec_av:cb_bus_call:REASON_FORMAT_ERROR"));
                }
            }
            g_error_free(error);
            g_free(msg1);
            g_free(txt);
            break;
        }
        case GST_MESSAGE_ELEMENT: {
            break;
        }
        default: break;
        }

        return true;
    }
}

void VideoThumbnail::CtrlMainLoop(eCtrlMainLoop type)
{    
    switch(type)
    {
        case INIT_LOOP:
            mLoop = g_main_loop_new(NULL, FALSE);
            break;
        case START_LOOP:
            if(mLoop)
                g_main_loop_run(mLoop);
            break;
        case STOP_LOOP:
            if(g_main_loop_is_running(mLoop))
                g_main_loop_quit(mLoop);
            break;
        default:
            break;        
    }
}
gboolean VideoThumbnail::cb_playtime(gpointer *that) {
    ERROR(("cb_playtime "));
    VideoThumbnail *oThumb = (VideoThumbnail*)that;
    if(oThumb)
    {
        oThumb->error = THUMBNAIL_TIMEOUT;
        oThumb->DeallocateGst();
    }
    return false;
}

#ifdef USE_PIXBUF
tErrorInfo VideoThumbnail::SaveImageFromPixbuf(const tThumbnailConfig config)
{
    ENTRY
    tErrorInfo res = THUMBNAIL_OK; 
    int width, height;
    GdkPixbuf *with_holes;
    GError *err = NULL;
    gboolean ret;

    if(!mPixbuf)
    {
        return -1;
    }    
    height = gdk_pixbuf_get_height (mPixbuf);
    width = gdk_pixbuf_get_width (mPixbuf);
    ETG_TRACE_USR1(("Height %d\n",height));
    ETG_TRACE_USR1(("Width %d\n",width));
    res = ScalePixbuf(mPixbuf,config.width,config.hight,&with_holes);
    if(with_holes)
    {
        switch(config.type)
        {
            case THUMBNAIL_PNG:
                ret = gdk_pixbuf_save (with_holes, config.dest.c_str(), "png", &err, NULL);
                break;
            case THUMBNAIL_BMP:
                ret = gdk_pixbuf_save (with_holes, config.dest.c_str(), "bmp", &err, NULL);
                break;
            case THUMBNAIL_JPEG:
            default:
                ret = gdk_pixbuf_save (with_holes, config.dest.c_str(), "jpeg", &err, NULL);
                break;
        }
        if (ret == FALSE) {
            if (err != NULL) {
                ETG_TRACE_ERR(("totem-video-thumbnailer couldn't write the thumbnail '%s' \n", err->message));
                g_error_free (err);
            } else {
                ETG_TRACE_ERR(("totem-video-thumbnailer couldn't write the thumbnail '%s' \n", err->message));
            }
            res = THUMBNAIL_ERROR;
        }

        g_object_unref(with_holes);
    }
    return res; 
}
tErrorInfo VideoThumbnail::ScalePixbuf (IN GdkPixbuf *pixbuf, IN const tU32 width, IN const tU32 height,OUT GdkPixbuf **oPixbuf)
{    
    tErrorInfo res = MP_NO_ERROR; 
    tU32 size = 0;

    d_height = height;
    d_width = width;
    size = height;
    ETG_TRACE_USR1(("Height %d\n",height));
    ETG_TRACE_USR1(("Width %d\n",width));
    if(*oPixbuf)
    {
        if (size <= 256) {

            *oPixbuf = gdk_pixbuf_scale_simple (pixbuf, d_width, d_height, GDK_INTERP_BILINEAR);
        } else {
            *oPixbuf = gdk_pixbuf_scale_simple (pixbuf, d_width, d_height, GDK_INTERP_BILINEAR);
        }
    }
    return res;
}
#endif
void VideoThumbnail::DeallocateGst()
{
    ERROR(("DeallocateGst"));

    if(mPlay)
    {
        SetPlaystate(mPlay, GST_STATE_NULL);
        gst_object_unref(mPlay);
    }
    if(mBuf)
    {
        gst_buffer_unref (mBuf);
        mBuf = NULL;
    }
    mPlay = NULL;
}
int VideoThumbnail::GetThumbnailBuf(unsigned char **imagedata, unsigned int &imagesize,unsigned int &hight, unsigned int &width)
{  
    int ret = THUMBNAIL_OK;
    unsigned char *strImage = NULL;

    ERROR(("GetThumbNailBuf "));
    if(mBuf)
    {

        ERROR(("GetThumbNailBuf before copying the gstreamer buffer"));
#if defined (ENABLE_GSTREAMER_1_0)
        //Imp For gst 1.0
        GstMapInfo map;
        if (gst_buffer_map (mBuf, &map, GST_MAP_READ))
        {
            *imagedata = map.data;
            imagesize = map.size;
            gst_buffer_unmap (mBuf, &map);
        }
#else
        //Imp For gst 0.10
        imagesize = GST_BUFFER_SIZE(mBuf);
        strImage = (unsigned char *)malloc(imagesize);
        if(strImage)
        {
            memcpy(strImage, GST_BUFFER_DATA (mBuf), imagesize);
            gst_buffer_unref (mBuf);
        }
        *imagedata = strImage;
#endif

        hight = outheight;
        width = outwidth;
        ERROR(("GetThumbNailBuf after copying the gstreamer buffer"));
    }
    else
    {
        ERROR(("GetThumbNailBuf gstreamer buffer map failed"));
        ret = THUMBNAIL_ERROR;
    }
    return ret;
}

int VideoThumbnail::StartSeek(GstElement* play,gint64    _time)
{

    syslog(LOG_INFO,"StartSeek Start %d", _time);
    int res = 0;
    GstStateChangeReturn msg;

    gst_element_seek(play, 1.0, GST_FORMAT_TIME,
            GST_SEEK_FLAG_FLUSH,GST_SEEK_TYPE_SET,
            _time * GST_MSECOND, GST_SEEK_TYPE_NONE, GST_CLOCK_TIME_NONE);
    ERROR(("StartSeek Start Done"));
    msg = gst_element_get_state (play, NULL, NULL, GST_SECOND * GST_STATE_CHANGE_TIMEOUT);
    if(msg != GST_STATE_CHANGE_SUCCESS)
    {
        res = THUMBNAIL_ERROR;
    }
    ERROR(("StartSeek End"));
    return res;
}
unsigned int VideoThumbnail::GetDuration(GstElement *element)
{
    GstQuery *query = NULL;
    gboolean res;
    gint64 duration = 0;

    query = gst_query_new_duration (GST_FORMAT_TIME);
    if(query)
    {
        res = gst_element_query (element, query);
        if (res) {
            gst_query_parse_duration (query, NULL, &duration);
            //g_print ("duration = %"GST_TIME_FORMAT, GST_TIME_ARGS (duration));
        }
        else {
           // g_print ("duration query failed...");
        }
        gst_query_unref (query);
    }
    return (duration/GST_SECOND);
}
