/*
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth 
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 * 	Jörn Magens <shuerhaaken@googlemail.com>
 * 	Matias De lellis <mati86dl@gmail.com>
 * 	Pavel Vasin <rat4vier@gmail.com>
 */


#ifndef TAGINFO_C_H
#define TAGINFO_C_H


#ifdef __cplusplus
extern "C" {
#endif

#ifndef BOOL
#define BOOL int
#endif


/*
 * TagInfo C Binding
 */



typedef struct { int dummy; } TagInfo_Info;

typedef enum {
  TagInfo_IMAGE_TYPE_UNKNOWN,
  TagInfo_IMAGE_TYPE_JPEG,
  TagInfo_IMAGE_TYPE_PNG
} TagInfo_ImageType;

typedef enum {
    MEDIA_FILE_TYPE_UNKNOWN,
    MEDIA_FILE_TYPE_AAC,
    MEDIA_FILE_TYPE_AIF,
    MEDIA_FILE_TYPE_APE,
    MEDIA_FILE_TYPE_ASF,
    MEDIA_FILE_TYPE_FLAC,
    MEDIA_FILE_TYPE_M4A,
    MEDIA_FILE_TYPE_M4B,
    MEDIA_FILE_TYPE_M4P,
    MEDIA_FILE_TYPE_MP3,
    MEDIA_FILE_TYPE_MP4,
    MEDIA_FILE_TYPE_MPC,
    MEDIA_FILE_TYPE_OGA,
    MEDIA_FILE_TYPE_OGG,
    MEDIA_FILE_TYPE_TTA,
    MEDIA_FILE_TYPE_WAV,
    MEDIA_FILE_TYPE_WMA,
    MEDIA_FILE_TYPE_WV,
    MEDIA_FILE_TYPE_SPEEX,
    MEDIA_FILE_TYPE_WMV
} TagInfo_MediaFileType;



// FACTORY CREATION
/*!
 * Creates a TagInfo file based on \a filename.  TagInfo will try to guess the file
 * type.
 *
 * \returns NULL if the file type cannot be determined or the file cannot
 * be opened. The returned object should be freed with taginfo_info_free() after usage
 */
TagInfo_Info *taginfo_info_factory_make(const char *filename);

TagInfo_Info *taginfo_info_factory_make_with_format(const char *filename, TagInfo_MediaFileType format);

TagInfo_Info *taginfo_info_factory_make_from_mime(const char *filename, const char *mime_type);

/*!
 * Frees and closes the info object.
 */
void taginfo_info_free(TagInfo_Info *info);


// READ / WRITE
BOOL taginfo_info_read(TagInfo_Info *info);
BOOL taginfo_info_write(TagInfo_Info *info);


// STRING MANAGEMENT
void taginfo_info_enable_string_management(BOOL management);
void taginfo_info_free_strings();


// TAG ACCESS
char *taginfo_info_get_artist(const TagInfo_Info *info);
void  taginfo_info_set_artist(TagInfo_Info *info, const char *artist);

char *taginfo_info_get_album(const TagInfo_Info *info);
void  taginfo_info_set_album(TagInfo_Info *info, const char *album);

char *taginfo_info_get_title(const TagInfo_Info *info);
void  taginfo_info_set_title(TagInfo_Info *info, const char *title);

char *taginfo_info_get_albumartist(const TagInfo_Info *info);
void  taginfo_info_set_albumartist(TagInfo_Info *info, const char *albumartist);

char *taginfo_info_get_comment(const TagInfo_Info *info);
void  taginfo_info_set_comment(TagInfo_Info *info, const char *comment);

char *taginfo_info_get_composer(const TagInfo_Info *info);
void  taginfo_info_set_composer(TagInfo_Info *info, const char *composer);

char *taginfo_info_get_genre(const TagInfo_Info *info);
void  taginfo_info_set_genre(TagInfo_Info *info, const char *genre);

int  taginfo_info_get_tracknumber(const TagInfo_Info *info);
void taginfo_info_set_tracknumber(TagInfo_Info *info, int tracknumber);

int taginfo_info_get_year(const TagInfo_Info *info);
void taginfo_info_set_year(TagInfo_Info *info, int year);

BOOL taginfo_info_get_has_image(const TagInfo_Info *info);

// rating 1 - 5; 0 -> not set
int  taginfo_info_get_rating(const TagInfo_Info *info);
void taginfo_info_set_rating(TagInfo_Info *info, int rating);

int  taginfo_info_get_playcount(const TagInfo_Info *info);
void taginfo_info_set_playcount(TagInfo_Info *info, int count);

int  taginfo_info_get_tracknumber(const TagInfo_Info *info);
void taginfo_info_set_tracknumber(TagInfo_Info *info, int tracknumber);

// deprecated - do not use
char *taginfo_info_get_disk_string(const TagInfo_Info *info);
// deprecated - do not use
void  taginfo_info_set_disk_string(TagInfo_Info *info, const char *disk_string);

int taginfo_info_get_disk_number(const TagInfo_Info *info);
void taginfo_info_set_disk_number(TagInfo_Info *info, int number);

BOOL taginfo_info_get_is_compilation(const TagInfo_Info *info);
void taginfo_info_set_is_compilation(TagInfo_Info *info, BOOL is_compilation);


// AUDIO PROPERTIES
int taginfo_info_get_length(const TagInfo_Info *info);
int taginfo_info_get_bitrate(const TagInfo_Info *info);
int taginfo_info_get_samplerate(const TagInfo_Info *info);
int taginfo_info_get_channels(const TagInfo_Info *info);

// USER LABELS
char **taginfo_info_get_track_labels(const TagInfo_Info *info,
                                     int *data_length);
void taginfo_info_set_track_labels(TagInfo_Info *info,
                                   const char* const* data, int data_length);

char **taginfo_info_get_album_labels(const TagInfo_Info *info,
                                     int *data_length);
void taginfo_info_set_album_labels(TagInfo_Info *info,
                                   const char* const* data, int data_length);

char **taginfo_info_get_artist_labels(const TagInfo_Info *info,
                                      int *data_length);
void taginfo_info_set_artist_labels(TagInfo_Info *info,
                                    const char* const* data, int data_length);


// IMAGE
BOOL taginfo_info_can_handle_image(const TagInfo_Info *info);
BOOL taginfo_info_get_image(const TagInfo_Info *info,
                            char** data, int *data_length, 
                            TagInfo_ImageType *image_type);
BOOL taginfo_info_set_image(TagInfo_Info *info,
                            const char* data, 
                            int data_length, 
                            TagInfo_ImageType image_type);
/*!
 * Frees a data returned by taginfo_info_get_image()
 */
void taginfo_free_image_data(char* data);


// LYRICS
BOOL taginfo_info_can_handle_lyrics(const TagInfo_Info *info);
char * taginfo_info_get_lyrics(const TagInfo_Info *info);
BOOL taginfo_info_set_lyrics(TagInfo_Info *info, const char *lyrics);



#ifdef __cplusplus
}
#endif
#endif
