
/*
 * ecnr-alsa.c
 *
 *
 * *************************************************
 * qwa-alsa.c
 *
 * Andy Lowe <alowe@mvista.com>
 *
 * 2010 (c) MontaVista Software, LLC. This file is licensed under
 * the terms of the AFL.
 *
 *************************************************
 * Author : Patrick Rey
 * Date: 21.08.2013
 * description: porting from qwa-alsa.c to G3g
 */

#include <glib/gi18n.h>
#include <glib-object.h>
#include <sched.h>
#include "math.h"
/*
 * Add the QWA header files to the list of library header files maintained
 * by lint.
 */
/*lint +libh("qwa.h","qwa_defs.h","qwa_err.h","qwa_rcs.h","qwa_rcs_err.h") */

#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
//#include "ecnr-common-defs.h"
#include "ecnr-alsa.h"
#include "ecnr-audio-recording.h"
#include "ecnr-audio-rts-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-engine-handler.h"
#include "ecnr-qwa-rcs-handler.h"
#include "ecnr-configuration.h"
#include "ecnr-ioctrl.h"

/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_ALSA
#include "trcGenProj/Header/ecnr-alsa.c.trc.h"
#endif


#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"



/*******************************************************************************
              Constant defintion
*******************************************************************************/

//#define D_RETRY_COUNT 30  // max (30 * 100 ms) retry cycle /*currently not used */

#define D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ 2

#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE))
    #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM
        #define D_DUPLICATE_MIC_0_STREAM_BUFFER
    #endif
#endif


#define FLOAT_KOEFF              0.01
#define AUD_FLOAT_ENVELOP_DELAY  0.9999

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
static stecnr_alsa_state *state;

static double fintegrator = 0;
static double fenvelope   = 0;


#ifdef D_USE_STATIC_MEMORY

static tsseAudio _MicIn   = {0};
static tsseAudio _RecvIn  = {0};
static tsseAudio _RefIn   = {0};
static tsseAudio _MicOut  = {0};
static tsseAudio _RecvOut = {0};

#define SMEM_SIZE 1024
typedef char tcSBuffer[SMEM_SIZE];

static tcSBuffer sbuf_i[4] = {0};
static tcSBuffer sbuf_o[2] = {0};

static tcSBuffer sbuf_AsrPrepare = {0};
#endif

/*
 * Helper routines to read QWA parameters.
 */


/*******************************************************************************
*
* FUNCTION: ecnr_get_MicInCnt
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
 static guint ecnr_get_MicInCnt(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_mic_in_cnt = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaMicInCnt, 0, &size, &ecnr_mic_in_cnt);
#else

    status = ecnr_sse_handler_get_data(sse_MicInCnt, 0, &size, &ecnr_mic_in_cnt);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaMicInCnt");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_MicInCnt");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaMicInCnt with value %d \n" ,ecnr_mic_in_cnt);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_MicInCnt with value %d \n" ,ecnr_mic_in_cnt);
#endif
    return ecnr_mic_in_cnt;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RefInCnt
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RefInCnt(void)
{
    /*Lint Warning 529: prio2:*/
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    int status = ECNR_ERR_OK;
#endif
    guint32 size;
    guint16 ecnr_ref_in_cnt = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    ecnr_ref_in_cnt = 1;
#else
    status = ecnr_sse_handler_get_data(sse_RefInCnt, 0, &size, &ecnr_ref_in_cnt);
#endif
    g_mutex_unlock(&state->lock);

/* Lint warning: Info 774: prio1: */
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    if (status)
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_RefInCnt");
    else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_RefInCnt with value %d \n" ,ecnr_ref_in_cnt);
#endif

    return ecnr_ref_in_cnt;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RecvInCnt
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RecvInCnt(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_recv_in_cnt = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRecvInCnt, 0, &size, &ecnr_recv_in_cnt);
#else

    status = ecnr_sse_handler_get_data(sse_RecvInCnt, 0, &size, &ecnr_recv_in_cnt);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRecvInCnt");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_RecvInCnt");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRecvInCnt with value %d \n" ,ecnr_recv_in_cnt);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_RecvInCnt with value %d \n" ,ecnr_recv_in_cnt);
#endif

  return ecnr_recv_in_cnt;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_FrameShiftIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_get_FrameShiftIn(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_frame_shift_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaFrameShiftIn, 0, &size, &ecnr_frame_shift_in);
#else

    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, &ecnr_frame_shift_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaFrameShiftIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_FrameShift");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaFrameShiftIn with value %d \n" ,ecnr_frame_shift_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_FrameShift with value %d \n" ,ecnr_frame_shift_in);
#endif

    return ecnr_frame_shift_in;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_FrameShiftOut
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_FrameShiftOut(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_frame_shift_out = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaFrameShiftOut, 0, &size, &ecnr_frame_shift_out);
#else

    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, &ecnr_frame_shift_out);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaFrameShiftOut");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_FrameShift");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaFrameShiftOut with value %d \n" ,ecnr_frame_shift_out);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_FrameShift with value %d \n" ,ecnr_frame_shift_out);
#endif

  return ecnr_frame_shift_out;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RecvFrameShiftIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RecvFrameShiftIn(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_recv_frame_shift_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRecvFrameShiftIn, 0, &size, &ecnr_recv_frame_shift_in);
#else

    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, &ecnr_recv_frame_shift_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRecvFrameShiftIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_FrameShift");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRecvFrameShiftIn with value %d \n" ,ecnr_recv_frame_shift_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_FrameShift with value %d \n" ,ecnr_recv_frame_shift_in);
#endif
  return ecnr_recv_frame_shift_in;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RecvFrameShiftOut
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RecvFrameShiftOut(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_recv_frame_shift_out = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRecvFrameShiftOut, 0, &size, &ecnr_recv_frame_shift_out);
#else

    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, &ecnr_recv_frame_shift_out);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRecvFrameShiftOut");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_FrameShift");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRecvFrameShiftOut with value %d \n" ,ecnr_recv_frame_shift_out);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_FrameShift with value %d \n" ,ecnr_recv_frame_shift_out);
#endif

  return ecnr_recv_frame_shift_out;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RefFrameShiftIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RefFrameShiftIn(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_ref_frame_shift_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRefFrameShiftIn, 0, &size, &ecnr_ref_frame_shift_in);
#else

    status = ecnr_sse_handler_get_data(sse_FrameShift, 0, &size, &ecnr_ref_frame_shift_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRefFrameShiftIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_FrameShift");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRefFrameShiftIn with value %d \n" ,ecnr_ref_frame_shift_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_FrameShift with value %d \n" ,ecnr_ref_frame_shift_in);
#endif

  return ecnr_ref_frame_shift_in;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_SampleRateIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_get_SampleRateIn(void)
{
    int status = ECNR_ERR_OK;;
    guint32 size;
    guint16 ecnr_sample_rate_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaSampleRateIn, 0, &size, &ecnr_sample_rate_in);
#else

    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, &ecnr_sample_rate_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaSampleRateIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_SampleRate");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaSampleRateIn with value %d \n" ,ecnr_sample_rate_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_SampleRate with value %d \n" ,ecnr_sample_rate_in);
#endif

  return ecnr_sample_rate_in;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_SampleRateOut
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_SampleRateOut(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_sample_rate_out = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaSampleRateOut, 0, &size, &ecnr_sample_rate_out);
#else

    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, &ecnr_sample_rate_out);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaSampleRateOut");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_SampleRate");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaSampleRateOut with value %d \n" ,ecnr_sample_rate_out);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_SampleRate with value %d \n" ,ecnr_sample_rate_out);
#endif

  return ecnr_sample_rate_out;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RecvSampleRateIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RecvSampleRateIn(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_recv_sample_rate_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRecvSampleRateIn, 0, &size, &ecnr_recv_sample_rate_in);
#else

    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, &ecnr_recv_sample_rate_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRecvSampleRateIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_SampleRate");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRecvSampleRateIn with value %d \n" ,ecnr_recv_sample_rate_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_SampleRate with value %d \n" ,ecnr_recv_sample_rate_in);
#endif

  return ecnr_recv_sample_rate_in;
}
/*******************************************************************************
*
* FUNCTION: ecnr_get_RecvSampleRateOut
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RecvSampleRateOut(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_recv_sample_rate_out = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRecvSampleRateOut, 0, &size, &ecnr_recv_sample_rate_out);
#else

    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, &ecnr_recv_sample_rate_out);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRecvSampleRateOut");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_SampleRate");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRecvSampleRateOut with value %d \n" ,ecnr_recv_sample_rate_out);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_SampleRate with value %d \n" ,ecnr_recv_sample_rate_out);
#endif

   return ecnr_recv_sample_rate_out;
}


/*******************************************************************************
*
* FUNCTION: ecnr_get_RefSampleRateIn
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_get_RefSampleRateIn(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint16 ecnr_ref_sample_rate_in = 0;
    size = sizeof(guint16);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaRefSampleRateIn, 0, &size, &ecnr_ref_sample_rate_in);
#else

    status = ecnr_sse_handler_get_data(sse_SampleRate, 0, &size, &ecnr_ref_sample_rate_in);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaRefSampleRateIn");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_SampleRate");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaRefSampleRateIn with value %d \n" ,ecnr_ref_sample_rate_in);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_SampleRate with value %d \n" ,ecnr_ref_sample_rate_in);
#endif

  return ecnr_ref_sample_rate_in;
}

static gchar* ecnr_get_audio_type_for_asr(guint32* osize)
{
    gchar* pdata = (gchar*)NULL;
    int status = ECNR_ERR_OK;
    guint32 size;

    size = sizeof(guint32);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    //TODO: ASR mode from QWA library
    status = ECNR_ERR_NO_RESULT;
    pr_debug("ASR mode from QWA library is not supported \n");
#else
    status = ecnr_sse_handler_get_data(sse_AudioTypeForASR , 0, &size, NULL);
    if(status == SSE_ERR_SIZE_ONLY)
    {
        pdata = (gchar*) g_malloc0 (size * sizeof(gchar));
        status = ecnr_sse_handler_get_data(sse_AudioTypeForASR , 0, &size, pdata);
    }
#endif
    g_mutex_unlock(&state->lock);

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
      //TODO: ASR mode from QWA library
    pr_debug("ASR mode from QWA library is not supported \n");

    (void*)osize;//to resolve lint
#else
    if (status)
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_AudioTypeForASR");
    else
    {
      guint32 i = 0;
        for (i = 0; (i < size) && (NULL != pdata) ; i++)
        {
            pr_debug("array[%d]:%d\n", i, pdata[i] );
        }
    }

    *osize = size;
#endif

    return pdata;
}

static gint ecnr_get_op_mode(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    gint32 ecnr_OpMode = 1; // (sseHandsFree = 1) or ( qwaHandsFree = 1)
    size = sizeof(guint32);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    status = ecnr_qwa_handler_get_data(qwaOpMode, 0, &size, (void *)&ecnr_OpMode);
#else
    status = ecnr_sse_handler_get_data(sse_OpMode, 0, &size, (void *)&ecnr_OpMode);
#endif
    g_mutex_unlock(&state->lock);

    if (status)
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_critical(status, "ecnr_qwa_handler_get_data for qwaOpMode");
#else
        ecnr_critical(status, "ecnr_sse_handler_get_data for sse_OpMode");
#endif
    else
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
      ecnr_debug(status, "ecnr_qwa_handler_get_data for parameter qwaOpMode with value %d \n" ,ecnr_OpMode);
#else
        ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_OpMode with value %d \n" ,ecnr_OpMode);
#endif

  return ecnr_OpMode;
}


static guint ecnr_get_asr_mode(void)
{
    int status = ECNR_ERR_OK;
    guint32 size;
    guint32 ecnr_ASRMode = 1; // (sseHandsFree = 1)

    size = sizeof(guint32);

    g_mutex_lock(&state->lock);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    //TODO: ASR mode from QWA library
    status = ECNR_ERR_NO_RESULT;
#else
    status = ecnr_sse_handler_get_data(sse_ASRMode, 0, &size, (void *)&ecnr_ASRMode);
#endif
    g_mutex_unlock(&state->lock);


#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
      //TODO: ASR mode from QWA library
#else
        if (status)
             ecnr_critical(status, "ecnr_sse_handler_get_data for sse_ASRMode");
        else
            ecnr_debug(status, "ecnr_sse_handler_get_data for parameter sse_ASRMode with value %d \n" ,ecnr_ASRMode);
#endif

  return ecnr_ASRMode;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_ecnr_parameters
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_get_ecnr_parameters(void)
{
    pr_debug("ENTERED\n");

    if(state)
    {
        state->ecnr_mic_in_cnt = ecnr_get_MicInCnt();
        state->ecnr_recv_in_cnt = ecnr_get_RecvInCnt();
        state->ecnr_ref_in_cnt = ecnr_get_RefInCnt();
        state->ecnr_frame_shift_in = ecnr_get_FrameShiftIn();
        state->ecnr_frame_shift_out = ecnr_get_FrameShiftOut();
        state->ecnr_recv_frame_shift_in = ecnr_get_RecvFrameShiftIn();
        state->ecnr_recv_frame_shift_out = ecnr_get_RecvFrameShiftOut();
        state->ecnr_ref_frame_shift_in = ecnr_get_RefFrameShiftIn();
        state->ecnr_sample_rate_in = ecnr_get_SampleRateIn();
        state->ecnr_sample_rate_out = ecnr_get_SampleRateOut();
        state->ecnr_recv_sample_rate_in = ecnr_get_RecvSampleRateIn();
        state->ecnr_recv_sample_rate_out = ecnr_get_RecvSampleRateOut();
        state->ecnr_ref_sample_rate_in = ecnr_get_RefSampleRateIn();
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_ecnr_parameters]: ecnr alsa instance 'state' -> is NULL"));
    }

    pr_debug("EXIT\n");
}
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_dev_name
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
/*
 * Get/Set routines for properties exported via DBus.
 */

void ecnr_alsa_set_dev_name(enum ecnr_audio ecnr_id, const gchar * name)
{
    if(state)
    {
        if(ecnr_id >= NUM_ECNR_AUDIO)
        {
            pr_debug("ecnr channel Id out of range\n");
            return;
        }

        g_mutex_lock(&state->lock);

        if(state->alsa_dev_name[ecnr_id])
            g_free(state->alsa_dev_name[ecnr_id]);

        if (name)
            state->alsa_dev_name[ecnr_id] = g_strdup(name);
        else
            state->alsa_dev_name[ecnr_id] = NULL;

        g_mutex_unlock(&state->lock);

        pr_debug("ecnr_alsa_set_dev_name with paramter %d and device name %s \n" ,ecnr_id, state->alsa_dev_name[ecnr_id]);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_dev_name]: ecnr alsa instance 'state' -> is NULL"));
    }
}




/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_terminate
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_audio_terminate(void)
{

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_alsa_audio_terminate]: ENTERED"));
    if(state)
    {
        g_mutex_lock(&state->lock);
        state->athread_active = FALSE;

        /* thread was terminated,  exit action is done  */
        state->exit = FALSE;
        state->run  = FALSE;
        state->initialized = FALSE;
        state->thread = NULL;
        g_mutex_unlock(&state->lock);

        /* audio streaming thread was terminated */
        state->athread_terminate_count++;

        /*  signal send to wake up from call of g_cond_wait_until()
            within the audproc_alsa_start_ent_instance() function */
        //g_cond_broadcast(&state->update);

        pr_debug("current alsa audio thread active state is %d\n",state->athread_active);
        //g_mutex_unlock(&state->lock);

        (void)ecnr_service_audio_terminate_cb();
        (void)ecnr_service_audio_unref();

    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_audio_terminate]: ecnr alsa instance 'state' -> is NULL"));
    }

    pr_message("EXIT\n");
    return;
}



/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_dev_name
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_mutli_channel_in_stream_mode(gboolean multi_chan)
{
    if(state)
    {
        g_mutex_lock(&state->lock);
        state->ecnr_alsa_multi_chan_in_dev_active = multi_chan;
        g_mutex_unlock(&state->lock);
        pr_debug(" set ecnr_alsa_multi_chan_in_dev_active: %d", state->ecnr_alsa_multi_chan_in_dev_active);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_mutli_channel_in_stream_mode]: ecnr alsa instance 'state' -> is NULL"));
    }
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_dev_name
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
/*
 * Get the name of an ALSA device.  The caller must free the returned string
 * with g_free().
 */
gchar *ecnr_alsa_get_dev_name(enum ecnr_audio ecnr_id)
{
    gchar *name = NULL;

    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return (gchar*)NULL;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        name = g_strdup(state->alsa_dev_name[ecnr_id]);
        g_mutex_unlock(&state->lock);
    }
    else
    {
      name = (gchar*)name;
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_dev_name]: ecnr alsa instance 'state' -> is NULL"));
    }

    return name;
}
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_num_channels
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_num_channels(enum ecnr_audio ecnr_id, guint num_channels)
{
    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->num_channels[ecnr_id] = num_channels;
        pr_debug("set number of channels %d for devId %d",num_channels, ecnr_id);
        ETG_TRACE_USR3(("[ecnr_alsa_set_num_channels]: set number of channels %d for devId %d",num_channels, ecnr_id ));
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_ERR(("[ecnr_alsa_set_num_channels]: ecnr alsa instance 'state' -> is NULL"));
    }

}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_num_channels
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_alsa_get_num_channels(enum ecnr_audio ecnr_id)
{
    guint num_channels =ECNR_MIC_IN0;

    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return ECNR_MIC_IN0;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        num_channels = state->num_channels[ecnr_id];
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_num_channels]: ecnr alsa instance 'state' -> is NULL"));
    }

    return num_channels;
}
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_channel
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_channel(enum ecnr_audio ecnr_id, guint channel)
{
    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->channel[ecnr_id] = channel;
        pr_debug("set active channel %d for devId %d",channel, ecnr_id);
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_channel]: ecnr alsa instance 'state' -> is NULL"));
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_channel
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_alsa_get_channel(enum ecnr_audio ecnr_id)
{
    guint channel = 0;

    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return channel;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        channel = state->channel[ecnr_id];
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_channel]: ecnr alsa instance 'state' -> is NULL"));
    }

    return channel;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_device_open_order
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_device_open_order(enum ecnr_audio ecnr_id, guint dev_order)
{

    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->dev_order[ecnr_id] = dev_order;
        pr_debug("set device %d with order %d",ecnr_id, dev_order);
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_device_open_order]: ecnr alsa instance 'state' -> is NULL"));
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_device_open_order
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_alsa_get_device_open_order(enum ecnr_audio ecnr_id)
{
    guint dev_order=0;

    if(ecnr_id >= NUM_ECNR_AUDIO)
    {
        pr_debug("ecnr channel Id out of range\n");
        return 0;
    }

    if(state)
    {
        g_mutex_lock(&state->lock);
        dev_order = state->dev_order[ecnr_id];
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_device_open_order]: ecnr alsa instance 'state' -> is NULL"));
    }

    return dev_order;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_device_prefill_value
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_device_prefill_value(guint prefill)
{
     ETG_TRACE_USR3(("[ecnr_alsa_set_device_prefill_value]: ENTERED"));

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->prefill = prefill;
        pr_debug("set alsa prefill to %d ms", prefill);
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_device_prefill_value]: ecnr alsa instance 'state' -> is NULL"));
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_device_prefill_value
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static guint ecnr_alsa_get_device_prefill_value(void)
{
    guint prefill =0;

    if(state)
    {
        g_mutex_lock(&state->lock);
        prefill = state->prefill;
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_device_prefill_value]: ecnr alsa instance 'state' -> is NULL"));
    }

    return prefill;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_print_dev_properties
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_alsa_print_dev_properties(void)
{
    gint err = ECNR_ERR_OK;
    gint i =0;

    if(state)
    {
        for (i=0;i<NUM_ECNR_AUDIO;i++)
        {
            pr_message("dev_idx[%d]:: alsa_based_dev:%s,  num_channels:%d, channel:%d, dev_order:%d",
                        i, state->alsa_dev_name[i], state->num_channels[i], state->channel[i], state->dev_order[i]);
        }
        pr_message("prefill:  %d, alsa_thread_priority: %d", state->prefill,state->alsa_thread_priority);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_print_dev_properties]: ecnr alsa instance 'state' -> is NULL"));
        err = ECNR_ERR_NULL_POINTER;
    }
    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_thread_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_thread_priority(guint thread_priority)
{
    ETG_TRACE_USR3(("[ecnr_alsa_set_thread_priority]: ENTERED"));

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->alsa_thread_priority = thread_priority;
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_set_thread_priority]: ecnr alsa instance 'state' -> is NULL"));
    }
    ETG_TRACE_USR3(("[ecnr_alsa_set_thread_priority]: EXIT"));

}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_thread_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
guint ecnr_alsa_get_thread_priority(void)
{
    guint thread_priority = 0;

    if(state)
    {
        g_mutex_lock(&state->lock);
        thread_priority = state->alsa_thread_priority;
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_thread_priority]: ecnr alsa instance 'state' -> is NULL"));
    }

    return thread_priority;
}

/*
 * Routines exported via DBus methods.
 */

/*
 * Start the audio thread and wait for 'timeout' microseconds for confirmation
 * that the thread is running and is waiting for input audio samples.  Returns
 * TRUE if the audio thread was successfully started, or FALSE if the audio
 * thread was not confirmed to be running before the timeout expired.
 */
gboolean ecnr_alsa_start(glong timeout_ms)
{
    gboolean started = FALSE;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_alsa_start]: ENTERED with time_out-> %d",timeout_ms));

    if(!state)
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_ERR(("[ecnr_alsa_start]:ecnr alsa instance 'state' -> is NULL"));
        return started;
    }

    /* initiate audio streaaming start, change run state to True
       and send signal to audio streaming thread to trigger streaming
       while release from asleep state (wait)
    */
    g_mutex_lock(&state->lock);
    if (state->run == FALSE)
    {
        pr_message("set run state to TRUE \n");
        ETG_TRACE_USR3(("[ecnr_alsa_start]: the run state is set to TRUE"));
        state->run = TRUE;
        state->req_start_stop_audio = TRUE;
        g_cond_broadcast(&state->update);
    }
    g_mutex_unlock(&state->lock);

    /* return immediately if no time out is required */
    if(!timeout_ms)
    {
        state->req_start_stop_audio = FALSE;
        return TRUE;
    }

    gint64 tout = (gint64)(timeout_ms) ;
    gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

    /* wait the audio streaming thread has started streaming and
       updated state initialized to true after at least two R/W
       was processed successfully
    */
    struct timespec tout_start, tout_stop;
    glong exec_time = 0;

    clock_gettime(CLOCK_MONOTONIC, &tout_start);

    g_mutex_lock(&state->lock);
    while (state->run && !state->initialized)
    {
      //if (!g_cond_timed_wait(&state->update, &state->lock, &abs_timeout)) //Depricated
      if (!g_cond_wait_until(&state->update, &state->lock, end_time))
        break;
    }

    /* streaming was started successfully return TRUE */
    if (state->run && state->initialized)
    {
        started = TRUE;

        /* reset count of device restart due to xrun detecttion */
        state->devices_restart_after_xrun_count = 0;
    }
    else
    {
        glong tout_time = 0;
        clock_gettime(CLOCK_MONOTONIC, &tout_stop);

        if(tout_start.tv_sec == tout_stop.tv_sec)
        {
            tout_time = tout_stop.tv_nsec - tout_start.tv_nsec;
        }
        else /* next second*/
        {
            tout_time = (1000000000 - tout_start.tv_nsec) + tout_stop.tv_nsec;
        }

        ETG_TRACE_USR3(("[ecnr_alsa_start]: stop start time out after [%d]nsec , an error has occured with repectively the alsa devices,  run(%d), initialized(%d)", tout_time, state->run, state->initialized));
    }

    state->req_start_stop_audio = FALSE;
    g_mutex_unlock(&state->lock);

    pr_message("EXIT with audio state run(%d), initialized(%d)\n", state->run, state->initialized );
    ETG_TRACE_USR3(("[ecnr_alsa_start]: EXIT with audio state run(%d), initialized(%d)", state->run, state->initialized));

    return started;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_audio_run_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: run state
*
*******************************************************************************/
gboolean ecnr_alsa_get_audio_run_state(void)
{
    gboolean run = FALSE;

    if(state)
    {
        g_mutex_lock(&state->lock);
        if(state)
            run = state->run;
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_get_audio_run_state]: ecnr alsa instance 'state' -> is NULL"));
    }

    return run;
}


/*
 * Stop the audio thread and wait for 'timeout' microseconds for confirmation
 * that the thread is stopped.  Returns TRUE if the audio thread was
 * successfully stopped, or FALSE if the audio thread was not confirmed to be
 * stopped before the timeout expired.
 */
gboolean ecnr_alsa_stop(glong timeout_ms)
{
    gboolean stopped = FALSE;

    pr_message("ENTERED\n");

    if(state)
    {
        g_mutex_lock(&state->lock);

        if (state->run == TRUE)
        {
            pr_message("send stop request\n");
            ETG_TRACE_USR3(("[ecnr_alsa_stop]: send stop request"));
            state->run = FALSE;
            state->req_start_stop_audio = TRUE;
            state->req_stop = TRUE;
            g_cond_broadcast(&state->update);
        }
        g_mutex_unlock(&state->lock);

        /* return immediately if no time out is required */
        if(!timeout_ms)
        {
            pr_message("stop immediatly\n");
            ETG_TRACE_USR3(("[ecnr_alsa_stop]: stop immediatly"));
            state->req_start_stop_audio = FALSE;
            return TRUE;
        }

        gint64 tout = (gint64)(timeout_ms) ;
        gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

        g_mutex_lock(&state->lock);
        ecnr_audio_state audio_state = ecnr_service_audio_get_state();

        while (    state->req_stop
                && !state->run
                &&(   state->initialized
                   ||(!state->initialized && (ECNR_AS_STREAM_WAIT_DATA == audio_state))))// 210917 && state->initialized)
        {
            pr_message("wait that the streaming thread has stopped\n");
            ETG_TRACE_USR3(("[ecnr_alsa_stop]: wait that the streaming thread has stopped"));
            if (!g_cond_wait_until(&state->update, &state->lock, end_time))
                break;
            audio_state = ecnr_service_audio_get_state();
            pr_message("run->%d, initialized->%d, audio_state->%d, req_stop->%d \n",state->run, state->initialized, audio_state, state->req_stop);
        }

        if (!state->run && !state->initialized)
            stopped = TRUE;

        state->req_start_stop_audio = FALSE;
        g_mutex_unlock(&state->lock);

        pr_debug("EXIT with audio state run(%d), initialized(%d), exit(%d)\n", state->run, state->initialized, state->exit);
        pr_message("stop successfully");
        ETG_TRACE_USR3(("[ecnr_alsa_stop]: stop successfully"));

    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_stop]: ecnr alsa instance 'state' -> is NULL"));
    }

    state->req_stop = FALSE;

    pr_message("EXIT");
    return stopped;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_exit_wait
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gboolean ecnr_alsa_exit_wait(glong timeout_ms)
{
    //gboolean exit = FALSE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: ENTERED"));

    if(state)
    {
        /* audio thread was terminated */
        if(!ecnr_alsa_audio_active_is())
        {
            pr_message("EXIT -> the als thread is already terminated \n");
            ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: EXIT -> the alsa thread is already terminated"));
            return TRUE;
        }

        g_mutex_lock(&state->lock);

        if (state->exit == FALSE)
        {
            pr_message("request termination of the alsa thread \n");
            state->exit = TRUE;
            g_cond_broadcast(&state->update);
        }
        else
        {
            /* fix me */
            //g_mutex_unlock(&state->lock);
            pr_message("the alsa thread is already terminated");
            //ecnr_alsa_print_current_audio_state();
        }

        g_mutex_unlock(&state->lock);

        if(!timeout_ms)
        {
            /*new: in the case of time out 0, the alsa thread will joined*/
             /* wait till the thread terminate */
            if(state->thread)
            {
                pr_message("wait alsa thread termination\n");
                ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: wait alsa thread termination"));
                /*(gint)*/ g_thread_join(state->thread);

                /* thread was unref by thread_join, the thread struct shold be nulled hier*/
                pr_message("audio alsa thread was terminated");
                ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: audio alsa thread was terminated"));
                ecnr_alsa_audio_terminate();
            }
            else
            {
                pr_message("invalid thread handle, exit immediatly wo waiting termination\n");
            }
        }
        else
        {
            gint64 tout = (gint64)(timeout_ms) ;
            gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;
            ecnr_audio_state audio_state = ecnr_service_audio_get_state();

            pr_message("wait alsa thread termination\n");
            while (ECNR_AS_STREAM_TERMINATED != audio_state)// 210917 && state->initialized)
            {
                pr_message("current state -> %s", ecnr_audio_state_str(audio_state));
                if (!g_cond_wait_until(&state->update, &state->lock, end_time))
                    break;
                audio_state = ecnr_service_audio_get_state();
            }
        }

        //g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("invalide ecnr alsa instance\n");
        ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: invalide ecnr alsa instance"));
    }

    ETG_TRACE_USR3(("[ecnr_alsa_exit_wait]: EXIT"));
    pr_message("EXIT\n");

    return TRUE;
}

/*
 * audio handling routines
 */

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_stream_close
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_stream_close(struct ecnr_alsa_stream *s)
{
    if(s)
    {
        pr_debug("%s \"%s\"",
                (s->dir == SND_PCM_STREAM_CAPTURE) ? "input" : "output",
                        s->dev_name);

        if(s->dev_name)
            g_free(s->dev_name);
        s->dev_name = NULL;
        g_slice_free1(sizeof(*s), s);
    }
    else
    {
        pr_critical("ecnr alsa stream instance 's' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_stream_close]: ecnr alsa stream instance 's' -> is NULL"));
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_statistic
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_get_statistic(void)
{
    trts_stat stat = {0};


    /*initialize xrun */
    stat.num_xruns = 0;

    /* retreive cuurent xrun value in rts*/
    ecnr_audio_rts_handler_statistic_get(&stat);

    pr_message("number of xrun: %d \n",stat.num_xruns);

}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_rts_is_adev_configured
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gboolean ecnr_alsa_rts_is_adev_configured(const gchar *dev_name, snd_pcm_stream_t dir)
{
    gint i = 0;
    trts_cfgadev* st;

    pr_debug("configure device: %s",dev_name );

    for (i = 0; i < NUM_ECNR_AUDIO; i++)
    {
        if(state->rstdata->adevs)
        {
            st = (trts_cfgadev*)&(state->rstdata->adevs[i]);

            if (!g_strcmp0(dev_name, st->pcmname) && (dir == st->dir) )
            {
                pr_debug("the device %s is already configured", dev_name);
                return TRUE;
            }
            pr_debug("rstdeata->adev{%d] = %s, new to configure = %s",i,st->pcmname, dev_name);
        }
        else
        {
            pr_debug("no device configuration available\n");
            return FALSE;
        }
    }

    pr_debug("EXIT");
    return FALSE;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_close_all_streams
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_close_all_streams(void)
{
    gint i;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_close_all_streams]: ENTERED"));

    if(state)
    {
        for (i = 0; i < NUM_ALSA_INPUTS; i++)
        {
            if (state->in_stream[i])
            {
                ecnr_alsa_stream_close(state->in_stream[i]);
                state->in_stream[i] = NULL;
            }
        }

        for (i = 0; i < NUM_ALSA_OUTPUTS; i++)
        {
            if (state->out_stream[i])
            {
                ecnr_alsa_stream_close(state->out_stream[i]);
                state->out_stream[i] = NULL;
            }
        }
    }
    else
  {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_close_all_streams]: ecnr alsa instance 'state' -> is NULL"));
  }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_free_all_ecnr_buffers
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_free_all_ecnr_buffers(void)
{
    short **ecnr_buffer;
    int i;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_free_all_ecnr_buffers]: ENTERED"));

    if(state)
    {

#ifdef D_USE_STATIC_MEMORY

        memset(sbuf_i, 0, sizeof(sbuf_i));
        memset(sbuf_o, 0, sizeof(sbuf_o));
        memset(sbuf_AsrPrepare, 0, sizeof(sbuf_AsrPrepare));

        pr_message("-> memset zeroed for static buffers \n");
#else
        if (state->MicIn)
        {
            ecnr_buffer = *state->MicIn;
            for (i = 0; i < SSE_MAX_AUDIO_CNT; i++)
            {
                if(ecnr_buffer[i])
                {
                    g_free(ecnr_buffer[i]);
                    ecnr_buffer[i] = NULL;
                }
            }

            if(state->MicIn)
                g_free(state->MicIn);
            state->MicIn = NULL;
        }

        if (state->MicOut)
        {
            ecnr_buffer = *state->MicOut;
            for (i = 0; i < SSE_MAX_AUDIO_CNT; i++)
            {
                if(ecnr_buffer[i])
                {
                    g_free(ecnr_buffer[i]);
                    ecnr_buffer[i] = NULL;
                }
            }
            if(state->MicOut)
                g_free(state->MicOut);
            state->MicOut = NULL;

            if(state->ecnr_enable_asr_preprocessing)
            {
                if(state->AsrPrepare)
                    g_free(state->AsrPrepare);
                state->AsrPrepare = NULL;
            }
        }

        if (state->RecvIn)
        {
            ecnr_buffer = *state->RecvIn;
            for (i = 0; i < SSE_MAX_AUDIO_CNT; i++)
            {
                if(ecnr_buffer[i])
                {
                    g_free(ecnr_buffer[i]);
                    ecnr_buffer[i] = NULL;
                }
            }
            if(state->RecvIn)
                g_free(state->RecvIn);
            state->RecvIn = NULL;
        }

        if (state->RecvOut)
        {
            ecnr_buffer = *state->RecvOut;
            for (i = 0; i < SSE_MAX_AUDIO_CNT; i++)
            {
                if(ecnr_buffer[i])
                {
                    g_free(ecnr_buffer[i]);
                    ecnr_buffer[i] = NULL;
                }
            }
            if(state->RecvOut)
                g_free(state->RecvOut);
            state->RecvOut = NULL;
        }

        if (state->RefIn)
        {
            ecnr_buffer = *state->RefIn;
            for (i = 0; i < SSE_MAX_AUDIO_CNT; i++)
            {
                if(ecnr_buffer[i])
                {
                    g_free(ecnr_buffer[i]);
                    ecnr_buffer[i] = NULL;
                }
            }
            if(state->RefIn)
                g_free(state->RefIn);
            state->RefIn = NULL;
        }
#endif

        /* close engine speoifical data  */
        if(state->ecnr_asr_data_type)
        {
            g_free(state->ecnr_asr_data_type);
            state->ecnr_asr_data_type = NULL;
        }
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_free_all_ecnr_buffers]: ecnr alsa instance 'state' -> is NULL"));
    }
}
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_rst_close
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_rst_close(void)
{
    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_rst_close]: ENTERED"));

    if(state)
    {

        if(state->rstdata)
        {
            if(state->rstdata->adevs)
            {
                g_free(state->rstdata->adevs);
                state->rstdata->adevs=NULL;
            }

            if(state->rstdata->streams)
            {
                g_free(state->rstdata->streams);
                state->rstdata->streams=NULL;
            }

            state->rstdata->num_adevs = 0;
            state->rstdata->num_streams = 0;
        }
        (void)ecnr_audio_rts_handler_destroy(state);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_rst_close]: ecnr alsa instance 'state' -> is NULL"));
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_free_all_buffer_descriptors
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_free_all_buffer_descriptors(void)
{
    gint i;
    pr_debug("ENTERED");

    if(state)
    {
        for (i = 0; i < NUM_ALSA_INPUTS; i++)
        {
            if(state->in_buffer[i])
                g_free(state->in_buffer[i]);
            state->in_buffer[i] = NULL;
        }

        for (i = 0; i < NUM_ALSA_OUTPUTS; i++)
        {
            if(state->out_buffer[i])
                g_free(state->out_buffer[i]);
            state->out_buffer[i] = NULL;
        }
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_free_all_buffer_descriptors]: ecnr alsa instance 'state' -> is NULL"));
    }
}

/*
 * Read a period from all ALSA inputs.
 * Returns a negative error code on failure, or 0 on success.
 */
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_read_all
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_read_all(void)
{
    gint err = 0;

    if(state)
    {
        err = ecnr_audio_rts_handler_read((void**)state->rstbuf_i);

        if(err)
            pr_debug("rts read returned with error:%d \n", err);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_read_all]: ecnr alsa instance 'state' -> is NULL"));
    }
    return err;
}

/*
 * Write a period to all ALSA outputs.
 * Returns a negative error code on failure, or 0 on success.
 */
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_write_all
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_write_all(void)
{
    gint err = 0;

    if(state)
    {
        err = ecnr_audio_rts_handler_write((void**)state->rstbuf_o);

        if(err)
            pr_debug("rts write returned with error:%d \n",err);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_write_all]: ecnr alsa instance 'state' -> is NULL"));
    }

    return err;
}


/*
 * Check if a given alsa device name already exists in the given list of alsa
 * streams.  Returns 0 if no match is found; otherwise, returns a pointer to the
 * stream structure where a match was found.
 */
 /*******************************************************************************
*
* FUNCTION: ecnr_alsa_dev_name_strcmp
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static struct ecnr_alsa_stream *ecnr_alsa_dev_name_strcmp(const gchar * dev_name, struct ecnr_alsa_stream
              **stream,
              gint num_streams)
{
    gint i;
    struct ecnr_alsa_stream* s;

    for (i = 0; i < num_streams; i++)
    {
        s = stream[i];
        if (s)
        {
            if (!g_strcmp0(dev_name, s->dev_name))
                return s;
        }
    }
    return NULL;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_finalize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_audio_finalize(gint audio_started)
{
    pr_debug(" finalize  %s \n", (audio_started)? "AFTER_AUDIO_STREAMING_HAPPEND":"NO_AUDIO_STREAMING_HAPPEND");
    ETG_TRACE_USR3(("[ecnr_alsa_audio_finalize]: ENTERED"));

    ecnr_alsa_get_statistic();

 #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    if(ecnr_qwa_rcs_get_active_state())
    {
        /* clean engine instance and associated resources in case of error */
        if(!audio_started)
            (void)ecnr_engine_clean_on_audio_end();
        ecnr_qwa_rcs_handler_finalize();
    }
 #endif
    ecnr_alsa_close_all_streams();
    ecnr_alsa_free_all_ecnr_buffers();
    ecnr_alsa_rst_close();

    ETG_TRACE_USR3(("[ecnr_alsa_audio_finalize]: EXIT"));
    pr_message("audio streaming related resources are freed");
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_stream_setup
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static struct ecnr_alsa_stream *ecnr_alsa_stream_setup
(
    const gchar         *name,
    snd_pcm_stream_t    dir,
    guint               num_channels,
    guint               rate,
    snd_pcm_uframes_t   period_size,
    guint               dev_order
)
{
    struct ecnr_alsa_stream *s;

    pr_debug("ENTERED");

    s = g_slice_alloc0(sizeof(*s));

    if(!s)
        return (struct ecnr_alsa_stream*)NULL;

    s->dev_name = g_strdup(name);
    s->dir = dir;
    s->num_channels = num_channels;
    s->rate = rate;
    s->period_size = period_size;
    s->dev_order = dev_order;

    pr_debug("%s \"%s\": num_channels %d, rate %d,"
        "period size %lu, device order %1u ",
        (s->dir == SND_PCM_STREAM_CAPTURE) ? "input" : "output",
        s->dev_name, num_channels, rate, period_size, dev_order);

    return s;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_create_buffer_descriptor
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

static struct ecnr_alsa_buffer *ecnr_alsa_create_buffer_descriptor
(
    struct  ecnr_alsa_stream    *s,
    enum    ecnr_audio          ecnr_id,
    guint                       channel
)
{
    struct ecnr_alsa_buffer *bd;

    pr_debug("%s \"%s\": ecnr_id %d, channel %d",
            (s->dir == SND_PCM_STREAM_CAPTURE) ? "input" : "output",
            s->dev_name, ecnr_id, channel);

    //if ((channel < 0) || (channel >= s->num_channels))
    if (channel >= s->num_channels)
    {
        pr_critical("%s \"%s\": ecnr_id %d, Bad channel %d",
                (s->dir == SND_PCM_STREAM_CAPTURE) ? "input" : "output",
                 s->dev_name, ecnr_id, channel);
        return NULL;
    }

    bd = g_malloc0(sizeof(*bd));
    bd->stream = s;
    bd->stream_channel = channel;
    bd->ecnr_id = ecnr_id;
    bd->num_samples = s->period_size;

    return bd;
}



/*******************************************************************************
*
* FUNCTION: ecnr_alsa_rst_set_dev_prefill_level
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_rst_set_dev_prefill_level(void)
{
    guint prefill_ms = 0;
    guint prefill_calc_ms = 0;
    guint i;
    gboolean found = FALSE;
    guint frame_shift, sample_rate;
    gboolean bfollowConfiguration = TRUE;

    frame_shift = ecnr_get_FrameShiftIn();
    sample_rate = ecnr_get_SampleRateIn();

    if(sample_rate)
    {
        prefill_calc_ms = (1000 * frame_shift ) / sample_rate;
    }

#ifdef D_FEAT_ADAPT_PREFILL_TO_BSD_FILE_CONTENT
    if(ecnr_sse_debug_check_enable_file())
    {
        pr_message("frame_shift -> %d, sample_rate -> %d\n", frame_shift, sample_rate);
        bfollowConfiguration = FALSE;
        prefill_ms = prefill_calc_ms;
        pr_message("the tuning mode is active, follow the bsd setting prefill -> %d\n", prefill_ms );
    }
    else if (state && state->enable_force_dev_fill_per_feature_setting)
#else
    if (state && state->enable_force_dev_fill_per_feature_setting)
#endif // D_FEAT_ADAPT_PREFILL_TO_BSD_FILE_CONTENT
    {
        prefill_ms = ecnr_alsa_get_device_prefill_value();
        pr_debug("enable prefill _force_dev_fill_per_feature_setting is active\n");
        pr_debug("set prefill (%d)\n", prefill_ms);
        bfollowConfiguration = FALSE;
    }
    else if(!state->ecnr_alsa_prefill_ctrl)
    {
        pr_warning("prefill control was activated \n");
        prefill_ms = ecnr_alsa_get_device_prefill_value();
        bfollowConfiguration = FALSE;
    }
    else
    {
        /*do nothing*/
    }

    if(prefill_ms < prefill_calc_ms)
    {
        pr_warning("the prefill buffer is not completly filled, we expect a lost of quality \n");
    }

    if(bfollowConfiguration) /* walk configuration table */
    {
        //frame_shift = ecnr_get_FrameShiftIn();
        //sample_rate = ecnr_get_SampleRateIn();

        for (i=0; i <STREAM_CFG_MAX;i++)
        {
            if (   ECNR_RTS_STREAM_CONF_table[i].frame_shift == frame_shift
                && ECNR_RTS_STREAM_CONF_table[i].sample_rate == sample_rate
                )
            {
                prefill_ms = ECNR_RTS_STREAM_CONF_table[i].prefill_ms;
                pr_message("set prefill level to %d", prefill_ms);
                found = TRUE;
                break;
            }
        }

        if(!found)
        {
            pr_warning("no configuration available");
        }
    }

    /* set prefill level */
    state->rstdata->prefill_ms = prefill_ms;

    /* reset feature setting  enable flag*/
    state->enable_force_dev_fill_per_feature_setting = FALSE;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_up_device_configuration
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

static guint ecnr_alsa_set_up_device_configuration(enum ecnr_audio dev)
{
    gchar *dev_name = NULL;
    guint i;
    gboolean found = FALSE;
    guint err = ECNR_ERR_OK;


    if(state && !state->enable_dev_configuration)
    {
        pr_warning("device configuration is disabled");
        return ECNR_ERR_OK;
    }

    for (i=0; i <DEV_CFG_MAX;i++)
    {
        if (ECNR_RTS_DEV_CONF_table[i].rts_dev == dev)
        {
            dev_name = ecnr_alsa_get_dev_name(dev);
            if(!g_strcmp0(dev_name,ECNR_RTS_DEV_CONF_table[i].adev_name))
            {
                ecnr_alsa_set_num_channels(dev, ECNR_RTS_DEV_CONF_table[i].num_channel);
                ecnr_alsa_set_channel(dev, ECNR_RTS_DEV_CONF_table[i].active_channel);
                ecnr_alsa_set_device_open_order(dev, ECNR_RTS_DEV_CONF_table[i].dev_open_order);

                pr_debug(" configure device %d : name: %s, num_channel:%d, active channel: %d, open_order: %d"
                    ,dev
                    ,dev_name
                    ,ECNR_RTS_DEV_CONF_table[i].num_channel
                    ,ECNR_RTS_DEV_CONF_table[i].active_channel
                    ,ECNR_RTS_DEV_CONF_table[i].dev_open_order
                );
                found = TRUE;
            }

            if(dev_name)
                g_free(dev_name);
            dev_name = NULL;
        }

        if(found)
            break;
    }

    if (!found)
    {
        pr_warning("no configuration available for dev channel(%d)\n", dev);
        err = ECNR_ERR_DEV_ID_NOT_AVAILABLE;
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_audio_rts_setup
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_audio_rts_setup(struct ecnr_alsa_stream *s, unsigned int stream_channel)
{
    trts_cfgadev*   st;
    gint            err = ECNR_ERR_OK;
    guint           dev_idx = 0;

    pr_debug("stream device: %s, stream channel:%d",s->dev_name, stream_channel);

    /* check dev order */
    if(s->dev_order > NUM_ECNR_AUDIO || !s->dev_order  )
        return -1;

    dev_idx = state->rstdata->num_adevs;
    if(!state->rstdata->adevs)
    {
        pr_debug("no device configuration\n");
        return ECNR_ERR_NULL_POINTER;
    }

    st = (trts_cfgadev*)&(state->rstdata->adevs[dev_idx]);

    /* check whether this device is already configured
     if not than configure new device
    */
    if(!ecnr_alsa_rts_is_adev_configured(s->dev_name, s->dir))
    {
        st->pcmname         = g_strdup(s->dev_name);
        st->dir             = s->dir;         /*SND_PCM_STREAM_PLAYBACK/..CAPTURE*/
        st->rate            = s->rate;
        st->period_frames   = (gint)s->period_size;
/* startup timeout hard coded value constantfor all devices */
        st->startup_tout   = 100;
/* startup timeout hard coded value constantfor all devices */
        st->format          = SND_PCM_FORMAT_S16;    /* */

#ifdef D_SET_INVIDUAL_PREFILL_PER_DEV
        if(!g_strcmp0("AdevMicro12AmpRef", st->pcmname))
        {
            st->extra_periods   =  20; /* set 20 ms prefill for the AdevMicro12AmpRef device*/
            pr_message("set prefill for the device \"AdevMicro12AmpRef\" -> %d\n",st->extra_periods);
        }
#endif

        /* update rts configuration structure  */
        state->rstdata->num_adevs++;
        s->dev_idx = dev_idx;

        pr_debug("rstdev %s \"%s\": stream_dir %d, num_channel %d, rate %d, buffer size %d, "
                "period size %u",
                (st->dir == SND_PCM_STREAM_CAPTURE) ? "input" : "output",
                 st->pcmname, st->dir, s->num_channels, st->rate, 0,
                 st->period_frames);
    }

    /* set channel */
    s->dev_active_chans[s->dev_active_ch_nb] = stream_channel;
    s->dev_active_ch_nb++;

    /* increase number of stream */
    state->rstdata->num_streams++;

    pr_debug("number of adev = %d", state->rstdata->num_adevs);
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_mic_in_0
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_mic_in_0(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err = 0;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn0 - this input is required by the ecnr filter */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MIC_IN0);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MIC_IN0);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {

        /* set up alsa device descriptor */
        state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_MIC_IN0),
                                state->ecnr_sample_rate_in,
                                state->ecnr_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_MIC_IN0));

        if (!state->in_stream[alsa_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* create buffer descriptor */
        state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                state->in_stream[alsa_index],
                                ECNR_MIC_IN0,
                                ecnr_alsa_get_channel(ECNR_MIC_IN0));

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* setup rts data structure*/
        err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MIC_IN0));

        alsa_index++;
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_dup_mic_out
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_dup_mic_out(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err = 0;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn0 - this input is required by the ecnr filter */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_DUP_MIC_OUT);
    dev_name = ecnr_alsa_get_dev_name(ECNR_DUP_MIC_OUT);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {

        /* set up alsa device descriptor */
        state->out_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_PLAYBACK,
                                ecnr_alsa_get_num_channels(ECNR_DUP_MIC_OUT),
                                state->ecnr_sample_rate_out,
                                state->ecnr_frame_shift_out,
                                ecnr_alsa_get_device_open_order(ECNR_DUP_MIC_OUT));

        if (!state->out_stream[alsa_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* create buffer descriptor */
        state->out_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                state->out_stream[alsa_index],
                                ECNR_DUP_MIC_OUT,
                                ecnr_alsa_get_channel(ECNR_DUP_MIC_OUT));

        if (!state->out_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* setup rts data structure*/
        err = ecnr_audio_rts_setup(state->out_stream[alsa_index], ecnr_alsa_get_channel(ECNR_DUP_MIC_OUT));

        alsa_index++;
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_mic_in_1
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_mic_in_1(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn0 - this input is required by the ecnr filter */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MIC_IN1);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MIC_IN1);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name, state->in_stream, NUM_ALSA_INPUTS);
        if (!s)
        {
            state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_MIC_IN1),
                                state->ecnr_sample_rate_in,
                                state->ecnr_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_MIC_IN1));

            if (!state->in_stream[alsa_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(state->in_stream[alsa_index], ECNR_MIC_IN1, ecnr_alsa_get_channel(ECNR_MIC_IN1));

            if (!state->in_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MIC_IN1));
            alsa_index++;
        }
        else
        {
            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(s, ECNR_MIC_IN1, ecnr_alsa_get_channel(ECNR_MIC_IN1));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_MIC_IN1));
        }

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }

    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_ref_in
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_ref_in(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    (void)ecnr_alsa_set_up_device_configuration(ECNR_REF_IN);
    dev_name = ecnr_alsa_get_dev_name(ECNR_REF_IN);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name, state->in_stream, NUM_ALSA_INPUTS);
        if (!s)
        {
            state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_REF_IN),
                                state->ecnr_ref_sample_rate_in,
                                state->ecnr_ref_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_REF_IN));

        if (!state->in_stream[alsa_index])
        {
            err = -ENODEV;
            goto error_out;
        }


            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                  state->in_stream[alsa_index],
                  ECNR_REF_IN,
                  ecnr_alsa_get_channel(ECNR_REF_IN));

            if (!state->in_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_REF_IN));
            alsa_index++;
        }
        else
        {
            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
            s,
            ECNR_REF_IN,
            ecnr_alsa_get_channel(ECNR_REF_IN));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_REF_IN));
        }

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }

    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_multi_chan_mic_in_0_rts
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_multi_chan_mic_in_0(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn0 - this input is required by the ecnr filter */
    /* use last setting done by applcication via ecnr interface,
     * if no setting was done by application than configuration data is taken
     */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_DEV);
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_MIC_0);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MULTI_CHAN_IN_DEV);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_MULTI_CHAN_IN_DEV),
                                state->ecnr_sample_rate_in,
                                state->ecnr_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_MULTI_CHAN_IN_DEV));

        if (!state->in_stream[alsa_index])
        {
            err = -ENODEV;
            goto error_out;
        }


        state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                state->in_stream[alsa_index],
                                ECNR_MULTI_CHAN_IN_MIC_0,
                                ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_0));

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* setup rts data structure*/
        err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_0));
        alsa_index++;
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_multi_chan_mic_in_1_rts
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_multi_chan_mic_in_1(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn1 - this input is required by the ecnr filter */
    /* use last setting done by applcication via ecnr interface,
     * if no setting was done by application than configuration data is taken
     */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_DEV);
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_MIC_1);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MULTI_CHAN_IN_DEV);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name, state->in_stream, NUM_ALSA_INPUTS);
        if (!s)
        {
            state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_MULTI_CHAN_IN_DEV),
                                state->ecnr_sample_rate_in,
                                state->ecnr_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_MULTI_CHAN_IN_DEV));

            if (!state->in_stream[alsa_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                  state->in_stream[alsa_index],
                                  ECNR_MULTI_CHAN_IN_MIC_1,
                                  ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_1));

            if (!state->in_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_1));
            alsa_index++;
        }
        else
        {
            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                      s,
                      ECNR_MULTI_CHAN_IN_MIC_1,
                      ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_1));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_MIC_1));
        }

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_multi_chan_ref_in_rts
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_multi_chan_ref_in(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* use last setting done by applcication via ecnr interface,
     * if no setting was done by application than configuration data is taken
     */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_DEV);
    (void)ecnr_alsa_set_up_device_configuration(ECNR_MULTI_CHAN_IN_REF);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MULTI_CHAN_IN_DEV);
    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name, state->in_stream, NUM_ALSA_INPUTS);
        if (!s)
        {
            state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_MULTI_CHAN_IN_DEV),
                                state->ecnr_ref_sample_rate_in,
                                state->ecnr_ref_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_MULTI_CHAN_IN_DEV));

            if (!state->in_stream[alsa_index])
            {
                err = -ENODEV;
                goto error_out;
            }


            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                  state->in_stream[alsa_index],
                                  ECNR_MULTI_CHAN_IN_REF,
                                  ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_REF));

            if (!state->in_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_REF));
            alsa_index++;
        }
        else
        {
            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                s,
                ECNR_MULTI_CHAN_IN_REF,
                ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_REF));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_MULTI_CHAN_IN_REF));
        }

        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }

    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_recv_in
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_recv_in(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    /* MicIn0 - this input is required by the ecnr filter */
    (void)ecnr_alsa_set_up_device_configuration(ECNR_RECV_IN);
    dev_name = ecnr_alsa_get_dev_name(ECNR_RECV_IN);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name, state->in_stream, NUM_ALSA_INPUTS);
        if (!s)
        {
            state->in_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_CAPTURE,
                                ecnr_alsa_get_num_channels(ECNR_RECV_IN),
                                state->ecnr_recv_sample_rate_in,
                                state->ecnr_recv_frame_shift_in,
                                ecnr_alsa_get_device_open_order(ECNR_RECV_IN));

            if (!state->in_stream[alsa_index])
            {
                err = -ENODEV;
                goto error_out;
            }


            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                  state->in_stream[alsa_index],
                  ECNR_RECV_IN,
                  ecnr_alsa_get_channel(ECNR_RECV_IN));

            if (!state->in_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->in_stream[alsa_index], ecnr_alsa_get_channel(ECNR_RECV_IN));
            alsa_index++;
        }
        else
        {
            state->in_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(s, ECNR_RECV_IN, ecnr_alsa_get_channel(ECNR_RECV_IN));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_RECV_IN));
        }
        if (!state->in_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;

    pr_debug("EXIT");
    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_mic_out_rts
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_mic_out(gint* alsa_idx, gint* bd_idx)
{
    gchar   *dev_name = NULL;
    gint    alsa_index = *alsa_idx;
    gint    bd_index = *bd_idx ;
    gint    err;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    (void)ecnr_alsa_set_up_device_configuration(ECNR_MIC_OUT);
    dev_name = ecnr_alsa_get_dev_name(ECNR_MIC_OUT);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        state->out_stream[alsa_index] =  ecnr_alsa_stream_setup(
                dev_name,
                SND_PCM_STREAM_PLAYBACK,
                ecnr_alsa_get_num_channels(ECNR_MIC_OUT),
                state->ecnr_sample_rate_out,
                state->ecnr_frame_shift_out,
                ecnr_alsa_get_device_open_order(ECNR_MIC_OUT));

        if (!state->out_stream[alsa_index])
        {
        err = -ENODEV;
        goto error_out;
        }

        state->out_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                                state->out_stream[alsa_index],
                                ECNR_MIC_OUT,
                                ecnr_alsa_get_channel(ECNR_MIC_OUT));

        if (!state->out_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }

        /* setup rts data structure*/
        err = ecnr_audio_rts_setup(state->out_stream[alsa_index], ecnr_alsa_get_channel(ECNR_MIC_OUT));
        alsa_index++;
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;
    pr_debug("EXIT");

    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize_recv_out_rts
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_audio_initialize_recv_out(gint* alsa_idx, gint* bd_idx)
{
    gchar *dev_name = NULL;
    gint alsa_index = *alsa_idx;
    gint bd_index = *bd_idx ;
    gint err;
    struct ecnr_alsa_stream *s;

    pr_debug("alsa_idx: %d, bd_idx: %d",alsa_index,bd_index );

    (void)ecnr_alsa_set_up_device_configuration(ECNR_RECV_OUT);
    dev_name = ecnr_alsa_get_dev_name(ECNR_RECV_OUT);

    if ((dev_name != NULL) && (strlen(dev_name) > 0))
    {
        s = ecnr_alsa_dev_name_strcmp(dev_name,state->out_stream, NUM_ALSA_OUTPUTS);
        if (!s)
        {
            state->out_stream[alsa_index] =  ecnr_alsa_stream_setup(
                                dev_name,
                                SND_PCM_STREAM_PLAYBACK,
                                ecnr_alsa_get_num_channels(ECNR_RECV_OUT),
                                state->ecnr_recv_sample_rate_out,
                                state->ecnr_recv_frame_shift_out,
                                ecnr_alsa_get_device_open_order(ECNR_RECV_OUT));

            if (!state->out_stream[alsa_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            state->out_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(
                  state->out_stream[alsa_index],
                  ECNR_RECV_OUT,
                  ecnr_alsa_get_channel(ECNR_RECV_OUT));

            if (!state->out_buffer[bd_index])
            {
                err = -ENODEV;
                goto error_out;
            }

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(state->out_stream[alsa_index], ecnr_alsa_get_channel(ECNR_RECV_OUT));
            alsa_index++;
        }
        else
        {
            state->out_buffer[bd_index] = ecnr_alsa_create_buffer_descriptor(s, ECNR_RECV_OUT, ecnr_alsa_get_channel(ECNR_RECV_OUT));

            /* setup rts data structure*/
            err = ecnr_audio_rts_setup(s, ecnr_alsa_get_channel(ECNR_RECV_OUT));
        }
        if (!state->out_buffer[bd_index])
        {
            err = -ENODEV;
            goto error_out;
        }
        bd_index++;
    }
    if(dev_name != NULL)
        g_free(dev_name);
    dev_name = NULL;

    /*return new alss and buffer descriptor index */
    *alsa_idx = alsa_index;
    *bd_idx   = bd_index;
    pr_debug("EXIT");

    return 0;

error_out:
    pr_debug("intilization failed with device: %s and error: %d ",dev_name, err );
    g_free(dev_name);
    dev_name = NULL;
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_rts_create_read_buffer
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_rts_create_read_buffer(struct ecnr_alsa_stream *s, gint channel, gint buf_idx)
{
    gint err = 0;
    gint i = 0;
    gchar* dev_name;
    struct ecnr_alsa_buffer *bd;
    short** buffer;
    gboolean found = FALSE;
    gulong num_samples;

    pr_debug("enter\n");

    if(!s)
        return ECNR_ERR_NULL_POINTER;

    /* go through the input buffer list and search the matching entry */
    for (i = 0; i < NUM_ALSA_INPUTS; i++)
    {
        bd = state->in_buffer[i];
        if (!bd)
            break;
        dev_name = bd->stream->dev_name;
        if(!g_strcmp0(dev_name, s->dev_name))
        {
            if(bd->stream_channel == channel)
            {
                found = TRUE;
                break; /*structure found*/
            }
        }
    }

    /* create buffers */
    if(found && bd)
    {
        /*rst buffer */
        num_samples = bd->num_samples;

#ifndef D_USE_STATIC_MEMORY
        state->rstbuf_i[buf_idx] = g_malloc0(num_samples * sizeof(short));
#endif

        switch (bd->ecnr_id)
        {
            case ECNR_MIC_IN0:
            case ECNR_MULTI_CHAN_IN_MIC_0:
#ifdef D_USE_STATIC_MEMORY
                state->rstbuf_i[buf_idx] = sbuf_i[0];
                if (!state->MicIn)
                 state->MicIn = _MicIn;
#else
                if (!state->MicIn)
                    state->MicIn = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->MicIn;
                buffer[0] = state->rstbuf_i[buf_idx];
                pr_debug("mapping of rstbuf_i[%d] with state->MicIn[0] with address %p\n", buf_idx,buffer[0]);
                break;
            case ECNR_MIC_IN1:
            case ECNR_MULTI_CHAN_IN_MIC_1:
#ifdef D_USE_STATIC_MEMORY
                state->rstbuf_i[buf_idx] = sbuf_i[1];
                if (!state->MicIn)
                    state->MicIn = _MicIn;
#else
                if (!state->MicIn)
                    state->MicIn = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->MicIn;
                buffer[1] = state->rstbuf_i[buf_idx];
                pr_debug("mapping of rstbuf_i[%d] with state->MicIn[1] with address %p\n", buf_idx,buffer[1]);
                break;
            case ECNR_RECV_IN:
#ifdef D_USE_STATIC_MEMORY
                state->rstbuf_i[buf_idx] = sbuf_i[2];
                if (!state->RecvIn)
                    state->RecvIn = _RecvIn;
#else
                if (!state->RecvIn)
                    state->RecvIn = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->RecvIn;
                buffer[0] = state->rstbuf_i[buf_idx];
                pr_debug("mapping of rstbuf_i[%d] with state->RecvIn[0] with address %p\n", buf_idx,buffer[0]);
                break;
            case ECNR_REF_IN:
            case ECNR_MULTI_CHAN_IN_REF:
#ifdef D_USE_STATIC_MEMORY
                state->rstbuf_i[buf_idx] = sbuf_i[3];
                if (!state->RefIn)
                  state->RefIn = _RefIn;
#else
                if (!state->RefIn)
                    state->RefIn = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->RefIn;
                buffer[0] = state->rstbuf_i[buf_idx];
                pr_debug("mapping of rstbuf_i[%d] with state->RefIn[0] with address %p\n", buf_idx,buffer[0]);
                break;
            case ECNR_RECV_OUT:
            case ECNR_MIC_OUT:
                default:
                break;
        }
    }/* found == FALSE*/
    else
    {
      pr_critical("no configuration");
    }

    pr_debug("set read buffer[%d] for device : %s and channel %d\n", buf_idx,s->dev_name,channel);
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_rts_create_write_buffer
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_rts_create_write_buffer(struct ecnr_alsa_stream *s, gint channel, gint buf_idx)
{
    gint err = 0;
    gint i = 0;
    gchar * dev_name = (gchar*)NULL;
    struct ecnr_alsa_buffer *bd;
    short **buffer;
    gboolean found = FALSE;
    gulong num_samples;

    pr_debug("ENTERED");

    if(!s)
        return ECNR_ERR_NULL_POINTER;

    /* go through the output buffer list and search the matching entry */
    for (i = 0; i < NUM_ALSA_OUTPUTS; i++)
    {
        bd = state->out_buffer[i];
        if (!bd)
            break;

        if(bd->stream && bd->stream->dev_name)
        {
            dev_name = bd->stream->dev_name;
            pr_debug("setup stream for device -> %s\n", dev_name);
        }
        else
        {
            pr_debug("stream descritor or stream device are invalid\n");
        }

        if(!g_strcmp0(dev_name, s->dev_name))
        {
            if(bd->stream_channel == channel)
            {
                pr_debug("stream descriptor for device -> %s found \n", dev_name);
                found = TRUE;
                break; /*structure found*/
            }
        }
    }

    /* create buffers */
    if(found && bd)
    {
        /*rst buffer */
        num_samples = bd->num_samples;
#ifndef D_USE_STATIC_MEMORY
        state->rstbuf_o[buf_idx] = g_malloc0(num_samples * sizeof(short));
#endif

        pr_debug("setup write buffer for the ecnr channel -> %d\n", bd->ecnr_id);

        switch (bd->ecnr_id)
        {
            case ECNR_RECV_OUT:
            {
#ifdef D_USE_STATIC_MEMORY
              state->rstbuf_o[buf_idx] = sbuf_o[0];
                if (!state->RecvOut)
                    state->RecvOut = _RecvOut;
#else
                if (!state->RecvOut)
                    state->RecvOut = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->RecvOut;
                buffer[0] = state->rstbuf_o[buf_idx];
                pr_message("mapping of rstbuf_o[%d] with state->RecvOut[0] with address %p\n", buf_idx,buffer[0]);
                break;
            }
            case ECNR_MIC_OUT:
            {
#ifdef D_USE_STATIC_MEMORY
                state->rstbuf_o[buf_idx] = sbuf_o[1];
                if (!state->MicOut)
                    state->MicOut = _MicOut;
#else
                if (!state->MicOut)
                    state->MicOut = g_malloc0(sizeof(tsseAudio));
#endif
                buffer = *state->MicOut;
                buffer[0] = state->rstbuf_o[buf_idx];
                pr_message("mapping of rstbuf_o[%d] with state->MicOut[0] with address %p\n", buf_idx, buffer[0]);

                /* new asr support */
                if(state->ecnr_enable_asr_preprocessing)
                {
                    if(!state->AsrPrepare)
                    {
#ifdef D_USE_STATIC_MEMORY
                        state->AsrPrepare = sbuf_AsrPrepare;
                        pr_message("create asr ssePrepareDataForASR  buffer with static size %d\n",SMEM_SIZE);
#else
                        state->AsrPrepare = g_malloc0(num_samples * sizeof(short));
                        pr_debug("create asr ssePrepareDataForASR  buffer with size %lu_bytes\n", (num_samples * sizeof(short)));
#endif
                    }
                }
                break;
            }
#ifdef D_DUPLICATE_MIC_0_STREAM_BUFFER
            case ECNR_DUP_MIC_OUT:
            {
                if (state->MicOut)
                {
                    /* duplicate the MicOut to an second output device */
                    buffer = *state->MicOut;
                    state->rstbuf_o[buf_idx] = buffer[0];
                    pr_message("duplicate the MicOut, mapping of rstbuf_o[%d] with state->MicOut[0] with address %p\n", buf_idx, buffer[0]);
                }
                else
                {
                    pr_debug("the micout channel buffer is not available, please create firts , tham duplicate\n");
                }
                break;
            }
#endif
            case ECNR_MIC_IN0:
            case ECNR_MULTI_CHAN_IN_MIC_0:
            case ECNR_MIC_IN1:
            case ECNR_MULTI_CHAN_IN_MIC_1:
            case ECNR_RECV_IN:
            case ECNR_REF_IN:
            case ECNR_MULTI_CHAN_IN_REF:
            default:
                break;
        }
    }/* found == FALSE*/
    else
    {
        pr_critical("no configuration");
    }

    pr_debug("set write buffer[%d] for device : %s and channel %d\n", buf_idx,s->dev_name,channel);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_print_all_dev_order
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_alsa_print_all_dev_order(void)
{
    gint                      i=0;
    struct ecnr_alsa_stream   *s;

    for (i = 0; i < NUM_ALSA_INPUTS ; i++)
    {
        s = (state->in_stream[i]);
        if(!s)
            break;
        pr_debug("in_stream[%d] with device %s current order is: %d ", i, s->dev_name, s->dev_order);
    }

    for (i = 0; i < NUM_ALSA_OUTPUTS ; i++)
    {
        s = (state->out_stream[i]);
        if(!s)
            break;
        pr_debug("out_stream[%d] with device %s current order is: %d ", i, s->dev_name, s->dev_order);
    }
}

#define D_SAFE_MAX_LOOP_ECNR_ALSA 100
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_rts_check_dev_order
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_rts_check_dev_order(void)
{
    gint        err = ECNR_ERR_OK;
    guint       dev_num = 0;
    guint       stream_num = 0;
    gint        i=0;
    guint       dev_order = 1;
    guint       dev_order_try = 0;
    struct ecnr_alsa_stream *s;
    gboolean    found = FALSE;
    gboolean    retry = FALSE;
    gint        loopctrl = 0;

    pr_debug("ENTERED");

    dev_num = state->rstdata->num_adevs;
    stream_num = state->rstdata->num_streams;

    if(!stream_num)
        return ECNR_ERR_INVALID_SIZE;

    pr_debug("Nb of device: %d and Nb of stream : %d", dev_num, stream_num);

    /* normalize the dev order within to all device structures to the max number of configured devices */
    for (i = 0; i < NUM_ALSA_INPUTS ; i++)
    {
        s = (state->in_stream[i]);
        if(!s)
            break;

        if(s->dev_order > dev_num || s->dev_order == 0)
        {
            pr_debug("in_stream[%d] with device %s changed order: %d to %d", i, s->dev_name, s->dev_order, dev_num);
            s->dev_order = dev_num;
        }
        s->dev_order_fixed = FALSE;
    }

    for (i = 0; i < NUM_ALSA_OUTPUTS ; i++)
    {
        s = (state->out_stream[i]);
        if(!s)
            break;

        if(s->dev_order > dev_num || s->dev_order == 0 )
        {
            pr_debug("out_stream[%d] with device %s changed order: %d to %d", i, s->dev_name, s->dev_order, dev_num);
            s->dev_order = dev_num;
        }
        s->dev_order_fixed = FALSE;
    }

    ecnr_alsa_print_all_dev_order();


    loopctrl = D_SAFE_MAX_LOOP_ECNR_ALSA;

    /* rearrange the open device order so order nr. are assigned only ones  */
    while ((dev_order <= dev_num) && loopctrl)
    {
        found = FALSE;
        loopctrl--;

        for (i = 0; i < NUM_ALSA_INPUTS ; i++)
        {
            s = (state->in_stream[i]);
            if(!s)
                break;

            if(!s->dev_order_fixed && s->dev_order == dev_order)
            {
                if(found)
                {
                    if((s->dev_order + 1) <= dev_num)
                        s->dev_order = dev_order + 1;
                }
                else
                {
                    s->dev_order_fixed = TRUE;
                    found = TRUE;
                    pr_debug("in_stream[%d] with device %s fix order: %d", i, s->dev_name, s->dev_order);
                }
            }

            if(retry && s->dev_order == dev_order_try)
            {
                s->dev_order_fixed = TRUE;
                found = TRUE;
                retry = FALSE;
                pr_debug("in_stream[%d] with device %s fix order: %d to %d", i, s->dev_name, s->dev_order, dev_order);
                s->dev_order = dev_order;
                break;
            }
        }

        for (i = 0; i < NUM_ALSA_OUTPUTS ; i++)
        {
            s = (state->out_stream[i]);
            if(!s)
                break;

            if(!s->dev_order_fixed && s->dev_order == dev_order)
            {
                if(found)
                {
                    if((s->dev_order + 1) <= dev_num)
                        s->dev_order = dev_order + 1;
                }
                else
                {
                    s->dev_order_fixed = TRUE;
                    found = TRUE;
                    pr_debug("out_stream[%d] with device %s fix order: %d to %d", i, s->dev_name, s->dev_order, dev_num);
                }
            }

            if(retry && s->dev_order == dev_order_try)
            {
                s->dev_order_fixed = TRUE;
                found = TRUE;
                retry = FALSE;
                pr_debug("out_stream[%d] with device %s fix order: %d to %d", i, s->dev_name, s->dev_order, dev_order);
                s->dev_order = dev_order;
                break;
            }
        }
        if(found)
        {
            dev_order++;
            retry = FALSE;
            dev_order_try = 0;
        }
        else
        {
            if(!retry)
            {
                retry = TRUE;
                dev_order_try = dev_order + 1;
            }
            else
                dev_order_try = dev_order_try + 1;

        }
    }

    if(!loopctrl)
        pr_warning("the device configuration contain some incompatible values\n");

    ecnr_alsa_print_all_dev_order();
    pr_debug("EXIT");

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_rts_set_streams
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gint ecnr_alsa_rts_set_streams(void)
{
    gint        dev_num = 0;
    guint       stream_num = 0;
    gint        err = 0;
    gint        i=0;
    guint       j=0;
    guint       str_idx=0;
    guint       dev_order = 1;
    struct ecnr_alsa_stream * s;
    trts_cfgstream *sst;
    gboolean    found=FALSE;
    gint        bufo_idx=0;
    gint        bufi_idx=0;

    pr_debug("ENTERED");


    if(!state->rstdata)
        return ECNR_ERR_NULL_POINTER;

    dev_num = state->rstdata->num_adevs;
    stream_num = state->rstdata->num_streams;

    if(!stream_num)
        return ECNR_ERR_INVALID_SIZE;

    state->rstdata->streams = g_malloc0(sizeof(trts_cfgstream)* stream_num);
    if(!state->rstdata->streams)
        return ECNR_ERR_NULL_POINTER;

    pr_debug("Nb of device: %d and Nb of stream : %d", dev_num, stream_num);

    /* configure all stream though */
    while (str_idx < stream_num)
    {
        found = FALSE;

        /* check for dev order match within the input devices  */
        for (i = 0; i < NUM_ALSA_INPUTS ; i++)
        {
            s = (state->in_stream[i]);
            if(!s)
                break;

            if(s->dev_name && (s->dev_order == dev_order))
            {
                found = TRUE;
                pr_debug("in_stream[%d] matches with dev_order:%d", i, dev_order);
                break;
            }
            else
                pr_debug("in_stream[%d].dev_order: %d not match with dev_order:%d", i,s->dev_order, dev_order);
        }

        /* check for dev order match within the output devices  */
        if(!found)
        {
            for (i = 0; i < NUM_ALSA_OUTPUTS ; i++)
            {
                s = (state->out_stream[i]);
                if(!s)
                    break;

                if(s->dev_name && (s->dev_order == dev_order))
                {
                    found = TRUE;
                    pr_debug("out_stream[%d] matches with dev_order:%d", i, dev_order);
                    break;
                }
                else
                    pr_debug("out_stream[%d].dev_order: %d not match with dev_order:%d", i,s->dev_order, dev_order);
            }
        }

        if(found && s)
        {
            pr_debug("nb of valid channel: %d", s->dev_active_ch_nb);

            /* go though all channel */
            for (j = 0; j <  s->dev_active_ch_nb ; j++)
            {
                if(!state->rstdata->streams)
                    break;

                sst = &(state->rstdata->streams[str_idx]);

                /* here set rts stream structur
                 * - with dev_idx
                 * - Active channel number of the dev
                 */
                sst->adevidx = s->dev_idx;
                sst->channel = s->dev_active_chans[j];

                pr_debug("rst streams[%d] with channel: %d and devIdx:%d",
                str_idx,sst->channel, sst->adevidx);
                str_idx++;

                /* here the rts buffer which are required for rts_read and rts_write
                 * calls are set.
                 * The buffer array are set in configured open order of the devices
                 */

                /* set rts buffer */
                if(s->dir ==  SND_PCM_STREAM_CAPTURE)
                {
                    ecnr_rts_create_read_buffer(s, sst->channel, bufi_idx);
                    bufi_idx++;
                }
                else
                {
                    ecnr_rts_create_write_buffer(s, sst->channel, bufo_idx);
                    bufo_idx++;
                }
            }
        }
        else
        {
            pr_critical("No device configuration founded with the dev_order : %d", dev_order);
        }

        dev_order++;
        if(dev_order > NUM_ECNR_MAX_RTS_DEV_ORDER)
            break;
    }
    pr_debug("EXIT");
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_audio_print_buffer_pointer
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static void ecnr_audio_print_buffer_pointer(void)
{
    short **buffer;

    pr_debug("rst buffers");

    pr_debug("set rst buffer rstbuf_i[0]: adress: %p ", state->rstbuf_i[0]);
    pr_debug("set rst buffer rstbuf_i[1]: adress: %p ", state->rstbuf_i[1]);
    pr_debug("set rst buffer rstbuf_o[0]: adress: %p ", state->rstbuf_o[0]);
    pr_debug("set rst buffer rstbuf_o[1]: adress: %p ", state->rstbuf_o[1]);

    if (state->MicIn)
    {
        buffer = *state->MicIn;
        pr_debug("set ecnr buffer MicIn[0]: adress: %p ", buffer[0]);
    }

    if (state->RecvIn)
    {
        buffer = *state->RecvIn;
        pr_debug("set ecnr buffer RecvIn[0]: adress: %p ", buffer[0]);
    }

    if (state->MicOut)
    {
        buffer = *state->MicOut;
        pr_debug("set ecnr buffer MicOut[0]: adress: %p ", buffer[0]);
    }

    if (state->RecvOut)
    {
        buffer = *state->RecvOut;
        pr_debug("set ecnr buffer RecvOut[0]: adress: %p ", buffer[0]);
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_feature_set_device
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static int ecnr_alsa_feature_set_device(unsigned short config_id)
{

#if (FEATURE_CFG_MAX == 0)
    pr_debug("No feature setting available\n");
    return ECNR_ERR_OK;
#else
    int err = ECNR_ERR_OK;
    const struct ecnr_id_spec_feature *p = (struct ecnr_id_spec_feature*)NULL;
    struct ecnr_feature_cfg     *pfc = (struct ecnr_feature_cfg*)NULL;
    struct stream_descriptor    *psd = (struct stream_descriptor*)NULL;
    int i = 0;
    int j = 0;


    if(state && !state->enable_dev_configuration)
    {
        pr_debug("device configuration is disabled by user while trying to set the device property with ecnr API \n");
        return ECNR_ERR_OK;
    }

    if( config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;


    pr_debug("feature device setting for config_id(%d)\n", config_id);

    for( i = 0; i < FEATURE_CFG_MAX; i++)
    {
        p = (struct ecnr_id_spec_feature*)&ECNR_FEATURE_CONF_table[i];

        if(p->config_id == config_id)
        {
            pfc = (struct ecnr_feature_cfg*)&(p->feat_cfg_table);
            if(pfc->is_device)
            {
                for( j = 0; j < D_FEATURE_DEV_CONF_MAX; j++)
                {
                    psd = (struct stream_descriptor*)&(pfc->str_desc[j]);
                    if(psd->adev_name)
                    {
                        if(state->dev_configuration_ctrl[psd->stream_ch_Id])
                        {
                            /* set stream channel with alsa device */
                            pr_debug("set stream channel (%d) with device (%s)\n", psd->stream_ch_Id, psd->adev_name);
                            ecnr_alsa_set_dev_name(psd->stream_ch_Id, (const gchar*)psd->adev_name);

                            /* set multichannel */
                            if(     psd->stream_ch_Id == ECNR_MULTI_CHAN_IN_DEV
                                ||  psd->stream_ch_Id == ECNR_MULTI_CHAN_IN_MIC_0
                                ||  psd->stream_ch_Id == ECNR_MULTI_CHAN_IN_MIC_1
                                ||  psd->stream_ch_Id == ECNR_MULTI_CHAN_IN_REF )
                            {
                                ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
                            }
                        }
                        else
                        {
                          pr_debug("dev configuration ctrl is disabled for channel (%d) with device (%s)\n", psd->stream_ch_Id, psd->adev_name);
                        }
                    }
                }
            }
            else
                pr_debug("set device feature is disabled\n");
        }
    }

    return err;
#endif
}


/*******************************************************************************
*
* FUNCTION: ecnr_configuration_is_io_ctrl_select
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static int ecnr_configuration_is_io_ctrl_select(enum eDataPool dpid, int feature)
{
    int err = ECNR_ERR_OK;
    int i = 0;
    int loopctrl = 0;


    pr_message("ENTERED with dp(%d), feature(%d)\n", dpid, feature);
    ETG_TRACE_USR3(("[ecnr_configuration_is_io_ctrl_select]: called with dpid[%d], Feature[%d]", dpid, feature));

    if(!(feature & CONFIG_FEATURE_IOCTRL_SEL))
    {
        pr_debug("no ioctrl feature configured\n");
        return ECNR_ERR_OK;
    }

    gboolean berr = ecnr_ioctrl_create();

    if(berr)
    {
        loopctrl = D_SAFE_MAX_LOOP_ECNR_CONFIGURATION;

        while (ECNR_CONFIG_FILE_table[i].file != NULL && loopctrl)
        {
            pr_debug("gpio(%d), io_level(%d)\n", ECNR_CONFIG_FILE_table[i].select_io_pin, ECNR_CONFIG_FILE_table[i].select_logical_level);

            if(dpid == ECNR_CONFIG_FILE_table[i].EcnrdPBaseKey)
            {
                pr_debug("config found with pin(%d) and level(%d)\n",ECNR_CONFIG_FILE_table[i].select_io_pin, ECNR_CONFIG_FILE_table[i].select_logical_level  );
                ecnr_ioctrl_set_pin_state(ECNR_CONFIG_FILE_table[i].select_io_pin, ECNR_CONFIG_FILE_table[i].select_logical_level);
            }

            i++;
            loopctrl--;
        }

        if(!loopctrl)
            pr_warning("the device configuration contains some incompatible values\n");
    }
    else
       pr_debug("ioctrl create failed\n");

    pr_message("EXIT with loopctrl(%d)\n", loopctrl);

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_feature_set_prefill
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static int ecnr_alsa_feature_set_prefill(unsigned short config_id)
{
#if (FEATURE_CFG_MAX == 0)
    pr_debug("No feature setting available\n");
    return ECNR_ERR_OK;
#else
    int err = ECNR_ERR_OK;
    const struct ecnr_id_spec_feature *p = (struct ecnr_id_spec_feature*)NULL;
    struct ecnr_feature_cfg     *pfc = (struct ecnr_feature_cfg*)NULL;
    int i = 0;


    if(state && !state->ecnr_alsa_prefill_ctrl)
    {
        pr_debug(" prefill configuration is disabled by user while trying to set the device property with ecnr API \n");
        return ECNR_ERR_OK;
    }

    if( config_id == 0)
        return ECNR_ERR_CONFID_NOT_AVAILABLE;

    pr_debug("feature prefill setting for config_id(%d)\n", config_id);

    for( i = 0; i < FEATURE_CFG_MAX; i++)
    {
        p = (struct ecnr_id_spec_feature*)&ECNR_FEATURE_CONF_table[i];

        if(p->config_id == config_id)
        {
            pfc = (struct ecnr_feature_cfg*)&(p->feat_cfg_table);
            if(pfc->is_preFill)
            {
                if(!pfc->prefill_value)
                    return ECNR_ERR_VALUE_RANGE_FAILURE;

                /* enable setting of pre-fill  */
                pr_debug("set prefill value (%d)\n", pfc->prefill_value);
                state->enable_force_dev_fill_per_feature_setting = TRUE;
                ecnr_alsa_set_device_prefill_value(pfc->prefill_value);
            }
            else
                pr_debug("set prefill feature is disabled\n");
        }
    }

    return err;
#endif
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_feature_WB_active
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_feature_stream_duplicate(int feature)
{
#ifdef D_DUPLICATE_MIC_0_STREAM_BUFFER
    if(feature & CONFIG_FEATURE_DUP_STREAM)
    {
        if(state)
        {
            state->ecnr_dup_mic_in_0 = TRUE;
            pr_message("duplicate stream device is activated\n");
            ETG_TRACE_USR3(("[ecnr_alsa_feature_stream_duplicate]: duplicate stream device is activated"));
        }
    }
#endif
    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_feature_WB_active
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
#if (FEATURE_CFG_MAX == 0)
static int ecnr_alsa_is_feature_WB_active(int feature, int class)
{
    int err = ECNR_ERR_OK;

    if (    class != CONFIG_CLASS_PHONE
         && class != CONFIG_CLASS_VOICE)
        return ECNR_ERR_OK;

    if (    feature == CONFIG_FEATURE_PHWB
         || feature == CONFIG_FEATURE_WB )
    {
        if(   !state->enable_dev_configuration
           && !state->ecnr_alsa_prefill_ctrl)
        {
            pr_message("device configuration is disable by user while setting device property with ecnr API");
        }
        else
        {
            pr_message("Phone class and Wide Band feature active, the default devices for WB are set");

            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_OUT])
                {
                    /* set always the ECNR_MIC_OUT for handsfree as "AdevBTPhoneTxOut" */
                    pr_debug("set ECNR_MIC_OUT channel to AdevBTPhoneTxOut\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_OUT,"AdevBTPhoneTxOut");
                }

                if(state->dev_configuration_ctrl[ECNR_RECV_IN])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevBTPhoneRxIn" */
                    pr_debug("set ECNR_RECV_IN channel to AdevBTPhoneRxIn\n");
                    ecnr_alsa_set_dev_name(ECNR_RECV_IN,"AdevBTPhoneRxIn");
                }

            }
    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_IN0])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevMicro1In" */
                    pr_debug("set ECNR_MIC_IN0 to AdevMicro1In\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_IN0,"AdevMicro1In");
                }
            }

            if(state->ecnr_alsa_prefill_ctrl)
            {
                /* enable setting of pre-fill and set pre-fill to 4  */
                state->enable_force_dev_fill_per_feature_setting = TRUE;
                ecnr_alsa_set_device_prefill_value(4);
            }
    #else
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MULTI_CHAN_IN_DEV])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevMicro12AmpRef" */
                    pr_debug("set ECNR_MULTI_CHAN_IN_DEV to AdevMicro12AmpRef\n");
                    ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_DEV,"AdevMicro12AmpRef");

                    /* set multichannel */
                    ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
                }
            }
    #endif
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevVoiceOut" */
                    pr_debug("set ECNR_RECV_OUT to AdevVoiceOut\n");
                    ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevVoiceOut");
                }
            }
        }
    }
    else
        return ECNR_ERR_OK;

    return err;
}
#endif //#if (FEATURE_CFG_MAX == 0)

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_feature_NB_active
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
#if (FEATURE_CFG_MAX == 0)
static int ecnr_alsa_is_feature_NB_active(int feature, int class)
{
    int err = ECNR_ERR_OK;

    if (   class != CONFIG_CLASS_PHONE
        && class != CONFIG_CLASS_VOICE)

        return ECNR_ERR_OK;

    if (   feature == CONFIG_FEATURE_PHNB
        || feature == CONFIG_FEATURE_NB )
    {
        if(   !state->enable_dev_configuration
           && !state->ecnr_alsa_prefill_ctrl)
        {
            pr_message("device configuration is disable by user while setting device property with ecnr API");
        }
        else
        {
            pr_message(" Phone class and Narrow Band feature are active, the default devices for NB are set");

            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_OUT])
                {
                    /* set always the ECNR_MIC_OUT for handsfree as "AdevBTPhoneTxOut" */
                    pr_debug("set ECNR_MIC_OUT channel to AdevBTPhoneTxOut\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_OUT,"AdevBTPhoneTxOut");
                }
                if(state->dev_configuration_ctrl[ECNR_RECV_IN])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevBTPhoneRxIn" */
                    pr_debug("set ECNR_RECV_IN channel to AdevBTPhoneRxIn\n");
                    ecnr_alsa_set_dev_name(ECNR_RECV_IN,"AdevBTPhoneRxIn");
                }
            }
    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_IN0])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevMicro1In" */
                    pr_debug("set ECNR_MIC_IN0 to AdevMicro1In\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_IN0,"AdevMicro1In");
                }
            }

            if(state->ecnr_alsa_prefill_ctrl)
            {
                /* enable setting of pre-fill and set pre-fill to 4  */
                state->enable_force_dev_fill_per_feature_setting = TRUE;
                ecnr_alsa_set_device_prefill_value(4);
            }
    #else
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MULTI_CHAN_IN_DEV])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevMicro12AmpRef" */
                    pr_debug("set ECNR_MULTI_CHAN_IN_DEV to AdevMicro12AmpRef\n");
                    ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_DEV,"AdevMicro12AmpRef");
                    /* set multichannel */
                    ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
                }
            }
    #endif
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevVoiceOut" */
                    pr_debug("set ECNR_RECV_OUT to AdevVoiceOut\n");
                    ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevVoiceOut");
                }
            }
        }
    }
    else
        return ECNR_ERR_OK;

    return err;
}
#endif //#if (FEATURE_CFG_MAX == 0)
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_feature_CP_active
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
#if (FEATURE_CFG_MAX == 0)
static int ecnr_alsa_is_feature_CP_active(int feature, int class)
{
    int err = ECNR_ERR_OK;


 #ifdef CONFIG_CLASS_DEBUG
    if (   class   == CONFIG_CLASS_DEBUG
        && feature == CONFIG_FEATURE_CP)
    {
        pr_debug("car play feature active, ECNR_MIC_OUT channel is set to AdevMicroWithECNROut\n" );
        ecnr_alsa_set_dev_name(ECNR_MIC_OUT,"AdevMicroWithECNROut");
    }
 #endif

    if (   class != CONFIG_CLASS_PHONE
        && class != CONFIG_CLASS_VOICE)

        return ECNR_ERR_OK;

    if(    feature == CONFIG_FEATURE_CPNB
        || feature == CONFIG_FEATURE_CPWB)
    {
        if(   !state->enable_dev_configuration
           && !state->ecnr_alsa_prefill_ctrl)
        {
            pr_message("device configuration is disable by user while setting device property with ecnr API");
        }
        else
        {
            pr_message("SPI class is active, the default devices for CP are set");

            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_OUT])
                {
                    /* set always the ECNR_MIC_OUT for handsfree as "AdevECNRInOut" */
                    pr_debug("set ECNR_MIC_OUT channel to AdevECNRInOut\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_OUT,"AdevECNRInOut");
                }
                if(state->dev_configuration_ctrl[ECNR_RECV_IN])
                {
                    /* set always the ECNR_RECV_IN for handsfree as "AdevECNRInOut" */
                    pr_debug("set ECNR_RECV_IN channel to AdevECNRInOut\n");
                    ecnr_alsa_set_dev_name(ECNR_RECV_IN,"AdevECNRInOut");
                }
            }

            if(state->ecnr_alsa_prefill_ctrl)
            {
                /* enable setting of pre-fill and set pre-fill to 16  */
                state->enable_force_dev_fill_per_feature_setting = TRUE;
                ecnr_alsa_set_device_prefill_value(16);
            }
    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            /* set always the ECNR_MIC_IN0 for handsfree as "AdevMicro1In" */
            //pr_debug("set ECNR_MIC_IN0 to AdevMicro1In\n");
            //ecnr_alsa_set_dev_name(ECNR_MIC_IN0,"AdevMicro1In");
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MIC_IN0])
                {
                    pr_debug("set ECNR_MIC_IN0 to AdevSPIMicro1\n");
                    ecnr_alsa_set_dev_name(ECNR_MIC_IN0,"AdevSPIMicro1");
                }
            }
    #else
            if(state->enable_dev_configuration)
            {
                if(state->dev_configuration_ctrl[ECNR_MULTI_CHAN_IN_DEV])
                {
                    /* set always the ECNR_MULTI_CHAN_IN_DEV for handsfree as "AdevMicro12AmpRef" */
                    pr_debug("set ECNR_MULTI_CHAN_IN_DEV to AdevMicro12AmpRef\n");
                    ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_DEV,"AdevMicro12AmpRef");

                    /* set multichannel */
                    ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
                }
            }
    #endif

            /* siri uses case */
            if(class == CONFIG_CLASS_VOICE)
            {
    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
                if(state->enable_dev_configuration)
                {
                    if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                    {
                        /* set always the ECNR_RECV_IN for handsfree as "AdevSPIAcousticoutSpeech " */
                        pr_debug("set ECNR_RECV_OUT to AdevSPIAcousticoutSpeech  for Siri uses case\n");
                        ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevSPIAcousticoutSpeech");
                    }
                }
    #else
                if(state->enable_dev_configuration)
                {
                    if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                    {
                        /* set always the ECNR_RECV_IN for handsfree as "AdevAcousticoutSpeech" */
                        pr_debug("set ECNR_RECV_OUT to AdevAcousticoutSpeech for Siri uses case\n");
                        ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevAcousticoutSpeech");
                    }
                }
    #endif

            }
            else
            {
    #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
                if(state->enable_dev_configuration)
                {
                    if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                    {
                        /* set always the ECNR_RECV_IN for handsfree as "AdevSPIVoiceOut " */
                        pr_debug("set ECNR_RECV_OUT to AdevSPIVoiceOut \n");
                        ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevSPIVoiceOut");
                    }
                }
    #else
                if(state->enable_dev_configuration)
                {
                    if(state->dev_configuration_ctrl[ECNR_RECV_OUT])
                    {
                        /* set always the ECNR_RECV_IN for handsfree as "AdevVoiceOut" */
                        pr_debug("set ECNR_RECV_OUT to AdevVoiceOut\n");
                        ecnr_alsa_set_dev_name(ECNR_RECV_OUT,"AdevVoiceOut");
                    }
                }
    #endif
            }
        }
    }
    else
        return ECNR_ERR_OK;

    return err;
}
#endif //#if (FEATURE_CFG_MAX == 0)
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_check_feature_setting
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_check_feature_setting(void)
{
    int feature = 0;
    int class = 0;
    unsigned short current_active_config_Id = 0;
#if (FEATURE_CFG_MAX > 0)
    unsigned short ecnr_config_id = 0;
#endif
    int err = ECNR_ERR_OK;

    /* retrieve cuurent active config Id*/
    current_active_config_Id = ecnr_engine_get_current_config_id();

    /* retrieve active feature */
    feature = ecnr_configuration_get_config_feature(current_active_config_Id);

    /* retrieve active class */
    class = ecnr_configuration_get_config_class(current_active_config_Id);

    pr_message("set feature for config Id: %d, feature: %d, class :%d", current_active_config_Id, feature, class);

#if (FEATURE_CFG_MAX > 0)
    ecnr_config_id = ecnr_configuration_get_ecnr_config_Id(current_active_config_Id);
    err = ecnr_alsa_feature_set_device(ecnr_config_id);
    if(err == ECNR_ERR_OK)
    {
        err = ecnr_alsa_feature_set_prefill(ecnr_config_id);
        #ifdef D_DUPLICATE_MIC_0_STREAM_BUFFER
        if(err == ECNR_ERR_OK)
            ecnr_alsa_feature_stream_duplicate(feature);
        #endif
    }

    ecnr_configuration_is_io_ctrl_select(ecnr_configuration_get_dp(ecnr_config_id), feature);

#else
    /* handling around WB and BN feature*/
    (void)ecnr_alsa_is_feature_WB_active(feature, class);

    /* handling around WB and BN feature*/
    (void)ecnr_alsa_is_feature_NB_active(feature, class);

    /* handling over CP (car play) feature*/
    (void)ecnr_alsa_is_feature_CP_active(feature, class);
#endif

}


static gint ecnr_alsa_check_operating_mode(void)
{
    gint err = ECNR_ERR_OK;

    if(!state)
    {
        pr_warning("no alsa module instance available\n");
        err = ECNR_ERR_ALSA_NO_INSTANCE;
        return err ;
    }


    if(state->ecnr_op_mode == sseVoiceRecog)
    {
        if(state->ecnr_asr_mode == sseSpeechProbLSB)
        {
            state->ecnr_enable_asr_preprocessing = TRUE;
            pr_debug("enable asr preprocessing\n");
        }
        else if(state->ecnr_asr_mode == sseSpeechProbAttached)
        {
            state->ecnr_enable_asr_preprocessing = FALSE;
            pr_critical("disable asr prepocessing. the mode sseSpeechProbAttached is not supported\n");
            err = ECNR_ERR_INVALID_ASR_MODE;
        }
        else
        {
        /* do nothing */
        }
    }
    else
    {
        state->ecnr_enable_asr_preprocessing = FALSE;

        /* handsfree mode */
        state->ecnr_asr_mode = -1;
    }


    /* emit always asr mode */
    (void)ecnr_object_emit_signal_ecnr_asr_mode_enable((gint)state->ecnr_enable_asr_preprocessing);

    pr_debug("operating mode: %d , ASR mode: %d \n", state->ecnr_op_mode, state->ecnr_asr_mode);

    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_initialize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint ecnr_alsa_audio_initialize(void)
{
    gint alsa_index, bd_index;
    gint err = 0;

    pr_debug("ENTERED");

    /* get a copy of the ecnr parameters we need */
    ecnr_alsa_get_ecnr_parameters();

    if(state)
    {
      state->ecnr_mic_lvl_watch = FALSE;
      state->ecnr_mic_lvl = 0;
    }

    /* walk through the configured ecnr feature */
    ecnr_alsa_check_feature_setting();

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM
    state->ecnr_alsa_multi_chan_in_dev_active = FALSE;
#endif
    /*
     *  set Input devices
     */
    alsa_index = 0;
    bd_index = 0;

    if(state->rstdata)
    {
        state->rstdata->adevs = g_malloc0(sizeof(trts_cfgadev)* NUM_ECNR_AUDIO);
    }
    else
    {
        err = ECNR_ERR_NULL_POINTER;
        goto error_out;
    }

    /* MicIn0 - this input is required by the ecnr filter */
    if (!state->ecnr_alsa_multi_chan_in_dev_active)
    {
        err = ecnr_alsa_audio_initialize_mic_in_0(&alsa_index, &bd_index);
    }
    else
    {
        err = ecnr_alsa_audio_initialize_multi_chan_mic_in_0(&alsa_index, &bd_index);
    }
    if (err)
        goto error_out;


    /* MicIn1 - this input is optional */
    if (state->ecnr_mic_in_cnt > 1)
    {
        if (!state->ecnr_alsa_multi_chan_in_dev_active)
        {
            err = ecnr_alsa_audio_initialize_mic_in_1(&alsa_index, &bd_index);
        }
        else
        {
            err = ecnr_alsa_audio_initialize_multi_chan_mic_in_1(&alsa_index, &bd_index);
        }
        if (err)
            goto error_out;
    }

    /* RefIn - this input is optional */
    if (state->ecnr_ref_in_cnt > 0)
    {
        if (!state->ecnr_alsa_multi_chan_in_dev_active)
        {
            err = ecnr_alsa_audio_initialize_ref_in(&alsa_index, &bd_index);
        }
        else
        {
            err = ecnr_alsa_audio_initialize_multi_chan_ref_in(&alsa_index, &bd_index);
        }
        if (err)
            goto error_out;
    }

    /* RecvIn - this input is optional */
    if (state->ecnr_recv_in_cnt > 0)
    {
        err = ecnr_alsa_audio_initialize_recv_in(&alsa_index, &bd_index);
        if (err)
            goto error_out;
    }


    /*
    * ALSA Outputs
    */
    alsa_index = 0;
    bd_index = 0;

    /* MicOut - this output is required */
    err =  ecnr_alsa_audio_initialize_mic_out(&alsa_index, &bd_index);
    if (err)
        goto error_out;

    /* RecvOut - this output is optional */
    if (state->ecnr_recv_in_cnt > 0)
    {
        err = ecnr_alsa_audio_initialize_recv_out(&alsa_index, &bd_index);
        if (err)
            goto error_out;
    }

#ifdef D_DUPLICATE_MIC_0_STREAM_BUFFER
    if (state->ecnr_dup_mic_in_0)
    {
        err = ecnr_alsa_audio_initialize_dup_mic_out(&alsa_index, &bd_index);
        if (err)
            goto error_out;
    }
#endif

    /* set up rts*/
#ifdef D_ENABLE_DEV_ORDER_CHECK
    err = ecnr_alsa_rts_check_dev_order();
#endif
    err = ecnr_alsa_rts_set_streams();

    ecnr_audio_print_buffer_pointer(); /* only for debug*/
    ecnr_alsa_rst_set_dev_prefill_level();

    /*new: call always ecnr_audio_rts_handler_destroy prior to a new create */
    (void)ecnr_audio_rts_handler_destroy(state);
    err = ecnr_audio_rts_handler_create(state, TRUE);
    if (err)
        goto error_out;

 #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    /* initialize only for qwa rcs */
    if(ecnr_qwa_rcs_get_active_state())
    {
        err = ecnr_qwa_rcs_handler_initialize();
        if (err)
            goto error_out;
    }
 #endif
    /* Device configuration for the next initialization is always set as default mode*/
    ecnr_alsa_enable_dev_configuration(NUM_ECNR_AUDIO);
    ecnr_alsa_dev_order_ctrl(TRUE);
    ecnr_alsa_prefill_ctrl(TRUE);

    pr_message("initialization done");
    return 0;

error_out:
    pr_message("initialization failed with error: %d ", err );
    ecnr_alsa_free_all_buffer_descriptors();
    ecnr_alsa_close_all_streams();
    ecnr_alsa_free_all_ecnr_buffers();
    ecnr_alsa_rst_close();
    ecnr_alsa_enable_dev_configuration(NUM_ECNR_AUDIO);
    ecnr_alsa_dev_order_ctrl(TRUE);
    ecnr_alsa_prefill_ctrl(TRUE);
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    if( ecnr_qwa_rcs_get_active_state())
        ecnr_qwa_rcs_handler_finalize();
#endif
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_set_scheduling_priority
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_set_scheduling_priority(void)
{
    gint err;
    struct sched_param sched_param;

    err = sched_getparam(0, &sched_param);
    if (err < 0)
    {
        pr_warning("sched_getparam error %d", err);
        return;
    }

    sched_param.sched_priority = ecnr_alsa_get_thread_priority();
    err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
    if (!err)
    {
        pr_message("Priority of ecnr_alsa_thread set to SCHED_FIFO %d", sched_param.sched_priority);
    }
    else
        pr_warning("sched_setscheduler error %d when attempting to set priority SCHED_FIFO %d", err, sched_param.sched_priority);
}


/* currently not set because still in evaluation in Project */
#if 0 // def VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
#define D_WA_XRUN_DETECTION
#endif


#ifdef D_WA_XRUN_DETECTION
#define D_WA_XRUN_DETECTION_CNT  33


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_xrun_occured
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gboolean ecnr_alsa_is_xrun_occured(void)
{
    gboolean bXRUN = FALSE;
    unsigned int xrun = 0;


    /* retrieve current xrun value in rts*/
    ecnr_audio_rts_handler_get_xrun(&xrun);

    if(xrun)
    {
        bXRUN = TRUE;
        pr_message("current xrun count(%d) \n",xrun);
    }

    return bXRUN;
}

#endif // D_WA_XRUN_DETECTION

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_thread
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
static gpointer ecnr_alsa_thread(gpointer data)
{
    gint err = 0, ecnr_err = 0;
    gint audio_started;
    gboolean debug_active_state = FALSE;
    gint restart = 0;
    //gint retry = 0;
    gint wr_success_recover_cnt = 0;
#ifdef D_WA_XRUN_DETECTION
    gint rw_nb = 0;
#endif

    /*
     * Suppress lint Info 715 "Symbol not referenced" message regarding
     * the formal parameter 'data'.
     */

    /*lint -efunc(715,ecnr_alsa_thread) */

    /*
     * Set the scheduling priority once before the loop starts so that the
     * audio thread has a real-time priority while waiting for the first
     * stream.  This avoids having a potentially longer wakeup latency on
     * the first stream than on subsequent streams.  We also set the
     * priority of the audio thread immediately after waking up so that
     * any changes to the priority will take effect.
     */

    pr_message("start ecnr alsa streaming thread");
    ETG_TRACE_USR3(("[ecnr_alsa_thread]: ENTERED"));

    //ecnr_alsa_print_current_audio_state();
    //ecnr_alsa_set_scheduling_priority();
    //ecnr_audio_rts_handler_clear_xrun();

    while (!state->exit)
    {
        pr_debug("entry top audio streaming loop");

        g_mutex_lock(&state->lock);
        if (err)
        {
            state->run = FALSE;
            ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_STOP);
        }
        //else
        //    ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_SILENCE);

        if (restart)
            state->run = TRUE;
        state->initialized = FALSE;

        /* alsa stop was terminated,
           the main thread which runs the ecnr_alsa_stop function is woken up
        */
        g_cond_broadcast(&state->update);

        /* initialize thread for next run*/
        err = ecnr_err = 0;
        restart = 0;
        state->retry = 0;

        //ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_SILENCE);

        /* wait release from ecnr_alsa_start() function */
        while (!state->run && !state->exit)
        {
            if(state->req_stop) /*new : a stop request is pending, trigger termination */
            {
               pr_message("pending stop audio streaming request\n");
               ETG_TRACE_USR3(("[ecnr_alsa_thread]: pending stop audio streaming request"));
               g_cond_broadcast(&state->update);
               state->req_stop = FALSE;
            }
            else /*new*/
            {
                pr_message("audio streaming awaiting\n");
                ETG_TRACE_USR3(("[ecnr_alsa_thread]: audio streaming awaiting"));
            }
            g_cond_wait(&state->update, &state->lock);
        }

        g_mutex_unlock(&state->lock);

        /* audio thread terminated with call ecnr_alsa_exit() ? */
        if (state->exit)
            continue;

        /* is the tuning mode activated ?*/
        err = ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_INIT);
        debug_active_state = ecnr_engine_is_debug_session_active();

        audio_started = 0;

        /* run audio streaming */
        while (!err && !restart && state->run && !state->exit)
        {
            /* read audio frames from input stream devices*/
            err = ecnr_alsa_read_all();
            if (err)
                goto rts_err;

            /* first execute pending request */
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            (void)ecnr_sse_process_pending_req();
#endif
            /* debug or tuning mode active?*/
            if(debug_active_state)
            {
                /* process debug data from tuning tools */
                err = ecnr_err = ecnr_engine_handler_process_audio_with_debug_data(state, &restart);
                if(restart)
                    pr_debug("restart rcs target was requested (%d)\n", restart);
            }
            else
                /* process audio stream data only */
                err = ecnr_err = ecnr_engine_handler_process_audio(state);

            /* calculate microphone level */
            if(state->ecnr_mic_lvl_watch)
               ecnr_alsa_mic_level_calculate();

            if (err || restart)
                continue;

            /* write audio frames to output stream devices*/
            err = ecnr_alsa_write_all();
            if (err)
                goto rts_err;

            /* count audio streaming loop as long as the initialization state is not reported*/
            if(!audio_started || !state->initialized)
                audio_started++;

            if(state->initialized && state->retry)
                wr_success_recover_cnt++;

rts_err:

            /* insert retry, with err ENODATA (61) or EPIPE (32)  */
            if( err == -ENODATA || err == -EPIPE  )
            {
                pr_debug("rts streaming failed with err(%d) and current retry count is (%d)\n",err, state->retry);

                err = ecnr_audio_rts_handler_recover(state);
                if(err)
                {
                    pr_warning("rts recover failed with err(%d) and current retry count is (%d)\n", err, state->retry);
                    ETG_TRACE_USR3(("[ecnr_alsa_thread]: rts recover failed with err -> %d and current retry count -> %d", err, state->retry));
                }

                err = ECNR_ERR_OK;

                if(state->initialized)
                    ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_WAIT_RECOVER);

                state->retry++;

                /* state no audio streaming */
                audio_started = 0;
                wr_success_recover_cnt = 0;
            }

            /*audio was started without failure, streaming is started  */
            if(    (audio_started > D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ)
                && !state->initialized)
            {
                g_mutex_lock(&state->lock);
                state->initialized = TRUE;

                ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_RUNNING);
                ETG_TRACE_USR3(("[ecnr_alsa_thread]: change to state ECNR_AS_STREAM_RUNNING"));

                /* signal send to wake up from call within "ecnr_alsa_start()" function,
                   the return code is returned to object interface
                */
                g_cond_broadcast(&state->update);
                g_mutex_unlock(&state->lock);
                pr_debug("audio streaming running with audio_started value(%d)", audio_started ); //pr_message("Starting audio");
            }

            if(!state->initialized)
                ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_WAIT_DATA);

            /*audio was started without failure, streaming is started  */
            if(    (wr_success_recover_cnt > D_MIN_R_W_COUNT_BEFORE_SIGNAL_READ)
                &&  state->initialized)
            {
                ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_RUNNING);
                ETG_TRACE_USR3(("[ecnr_alsa_thread]: change to state ECNR_AS_STREAM_RUNNING"));
                pr_message("Streaming recovered after %d retries \n", state->retry );
                state->retry = 0;
                wr_success_recover_cnt = 0;
            }


            /* request for log statistic, should be done from alsa_thread context,
               otherwise rts lib assert an abort.
               req_statistic flag is released within ecnr_alsa_print_current_audio_state()
            */
            if(state->req_statistic)
            {
                /* fix me */
                //ecnr_alsa_print_current_audio_state();
            }

            /* issue distortion heal from xrun*/
#ifdef D_WA_XRUN_DETECTION
            if(   state->initialized
                &&(rw_nb > D_WA_XRUN_DETECTION_CNT))
            {
                if(ecnr_alsa_is_xrun_occured())
                {
                    err = ecnr_audio_rts_handler_destroy(state);
                    if(err == ECNR_ERR_OK)
                        err = ecnr_audio_rts_handler_create(state, FALSE);
                    pr_message("workaround: recover from XRUN");

                    /* increment count of restart */
                    state->devices_restart_after_xrun_count++;
                }
                rw_nb = 0;
            }
            else
            {
                rw_nb++;
            }
#endif
        }

        ETG_TRACE_USR4(("[ecnr_alsa_thread]: reason of terminatton of the ecnr processing loop : err -> %d, restart -> %d, run -> %d, exit -> %d",err,restart,state->run,state->exit));

        /* handle with error */
        if (err)
        {
            if (ecnr_err)
                ecnr_critical(ecnr_err,"audio terminated by ECNR error");
            else
            {
                ecnr_alsa_get_statistic();
                ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_ERROR);
                pr_critical("audio terminated by error %d",err);
            }
        }

        /*new*/
        if(state->initialized)
        {
            if(!audio_started)
            {
                pr_message("close after successfull streaming, variable audio_started is forced to 1\n");
                audio_started = 1;
            }
            pr_message("close after successfull streaming with audio_started(%d)\n", audio_started );
        }
        /*new*/
        ecnr_alsa_audio_finalize(audio_started);
        pr_message("audio streaming stopped");
    }

    ecnr_service_audio_change_state((ecnr_audio_state)ECNR_AS_STREAM_TERMINATED);
    g_cond_broadcast(&state->update); /*new*/


    pr_debug("thread terminated with run(%d), exit(%d), initialized(%d) and err(%d)\n",state->run, state->exit , state->initialized, err);
    pr_message("EXIT\n");
    ETG_TRACE_USR3(("[ecnr_alsa_thread]: EXIT"));

    return 0;
}


 /*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_thread_new
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint ecnr_alsa_audio_thread_new (void)
{
    gint err = ECNR_ERR_OK;
    GError *error = NULL;

    ETG_TRACE_USR3(("ecnr_alsa_audio_thread_new]: ENTERED"));
    pr_message("create audio alsa thread ENTERED");

    if(state)
    {
        pr_debug("current value state->exit(%d)\n", state->exit);

        g_mutex_lock(&state->lock);
        if (state->thread)
        {
            ETG_TRACE_USR3(("ecnr_alsa_audio_thread_new]: alsa thtread currently active"));
            pr_message("alsa thtread currently active\n");
            g_mutex_unlock(&state->lock);
            return ECNR_ERR_OK;
        }

        g_mutex_unlock(&state->lock);

        /* create a new thread only if not still exist */
        if (!state->thread)
        {
            state->thread = g_thread_try_new ("alsa_thread", ecnr_alsa_thread, NULL, &error);

            if (!state->thread)
            {
                pr_warning("Can't create ecnr_alsa_thread\n");
                ETG_TRACE_USR3(("ecnr_alsa_audio_thread_new]: Can't create ecnr_alsa_thread"));
                (void)ecnr_service_audio_terminate_cb();
                err = ECNR_ERR_ALSA_NO_INSTANCE;

            }
            else
            {
                pr_message("create new alsa thread \n");
                ETG_TRACE_USR3(("ecnr_alsa_audio_thread_new]: create new alsa thread"));
                ecnr_alsa_audio_started();

            }
        }
        else
        {
            pr_warning("thread already exist\n");
        }

        //g_mutex_unlock(&state->lock);
    }

    pr_message("EXIT\n");

    return err;
 }


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_active_is
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*               true : alsa thread is active
*               false: alsa thread was terminated
*******************************************************************************/
extern gboolean ecnr_alsa_audio_active_is(void)
{
    gboolean retstate = FALSE;

    if(state)
    {
        g_mutex_lock(&state->lock);
        retstate = state->athread_active;
        pr_debug("current alsa audio thread state is %d",retstate);
        g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_audio_active_is]: ecnr alsa instance 'state' -> is NULL"));
    }

    return retstate;
}





/*******************************************************************************
*
* FUNCTION: ecnr_alsa_audio_started
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
extern void ecnr_alsa_audio_started(void)
{

    pr_message("ENTERED\n");
    if(state)
    {
        g_mutex_lock(&state->lock);
        state->athread_active = TRUE;

        /* new audio streaming thread instance */
        state->athread_create_count++;

        pr_message("current alsa_thread active state -> %d\n", state->athread_active);

        g_mutex_unlock(&state->lock);

        /* increment ref count alsa thtread in service module */
        //ecnr_service_alsa_thread_ref();
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_audio_started]: ecnr alsa instance 'state' -> is NULL"));
    }
    pr_message("EXIT\n");

    return;
}

/*
 * Routines to initialize and finalize the audio handling thread.
 */
/*******************************************************************************
*
* FUNCTION: ecnr_alsa_finalize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_finalize(void)
{
    pr_message("remove all system resources owned by the module ecnr_alsa");
    ETG_TRACE_USR3(("[ecnr_alsa_finalize]: ENTERED"));

    if(state)
    {
        /* wait a start audio request is terminated */
        while(state->req_start_stop_audio)
            usleep(1000);


        /* free all resources */
        ecnr_alsa_free_all_buffer_descriptors();
        ecnr_alsa_close_all_streams();
        ecnr_alsa_free_all_ecnr_buffers();

        /* free rts related resources */
        if(state->rstdata)
        {
            //if(state->rstdata)
            g_free(state->rstdata);
            state->rstdata=NULL;
        }

        /* free the ALSA device names */
        ecnr_alsa_set_dev_name(ECNR_MIC_IN0, NULL);
        ecnr_alsa_set_dev_name(ECNR_MIC_IN1, NULL);
        ecnr_alsa_set_dev_name(ECNR_MIC_OUT, NULL);
        ecnr_alsa_set_dev_name(ECNR_RECV_IN, NULL);
        ecnr_alsa_set_dev_name(ECNR_RECV_OUT, NULL);
        ecnr_alsa_set_dev_name(ECNR_REF_IN, NULL);
        ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_DEV, NULL);
        ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_MIC_0, NULL);
        ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_MIC_1, NULL);
        ecnr_alsa_set_dev_name(ECNR_MULTI_CHAN_IN_REF, NULL);

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        ecnr_qwa_handler_close();
#endif

        (void)ecnr_service_audio_get_ref();

#if 0
        g_mutex_clear(&state->lock);
        g_cond_clear(&state->update);

        //(void)ecnr_service_audio_get_ref();

        if(state)
            g_free(state);
        state = NULL;
#endif

    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_finalize]: ecnr alsa instance 'state' -> is NULL"));
    }

    ETG_TRACE_USR3(("[ecnr_alsa_finalize]: EXIT"));
    pr_message("EXIT");

    return;
}

static void ecnr_alsa_configuration_init_dev_name(void)
{
    gint devid = 0;
    const gchar* dev = NULL;
    gboolean is_default_dev = FALSE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_alsa_configuration_init_dev_name]: ENTERED"));

    for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
    {
        dev = ecnr_configuration_get_default_dev_name((enum ecnr_audio)devid, &is_default_dev);
        ecnr_alsa_set_dev_name((enum ecnr_audio)devid, dev);

        /* use case specifical device handling */
        if( (is_default_dev == TRUE )
            && (ECNR_MULTI_CHAN_IN_DEV == (enum ecnr_audio)devid)
            && !g_strcmp0(dev, "AdevMicro12AmpRef")
          )
        {
            /* set multichannel */
            pr_debug("set multichannel dev active");
            ETG_TRACE_USR4(("[ecnr_alsa_configuration_init_dev_name]: set multichannel dev active"));
            state->ecnr_alsa_multi_chan_in_dev_active = TRUE;
        }

        /* init flag device configuraiton control, per default configuration is available */
        state->dev_configuration_ctrl[(enum ecnr_audio)devid] = TRUE;
    }

    pr_message("EXIT\n");
    ETG_TRACE_USR3(("[ecnr_alsa_configuration_init_dev_name]: EXIT"));
}

static void ecnr_alsa_configuration_init_num_channels(void)
{
    gint devid = 0;
    unsigned int val = 0;
    ETG_TRACE_USR3(("[ecnr_alsa_configuration_init_num_channels]: ENTERED"));

    for(devid=ECNR_MIC_IN0 ;devid < NUM_ECNR_AUDIO ;devid++)
    {
        val = ecnr_configuration_get_default_num_channels((enum ecnr_audio)devid);
        ecnr_alsa_set_num_channels((enum ecnr_audio)devid, val);
    }
}

static void ecnr_alsa_configuration_init_active_channel(void)
{
    gint devid = 0;
    unsigned int val = 0;

    ETG_TRACE_USR3(("[ecnr_alsa_configuration_init_active_channel]: ENTERED"));

    for(devid=ECNR_MIC_IN0 ;devid < NUM_ECNR_AUDIO ;devid++)
    {
        val = ecnr_configuration_get_default_active_channel((enum ecnr_audio)devid);
        ecnr_alsa_set_channel((enum ecnr_audio)devid, val);
    }
}

static void ecnr_alsa_configuration_init_dev_open_order(void)
{
    gint devid = 0;
    unsigned int val = 0;

    ETG_TRACE_USR3(("[ecnr_alsa_configuration_init_dev_open_order]: ENTERED"));

    for(devid=ECNR_MIC_IN0 ;devid < NUM_ECNR_AUDIO ;devid++)
    {
        val = ecnr_configuration_get_default_dev_open_order((enum ecnr_audio)devid);
        ecnr_alsa_set_device_open_order((enum ecnr_audio)devid, val);
    }
}


static void ecnr_alsa_print_dev_configuration(void)
{
    gint devid = 0;


    ETG_TRACE_USR3(("[ecnr_alsa_print_dev_configuration]: ENTERED"));

    if(state)
    {
        for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ; devid++)
        {
            pr_debug("default_dev[%d] : %s\n", devid, state->alsa_dev_name[devid]);
        }

        for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
        {
            pr_debug("default_dev_num_channels[%d] : %d\n", devid, state->num_channels[devid]);
        }

        for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
        {
            pr_debug("default_dev_active_channel[%d] : %d\n", devid, state->channel[devid]);
        }

        for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
        {
            pr_debug("default_dev_order[%d] : %d\n", devid, state->dev_order[devid]);
        }
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_print_dev_configuration]: ecnr alsa instance 'state' -> is NULL"));
    }

    return;
}

gboolean  ecnr_alsa_new(void)
{

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_new]: ENTERED"));

    if(state)
    {
        pr_message("alsa module instance already active \n");
        ETG_TRACE_ERR(("[ecnr_alsa_new]: alsa module instance already active"));
        return TRUE;
    }

    /* create data struture for alsa module meneber variables */
    state = g_malloc0(sizeof(*state));
    if(!state)
    {
        pr_critical("fails to allocate alsa module instance\n");
        ETG_TRACE_ERR(("[ecnr_alsa_new]: fails to allocate alsa module instance"));
        return FALSE;
    }
    /* create sync objects*/
    g_mutex_init(&state->lock);
    g_cond_init(&state->update);
    state->thread = NULL;
    ecnr_audio_recording_new();

    /* parameter which are initialize only once */
    state->ecnr_debug_active = FALSE; /*fix me*/

    return TRUE;
}

void ecnr_alsa_delete(void)
{

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_delete]: ENTERED"));

    if(state)
    {
        g_mutex_clear(&state->lock);
        g_cond_clear(&state->update);

        if(state)
            g_free(state);
        state = NULL;

        pr_critical("delete alsa module data instance \n");
        ETG_TRACE_ERR(("[ecnr_alsa_delete]: delete alsa module data instance"));
    }

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_initialize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_initialize(void)
{

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_initialize]: ENTERED"));

    /* create instnce alsa module member data structure */
    if(!ecnr_alsa_new())
    {
        pr_critical("fails to allocate memory for the alsa instance 'state' -> is NULL\n");
        ETG_TRACE_ERR(("[ecnr_alsa_initialize]: fails to allocate memory for the alsa instance"));
        return;
    }


    //state->ecnr_debug_active = FALSE;
    state->ecnr_alsa_multi_chan_in_dev_active = FALSE;
    state->enable_dev_configuration = TRUE;
    state->enable_force_dev_fill_per_feature_setting = FALSE;
    state->ecnr_alsa_prefill_ctrl = TRUE;
    state->ecnr_alsa_dev_order_ctrl = TRUE;
    state->AsrPrepare = (short int*)NULL;
    state->ecnr_asr_mode = -1;
    state->ecnr_dup_mic_in_0 = FALSE;

    /* initiate audio thread state und action */
    state->exit = FALSE;
    state->run = FALSE;
    state->req_stop = FALSE;
    state->initialized = FALSE;
    state->retry = 0;
    state->athread_create_count = 0;
    state->athread_terminate_count = 0;
    state->req_statistic = FALSE;
    state->req_start_stop_audio = FALSE;
    state->devices_restart_after_xrun_count=0;


    /* create rts data structure*/
    if(state->rstdata )
        g_free(state->rstdata);

    state->rstdata = g_malloc0(sizeof(trts_cfg));

    /* set default names for the ALSA devices */
    ecnr_alsa_configuration_init_dev_name();

    /* set default number of channels for the ALSA devices */
    ecnr_alsa_configuration_init_num_channels();

    /* set default channel for the ALSA devices */
    ecnr_alsa_configuration_init_active_channel();

    /* set default channel for the ALSA devices */
    ecnr_alsa_configuration_init_dev_open_order();

    /* print the device configuration */
    ecnr_alsa_print_dev_configuration();

    /* set default prefill value of the alsa devices */
    ecnr_alsa_set_device_prefill_value(RTS_DEFAULT_SILENCE_PREFILL);

    /* set default scheduling priority for the ALSA thread */
    ecnr_alsa_set_thread_priority(sched_get_priority_max(SCHED_FIFO) - 1);

    /* initiate the configuration module */
    ecnr_configuration_init();
    ecnr_audio_rts_handler_init();


#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    ecnr_qwa_handler_init();
#endif

    pr_message("EXIT");
    ETG_TRACE_USR3(("[ecnr_alsa_initialize]: EXIT"));
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_reinitialize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_reinitialize(void)
{

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_alsa_reinitialize]: ENTERED"));


    /* here initialize alsa module variable */
    ecnr_alsa_initialize();

    /* enable device configuration to true */
    ecnr_alsa_enable_dev_configuration(NUM_ECNR_AUDIO);
    state->ecnr_enable_asr_preprocessing = FALSE;

    /* init value */
    state->ecnr_asr_mode = -1;

    pr_message("EXIT");
    ETG_TRACE_USR3(("[ecnr_alsa_reinitialize]: EXIT"));
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_get_instance
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
stecnr_alsa_state* ecnr_alsa_get_instance(void)
{
   return state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_enable_sse_debug
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

int ecnr_alsa_enable_sse_debug(void)
{
    int ret = ECNR_ERR_OK;

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->ecnr_debug_active = TRUE;
        pr_message("Debug mode enabled");
        g_mutex_unlock(&state->lock);
    }
    else
        ret = ECNR_ERR_ALSA_NO_INSTANCE;

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_disable_sse_debug
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_alsa_disable_sse_debug(void)
{
    int ret = ECNR_ERR_OK;

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->ecnr_debug_active = FALSE;
        pr_message("Debug mode disabled");
        g_mutex_unlock(&state->lock);
    }
    else
        ret = ECNR_ERR_ALSA_NO_INSTANCE;

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_is_sse_debug_active
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/

gboolean  ecnr_alsa_is_sse_debug_active(void)
{
    gboolean retstate = FALSE;

    if(state)
    {
        g_mutex_lock(&state->lock);
        retstate = state->ecnr_debug_active;
        g_mutex_unlock(&state->lock);
    }

    pr_message("current sse debug state is %d",retstate);
    return retstate;
}




/*******************************************************************************
*
* FUNCTION: ecnr_alsa_disable_dev_configuration
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
extern int ecnr_alsa_disable_dev_configuration(enum ecnr_audio ecnr_id)
{
    int err = ECNR_ERR_OK;
    gint devid = 0;

    if(state)
    {
        g_mutex_lock(&state->lock);

        if(ecnr_id > NUM_ECNR_AUDIO)
            err = ECNR_ERR_INVALID_SIZE;
        else
        {
            if(ecnr_id == NUM_ECNR_AUDIO)
            {
                for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
                {
                    /* init flag device configuration control, per default configuration is available */
                    state->dev_configuration_ctrl[(enum ecnr_audio)devid] = FALSE;
                }
                state->enable_dev_configuration = FALSE;
            }
            else
                state->dev_configuration_ctrl[(enum ecnr_audio)ecnr_id] = FALSE;
        }
        //state->enable_dev_configuration = FALSE;
        g_mutex_unlock(&state->lock);
    }
    else
        err = ECNR_ERR_ALSA_NO_INSTANCE;

    pr_message("device configuration is disabled");
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_enable_dev_configuration
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
extern int ecnr_alsa_enable_dev_configuration(enum ecnr_audio ecnr_id)
{
    int err = ECNR_ERR_OK;
    gint devid = 0;

    ETG_TRACE_USR3(("[ecnr_alsa_enable_dev_configuration]: ENTER"));

    if(state)
    {
        g_mutex_lock(&state->lock);

        if(ecnr_id > NUM_ECNR_AUDIO)
            err = ECNR_ERR_INVALID_SIZE;
        else
        {
            if(ecnr_id == NUM_ECNR_AUDIO)
            {
                ETG_TRACE_USR3(("[ecnr_alsa_enable_dev_configuration]: initialize dev control"));

                for(devid = ECNR_MIC_IN0 ; devid < NUM_ECNR_AUDIO ;devid++)
                {
                    /* init flag device configuration control, per default configuration is available */
                    state->dev_configuration_ctrl[(enum ecnr_audio)devid] = TRUE;
                }
                state->enable_dev_configuration = TRUE;
            }
            else
                state->dev_configuration_ctrl[(enum ecnr_audio)ecnr_id] = TRUE;
        }
        //state->enable_dev_configuration = TRUE;
        g_mutex_unlock(&state->lock);
    }
    else
        err = ECNR_ERR_ALSA_NO_INSTANCE;

    ETG_TRACE_USR3(("[ecnr_alsa_enable_dev_configuration]: EXIT"));
    pr_message("device configuration is enabled");
    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_prefill_ctrl
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_alsa_prefill_ctrl(gboolean prefill)
{
    int ret = ECNR_ERR_OK;

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->ecnr_alsa_prefill_ctrl = prefill;
        g_mutex_unlock(&state->lock);
    }
    else
        ret = ECNR_ERR_ALSA_NO_INSTANCE;

    pr_message("current prefill control is (%d)\n", prefill);
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_dev_order_ctrl
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int ecnr_alsa_dev_order_ctrl(gboolean dev_order)
{
    int ret = ECNR_ERR_OK;

    if(state)
    {
        g_mutex_lock(&state->lock);
        state->ecnr_alsa_dev_order_ctrl = dev_order;
        g_mutex_unlock(&state->lock);
    }
    else
        ret = ECNR_ERR_ALSA_NO_INSTANCE;

    pr_message("current device order control is (%d)\n", dev_order);
    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_print_current_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_print_current_audio_state(void)
{
    if(state)
    {
        pr_message("print current audio state\n");
        ETG_TRACE_USR3(("[ecnr_alsa_print_current_audio_state]: print current audio state"));

        /* print audio streaminng thread state */
        pr_message("**** current state control variable audio streaming thread\n");
        pr_message("state->run                              : %d\n", state->run);
        pr_message("state->exit                             : %d\n", state->exit);
        pr_message("state->initialized                      : %d\n", state->initialized);
        pr_message("state->athread_active                   : %d\n", state->athread_active);
        pr_message("state->athread_create_count             : %d\n", state->athread_create_count);
        pr_message("state->athread_terminate_count          : %d\n", state->athread_terminate_count);
        pr_message("state->devices_restart_after_xrun_count : %d\n", state->devices_restart_after_xrun_count);

        pr_message("**** print current rts state\n");



        /* print rts audio statistic */
        ecnr_alsa_get_statistic();

        /*fix me*/
        //g_mutex_lock(&state->lock);
        state->req_statistic = FALSE;
        //g_mutex_unlock(&state->lock);
    }
    else
    {
        pr_critical("ecnr alsa instance 'state' -> is NULL\n");
        ETG_TRACE_USR3(("[ecnr_alsa_print_current_audio_state]: ecnr alsa instance 'state' -> is NULL"));
    }

    ETG_TRACE_USR3(("[ecnr_alsa_print_current_audio_state]: EXIT"));
    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_alsa_request_print_current_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void ecnr_alsa_request_print_current_audio_state(void)
{

    /* call statistic directly only if the audio streaming thread is not active */
    if(ecnr_audio_rts_get_state(state))
    {
        g_mutex_lock(&state->lock);
        state->req_statistic = TRUE;
        g_mutex_unlock(&state->lock);
    }
    else
        ecnr_alsa_print_current_audio_state();

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_notify_asr_mode_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint  ecnr_alsa_notify_asr_mode_state(void)
{
    gint err = ECNR_ERR_OK;

    if(!state)
    {
        pr_warning("no alsa module instance available\n");
        return ECNR_ERR_NULL_POINTER;
    }

    /* retrieve asr mode releavnt sse parameter */
    state->ecnr_op_mode       = ecnr_get_op_mode();
    state->ecnr_asr_mode      = (int)ecnr_get_asr_mode();
    state->ecnr_asr_data_type = ecnr_get_audio_type_for_asr(&state->ecnr_asr_data_type_size);

    /* notify state to the client */
    err = ecnr_alsa_check_operating_mode();

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_alsa_asr_mode_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
gint  ecnr_alsa_asr_mode_state(void)
{

  if(!state)
  {
    pr_warning("no alsa module instance available\n");
    return FALSE;
  }

  pr_debug("current asr mode active state: %d, asr mode: %d \n", state->ecnr_enable_asr_preprocessing, state->ecnr_asr_mode);


  return state->ecnr_asr_mode;
}


/******************************************************************************
 * FUNCTION     : ecnr_alsa_set_mic_level_watch
 * DESCRIPTION  : set the MIC level monitor watch either to start/stop
 * PARAMETER    : watch_state -> TRUE for Start/FALSE for Stop
 * RETURN VALUE : OK on valid ALSA instance else error code
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
int ecnr_alsa_set_mic_level_watch(gboolean watch_state)
{
  int status = ECNR_ERR_OK;
  pr_debug("ENTERED\n");

  if(state)
  {
        g_mutex_lock(&state->lock);
    state->ecnr_mic_lvl_watch = watch_state;
        g_mutex_unlock(&state->lock);
  }
  else
    status = ECNR_ERR_ALSA_NO_INSTANCE;

  pr_debug("EXIT\n");
  return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_alsa_get_mic_level_watch
 * DESCRIPTION  : get the current state of microphone level monitoring
 * PARAMETER    : none
 * RETURN VALUE : returns TRUE/FALSE based on watch state
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
gboolean ecnr_alsa_get_mic_level_watch(void)
{
  gboolean ecnr_mic_lvl_watch_state = FALSE;
  pr_debug("ENTERED\n");

  if(state)
  {
        g_mutex_lock(&state->lock);
        ecnr_mic_lvl_watch_state = state->ecnr_mic_lvl_watch;
        g_mutex_unlock(&state->lock);
  }

  pr_debug("EXIT\n");
  return ecnr_mic_lvl_watch_state;
}

/******************************************************************************
 * FUNCTION     : ecnr_alsa_mic_level_calculate
 * DESCRIPTION  : calculation of microphone levels using processed
 *              : MIC out signals and the number of samples and the levels are
 *              : stored in state variable.
 * PARAMETER    : void
 * RETURN VALUE : OK on valid ALSA instance else error code
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Patrick Rey             | Initial Revision
 *****************************************************************************/
int ecnr_alsa_mic_level_calculate(void)
{
  int status = ECNR_ERR_OK;
  pr_debug("ENTERED\n");

  if(state)
  {
    int          index = 0;
    int    sample_num  = state->out_buffer[0]->num_samples;
    short int* samples = *state->MicOut[0];
    double fsample_row, fsample_abs;

    for ( index = 0; index < sample_num ; index++)
    {
      fsample_row = (float)samples[index];

      fsample_abs = fabs(fsample_row);
      fsample_row = fsample_abs * fsample_abs;
      fintegrator = fsample_row * FLOAT_KOEFF + fintegrator * (1-FLOAT_KOEFF);

      if (fenvelope < fsample_abs)
      {
        fenvelope = fsample_abs;
      }
      else
      {
        fenvelope = fenvelope * AUD_FLOAT_ENVELOP_DELAY;
      }
    }

    fsample_abs      = sqrt(fintegrator);
    state->ecnr_mic_lvl = (guint32)fsample_abs;

    pr_message("microphone level -> (%u)\n",state->ecnr_mic_lvl);
    ETG_TRACE_USR4(("[ecnr_alsa_mic_level_calculate]: microphone level -> (%u)", state->ecnr_mic_lvl));
  }
  else
    status = ECNR_ERR_ALSA_NO_INSTANCE;

  pr_debug("EXIT\n");
  return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_alsa_get_mic_level
 * DESCRIPTION  : get stored microphone level
 * PARAMETER    : none
 * RETURN VALUE : stored microphone level
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
guint32 ecnr_alsa_get_mic_level(void)
{
  guint32 ecnr_mic_lvl = 0;
  pr_debug("ENTERED\n");

  if(state)
  {
    g_mutex_lock(&state->lock);
    ecnr_mic_lvl = state->ecnr_mic_lvl;
    g_mutex_unlock(&state->lock);
  }

  pr_debug("EXIT\n");
  return ecnr_mic_lvl;
}


