/*!
*******************************************************************************
* \file              spi_tclWBLClientHandler.h
* \brief             WBL Client handler class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    WBL Client handler class
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                           | Modifications
 17.11.2016 |  Unmukt Jain (RBEI/ECP2)          | Initial Version
\endverbatim
******************************************************************************/

#ifndef _SPI_TCLWBLCLIENTHANDLER_H_
#define _SPI_TCLWBLCLIENTHANDLER_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/

//!Include Application Help Library.

#include "SPITypes.h"
#include "org-bosch-wbl-conflictmanagement-generated.h"
#include "org-bosch-wbl-wifisetup-generated.h"
#include "org-freedesktop-properties.h"
#include "spi_WifiTypedefs.h"
#include "Threadable.h"
#include <map>
#include <string>



/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/* Forward Declarations. */

/*!
* \class spi_tclWBLClientHandler
* \brief Wifi client handler class that realizes the WifiPolicy interface.
*/
class spi_tclWBLClientHandler : public shl::thread::Threadable
{

public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclWBLClientHandler::spi_tclWBLClientHandler(ahl_tclBaseOneThreadApp...)
   **************************************************************************/
   /*!
   * \fn      spi_tclWBLClientHandler()
   * \brief   Constructor
   **************************************************************************/
	   spi_tclWBLClientHandler();

   /***************************************************************************
   ** FUNCTION:  spi_tclWBLClientHandler::~spi_tclWBLClientHandler()
   **************************************************************************/
   /*!
   * \fn      ~spi_tclWBLClientHandler()
   * \brief   Destructor
   **************************************************************************/
   virtual ~spi_tclWBLClientHandler();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclWBLClientHandler::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterCallbacks(trWiFiCallbacks rWiFiCallbacks)
   * \brief   Interface to register for Wifi callbacks.
   * \param   [IN] rWiFiCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterCallbacks(trWiFiCallbacks rWiFiCallbacks);

   /***************************************************************************
   ** FUNCTION:  spi_tclWBLClientHandler::vConfigureWiFi(tenDeviceCategory enDevCat)
   **************************************************************************/
   /*!
   * \fn      spi_tclWBLClientHandler::vConfigureWiFi(tenDeviceCategory enDevCat)
   * \brief   WiFi applications (which uses the STA, AP, P2P wifi modes) initially
			  shall request WBL to prepare their setup before they start any operations.
   * \param   enDevCat: Device type for which WiFi AP needs to be configured
   **************************************************************************/
   t_Void vConfigureWiFi(tenDeviceCategory enDevCat);
   
   /***************************************************************************
    ** FUNCTION: t_Void spi_tclWBLClientHandler::vSetWiFiLimitationMode()
    ***************************************************************************/
   /*!
    * \fn      vSetWiFiLimitationMode()
    * \brief   This is to enable/disable restrictions on WiFi feature set when a SPI session is active
    * \param   enDevCategory: Device Category
    * \param   bActivate: TRUE - Activate. FALSE -Deactivate
    **************************************************************************/
   t_Void vSetWiFiLimitationMode(tenDeviceCategory enDevCategory,t_Bool bActivate);


   
private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclWBLClientHandler::spi_tclWBLClientHandler(const spi_tclWBLClientHandler &rhs)
   **************************************************************************/
   /*!
   * \fn      spi_tclWBLClientHandler(const spi_tclWBLClientHandler &rhs)
   * \brief   Copy Constructor. Will not be implemented.
   **************************************************************************/
   spi_tclWBLClientHandler(const spi_tclWBLClientHandler &rhs);

   /***************************************************************************
   ** FUNCTION:  spi_tclWBLClientHandler::operator=(const spi_tclWBLClientHandler &rhs)
   **************************************************************************/
   /*!
   * \fn      operator=(const spi_tclWBLClientHandler &rhs)
   * \brief   overloaded operator=. Will not be implemented.
   **************************************************************************/
   spi_tclWBLClientHandler& operator=(const spi_tclWBLClientHandler &rhs);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vExecute
    ***************************************************************************/
   /*!
    * \fn     vExecute()
    * \brief  Executes in a separate thread. This creates a Mainloop which waits
    *         for DBus asynchronous responses/signals
    **************************************************************************/
   t_Void vExecute();

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vDbusNameAppearedCb
    ***************************************************************************/
   /*!
    * \fn     vDbusNameAppearedCb
    * \brief  Invoked when the name being watched is known to have to have a owner.
    * \param  connection: The GDBusConnection the name is being watched on
    * \param  name: The name being watched.
    * \param  name_owner: Unique name of the owner of the name being watched.
    * \param  pUserData: User data passed to g_bus_watch_name().
    **************************************************************************/
   static t_Void vDbusNameAppearedCb(GDBusConnection *connection, const gchar *name,
		                                 const gchar *name_owner, gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vDbusNameVanishedCb
    ***************************************************************************/
   /*!
    * \fn     vDbusNameVanishedCb
    * \brief  Invoked when the name being watched is known not to have to have a owner.
    * \param  connection: The GDBusConnection the name is being watched on or NULL.
    * \param  name: The name being watched.
    * \param  pUserData: User data passed to g_bus_watch_name().
    **************************************************************************/
   static t_Void vDbusNameVanishedCb(GDBusConnection *connection, const gchar *name, gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vConfilctManagementProxyCb
    ***************************************************************************/
   /*!
    * \fn     vConfilctManagementProxyCb
    * \brief  Invoked when org_bosch_wbl_conflictmanagement_proxy_new() operation finishes.
    * \param  poSourceObject: the object the asynchronous operation was started with
    * \param  poResult: a GAsyncResult.
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vConfilctManagementProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vConfilctManagementPropertiesProxyCb
    ***************************************************************************/
   /*!
    * \fn     vConfilctManagementPropertiesProxyCb
    * \brief  Invoked when org_freedesktop_dbus_properties_proxy_new() operation finishes.
    * \param  poSourceObject: the object the asynchronous operation was started with
    * \param  poResult: a GAsyncResult.
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vConfilctManagementPropertiesProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vWifisetupProxyCb
    ***************************************************************************/
   /*!
    * \fn     vWifisetupProxyCb
    * \brief  Invoked when org_bosch_wbl_wifisetup_proxy_new() operation finishes.
    * \param  poSourceObject: the object the asynchronous operation was started with
    * \param  poResult: a GAsyncResult.
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vWifisetupProxyCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);
   
   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vPrepareSetupCb
    ***************************************************************************/
   /*!
    * \fn     vPrepareSetupCb
    * \brief  Invoked when org_bosch_wbl_conflictmanagement_call_prepare_setup() operation finishes.
    * \param  poSourceObject: the object the asynchronous operation was started with
    * \param  poResult: a GAsyncResult.
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vPrepareSetupCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);
   
   
   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vSetRestrictionCb
    ***************************************************************************/
   /*!
    * \fn     vSetRestrictionCb
    * \brief  Invoked when org_bosch_wbl_wifisetup_call_set_restrictions() operation finishes.
    * \param  poSourceObject: the object the asynchronous operation was started with
    * \param  poResult: a GAsyncResult.
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vSetRestrictionCb(GObject *poSourceObject, GAsyncResult *poResult, gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vOnFreeDesktopPropertiesChanged
    ***************************************************************************/
   /*!
    * \fn     vOnFreeDesktopPropertiesChanged
    * \brief  Invoked when PropertiesChanged signal is emitted on DBUS by FreeDesktop interface.
    * \param  object: the object the asynchronous operation was started with.
    * \param  arg_interface_name: name of interface that emitted properties changed.
    * \param  arg_changed_properties: array of changed properties.
    * \param  arg_invalidated_properties: array of invalidated properties
    * \param  pUserData: user data passed to the callback..
    **************************************************************************/
   static t_Void vOnFreeDesktopPropertiesChanged(OrgFreedesktopDBusProperties *object, const gchar *arg_interface_name,
		                              GVariant *arg_changed_properties, const gchar *const *arg_invalidated_properties,
		                              gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vOnConflictManagementPropertiesChanged
    ***************************************************************************/
   /*!
    * \fn     vOnConflictManagementPropertiesChanged
    * \brief  Invoked when PropertiesChanged signal is emitted on DBUS by ConflictManagement interface.
    * \param  object: the object the asynchronous operation was started with.
    * \param  arg_interface_name: name of interface that emitted properties changed.
    * \param  arg_changed_properties: array of changed properties.
    * \param  arg_invalidated_properties: array of invalidated properties
    * \param  pUserData: user data passed to the callback..vOnConflictManagementPropertiesChanged..vOnFreeDesktopPropertiesChanged
    **************************************************************************/
   static t_Void vOnConflictManagementPropertiesChanged(OrgBoschWblConflictmanagement *object, const gchar *arg_interface_name,
   		                              GVariant *arg_changed_properties, const gchar *const *arg_invalidated_properties,
   		                              gpointer pUserData);


   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vOnWiFiSetUpPropertiesChanged
    ***************************************************************************/
   /*!
    * \fn     vOnWiFiSetUpPropertiesChanged
    * \brief  Invoked when PropertiesChanged signal is emitted on DBUS by ConflictManagement interface.
    * \param  object: the object the asynchronous operation was started with.
    * \param  arg_interface_name: name of interface that emitted properties changed.
    * \param  arg_changed_properties: array of changed properties.
    * \param  arg_invalidated_properties: array of invalidated properties
    * \param  pUserData: user data passed to the callback..vOnConflictManagementPropertiesChanged..vOnFreeDesktopPropertiesChanged
    **************************************************************************/
   static t_Void vOnWiFiSetUpPropertiesChanged(OrgBoschWblConflictmanagement *object, const gchar *arg_interface_name,
   		                              GVariant *arg_changed_properties, const gchar *const *arg_invalidated_properties,
   		                              gpointer pUserData);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vParseDbusDict
    ***************************************************************************/
   /*!
    * \fn     vHandlePropertiesChanged
    * \brief  Handles the PropertiesChanged signal.
    * \param  arg_interface_name: name of interface that emitted properties changed.
    * \param  arg_changed_properties: array of changed properties.
    * \param  arg_invalidated_properties: array of invalidated properties
    **************************************************************************/
   t_Void vHandlePropertiesChanged(const gchar *arg_interface_name,GVariant *arg_changed_properties,
		   	   	   	   	   	   	   const gchar *const *arg_invalidated_properties);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vParseDbusDict
    ***************************************************************************/
   /*!
    * \fn     vOnPropertiesChanged
    * \brief  Parses the DBUS dictionary of type a{sv} and collects key-value pairs.
    * \param  poDict: pointer to the dictionary to be parsed.
    * \param  vec: OUT parameter. vec will be populated with key-value pairs
    **************************************************************************/
   t_Void vParseDbusDict(GVariant *poDict, std::map<std::string, void*> &vec);

   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::vStoreAPConfig
    ***************************************************************************/
   /*!
    * \fn     vStoreWiFiConfig
    * \brief  Stores the ActiveWifiSetups property information in member structure
    * \param  vec: vector having the key-value pairs.
    **************************************************************************/
   t_Void vStoreWiFiConfig(std::map<std::string, void*> &vec);
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclWBLClientHandler::vGetActiveWiFiSetUpsCb
    ***************************************************************************/
   /*!
    * \fn     t_Void vGetActiveWiFiSetUpsCb (GObject *object,GAsyncResult *res,gpointer user_data);
    * \brief  Method to get the ActiveWifiSetups property information on startup
    * \param  object: Object path
	* \param  res : Result
	* \param  user_data : User context
    **************************************************************************/ 
   static t_Void vGetActiveWiFiSetUpsCb(GObject *object,GAsyncResult *res,gpointer user_data);
   
 
   /***************************************************************************
    ** FUNCTION:  spi_tclWBLClientHandler::bParseStationDict
    ***************************************************************************/
   /*!
    * \fn     t_Bool bParseStationDict()
    * \brief  Method to detect whether an Associated station information is available.
	*         If it is available parse the station dictionary and provide the STA info.
    * \param  gVariant: [IN] pointer to the dictionary to be parsed.
    * \param  rfrStationInfo: [OUT] Station info
	* \retval t_Bool: TRUE - if a STA information retrived/available in the Dict provided.
    **************************************************************************/ 
   t_Bool bParseStationDict(GVariant *gVariant,trStationInfo& rfrStationInfo);
   

   //! Pointer to MainLoop which waits for the responses, signals from DBus
   GMainLoop *m_poMainLoop;

   t_String m_szObjectPath;

   trWiFiCallbacks m_rWiFiCallbacks;

   //! Pointer to Conflict Management proxy
   OrgBoschWblConflictmanagement *m_poOrgBoschWblConflictmanagementProxy;

   //! Pointer to Conflict Management Properties proxy
   OrgFreedesktopDBusProperties *m_poConflictManagementPropertiesProxy;

   OrgBoschWblWifisetup *m_poWifisetupProxy;
   
   //To maintain the list of connected stations info, till it is posted to SPI_core
   std::vector<trStationInfo> m_vecStationsInfo;
};

#endif // _SPI_TCLWBLCLIENTHANDLER_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
