/**
 * \file sdc_op_conv.h
 *
 * \brief libSDC: Definition of functions etc. of crytographic operations (convenience API)
 *
 * Applications needing to perform wrap/unwrap, encrypt/decrypt or sign/verify
 * in an easy way should include this header.\n
 * The common operation header or generic types like error codes are included internally.
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup operations_encdec
 * \defgroup operations_encdec_conv Convenience API
 * Convenience API
 *
 */
/**
 * \ingroup operations_encdec_conv
 * \defgroup operations_encdec_conv_form Formatted
 * All parameters (data,IV,...) are handled in a formatted memory buffer having
 * an proprietary format.\n
 * An extended version can be found in \ref operations_encdec_adv_ext_form
 */
/**
 * \ingroup operations_encdec_conv
 * \defgroup operations_encdec_conv_norm Non-Formatted
 * All parameters (data,IV,...) are specified by application
 */
/**
 * \ingroup operations_signverify
 * \defgroup operations_signverify_conv Convenience API
 * Convenience API
 *
 */
/**
 * \ingroup operations_signverify_conv
 * \defgroup operations_signverify_conv_form Formatted
 * All parameters (data,IV,...) are handled in a formatted memory buffer having
 * an proprietary format.\n
 * An extended version can be found in \ref operations_signverify_adv_ext_form
 */
/**
 * \ingroup operations_signverify_conv
 * \defgroup operations_signverify_conv_norm Non-Formatted
 * All parameters (data,IV,...) are specified by application
 */
/**
 * \ingroup operations_digest
 * \defgroup operations_digest_conv Convenience API
 * Convenience API
 *
 */
/**
 * \ingroup operations_digest_conv
 * \defgroup operations_digest_conv_norm Non-Formatted
 * All parameters (data,...) are specified by application
 */
/**
 * \ingroup operations_wrapunwrap
 * \defgroup operations_wrapunwrap_conv Convenience API
 * Convenience API
 *
 */
/**
 * \ingroup operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_form Formatted
 * All parameters (data,IV,...) are handled in a formatted memory buffer having
 * an proprietary format.\n
 * An extended version can be found in \ref operations_wrapunwrap_adv_ext_form
 */
/**
 * \ingroup operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_norm Non-Formatted
 * All parameters (data,IV,...) are specified by application
 */
/**
 * \ingroup  operations_wrapunwrap_conv
 * \defgroup operations_wrapunwrap_conv_buffer_handling Buffer handling rules
 *
 * For buffers providing input data only a simple pointer is used.\n
 * In case a buffer can provide output data a pointer to a pointer is used.\n
 * \n
 * In case the value of the pointer to the actual buffer is != NULL the data is provided externally.\n
 * In case this pointer is NULL the data is internally generated. If the buffer is an input/output or an output only buffer
 * the content can be used by application after successful execution of the function.\n
 * In case the buffer is externally provided the buffer length has to be provided explicitly as well.\n
 * Some internal generated buffers require an externally specified length (e.g. IV of \ref sdc_wrap).\n
 * This specification can be done either explicitly or the architecture and type specific default can be referenced.
 * In order to use this default the special values need to be assigned to the length values (i.e. SDC_IV_USE_DEFAULT, SDC_TAG_USE_DEFAULT).\n
 * In case the length and the buffer content can not be specified externally (e.g. output data buffer of \ref sdc_wrap) the pointer to the buffer has to be NULL while the length is ignored.\n
 * \n
 * Internally generated output buffers will return length 0 and pointer NULL in case of an API error.\n
 * In case of API success the application has to free internal generated buffers after usage (e.g. using \ref sdc_plain_buffer_free, \ref sdc_cipher_buffer_free).\n
 *
 * Please note: For externally provided buffers with explicit length 0 the buffer pointer might be NULL.
 */



#ifndef __SDC_LIB_OP_CONV_H_
#define __SDC_LIB_OP_CONV_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>
#include <sdc_session.h>
#include <sdc_op_common.h>

/* Definitions types and defaults */

/* Functions */

/**
 * \ingroup operations_wrapunwrap_conv_norm
 * \brief Perform wrap of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers is NULL/0.\n
 * This is not true for the IV in case application has specified the IV explicitly (*\p iv != NULL).
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in] aad_data  pointer to buffer containing additional authenticated data (externally provided buffer)
 * \param[in] aad_len   length of \p aad_data \n
 *                      Currently not supported (has to be 0)\n
 *                      Reserved for future use
 * \param[in,out] iv    pointer to the IV buffer pointer\n
 *                      IV can be provided externally (*\p iv != NULL) or\n
 *                      internally generated (*\p iv == NULL)\n
 *                      In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len    pointer to the length of the IV\n
 *                      The length has to be specified externally\n
 *                      For the internally generated IV the architecture and \p type specific default can be selected by using
 *                      *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                      \p iv_len must not be NULL
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the wrapped data.\n
 *                      \p out_data must not be NULL\n
 *                      Initially *\p out_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] out_len   pointer to return the length of \p out_data \n
 *                      The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                      \p out_len must not be NULL\n
 * \param[out] tag_data  pointer to return a pointer for the internally allocated buffer storing the tag (mac or signature).\n
 *                      \p tag_data must not be NULL\n
 *                      Initially *\p tag_data has to be NULL\n
 *                      Application has to free this buffer after usage
 * \param[in,out] tag_len   pointer to the length of the tag\n
 *                      The length of the tag needs to be specified externally\n
 *                      In case *\p tag_len == SDC_TAG_USE_DEFAULT the architecture and \p type specific default is used.\n
 *                      In this case *\p tag_len will contain the actual length on success\n
 *                      \p tag_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap(sdc_session_t *session,
                             const sdc_wrap_unwrap_type_t *type,
                             const uint8_t *in_data, const size_t in_len,
                             const uint8_t *aad_data, const size_t aad_len,
                             uint8_t **iv, size_t *iv_len,
                             uint8_t **out_data, size_t *out_len,
                             uint8_t **tag_data, size_t *tag_len);

/**
 * \ingroup operations_wrapunwrap_conv_norm
 * \brief Perform unwrap of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of the output buffer is NULL/0.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the cipher data to be unwrapped (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in] aad_data  pointer to buffer containing additional authenticated data (externally provided buffer)
 * \param[in] aad_len   length of \p aad_data \n
 *                      Currently not supported (has to be 0)\n
 *                      Reserved for future use
 * \param[in] iv        pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len    length of the IV  (external specified length)\n
 * \param[in] tag_data  pointer to the tag (mac or signature) buffer (externally provided buffer)\n
 * \param[in] tag_len   length of the tag  (external specified length)\n
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the unwrapped data.\n
 *                      \p out_data must not be NULL\n
 *                      Initially *\p out_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] out_len   pointer to return the length of \p out_data \n
 *                      The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                      \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data or length pointer is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the tag of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap(sdc_session_t *session,
                               const sdc_wrap_unwrap_type_t *type,
                               const uint8_t *in_data, const size_t in_len,
                               const uint8_t *aad_data, const size_t aad_len,
                               const uint8_t *iv, const size_t iv_len,
                               const uint8_t *tag_data, const size_t tag_len,
                               uint8_t **out_data, size_t *out_len);



/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Perform wrap of a memory buffer and generate a formatted output buffer
 *
 * The used key for wrapping is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and block mode,
 * - automatically generated iv
 * - wrapped data
 * - tag information (mac or signature)
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be wrapped (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted wrapped data.\n
 *                      \p formatted_data must not be NULL\n
 *                      Initially *\p formatted_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                      The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                      \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_wrap_formatted(sdc_session_t *session,
                                       const sdc_wrap_unwrap_type_t *type,
                                       const uint8_t *in_data, const size_t in_len,
                                       uint8_t **formatted_data, size_t *formatted_len);



/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Perform unwrap of a the formatted memory buffer and provide output data
 *
 * The used key for wrapping is the key currently assigned to the session.
 * This key might be set using \ref sdc_unwrap_formatted_autoload_key.\n
 * \n
 * Furthermore it is required by the application to extract the algorithm,
 * blockmode using \ref sdc_unwrap_formatted_extract_type, check it
 * (algorithm and block mode sufficient to fulfill its own security requirements)
 * and provide it as argument.\n
 * \n
 * IV and tag (mac or signature) are extracted internally.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the unwrapped data.\n
 *                      \p out_data must not be NULL\n
 *                      Initially *\p out_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] out_len   pointer to return the length of \p out_data \n
 *                      The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                      \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_IV_INVALID - iv within formatted data is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data within formatted data is invalid
 * \return \ref SDC_AAD_DATA_INVALID - aad data within formatted data is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the tag of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_formatted(sdc_session_t *session,
                                         const sdc_wrap_unwrap_type_t *type,
                                         const uint8_t *formatted_data, const size_t formatted_len,
                                         uint8_t **out_data, size_t *out_len);


/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Try to load the key described in the formatted memory buffer
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_formatted_autoload_key(sdc_session_t *session,
                                                      const uint8_t *formatted_data, const size_t formatted_len);

/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Try to load the key described in the formatted memory buffer (with secret modifier)
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \p secret_mod_data must only provided if a key with secret modifier has been used when generating
 * the formatted data.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[in] secret_mod_data  pointer to the secret modifier (NULL if no modifier)
 * \param[in] secret_mod_len   length of the \p secret_mod_data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return \ref SDC_MODIFIER_INVALID - the secret modifier is invalid for the key or its length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len);


/**
 * \ingroup operations_wrapunwrap_conv_form
 * \brief Extract \ref sdc_wrap_unwrap_type_t from formatted memory buffer
 *
 * Based on the results the application has to check if its own
 * security requirements are fulfilled.\n
 * After using the type for unwrapping it needs to be freed using
 * \ref sdc_wrap_unwrap_type_free.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[out] type            type descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_unwrap_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_wrap_unwrap_type_t **type);



/**
 * \ingroup operations_encdec_conv_norm
 * \brief Perform encrypt of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers is NULL/0.\n
 * This is not true for the IV in case application has specified the IV explicitly (*\p iv != NULL).
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in,out] iv    pointer to the IV buffer pointer\n
 *                      IV can be provided externally (*\p iv != NULL) or\n
 *                      internally generated (*\p iv == NULL)\n
 *                      In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len    pointer to the length of the IV\n
 *                      The length has to be specified externally\n
 *                      For the internally generated IV the architecture and \p type specific default can be selected by using
 *                      *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                      \p iv_len must not be NULL
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the encrypted data.\n
 *                      \p out_data must not be NULL\n
 *                      Initially *\p out_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] out_len   pointer to return the length of \p out_data \n
 *                      The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                      \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt(sdc_session_t *session,
                                const sdc_encrypt_decrypt_type_t *type,
                                const uint8_t *in_data, const size_t in_len,
                                uint8_t **iv, size_t *iv_len,
                                uint8_t **out_data, size_t *out_len);


/**
 * \ingroup operations_encdec_conv_norm
 * \brief Perform decrypt of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of the output buffer is NULL/0.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session  session handle created using ::sdc_open_session
 * \param[in] type     type descriptor
 * \param[in] in_data  pointer to the buffer containing the cipher data to be decrypted (externally provided buffer)
 * \param[in] in_len   length of the \p in_data buffer (external specified length)
 * \param[in] iv       pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len   length of the IV  (external specified length)\n
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the decrypted data.\n
 *                     \p out_data must not be NULL\n
 *                     Initially *\p out_data has to be NULL!\n
 *                     Application has to free this buffer after usage
 * \param[out] out_len   pointer to return the length of \p out_data \n
 *                     The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                     \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt(sdc_session_t *session,
                                const sdc_encrypt_decrypt_type_t *type,
                                const uint8_t *in_data, const size_t in_len,
                                const uint8_t *iv, const size_t iv_len,
                                uint8_t **out_data, size_t *out_len);


/**
 * \ingroup operations_encdec_conv_form
 * \brief Perform encryption of a memory buffer and generate a formatted output buffer
 *
 * The used key for encryption is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and block mode,
 * - automatically generated iv
 * - encrypted data
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be encrypted (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted encrypted data.\n
 *                     \p formatted_data must not be NULL\n
 *                     Initially *\p formatted_data has to be NULL!\n
 *                     Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                     The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                     \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_encrypt_formatted(sdc_session_t *session,
                                          const sdc_encrypt_decrypt_type_t *type,
                                          const uint8_t *in_data, const size_t in_len,
                                          uint8_t **formatted_data, size_t *formatted_len);



/**
 * \ingroup operations_encdec_conv_form
 * \brief Perform decryption of a the formatted memory buffer and provide output data
 *
 * The used key for decryption is the key currently assigned to the session.
 * This key might be set using \ref sdc_decrypt_formatted_autoload_key.\n
 * \n
 * Furthermore it is required by the application to extract the algorithm,
 * blockmode using \ref sdc_decrypt_formatted_extract_type, check it
 * (algorithm and block mode sufficient to fulfill its own security requirements)
 * and provide it as argument.\n
 * \n
 * IV is extracted internally.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session  session handle created using ::sdc_open_session
 * \param[in] type     type descriptor
 * \param[in] formatted_data pointer to the buffer containing the formatted data to be decrypted (externally provided buffer)
 * \param[in] formatted_len  length of the \p formatted_data buffer (external specified length)
 * \param[out] out_data  pointer to return a pointer for the internally allocated buffer storing the decrypted data.\n
 *                     \p out_data must not be NULL\n
 *                     Initially *\p out_data has to be NULL!\n
 *                     Application has to free this buffer after usage
 * \param[out] out_len  pointer to return the length of \p out_data \n
 *                     The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                     \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_IV_INVALID - iv within formatted data is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_formatted(sdc_session_t *session,
                                          const sdc_encrypt_decrypt_type_t *type,
                                          const uint8_t *formatted_data, const size_t formatted_len,
                                          uint8_t **out_data, size_t *out_len);

/**
 * \ingroup operations_encdec_conv_form
 * \brief Try to load the key described in the formatted memory buffer
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_formatted_autoload_key(sdc_session_t *session,
                                                       const uint8_t *formatted_data, const size_t formatted_len);

/**
 * \ingroup operations_encdec_conv_form
 * \brief Try to load the key described in the formatted memory buffer (with secret modifier)
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \p secret_mod_data must only provided if a key with secret modifier has been used when generating
 * the formatted data.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[in] secret_mod_data  pointer to the secret modifier (NULL if no modifier)
 * \param[in] secret_mod_len   length of the \p secret_mod_data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return \ref SDC_MODIFIER_INVALID - the secret modifier is invalid for the key or its length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len);

/**
 * \ingroup operations_encdec_conv_form
 * \brief Extract \ref sdc_encrypt_decrypt_type_t from formatted memory buffer
 *
 * Based on the results the application has to check if its own
 * security requirements are fulfilled.\n
 * After using the type for decryption it needs to be freed using
 * \ref sdc_encrypt_decrypt_type_free.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be decrypted (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[out] type      type descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_decrypt_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_encrypt_decrypt_type_t **type);




/**
 * \ingroup operations_signverify_conv_norm
 * \brief Generate tag (mac or signature) of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers is NULL/0.\n
 * This is not true for the IV in case application has specified the IV explicitly (*\p iv != NULL).
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session  session handle created using ::sdc_open_session
 * \param[in] type     type descriptor
 * \param[in] in_data  pointer to the buffer containing the data to be signed (externally provided buffer)
 * \param[in] in_len   length of the \p in_data buffer (external specified length)
 * \param[in,out] iv   pointer to the IV buffer pointer\n
 *                     IV can be provided externally (*\p iv != NULL) or\n
 *                     internally generated (*\p iv == NULL)\n
 *                     In both cases \p iv must not be NULL\n
 * \param[in,out] iv_len  pointer to the length of the IV\n
 *                     The length has to be specified externally\n
 *                     For the internally generated IV the architecture and \p type specific default can be selected by using
 *                     *\p iv_len == SDC_IV_USE_DEFAULT. In this case *\p iv_len will contain the actual length on success\n
 *                     \p iv_len must not be NULL
 * \param[out] tag_data  pointer to return a pointer for the internally allocated buffer storing the tag (mac or signature).\n
 *                     \p tag_data must not be NULL\n
 *                     Initially *\p tag_data has to be NULL\n
 *                     Application has to free this buffer after usage
 * \param[out] tag_len   pointer to the length of the tag\n
 *                     The length of the tag needs to be specified externally\n
 *                     In case *\p tag_len == SDC_TAG_USE_DEFAULT the architecture and \p type specific default is used.\n
 *                     In this case *\p tag_len will contain the actual length on success\n
 *                     \p tag_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign(sdc_session_t *session,
                             const sdc_sign_verify_type_t *type,
                             const uint8_t *in_data, const size_t in_len,
                             uint8_t **iv, size_t *iv_len,
                             uint8_t **tag_data, size_t *tag_len);


/**
 * \ingroup operations_signverify_conv_norm
 * \brief Perform verification of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the data to be verified (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[in] iv        pointer to the IV buffer (externally provided buffer)
 * \param[in] iv_len    length of the IV  (external specified length)\n
 * \param[in] tag_data  pointer to the tag (mac or signature) buffer (externally provided buffer)\n
 * \param[in] tag_len   length of the tag  (external specified length)\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_IV_INVALID - iv data or length pointer is invalid or invalid length specified
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data or length pointer is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the tag of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify(sdc_session_t *session,
                               const sdc_sign_verify_type_t *type,
                               const uint8_t *in_data, const size_t in_len,
                               const uint8_t *iv, const size_t iv_len,
                               const uint8_t *tag_data, const size_t tag_len);


/**
 * \ingroup operations_signverify_conv_form
 * \brief Perform sign of a memory buffer and generate a formatted output buffer
 *
 * The used key for authentication is the key currently assigned to the session.
 * This formatted output buffer contains\n
 * - key information,
 * - information on algorithm and hash used,
 * - automatically generated iv
 * - plain data (authenticated by tag)
 * - tag information (mac or signature)
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] in_data   pointer to the buffer containing the plain data to be signed and stored in formatted_data (externally provided buffer)
 * \param[in] in_len    length of the \p in_data buffer (external specified length)
 * \param[out] formatted_data  pointer to return a pointer for the internally allocated buffer storing the formatted_data.\n
 *                      \p formatted_data must not be NULL\n
 *                      Initially *\p formatted_data has to be NULL!\n
 *                      Application has to free this buffer after usage
 * \param[out] formatted_len   pointer to return the length of \p formatted_data \n
 *                      The length is determined internally - therefore the value of *\p formatted_len is ignored\n
 *                      \p formatted_len must not be NULL\n
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_sign_formatted(sdc_session_t *session,
                                       const sdc_sign_verify_type_t *type,
                                       const uint8_t *in_data, const size_t in_len,
                                       uint8_t **formatted_data, size_t *formatted_len);



/**
 * \ingroup operations_signverify_conv_form
 * \brief Extract authenticated data of a the formatted memory buffer
 *
 * The used key for authentication is the key currently assigned to the session.
 * This key might be set using \ref sdc_verify_formatted_autoload_key.\n
 * \n
 * Furthermore it is required by the application to extract the algorithm,
 * hash using \ref sdc_verify_formatted_extract_type, check it
 * (mac and hash sufficient to fulfill its own security requirements)
 * and provide it as argument.\n
 * \n
 * IV and tag (mac or signature) are extracted internally.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] type      type descriptor
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be authenticated (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[out] out_data pointer to return a pointer to the authenticated data within the constant formatted data input.\n
 *                      \p out_data must not be NULL\n
 *                      Initially *\p out_data has to be NULL!\n
 *                      Application must not free this buffer after usage - it will be freed when formatted data is freed.
 * \param[out] out_len  pointer to return the length of \p out_data \n
 *                      The length is determined internally - therefore the value of *\p out_len is ignored\n
 *                      \p out_len must not be NULL\n
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_OUT_DATA_INVALID - out data or length pointer is invalid
 * \return \ref SDC_IV_INVALID - iv within formatted data is invalid
 * \return \ref SDC_TAG_DATA_INVALID - tag (mac or signature) data within formatted data is invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to check the tag of the input data
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_formatted(sdc_session_t *session,
                                         const sdc_sign_verify_type_t *type,
                                         const uint8_t *formatted_data, const size_t formatted_len,
                                         const uint8_t **out_data, size_t *out_len);

/**
 * \ingroup operations_signverify_conv_form
 * \brief Try to load the key described in the formatted memory buffer
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_formatted_autoload_key(sdc_session_t *session,
                                                      const uint8_t *formatted_data, const size_t formatted_len);

/**
 * \ingroup operations_signverify_conv_form
 * \brief Try to load the key described in the formatted memory buffer (with secret modifier)
 *
 * Load the key described in the formatted memory buffer.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \p secret_mod_data must only provided if a key with secret modifier has been used when generating
 * the formatted data.
 *
 * \param[in] session   session handle created using ::sdc_open_session
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be unwrapped (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[in] secret_mod_data  pointer to the secret modifier (NULL if no modifier)
 * \param[in] secret_mod_len   length of the \p secret_mod_data
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_NOT_SUPPORTED - Key kind is not supported
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to perform any operation using the given key
 * \return \ref SDC_KEY_NOT_READY - key not ready - try again later
 * \return \ref SDC_KEY_LOCKED - key store currently locked - try again later
 * \return \ref SDC_KEY_TAMPERED - loading the key aborted as it seems to be tampered
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_AUTOLOAD_KEY_UNSUPPORTED - the key type of the formatted data is not supported
 * \return \ref SDC_AUTOLOAD_KEY_WITH_SECRET - the key of the formatted data requires a secret modifier to be specified
 * \return \ref SDC_MODIFIER_INVALID - the secret modifier is invalid for the key or its length is invalid
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_formatted_autoload_key_with_secret_mod(
    sdc_session_t *session,
    const uint8_t *formatted_data, const size_t formatted_len,
    const uint8_t *secret_mod_data, const size_t secret_mod_len);

/**
 * \ingroup operations_signverify_conv_form
 * \brief Extract \ref sdc_sign_verify_type_t from formatted memory buffer
 *
 * Based on the results the application has to check if its own
 * security requirements are fulfilled.\n
 * After using the type for authentication it needs to be freed using
 * \ref sdc_sign_verify_type_free.
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] formatted_data   pointer to the buffer containing the formatted data to be authenticated (externally provided buffer)
 * \param[in] formatted_len    length of the \p formatted_data buffer (external specified length)
 * \param[out] type      type descriptor
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_verify_formatted_extract_type(
    const uint8_t *formatted_data, const size_t formatted_len,
    sdc_sign_verify_type_t **type);


/**
 * \ingroup operations_digest_conv_norm
 * \brief Calculate digest of a memory buffer with a single call from application
 *
 * Internally the request is handled by one or multiple calls to the kernel.\n
 * In case of an error the buffer pointer and the length of output buffers is NULL/0.\n
 *
 * Please refer to \ref operations_wrapunwrap_conv_buffer_handling for general buffer handling descriptions.
 *
 * \param[in] session  session handle created using ::sdc_open_session
 * \param[in] type     type descriptor
 * \param[in] in_data  pointer to the buffer containing the data (externally provided buffer)
 * \param[in] in_len   length of the \p in_data buffer (external specified length)
 * \param[out] digest_data  pointer to return a pointer for the internally allocated buffer storing the digest.\n
 *                     \p digest_data must not be NULL\n
 *                     Initially *\p digest_data has to be NULL\n
 *                     Application has to free this buffer after usage
 * \param[out] digest_len   pointer to the length of the digest\n
 *                     The length of the digest needs to be specified externally\n
 *                     In case *\p digest_len == SDC_DGST_USE_DEFAULT the architecture and \p type specific default is used.\n
 *                     In this case *\p digest_len will contain the actual length on success\n
 *                     \p digest_len must not be NULL
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_ALG_MODE_INVALID - type invalid
 * \return \ref SDC_IN_DATA_INVALID - in data pointer or length is invalid
 * \return \ref SDC_DGST_DATA_INVALID - digest data or length pointer is invalid
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_dgst(sdc_session_t *session,
                             const sdc_dgst_type_t *type,
                             const uint8_t *in_data, const size_t in_len,
                             uint8_t **digest_data, size_t *digest_len);

#ifdef __cplusplus
}
#endif

#endif
