/**
 * \file sdc_keystore_keys.h
 *
 * \brief LIBSDC: Definition of functions etc. required by applications to manage
 * keystorage keys
 *
 * Applications requiring to install/remove key-storage keys need to
 * include this header.\n
 * The functions required to manage permissions are included automatically.\n
 * The same is true for generic types like error codes.
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \ingroup keymanagement
 * \defgroup keymanagement_keystore Key storage
 * Functions to generate, insert, import or remove keys of stored in the
 * key storage (persistent or volatile)
 */

#ifndef __SDC_LIB_KEYSTORE_KEYS_H_
#define __SDC_LIB_KEYSTORE_KEYS_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>
#include <sdc_perm.h>
#include <sdc_session.h>
#include <sdc_op_common.h>

/* Definitions types and defaults */

/**
 * \ingroup keymanagement_keystore
 * \brief Specify if the KID is specified or automatically determined during key creation
 */
typedef enum {
    SDC_CREATE_KEY_OPT_FIRST = 0,
    SDC_CREATE_KEY_AUTOMATIC_ID = SDC_CREATE_KEY_OPT_FIRST,    /**< determine KID automatically */
    SDC_CREATE_KEY_FIXED_ID,         /**< use explicitly specified KID */
    SDC_CREATE_KEY_OPT_LAST = SDC_CREATE_KEY_FIXED_ID
} sdc_key_id_options_t;

/**
 * \ingroup keymanagement_keystore
 * \brief Specify if the created key shall be volatile or persistent
 */
typedef enum {
    SDC_STORAGE_KEY_OPT_FIRST = 0,
    SDC_VOLATILE_STORAGE_KEY = SDC_STORAGE_KEY_OPT_FIRST,   /**< keystorage key will be volatile (lost in next power cycle) */
    SDC_PERSISTENT_STORAGE_KEY,  /**< keystorage key will persist over power cycles */
    SDC_STORAGE_KEY_OPT_LAST = SDC_PERSISTENT_STORAGE_KEY
} sdc_key_storage_options_t;


/* Functions */


/**
 * \ingroup keymanagement_keystore
 * \brief Install a key to the key storage using a user specified plain key secret
 *
 * Note: keystore keys only support uid set to current effective uid
 *
 * If \p kid_opt is \ref SDC_CREATE_KEY_FIXED_ID \p kid is used as input.\n
 * If \p kid_opt is \ref SDC_CREATE_KEY_AUTOMATIC_ID the kernel will search for a free ID and return it using \p kid.
 *
 * \param[in,out]   kid key identifier
 * \param[in]       kid_opt specify if \p kid is used as input or output
 * \param[in]       plain_key buffer containing the key secret of the key
 * \param[in]       plain_key_len length of the plain_key buffer (in bytes)
 * \param[in]       key_stor_opt specify if the key shall be volatile or persistent
 * \param[in]       permissions permissions (incl. group) of the key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given \p kid or \p plain_key is invalid
 * \return \ref SDC_KEYLEN_INVALID - \p plain_key_len is invalid
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_insert_plain_storage_key(
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const uint8_t *plain_key,
    const size_t plain_key_len,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_keystore
 * \brief Install a key to the key storage using internally generated random key secret
 *
 * Note: keystore keys only support uid set to current effective uid
 *
 * If \p kid_opt is \ref SDC_CREATE_KEY_FIXED_ID \p kid is used as input.\n
 * If \p kid_opt is \ref SDC_CREATE_KEY_AUTOMATIC_ID the kernel will search for a free ID and return it using \p kid.
 *
 * \param[in,out]   kid key identifier
 * \param[in]       kid_opt specify if \p kid is used as input or output
 * \param[in]       key_len desired key length (in bytes)
 * \param[in]       key_stor_opt specify if the key shall be volatile or persistent
 * \param[in]       permissions permissions (incl. group) of the key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given \p kid is invalid
 * \return \ref SDC_KEYLEN_INVALID - \p key_len is invalid
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_generate_random_storage_key(
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const size_t key_len,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions);

/**
 * \ingroup keymanagement_keystore
 * \brief Install a key to the key storage using internally generated random key secret
 *
 * Note: keystore keys only support uid set to current effective uid
 *
 * If \p kid_opt is \ref SDC_CREATE_KEY_FIXED_ID \p kid is used as input.\n
 * If \p kid_opt is \ref SDC_CREATE_KEY_AUTOMATIC_ID the kernel will search for a free ID and return it using \p kid.
 *
 * \param[in]       fmt desired key format
 * \param[in]       len desired key length
 * \param[in,out]   kid key identifier
 * \param[in]       kid_opt specify if \p kid is used as input or output
 * \param[in]       key_stor_opt specify if the key shall be volatile or persistent
 * \param[in]       permissions permissions (incl. group) of the key
 * \param[in]       key_secret_enc externally defined encoding of the key_secret (e.g. PEM for RSA keys)
 * \param[in]       key_secret buffer containing the key secret of the key
 * \param[in]       key_secret_len length of the plain_key buffer
 *
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_FMT_INVALID - \p fmt is invalid
 * \return \ref SDC_KEY_ENC_INVALID - \p key_secret_enc is invalid
 * \return \ref SDC_KEYLEN_INVALID - \p len is invalid
 * \return \ref SDC_KEY_INVALID - given \p kid or \p key_secret is invalid
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_insert_storage_key(
    const sdc_key_fmt_t fmt,
    const sdc_key_len_t len,
    sdc_key_id_t *kid,
    const sdc_key_id_options_t kid_opt,
    const sdc_key_storage_options_t key_stor_opt,
    const sdc_permissions_t *permissions,
    sdc_keysecret_enc_t key_secret_enc,
    const uint8_t *key_secret,
    const size_t key_secret_len);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup keymanagement_keystore
 * \brief Import a key to key_storage.
 *
 * In contrast to \ref sdc_insert_storage_key the key needs to be unwrapped using
 * the current key loaded to the session.
 *
 * NOTE: No element of sdc_permissions_t must exceed the upper limit
 * specified by the inheritance mask of the key(currently loaded) used to wrap \p wrapped_key.
 *
 * \param[in] session          session handle created using ::sdc_open_session
 * \param[in] fmt              format of the key
 * \param[in] len              length of the key
 * \param[in,out]   kid        key identifier
 * \param[in] kid_opt          specify if \p kid is used as input or output
 * \param[in] key_stor_opt     specify if the key shall be volatile or persistent
 * \param[in] permissions      permissions (incl. group) of the key
 * \param[in] wrap_type        type descriptor for wrap used to encrypt/authenticate key
 * \param[in] wrapped_key_enc  Encoding of the data inside \p wrapped_key
 * \param[in] wrap_iv          iv used when wrapping \p wrapped_key
 * \param[in] wrap_iv_len      length of \p wrap_iv
 * \param[in] wrap_tag         tag associated with \p wrapped_key
 * \param[in] wrap_tag_len     length of \p wrap_tag
 * \param[in] wrapped_key      buffer containing the wrapped key
 * \param[in] wrapped_key_len  length of \p wrapped_key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_KEY_FMT_INVALID - \p fmt is invalid
 * \return \ref SDC_KEY_ENC_INVALID - \p wrapped_key_enc is invalid
 * \return \ref SDC_KEYLEN_INVALID - \p len is invalid
 * \return \ref SDC_KEY_INVALID - given \p kid is invalid
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return \ref SDC_KEY_UNSET - no key (for unwrap) loaded to session
 * \return \ref SDC_ALG_MODE_INVALID - wrap_type invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to authenticate the wrapped key
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_import_wrapped_storage_key(sdc_session_t *session,
                                                   const sdc_key_fmt_t fmt,
                                                   const sdc_key_len_t len,
                                                   sdc_key_id_t *kid,
                                                   const sdc_key_id_options_t kid_opt,
                                                   const sdc_key_storage_options_t key_stor_opt,
                                                   const sdc_permissions_t *permissions,
                                                   const sdc_wrap_unwrap_type_t *wrap_type,
                                                   sdc_keysecret_enc_t wrapped_key_enc,
                                                   const uint8_t *wrap_iv, const size_t wrap_iv_len,
                                                   const uint8_t *wrap_tag, const size_t wrap_tag_len,
                                                   const uint8_t *wrapped_key, const size_t wrapped_key_len);

/* Note: validity of session needs to be checked internally */
/**
 * \ingroup keymanagement_keystore
 * \brief Import a key from formatted to key_storage.
 *
 * In contrast to \ref sdc_import_wrapped_storage_key wrap type, iv and key data are read
 * from formatted memory block.
 *
 * Please call \ref sdc_import_formatted_autoload_key_with_secret_mod first.
 * This function will load the required key to the session.
 * Note: Manually loading the key is required in case a session key is used
 * to protect the key inside the formatted data.
 *
 * In future this function will support two different types of formatted data
 *  - formatted wrapped data : like \ref sdc_wrap_formatted, sdc_unwrap_formatted
 *  - special formatted key data
 * In the first case \p fmt/\p len/\p key_enc need to be explicitly specified by application
 * In the latter case it might be retrieved using a special function
 *
 * NOTE: No element of sdc_permissions_t must exceed the upper limit
 * specified by the inheritance mask of the key used to generate the formatted data
 *
 * \param[in] session          session handle created using ::sdc_open_session
 * \param[in] fmt              format of the key
 * \param[in] len              length of the key
 * \param[in] key_enc          encoding used for the key inside formatted memory buffer
 * \param[in,out]   kid        key identifier
 * \param[in] kid_opt          specify if \p kid is used as input or output
 * \param[in] key_stor_opt     specify if the key shall be volatile or persistent
 * \param[in] permissions      permissions (incl. group) of the key
 * \param[in] formatted_key     buffer containing the wrapped key
 * \param[in] formatted_key_len length of \p wrapped_key
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_SESSION_INVALID - given session is invalid
 * \return \ref SDC_FORMATTED_DATA_INVALID - formatted data or length pointer is invalid
 * \return \ref SDC_KEY_FMT_INVALID - \p fmt is invalid
 * \return \ref SDC_KEYLEN_INVALID - \p len is invalid
 * \return \ref SDC_KEY_INVALID - given \p kid is invalid
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_PERM_INVALID - given \p permissions are invalid
 * \return \ref SDC_INVALID_PARAMETER - any other parameter is invalid
 * \return \ref SDC_KID_EXISTS - the fixed (\ref SDC_CREATE_KEY_FIXED_ID) KID is already used
 * \return \ref SDC_KID_NOT_AVAILABLE - failed to find any free KID (\ref SDC_CREATE_KEY_AUTOMATIC_ID)
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return \ref SDC_KEY_UNSET - no key (for unwrap) loaded to session
 * \return \ref SDC_ALG_MODE_INVALID - wrap_type invalid
 * \return \ref SDC_AUTHENTICATION_FAILED - failed to authenticate the wrapped key
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_import_formatted_storage_key(sdc_session_t *session,
                                                     const sdc_key_fmt_t fmt,
                                                     const sdc_key_len_t len,
                                                     const sdc_keysecret_enc_t key_enc,
                                                     sdc_key_id_t *kid,
                                                     const sdc_key_id_options_t kid_opt,
                                                     const sdc_key_storage_options_t key_stor_opt,
                                                     const sdc_permissions_t *permissions,
                                                     const uint8_t *formatted_key, const size_t formatted_key_len);

/**
 * \ingroup keymanagement_keystore
 * \brief Remove key with given ID from the key storage
 *
 * \param[in] kid        ID of key to remove
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_KEY_INVALID - given KID is invalid
 * \return \ref SDC_KID_NOT_AVAILABLE - the given KID is valid but not available in the key storage
 * \return \ref SDC_KEY_NOT_READY - in case the keystore is not ready to handle install/remove
 * \return \ref SDC_KEY_LOCKED - the key can't be removed as it is currently assigned to one or multiple sessions
 * \return \ref SDC_ACCESS_DENIED - the process has no permission to delete the key
 * \return \ref SDC_NO_MEM - failed to allocate memory
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to fetch SDC daemon socket config
 * \return \ref SDC_CONFIG_MISSING - SDC daemon socket missing in config
 * \return \ref SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
 * \return \ref SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time
 * \return \ref SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access key storage
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_remove_storage_key(const sdc_key_id_t kid);

#ifdef __cplusplus
}
#endif

#endif
