/**
 * \file sdc.h
 *
 * \brief libSDC: Definition of generic functions and types (e.g. error codes) - <b>should not be included directly!</b>
 *
 * Application should not include this header directly!
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \defgroup libSDC_API libSDC API
 */

/**
 * \ingroup libSDC_API
 * \defgroup generictype Generic types / enums
 */

/**
 * \ingroup libSDC_API
 * \defgroup genericfunc Generic functions
 * Functions to resolve error codes, free memory etc.
 */

/**
 * \ingroup libSDC_API
 * \defgroup keymanagement Key management
 * Functions to generate, insert, import or remove keys
 */

#ifndef __SDC_LIB_H_
#define __SDC_LIB_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <sys/types.h>
#include <stdbool.h>
#include <stdint.h>

/* Definitions types and defaults */

/**
 * \ingroup libSDC
 * \brief Used to specify that this function is part of the "normal" API of libSDC
 *
 * These function will be visible to applications including libSDC library */
#define SDC_API __attribute__((__visibility__("default")))

/**
 * \ingroup libSDC
 * \brief Used to specify that this function is part of the architecture specific API of libSDC
 *
 * These function won't be visible to applications including libSDC library */
#define SDC_ARCH_API __attribute__((__visibility__("internal")))

/**
 * \ingroup libSDC
 * \brief Used to specify that this function is part of the architecture specific API of libSDC
 * but needs to be available out of libSDC (e.g. for khd)
 *
 * These function won't be visible to applications including libSDC library */
#define SDC_ARCH_EXTERN_API __attribute__((__visibility__("default")))

/**
 * \ingroup keymanagement_keystore
 * \brief KID used to flag an invalid key storage key
 */
#define SDC_FLAG_INVALID_KID             0


/**
 * \ingroup keymanagement
 * \brief special UID to mark an invalid user
 * For keys owned by this user permissions of the owner are not relevant and should be cleared.
 */
#define SDC_FLAG_INVALID_UID ((uid_t)-1)

/**
 * \ingroup keymanagement
 * \brief special GID to mark an invalid group
 * For keys owned by this group permissions of the owner are not relevant and should be cleared.
 */
#define SDC_FLAG_INVALID_GID ((gid_t)-1)

/**
 * \ingroup keymanagement_keystore
 * \brief Last possible KID
 */
#define SDC_LAST_KID UINT32_MAX

/**
 * \ingroup keymanagement_keystore
 * \brief Start of a automatic key range
 *
 * When a keystore key is installed without explicit key ID
 * the ID will be automatically determined in the range \ref SDC_AUTOMATIC_KID_RANGE_MIN to \ref SDC_AUTOMATIC_KID_RANGE_MAX
 */
#define SDC_AUTOMATIC_KID_RANGE_MIN 20001
/**
 * \ingroup keymanagement_keystore
 * \brief End of a automatic key range
 */
#define SDC_AUTOMATIC_KID_RANGE_MAX SDC_LAST_KID

/**
 * \ingroup keymanagement_keystore
 * \brief Type for key identifier
 */
typedef uint32_t sdc_key_id_t;

/**
 * \ingroup generictype
 * \brief libSDC error codes
 */
typedef enum {
    SDC_OK = 0,                     /**< SDC_OK - no error */

    SDC_NO_MEM,                     /**< SDC_NO_MEM - no memory available */

    SDC_INVALID_VERSION,            /**< SDC_INVALID_VERSION - Mismatch of kernel and user-space version detected
                                         only used by version check functions like sdc_kernel_version_verify which is not
                                         available for normal applications using libSDC */

    SDC_CONFIG_MISSING,             /**< SDC_CONFIG_MISSING - no such SDC config entry */

    /* error codes will be returned if a libkeyman or libsdc library function is called by external application with invalid parameters */
    SDC_INVALID_PARAMETER,          /**< SDC_INVALID_PARAMETER - unspecified parameters is invalid */
    SDC_KEY_INVALID,                /**< SDC_KEY_INVALID - the given key is invalid */
    SDC_ALG_MODE_INVALID,           /**< SDC_ALG_MODE_INVALID - the given algorithm or mode is invalid */
    SDC_IV_INVALID,                 /**< SDC_IV_INVALID - the given IV is invalid */
    SDC_IN_DATA_INVALID,            /**< SDC_IN_DATA_INVALID - the given in data is invalid */
    SDC_OUT_DATA_INVALID,           /**< SDC_OUT_DATA_INVALID - the given out data is invalid */
    SDC_AAD_DATA_INVALID,           /**< SDC_AAD_DATA_INVALID - the given additional authenticated data (aad) is invalid */
    SDC_TAG_DATA_INVALID,           /**< SDC_TAG_DATA_INVALID - the given tag (mac or signature) is invalid */
    SDC_SESSION_INVALID,            /**< SDC_SESSION_INVALID - the given session is invalid */
    SDC_PERM_INVALID,               /**< SDC_PERM_INVALID - the given permissions are invalid */
    SDC_FORMATTED_DATA_INVALID,     /**< SDC_FORMATTED_DATA_INVALID - an issue with the given formatted data has been detected */
    SDC_KEYLEN_INVALID,             /**< SDC_KEYLEN_INVALID - the given key-length is invalid */

    SDC_DAEMON_COMMUNICATION_ERROR, /**< SDC_DAEMON_COMMUNICATION_ERROR - communication to SDC daemon failed
                                         most likely some permissions are missing or the daemon has not been started */
    SDC_DAEMON_REQUEST_REJECTED,    /**< SDC_DAEMON_REQUEST_REJECTED - the request has been rejected by the daemon */
    SDC_DAEMON_INVALID_RESPONSE,    /**< SDC_DAEMON_INVALID_RESPONSE - checking response of daemon failed */
    SDC_DAEMON_TIMEOUT_ERROR,       /**< SDC_DAEMON_TIMEOUT_ERROR - response of daemon not received in time */
    SDC_DAEMON_SOCKET_CREATION_FAILED,  /**< SDC_DAEMON_SOCKET_CREATION_FAILED - Daemon failed to create socket */

    SDC_KEY_STORAGE_ALREADY_PRESENT, /**< SDC_KEY_STORAGE_ALREADY_PRESENT - Key-storage does already exist */
    SDC_KEY_STORAGE_ACCESS_FAILED,  /**< SDC_KEY_STORAGE_ACCESS_FAILED - daemon failed to access the key storage */
    SDC_KEY_STORAGE_CORRUPT,        /**< SDC_KEY_STORAGE_CORRUPT - daemon failed to load corrupted key storage */
    SDC_KEY_CONTAINER_IMPORT_FAILED, /**< SDC_KEY_CONTAINER_IMPORT_FAILED - daemon failed to import a key container */

    SDC_KID_NOT_AVAILABLE,          /**< SDC_KID_NOT_AVAILABLE - the KID does not exist (e.g. for load or remove) */
    SDC_KID_EXISTS,                 /**< SDC_KID_EXISTS - could not install key as the KID is already used */

    SDC_KEY_UNSET,                  /**< SDC_KEY_UNSET - operation fails as no key has been assigned to a session */
    SDC_KEY_LOCKED,                 /**< SDC_KEY_LOCKED - operation aborted as key or key storage is currently locked - try again later */
    SDC_KEY_NOT_READY,              /**< SDC_KEY_NOT_READY - operation aborted as key is not ready to be used - try again later */
    SDC_KEY_TAMPERED,               /**< SDC_KEY_TAMPERED - operation aborted as the key seems to be tampered */

    SDC_ACCESS_DENIED,              /**< SDC_ACCESS_DENIED - the process has not sufficient permissions to perform the requested operation */
    SDC_OP_NOT_SUPPORTED,           /**< SDC_OP_NOT_SUPPORTED - Operation not supported
                                         e.g. the key does not fit to the operation e.g. DES and 256bit key,
                                         e.g. init/update/finalize API not allowed for this particular algorithm/mode */
    SDC_AUTHENTICATION_FAILED,      /**< SDC_AUTHENTICATION_FAILED - authentication of data failed */

    SDC_PADDING_ERROR,              /**< SDC_PADDING_ERROR - invalid padding detected when removing padding */
    SDC_INTERNAL_ERROR,             /**< SDC_INTERNAL_ERROR - these errors should not occur if all API layers are implemented correctly */
    SDC_NOT_SUPPORTED,              /**< SDC_NOT_SUPPORTED - the function called is currently not supported for the current architecture (i.e. not implemented at all) */
    SDC_UNKNOWN_ERROR,              /**< SDC_UNKNOWN_ERROR - ? */
    SDC_CONFIG_ACCESS_FAILED,       /**< SDC_CONFIG_ACCESS_FAILED - failed to read SDC config
                                         most likely the credentials of the libSDC config */
    SDC_CONFIG_INVALID,             /**< SDC_CONFIG_INVALID - entry in SDC config is invalid */
    SDC_DAEMON_MKDIR_FAILED,        /**< SDC_DAEMON_MKDIR_FAILED - SDC daemon failed to create missing directory */

    SDC_KEY_INFO_INVALID,           /**< SDC_KEY_INFO_INVALID - failed to fill key info */
    SDC_AUTOLOAD_KEY_UNSUPPORTED,   /**< SDC_AUTOLOAD_KEY_UNSUPPORTED - The key type of the formatted data is not supported */
    SDC_AUTOLOAD_KEY_WITH_SECRET,   /**< SDC_AUTOLOAD_KEY_WITH_SECRET - The key of the formatted data requires a secret modifier to be specified */
    SDC_MODIFIER_INVALID,           /**< SDC_MODIFIER_INVALID - the modifier is invalid for the key or its length is invalid */
    SDC_DGST_DATA_INVALID,          /**< SDC_DGST_DATA_INVALID - the given digest is invalid */
    SDC_KEY_FMT_INVALID,            /**< SDC_KEY_FMT_INVALID - the given key-format is invalid */
    SDC_KEY_ENC_INVALID,            /**< SDC_KEY_ENC_INVALID - the encoding of the key secret is invalid */

    SDC_OP_SEQUENCE_INVALID,        /**< SDC_OP_SEQUENCE_INVALID - the operation sequence is invalid */

    SDC_ARCHITECTURE_SPECIFIC_ERROR_01, /**< SDC_ARCHITECTURE_SPECIFIC_ERROR_XX - architecture specific error */
    SDC_ARCHITECTURE_SPECIFIC_ERROR_02,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_03,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_04,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_05,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_06,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_07,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_08,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_09,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_10,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_11,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_12,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_13,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_14,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_15,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_16,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_17,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_18,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_19,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_20,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_21,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_22,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_23,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_24,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_25,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_26,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_27,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_28,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_29,
    SDC_ARCHITECTURE_SPECIFIC_ERROR_30
} sdc_error_t;

/**
 * \ingroup generictype
 * \brief First value of \ref sdc_error_t
 */
#define SDC_ERROR_FIRST SDC_OK

/**
 * \ingroup generictype
 * \brief Last value of \ref sdc_error_t
 */
#define SDC_ERROR_LAST  SDC_ARCHITECTURE_SPECIFIC_ERROR_30

/**
 * \ingroup generictype
 * \brief Deprecated : SDC_SIG_DATA_INVALID is now named SDC_TAG_DATA_INVALID
 */
#define SDC_SIG_DATA_INVALID    SDC_TAG_DATA_INVALID


/**
 * \ingroup generictype
 * \brief libSDC supported key formats
 */
enum sdc_key_fmt {
    SDC_KEY_FMT_UNKNOWN        = 0,                 /**< Invalid key format */
    SDC_KEY_FMT_FIRST          = 1,                 /**< helper to iterate over key formats */
    SDC_KEY_FMT_SIMPLE_SYM     = SDC_KEY_FMT_FIRST, /**< Symmetric key using plain key secret - no encoding, no parity */
    SDC_KEY_FMT_RSA_PUBLIC,                         /**< Asymmetric RSA public key */
    SDC_KEY_FMT_RSA_PRIVATE,                        /**< Asymmetric RSA private key */
    SDC_KEY_FMT_END
};
/**
 * \ingroup generictype
 * \brief typedef for enum \ref sdc_key_fmt
 */
typedef enum sdc_key_fmt sdc_key_fmt_t;

/**
 * \ingroup generictype
 * \brief libSDC supported key secret encodings
 */
enum sdc_keysecret_enc {
    SDC_KEY_ENC_UNKNOWN        = 0,                 /**< Invalid key secret encodings */
    SDC_KEY_ENC_FIRST          = 1,                 /**< helper to iterate over key formats */
    SDC_KEY_ENC_PLAIN          = SDC_KEY_ENC_FIRST, /**< Symmetric keys - sequence of bytes */
    SDC_KEY_ENC_DER,                                /**< Asymmetric keys - DER encoding */
    SDC_KEY_ENC_PEM,                                /**< Asymmetric keys - PEM encoding */
    SDC_KEY_ENC_END
};
/**
 * \ingroup generictype
 * \brief typedef for enum \ref sdc_keysecret_enc
 */
typedef enum sdc_keysecret_enc sdc_keysecret_enc_t;

/**
 * \ingroup generictype
 * \brief Mapping of \ref sdc_key_fmt_t to individual bits of \ref sdc_key_fmt_bmsk_t
 */
#define SDC_KEY_FMT_TO_BMSK(X)              (1<<((uint32_t)(X-SDC_KEY_FMT_FIRST)))
/**
 * \ingroup generictype
 * \brief \ref sdc_key_fmt_bmsk_t value of \ref SDC_KEY_FMT_SIMPLE_SYM
 */
#define SDC_KEY_FMT_BMSK_SIMPLE_SYM         SDC_KEY_FMT_TO_BMSK(SDC_KEY_FMT_SIMPLE_SYM)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_fmt_bmsk_t value of \ref SDC_KEY_FMT_RSA_PUBLIC
 */
#define SDC_KEY_FMT_BMSK_RSA_PUBLIC         SDC_KEY_FMT_TO_BMSK(SDC_KEY_FMT_RSA_PUBLIC)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_fmt_bmsk_t value of \ref SDC_KEY_FMT_RSA_PRIVATE
 */
#define SDC_KEY_FMT_BMSK_RSA_PRIVATE        SDC_KEY_FMT_TO_BMSK(SDC_KEY_FMT_RSA_PRIVATE)

/**
 * \ingroup generictype
 * \brief libSDC bitmask to return the supported key formats
 *
 * Depending on the supported formats one or multiple bits are set.
 * Key format values according to \ref sdc_key_fmt_t
 */
typedef unsigned int sdc_key_fmt_bmsk_t;

/**
 * \ingroup generictype
 * \brief define for all libSDC supported key formats
 */
#define SDC_KEY_FMT_BMSK_ALL (SDC_KEY_FMT_BMSK_SIMPLE_SYM| \
                              SDC_KEY_FMT_BMSK_RSA_PUBLIC| \
                              SDC_KEY_FMT_BMSK_RSA_PRIVATE)
/**
 * \ingroup generictype
 * \brief libSDC supported key lengths
 */
enum sdc_key_len {
    SDC_KEY_LEN_UNKNOWN= 0,         /**< Invalid key length */
    SDC_KEY_LEN_FIRST   = 1,        /**< helper to iterate over keylength */
    SDC_KEY_LEN_56bit   = SDC_KEY_LEN_FIRST,    /**< Key length 56bit = 7bytes */
    SDC_KEY_LEN_64bit,                          /**< Key length 64bit = 8bytes */
    SDC_KEY_LEN_80bit,                          /**< Key length 80bit = 10bytes */
    SDC_KEY_LEN_96bit,                          /**< Key length 96bit = 12bytes */
    SDC_KEY_LEN_112bit,                         /**< Key length 112bit = 14bytes */
    SDC_KEY_LEN_128bit,                         /**< Key length 128bit = 16bytes */
    SDC_KEY_LEN_192bit,                         /**< Key length 192bit = 24bytes */
    SDC_KEY_LEN_256bit,                         /**< Key length 256bit = 32bytes */
    SDC_KEY_LEN_384bit,                         /**< Key length 384bit = 48bytes */
    SDC_KEY_LEN_512bit,                         /**< Key length 512bit = 64bytes */
    SDC_KEY_LEN_1024bit,                        /**< Key length 1024bit = 128bytes */
    SDC_KEY_LEN_2048bit,                        /**< Key length 2048bit = 256bytes */
    SDC_KEY_LEN_4096bit,                        /**< Key length 4096bit = 512bytes */
    SDC_KEY_LEN_END
};
/**
 * \ingroup generictype
 * \brief typedef for enum \ref sdc_key_len
 */
typedef enum sdc_key_len sdc_key_len_t;

/**
 * \ingroup generictype
 * \brief Mapping of \ref sdc_key_len_t to individual bits of \ref sdc_key_len_bmsk_t
 */
#define SDC_KEY_LEN_TO_BMSK(X)     (1<<((uint32_t)(X-SDC_KEY_LEN_FIRST)))
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_56bit
 */
#define SDC_KEY_LEN_BMSK_56bit     SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_56bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_64bit
 */
#define SDC_KEY_LEN_BMSK_64bit     SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_64bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_80bit
 */
#define SDC_KEY_LEN_BMSK_80bit     SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_80bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_96bit
 */
#define SDC_KEY_LEN_BMSK_96bit     SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_96bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_112bit
 */
#define SDC_KEY_LEN_BMSK_112bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_112bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_128bit
 */
#define SDC_KEY_LEN_BMSK_128bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_128bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_192bit
 */
#define SDC_KEY_LEN_BMSK_192bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_192bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_256bit
 */
#define SDC_KEY_LEN_BMSK_256bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_256bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_384bit
 */
#define SDC_KEY_LEN_BMSK_384bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_384bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_512bit
 */
#define SDC_KEY_LEN_BMSK_512bit    SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_512bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_1024bit
 */
#define SDC_KEY_LEN_BMSK_1024bit   SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_1024bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_2048bit
 */
#define SDC_KEY_LEN_BMSK_2048bit   SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_2048bit)
/**
 * \ingroup generictype
 * \brief \ref sdc_key_len_bmsk_t value of \ref SDC_KEY_LEN_4096bit
 */
#define SDC_KEY_LEN_BMSK_4096bit   SDC_KEY_LEN_TO_BMSK(SDC_KEY_LEN_4096bit)

/**
 * \ingroup generictype
 * \brief define to mark no key length supported
 */
#define SDC_KEY_LEN_BMSK_NONE 0

/**
 * \ingroup generictype
 * \brief define for all libSDC supported key lengths
 */
#define SDC_KEY_LEN_BMSK_ALL ( \
        SDC_KEY_LEN_BMSK_64bit|SDC_KEY_LEN_BMSK_80bit|SDC_KEY_LEN_BMSK_96bit| \
        SDC_KEY_LEN_BMSK_112bit|SDC_KEY_LEN_BMSK_128bit|SDC_KEY_LEN_BMSK_192bit| \
        SDC_KEY_LEN_BMSK_256bit|SDC_KEY_LEN_BMSK_384bit| \
        SDC_KEY_LEN_BMSK_512bit|SDC_KEY_LEN_BMSK_1024bit| \
        SDC_KEY_LEN_BMSK_2048bit|SDC_KEY_LEN_BMSK_4096bit)

/**
 * \ingroup generictype
 * \brief libSDC bitmask to return the supported key lengths
 *
 * Depending on the supported key lengths one or multiple bits are set.
 * Key length values according to \ref sdc_key_len_t
 */
typedef unsigned int sdc_key_len_bmsk_t;



/**
 * \ingroup sessionmanagement
 * \brief Session handle
 *
 * In order to start a new SDC session a new session handle needs to be created
 * using ::sdc_open_session.
 * Afterwards the session handle needs to be passed to any session related
 * operation (e.g. ::sdc_session_load_storage_key).
 * After usage it needs to be freed using ::sdc_close_session.
 *
 * For each session it is possible to execute a single cryptographic
 * operation (wrap, unwrap, encrypt, decrypt, sign, verify or
 * random number generation) at once.
 * Starting a second operation while the first is still ongoing results
 * in an error.
 *
 */
typedef struct sdc_session sdc_session_t;

/* Functions */

/**
 * \ingroup genericfunc
 * \brief Get a string message describing the given error code.
 *
 * \param[in] err \ref sdc_error_t error code
 * \return error message
 */
SDC_API const char* sdc_get_error_string (sdc_error_t err);


/**
 * \ingroup genericfunc
 * \brief Overwrite the buffer containing secret data with 0
 *
 *  The asm volatile with memory in the clobber list is intended
 *  to ensure that the compiler does not reorder instructions
 *  and removes memset in optimization.
 *
 * \param[in] secret    Buffer containing potentially secret data
 * \param[in] secretlen Length of the buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - data NULL or len 0
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_overwrite_secret(uint8_t *secret, size_t secretlen);

/**
 * \ingroup genericfunc
 * \brief Free a plain buffer which might contain confidential data.
 *
 * In contrast to ::sdc_cipher_buffer_free this function will first
 * write all 0 to the buffer using same functionality as \ref sdc_overwrite_secret.
 *
 * \param[in] plain_data    pointer to buffer to be freed
 * \param[in] plain_len     length of the buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - data NULL or len 0
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_plain_buffer_free (uint8_t *plain_data, const size_t plain_len);

/**
 * \ingroup genericfunc
 * \brief Free a buffer which contains encrypted or wrapped data.
 *
 * As it is uncritical to leave encrypted or wrapped data in memory this
 * function will only call free.
 *
 * \param[in] cipher_data    pointer to buffer to be freed
 * \param[in] cipher_len     length of the buffer
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - data NULL or len 0
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_cipher_buffer_free (uint8_t *cipher_data, const size_t cipher_len);

/**
 * \ingroup genericfunc
 * \brief Determine the bits of the given key-length value
 *
 * \param[in]  keylen_val    \ref sdc_key_len_t value
 * \param[out] keylen_bits   pointer to return the bits
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_bits
 * \return \ref SDC_KEYLEN_INVALID - the \p keylen_val value is not a element of \ref sdc_key_len_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_get_bits(sdc_key_len_t keylen_val, size_t *keylen_bits);

/**
 * \ingroup genericfunc
 * \brief Determine the bytes of the given key-length value
 *
 * \param[in]  keylen_val    \ref sdc_key_len_t value
 * \param[out] keylen_bytes  pointer to return the bytes
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_bytes
 * \return \ref SDC_KEYLEN_INVALID - the \p keylen_val value is not a element of \ref sdc_key_len_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_get_bytes(sdc_key_len_t keylen_val, size_t *keylen_bytes);

/**
 * \ingroup genericfunc
 * \brief Determine the \ref sdc_key_len_t for given bits
 *
 * \param[in]	bits			length in bits
 * \param[out]  keylen_val		\ref sdc_key_len_t value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return \ref SDC_KEYLEN_INVALID - \p bits does not match to any member of \ref sdc_key_len_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_from_bits(size_t bits, sdc_key_len_t *keylen_val);

/**
 * \ingroup genericfunc
 * \brief Determine the \ref sdc_key_len_t for given bytes
 *
 * \param[in]	bytes			length in bytes
 * \param[out]  keylen_val		\ref sdc_key_len_t value
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return \ref SDC_KEYLEN_INVALID - \p bytes does not match to any member of \ref sdc_key_len_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_from_bytes(size_t bytes, sdc_key_len_t *keylen_val);

/**
 * \ingroup genericfunc
 * \brief Using \ref sdc_key_len_bmsk_first and \ref sdc_key_len_bmsk_next it is
 * possible to iterate over the key length given in \p keylen_bmsk one by one
 *
 * This function is just a shortcut to initialize \p keylen_val to \ref SDC_KEY_LEN_UNKNOWN.
 * Afterwards it will internally call \ref sdc_key_len_bmsk_next.
 * Please refer to \ref sdc_key_len_bmsk_next for more details.
 *
 * \param[in]  keylen_bmsk  \ref sdc_key_len_bmsk_t value with all bits set
 * \param[out] keylen_val   \ref sdc_key_len_t to return the first key-length
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_bmsk_first(sdc_key_len_bmsk_t keylen_bmsk, sdc_key_len_t *keylen_val);

/**
 * \ingroup genericfunc
 * \brief Using \ref sdc_key_len_bmsk_first and \ref sdc_key_len_bmsk_next it is
 * possible to iterate over the key length given in \p keylen_bmsk one by one
 *
 * While iterating \p keylen_val has to be the value returned in the previous call.
 * For initialization either use \ref sdc_key_len_bmsk_first or set
 * \p keylen_val to \ref SDC_KEY_LEN_UNKNOWN.
 *
 * In case no other key-length is found \p keylen_val will be set to \ref SDC_KEY_LEN_UNKNOWN.
 * Note : SDC_OK is returned in this case. Calling sdc_key_len_bmsk_next again, will result
 * in a loop.
 *
 * \param[in] keylen_bmsk   \ref sdc_key_len_bmsk_t value with all bits set
 * \param[in,out] keylen_val   \ref sdc_key_len_t to return the next key-length - use the last result when calling
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p keylen_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_bmsk_next(sdc_key_len_bmsk_t keylen_bmsk, sdc_key_len_t *keylen_val);

/**
 * \ingroup genericfunc
 * \brief Get name of the key format

 * \param[in] key_fmt    \ref sdc_key_fmt_t value
 *
 * \return name or NULL if invalid key format
 */
SDC_API const char *sdc_key_fmt_get_name(sdc_key_fmt_t key_fmt);

/**
 * \ingroup genericfunc
 * \brief Using \ref sdc_key_fmt_bmsk_first and \ref sdc_key_fmt_bmsk_next it is
 * possible to iterate over the key formats given in \p key_fmt_bmsk one by one
 *
 * This function is just a shortcut to initialize \p key_fmt_val to SDC_KEY_FMT_UNKNOWN.
 * Afterwards it will internally call \ref sdc_key_fmt_bmsk_next.
 * Please refer to \ref sdc_key_fmt_bmsk_next for more details.
 *
 * \param[in]  key_fmt_bmsk  \ref sdc_key_fmt_bmsk_t value with all bits set
 * \param[out] key_fmt_val   \ref sdc_key_fmt_t to return the first key format
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p key_fmt_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_fmt_bmsk_first(sdc_key_fmt_bmsk_t key_fmt_bmsk, sdc_key_fmt_t *key_fmt_val);

/**
 * \ingroup genericfunc
 * \brief Using \ref sdc_key_fmt_bmsk_first and \ref sdc_key_fmt_bmsk_next it is
 * possible to iterate over the key formats given in \p key_fmt_bmsk one by one
 *
 * While iterating \p key_fmt_val has to be the value returned in the previous call.
 * For initialization either use \ref sdc_key_fmt_bmsk_first or set
 * \p key_fmt_val to \ref SDC_KEY_FMT_UNKNOWN.
 *
 * In case no other key format is found \p key_fmt_val will be set to \ref SDC_KEY_FMT_UNKNOWN.
 * Note : SDC_OK is returned in this case. Calling sdc_key_fmt_bmsk_next again, will result
 * in a loop.
 *
 * \param[in] key_fmt_bmsk   \ref sdc_key_fmt_bmsk_t value with all bits set
 * \param[in,out] key_fmt_val   \ref sdc_key_fmt_t to return the next key format - use the last result when calling
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p key_fmt_val
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_fmt_bmsk_next(sdc_key_fmt_bmsk_t key_fmt_bmsk, sdc_key_fmt_t *key_fmt_val);

/**
 * \ingroup genericfunc
 * \brief Function to return the key lengths supported by the architecture for given key format
 *
 * Please note: That not all algorithms and block modes need to support these values
 * for processing. But the system has to at least support key management for these
 * key lengths.
 *
 * \param[in] key_fmt   \ref sdc_key_fmt_t key format
 * \param[out] arch_keylen_bmsk	\ref sdc_key_len_bmsk_t pointer to return the supported key lengths for \p key_fmt
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - NULL pointer for \p arch_keylen_bmsk
 * \return \ref SDC_KEY_FMT_INVALID - \p key_fmt is not a valid member of \ref sdc_key_fmt_t
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_API sdc_error_t sdc_key_len_bmsks(sdc_key_fmt_t key_fmt, sdc_key_len_bmsk_t *arch_keylen_bmsk);


#ifdef __cplusplus
}
#endif

#endif
