// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H

#include "common.h"
#include "IControllerCallbacks.h"
#include "ProtocolEndpointBase.h"

/**
 * @internal
 * This class manages the internal GAL state machine, authentication and common messaging. You
 * should not need to directly interact with this class.
 */
class Controller : public ProtocolEndpointBase {
public:
    Controller(MessageRouter* router) :
            ProtocolEndpointBase(CONTROLLER_SERVICE_ID, router, false),
            mHealthy(false),
            mDriverPosition(DRIVER_POSITION_UNKNOWN),
            mSessionConfiguration(0) { }
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void start();
    void shutdown() {
        mSslWrapper.shutdown();
    }
    /**
     * @internal
     * Sets information that describes this car. Should be called before calling
     * start because this information is exchanged in the service discovery phase.
     * Send empty strings for any fields that you cannot provide.
     * @param make The car make.
     * @param model The car model.
     * @param year Car model year.
     * @param id Typically an Id that that can identify this car but can also be
     *        reset by the user. By definition, this cannot be the vin, it should
     *        be a separately generated uuid for instance which can be reset on
     *        user request.
     */
    void setIdentityInfo(const string& make, const string& model, const string& year, const string& id) {
        mMake = make;
        mModel = model;
        mYear = year;
        mId = id;
    }
    /**
     * @internal
     * Set up information about the head unit. Send empty strings for any fields that you cannot provide.
     * @param huMake The make of the head unit.
     * @param huModel The model of the head unit.
     * @param huSwBuild The software build of the head unit.
     * @param huSwVersion The software version of the head unit.
     */
    void setHeadUnitInfo(const string& huMake, const string& huModel,
            const string& huSwBuild, const string& huSwVersion) {
        mHuMake = huMake;
        mHuModel = huModel;
        mHuSwBuild = huSwBuild;
        mHuSwVersion = huSwVersion;
    }
    /**
     * @internal
     * Lets the phone know which side the driver sits on.
     * @param position See the DriverPosition proto.
     */
    void setDriverPosition(int position) { mDriverPosition = position; }
    /**
     * @internal
     * Sets the session configuration bitmask
     * @param sessionConfiguration See SessionConfiguration proto
     */
    void setSessionConfiguration(int sessionConfiguration) {
        mSessionConfiguration = sessionConfiguration;
    }
    void registerCallbacks(
            const shared_ptr<IControllerCallbacks>& controllerCallbacks) {
        mControllerCallbacks = controllerCallbacks;
    }
    void unrecoverableError(MessageStatus err) {
        mControllerCallbacks->unrecoverableErrorCallback(err);
    }
    /**
     * Ping the other side to check to see if it is alive.
     * @param timestamp The local timestamp. This is repeated by the other side
     *        in the response so it can be used to do interesting things like
     *        computing round trip delays.
     * @param bugReport Should the other side save away a bug report locally. The
     *        other side may choose to ignore this request if it is deemed to be
     *        too high frequency.
     */
    void sendPingRequest(int64_t timestamp, bool bugReport);
    bool setClientCreds(const string& rootCert, const string& clientCert, const string& privKey);
    void setNavigationFocus(NavFocusType type);
    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr) { };
    /**
     * Send ByeByeRequest to the other side. After sending this, ByeByeResponse will be coming,
     * and that is the end of the current connection.
     * @param reason reason for finishing.
     */
    void sendByeByeRequest(ByeByeReason reason);
    /**
     * Send ByeByeResponse as a response to ByeByeRequest. This should be sent only after getting
     * ByeByeRequest, which will be notified via byeByeRequestCallback(..).
     */
    void sendByeByeResponse();
    /**
     * Set audio focus.
     * @param focusState Can be one of AUDIO_FOCUS_STATE_GAIN, AUDIO_FOCUS_STATE_LOSS,
     *        AUDIO_FOCUS_STATE_LOSS_TRANSIENT.
     * @param unsolicited whether this is a response to AudioFocusRequestNotification (false) or
     *        sink's notification (true).
     */
    void setAudioFocus(AudioFocusStateType focusState, bool unsolicited);
    void setCertificateVerificationTime(time_t time) {
        mSslWrapper.setCertificateVerificationTime(time);
    }

private:
    int sendVersionRequest();
    int handleVersionResponse(void* msg, size_t len);
    int startAuthentication();
    int handleServiceDiscoveryRequest(const ServiceDiscoveryRequest& req);
    int sendServiceDiscoveryResponse();
    int handlePingRequest(const PingRequest& req);
    int handlePingResponse(const PingResponse& resp);
    void sendAuthResponse(int status);

    bool mHealthy;
    string mMake;
    string mModel;
    string mYear;
    string mId;
    string mHuMake;
    string mHuModel;
    string mHuSwBuild;
    string mHuSwVersion;
    int mDriverPosition;
    int mSessionConfiguration;
    shared_ptr<IControllerCallbacks> mControllerCallbacks;
    SslWrapper mSslWrapper;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_CONTROLLER_H
