// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SINK_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SINK_H

#include "common.h"
#include "IAudioSinkCallbacks.h"
#include "MediaSinkBase.h"

/**
 * This sink handles audio data that must be played by the car. Each audio sink represents a
 * distinct audio stream type, so there would be one for media, one for voice prompts etc.
 * The general flow of how such a sink would operate is described below, you must repeat it
 * once for each stream type.
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      AudioSink* endpoint = new AudioSink(serviceId, galReceiver->messageRouter());
 *      endpoint->registerCallbacks(callbacks); // Subclassed from IAudioSinkCallbacks.
 *      endpoint->setAudioType(...);
 *      endpoint->addSupportedConfiguration(...);
 *      endpoint->addSupportedConfiguration(...);
 *      endpoint->setMaxUnackedFrames(...);
 *
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 *
 *      setupCallback() gets invoked.
 *          perform any initialization necessary for playback for this codec.
 *
 *      codecConfigCallback() gets invoked.
 *          perform codec specific setup steps.
 *
 *      sourceVideoConfigCallback() gets invoked.
 *          ensure you have a surface of the appropriate resolution to begin playback.
 *
 *      playbackStartCallback gets invoked.
 *
 *      dataAvailableCallback gets invoked.
 *          schedule new frame for decoding
 *
 *      A frame completes playing.
 *          call endpoint->ackFrames(1); // MediaSinkBase::ackFrames();
 * </pre>
 * <br>
 * Audio Stream types can be classified as:
 * <ul>
 *      <li>AUDIO_STREAM_GUIDANCE: Navigation and other speech to text prompts. </li>
 *      <li>AUDIO_STREAM_SYSTEM_AUDIO: Notification sounds, alarms, UI clicks etc. </li>
 *      <li>AUDIO_STREAM_MEDIA: Any kind of music.</li>
 *      <li>AUDIO_STREAM_TELEPHONY: Call audio. This is currently not supported, voice calls
 *          are handled over bluetooth. </li>
 * </ul>
 */
class AudioSink : public MediaSinkBase {
public:
    AudioSink (uint8_t id, MessageRouter* router) : MediaSinkBase(id, router) { }
    void addDiscoveryInfo(ServiceDiscoveryResponse* sdr);
    void registerCallbacks(const shared_ptr<IAudioSinkCallbacks>& callbacks) {
        mCallbacks = callbacks;
    }

    /**
     * Add a supported audio configuration. Multiple audio configurations may be supported and
     * so this method can be called repeatedly to add all of them.
     * @param samplingRate The sampling rate.
     * @param numberOfBits Number of bits per sample.
     * @param numberOfChannels Number of channels.
     * @return index The index of this configuration.
     */
    uint32_t addSupportedConfiguration(uint32_t samplingRate, uint32_t numberOfBits,
            uint32_t numberOfChannels);

    /**
     * Set the maximum number of frames that can be outstanding.
     * @param maxUnackedFrames The maximum number of frames that will be kept outstanding. Once
     *        this limit is hit, the other end will no longer send frames.
     */
    void setMaxUnackedFrames(int maxUnackedFrames) {
        mMaxUnackedFrames = maxUnackedFrames;
    }

    /**
     * Set the audio stream type of this audio channel.
     * @param audioType Can be one of AUDIO_STREAM_GUIDANCE, AUDIO_STREAM_SYSTEM_AUDIO,
     *        AUDIO_STREAM_MEDIA, AUDIO_STREAM_TELEPHONY.
     */
    void setAudioType(int audioType) {
        mAudioType = (AudioStreamType) audioType;
    }

    /**
     * Call this *only* if you want to change the preferred configuration from what was
     * set during registration time. This will cause all future audio setup transactions
     * to have the new preferred configuration. It will not affect any already started
     * audio channels. To have this take effect, you must stop and start the channel.
     * If this is not specified, configuration will be preferred by the order of addition: First
     * added will have higher priority.
     * @param configIndices Array of configuration indices.
     * @param size number of indices in configIndices.
     * @return true if the index is valid, false if not.
     */
    bool resetConfigurationPreference(uint32_t* configIndices, int size);

protected:
    virtual int handleSetup(int mediaCodecType);
    virtual int handleCodecConfig(void* config, size_t len);
    void playbackStart(int32_t sessionId);
    void playbackStop(int32_t sessionId);

    bool handleVideoFocusRequest(const VideoFocusRequestNotification& req) { return false; }
    int handleMediaConfiguration(int index);
    unsigned int getNumberOfConfigurations();

private:
    void handleDataAvailable(uint64_t timestamp, const shared_ptr<IoBuffer>& data, size_t offset);

    shared_ptr<IAudioSinkCallbacks> mCallbacks;
    vector<shared_ptr<AudioConfiguration> > mAudioConfigurations;
    AudioStreamType mAudioType;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_AUDIO_SINK_H
